/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.s3;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.invoke.MethodHandles;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import org.apache.solr.s3.S3Exception;
import org.apache.solr.s3.S3StorageClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.CompletedPart;
import software.amazon.awssdk.services.s3.model.UploadPartRequest;
import software.amazon.awssdk.services.s3.model.UploadPartResponse;

public class S3OutputStream
extends OutputStream {
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    static final int PART_SIZE = 0x1000000;
    static final int MIN_PART_SIZE = 0x500000;
    private final S3Client s3Client;
    private final String bucketName;
    private final String key;
    private volatile boolean closed;
    private final ByteBuffer buffer;
    private MultipartUpload multiPartUpload;

    public S3OutputStream(S3Client s3Client, String key, String bucketName) {
        this.s3Client = s3Client;
        this.bucketName = bucketName;
        this.key = key;
        this.closed = false;
        this.buffer = ByteBuffer.allocate(0x1000000);
        this.multiPartUpload = null;
        if (log.isDebugEnabled()) {
            log.debug("Created S3OutputStream for bucketName '{}' key '{}'", (Object)bucketName, (Object)key);
        }
    }

    @Override
    public void write(int b) throws IOException {
        if (this.closed) {
            throw new IOException("Stream closed");
        }
        this.buffer.put((byte)b);
        if (!this.buffer.hasRemaining()) {
            this.uploadPart();
        }
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        int lenRemaining;
        int firstPart;
        if (this.closed) {
            throw new IOException("Stream closed");
        }
        if (S3OutputStream.outOfRange(off, b.length) || len < 0 || S3OutputStream.outOfRange(off + len, b.length)) {
            throw new IndexOutOfBoundsException();
        }
        if (len == 0) {
            return;
        }
        int currentOffset = off;
        for (lenRemaining = len; this.buffer.remaining() < lenRemaining; lenRemaining -= firstPart) {
            firstPart = this.buffer.remaining();
            this.buffer.put(b, currentOffset, firstPart);
            this.uploadPart();
            currentOffset += firstPart;
        }
        if (lenRemaining > 0) {
            this.buffer.put(b, currentOffset, lenRemaining);
        }
    }

    private static boolean outOfRange(int off, int len) {
        return off < 0 || off > len;
    }

    private void uploadPart() throws IOException {
        int size = this.buffer.position() - this.buffer.arrayOffset();
        if (size == 0) {
            return;
        }
        if (this.multiPartUpload == null) {
            if (log.isDebugEnabled()) {
                log.debug("New multi-part upload for bucketName '{}' key '{}'", (Object)this.bucketName, (Object)this.key);
            }
            this.multiPartUpload = this.newMultipartUpload();
        }
        try (ByteArrayInputStream inputStream = new ByteArrayInputStream(this.buffer.array(), this.buffer.arrayOffset(), size);){
            this.multiPartUpload.uploadPart(inputStream, size);
        }
        catch (Exception e) {
            if (this.multiPartUpload != null) {
                this.multiPartUpload.abort();
                if (log.isDebugEnabled()) {
                    log.debug("Multipart upload aborted for bucketName '{}' key '{}'.", (Object)this.bucketName, (Object)this.key);
                }
            }
            throw new S3Exception("Part upload failed: ", e);
        }
        this.buffer.clear();
    }

    @Override
    public void flush() throws IOException {
        if (this.closed) {
            throw new IOException("Stream closed");
        }
        if (this.buffer.position() - this.buffer.arrayOffset() >= 0x500000) {
            this.uploadPart();
        }
    }

    @Override
    public void close() throws IOException {
        if (this.closed) {
            return;
        }
        if (this.multiPartUpload != null && this.multiPartUpload.aborted) {
            this.multiPartUpload = null;
            this.closed = true;
            return;
        }
        this.uploadPart();
        if (this.multiPartUpload != null) {
            this.multiPartUpload.complete();
        }
        this.multiPartUpload = null;
        this.closed = true;
    }

    private MultipartUpload newMultipartUpload() throws IOException {
        try {
            return new MultipartUpload(this.s3Client.createMultipartUpload(b -> b.bucket(this.bucketName).key(this.key)).uploadId());
        }
        catch (SdkException e) {
            throw S3StorageClient.handleAmazonException(e);
        }
    }

    private class MultipartUpload {
        private final String uploadId;
        private final List<CompletedPart> completedParts;
        private boolean aborted = false;

        public MultipartUpload(String uploadId) {
            this.uploadId = uploadId;
            this.completedParts = new ArrayList<CompletedPart>();
            if (log.isDebugEnabled()) {
                log.debug("Initiated multi-part upload for bucketName '{}' key '{}' with id '{}'", new Object[]{S3OutputStream.this.bucketName, S3OutputStream.this.key, uploadId});
            }
        }

        void uploadPart(ByteArrayInputStream inputStream, long partSize) {
            if (this.aborted) {
                throw new IllegalStateException("Can't upload new parts on a MultipartUpload that was aborted. id '" + this.uploadId + "'");
            }
            int currentPartNumber = this.completedParts.size() + 1;
            UploadPartRequest request = (UploadPartRequest)UploadPartRequest.builder().key(S3OutputStream.this.key).bucket(S3OutputStream.this.bucketName).uploadId(this.uploadId).partNumber(Integer.valueOf(currentPartNumber)).build();
            if (log.isDebugEnabled()) {
                log.debug("Uploading part {} for id '{}'", (Object)currentPartNumber, (Object)this.uploadId);
            }
            UploadPartResponse response = S3OutputStream.this.s3Client.uploadPart(request, RequestBody.fromInputStream((InputStream)inputStream, (long)partSize));
            this.completedParts.add((CompletedPart)CompletedPart.builder().partNumber(Integer.valueOf(currentPartNumber)).eTag(response.eTag()).build());
        }

        void complete() {
            if (this.aborted) {
                throw new IllegalStateException("Can't complete a MultipartUpload that was aborted. id '" + this.uploadId + "'");
            }
            if (log.isDebugEnabled()) {
                log.debug("Completing multi-part upload for key '{}', id '{}'", (Object)S3OutputStream.this.key, (Object)this.uploadId);
            }
            S3OutputStream.this.s3Client.completeMultipartUpload(b -> b.bucket(S3OutputStream.this.bucketName).key(S3OutputStream.this.key).uploadId(this.uploadId).multipartUpload(mub -> mub.parts(this.completedParts)));
        }

        public void abort() {
            if (log.isWarnEnabled()) {
                log.warn("Aborting multi-part upload with id '{}'", (Object)this.uploadId);
            }
            try {
                S3OutputStream.this.s3Client.abortMultipartUpload(b -> b.bucket(S3OutputStream.this.bucketName).key(S3OutputStream.this.key).uploadId(this.uploadId));
            }
            catch (Exception e) {
                log.error("Unable to abort multipart upload, you may need to purge uploaded parts: ", (Throwable)e);
            }
            this.aborted = true;
        }
    }
}

