package eachers

import (
	"errors"
	"fmt"
	"reflect"
	"strings"

	"github.com/onsi/gomega/types"
)

type matchFailure struct {
	name             string
	actual, expected interface{}
}

type arg struct {
	channel reflect.Value
	name    string
}

// Called is a gomega/types.GomegaMatcher that asserts on method calls
// for mocks generated by github.com/nelsam/hel
type Called struct {
	calls         [][]interface{}
	matched       [][]interface{}
	matchFailures []string
}

// BeCalled takes 0 or more slices of arguments and returns a Called
// matcher for those arguments.
func BeCalled(calls ...[]interface{}) *Called {
	return &Called{calls: calls}
}

// With is just sugar so you can type `BeCalled(With(arg0, arg1))`
// instead of `BeCalled([]interface{}{arg0, arg1})`
func With(args ...interface{}) []interface{} {
	return args
}

// Match asserts that actual was called.  If there are any argument
// slices in c, it will assert that actual was called with those
// arguments.
func (c *Called) Match(actual interface{}) (success bool, err error) {
	val := reflect.ValueOf(actual)
	if val.Kind() != reflect.Struct && val.Kind() != reflect.Chan {
		return false, errors.New("BeCalled requires a channel or struct full of channel fields")
	}
	if len(c.calls) == 0 {
		c.calls = append(c.calls, []interface{}{})
	}
	success = true
	for _, call := range c.calls {
		args := args(val)
		actualCall := make([]interface{}, 0, len(args))
		for i, arg := range args {
			recvd, ok := receiveArg(arg)
			if !ok {
				return false, nil
			}
			actualCall = append(actualCall, recvd)
			if i >= len(call) {
				continue
			}
			switch src := call[i].(type) {
			case types.GomegaMatcher:
				matched, err := src.Match(recvd)
				if err != nil {
					return false, err
				}
				if !matched {
					success = false
					c.matchFailures = append(c.matchFailures, fmt.Sprintf("%s: %s", arg.name, src.FailureMessage(recvd)))
				}
			default:
				if !reflect.DeepEqual(call[i], recvd) {
					success = false
					c.matchFailures = append(c.matchFailures, fmt.Sprintf("%s: %v (expected) != %v (actual)", arg.name, call[i], recvd))
				}
			}
		}
		if !success {
			// This was actually an unmatched call, so reset matched
			c.calls = append(c.calls, c.matched...)
			c.matched = nil
			break
		}
		c.matched = append([][]interface{}{call}, c.matched...)
		c.calls = c.calls[1:]
	}
	return success, nil
}

func receiveArg(arg arg) (interface{}, bool) {
	cases := []reflect.SelectCase{
		{Dir: reflect.SelectRecv, Chan: arg.channel},
		{Dir: reflect.SelectDefault},
	}
	_, recvd, _ := reflect.Select(cases)
	if recvd.Kind() == reflect.Invalid {
		return nil, false
	}
	return recvd.Interface(), true
}

// FailureMessage returns the message when c failed to match a call.
func (c *Called) FailureMessage(actual interface{}) (message string) {
	message = fmt.Sprintf("Expected %#v to be called", actual)
	if len(c.matchFailures) == 0 {
		return message
	}
	return message + " with args:\n" + strings.Join(c.matchFailures, "\n")
}

// NegatedFailureMessage returns the message when c matched a call and
// shouldn't have.
func (c *Called) NegatedFailureMessage(actual interface{}) (message string) {
	msg := fmt.Sprintf("Expected %#v not to be called", actual)
	if len(c.matched[0]) == 0 {
		return msg
	}
	return fmt.Sprintf("%s with args (%s)", msg, joinInter(c.matched[0], ", "))
}

func joinInter(a []interface{}, sep string) string {
	strs := make([]string, 0, len(a))
	for _, v := range a {
		strs = append(strs, fmt.Sprintf("%v", v))
	}
	return strings.Join(strs, sep)
}

func args(val reflect.Value) []arg {
	if val.Kind() == reflect.Chan {
		return []arg{{channel: val}}
	}
	args := make([]arg, 0, val.NumField())
	for i := 0; i < val.NumField(); i++ {
		arg := arg{
			name:    val.Type().Field(i).Name,
			channel: val.Field(i),
		}
		args = append(args, arg)
	}
	return args
}
