package ridata

import (
	"bytes"
	"testing"

	"encoding/hex"

	"github.com/dsoprea/go-logging"
	"github.com/dsoprea/go-utility/v2/filesystem"
)

func TestGetMimetypeFromContent(t *testing.T) {
	raw := "ffd8ffe000104a46494600010101004800480000ffe1001845786966000049492a00080000000000000000000000ffed004850686f746f73686f7020332e30003842494d040400000000000f1c015a00031b25471c020000020002003842494d0425000000000010fce11f89c8b7c9782f346234075877ebffdb0043000403030403030404030405040405060a07060606060d090a080a0f0d10100f0d0f0e11131814111217120e0f151c151719191b1b1b10141d1f1d1a1f181a1b1affdb0043010405050605060c07070c1a110f111a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1affc20011080313057803012200021101031101ffc4001c0000000701010000000000000000000000000102030405060708ffc4001b01010101010101010100000000000000000001020304050607ffda000c03010002100310000001d2b6b73c7eb94fa55e6f4d25cd2683a48d16533c6cc4854e95ea42dcdaafb5abd490c496196c386c4427c544789d582b52ed4224b4a4971329a8094a8f429db0b49d5074e7c460f7daeebcb8a1f4da3edcf1e57755be6c2907a1181000140c800cca12879c21aa63234b5245bb1d79932656ab17b0f60f244ee4f638e4bd4bad9000eb800000000000000085a21600a000000000000000057c5833cb3957176be558"

	buffer, err := hex.DecodeString(raw)
	log.PanicIf(err)

	b := bytes.NewBuffer(buffer)

	mimetype, err := GetMimetypeFromContent(b, 0)
	log.PanicIf(err)

	if mimetype != "image/jpeg" {
		t.Fatalf("Mime-type not correct: [%s]", mimetype)
	}
}

func TestGetMimetypeFromContent__ShortData_Error(t *testing.T) {
	raw := "ffd8ffe000104a46494600010101004800480000ffe1001845786966000049492a00080000000000000000000000ffed004850686f746f73686f7020332e30003842494d040400000000000f1c015a00031b25471c020000020002003842494d0425000000000010fce11f89c8b7c9782f346234075877ebffdb0043000403030403030404030405040405060a07060606060d090a080a0f0d10100f0d0f0e11131814111217120e0f151c151719191b1b1b10141d1f1d1a1f181a1b1affdb0043010405050605060c07070c1a110f111a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1affc20011080313057803012200021101031101ffc4001c0000000701010000000000000000000000000102030405060708ffc4001b01010101010101010100000000000000000001020304050607ffda000c03010002100310000001d2b6b73c7eb94fa55e6f4d25cd2683a48d16533c6cc4854e95ea42dcdaafb5abd490c496196c386c4427c544789d582b52ed4224b4a4971329a8094a8f429db0b49d5074e7c460f7daeebcb8a1f4da3edcf1e57755be6c2907a1181000140c800cca12879c21aa63234b5245bb1d79932656ab17b0f60f244ee4f638e4bd4bad9000eb800000000000000085a21600a000000000000000057c5833cb3957176be558"

	buffer, err := hex.DecodeString(raw)
	log.PanicIf(err)

	buffer = buffer[:256]

	b := rifs.NewSeekableBufferWithBytes(buffer)

	_, err = GetMimetypeFromContent(b, 0)
	if err == nil {
		t.Fatalf("Expected error for not enough data.")
	} else if err.Error() != "unexpected EOF" {
		log.Panic(err)
	}
}

func TestGetMimetypeFromContent__ShortData_Success(t *testing.T) {
	raw := "ffd8ffe000104a46494600010101004800480000ffe1001845786966000049492a00080000000000000000000000ffed004850686f746f73686f7020332e30003842494d040400000000000f1c015a00031b25471c020000020002003842494d0425000000000010fce11f89c8b7c9782f346234075877ebffdb0043000403030403030404030405040405060a07060606060d090a080a0f0d10100f0d0f0e11131814111217120e0f151c151719191b1b1b10141d1f1d1a1f181a1b1affdb0043010405050605060c07070c1a110f111a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1a1affc20011080313057803012200021101031101ffc4001c0000000701010000000000000000000000000102030405060708ffc4001b01010101010101010100000000000000000001020304050607ffda000c03010002100310000001d2b6b73c7eb94fa55e6f4d25cd2683a48d16533c6cc4854e95ea42dcdaafb5abd490c496196c386c4427c544789d582b52ed4224b4a4971329a8094a8f429db0b49d5074e7c460f7daeebcb8a1f4da3edcf1e57755be6c2907a1181000140c800cca12879c21aa63234b5245bb1d79932656ab17b0f60f244ee4f638e4bd4bad9000eb800000000000000085a21600a000000000000000057c5833cb3957176be558"

	buffer, err := hex.DecodeString(raw)
	log.PanicIf(err)

	buffer = buffer[:256]

	b := rifs.NewSeekableBufferWithBytes(buffer)

	mimetype, err := GetMimetypeFromContent(b, int64(len(buffer)))
	log.PanicIf(err)

	if mimetype != "image/jpeg" {
		t.Fatalf("Mime-type not correct: [%s]", mimetype)
	}
}
