/*
 * PGFConsole: A PGF-codec demonstration
 * $Date: 2006-05-09 20:13:33 +0200 (Di, 09 Mai 2006) $
 * $Revision: 187 $

 * This file Copyright (C) 2006 xeraina GmbH, Switzerland
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "CImage.h"
#include "FreeImagePlus.h"
#include <cassert>

/////////////////////////////////////////////////////////////////////////////
CImage::CImage() { 
	m_image = new fipImage(); 

#ifdef __PNMEXSUPPORT__
	SetMaxValue(0);
#endif
}

/////////////////////////////////////////////////////////////////////////////
CImage::~CImage() { 
	delete m_image; 
}

/////////////////////////////////////////////////////////////////////////////
// default create function (standard bitmap with color depth)
bool CImage::Create(int width, int height, int bpp) {
	return TRUE == m_image->setSize(FIT_BITMAP, (WORD)width, (WORD)height, (WORD)bpp);
}

/////////////////////////////////////////////////////////////////////////////
// image format and bpp are chosen by given type
bool CImage::Create(int width, int height, unsigned char type) {
	FREE_IMAGE_TYPE FItype;
	int bpp;
	switch (type) {
		case ImageModeBitmap:
			FItype = FIT_BITMAP;
			bpp = 1;
			break;

		case ImageModeIndexedColor:
		case ImageModeGrayScale:
			FItype = FIT_BITMAP;
			bpp = 8;	// only 8-bit is supported
			break;
		
		case ImageModeGray16:
			FItype = FIT_UINT16;
			bpp = 16;
			break;

		case ImageModeRGB16:
			FItype = FIT_BITMAP;
			bpp = 16;
			// RGB565 image needs RGB mask
			return TRUE == m_image->setSize(FItype, (WORD)width, (WORD)height, (WORD)bpp, FI16_565_RED_MASK, FI16_565_GREEN_MASK, FI16_565_BLUE_MASK);

		case ImageModeRGBColor:
			FItype = FIT_BITMAP;
			bpp = 24;
			break;

		case ImageModeRGB48:
			FItype = FIT_RGB16;
			bpp = 48;
			break;

		case ImageModeRGBA:
			FItype = FIT_BITMAP;
			bpp = 32;
			break;

		default:
			return false;
	}
	return TRUE == m_image->setSize(FItype, (WORD)width, (WORD)height, (WORD)bpp);
}

/////////////////////////////////////////////////////////////////////////////
unsigned char* CImage::GetBits() const {
	return (unsigned char*)m_image->accessPixels();
}

/////////////////////////////////////////////////////////////////////////////
int CImage::GetPitch() const {
	return m_image->getScanWidth();
}

/////////////////////////////////////////////////////////////////////////////
unsigned char CImage::GetBPP() const {
	return (unsigned char)m_image->getBitsPerPixel();
}

/////////////////////////////////////////////////////////////////////////////
unsigned int CImage::GetHeight() const {
	return m_image->getHeight();
}

/////////////////////////////////////////////////////////////////////////////
unsigned int CImage::GetWidth() const {
	return m_image->getWidth();
}

/////////////////////////////////////////////////////////////////////////////
bool CImage::Save(const char* dest) {
	return TRUE == m_image->save(dest, 0);
}

/////////////////////////////////////////////////////////////////////////////
bool CImage::Load(const char* source) {
	if (m_image->load(source, 0)) {
		if (GetColorType() == ImageModeRGB16) {
			// force RGB-565 for 16-bit images since PGF does not support RGB-555
			m_image->convertTo16Bits565();
		}
		return true;
	}
	return false;
}

/////////////////////////////////////////////////////////////////////////////
bool CImage::IsTransparencySupported() {
	return (m_image->getColorType() == FIC_RGBALPHA);
}

/////////////////////////////////////////////////////////////////////////////
bool CImage::IsIndexed() {
	return ((m_image->getColorType() == FIC_PALETTE) ||
		(m_image->getColorType() == FIC_MINISBLACK) ||
		(m_image->getColorType() == FIC_MINISWHITE));
}

/////////////////////////////////////////////////////////////////////////////
int CImage::GetMaxColorTableEntries() {
	return m_image->getColorsUsed();
}

/////////////////////////////////////////////////////////////////////////////
void CImage::GetColorTable(int firstColor, int numColors, void* prgbColors) {
	RGBQUAD* palette = m_image->getPalette();
	int max = firstColor + numColors;
	// ugly hack but necessary due to conflict between PGFplatform.h and FreeImage.h
	RGBQUAD* target = (RGBQUAD*)prgbColors;
	for (int i = firstColor; i < max; ++i) {
		target[i] = palette[i];	
	}
}

/////////////////////////////////////////////////////////////////////////////
void CImage::SetColorTable(int firstColor, int numColors, const void* prgbColors) {
	RGBQUAD* palette = m_image->getPalette();
	int max = firstColor + numColors;
	// ugly hack but necessary due to conflict between PGFplatform.h and FreeImage.h
	RGBQUAD* source = (RGBQUAD*)prgbColors;
	for (int i = firstColor; i < max; ++i) {
		palette[i] = source[i];
	}
}

/////////////////////////////////////////////////////////////////////////////
unsigned char CImage::GetColorType() const {
	switch (m_image->getImageType()) {
		case FIT_BITMAP:
			// further analysis necessary
			switch (m_image->getColorType()) {
				case FIC_MINISBLACK:
				case FIC_MINISWHITE:
					switch (m_image->getBitsPerPixel()) {
						case 1:
							return ImageModeBitmap;
						
						case 8:
							return ImageModeGrayScale;

						case 16:
							return ImageModeGray16;

						default:
							return ImageModeUnknown;
					}

				case FIC_PALETTE:
					return ImageModeIndexedColor;

				case FIC_RGB:
					switch (m_image->getBitsPerPixel()) {
						case 16:
							return ImageModeRGB16;

						case 24:
						case 32:
							return ImageModeRGBColor;

						case 48:
						case 64:
							return ImageModeRGB48;

						default:
							return ImageModeUnknown;
					}
					break;

				case FIC_RGBALPHA:
					switch (m_image->getBitsPerPixel()) {
						case 32:
							return ImageModeRGBA;

						default:
							return ImageModeUnknown;
					}
					break;

				default:
					return ImageModeUnknown;
			}
			break;

		case FIT_UINT16:
			if (m_image->getColorType() == FIC_RGB)
				return ImageModeRGB16;
			else
				return ImageModeGray16;

		case FIT_RGB16:
			return ImageModeRGB48;

		default:
			return ImageModeUnknown;
	}
}

/////////////////////////////////////////////////////////////////////////////
int CImage::GetChannelDepth() const {
	switch(GetColorType()) {
		case ImageModeBitmap:
			return 1;

		case ImageModeRGB16:
			// it's actually RGB-565, but returning 5 is ok
			return 5;

		case ImageModeGrayScale:
		case ImageModeIndexedColor:
		case ImageModeRGBColor:
		case ImageModeRGBA:
			return 8;

		case ImageModeGray16:
		case ImageModeRGB48:
			return 16;

		default:
			assert(false);
			return 0;
	}
}

/////////////////////////////////////////////////////////////////////////////
int CImage::GetChannels() const {
	switch(GetColorType()) {
		case ImageModeBitmap:
		case ImageModeGrayScale:
		case ImageModeGray16:
		case ImageModeIndexedColor:
			return 1;

		case ImageModeRGB16:
		case ImageModeRGBColor:
		case ImageModeRGB48:
			return 3;

		case ImageModeRGBA:
			return 4;

		default:
			assert(false);
			return 0;
	}
}

#ifdef __PNMEXSUPPORT__
/////////////////////////////////////////////////////////////////////////////
// Register another PGM plugin
extern void DLL_CALLCONV InitPNM(Plugin *plugin, int format_id);

void CImage::RegisterPNM() {
	FREE_IMAGE_FORMAT fif = FreeImage_RegisterLocalPlugin(InitPNM);
	// disable original pnm plugin
	FreeImage_SetPluginEnabled(FIF_PBMRAW, FALSE);
	FreeImage_SetPluginEnabled(FIF_PBM, FALSE);
	FreeImage_SetPluginEnabled(FIF_PGMRAW, FALSE);
	FreeImage_SetPluginEnabled(FIF_PGM, FALSE);
	FreeImage_SetPluginEnabled(FIF_PPMRAW, FALSE);
	FreeImage_SetPluginEnabled(FIF_PPM, FALSE);
	// enable new pnm plugin
	FreeImage_SetPluginEnabled(fif, TRUE);
}

/////////////////////////////////////////////////////////////////////////////
extern int s_maxval;

void CImage::SetMaxValue(int maxValue) {
	s_maxval = maxValue;
}

int CImage::GetMaxValue() const {
	return (s_maxval > 0) ? s_maxval : ((1 << GetChannelDepth()) - 1);
}

#endif

