// Copyright 2018 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package validate

import (
	"testing"

	"github.com/google/go-cmp/cmp"
	"github.com/google/go-cmp/cmp/cmpopts"
	"github.com/openconfig/gnmi/errdiff"
	"github.com/openconfig/ygot/exampleoc"
	"github.com/openconfig/ygot/exampleoc/opstateoc"
	"github.com/openconfig/ygot/uexampleoc"
	"github.com/openconfig/ygot/ygot"
	"github.com/openconfig/ygot/ytypes"
	"google.golang.org/protobuf/proto"

	gpb "github.com/openconfig/gnmi/proto/gnmi"
)

func mustSchema(fn func() (*ytypes.Schema, error)) *ytypes.Schema {
	s, err := fn()
	if err != nil {
		panic(err)
	}
	return s
}

func TestGetOrCreateMap(t *testing.T) {
	root := &exampleoc.Device{}
	m := root.GetOrCreateInterfaceMap()
	name := "foo"
	m[name] = &exampleoc.Interface{Name: ygot.String(name)}
	if root.GetInterface(name) == nil {
		t.Errorf("must not be nil")
	}
}

func TestSet(t *testing.T) {
	invalidSchemaDueToMissingKeyField := mustSchema(exampleoc.Schema)
	invalidRoot := invalidSchemaDueToMissingKeyField.Root.(*exampleoc.Device)
	aclSet := invalidRoot.GetOrCreateAcl().GetOrCreateAclSet("foo", exampleoc.Acl_ACL_TYPE_ACL_IPV4)
	aclSet.Name = nil

	tests := []struct {
		desc             string
		inSchema         *ytypes.Schema
		inPath           *gpb.Path
		inValue          *gpb.TypedValue
		inOpts           []ytypes.SetNodeOpt
		inGetOpts        []ytypes.GetNodeOpt
		wantErrSubstring string
		wantNode         *ytypes.TreeNode
	}{{
		desc:     "set-on-union-with-invalid-string-but-valid-enum",
		inSchema: mustSchema(exampleoc.Schema),
		inPath: &gpb.Path{
			Elem: []*gpb.PathElem{{
				Name: "routing-policy",
			}, {
				Name: "policy-definitions",
			}, {
				Name: "policy-definition",
				Key: map[string]string{
					"name": "test",
				},
			}, {
				Name: "statements",
			}, {
				Name: "statement",
				Key: map[string]string{
					"name": "test-stmt",
				},
			}, {
				Name: "actions",
			}, {
				Name: "bgp-actions",
			}, {
				Name: "set-community",
			}, {
				Name: "inline",
			}, {
				Name: "config",
			}, {
				Name: "communities",
			}},
		},
		inValue: &gpb.TypedValue{
			Value: &gpb.TypedValue_LeaflistVal{
				LeaflistVal: &gpb.ScalarArray{
					Element: []*gpb.TypedValue{{
						Value: &gpb.TypedValue_StringVal{StringVal: "openconfig-bgp-types:NO_ADVERTISE"},
					}},
				},
			},
		},
		inOpts: []ytypes.SetNodeOpt{&ytypes.InitMissingElements{}},
		wantNode: &ytypes.TreeNode{
			Path: &gpb.Path{
				Elem: []*gpb.PathElem{{
					Name: "routing-policy",
				}, {
					Name: "policy-definitions",
				}, {
					Name: "policy-definition",
					Key: map[string]string{
						"name": "test",
					},
				}, {
					Name: "statements",
				}, {
					Name: "statement",
					Key: map[string]string{
						"name": "test-stmt",
					},
				}, {
					Name: "actions",
				}, {
					Name: "bgp-actions",
				}, {
					Name: "set-community",
				}, {
					Name: "inline",
				}, {
					Name: "config",
				}, {
					Name: "communities",
				}},
			},
			Data: []exampleoc.RoutingPolicy_PolicyDefinition_Statement_Actions_BgpActions_SetCommunity_Inline_Communities_Union{exampleoc.BgpTypes_BGP_WELL_KNOWN_STD_COMMUNITY_NO_ADVERTISE},
		},
	}, {
		desc:     "set-on-union-with-invalid-string-but-valid-enum-no-module",
		inSchema: mustSchema(exampleoc.Schema),
		inPath: &gpb.Path{
			Elem: []*gpb.PathElem{{
				Name: "routing-policy",
			}, {
				Name: "policy-definitions",
			}, {
				Name: "policy-definition",
				Key: map[string]string{
					"name": "test",
				},
			}, {
				Name: "statements",
			}, {
				Name: "statement",
				Key: map[string]string{
					"name": "test-stmt",
				},
			}, {
				Name: "actions",
			}, {
				Name: "bgp-actions",
			}, {
				Name: "set-community",
			}, {
				Name: "inline",
			}, {
				Name: "config",
			}, {
				Name: "communities",
			}},
		},
		inValue: &gpb.TypedValue{
			Value: &gpb.TypedValue_LeaflistVal{
				LeaflistVal: &gpb.ScalarArray{
					Element: []*gpb.TypedValue{{
						Value: &gpb.TypedValue_StringVal{StringVal: "NO_ADVERTISE"},
					}},
				},
			},
		},
		inOpts: []ytypes.SetNodeOpt{&ytypes.InitMissingElements{}},
		wantNode: &ytypes.TreeNode{
			Path: &gpb.Path{
				Elem: []*gpb.PathElem{{
					Name: "routing-policy",
				}, {
					Name: "policy-definitions",
				}, {
					Name: "policy-definition",
					Key: map[string]string{
						"name": "test",
					},
				}, {
					Name: "statements",
				}, {
					Name: "statement",
					Key: map[string]string{
						"name": "test-stmt",
					},
				}, {
					Name: "actions",
				}, {
					Name: "bgp-actions",
				}, {
					Name: "set-community",
				}, {
					Name: "inline",
				}, {
					Name: "config",
				}, {
					Name: "communities",
				}},
			},
			Data: []exampleoc.RoutingPolicy_PolicyDefinition_Statement_Actions_BgpActions_SetCommunity_Inline_Communities_Union{exampleoc.BgpTypes_BGP_WELL_KNOWN_STD_COMMUNITY_NO_ADVERTISE},
		},
	}, {
		desc:     "set-on-union-with-invalid-string-but-valid-enum-json-container",
		inSchema: mustSchema(opstateoc.Schema),
		inPath: &gpb.Path{
			Elem: []*gpb.PathElem{{
				Name: "routing-policy",
			}, {
				Name: "policy-definitions",
			}, {
				Name: "policy-definition",
				Key: map[string]string{
					"name": "foo",
				},
			}},
		},
		inValue: &gpb.TypedValue{
			Value: &gpb.TypedValue_JsonIetfVal{
				JsonIetfVal: []byte("{\n  \"openconfig-routing-policy:state\": {\n    \"name\": \"foo\"\n  },\n  \"openconfig-routing-policy:name\": \"foo\",\n  \"openconfig-routing-policy:statements\": {\n    \"statement\": [\n      {\n        \"actions\": {\n          \"openconfig-bgp-policy:bgp-actions\": {\n            \"set-community\": {\n              \"state\": {\n                \"method\": \"INLINE\"\n              },\n              \"inline\": {\n                \"state\": {\n                  \"communities\": [\n                    \"openconfig-bgp-types:NO_ADVERTISE\"\n                  ]\n                }\n              }\n            }\n          }\n        },\n        \"state\": {\n          \"name\": \"foo-stmt\"\n        },\n        \"name\": \"foo-stmt\"\n      }\n    ]\n  }\n}"),
			},
		},
		inOpts: []ytypes.SetNodeOpt{&ytypes.InitMissingElements{}},
		wantNode: &ytypes.TreeNode{
			Path: &gpb.Path{
				Elem: []*gpb.PathElem{{
					Name: "routing-policy",
				}, {
					Name: "policy-definitions",
				}, {
					Name: "policy-definition",
					Key: map[string]string{
						"name": "foo",
					},
				}, {
					Name: "statements",
				}, {
					Name: "statement",
					Key: map[string]string{
						"name": "foo-stmt",
					},
				}, {
					Name: "actions",
				}, {
					Name: "bgp-actions",
				}, {
					Name: "set-community",
				}, {
					Name: "inline",
				}, {
					Name: "state",
				}, {
					Name: "communities",
				}},
			},
			Data: []opstateoc.RoutingPolicy_PolicyDefinition_Statement_Actions_BgpActions_SetCommunity_Inline_Communities_Union{opstateoc.OpenconfigBgpTypes_BGP_WELL_KNOWN_STD_COMMUNITY_NO_ADVERTISE},
		},
	}, {
		desc:     "set-on-union-with-invalid-string-but-valid-enum-json-container-shadow-path",
		inSchema: mustSchema(opstateoc.Schema),
		inPath: &gpb.Path{
			Elem: []*gpb.PathElem{{
				Name: "routing-policy",
			}, {
				Name: "policy-definitions",
			}, {
				Name: "policy-definition",
				Key: map[string]string{
					"name": "foo",
				},
			}},
		},
		inValue: &gpb.TypedValue{
			Value: &gpb.TypedValue_JsonIetfVal{
				JsonIetfVal: []byte("{\n  \"openconfig-routing-policy:config\": {\n    \"name\": \"foo\"\n  },\n  \"openconfig-routing-policy:name\": \"foo\",\n  \"openconfig-routing-policy:statements\": {\n    \"statement\": [\n      {\n        \"actions\": {\n          \"openconfig-bgp-policy:bgp-actions\": {\n            \"set-community\": {\n              \"config\": {\n                \"method\": \"INLINE\"\n              },\n              \"inline\": {\n                \"config\": {\n                  \"communities\": [\n                    \"openconfig-bgp-types:NO_ADVERTISE\"\n                  ]\n                }\n              }\n            }\n          }\n        },\n        \"config\": {\n          \"name\": \"foo-stmt\"\n        },\n        \"name\": \"foo-stmt\"\n      }\n    ]\n  }\n}"),
			},
		},
		inOpts:    []ytypes.SetNodeOpt{&ytypes.InitMissingElements{}, &ytypes.PreferShadowPath{}},
		inGetOpts: []ytypes.GetNodeOpt{&ytypes.PreferShadowPath{}},
		wantNode: &ytypes.TreeNode{
			Path: &gpb.Path{
				Elem: []*gpb.PathElem{{
					Name: "routing-policy",
				}, {
					Name: "policy-definitions",
				}, {
					Name: "policy-definition",
					Key: map[string]string{
						"name": "foo",
					},
				}, {
					Name: "statements",
				}, {
					Name: "statement",
					Key: map[string]string{
						"name": "foo-stmt",
					},
				}, {
					Name: "actions",
				}, {
					Name: "bgp-actions",
				}, {
					Name: "set-community",
				}, {
					Name: "inline",
				}, {
					Name: "config",
				}, {
					Name: "communities",
				}},
			},
			Data: []opstateoc.RoutingPolicy_PolicyDefinition_Statement_Actions_BgpActions_SetCommunity_Inline_Communities_Union{opstateoc.OpenconfigBgpTypes_BGP_WELL_KNOWN_STD_COMMUNITY_NO_ADVERTISE},
		},
	}, {
		desc:     "set leafref with mismatched name - compressed schema",
		inSchema: mustSchema(exampleoc.Schema),
		inPath: &gpb.Path{
			Elem: []*gpb.PathElem{{
				Name: "components",
			}, {
				Name: "component",
				Key: map[string]string{
					"name": "OCH-1-2",
				},
			}, {
				Name: "optical-channel",
			}, {
				Name: "config",
			}, {
				Name: "line-port",
			}},
		},
		inValue: &gpb.TypedValue{
			Value: &gpb.TypedValue_StringVal{"XCVR-1-2"},
		},
		inOpts: []ytypes.SetNodeOpt{&ytypes.InitMissingElements{}},
		wantNode: &ytypes.TreeNode{
			Path: &gpb.Path{
				Elem: []*gpb.PathElem{{
					Name: "components",
				}, {
					Name: "component",
					Key: map[string]string{
						"name": "OCH-1-2",
					},
				}, {
					Name: "optical-channel",
				}, {
					Name: "config",
				}, {
					Name: "line-port",
				}},
			},
			Data: ygot.String("XCVR-1-2"),
		},
	}, {
		desc:     "set leafref with mismatched name - uncompressed schema",
		inSchema: mustSchema(uexampleoc.Schema),
		inPath: &gpb.Path{
			Elem: []*gpb.PathElem{{
				Name: "components",
			}, {
				Name: "component",
				Key: map[string]string{
					"name": "OCH-1-2",
				},
			}, {
				Name: "optical-channel",
			}, {
				Name: "state",
			}, {
				Name: "line-port",
			}},
		},
		inValue: &gpb.TypedValue{
			Value: &gpb.TypedValue_StringVal{"XCVR-1-2"},
		},
		inOpts: []ytypes.SetNodeOpt{&ytypes.InitMissingElements{}},
		wantNode: &ytypes.TreeNode{
			Path: &gpb.Path{
				Elem: []*gpb.PathElem{{
					Name: "components",
				}, {
					Name: "component",
					Key: map[string]string{
						"name": "OCH-1-2",
					},
				}, {
					Name: "optical-channel",
				}, {
					Name: "state",
				}, {
					Name: "line-port",
				}},
			},
			Data: ygot.String("XCVR-1-2"),
		},
	}, {
		desc:     "set list with union type - compressed schema",
		inSchema: mustSchema(exampleoc.Schema),
		inPath: &gpb.Path{
			Elem: []*gpb.PathElem{{
				Name: "network-instances",
			}, {
				Name: "network-instance",
				Key: map[string]string{
					"name": "OCH-1-2",
				},
			}, {
				Name: "afts",
			}, {
				Name: "mpls",
			}, {
				Name: "label-entry",
				Key: map[string]string{
					"label": "483414",
				},
			}, {
				Name: "state",
			}, {
				Name: "next-hop-group",
			}},
		},
		inValue: &gpb.TypedValue{
			Value: &gpb.TypedValue_UintVal{5},
		},
		inOpts: []ytypes.SetNodeOpt{&ytypes.InitMissingElements{}},
		wantNode: &ytypes.TreeNode{
			Path: &gpb.Path{
				Elem: []*gpb.PathElem{{
					Name: "network-instances",
				}, {
					Name: "network-instance",
					Key: map[string]string{
						"name": "OCH-1-2",
					},
				}, {
					Name: "afts",
				}, {
					Name: "mpls",
				}, {
					Name: "label-entry",
					Key: map[string]string{
						"label": "483414",
					},
				}, {
					Name: "state",
				}, {
					Name: "next-hop-group",
				}},
			},
			Data: ygot.Uint64(5),
		},
	}, {
		desc:     "set leafref with mismatched name - operational state (compressed) schema",
		inSchema: mustSchema(opstateoc.Schema),
		inPath: &gpb.Path{
			Elem: []*gpb.PathElem{{
				Name: "components",
			}, {
				Name: "component",
				Key: map[string]string{
					"name": "OCH-1-2",
				},
			}, {
				Name: "optical-channel",
			}, {
				Name: "state",
			}, {
				Name: "line-port",
			}},
		},
		inValue: &gpb.TypedValue{
			Value: &gpb.TypedValue_StringVal{"XCVR-1-2"},
		},
		inOpts: []ytypes.SetNodeOpt{&ytypes.InitMissingElements{}},
		wantNode: &ytypes.TreeNode{
			Path: &gpb.Path{
				Elem: []*gpb.PathElem{{
					Name: "components",
				}, {
					Name: "component",
					Key: map[string]string{
						"name": "OCH-1-2",
					},
				}, {
					Name: "optical-channel",
				}, {
					Name: "state",
				}, {
					Name: "line-port",
				}},
			},
			Data: ygot.String("XCVR-1-2"),
		},
	}, {
		desc:     "set config (shadowed schema) list key - operational state (compressed) schema - schema ignores shadow-path",
		inSchema: mustSchema(opstateoc.Schema),
		inPath: &gpb.Path{
			Elem: []*gpb.PathElem{{
				Name: "components",
			}, {
				Name: "component",
				Key: map[string]string{
					"name": "OCH-1-2",
				},
			}, {
				Name: "optical-channel",
			}, {
				Name: "config",
			}, {
				Name: "line-port",
			}},
		},
		inValue: &gpb.TypedValue{
			Value: &gpb.TypedValue_StringVal{"XCVR-1-2"},
		},
		inOpts: []ytypes.SetNodeOpt{&ytypes.InitMissingElements{}},
		wantNode: &ytypes.TreeNode{
			Path: &gpb.Path{
				Elem: []*gpb.PathElem{{
					Name: "components",
				}, {
					Name: "component",
					Key: map[string]string{
						"name": "OCH-1-2",
					},
				}, {
					Name: "optical-channel",
				}, {
					Name: "config",
				}, {
					Name: "line-port",
				}},
			},
			// No error, but leaf is not set when shadow-path is provided.
			Data: nil,
		},
	}, {
		desc:     "set state (shadowed schema) key list - compressed schema - schema doesn't contain shadow-path",
		inSchema: mustSchema(exampleoc.Schema),
		inPath: &gpb.Path{
			Elem: []*gpb.PathElem{{
				Name: "components",
			}, {
				Name: "component",
				Key: map[string]string{
					"name": "OCH-1-2",
				},
			}, {
				Name: "optical-channel",
			}, {
				Name: "state",
			}, {
				Name: "line-port",
			}},
		},
		inValue: &gpb.TypedValue{
			Value: &gpb.TypedValue_StringVal{"XCVR-1-2"},
		},
		inOpts:           []ytypes.SetNodeOpt{&ytypes.InitMissingElements{}},
		wantErrSubstring: "no match found",
	}, {
		desc:     "bad path",
		inSchema: mustSchema(uexampleoc.Schema),
		inPath: &gpb.Path{
			Elem: []*gpb.PathElem{{
				Name: "doesnt-exist",
			}},
		},
		inValue: &gpb.TypedValue{
			Value: &gpb.TypedValue_IntVal{42},
		},
		wantErrSubstring: "no match found",
	}, {
		desc:     "wrong type",
		inSchema: mustSchema(uexampleoc.Schema),
		inPath: &gpb.Path{
			Elem: []*gpb.PathElem{{
				Name: "system",
			}, {
				Name: "config",
			}, {
				Name: "hostname",
			}},
		},
		inValue: &gpb.TypedValue{
			Value: &gpb.TypedValue_UintVal{42},
		},
		inOpts:           []ytypes.SetNodeOpt{&ytypes.InitMissingElements{}},
		wantErrSubstring: "failed to unmarshal",
	}, {
		desc:     "set of a leaf in prefer opstate, without prefer shadow path",
		inSchema: mustSchema(opstateoc.Schema),
		inPath: &gpb.Path{
			Elem: []*gpb.PathElem{{
				Name: "system",
			}, {
				Name: "config",
			}, {
				Name: "hostname",
			}},
		},
		inValue: &gpb.TypedValue{
			Value: &gpb.TypedValue_StringVal{
				StringVal: "hello world",
			},
		},
		inOpts: []ytypes.SetNodeOpt{&ytypes.InitMissingElements{}},
		wantNode: &ytypes.TreeNode{
			Data: &opstateoc.Device{
				System: &opstateoc.System{
					// Not set because we set the compressed-out version.
				},
			},
		},
	}, {
		desc:     "set of a leaf in prefer opstate, with prefer shadow path",
		inSchema: mustSchema(opstateoc.Schema),
		inPath: &gpb.Path{
			Elem: []*gpb.PathElem{{
				Name: "system",
			}, {
				Name: "config",
			}, {
				Name: "hostname",
			}},
		},
		inValue: &gpb.TypedValue{
			Value: &gpb.TypedValue_StringVal{
				StringVal: "hello world",
			},
		},
		inOpts: []ytypes.SetNodeOpt{&ytypes.InitMissingElements{}, &ytypes.PreferShadowPath{}},
		wantNode: &ytypes.TreeNode{
			Data: &opstateoc.Device{
				System: &opstateoc.System{
					Hostname: ygot.String("hello world"),
				},
			},
		},
	}, {
		desc:     "set of a leaf-list in prefer opstate",
		inSchema: mustSchema(opstateoc.Schema),
		inPath: &gpb.Path{
			Elem: []*gpb.PathElem{{
				Name: "system",
			}, {
				Name: "dns",
			}, {
				Name: "config",
			}, {
				Name: "search",
			}},
		},
		inValue: &gpb.TypedValue{
			Value: &gpb.TypedValue_LeaflistVal{
				LeaflistVal: &gpb.ScalarArray{
					Element: []*gpb.TypedValue{{
						Value: &gpb.TypedValue_StringVal{
							StringVal: "hello",
						},
					}, {
						Value: &gpb.TypedValue_StringVal{
							StringVal: "world",
						},
					}},
				},
			},
		},
		inOpts: []ytypes.SetNodeOpt{&ytypes.InitMissingElements{}},
		wantNode: &ytypes.TreeNode{
			Data: &opstateoc.Device{
				System: &opstateoc.System{
					Dns: &opstateoc.System_Dns{
						// Not set because we are still preferring the 'state' version over the 'config' version.
					},
				},
			},
		},
	}, {
		desc:     "set of a leaf-list in prefer opstate, with prefer shadow path",
		inSchema: mustSchema(opstateoc.Schema),
		inPath: &gpb.Path{
			Elem: []*gpb.PathElem{{
				Name: "system",
			}, {
				Name: "dns",
			}, {
				Name: "config",
			}, {
				Name: "search",
			}},
		},
		inValue: &gpb.TypedValue{
			Value: &gpb.TypedValue_LeaflistVal{
				LeaflistVal: &gpb.ScalarArray{
					Element: []*gpb.TypedValue{{
						Value: &gpb.TypedValue_StringVal{
							StringVal: "hello",
						},
					}, {
						Value: &gpb.TypedValue_StringVal{
							StringVal: "world",
						},
					}},
				},
			},
		},
		inOpts: []ytypes.SetNodeOpt{&ytypes.InitMissingElements{}, &ytypes.PreferShadowPath{}},
		wantNode: &ytypes.TreeNode{
			Data: &opstateoc.Device{
				System: &opstateoc.System{
					Dns: &opstateoc.System_Dns{
						// Set because we asked to prefer the 'config' version over the state version.
						Search: []string{"hello", "world"},
					},
				},
			},
		},
	}, {
		// This test case is not expecting an error since we expect
		// ygot to be able to traverse using the key specified in the
		// map as a fallback when the key values in the list element is
		// not available.
		desc:     "invalidSchemaDueToMissingKeyField",
		inSchema: invalidSchemaDueToMissingKeyField,
		inPath: &gpb.Path{
			Elem: []*gpb.PathElem{{
				Name: "acl",
			}, {
				Name: "acl-sets",
			}, {
				Name: "acl-set",
				Key: map[string]string{
					"name": "foo",
					"type": "ACL_IPV4",
				},
			}, {
				Name: "config",
			}, {
				Name: "description",
			}},
		},
		inValue: &gpb.TypedValue{
			Value: &gpb.TypedValue_StringVal{StringVal: "desc"},
		},
		wantNode: &ytypes.TreeNode{
			Path: &gpb.Path{
				Elem: []*gpb.PathElem{{
					Name: "acl",
				}, {
					Name: "acl-sets",
				}, {
					Name: "acl-set",
					Key: map[string]string{
						"name": "foo",
						"type": "ACL_IPV4",
					},
				}, {
					Name: "config",
				}, {
					Name: "description",
				}},
			},
			Data: ygot.String("desc"),
		},
	}}

	for _, tt := range tests {
		t.Run(tt.desc, func(t *testing.T) {
			err := ytypes.SetNode(tt.inSchema.RootSchema(), tt.inSchema.Root, tt.inPath, tt.inValue, tt.inOpts...)
			if diff := errdiff.Substring(err, tt.wantErrSubstring); diff != "" {
				t.Fatalf("did not get expected error, %s", diff)
			}

			if tt.wantNode == nil {
				return
			}

			got, err := ytypes.GetNode(tt.inSchema.RootSchema(), tt.inSchema.Root, tt.wantNode.Path, tt.inGetOpts...)
			if err != nil {
				t.Fatalf("cannot perform get, %v", err)
			}
			if len(got) != 1 {
				t.Fatalf("unexpected number of nodes, want: 1, got: %d", len(got))
			}

			opts := []cmp.Option{
				cmpopts.IgnoreFields(ytypes.TreeNode{}, "Schema"),
				cmp.Comparer(proto.Equal),
			}

			if !cmp.Equal(got[0], tt.wantNode, opts...) {
				diff := cmp.Diff(tt.wantNode, got[0], opts...)
				t.Fatalf("did not get expected node, got: %v, want: %v, diff (-want, +got):\n%s", got[0], tt.wantNode, diff)
			}
		})
	}
}
