import * as React from 'react'
import * as Kb from '../../common-adapters'
import * as Styles from '../../styles'
import {AccountID} from '../../constants/types/wallets'
import WalletRow from './wallet-row/container'

type AddProps = {
  onAddNew: () => void
  onLinkExisting: () => void
}

const rowHeight = 48

const _AddWallet = (props: AddProps & Kb.OverlayParentProps) => (
  <Kb.ClickableBox onClick={props.toggleShowingMenu} ref={props.setAttachmentRef}>
    <Kb.Box2
      style={styles.addContainerBox}
      direction="horizontal"
      fullWidth={true}
      className="hover_background_color_blueGreyDark"
    >
      <Kb.Icon type="icon-wallet-placeholder-add-32" style={styles.icon} />
      <Kb.Text type="BodySemibold">Add an account</Kb.Text>
    </Kb.Box2>
    <Kb.FloatingMenu
      attachTo={props.getAttachmentRef}
      closeOnSelect={true}
      items={[
        {icon: 'iconfont-new', onClick: () => props.onAddNew(), title: 'Create a new account'},
        {
          icon: 'iconfont-identity-stellar',
          onClick: () => props.onLinkExisting(),
          title: 'Link an existing Stellar account',
        },
      ]}
      onHidden={props.toggleShowingMenu}
      visible={props.showingMenu}
      position="bottom center"
    />
  </Kb.ClickableBox>
)

const AddWallet = Kb.OverlayParentHOC(_AddWallet)

const WhatIsStellar = (props: {onWhatIsStellar: () => void}) => (
  <Kb.ClickableBox onClick={props.onWhatIsStellar} style={styles.whatIsStellar}>
    <Kb.Box2 centerChildren={true} direction="horizontal">
      <Kb.Icon sizeType="Small" type="iconfont-info" />
      <Kb.Text style={styles.infoText} type="BodySmallSemibold">
        What is Stellar?
      </Kb.Text>
    </Kb.Box2>
  </Kb.ClickableBox>
)

export type Props = {
  accountIDs: Array<AccountID>
  style?: Styles.StylesCrossPlatform
  loading: boolean
  onAddNew: () => void
  onLinkExisting: () => void
  onWhatIsStellar: () => void
  title: string
}

type Row =
  | {
      type: 'wallet'
      accountID: AccountID
      key?: string
    }
  | {
      type: 'add wallet'
      key?: string
    }

class WalletList extends React.Component<Props> {
  _renderRow = (_: number, row: Row) => {
    switch (row.type) {
      case 'wallet':
        return <WalletRow key={row.accountID} accountID={row.accountID} />
      case 'add wallet':
        return (
          <AddWallet
            key={row.type}
            onAddNew={this.props.onAddNew}
            onLinkExisting={this.props.onLinkExisting}
          />
        )
      default:
        throw new Error(`Impossible case encountered: ${row}`)
    }
  }

  render() {
    if (this.props.loading && !this.props.accountIDs.length) {
      return (
        <Kb.Box2 direction="vertical" fullWidth={true} fullHeight={true} centerChildren={true}>
          <Kb.ProgressIndicator style={styles.progressIndicator} />
        </Kb.Box2>
      )
    }

    const rows: Row[] = this.props.accountIDs.map(
      accountID => ({accountID, key: accountID, type: 'wallet'} as const)
    )

    return (
      <>
        {this.props.loading && <Kb.ProgressIndicator style={styles.progressHeader} />}
        <Kb.BoxGrow>
          <Kb.List items={rows} renderItem={this._renderRow} keyProperty="key" style={this.props.style} />
        </Kb.BoxGrow>
        <WhatIsStellar onWhatIsStellar={this.props.onWhatIsStellar} />
      </>
    )
  }
}

const styles = Styles.styleSheetCreate(
  () =>
    ({
      addContainerBox: {alignItems: 'center', height: rowHeight},
      icon: {
        height: Styles.globalMargins.mediumLarge,
        marginLeft: Styles.globalMargins.tiny,
        marginRight: Styles.globalMargins.tiny,
        width: Styles.globalMargins.mediumLarge,
      },
      infoText: {
        paddingLeft: Styles.globalMargins.tiny,
        position: 'relative',
        top: -1,
      },
      progressHeader: {
        height: 18,
        left: 40,
        position: 'absolute',
        top: 9,
        width: 18,
        zIndex: 2,
      },
      progressIndicator: {height: 30, width: 30},
      whatIsStellar: {
        height: Styles.globalMargins.large,
        justifyContent: 'center',
        width: '100%',
      },
    } as const)
)

export {WalletList}
