/* SPDX-License-Identifier: GPL-2.0-only */

#include <intelblocks/xhci.h>
#include <soc/gpe.h>

/*
 * USB Port Wake Enable (UPWE) on usb attach/detach
 * Arg0 - Port Number
 * Arg1 - Port 1 Status and control offset
 * Arg2 - xHCI Memory-mapped address
 */
Method (UPWE, 3, Serialized)
{
	Local0 = Arg1 + ((Arg0 - 1) * 0x10)

	/* Map ((XMEM << 16) + Local0 in PSCR */
	OperationRegion (PSCR, SystemMemory, (Arg2 << 16) + Local0, 0x10)
	Field (PSCR, DWordAcc, NoLock, Preserve)
	{
		PSCT, 32,
	}
	Local0 = PSCT
	Local0 &= PORTSCN_BITS_OFF_MASK
	Local0 |= PORTSCN_WAKE_ON_BOTH_CONNECT_DISCONNECT_ENABLE
	PSCT = Local0
}

/*
 * USB Wake Enable Setup (UWES)
 * Arg0 - Port enable bitmap
 * Arg1 - Port 1 Status and control offset
 * Arg2 - xHCI Memory-mapped address
 */
Method (UWES, 3, Serialized)
{
	Local0 = Arg0

	While (1) {
		FindSetRightBit (Local0, Local1)
		If (Local1 == 0) {
			Break
		}
		UPWE (Local1, Arg1, Arg2)
		/*
		 * Clear the lowest set bit in Local0 since it was
		 * processed.
		 */
		Local0 &= (Local0 - 1)
	}
}

/* XHCI Controller 0:14.0 */

Device (XHCI)
{
	Name (_ADR, 0x00140000)

	Name (_PRW, Package () { GPE0_PME_B0, 3 })

	OperationRegion (XPRT, PCI_Config, 0x00, 0x100)
	Field (XPRT, AnyAcc, NoLock, Preserve)
	{
		Offset (0x10),
		, 16,
		XMEM, 16,	/* MEM_BASE */
	}

	Method (_DSW, 3)
	{
		UWES ((\U2WE & 0xFFF), PORTSCN_OFFSET, XMEM)
		UWES ((\U3WE & 0x3F ), PORTSCXUSB3_OFFSET, XMEM)
	}

	Name (_S3D, 3)	/* D3 supported in S3 */
	Name (_S0W, 3)	/* D3 can wake device in S0 */
	Name (_S3W, 3)	/* D3 can wake system from S3 */

	Method (_PS0, 0, Serialized)
	{

	}

	Method (_PS3, 0, Serialized)
	{

	}
#if CONFIG(SOC_INTEL_ALDERLAKE_PCH_S)
	/* Root Hub for Alder Lake-P PCH */
	Device (RHUB)
	{
		Name (_ADR, 0)

		/* USB2 */
		Device (HS01) { Name (_ADR, 1) }
		Device (HS02) { Name (_ADR, 2) }
		Device (HS03) { Name (_ADR, 3) }
		Device (HS04) { Name (_ADR, 4) }
		Device (HS05) { Name (_ADR, 5) }
		Device (HS06) { Name (_ADR, 6) }
		Device (HS07) { Name (_ADR, 7) }
		Device (HS08) { Name (_ADR, 8) }
		Device (HS09) { Name (_ADR, 9) }
		Device (HS10) { Name (_ADR, 10) }
		Device (HS11) { Name (_ADR, 11) }
		Device (HS12) { Name (_ADR, 12) }
		Device (HS13) { Name (_ADR, 13) }
		Device (HS14) { Name (_ADR, 14) }
		/* USB3 */
		Device (SS01) { Name (_ADR, 15) }
		Device (SS02) { Name (_ADR, 16) }
		Device (SS03) { Name (_ADR, 17) }
		Device (SS04) { Name (_ADR, 18) }
		Device (SS05) { Name (_ADR, 19) }
		Device (SS06) { Name (_ADR, 20) }
		Device (SS07) { Name (_ADR, 21) }
		Device (SS08) { Name (_ADR, 22) }
		Device (SS09) { Name (_ADR, 23) }
		Device (SS10) { Name (_ADR, 24) }
	}
#else
	/* Root Hub for Alder Lake-P PCH */
	Device (RHUB)
	{
		Name (_ADR, 0)

		/* USB2 */
		Device (HS01) { Name (_ADR, 1) }
		Device (HS02) { Name (_ADR, 2) }
		Device (HS03) { Name (_ADR, 3) }
		Device (HS04) { Name (_ADR, 4) }
		Device (HS05) { Name (_ADR, 5) }
		Device (HS06) { Name (_ADR, 6) }
		Device (HS07) { Name (_ADR, 7) }
		Device (HS08) { Name (_ADR, 8) }
		Device (HS09) { Name (_ADR, 9) }
		Device (HS10) { Name (_ADR, 10) }
		/* USB3 */
		Device (SS01) { Name (_ADR, 13) }
		Device (SS02) { Name (_ADR, 14) }
		Device (SS03) { Name (_ADR, 15) }
		Device (SS04) { Name (_ADR, 16) }
	}
#endif
}
