// SPDX-License-Identifier: Unlicense OR MIT

package main

import (
	"bytes"
	"flag"
	"fmt"
	"go/format"
	"math"
	"os"
)

func main() {
	out := flag.String("out", "", "output file")
	flag.Parse()

	var b bytes.Buffer
	printf := func(content string, args ...interface{}) {
		fmt.Fprintf(&b, content, args...)
	}

	printf("// SPDX-License-Identifier: Unlicense OR MIT\n\n")
	printf("// Code generated by f32colorgen. DO NOT EDIT.\n")
	printf("\n")
	printf("package f32color\n\n")

	printf("// table corresponds to sRGBToLinear(float32(index)/0xff)\n")
	printf("var srgb8ToLinear = [...]float32{")
	for b := 0; b <= 0xFF; b++ {
		if b%0x10 == 0 {
			printf("\n\t")
		}
		v := sRGBToLinear(float32(b) / 0xff)
		printf("%#v,", v)
	}
	printf("\n}\n")

	data, err := format.Source(b.Bytes())
	if err != nil {
		fmt.Fprint(os.Stderr, b.String())
		panic(err)
	}

	err = os.WriteFile(*out, data, 0755)
	if err != nil {
		panic(err)
	}
}

// sRGBToLinear transforms color value from sRGB to linear.
func sRGBToLinear(c float32) float32 {
	// Formula from EXT_sRGB.
	if c <= 0.04045 {
		return c / 12.92
	} else {
		return float32(math.Pow(float64((c+0.055)/1.055), 2.4))
	}
}
