package french_vocab_test

import (
	"testing"

	"github.com/blevesearch/snowball/french"
)

// Test a large set of words for which we know
// the correct stemmed form.
//
func Test_LargeFrenchVocabulary(t *testing.T) {
	testCases := []struct {
		in  string
		out string
	}{
		{"a", "a"},
		{"à", "à"},
		{"abailard", "abailard"},
		{"abaissait", "abaiss"},
		{"abaissant", "abaiss"},
		{"abaisse", "abaiss"},
		{"abaissé", "abaiss"},
		{"abaissement", "abaissement"},
		{"abaissent", "abaissent"},
		{"abaisser", "abaiss"},
		{"abaisserai", "abaiss"},
		{"abandon", "abandon"},
		{"abandonna", "abandon"},
		{"abandonnait", "abandon"},
		{"abandonnant", "abandon"},
		{"abandonne", "abandon"},
		{"abandonné", "abandon"},
		{"abandonnée", "abandon"},
		{"abandonner", "abandon"},
		{"abandonnera", "abandon"},
		{"abandonnerait", "abandon"},
		{"abandonnés", "abandon"},
		{"abandonnez", "abandon"},
		{"abasourdi", "abasourd"},
		{"abat", "abat"},
		{"abattant", "abatt"},
		{"abattement", "abatt"},
		{"abattit", "abatt"},
		{"abattre", "abattr"},
		{"abbaye", "abbay"},
		{"abbé", "abbé"},
		{"abbés", "abbé"},
		{"abbesse", "abbess"},
		{"abeille", "abeil"},
		{"abhorrait", "abhorr"},
		{"abhorre", "abhorr"},
		{"abhorré", "abhorr"},
		{"abîmait", "abîm"},
		{"abîme", "abîm"},
		{"abîmé", "abîm"},
		{"abîmée", "abîm"},
		{"abject", "abject"},
		{"abjurant", "abjur"},
		{"ablutions", "ablut"},
		{"abnégation", "abneg"},
		{"aboiements", "aboi"},
		{"aboiera", "aboi"},
		{"abolir", "abol"},
		{"abominable", "abomin"},
		{"abominablement", "abomin"},
		{"abominables", "abomin"},
		{"abomination", "abomin"},
		{"abondaient", "abond"},
		{"abondamment", "abond"},
		{"abondance", "abond"},
		{"abondant", "abond"},
		{"abondante", "abond"},
		{"abonde", "abond"},
		{"abonné", "abon"},
		{"abonnement", "abon"},
		{"abord", "abord"},
		{"aborda", "abord"},
		{"abordaient", "abord"},
		{"abordait", "abord"},
		{"abordant", "abord"},
		{"abordée", "abord"},
		{"aborder", "abord"},
		{"abordèrent", "abord"},
		{"abordés", "abord"},
		{"abords", "abord"},
		{"aboutit", "about"},
		{"aboyaient", "aboi"},
		{"aboyèrent", "aboi"},
		{"abraham", "abraham"},
		{"abrège", "abreg"},
		{"abrégé", "abreg"},
		{"abrégea", "abreg"},
		{"abrégeaient", "abreg"},
		{"abrégeait", "abreg"},
		{"abrégeant", "abreg"},
		{"abréger", "abreg"},
		{"abrégera", "abreg"},
		{"abrégerait", "abreg"},
		{"abrégés", "abreg"},
		{"abreuvoir", "abreuvoir"},
		{"abréviations", "abrévi"},
		{"abri", "abri"},
		{"abritât", "abrit"},
		{"abrité", "abrit"},
		{"abritée", "abrit"},
		{"abritent", "abritent"},
		{"abrités", "abrit"},
		{"abrutis", "abrut"},
		{"abrutissant", "abrut"},
		{"abrutissement", "abrut"},
		{"abruzze", "abruzz"},
		{"absence", "absenc"},
		{"absences", "absenc"},
		{"absent", "absent"},
		{"absentait", "absent"},
		{"absente", "absent"},
		{"absents", "absent"},
		{"absolu", "absolu"},
		{"absolue", "absolu"},
		{"absolument", "absolu"},
		{"absolus", "absolus"},
		{"absolutisme", "absolut"},
		{"absorba", "absorb"},
		{"absorbaient", "absorb"},
		{"absorbait", "absorb"},
		{"absorbé", "absorb"},
		{"absorbée", "absorb"},
		{"absorbés", "absorb"},
		{"absoudre", "absoudr"},
		{"abstenait", "absten"},
		{"abstenez", "absten"},
		{"abstiendrai", "abstiendr"},
		{"abstraction", "abstract"},
		{"abstrait", "abstrait"},
		{"absurde", "absurd"},
		{"absurdes", "absurd"},
		{"absurdité", "absurd"},
		{"absurdités", "absurd"},
		{"abus", "abus"},
		{"abusa", "abus"},
		{"abusait", "abus"},
		{"abusant", "abus"},
		{"abuse", "abus"},
		{"abusé", "abus"},
		{"abuser", "abus"},
		{"abusez", "abus"},
		{"abusif", "abus"},
		{"acabit", "acab"},
		{"acacia", "acaci"},
		{"acacias", "acaci"},
		{"académicien", "académicien"},
		{"académie", "academ"},
		{"académies", "academ"},
		{"académique", "academ"},
		{"acajou", "acajou"},
		{"accabla", "accabl"},
		{"accablaient", "accabl"},
		{"accablait", "accabl"},
		{"accablant", "accabl"},
		{"accablante", "accabl"},
		{"accable", "accabl"},
		{"accablé", "accabl"},
		{"accablée", "accabl"},
		{"accablement", "accabl"},
		{"accabler", "accabl"},
		{"accableraient", "accabl"},
		{"accablèrent", "accabl"},
		{"accalmie", "accalm"},
		{"accaparé", "accapar"},
		{"accède", "acced"},
		{"accéder", "acced"},
		{"accéléra", "accéler"},
		{"accélération", "accéler"},
		{"accent", "accent"},
		{"accents", "accent"},
		{"accentua", "accentu"},
		{"accepta", "accept"},
		{"acceptables", "accept"},
		{"acceptais", "accept"},
		{"acceptait", "accept"},
		{"accepte", "accept"},
		{"accepté", "accept"},
		{"acceptée", "accept"},
		{"acceptent", "acceptent"},
		{"accepter", "accept"},
		{"acceptera", "accept"},
		{"accepterai", "accept"},
		{"accepterait", "accept"},
		{"accepteriez", "accept"},
		{"acceptes", "accept"},
		{"acceptez", "accept"},
		{"acception", "accept"},
		{"acceptons", "accepton"},
		{"accès", "acces"},
		{"accident", "accident"},
		{"accidenté", "accident"},
		{"accidents", "accident"},
		{"acclamation", "acclam"},
		{"acclamations", "acclam"},
		{"accommodé", "accommod"},
		{"accompagnaient", "accompagn"},
		{"accompagnait", "accompagn"},
		{"accompagnant", "accompagn"},
		{"accompagne", "accompagn"},
		{"accompagné", "accompagn"},
		{"accompagnée", "accompagn"},
		{"accompagnement", "accompagn"},
		{"accompagnent", "accompagnent"},
		{"accompagner", "accompagn"},
		{"accompagnèrent", "accompagn"},
		{"accompagniez", "accompagn"},
		{"accompli", "accompl"},
		{"accomplie", "accompl"},
		{"accomplies", "accompl"},
		{"accomplir", "accompl"},
		{"accomplirait", "accompl"},
		{"accomplis", "accompl"},
		{"accomplissaient", "accompl"},
		{"accomplissais", "accompl"},
		{"accomplissait", "accompl"},
		{"accomplissant", "accompl"},
		{"accomplissement", "accompl"},
		{"accomplissent", "accompl"},
		{"accomplit", "accompl"},
		{"accord", "accord"},
		{"accorda", "accord"},
		{"accordait", "accord"},
		{"accordant", "accord"},
		{"accorde", "accord"},
		{"accordé", "accord"},
		{"accordée", "accord"},
		{"accordées", "accord"},
		{"accorder", "accord"},
		{"accorderai", "accord"},
		{"accorderait", "accord"},
		{"accordèrent", "accord"},
		{"accorderez", "accord"},
		{"accordez", "accord"},
		{"accosta", "accost"},
		{"accostait", "accost"},
		{"accosté", "accost"},
		{"accoster", "accost"},
		{"accoté", "accot"},
		{"accouchement", "accouch"},
		{"accouraient", "accour"},
		{"accourait", "accour"},
		{"accourant", "accour"},
		{"accourez", "accour"},
		{"accourir", "accour"},
		{"accourue", "accouru"},
		{"accoururent", "accoururent"},
		{"accourus", "accourus"},
		{"accourut", "accourut"},
		{"accoutrement", "accoutr"},
		{"accoutumât", "accoutum"},
		{"accoutume", "accoutum"},
		{"accoutumé", "accoutum"},
		{"accoutumée", "accoutum"},
		{"accoutumées", "accoutum"},
		{"accoutumer", "accoutum"},
		{"accoutumes", "accoutum"},
		{"accoutumés", "accoutum"},
		{"accrédité", "accred"},
		{"accréditer", "accrédit"},
		{"accrochaient", "accroch"},
		{"accrochant", "accroch"},
		{"accroché", "accroch"},
		{"accrocher", "accroch"},
		{"accroissaient", "accroiss"},
		{"accroissait", "accroiss"},
		{"accroissant", "accroiss"},
		{"accroître", "accroîtr"},
		{"accru", "accru"},
		{"accrue", "accru"},
		{"accrut", "accrut"},
		{"accueil", "accueil"},
		{"accueillait", "accueil"},
		{"accueillent", "accueillent"},
		{"accueilli", "accueil"},
		{"accueillie", "accueil"},
		{"accueillies", "accueil"},
		{"accueillir", "accueil"},
		{"accueillis", "accueil"},
		{"accumulation", "accumul"},
		{"accumulé", "accumul"},
		{"accumulées", "accumul"},
		{"accumulent", "accumulent"},
		{"accumuler", "accumul"},
		{"accumulés", "accumul"},
		{"accusa", "accus"},
		{"accusait", "accus"},
		{"accusant", "accus"},
		{"accusateur", "accus"},
		{"accusation", "accus"},
		{"accusations", "accus"},
		{"accuse", "accus"},
		{"accusé", "accus"},
		{"accuser", "accus"},
		{"accuseront", "accus"},
		{"accusés", "accus"},
		{"accusez", "accus"},
		{"achalandage", "achalandag"},
		{"acharnant", "acharn"},
		{"acharnés", "acharn"},
		{"achat", "achat"},
		{"achemina", "achemin"},
		{"acheminait", "achemin"},
		{"acheminé", "achemin"},
		{"acheta", "achet"},
		{"achetait", "achet"},
		{"achetant", "achet"},
		{"achete", "achet"},
		{"achète", "achet"},
		{"acheté", "achet"},
		{"achetées", "achet"},
		{"acheter", "achet"},
		{"achèterons", "achet"},
		{"achètes", "achet"},
		{"achetés", "achet"},
		{"achetez", "achet"},
		{"acheva", "achev"},
		{"achevaient", "achev"},
		{"achevait", "achev"},
		{"achevant", "achev"},
		{"achève", "achev"},
		{"achevé", "achev"},
		{"achevée", "achev"},
		{"achèvement", "achev"},
		{"achever", "achev"},
		{"achèvera", "achev"},
		{"achevez", "achev"},
		{"achille", "achill"},
		{"acier", "aci"},
		{"acolyte", "acolyt"},
		{"acoustiques", "acoust"},
		{"acquérait", "acquer"},
		{"acquerrait", "acquerr"},
		{"acquiert", "acquiert"},
		{"acquis", "acquis"},
		{"acquise", "acquis"},
		{"acquises", "acquis"},
		{"acquisition", "acquisit"},
		{"acquisitions", "acquisit"},
		{"acquit", "acquit"},
		{"acquitta", "acquitt"},
		{"acquittait", "acquitt"},
		{"acquitte", "acquitt"},
		{"acquitté", "acquitt"},
		{"acquittement", "acquitt"},
		{"acquitter", "acquitt"},
		{"acquittés", "acquitt"},
		{"acrobates", "acrobat"},
		{"acrobatique", "acrobat"},
		{"acte", "acte"},
		{"actes", "acte"},
		{"acteur", "acteur"},
		{"acteurs", "acteur"},
		{"actif", "actif"},
		{"action", "action"},
		{"actionnaires", "actionnair"},
		{"actions", "action"},
		{"active", "activ"},
		{"activement", "activ"},
		{"actives", "activ"},
		{"activité", "activ"},
		{"actrice", "actric"},
		{"actrices", "actric"},
		{"actuel", "actuel"},
		{"actuelle", "actuel"},
		{"actuellement", "actuel"},
		{"actuelles", "actuel"},
		{"actuels", "actuel"},
		{"ad", "ad"},
		{"adaptant", "adapt"},
		{"additions", "addit"},
		{"address", "address"},
		{"adeline", "adelin"},
		{"aden", "aden"},
		{"adieu", "adieu"},
		{"adieux", "adieux"},
		{"adjoint", "adjoint"},
		{"adjonction", "adjonct"},
		{"adjudant", "adjud"},
		{"adjudication", "adjud"},
		{"adjudications", "adjud"},
		{"adjugeait", "adjug"},
		{"adjuger", "adjug"},
		{"admet", "admet"},
		{"admettait", "admet"},
		{"admettant", "admet"},
		{"admettent", "admettent"},
		{"admettons", "admetton"},
		{"admettre", "admettr"},
		{"administra", "administr"},
		{"administrateur", "administr"},
		{"administrateurs", "administr"},
		{"administration", "administr"},
		{"administre", "administr"},
		{"administré", "administr"},
		{"administrer", "administr"},
		{"administrés", "administr"},
		{"administrez", "administr"},
		{"admira", "admir"},
		{"admirable", "admir"},
		{"admirablement", "admir"},
		{"admirables", "admir"},
		{"admirais", "admir"},
		{"admirait", "admir"},
		{"admirant", "admir"},
		{"admirateurs", "admir"},
		{"admiration", "admir"},
		{"admire", "admir"},
		{"admiré", "admir"},
		{"admirée", "admir"},
		{"admirées", "admir"},
		{"admirent", "admirent"},
		{"admirer", "admir"},
		{"admirera", "admir"},
		{"admirèrent", "admir"},
		{"admirés", "admir"},
		{"admiriez", "admir"},
		{"admis", "admis"},
		{"admise", "admis"},
		{"admises", "admis"},
		{"admission", "admiss"},
		{"admît", "admît"},
		{"admonestant", "admonest"},
		{"adolescent", "adolescent"},
		{"adolphe", "adolph"},
		{"adonné", "adon"},
		{"adonnée", "adon"},
		{"adonnent", "adonnent"},
		{"adopta", "adopt"},
		{"adopte", "adopt"},
		{"adopté", "adopt"},
		{"adoptée", "adopt"},
		{"adopter", "adopt"},
		{"adoptif", "adopt"},
		{"adorable", "ador"},
		{"adoraient", "ador"},
		{"adorais", "ador"},
		{"adorait", "ador"},
		{"adorant", "ador"},
		{"adorateur", "ador"},
		{"adorateurs", "ador"},
		{"adoration", "ador"},
		{"adorations", "ador"},
		{"adoratrices", "ador"},
		{"adore", "ador"},
		{"adoré", "ador"},
		{"adorée", "ador"},
		{"adorent", "adorent"},
		{"adorer", "ador"},
		{"adorera", "ador"},
		{"adouci", "adouc"},
		{"adoucir", "adouc"},
		{"adoucissait", "adouc"},
		{"adoucissent", "adouc"},
		{"adressa", "adress"},
		{"adressaient", "adress"},
		{"adressait", "adress"},
		{"adressant", "adress"},
		{"adressât", "adress"},
		{"adresse", "adress"},
		{"adressé", "adress"},
		{"adressée", "adress"},
		{"adressées", "adress"},
		{"adressent", "adressent"},
		{"adresser", "adress"},
		{"adresserai", "adress"},
		{"adressèrent", "adress"},
		{"adresses", "adress"},
		{"adressés", "adress"},
		{"adressez", "adress"},
		{"adrien", "adrien"},
		{"adroit", "adroit"},
		{"adroite", "adroit"},
		{"adroitement", "adroit"},
		{"adroites", "adroit"},
		{"adroits", "adroit"},
		{"adsum", "adsum"},
		{"adultère", "adulter"},
		{"adverbe", "adverb"},
		{"adversaire", "adversair"},
		{"adversaires", "adversair"},
		{"adverse", "advers"},
		{"adversité", "advers"},
		{"advint", "advint"},
		{"aérienne", "aérien"},
		{"affabilité", "affabl"},
		{"affadissantes", "affad"},
		{"affaibli", "affaibl"},
		{"affaiblie", "affaibl"},
		{"affaiblissait", "affaibl"},
		{"affaiblissement", "affaibl"},
		{"affaiblit", "affaibl"},
		{"affaire", "affair"},
		{"affairé", "affair"},
		{"affaires", "affair"},
		{"affairés", "affair"},
		{"affaissé", "affaiss"},
		{"affamé", "affam"},
		{"affamer", "affam"},
		{"affamés", "affam"},
		{"affecta", "affect"},
		{"affectaient", "affect"},
		{"affectait", "affect"},
		{"affectant", "affect"},
		{"affectation", "affect"},
		{"affectations", "affect"},
		{"affecte", "affect"},
		{"affecté", "affect"},
		{"affectée", "affect"},
		{"affectées", "affect"},
		{"affecter", "affect"},
		{"affectés", "affect"},
		{"affection", "affect"},
		{"affectionnait", "affection"},
		{"affectionné", "affection"},
		{"affections", "affect"},
		{"affectueux", "affectu"},
		{"affermissant", "afferm"},
		{"affermit", "afferm"},
		{"affichait", "affich"},
		{"affiche", "affich"},
		{"affiché", "affich"},
		{"affichée", "affich"},
		{"affichent", "affichent"},
		{"afficher", "affich"},
		{"affiches", "affich"},
		{"afficheur", "afficheur"},
		{"affidé", "affid"},
		{"affidés", "affid"},
		{"affilée", "affil"},
		{"affilié", "affili"},
		{"affilier", "affili"},
		{"affinités", "affin"},
		{"affirmait", "affirm"},
		{"affirmant", "affirm"},
		{"affirmatif", "affirm"},
		{"affirme", "affirm"},
		{"affirmé", "affirm"},
		{"affirment", "affirment"},
		{"affirmer", "affirm"},
		{"affirmerai", "affirm"},
		{"afflictive", "afflict"},
		{"afflige", "afflig"},
		{"affligé", "afflig"},
		{"affligea", "afflig"},
		{"affligée", "afflig"},
		{"affliger", "afflig"},
		{"affligez", "afflig"},
		{"afflua", "afflu"},
		{"affluaient", "afflu"},
		{"affluence", "affluenc"},
		{"affluent", "affluent"},
		{"affluents", "affluent"},
		{"affluer", "afflu"},
		{"affolait", "affol"},
		{"affolées", "affol"},
		{"affranchi", "affranch"},
		{"affrète", "affret"},
		{"affreuse", "affreux"},
		{"affreuses", "affreux"},
		{"affreux", "affreux"},
		{"affriolait", "affriol"},
		{"affront", "affront"},
		{"affrontais", "affront"},
		{"affronter", "affront"},
		{"affublé", "affubl"},
		{"afin", "afin"},
		{"afrique", "afriqu"},
		{"agde", "agde"},
		{"age", "age"},
		{"âge", "âge"},
		{"âgé", "âgé"},
		{"âgée", "âgé"},
		{"âgées", "âgé"},
		{"agen", "agen"},
		{"agencées", "agenc"},
		{"agenouilla", "agenouill"},
		{"agenouillant", "agenouill"},
		{"agenouillé", "agenouill"},
		{"agenouillée", "agenouill"},
		{"agent", "agent"},
		{"agents", "agent"},
		{"âges", "âge"},
		{"âgés", "âgé"},
		{"agglomération", "agglomer"},
		{"aggrottato", "aggrottato"},
		{"agi", "agi"},
		{"agile", "agil"},
		{"agilité", "agil"},
		{"agir", "agir"},
		{"agirait", "agir"},
		{"agis", "agis"},
		{"agissaient", "agiss"},
		{"agissait", "agiss"},
		{"agissant", "agiss"},
		{"agissante", "agiss"},
		{"agisse", "agiss"},
		{"agissent", "agissent"},
		{"agissez", "agiss"},
		{"agissons", "agisson"},
		{"agit", "agit"},
		{"agita", "agit"},
		{"agitaient", "agit"},
		{"agitait", "agit"},
		{"agitant", "agit"},
		{"agitation", "agit"},
		{"agite", "agit"},
		{"agité", "agit"},
		{"agitée", "agit"},
		{"agitées", "agit"},
		{"agitent", "agitent"},
		{"agiter", "agit"},
		{"agitèrent", "agit"},
		{"agités", "agit"},
		{"agneaux", "agneau"},
		{"agonie", "agon"},
		{"agra", "agra"},
		{"agrandis", "agrand"},
		{"agréable", "agréabl"},
		{"agréablement", "agréabl"},
		{"agréables", "agréabl"},
		{"agréer", "agré"},
		{"agréez", "agré"},
		{"agrément", "agrément"},
		{"agrémenté", "agrément"},
		{"agrémentée", "agrément"},
		{"agrémentées", "agrément"},
		{"agrémentés", "agrément"},
		{"agréments", "agrément"},
		{"agrès", "agres"},
		{"agriculteur", "agriculteur"},
		{"agriculture", "agricultur"},
		{"ah", "ah"},
		{"ahméhnagara", "ahméhnagar"},
		{"ahuri", "ahur"},
		{"ai", "ai"},
		{"aida", "aid"},
		{"aidait", "aid"},
		{"aidant", "aid"},
		{"aide", "aid"},
		{"aidé", "aid"},
		{"aidée", "aid"},
		{"aidées", "aid"},
		{"aider", "aid"},
		{"aidera", "aid"},
		{"aiderai", "aid"},
		{"aidèrent", "aid"},
		{"aiderez", "aid"},
		{"aides", "aid"},
		{"aidés", "aid"},
		{"aidez", "aid"},
		{"aidiez", "aid"},
		{"aie", "aie"},
		{"aient", "aient"},
		{"aies", "aie"},
		{"aïeul", "aïeul"},
		{"aïeux", "aïeux"},
		{"aigle", "aigl"},
		{"aigre", "aigr"},
		{"aigrelets", "aigrelet"},
		{"aigrelettes", "aigrelet"},
		{"aigrement", "aigr"},
		{"aigres", "aigr"},
		{"aigreur", "aigreur"},
		{"aigrirait", "aigr"},
		{"aiguille", "aiguill"},
		{"aiguilles", "aiguill"},
		{"aiguillon", "aiguillon"},
		{"aiguiser", "aiguis"},
		{"aigus", "aigus"},
		{"aile", "ail"},
		{"ailes", "ail"},
		{"aille", "aill"},
		{"ailles", "aill"},
		{"ailleurs", "ailleur"},
		{"aima", "aim"},
		{"aimable", "aimabl"},
		{"aimables", "aimabl"},
		{"aimaient", "aim"},
		{"aimais", "aim"},
		{"aimait", "aim"},
		{"aimant", "aim"},
		{"aimât", "aim"},
		{"aime", "aim"},
		{"aimé", "aim"},
		{"aimée", "aim"},
		{"aimées", "aim"},
		{"aiment", "aiment"},
		{"aimer", "aim"},
		{"aimera", "aim"},
		{"aimerai", "aim"},
		{"aimerais", "aim"},
		{"aimerait", "aim"},
		{"aimerez", "aim"},
		{"aimes", "aim"},
		{"aimés", "aim"},
		{"aimez", "aim"},
		{"aimiez", "aim"},
		{"aimons", "aimon"},
		{"aine", "ain"},
		{"aîné", "aîn"},
		{"aînée", "aîn"},
		{"aînés", "aîn"},
		{"ainsi", "ains"},
		{"air", "air"},
		{"aire", "air"},
		{"airs", "air"},
		{"aisance", "aisanc"},
		{"aise", "ais"},
		{"aisé", "ais"},
		{"aisées", "ais"},
		{"aisément", "ais"},
		{"aisés", "ais"},
		{"ait", "ait"},
		{"ajournement", "ajourn"},
		{"ajouta", "ajout"},
		{"ajoutaient", "ajout"},
		{"ajoutait", "ajout"},
		{"ajoutant", "ajout"},
		{"ajoute", "ajout"},
		{"ajouté", "ajout"},
		{"ajoutées", "ajout"},
		{"ajouter", "ajout"},
		{"ajoutera", "ajout"},
		{"ajouterai", "ajout"},
		{"ajouteraient", "ajout"},
		{"ajouterais", "ajout"},
		{"ajoutèrent", "ajout"},
		{"ajouterez", "ajout"},
		{"ajouterons", "ajout"},
		{"ajuster", "ajust"},
		{"al", "al"},
		{"alabama", "alabam"},
		{"alacoque", "alacoqu"},
		{"alari", "alar"},
		{"alarma", "alarm"},
		{"alarmaient", "alarm"},
		{"alarmants", "alarm"},
		{"alarme", "alarm"},
		{"alarmé", "alarm"},
		{"alarmée", "alarm"},
		{"alarmer", "alarm"},
		{"alarmes", "alarm"},
		{"alas", "alas"},
		{"albâtre", "albâtr"},
		{"albermale", "albermal"},
		{"album", "album"},
		{"alcooliques", "alcool"},
		{"ale", "ale"},
		{"aléa", "alé"},
		{"alembert", "alembert"},
		{"alençon", "alençon"},
		{"alerte", "alert"},
		{"alertes", "alert"},
		{"alexandre", "alexandr"},
		{"alfieri", "alfier"},
		{"alfred", "alfred"},
		{"alger", "alger"},
		{"alhambra", "alhambr"},
		{"aliène", "alien"},
		{"alignées", "align"},
		{"aligre", "aligr"},
		{"aliment", "aliment"},
		{"alimentation", "aliment"},
		{"alimenter", "aliment"},
		{"alimento", "alimento"},
		{"alinéa", "aliné"},
		{"all", "all"},
		{"alla", "alla"},
		{"allahabad", "allahabad"},
		{"allaient", "allaient"},
		{"allais", "allais"},
		{"allait", "allait"},
		{"allant", "allant"},
		{"allât", "allât"},
		{"allé", "allé"},
		{"alléché", "allech"},
		{"alléchés", "allech"},
		{"allée", "allé"},
		{"allées", "allé"},
		{"allégation", "alleg"},
		{"allege", "alleg"},
		{"allège", "alleg"},
		{"allègre", "allegr"},
		{"allégresse", "allégress"},
		{"alléguant", "allégu"},
		{"allemagne", "allemagn"},
		{"allemand", "allemand"},
		{"allemande", "allemand"},
		{"allemands", "allemand"},
		{"aller", "aller"},
		{"allèrent", "allèrent"},
		{"allés", "allé"},
		{"allez", "allez"},
		{"alliage", "alliag"},
		{"alliance", "allianc"},
		{"allié", "alli"},
		{"alliés", "alli"},
		{"alliez", "alli"},
		{"alligators", "alligator"},
		{"allions", "allion"},
		{"allocution", "allocu"},
		{"allonge", "allong"},
		{"allongeait", "allong"},
		{"allongée", "allong"},
		{"allons", "allon"},
		{"alloue", "allou"},
		{"alluma", "allum"},
		{"allumait", "allum"},
		{"allume", "allum"},
		{"allumé", "allum"},
		{"allumée", "allum"},
		{"allumées", "allum"},
		{"allumer", "allum"},
		{"allumés", "allum"},
		{"allure", "allur"},
		{"allures", "allur"},
		{"allusion", "allus"},
		{"allusions", "allus"},
		{"almanach", "almanach"},
		{"almaviva", "almaviv"},
		{"alors", "alor"},
		{"alouettes", "alouet"},
		{"alourdie", "alourd"},
		{"alpes", "alpe"},
		{"alphabet", "alphabet"},
		{"alphabets", "alphabet"},
		{"alsace", "alsac"},
		{"also", "also"},
		{"altamira", "altam"},
		{"altéra", "alter"},
		{"altérait", "alter"},
		{"altéré", "alter"},
		{"altérée", "alter"},
		{"altérer", "alter"},
		{"altérés", "alter"},
		{"alternative", "altern"},
		{"alternatives", "altern"},
		{"altesse", "altess"},
		{"altesses", "altess"},
		{"altier", "alti"},
		{"altière", "altier"},
		{"altitude", "altitud"},
		{"alvizi", "alviz"},
		{"alzar", "alzar"},
		{"ama", "ama"},
		{"amabilité", "amabl"},
		{"amadoué", "amadou"},
		{"amadouer", "amadou"},
		{"amai", "amai"},
		{"amaigris", "amaigr"},
		{"amand", "amand"},
		{"amanda", "amand"},
		{"amant", "amant"},
		{"amantes", "amant"},
		{"amants", "amant"},
		{"amarres", "amarr"},
		{"amarrés", "amarr"},
		{"amas", "amas"},
		{"amassaient", "amass"},
		{"amassé", "amass"},
		{"amassée", "amass"},
		{"amateur", "amateur"},
		{"amateurs", "amateur"},
		{"ambassade", "ambassad"},
		{"ambassadeur", "ambassadeur"},
		{"ambassadeurs", "ambassadeur"},
		{"ambiguë", "ambigu"},
		{"ambitieuse", "ambiti"},
		{"ambitieux", "ambiti"},
		{"ambition", "ambit"},
		{"ambulance", "ambul"},
		{"ambulant", "ambul"},
		{"ambulantes", "ambul"},
		{"ambulants", "ambul"},
		{"âme", "âme"},
		{"amena", "amen"},
		{"aménagé", "aménag"},
		{"aménagements", "aménag"},
		{"amenait", "amen"},
		{"amende", "amend"},
		{"amendes", "amend"},
		{"amène", "amen"},
		{"amené", "amen"},
		{"amenée", "amen"},
		{"amener", "amen"},
		{"amènera", "amen"},
		{"amènerai", "amen"},
		{"amènerait", "amen"},
		{"amenez", "amen"},
		{"ameni", "amen"},
		{"amer", "amer"},
		{"amère", "amer"},
		{"amèrement", "amer"},
		{"amères", "amer"},
		{"américain", "américain"},
		{"américaine", "américain"},
		{"américains", "américain"},
		{"american", "american"},
		{"amérique", "amer"},
		{"amers", "amer"},
		{"amertume", "amertum"},
		{"âmes", "âme"},
		{"ameublement", "ameubl"},
		{"ameutant", "ameut"},
		{"ami", "ami"},
		{"amiable", "amiabl"},
		{"amical", "amical"},
		{"amicale", "amical"},
		{"amicalement", "amical"},
		{"amicizia", "amicizi"},
		{"amie", "ami"},
		{"amiens", "amien"},
		{"amies", "ami"},
		{"amis", "amis"},
		{"amitié", "amiti"},
		{"amitiés", "amiti"},
		{"amoindri", "amoindr"},
		{"amor", "amor"},
		{"amorce", "amorc"},
		{"amorces", "amorc"},
		{"amortir", "amort"},
		{"amortirait", "amort"},
		{"amour", "amour"},
		{"amourachée", "amourach"},
		{"amoureuse", "amour"},
		{"amoureusement", "amour"},
		{"amoureux", "amour"},
		{"amours", "amour"},
		{"amphigouriques", "amphigour"},
		{"amphion", "amphion"},
		{"amphithéâtre", "amphithéâtr"},
		{"ample", "ample"},
		{"amplification", "amplif"},
		{"amplifications", "amplif"},
		{"amputés", "amput"},
		{"amusa", "amus"},
		{"amusaient", "amus"},
		{"amusait", "amus"},
		{"amusant", "amus"},
		{"amusante", "amus"},
		{"amusantes", "amus"},
		{"amusants", "amus"},
		{"amuse", "amus"},
		{"amusé", "amus"},
		{"amusement", "amus"},
		{"amuser", "amus"},
		{"amusera", "amus"},
		{"an", "an"},
		{"anachorète", "anachoret"},
		{"anachronisme", "anachron"},
		{"analyser", "analys"},
		{"ananas", "anan"},
		{"anathème", "anathem"},
		{"anathèmes", "anathem"},
		{"ancêtres", "ancêtr"},
		{"ancien", "ancien"},
		{"ancienne", "ancien"},
		{"anciennes", "ancien"},
		{"anciens", "ancien"},
		{"ancre", "ancre"},
		{"ancrés", "ancré"},
		{"and", "and"},
		{"andaman", "andaman"},
		{"anderson", "anderson"},
		{"andrew", "andrew"},
		{"andryane", "andryan"},
		{"anéanti", "anéant"},
		{"anéantie", "anéant"},
		{"anéantir", "anéant"},
		{"anéantis", "anéant"},
		{"anéantissait", "anéant"},
		{"anéantit", "anéant"},
		{"anecdote", "anecdot"},
		{"anecdotes", "anecdot"},
		{"anecdotique", "anecdot"},
		{"ânerie", "âner"},
		{"anetta", "anet"},
		{"ange", "ange"},
		{"angela", "angel"},
		{"angelica", "angelic"},
		{"angélina", "angélin"},
		{"angélique", "angel"},
		{"anges", "ange"},
		{"anglais", "anglais"},
		{"anglaise", "anglais"},
		{"anglaises", "anglais"},
		{"angle", "angle"},
		{"angles", "angle"},
		{"angleterre", "angleterr"},
		{"anglo", "anglo"},
		{"angoisse", "angoiss"},
		{"angoisses", "angoiss"},
		{"angoulême", "angoulêm"},
		{"aniken", "aniken"},
		{"anima", "anim"},
		{"animaient", "anim"},
		{"animait", "anim"},
		{"animal", "animal"},
		{"animant", "anim"},
		{"animation", "anim"},
		{"animaux", "animal"},
		{"anime", "anim"},
		{"animé", "anim"},
		{"animée", "anim"},
		{"animées", "anim"},
		{"animer", "anim"},
		{"animèrent", "anim"},
		{"animés", "anim"},
		{"annales", "annal"},
		{"annam", "annam"},
		{"anneau", "anneau"},
		{"année", "anné"},
		{"années", "anné"},
		{"annexait", "annex"},
		{"annibal", "annibal"},
		{"anno", "anno"},
		{"annonça", "annonc"},
		{"annonçaient", "annonc"},
		{"annonçait", "annonc"},
		{"annonçant", "annonc"},
		{"annonce", "annonc"},
		{"annoncé", "annonc"},
		{"annoncée", "annonc"},
		{"annoncées", "annonc"},
		{"annoncent", "annoncent"},
		{"annoncer", "annonc"},
		{"annoncerait", "annonc"},
		{"annoncèrent", "annonc"},
		{"annoncerez", "annonc"},
		{"annonces", "annonc"},
		{"annoncés", "annonc"},
		{"annuel", "annuel"},
		{"annuelle", "annuel"},
		{"annuellement", "annuel"},
		{"annulé", "annul"},
		{"annulée", "annul"},
		{"annuleront", "annul"},
		{"anobli", "anobl"},
		{"anoblir", "anobl"},
		{"anoblis", "anobl"},
		{"anoblissement", "anobl"},
		{"anoblit", "anobl"},
		{"anodines", "anodin"},
		{"ânonna", "ânon"},
		{"anonyme", "anonym"},
		{"anonymement", "anonym"},
		{"anonymes", "anonym"},
		{"ans", "an"},
		{"anse", "anse"},
		{"antagoniste", "antagon"},
		{"antérieure", "antérieur"},
		{"antérieurs", "antérieur"},
		{"anthropophages", "anthropophag"},
		{"anti", "anti"},
		{"antibes", "antib"},
		{"antichambre", "antichambr"},
		{"antichambres", "antichambr"},
		{"antidata", "antidat"},
		{"antidatée", "antidat"},
		{"antigénéreuses", "antigéner"},
		{"antijacobine", "antijacobin"},
		{"antimonarchique", "antimonarch"},
		{"antipathie", "antipath"},
		{"antipathique", "antipath"},
		{"antipode", "antipod"},
		{"antipodes", "antipod"},
		{"antiquaires", "antiquair"},
		{"antique", "antiqu"},
		{"antiques", "antiqu"},
		{"antiquité", "antiqu"},
		{"antiquités", "antiqu"},
		{"antithèse", "antithes"},
		{"antoine", "antoin"},
		{"anxiété", "anxiet"},
		{"anxieux", "anxieux"},
		{"any", "any"},
		{"aouda", "aoud"},
		{"août", "août"},
		{"apaisa", "apais"},
		{"apaisée", "apais"},
		{"apaiser", "apais"},
		{"apaisera", "apais"},
		{"apathie", "apath"},
		{"apathique", "apath"},
		{"apennin", "apennin"},
		{"apens", "apen"},
		{"apercevait", "apercev"},
		{"apercevant", "apercev"},
		{"apercevoir", "apercevoir"},
		{"apercevons", "apercevon"},
		{"apercevrait", "apercevr"},
		{"aperçois", "aperçois"},
		{"aperçoit", "aperçoit"},
		{"aperçoive", "aperço"},
		{"aperçu", "aperçu"},
		{"aperçue", "aperçu"},
		{"aperçurent", "aperçurent"},
		{"aperçus", "aperçus"},
		{"aperçut", "aperçut"},
		{"aperçût", "aperçût"},
		{"aplaties", "aplat"},
		{"aplatit", "aplat"},
		{"aplomb", "aplomb"},
		{"apocalypse", "apocalyps"},
		{"apocalyptique", "apocalypt"},
		{"apoplectique", "apoplect"},
		{"apoplexie", "apoplex"},
		{"apoplexies", "apoplex"},
		{"apostat", "apostat"},
		{"aposté", "apost"},
		{"apostille", "apostill"},
		{"apostolique", "apostol"},
		{"apostoliques", "apostol"},
		{"apothicaire", "apothicair"},
		{"apôtre", "apôtr"},
		{"apôtres", "apôtr"},
		{"apparaissaient", "apparaiss"},
		{"apparaissait", "apparaiss"},
		{"apparait", "appar"},
		{"apparaît", "apparaît"},
		{"apparaîtrait", "apparaîtr"},
		{"apparaître", "apparaîtr"},
		{"apparat", "apparat"},
		{"appareil", "appareil"},
		{"appareillait", "appareil"},
		{"appareiller", "appareil"},
		{"appareils", "appareil"},
		{"apparemment", "apparent"},
		{"apparence", "apparent"},
		{"apparences", "apparent"},
		{"apparent", "apparent"},
		{"apparente", "apparent"},
		{"apparition", "apparit"},
		{"apparitions", "apparit"},
		{"appartement", "appart"},
		{"appartements", "appart"},
		{"appartenaient", "apparten"},
		{"appartenait", "apparten"},
		{"appartenant", "apparten"},
		{"appartenir", "apparten"},
		{"appartenu", "appartenu"},
		{"appartiendront", "appartiendront"},
		{"appartiennent", "appartiennent"},
		{"appartient", "appartient"},
		{"appartint", "appartint"},
		{"appartînt", "appartînt"},
		{"apparu", "apparu"},
		{"apparut", "apparut"},
		{"appât", "appât"},
		{"appel", "appel"},
		{"appela", "appel"},
		{"appelaient", "appel"},
		{"appelait", "appel"},
		{"appelant", "appel"},
		{"appelât", "appel"},
		{"appelé", "appel"},
		{"appelée", "appel"},
		{"appelées", "appel"},
		{"appeler", "appel"},
		{"appelés", "appel"},
		{"appelez", "appel"},
		{"appelle", "appel"},
		{"appellent", "appellent"},
		{"appellera", "appel"},
		{"appellerai", "appel"},
		{"appellerait", "appel"},
		{"appellerez", "appel"},
		{"appelleront", "appel"},
		{"appelles", "appel"},
		{"appendices", "appendic"},
		{"appert", "appert"},
		{"appesantie", "appesant"},
		{"appesantis", "appesant"},
		{"appétit", "appet"},
		{"applaudi", "applaud"},
		{"applaudie", "applaud"},
		{"applaudir", "applaud"},
		{"applaudirent", "applaud"},
		{"applaudissaient", "applaud"},
		{"applaudissements", "applaud"},
		{"applaudit", "applaud"},
		{"applicables", "applic"},
		{"application", "appliqu"},
		{"appliqua", "appliqu"},
		{"appliquait", "appliqu"},
		{"applique", "appliqu"},
		{"appliqué", "appliqu"},
		{"appliquée", "appliqu"},
		{"appliquer", "appliqu"},
		{"appliquez", "appliqu"},
		{"appoint", "appoint"},
		{"appointements", "appoint"},
		{"appointés", "appoint"},
		{"apporta", "apport"},
		{"apportaient", "apport"},
		{"apportait", "apport"},
		{"apportant", "apport"},
		{"apporte", "apport"},
		{"apporté", "apport"},
		{"apportée", "apport"},
		{"apportées", "apport"},
		{"apporter", "apport"},
		{"apporterai", "apport"},
		{"apporterait", "apport"},
		{"apportés", "apport"},
		{"apportez", "apport"},
		{"apportiez", "apport"},
		{"apposa", "appos"},
		{"apposer", "appos"},
		{"apposés", "appos"},
		{"apprécie", "apprec"},
		{"appréciée", "appréci"},
		{"apprécier", "appréci"},
		{"appréciés", "appréci"},
		{"appréhension", "appréhens"},
		{"apprenaient", "appren"},
		{"apprenait", "appren"},
		{"apprenant", "appren"},
		{"apprend", "apprend"},
		{"apprendra", "apprendr"},
		{"apprendrais", "apprendr"},
		{"apprendrait", "apprendr"},
		{"apprendre", "apprendr"},
		{"apprendrez", "apprendr"},
		{"apprendrions", "apprendr"},
		{"apprendront", "apprendront"},
		{"apprends", "apprend"},
		{"apprenez", "appren"},
		{"apprennes", "appren"},
		{"apprêtait", "apprêt"},
		{"apprêter", "apprêt"},
		{"apprêtons", "apprêton"},
		{"apprirent", "apprirent"},
		{"appris", "appris"},
		{"apprise", "appris"},
		{"apprises", "appris"},
		{"apprit", "apprit"},
		{"apprivoisés", "apprivois"},
		{"approbation", "approb"},
		{"approcha", "approch"},
		{"approchaient", "approch"},
		{"approchait", "approch"},
		{"approchant", "approch"},
		{"approche", "approch"},
		{"approché", "approch"},
		{"approchée", "approch"},
		{"approchent", "approchent"},
		{"approcher", "approch"},
		{"approchèrent", "approch"},
		{"approches", "approch"},
		{"approchés", "approch"},
		{"approfondie", "approfond"},
		{"approprié", "appropri"},
		{"approuva", "approuv"},
		{"approuvé", "approuv"},
		{"approuvée", "approuv"},
		{"approuvera", "approuv"},
		{"approuves", "approuv"},
		{"appui", "appui"},
		{"appuie", "appui"},
		{"appuya", "appui"},
		{"appuyaient", "appui"},
		{"appuyais", "appui"},
		{"appuyait", "appui"},
		{"appuyant", "appui"},
		{"appuyé", "appui"},
		{"appuyée", "appui"},
		{"appuyées", "appui"},
		{"appuyer", "appui"},
		{"âpre", "âpre"},
		{"après", "apres"},
		{"april", "april"},
		{"aptitude", "aptitud"},
		{"aquarelle", "aquarel"},
		{"aquetta", "aquet"},
		{"aquilin", "aquilin"},
		{"aquitaine", "aquitain"},
		{"arabe", "arab"},
		{"arabes", "arab"},
		{"arabique", "arab"},
		{"araceli", "aracel"},
		{"arbitraire", "arbitrair"},
		{"arbitraires", "arbitrair"},
		{"arbitre", "arbitr"},
		{"arbitres", "arbitr"},
		{"arboré", "arbor"},
		{"arborer", "arbor"},
		{"arborescentes", "arborescent"},
		{"arbre", "arbre"},
		{"arbres", "arbre"},
		{"arbrisseaux", "arbrisseau"},
		{"arc", "arc"},
		{"arceaux", "arceau"},
		{"arche", "arche"},
		{"archet", "archet"},
		{"archevêché", "archevêch"},
		{"archevêque", "archevêqu"},
		{"archevêques", "archevêqu"},
		{"archiduc", "archiduc"},
		{"archiépiscopal", "archiépiscopal"},
		{"archiépiscopale", "archiépiscopal"},
		{"archifolle", "archifoll"},
		{"archiprêtre", "archiprêtr"},
		{"architecte", "architect"},
		{"architecture", "architectur"},
		{"archives", "archiv"},
		{"arcole", "arcol"},
		{"arçon", "arçon"},
		{"ardemment", "ardemment"},
		{"ardent", "ardent"},
		{"ardente", "ardent"},
		{"ardentes", "ardent"},
		{"ardents", "ardent"},
		{"ardeur", "ardeur"},
		{"ardu", "ardu"},
		{"are", "are"},
		{"arecs", "arec"},
		{"aresi", "ares"},
		{"arêtes", "arêt"},
		{"argent", "argent"},
		{"argentée", "argent"},
		{"argile", "argil"},
		{"argument", "argument"},
		{"arguments", "argument"},
		{"arides", "arid"},
		{"arioste", "ariost"},
		{"aristocrate", "aristocrat"},
		{"aristocratie", "aristocrat"},
		{"aristocratique", "aristocrat"},
		{"aristocratiques", "aristocrat"},
		{"arithmétique", "arithmet"},
		{"arkansas", "arkans"},
		{"arlequin", "arlequin"},
		{"arma", "arma"},
		{"armait", "armait"},
		{"armant", "armant"},
		{"armateur", "armateur"},
		{"armateurs", "armateur"},
		{"arme", "arme"},
		{"armé", "armé"},
		{"armée", "armé"},
		{"armées", "armé"},
		{"arméniennes", "arménien"},
		{"arméniens", "arménien"},
		{"armer", "armer"},
		{"armes", "arme"},
		{"armés", "armé"},
		{"armide", "armid"},
		{"armoire", "armoir"},
		{"armoires", "armoir"},
		{"armoiries", "armoir"},
		{"armonica", "armonic"},
		{"armurier", "armuri"},
		{"arnauld", "arnauld"},
		{"arpent", "arpent"},
		{"arpents", "arpent"},
		{"arquebuser", "arquebus"},
		{"arquées", "arqué"},
		{"arqués", "arqué"},
		{"arracha", "arrach"},
		{"arrachait", "arrach"},
		{"arrachant", "arrach"},
		{"arrache", "arrach"},
		{"arraché", "arrach"},
		{"arrachée", "arrach"},
		{"arrachent", "arrachent"},
		{"arracher", "arrach"},
		{"arrachèrent", "arrach"},
		{"arrachés", "arrach"},
		{"arrange", "arrang"},
		{"arrangé", "arrang"},
		{"arrangea", "arrang"},
		{"arrangeaient", "arrang"},
		{"arrangeait", "arrang"},
		{"arrangée", "arrang"},
		{"arrangées", "arrang"},
		{"arrangement", "arrang"},
		{"arrangements", "arrang"},
		{"arrangent", "arrangent"},
		{"arranger", "arrang"},
		{"arrangera", "arrang"},
		{"arrangerait", "arrang"},
		{"arrangés", "arrang"},
		{"arrérages", "arrérag"},
		{"arrestation", "arrest"},
		{"arrestations", "arrest"},
		{"arrêt", "arrêt"},
		{"arrêta", "arrêt"},
		{"arrêtaient", "arrêt"},
		{"arrêtait", "arrêt"},
		{"arrêtant", "arrêt"},
		{"arrête", "arrêt"},
		{"arrêté", "arrêt"},
		{"arrêtée", "arrêt"},
		{"arrêtées", "arrêt"},
		{"arrêtent", "arrêtent"},
		{"arrêter", "arrêt"},
		{"arrêtera", "arrêt"},
		{"arrêterai", "arrêt"},
		{"arrêterait", "arrêt"},
		{"arrêtèrent", "arrêt"},
		{"arrêterons", "arrêt"},
		{"arrêteront", "arrêt"},
		{"arrêtés", "arrêt"},
		{"arrêtez", "arrêt"},
		{"arrêtiez", "arrêt"},
		{"arrêtons", "arrêton"},
		{"arrêts", "arrêt"},
		{"arrhes", "arrhe"},
		{"arrien", "arrien"},
		{"arrière", "arrier"},
		{"arriva", "arriv"},
		{"arrivai", "arriv"},
		{"arrivaient", "arriv"},
		{"arrivais", "arriv"},
		{"arrivait", "arriv"},
		{"arrivant", "arriv"},
		{"arrivants", "arriv"},
		{"arrivât", "arriv"},
		{"arrive", "arriv"},
		{"arrivé", "arriv"},
		{"arrivée", "arriv"},
		{"arrivées", "arriv"},
		{"arrivent", "arrivent"},
		{"arriver", "arriv"},
		{"arrivera", "arriv"},
		{"arriveraient", "arriv"},
		{"arriverais", "arriv"},
		{"arriverait", "arriv"},
		{"arrivèrent", "arriv"},
		{"arriverez", "arriv"},
		{"arriverions", "arriv"},
		{"arriverons", "arriv"},
		{"arriveront", "arriv"},
		{"arrives", "arriv"},
		{"arrivés", "arriv"},
		{"arrivez", "arriv"},
		{"arrivons", "arrivon"},
		{"arrogante", "arrog"},
		{"arrondies", "arrond"},
		{"arrondis", "arrond"},
		{"arrondissait", "arrond"},
		{"arrondissement", "arrond"},
		{"arrondit", "arrond"},
		{"arrosé", "arros"},
		{"arrosées", "arros"},
		{"arsenal", "arsenal"},
		{"arsène", "arsen"},
		{"arsenic", "arsenic"},
		{"art", "art"},
		{"arte", "arte"},
		{"artère", "arter"},
		{"arthémidore", "arthémidor"},
		{"article", "articl"},
		{"articles", "articl"},
		{"articulait", "articul"},
		{"articulant", "articul"},
		{"articulée", "articul"},
		{"articuler", "articul"},
		{"articulés", "articul"},
		{"artifice", "artific"},
		{"artillerie", "artiller"},
		{"artisans", "artisan"},
		{"artiste", "artist"},
		{"artistes", "artist"},
		{"arts", "art"},
		{"as", "as"},
		{"ascagne", "ascagn"},
		{"ascanio", "ascanio"},
		{"ascétique", "ascet"},
		{"asie", "asi"},
		{"asile", "asil"},
		{"aspect", "aspect"},
		{"aspects", "aspect"},
		{"aspérités", "asper"},
		{"asphaltite", "asphaltit"},
		{"asphyxie", "asphyx"},
		{"asphyxié", "asphyxi"},
		{"aspiciam", "aspiciam"},
		{"aspira", "aspir"},
		{"aspirait", "aspir"},
		{"aspirants", "aspir"},
		{"aspirations", "aspir"},
		{"aspire", "aspir"},
		{"aspirer", "aspir"},
		{"aspirez", "aspir"},
		{"assaillaient", "assaill"},
		{"assaillant", "assaill"},
		{"assaillants", "assaill"},
		{"assailli", "assaill"},
		{"assaillie", "assaill"},
		{"assaillis", "assaill"},
		{"assassin", "assassin"},
		{"assassinait", "assassin"},
		{"assassinat", "assassinat"},
		{"assassine", "assassin"},
		{"assassiné", "assassin"},
		{"assassiner", "assassin"},
		{"assassinés", "assassin"},
		{"assassins", "assassin"},
		{"assaut", "assaut"},
		{"assembla", "assembl"},
		{"assemblée", "assembl"},
		{"assembler", "assembl"},
		{"assentiment", "assent"},
		{"asseoir", "asseoir"},
		{"assertions", "assert"},
		{"asseyaient", "assei"},
		{"asseyait", "assei"},
		{"asseyant", "assei"},
		{"asseyez", "assei"},
		{"assez", "assez"},
		{"assidue", "assidu"},
		{"assiduité", "assidu"},
		{"assiduités", "assidu"},
		{"assidûment", "assidû"},
		{"assied", "assied"},
		{"assiégé", "assieg"},
		{"assiégée", "assieg"},
		{"assiéger", "assieg"},
		{"assiégés", "assieg"},
		{"assiette", "assiet"},
		{"assignats", "assignat"},
		{"assigne", "assign"},
		{"assigné", "assign"},
		{"assigner", "assign"},
		{"assirent", "assirent"},
		{"assis", "assis"},
		{"assise", "assis"},
		{"assises", "assis"},
		{"assista", "assist"},
		{"assistaient", "assist"},
		{"assistait", "assist"},
		{"assistance", "assist"},
		{"assistant", "assist"},
		{"assistants", "assist"},
		{"assiste", "assist"},
		{"assisté", "assist"},
		{"assister", "assist"},
		{"assistera", "assist"},
		{"assit", "assit"},
		{"associaient", "associ"},
		{"association", "associ"},
		{"associations", "associ"},
		{"associe", "assoc"},
		{"associé", "associ"},
		{"assoient", "assoient"},
		{"assombri", "assombr"},
		{"assombrit", "assombr"},
		{"assommant", "assomm"},
		{"assommé", "assomm"},
		{"assommer", "assomm"},
		{"assommés", "assomm"},
		{"assommoir", "assommoir"},
		{"assoupie", "assoup"},
		{"assoupissantes", "assoup"},
		{"assoupissement", "assoup"},
		{"assoupit", "assoup"},
		{"assourdissant", "assourd"},
		{"assourdit", "assourd"},
		{"assura", "assur"},
		{"assuraient", "assur"},
		{"assurait", "assur"},
		{"assurance", "assur"},
		{"assurances", "assur"},
		{"assure", "assur"},
		{"assuré", "assur"},
		{"assurée", "assur"},
		{"assurées", "assur"},
		{"assurément", "assur"},
		{"assurer", "assur"},
		{"assurerait", "assur"},
		{"assurerez", "assur"},
		{"assureront", "assur"},
		{"assurghur", "assurghur"},
		{"astéroïdes", "astéroïd"},
		{"asthme", "asthme"},
		{"asti", "asti"},
		{"astre", "astre"},
		{"astreindre", "astreindr"},
		{"astreins", "astrein"},
		{"astres", "astre"},
		{"astrologie", "astrolog"},
		{"astrologique", "astrolog"},
		{"astrologiques", "astrolog"},
		{"astrologues", "astrologu"},
		{"astronomes", "astronom"},
		{"astronomie", "astronom"},
		{"astronomique", "astronom"},
		{"astuce", "astuc"},
		{"at", "at"},
		{"atelier", "ateli"},
		{"athènes", "athen"},
		{"atlantique", "atlant"},
		{"atmosphère", "atmospher"},
		{"atome", "atom"},
		{"atours", "atour"},
		{"atouts", "atout"},
		{"atroce", "atroc"},
		{"atrocement", "atroc"},
		{"atroces", "atroc"},
		{"atrocité", "atroc"},
		{"attacha", "attach"},
		{"attachaient", "attach"},
		{"attachait", "attach"},
		{"attachant", "attach"},
		{"attache", "attach"},
		{"attaché", "attach"},
		{"attachée", "attach"},
		{"attachées", "attach"},
		{"attachement", "attach"},
		{"attachent", "attachent"},
		{"attacher", "attach"},
		{"attachera", "attach"},
		{"attacherai", "attach"},
		{"attacherait", "attach"},
		{"attachèrent", "attach"},
		{"attachés", "attach"},
		{"attaqua", "attaqu"},
		{"attaquaient", "attaqu"},
		{"attaquait", "attaqu"},
		{"attaque", "attaqu"},
		{"attaqué", "attaqu"},
		{"attaquée", "attaqu"},
		{"attaquer", "attaqu"},
		{"attaques", "attaqu"},
		{"attaqués", "attaqu"},
		{"attardé", "attard"},
		{"atteignaient", "atteign"},
		{"atteignait", "atteign"},
		{"atteignirent", "atteign"},
		{"atteignit", "atteign"},
		{"atteindrait", "atteindr"},
		{"atteindre", "atteindr"},
		{"atteint", "atteint"},
		{"atteintes", "atteint"},
		{"atteints", "atteint"},
		{"attelage", "attelag"},
		{"attelages", "attelag"},
		{"attelait", "attel"},
		{"attelé", "attel"},
		{"attelée", "attel"},
		{"attelés", "attel"},
		{"attelle", "attel"},
		{"attenait", "atten"},
		{"attenant", "atten"},
		{"attenante", "atten"},
		{"attend", "attend"},
		{"attendaient", "attend"},
		{"attendais", "attend"},
		{"attendait", "attend"},
		{"attendant", "attend"},
		{"attendent", "attendent"},
		{"attendez", "attend"},
		{"attendirent", "attend"},
		{"attendit", "attend"},
		{"attendons", "attendon"},
		{"attendra", "attendr"},
		{"attendrai", "attendr"},
		{"attendrait", "attendr"},
		{"attendre", "attendr"},
		{"attendri", "attendr"},
		{"attendrie", "attendr"},
		{"attendrir", "attendr"},
		{"attendriraient", "attendr"},
		{"attendris", "attendr"},
		{"attendrissant", "attendr"},
		{"attendrissement", "attendr"},
		{"attendrit", "attendr"},
		{"attends", "attend"},
		{"attendu", "attendu"},
		{"attendue", "attendu"},
		{"attendus", "attendus"},
		{"attente", "attent"},
		{"attenté", "attent"},
		{"attenter", "attent"},
		{"attentif", "attent"},
		{"attentifs", "attent"},
		{"attention", "attent"},
		{"attentions", "attent"},
		{"attentive", "attent"},
		{"attentivement", "attent"},
		{"atterra", "atterr"},
		{"atterrages", "atterrag"},
		{"atterré", "atterr"},
		{"atterrée", "atterr"},
		{"attestaient", "attest"},
		{"attiédi", "attied"},
		{"attira", "attir"},
		{"attiraient", "attir"},
		{"attirait", "attir"},
		{"attire", "attir"},
		{"attiré", "attir"},
		{"attirent", "attirent"},
		{"attirer", "attir"},
		{"attirerait", "attir"},
		{"attirèrent", "attir"},
		{"attirés", "attir"},
		{"attitude", "attitud"},
		{"attraction", "attract"},
		{"attrait", "attrait"},
		{"attrapé", "attrap"},
		{"attrapera", "attrap"},
		{"attrayant", "attrai"},
		{"attrayantes", "attrai"},
		{"attribuaient", "attribu"},
		{"attribuait", "attribu"},
		{"attribué", "attribu"},
		{"attribuée", "attribu"},
		{"attribuer", "attribu"},
		{"attribuera", "attribu"},
		{"attributions", "attribu"},
		{"attristait", "attrist"},
		{"attristé", "attrist"},
		{"attristée", "attrist"},
		{"attrister", "attrist"},
		{"attroupement", "attroup"},
		{"attrouper", "attroup"},
		{"au", "au"},
		{"aubaine", "aubain"},
		{"aubaines", "aubain"},
		{"aube", "aub"},
		{"auberge", "auberg"},
		{"auberges", "auberg"},
		{"aubergistes", "auberg"},
		{"aubigné", "aubign"},
		{"aubry", "aubry"},
		{"auburn", "auburn"},
		{"aucun", "aucun"},
		{"aucune", "aucun"},
		{"aucunement", "aucun"},
		{"audace", "audac"},
		{"audacieuse", "audaci"},
		{"audacieuses", "audaci"},
		{"audacieux", "audaci"},
		{"audience", "audienc"},
		{"audiences", "audienc"},
		{"audit", "audit"},
		{"auditeur", "auditeur"},
		{"auditeurs", "auditeur"},
		{"auditoire", "auditoir"},
		{"augmenta", "augment"},
		{"augmentaient", "augment"},
		{"augmentait", "augment"},
		{"augmentant", "augment"},
		{"augmentation", "augment"},
		{"augmente", "augment"},
		{"augmenté", "augment"},
		{"augmentée", "augment"},
		{"augmenter", "augment"},
		{"augmentera", "augment"},
		{"augmenterait", "augment"},
		{"augure", "augur"},
		{"augurer", "augur"},
		{"auguste", "august"},
		{"augustes", "august"},
		{"augustin", "augustin"},
		{"aujourd", "aujourd"},
		{"aumône", "aumôn"},
		{"aumônes", "aumôn"},
		{"aumônier", "aumôni"},
		{"aunes", "aun"},
		{"auparavant", "auparav"},
		{"auprès", "aupres"},
		{"auquel", "auquel"},
		{"aura", "aur"},
		{"aurai", "aur"},
		{"auraient", "aur"},
		{"aurais", "aur"},
		{"aurait", "aur"},
		{"auras", "aur"},
		{"aureng", "aureng"},
		{"aurez", "aur"},
		{"auriez", "aur"},
		{"aurions", "aurion"},
		{"aurons", "auron"},
		{"auront", "auront"},
		{"aurungabad", "aurungabad"},
		{"aussi", "auss"},
		{"aussitôt", "aussitôt"},
		{"austère", "auster"},
		{"austères", "auster"},
		{"austérité", "auster"},
		{"austral", "austral"},
		{"autant", "aut"},
		{"autel", "autel"},
		{"autels", "autel"},
		{"auteur", "auteur"},
		{"auteurs", "auteur"},
		{"authenticité", "authent"},
		{"auto", "auto"},
		{"automate", "automat"},
		{"automatique", "automat"},
		{"automne", "automn"},
		{"autorisait", "autoris"},
		{"autorisation", "autoris"},
		{"autorisations", "autoris"},
		{"autorise", "autoris"},
		{"autorisé", "autoris"},
		{"autorisée", "autoris"},
		{"autorisent", "autorisent"},
		{"autorisera", "autoris"},
		{"autoriserait", "autoris"},
		{"autorité", "autor"},
		{"autorités", "autor"},
		{"autour", "autour"},
		{"autre", "autr"},
		{"autrefois", "autrefois"},
		{"autrement", "autr"},
		{"autres", "autr"},
		{"autriche", "autrich"},
		{"autrichien", "autrichien"},
		{"autrichienne", "autrichien"},
		{"autrichiennes", "autrichien"},
		{"autrichiens", "autrichien"},
		{"aux", "aux"},
		{"auxiliaire", "auxiliair"},
		{"auxquelles", "auxquel"},
		{"auxquels", "auxquel"},
		{"av", "av"},
		{"avaient", "avaient"},
		{"avais", "avais"},
		{"avait", "avait"},
		{"avala", "aval"},
		{"avalé", "aval"},
		{"avança", "avanc"},
		{"avançaient", "avanc"},
		{"avançait", "avanc"},
		{"avançant", "avanc"},
		{"avance", "avanc"},
		{"avancé", "avanc"},
		{"avancée", "avanc"},
		{"avancement", "avanc"},
		{"avancements", "avanc"},
		{"avancent", "avancent"},
		{"avancer", "avanc"},
		{"avancera", "avanc"},
		{"avancerait", "avanc"},
		{"avancèrent", "avanc"},
		{"avances", "avanc"},
		{"avancés", "avanc"},
		{"avancez", "avanc"},
		{"avanie", "avan"},
		{"avanies", "avan"},
		{"avant", "avant"},
		{"avantage", "avantag"},
		{"avantages", "avantag"},
		{"avantageuse", "avantag"},
		{"avantageusement", "avantag"},
		{"avantageux", "avantag"},
		{"avare", "avar"},
		{"avarice", "avaric"},
		{"avarie", "avar"},
		{"avaries", "avar"},
		{"ave", "ave"},
		{"avec", "avec"},
		{"avenir", "aven"},
		{"aventure", "aventur"},
		{"aventurent", "aventurent"},
		{"aventurer", "aventur"},
		{"aventures", "aventur"},
		{"aventureuse", "aventur"},
		{"aventureux", "aventur"},
		{"avenues", "avenu"},
		{"avéré", "aver"},
		{"averse", "avers"},
		{"aversion", "avers"},
		{"averti", "avert"},
		{"avertie", "avert"},
		{"averties", "avert"},
		{"avertir", "avert"},
		{"avertira", "avert"},
		{"avertis", "avert"},
		{"avertissait", "avert"},
		{"avertissant", "avert"},
		{"avertissement", "avert"},
		{"avertit", "avert"},
		{"aveu", "aveu"},
		{"aveugle", "aveugl"},
		{"aveuglé", "aveugl"},
		{"aveuglée", "aveugl"},
		{"aveuglement", "aveugl"},
		{"aveuglément", "aveugl"},
		{"aveugles", "aveugl"},
		{"aveux", "aveux"},
		{"aveyron", "aveyron"},
		{"avez", "avez"},
		{"avide", "avid"},
		{"avidement", "avid"},
		{"avidité", "avid"},
		{"aviez", "avi"},
		{"avilie", "avil"},
		{"avilir", "avil"},
		{"avilirait", "avil"},
		{"avilissant", "avil"},
		{"avilissantes", "avil"},
		{"avilit", "avil"},
		{"avions", "avion"},
		{"aviron", "aviron"},
		{"avis", "avis"},
		{"avisa", "avis"},
		{"avisais", "avis"},
		{"avisait", "avis"},
		{"avise", "avis"},
		{"avisé", "avis"},
		{"avisée", "avis"},
		{"aviser", "avis"},
		{"aviserai", "avis"},
		{"aviserait", "avis"},
		{"aviserons", "avis"},
		{"avises", "avis"},
		{"avocat", "avocat"},
		{"avocats", "avocat"},
		{"avoine", "avoin"},
		{"avoir", "avoir"},
		{"avons", "avon"},
		{"avoua", "avou"},
		{"avouai", "avou"},
		{"avouait", "avou"},
		{"avouât", "avou"},
		{"avoue", "avou"},
		{"avoué", "avou"},
		{"avouent", "avouent"},
		{"avouer", "avou"},
		{"avouera", "avou"},
		{"avouerai", "avou"},
		{"avouerons", "avou"},
		{"avoués", "avou"},
		{"avouez", "avou"},
		{"avril", "avril"},
		{"avviamento", "avviamento"},
		{"away", "away"},
		{"axe", "axe"},
		{"ayant", "ayant"},
		{"ayez", "ayez"},
		{"aymon", "aymon"},
		{"ayons", "ayon"},
		{"azur", "azur"},
		{"b", "b"},
		{"bab", "bab"},
		{"bâbord", "bâbord"},
		{"babouches", "babouch"},
		{"babylone", "babylon"},
		{"bac", "bac"},
		{"bacchus", "bacchus"},
		{"back", "back"},
		{"bad", "bad"},
		{"badaud", "badaud"},
		{"badauderie", "badauder"},
		{"baden", "baden"},
		{"badin", "badin"},
		{"badine", "badin"},
		{"bafouait", "bafou"},
		{"bagage", "bagag"},
		{"bagages", "bagag"},
		{"bagarre", "bagarr"},
		{"bagatelle", "bagatel"},
		{"bagne", "bagn"},
		{"bague", "bagu"},
		{"bagues", "bagu"},
		{"baguette", "baguet"},
		{"bah", "bah"},
		{"baie", "bai"},
		{"baigné", "baign"},
		{"baignée", "baign"},
		{"baigner", "baign"},
		{"baignés", "baign"},
		{"bail", "bail"},
		{"bâillait", "bâill"},
		{"bâillant", "bâill"},
		{"bâillement", "bâill"},
		{"bâiller", "bâill"},
		{"bâillon", "bâillon"},
		{"baïonnette", "baïonnet"},
		{"baïonnettes", "baïonnet"},
		{"baisa", "bais"},
		{"baisai", "bais"},
		{"baisait", "bais"},
		{"baisant", "bais"},
		{"baise", "bais"},
		{"baiser", "bais"},
		{"baisers", "baiser"},
		{"baissa", "baiss"},
		{"baissait", "baiss"},
		{"baissant", "baiss"},
		{"baisse", "baiss"},
		{"baissé", "baiss"},
		{"baissée", "baiss"},
		{"baisser", "baiss"},
		{"baissèrent", "baiss"},
		{"baissés", "baiss"},
		{"bajocs", "bajoc"},
		{"bal", "bal"},
		{"baladins", "baladin"},
		{"balançait", "balanc"},
		{"balance", "balanc"},
		{"balancée", "balanc"},
		{"balancement", "balanc"},
		{"balancer", "balanc"},
		{"balances", "balanc"},
		{"balancier", "balanci"},
		{"balayé", "balai"},
		{"balayés", "balai"},
		{"balbi", "balb"},
		{"balbutia", "balbuti"},
		{"balbutiant", "balbuti"},
		{"balcon", "balcon"},
		{"balcons", "balcon"},
		{"baldaquin", "baldaquin"},
		{"baldi", "bald"},
		{"balivernes", "balivern"},
		{"balland", "balland"},
		{"ballants", "ball"},
		{"balle", "ball"},
		{"balles", "ball"},
		{"ballet", "ballet"},
		{"ballon", "ballon"},
		{"balourd", "balourd"},
		{"balourdise", "balourdis"},
		{"balourdises", "balourdis"},
		{"bals", "bal"},
		{"balustrade", "balustrad"},
		{"bamboches", "bamboch"},
		{"bambou", "bambou"},
		{"bambous", "bambous"},
		{"bambousiers", "bambousi"},
		{"ban", "ban"},
		{"bananiers", "banani"},
		{"banc", "banc"},
		{"bancs", "banc"},
		{"bande", "band"},
		{"bandé", "band"},
		{"banderoles", "banderol"},
		{"bandes", "band"},
		{"bandits", "bandit"},
		{"bandoulière", "bandouli"},
		{"banians", "banian"},
		{"bank", "bank"},
		{"banni", "bann"},
		{"bannie", "bann"},
		{"bannières", "banni"},
		{"bannir", "bann"},
		{"banque", "banqu"},
		{"banqueroute", "banquerout"},
		{"banquette", "banquet"},
		{"banquettes", "banquet"},
		{"banquier", "banqui"},
		{"banquière", "banqui"},
		{"banquiers", "banqui"},
		{"baptisé", "baptis"},
		{"bar", "bar"},
		{"baraque", "baraqu"},
		{"barbare", "barbar"},
		{"barbares", "barbar"},
		{"barbarie", "barbar"},
		{"barbe", "barb"},
		{"barbiche", "barbich"},
		{"barbier", "barbi"},
		{"barbone", "barbon"},
		{"barbotement", "barbot"},
		{"barbouillait", "barbouill"},
		{"barême", "barêm"},
		{"bari", "bar"},
		{"baring", "baring"},
		{"bariolée", "bariol"},
		{"barnave", "barnav"},
		{"barnum", "barnum"},
		{"baromètre", "barometr"},
		{"baromètres", "barometr"},
		{"baron", "baron"},
		{"baronne", "baron"},
		{"baronnie", "baron"},
		{"barons", "baron"},
		{"baroque", "baroqu"},
		{"barque", "barqu"},
		{"barques", "barqu"},
		{"barra", "barr"},
		{"barre", "barr"},
		{"barreaux", "barreau"},
		{"barrer", "barr"},
		{"barricader", "barricad"},
		{"barricadés", "barricad"},
		{"barrière", "barri"},
		{"barrières", "barri"},
		{"barthélemy", "barthélemy"},
		{"bas", "bas"},
		{"basalte", "basalt"},
		{"base", "bas"},
		{"basée", "bas"},
		{"bases", "bas"},
		{"basile", "basil"},
		{"basilique", "basil"},
		{"basse", "bass"},
		{"bassement", "bass"},
		{"basses", "bass"},
		{"bassesse", "bassess"},
		{"bassesses", "bassess"},
		{"bassin", "bassin"},
		{"bassinet", "bassinet"},
		{"bassins", "bassin"},
		{"bassompierre", "bassompierr"},
		{"bastingages", "bastingag"},
		{"bastion", "bastion"},
		{"bastions", "bastion"},
		{"bat", "bat"},
		{"bataille", "bataill"},
		{"batailles", "bataill"},
		{"bataillon", "bataillon"},
		{"bataillons", "bataillon"},
		{"bâtardise", "bâtardis"},
		{"bateau", "bateau"},
		{"bateaux", "bateau"},
		{"batelier", "bateli"},
		{"bateliers", "bateli"},
		{"bathurst", "bathurst"},
		{"bâti", "bât"},
		{"bâtie", "bât"},
		{"bâties", "bât"},
		{"bâtiment", "bât"},
		{"bâtiments", "bât"},
		{"bâtir", "bât"},
		{"bâtis", "bât"},
		{"bâtissait", "bât"},
		{"bâtissant", "bât"},
		{"bâtisses", "bât"},
		{"bâtisseur", "bâtisseur"},
		{"bâtit", "bât"},
		{"bâton", "bâton"},
		{"bâtonne", "bâton"},
		{"bâtons", "bâton"},
		{"bats", "bat"},
		{"battaient", "batt"},
		{"battait", "batt"},
		{"battant", "batt"},
		{"battante", "batt"},
		{"battants", "batt"},
		{"batte", "batt"},
		{"battement", "batt"},
		{"battements", "batt"},
		{"battent", "battent"},
		{"batterie", "batter"},
		{"battez", "batt"},
		{"battistin", "battistin"},
		{"battit", "batt"},
		{"battrai", "battr"},
		{"battrais", "battr"},
		{"battras", "battr"},
		{"battre", "battr"},
		{"battrez", "battr"},
		{"battriez", "battr"},
		{"battu", "battu"},
		{"battue", "battu"},
		{"battus", "battus"},
		{"batulcar", "batulcar"},
		{"baume", "baum"},
		{"bavard", "bavard"},
		{"bavardage", "bavardag"},
		{"bavardages", "bavardag"},
		{"bavarde", "bavard"},
		{"bavardes", "bavard"},
		{"bavards", "bavard"},
		{"bayadères", "bayader"},
		{"bayard", "bayard"},
		{"baylen", "baylen"},
		{"bazar", "bazar"},
		{"bazars", "bazar"},
		{"bazin", "bazin"},
		{"be", "be"},
		{"béante", "bé"},
		{"béatitudes", "béatitud"},
		{"beau", "beau"},
		{"beaucoup", "beaucoup"},
		{"beauharnais", "beauharn"},
		{"beaujolais", "beaujol"},
		{"beaumarchais", "beaumarch"},
		{"beauté", "beaut"},
		{"beautés", "beaut"},
		{"beauty", "beauty"},
		{"beauvais", "beauv"},
		{"beauvaisis", "beauvais"},
		{"beauvoisis", "beauvois"},
		{"beaux", "beau"},
		{"bec", "bec"},
		{"bêchait", "bêch"},
		{"bêché", "bêch"},
		{"bêcher", "bêch"},
		{"becs", "bec"},
		{"bédouin", "bédouin"},
		{"beefsteacks", "beefsteack"},
		{"bégayait", "bégai"},
		{"béhar", "béhar"},
		{"behind", "behind"},
		{"békir", "bek"},
		{"bel", "bel"},
		{"belagio", "belagio"},
		{"bélan", "bélan"},
		{"belgique", "belgiqu"},
		{"belgirate", "belgirat"},
		{"belle", "bel"},
		{"belles", "bel"},
		{"belliqueux", "belliqu"},
		{"belphégor", "belphégor"},
		{"belvédère", "belvéder"},
		{"ben", "ben"},
		{"bénarès", "bénares"},
		{"bénédiction", "bénédict"},
		{"bénédictions", "bénédict"},
		{"benefaciendo", "benefaciendo"},
		{"bénéfice", "bénéfic"},
		{"bénéfices", "bénéfic"},
		{"benêt", "benêt"},
		{"bengale", "bengal"},
		{"bénis", "ben"},
		{"bénit", "ben"},
		{"bénite", "bénit"},
		{"bénitier", "béniti"},
		{"benjamin", "benjamin"},
		{"benten", "benten"},
		{"bentivoglio", "bentivoglio"},
		{"béranger", "bérang"},
		{"berceau", "berceau"},
		{"bercer", "berc"},
		{"bérésina", "bérésin"},
		{"bergère", "berger"},
		{"bergères", "berger"},
		{"bernard", "bernard"},
		{"berne", "bern"},
		{"bertolotti", "bertolott"},
		{"bertuccio", "bertuccio"},
		{"besace", "besac"},
		{"besançon", "besançon"},
		{"besenval", "besenval"},
		{"besogne", "besogn"},
		{"besoin", "besoin"},
		{"besoins", "besoin"},
		{"bêta", "bêt"},
		{"bétail", "bétail"},
		{"bête", "bêt"},
		{"bétel", "bétel"},
		{"bêtement", "bêt"},
		{"bêtes", "bêt"},
		{"bêtise", "bêtis"},
		{"bêtises", "bêtis"},
		{"betrays", "betray"},
		{"bettina", "bettin"},
		{"beuglements", "beugl"},
		{"beurre", "beurr"},
		{"beuve", "beuv"},
		{"bévue", "bévu"},
		{"bévues", "bévu"},
		{"biais", "bi"},
		{"biaiser", "biais"},
		{"bible", "bibl"},
		{"bibliomane", "biblioman"},
		{"bibliothèque", "bibliothequ"},
		{"bibliothèques", "bibliothequ"},
		{"biblique", "bibliqu"},
		{"bibliques", "bibliqu"},
		{"bielle", "biel"},
		{"bien", "bien"},
		{"bienfait", "bienf"},
		{"bienfaiteur", "bienfaiteur"},
		{"bienfaitrice", "bienfaitric"},
		{"bienfaits", "bienfait"},
		{"biens", "bien"},
		{"bientôt", "bientôt"},
		{"bienveillance", "bienveil"},
		{"bienveillant", "bienveil"},
		{"bienvenu", "bienvenu"},
		{"bière", "bier"},
		{"biftecks", "bifteck"},
		{"bigarrée", "bigarr"},
		{"bijou", "bijou"},
		{"bijoux", "bijoux"},
		{"bilieuse", "bilieux"},
		{"bilieux", "bilieux"},
		{"billard", "billard"},
		{"billards", "billard"},
		{"billet", "billet"},
		{"billets", "billet"},
		{"binder", "bind"},
		{"binet", "binet"},
		{"biographie", "biograph"},
		{"birmingham", "birmingham"},
		{"bisaïeul", "bisaïeul"},
		{"biscaïens", "biscaïen"},
		{"biscuit", "biscuit"},
		{"biscuits", "biscuit"},
		{"bisogna", "bisogn"},
		{"bisons", "bison"},
		{"bisontine", "bisontin"},
		{"bisontins", "bisontin"},
		{"bisontium", "bisontium"},
		{"bitter", "bitt"},
		{"bivouac", "bivouac"},
		{"bivouacs", "bivouac"},
		{"bivouaquaient", "bivouaqu"},
		{"bizarre", "bizarr"},
		{"bizarrerie", "bizarrer"},
		{"bizarres", "bizarr"},
		{"blackest", "blackest"},
		{"blafarde", "blafard"},
		{"blâma", "blâm"},
		{"blâmable", "blâmabl"},
		{"blâmables", "blâmabl"},
		{"blâmait", "blâm"},
		{"blâme", "blâm"},
		{"blâmé", "blâm"},
		{"blâmée", "blâm"},
		{"blâmer", "blâm"},
		{"blâmés", "blâm"},
		{"blanc", "blanc"},
		{"blanche", "blanch"},
		{"blanches", "blanch"},
		{"blancheur", "blancheur"},
		{"blanchie", "blanch"},
		{"blanchies", "blanch"},
		{"blanchis", "blanch"},
		{"blanchisseuse", "blanchiss"},
		{"blancs", "blanc"},
		{"bland", "bland"},
		{"blanès", "blanes"},
		{"blason", "blason"},
		{"blasphème", "blasphem"},
		{"blasphémé", "blasphem"},
		{"blé", "blé"},
		{"blême", "blêm"},
		{"blés", "blé"},
		{"blessa", "bless"},
		{"blessaient", "bless"},
		{"blessant", "bless"},
		{"blesse", "bless"},
		{"blessé", "bless"},
		{"blessée", "bless"},
		{"blesser", "bless"},
		{"blesserait", "bless"},
		{"blessés", "bless"},
		{"blessez", "bless"},
		{"blessure", "blessur"},
		{"blessures", "blessur"},
		{"bleu", "bleu"},
		{"bleuâtre", "bleuâtr"},
		{"bleuâtres", "bleuâtr"},
		{"bleue", "bleu"},
		{"bleues", "bleu"},
		{"bleus", "bleus"},
		{"bloc", "bloc"},
		{"blocs", "bloc"},
		{"blond", "blond"},
		{"blonde", "blond"},
		{"blondin", "blondin"},
		{"blonds", "blond"},
		{"blood", "blood"},
		{"blottie", "blott"},
		{"blottirent", "blott"},
		{"blottit", "blott"},
		{"blue", "blu"},
		{"bluffs", "bluff"},
		{"blushes", "blush"},
		{"boat", "boat"},
		{"boats", "boat"},
		{"boeuf", "boeuf"},
		{"boeufs", "boeuf"},
		{"boire", "boir"},
		{"bois", "bois"},
		{"boisé", "bois"},
		{"boiserie", "boiser"},
		{"boisson", "boisson"},
		{"boissons", "boisson"},
		{"boit", "boit"},
		{"boîte", "boît"},
		{"boîtes", "boît"},
		{"boiteux", "boiteux"},
		{"bologne", "bologn"},
		{"bombace", "bombac"},
		{"bombardé", "bombard"},
		{"bombay", "bombay"},
		{"bombe", "bomb"},
		{"bombes", "bomb"},
		{"bombés", "bomb"},
		{"bon", "bon"},
		{"bona", "bon"},
		{"bonaparte", "bonapart"},
		{"bonapartiste", "bonapart"},
		{"bonaventure", "bonaventur"},
		{"bondir", "bond"},
		{"bondissait", "bond"},
		{"bondissant", "bond"},
		{"bondit", "bond"},
		{"bonds", "bond"},
		{"bone", "bon"},
		{"bonheur", "bonheur"},
		{"bonheurs", "bonheur"},
		{"bonhomie", "bonhom"},
		{"bonhomme", "bonhomm"},
		{"boniface", "bonifac"},
		{"bonne", "bon"},
		{"bonnement", "bon"},
		{"bonnes", "bon"},
		{"bonnet", "bonnet"},
		{"bonnets", "bonnet"},
		{"bons", "bon"},
		{"bonshommes", "bonshomm"},
		{"bonsoir", "bonsoir"},
		{"bonté", "bont"},
		{"bontés", "bont"},
		{"bonzeries", "bonzer"},
		{"bonzes", "bonz"},
		{"booby", "booby"},
		{"bord", "bord"},
		{"borda", "bord"},
		{"bordaient", "bord"},
		{"bordait", "bord"},
		{"bordé", "bord"},
		{"bordeaux", "bordeau"},
		{"bordée", "bord"},
		{"bordées", "bord"},
		{"bordent", "bordent"},
		{"bordés", "bord"},
		{"bords", "bord"},
		{"bordures", "bordur"},
		{"borgo", "borgo"},
		{"borna", "born"},
		{"bornaient", "born"},
		{"bornait", "born"},
		{"borne", "born"},
		{"borné", "born"},
		{"bornes", "born"},
		{"borone", "boron"},
		{"borso", "borso"},
		{"bosco", "bosco"},
		{"bosse", "boss"},
		{"bosses", "boss"},
		{"bossi", "boss"},
		{"bossuet", "bossuet"},
		{"botanique", "botan"},
		{"botte", "bott"},
		{"bottes", "bott"},
		{"bottier", "botti"},
		{"bouche", "bouch"},
		{"boucher", "bouch"},
		{"boucherie", "boucher"},
		{"bouchers", "boucher"},
		{"bouches", "bouch"},
		{"boucles", "boucl"},
		{"bouclés", "boucl"},
		{"bouclez", "boucl"},
		{"bouclier", "boucli"},
		{"bouddhisme", "bouddhism"},
		{"bouddhiste", "bouddhist"},
		{"bouder", "boud"},
		{"boudeur", "boudeur"},
		{"boudeurs", "boudeur"},
		{"boue", "bou"},
		{"bouffe", "bouff"},
		{"bouffée", "bouff"},
		{"bouffées", "bouff"},
		{"bouffes", "bouff"},
		{"bouffonnerie", "bouffonner"},
		{"bouffonneries", "bouffonner"},
		{"bougeait", "boug"},
		{"bougeoir", "bougeoir"},
		{"bouger", "boug"},
		{"bougera", "boug"},
		{"bougez", "boug"},
		{"bougie", "boug"},
		{"bougies", "boug"},
		{"bougon", "bougon"},
		{"bouillait", "bouill"},
		{"bouillant", "bouill"},
		{"bouillante", "bouill"},
		{"bouilli", "bouill"},
		{"bouillir", "bouill"},
		{"bouillon", "bouillon"},
		{"boulangers", "boulanger"},
		{"boule", "boul"},
		{"bouledogue", "bouledogu"},
		{"boulet", "boulet"},
		{"boulets", "boulet"},
		{"boulettes", "boulet"},
		{"boulevard", "boulevard"},
		{"bouleversa", "boulevers"},
		{"bouleversaient", "boulevers"},
		{"bouleverse", "boulevers"},
		{"bouleversé", "boulevers"},
		{"bouleversée", "boulevers"},
		{"bouleversés", "boulevers"},
		{"boulogne", "boulogn"},
		{"boulons", "boulon"},
		{"boulot", "boulot"},
		{"bouquer", "bouqu"},
		{"bouquet", "bouquet"},
		{"bouquets", "bouquet"},
		{"bourbeux", "bourbeux"},
		{"bourbier", "bourbi"},
		{"bourbiers", "bourbi"},
		{"bourbons", "bourbon"},
		{"bourdon", "bourdon"},
		{"bourdonnement", "bourdon"},
		{"bourdonnements", "bourdon"},
		{"bourg", "bourg"},
		{"bourgade", "bourgad"},
		{"bourgades", "bourgad"},
		{"bourgeois", "bourgeois"},
		{"bourgeoise", "bourgeois"},
		{"bourgeoises", "bourgeois"},
		{"bourgeoisie", "bourgeois"},
		{"bourgeons", "bourgeon"},
		{"bourgogne", "bourgogn"},
		{"bourguignon", "bourguignon"},
		{"bourguignons", "bourguignon"},
		{"bourrasque", "bourrasqu"},
		{"bourrasques", "bourrasqu"},
		{"bourreau", "bourreau"},
		{"bourreaux", "bourreau"},
		{"bourrées", "bourr"},
		{"bourrelée", "bourrel"},
		{"bourrèlements", "bourrel"},
		{"bourse", "bours"},
		{"bourses", "bours"},
		{"boursicot", "boursicot"},
		{"bousculés", "bouscul"},
		{"bout", "bout"},
		{"boutade", "boutad"},
		{"bouteille", "bouteil"},
		{"bouteilles", "bouteil"},
		{"boutique", "boutiqu"},
		{"boutiques", "boutiqu"},
		{"boutiquiers", "boutiqui"},
		{"bouton", "bouton"},
		{"boutonné", "bouton"},
		{"boutonnière", "boutonni"},
		{"boutons", "bouton"},
		{"bouts", "bout"},
		{"bow", "bow"},
		{"boxe", "box"},
		{"bracelets", "bracelet"},
		{"bradshaw", "bradshaw"},
		{"brahma", "brahm"},
		{"brahmanes", "brahman"},
		{"brahmanique", "brahman"},
		{"brahmaniques", "brahman"},
		{"branchages", "branchag"},
		{"branche", "branch"},
		{"branches", "branch"},
		{"brandissant", "brand"},
		{"brandy", "brandy"},
		{"branlante", "branl"},
		{"branle", "branl"},
		{"brantôme", "brantôm"},
		{"bras", "bras"},
		{"brasses", "brass"},
		{"brasseur", "brasseur"},
		{"bravade", "bravad"},
		{"bravait", "brav"},
		{"brave", "brav"},
		{"bravé", "brav"},
		{"bravée", "brav"},
		{"bravement", "brav"},
		{"bravent", "bravent"},
		{"braver", "brav"},
		{"braverais", "brav"},
		{"braves", "brav"},
		{"bravi", "brav"},
		{"bravissimo", "bravissimo"},
		{"bravo", "bravo"},
		{"bravoure", "bravour"},
		{"bray", "bray"},
		{"brebis", "breb"},
		{"bref", "bref"},
		{"brefs", "bref"},
		{"brescia", "bresci"},
		{"brésil", "brésil"},
		{"bretagne", "bretagn"},
		{"brève", "brev"},
		{"brèves", "brev"},
		{"brevet", "brevet"},
		{"brevets", "brevet"},
		{"bréviaire", "bréviair"},
		{"brick", "brick"},
		{"bride", "brid"},
		{"bridés", "brid"},
		{"bridger", "bridg"},
		{"bridon", "bridon"},
		{"brigade", "brigad"},
		{"brigadier", "brigadi"},
		{"brigadiers", "brigadi"},
		{"brigand", "brigand"},
		{"brigands", "brigand"},
		{"brigantine", "brigantin"},
		{"brigham", "brigham"},
		{"brighella", "brighel"},
		{"brilla", "brill"},
		{"brillaient", "brill"},
		{"brillait", "brill"},
		{"brillamment", "brill"},
		{"brillant", "brill"},
		{"brillante", "brill"},
		{"brillantes", "brill"},
		{"brillants", "brill"},
		{"brille", "brill"},
		{"brillé", "brill"},
		{"brillent", "brillent"},
		{"briller", "brill"},
		{"brillèrent", "brill"},
		{"brindisi", "brindis"},
		{"brins", "brin"},
		{"brio", "brio"},
		{"brique", "briqu"},
		{"briques", "briqu"},
		{"brisa", "bris"},
		{"brisaient", "bris"},
		{"brisait", "bris"},
		{"brise", "bris"},
		{"brisé", "bris"},
		{"brisée", "bris"},
		{"brisées", "bris"},
		{"briser", "bris"},
		{"brisèrent", "bris"},
		{"brises", "bris"},
		{"brisés", "bris"},
		{"britannique", "britann"},
		{"britannisée", "britannis"},
		{"british", "british"},
		{"broadway", "broadway"},
		{"brocanteur", "brocanteur"},
		{"brocart", "brocart"},
		{"brochant", "broch"},
		{"brochées", "broch"},
		{"brochure", "brochur"},
		{"brocs", "broc"},
		{"brodé", "brod"},
		{"brodées", "brod"},
		{"broderie", "broder"},
		{"broderies", "broder"},
		{"brodés", "brod"},
		{"bronché", "bronch"},
		{"broncher", "bronch"},
		{"bronze", "bronz"},
		{"brosser", "bross"},
		{"brouette", "brouet"},
		{"brouettes", "brouet"},
		{"brougham", "brougham"},
		{"brouhaha", "brouhah"},
		{"brouillaient", "brouill"},
		{"brouillais", "brouill"},
		{"brouillard", "brouillard"},
		{"brouillards", "brouillard"},
		{"brouille", "brouill"},
		{"brouillé", "brouill"},
		{"brouillée", "brouill"},
		{"brouiller", "brouill"},
		{"brouillerie", "brouiller"},
		{"brouilleriez", "brouill"},
		{"brouillerons", "brouill"},
		{"brouilles", "brouill"},
		{"brouillon", "brouillon"},
		{"brouillons", "brouillon"},
		{"broussailles", "broussaill"},
		{"bruissement", "bruissement"},
		{"bruit", "bruit"},
		{"bruits", "bruit"},
		{"brûla", "brûl"},
		{"brûlaient", "brûl"},
		{"brûlait", "brûl"},
		{"brûlant", "brûl"},
		{"brûlante", "brûl"},
		{"brûlantes", "brûl"},
		{"brûle", "brûl"},
		{"brûlé", "brûl"},
		{"brûlée", "brûl"},
		{"brûlées", "brûl"},
		{"brûler", "brûl"},
		{"brûlerait", "brûl"},
		{"brûlerez", "brûl"},
		{"brûlés", "brûl"},
		{"brûlons", "brûlon"},
		{"brûlure", "brûlur"},
		{"brume", "brum"},
		{"brumes", "brum"},
		{"brun", "brun"},
		{"brune", "brun"},
		{"bruno", "bruno"},
		{"bruns", "brun"},
		{"brusque", "brusqu"},
		{"brusquement", "brusqu"},
		{"brusquer", "brusqu"},
		{"brusques", "brusqu"},
		{"brut", "brut"},
		{"brutal", "brutal"},
		{"brutale", "brutal"},
		{"brutalement", "brutal"},
		{"brutalité", "brutal"},
		{"brutes", "brut"},
		{"brutus", "brutus"},
		{"bruxelles", "bruxel"},
		{"bruyamment", "brui"},
		{"bruyant", "brui"},
		{"bruyante", "brui"},
		{"bruyantes", "brui"},
		{"bruyants", "brui"},
		{"bu", "bu"},
		{"bubna", "bubn"},
		{"bûcher", "bûch"},
		{"budget", "budget"},
		{"bue", "bu"},
		{"buffa", "buff"},
		{"buffalos", "buffalos"},
		{"buffet", "buffet"},
		{"buffles", "buffl"},
		{"buis", "buis"},
		{"buissons", "buisson"},
		{"buli", "bul"},
		{"bull", "bull"},
		{"bulle", "bull"},
		{"bulletin", "bulletin"},
		{"bulletins", "bulletin"},
		{"bulo", "bulo"},
		{"bundelkund", "bundelkund"},
		{"bungalow", "bungalow"},
		{"bungalows", "bungalow"},
		{"bunhyas", "bunhi"},
		{"bunsby", "bunsby"},
		{"buonaparte", "buonapart"},
		{"buonaparté", "buonapart"},
		{"burati", "burat"},
		{"burdivan", "burdivan"},
		{"burdwan", "burdwan"},
		{"bureau", "bureau"},
		{"bureaucratique", "bureaucrat"},
		{"bureaux", "bureau"},
		{"burhampour", "burhampour"},
		{"burlesque", "burlesqu"},
		{"burlington", "burlington"},
		{"burning", "burning"},
		{"busqué", "busqu"},
		{"buste", "bust"},
		{"bustos", "bustos"},
		{"but", "but"},
		{"butcher", "butch"},
		{"butte", "butt"},
		{"buvait", "buv"},
		{"buvez", "buv"},
		{"buvions", "buvion"},
		{"buvons", "buvon"},
		{"buxar", "buxar"},
		{"by", "by"},
		{"byron", "byron"},
		{"c", "c"},
		{"ça", "ça"},
		{"çà", "çà"},
		{"cab", "cab"},
		{"cabale", "cabal"},
		{"cabane", "caban"},
		{"cabanes", "caban"},
		{"cabaret", "cabaret"},
		{"cabarets", "cabaret"},
		{"cabine", "cabin"},
		{"cabines", "cabin"},
		{"cabinet", "cabinet"},
		{"cabinets", "cabinet"},
		{"cabotage", "cabotag"},
		{"cabra", "cabr"},
		{"cabrait", "cabr"},
		{"cabrer", "cabr"},
		{"cabriolet", "cabriolet"},
		{"cacha", "cach"},
		{"cachaient", "cach"},
		{"cachais", "cach"},
		{"cachait", "cach"},
		{"cachant", "cach"},
		{"cache", "cach"},
		{"caché", "cach"},
		{"cachée", "cach"},
		{"cachées", "cach"},
		{"cachemire", "cachemir"},
		{"cachent", "cachent"},
		{"cacher", "cach"},
		{"cachera", "cach"},
		{"cacherai", "cach"},
		{"cacherait", "cach"},
		{"cacherons", "cach"},
		{"cachés", "cach"},
		{"cachet", "cachet"},
		{"cacheté", "cachet"},
		{"cacheter", "cachet"},
		{"cachette", "cachet"},
		{"cachettes", "cachet"},
		{"cachons", "cachon"},
		{"cachot", "cachot"},
		{"cachots", "cachot"},
		{"cacolet", "cacolet"},
		{"cacolets", "cacolet"},
		{"cadavre", "cadavr"},
		{"cadavres", "cadavr"},
		{"cadeau", "cadeau"},
		{"cadeaux", "cadeau"},
		{"cadenabia", "cadenabi"},
		{"cadenas", "caden"},
		{"cadet", "cadet"},
		{"cadets", "cadet"},
		{"cadette", "cadet"},
		{"cadran", "cadran"},
		{"cadre", "cadr"},
		{"cadres", "cadr"},
		{"café", "caf"},
		{"caféiers", "caféi"},
		{"cafés", "caf"},
		{"cage", "cag"},
		{"cages", "cag"},
		{"cagnola", "cagnol"},
		{"cahier", "cahi"},
		{"cahutes", "cahut"},
		{"caillé", "caill"},
		{"cailles", "caill"},
		{"cailloux", "cailloux"},
		{"caisse", "caiss"},
		{"caisses", "caiss"},
		{"caissier", "caissi"},
		{"caissons", "caisson"},
		{"cajoleries", "cajoler"},
		{"calais", "cal"},
		{"calcul", "calcul"},
		{"calculait", "calcul"},
		{"calculant", "calcul"},
		{"calculateur", "calcul"},
		{"calculé", "calcul"},
		{"calculée", "calcul"},
		{"calculées", "calcul"},
		{"calculer", "calcul"},
		{"calculs", "calcul"},
		{"calcutta", "calcutt"},
		{"calèche", "calech"},
		{"calembour", "calembour"},
		{"calembours", "calembour"},
		{"calendrier", "calendri"},
		{"californie", "californ"},
		{"californienne", "californien"},
		{"califourchon", "califourchon"},
		{"câline", "câlin"},
		{"call", "call"},
		{"called", "called"},
		{"callyan", "callyan"},
		{"calm", "calm"},
		{"calma", "calm"},
		{"calme", "calm"},
		{"calmer", "calm"},
		{"calmerai", "calm"},
		{"calmèrent", "calm"},
		{"calmes", "calm"},
		{"calmez", "calm"},
		{"calomnie", "calomn"},
		{"calomnié", "calomni"},
		{"calomniée", "calomni"},
		{"calomniées", "calomni"},
		{"calomnier", "calomni"},
		{"calomnieront", "calomni"},
		{"calomnies", "calomn"},
		{"calomnieuses", "calomni"},
		{"calotin", "calotin"},
		{"calotte", "calott"},
		{"camarade", "camarad"},
		{"camarades", "camarad"},
		{"camarilla", "camarill"},
		{"cambaye", "cambay"},
		{"cambodge", "cambodg"},
		{"cambrai", "cambr"},
		{"cambrure", "cambrur"},
		{"camélias", "caméli"},
		{"cameriere", "camerier"},
		{"camériste", "camer"},
		{"camp", "camp"},
		{"campagnards", "campagnard"},
		{"campagne", "campagn"},
		{"campagnes", "campagn"},
		{"campaient", "camp"},
		{"campé", "camp"},
		{"can", "can"},
		{"canaille", "canaill"},
		{"canal", "canal"},
		{"canapé", "canap"},
		{"canards", "canard"},
		{"cancer", "canc"},
		{"candeur", "candeur"},
		{"candidat", "candidat"},
		{"candidats", "candidat"},
		{"candidature", "candidatur"},
		{"candide", "candid"},
		{"cangos", "cangos"},
		{"cani", "can"},
		{"caniches", "canich"},
		{"canif", "canif"},
		{"canne", "cann"},
		{"cannelle", "cannel"},
		{"cannes", "cann"},
		{"canon", "canon"},
		{"canonique", "canon"},
		{"canonnade", "canonnad"},
		{"canonniers", "canonni"},
		{"canons", "canon"},
		{"canot", "canot"},
		{"canots", "canot"},
		{"canova", "canov"},
		{"canta", "cant"},
		{"cantate", "cantat"},
		{"cantatrice", "cantatric"},
		{"cantilène", "cantilen"},
		{"cantinière", "cantini"},
		{"cantiques", "cantiqu"},
		{"canton", "canton"},
		{"cantonnées", "canton"},
		{"cantons", "canton"},
		{"cap", "cap"},
		{"capable", "capabl"},
		{"capacité", "capac"},
		{"caparaçonnés", "caparaçon"},
		{"cape", "cap"},
		{"capharnaüm", "capharnaüm"},
		{"capillaire", "capillair"},
		{"capitaine", "capitain"},
		{"capitaines", "capitain"},
		{"capital", "capital"},
		{"capitale", "capital"},
		{"capitani", "capitan"},
		{"capitulation", "capitul"},
		{"capon", "capon"},
		{"caporal", "caporal"},
		{"capot", "capot"},
		{"caprice", "capric"},
		{"caprices", "capric"},
		{"capricieuse", "caprici"},
		{"capricieusement", "caprici"},
		{"capricieuses", "caprici"},
		{"captivait", "captiv"},
		{"captivité", "captiv"},
		{"capture", "captur"},
		{"capuche", "capuch"},
		{"capuchons", "capuchon"},
		{"capucin", "capucin"},
		{"car", "car"},
		{"carabine", "carabin"},
		{"carabines", "carabin"},
		{"carabiniers", "carabini"},
		{"caractère", "caracter"},
		{"caractères", "caracter"},
		{"caractérisait", "caractéris"},
		{"caractérise", "caractéris"},
		{"caractéristique", "caractérist"},
		{"carafe", "caraf"},
		{"carbonaro", "carbonaro"},
		{"carburé", "carbur"},
		{"carcasse", "carc"},
		{"cardif", "cardif"},
		{"cardinal", "cardinal"},
		{"cardinaux", "cardinal"},
		{"caressa", "caress"},
		{"caressait", "caress"},
		{"caressant", "caress"},
		{"caressants", "caress"},
		{"caresse", "caress"},
		{"caressent", "caressent"},
		{"caresser", "caress"},
		{"caresseraient", "caress"},
		{"caresses", "caress"},
		{"cargaison", "cargaison"},
		{"caricature", "caricatur"},
		{"carlino", "carlino"},
		{"carlo", "carlo"},
		{"carlone", "carlon"},
		{"carnassiers", "carnassi"},
		{"carnatic", "carnatic"},
		{"carnaval", "carnaval"},
		{"carnet", "carnet"},
		{"carnot", "carnot"},
		{"caro", "caro"},
		{"caroline", "carolin"},
		{"caron", "caron"},
		{"caroubiers", "caroubi"},
		{"carpe", "carp"},
		{"carré", "carr"},
		{"carreau", "carreau"},
		{"carreaux", "carreau"},
		{"carrée", "carr"},
		{"carrefours", "carrefour"},
		{"carrément", "carr"},
		{"carrés", "carr"},
		{"carrière", "carri"},
		{"carrières", "carri"},
		{"carrosse", "carross"},
		{"carrosses", "carross"},
		{"cars", "car"},
		{"carson", "carson"},
		{"carta", "cart"},
		{"carte", "cart"},
		{"cartes", "cart"},
		{"carthage", "carthag"},
		{"carton", "carton"},
		{"cartons", "carton"},
		{"cartouche", "cartouch"},
		{"cartouches", "cartouch"},
		{"cas", "cas"},
		{"casa", "cas"},
		{"casal", "casal"},
		{"casanier", "casani"},
		{"cascade", "cascad"},
		{"cascades", "cascad"},
		{"case", "cas"},
		{"caserne", "casern"},
		{"cases", "cas"},
		{"casi", "cas"},
		{"casimir", "casim"},
		{"casino", "casino"},
		{"casque", "casqu"},
		{"casques", "casqu"},
		{"cassa", "cass"},
		{"cassait", "cass"},
		{"cassandre", "cassandr"},
		{"cassano", "cassano"},
		{"cassât", "cass"},
		{"casse", "cass"},
		{"cassé", "cass"},
		{"cassée", "cass"},
		{"cassées", "cass"},
		{"cassent", "cassent"},
		{"casser", "cass"},
		{"casserais", "cass"},
		{"casseroles", "casserol"},
		{"cassette", "casset"},
		{"cassine", "cassin"},
		{"castanède", "castaned"},
		{"caste", "cast"},
		{"castelnovo", "castelnovo"},
		{"casti", "cast"},
		{"casto", "casto"},
		{"castorine", "castorin"},
		{"castres", "castr"},
		{"castries", "castr"},
		{"casuel", "casuel"},
		{"casuistes", "casuist"},
		{"catanzara", "catanzar"},
		{"catastrophe", "catastroph"},
		{"catéchisme", "catech"},
		{"catena", "caten"},
		{"cathédrale", "cathédral"},
		{"cathelineau", "cathelineau"},
		{"catherine", "catherin"},
		{"cattaro", "cattaro"},
		{"cauchemar", "cauchemar"},
		{"cauchemars", "cauchemar"},
		{"caudataire", "caudatair"},
		{"causa", "caus"},
		{"causait", "caus"},
		{"causant", "caus"},
		{"cause", "caus"},
		{"causé", "caus"},
		{"causée", "caus"},
		{"causer", "caus"},
		{"causerait", "caus"},
		{"causèrent", "caus"},
		{"causes", "caus"},
		{"causés", "caus"},
		{"causeur", "causeur"},
		{"caustique", "caustiqu"},
		{"cauteleuse", "cautel"},
		{"cauteleux", "cautel"},
		{"caution", "caution"},
		{"cautions", "caution"},
		{"cavalerie", "cavaler"},
		{"cavalier", "cavali"},
		{"cavaliers", "cavali"},
		{"cavallo", "cavallo"},
		{"cave", "cav"},
		{"caverne", "cavern"},
		{"caves", "cav"},
		{"cavi", "cav"},
		{"caylus", "caylus"},
		{"ce", "ce"},
		{"ceci", "cec"},
		{"céda", "ced"},
		{"cédait", "ced"},
		{"cédant", "ced"},
		{"cédar", "cédar"},
		{"cède", "ced"},
		{"cédé", "ced"},
		{"céder", "ced"},
		{"cédera", "ced"},
		{"céderons", "ced"},
		{"cédez", "ced"},
		{"cèdre", "cedr"},
		{"cèdres", "cedr"},
		{"ceignait", "ceign"},
		{"ceinture", "ceintur"},
		{"cela", "cel"},
		{"célébra", "célebr"},
		{"célébraient", "célebr"},
		{"célébrait", "célebr"},
		{"célèbre", "célebr"},
		{"célébré", "célebr"},
		{"célébrée", "célebr"},
		{"célébrèrent", "célebr"},
		{"célèbres", "célebr"},
		{"célébrés", "célebr"},
		{"céleste", "célest"},
		{"célibataires", "célibatair"},
		{"cella", "cel"},
		{"celle", "cel"},
		{"celles", "cel"},
		{"cellier", "celli"},
		{"cellule", "cellul"},
		{"celui", "celui"},
		{"cendré", "cendr"},
		{"cendres", "cendr"},
		{"cendrés", "cendr"},
		{"cenis", "cen"},
		{"censure", "censur"},
		{"cent", "cent"},
		{"centaine", "centain"},
		{"centaines", "centain"},
		{"centième", "centiem"},
		{"centièmes", "centiem"},
		{"centime", "centim"},
		{"centimes", "centim"},
		{"central", "central"},
		{"centrale", "central"},
		{"centre", "centr"},
		{"centres", "centr"},
		{"cents", "cent"},
		{"centuplais", "centupl"},
		{"centuplait", "centupl"},
		{"centuplée", "centupl"},
		{"cependant", "cepend"},
		{"cercle", "cercl"},
		{"cercles", "cercl"},
		{"cérémonie", "cérémon"},
		{"cérémonies", "cérémon"},
		{"cérémonieuse", "cérémoni"},
		{"cérès", "céres"},
		{"cerf", "cerf"},
		{"cerisiers", "cerisi"},
		{"certain", "certain"},
		{"certaine", "certain"},
		{"certainement", "certain"},
		{"certaines", "certain"},
		{"certains", "certain"},
		{"certes", "cert"},
		{"certi", "cert"},
		{"certificat", "certificat"},
		{"certificats", "certificat"},
		{"certifiera", "certifi"},
		{"certitude", "certitud"},
		{"certitudes", "certitud"},
		{"cerveau", "cerveau"},
		{"cervelle", "cervel"},
		{"ces", "ce"},
		{"césar", "césar"},
		{"cesare", "cesar"},
		{"cessa", "cess"},
		{"cessaient", "cess"},
		{"cessait", "cess"},
		{"cessant", "cess"},
		{"cessante", "cess"},
		{"cessation", "cessat"},
		{"cesse", "cess"},
		{"cessé", "cess"},
		{"cesser", "cess"},
		{"cessera", "cess"},
		{"cesserais", "cess"},
		{"cesseras", "cess"},
		{"cessèrent", "cess"},
		{"cesseront", "cess"},
		{"cessiez", "cess"},
		{"cessions", "cession"},
		{"cet", "cet"},
		{"cette", "cet"},
		{"ceux", "ceux"},
		{"ceylan", "ceylan"},
		{"ceylandais", "ceyland"},
		{"chacun", "chacun"},
		{"chacune", "chacun"},
		{"chagrin", "chagrin"},
		{"chagrine", "chagrin"},
		{"chagriner", "chagrin"},
		{"chagrins", "chagrin"},
		{"chaîne", "chaîn"},
		{"chaînes", "chaîn"},
		{"chaînon", "chaînon"},
		{"chaînons", "chaînon"},
		{"chair", "chair"},
		{"chaire", "chair"},
		{"chaise", "chais"},
		{"chaises", "chais"},
		{"chalands", "chaland"},
		{"châle", "châl"},
		{"chaleur", "chaleur"},
		{"chaleurs", "chaleur"},
		{"châlon", "châlon"},
		{"châlons", "châlon"},
		{"chalvet", "chalvet"},
		{"chamarrées", "chamarr"},
		{"chamarrés", "chamarr"},
		{"chambellan", "chambellan"},
		{"chambellans", "chambellan"},
		{"chambranle", "chambranl"},
		{"chambre", "chambr"},
		{"chambres", "chambr"},
		{"chambrette", "chambret"},
		{"chameau", "chameau"},
		{"champ", "champ"},
		{"champagne", "champagn"},
		{"champêtre", "champêtr"},
		{"champions", "champion"},
		{"champs", "champ"},
		{"chance", "chanc"},
		{"chancelant", "chancel"},
		{"chancelier", "chanceli"},
		{"chancellerie", "chanceller"},
		{"chances", "chanc"},
		{"chanceux", "chanceux"},
		{"chandelle", "chandel"},
		{"chandernagor", "chandernagor"},
		{"change", "chang"},
		{"changé", "chang"},
		{"changea", "chang"},
		{"changeaient", "chang"},
		{"changeait", "chang"},
		{"changeant", "chang"},
		{"changeante", "chang"},
		{"changeants", "chang"},
		{"changée", "chang"},
		{"changement", "chang"},
		{"changements", "chang"},
		{"changent", "changent"},
		{"changer", "chang"},
		{"changerait", "chang"},
		{"changèrent", "chang"},
		{"changeront", "chang"},
		{"changés", "chang"},
		{"chanoine", "chanoin"},
		{"chanoines", "chanoin"},
		{"chanoinesse", "chanoiness"},
		{"chanson", "chanson"},
		{"chansons", "chanson"},
		{"chant", "chant"},
		{"chanta", "chant"},
		{"chantai", "chant"},
		{"chantaient", "chant"},
		{"chantait", "chant"},
		{"chantant", "chant"},
		{"chante", "chant"},
		{"chanté", "chant"},
		{"chantée", "chant"},
		{"chanter", "chant"},
		{"chantèrent", "chant"},
		{"chantés", "chant"},
		{"chanteur", "chanteur"},
		{"chanteuses", "chanteux"},
		{"chantiez", "chant"},
		{"chantonnant", "chanton"},
		{"chants", "chant"},
		{"chanvre", "chanvr"},
		{"chaos", "chaos"},
		{"chap", "chap"},
		{"chapeau", "chapeau"},
		{"chapeaux", "chapeau"},
		{"chapelet", "chapelet"},
		{"chapelle", "chapel"},
		{"chapes", "chap"},
		{"chapitre", "chapitr"},
		{"chapitres", "chapitr"},
		{"chapons", "chapon"},
		{"chaque", "chaqu"},
		{"char", "char"},
		{"charade", "charad"},
		{"charbon", "charbon"},
		{"charbonnier", "charbonni"},
		{"charcot", "charcot"},
		{"charge", "charg"},
		{"chargé", "charg"},
		{"chargea", "charg"},
		{"chargeaient", "charg"},
		{"chargeait", "charg"},
		{"chargeant", "charg"},
		{"chargée", "charg"},
		{"chargées", "charg"},
		{"chargement", "charg"},
		{"charger", "charg"},
		{"chargera", "charg"},
		{"chargerais", "charg"},
		{"chargerait", "charg"},
		{"chargèrent", "charg"},
		{"charges", "charg"},
		{"chargés", "charg"},
		{"chargez", "charg"},
		{"charing", "charing"},
		{"charitable", "charit"},
		{"charité", "charit"},
		{"charlatanisme", "charlatan"},
		{"charlatans", "charlatan"},
		{"charlemagne", "charlemagn"},
		{"charleroi", "charleroi"},
		{"charles", "charl"},
		{"charlotte", "charlott"},
		{"charma", "charm"},
		{"charmant", "charm"},
		{"charmante", "charm"},
		{"charmantes", "charm"},
		{"charmants", "charm"},
		{"charme", "charm"},
		{"charmé", "charm"},
		{"charmée", "charm"},
		{"charmer", "charm"},
		{"charmes", "charm"},
		{"charmier", "charmi"},
		{"charpente", "charpent"},
		{"charpentier", "charpenti"},
		{"charrette", "charret"},
		{"charrettes", "charret"},
		{"charron", "charron"},
		{"charrues", "charru"},
		{"chars", "char"},
		{"charte", "chart"},
		{"chartreuse", "chartreux"},
		{"chartreux", "chartreux"},
		{"chas", "chas"},
		{"chassa", "chass"},
		{"chassaient", "chass"},
		{"chassait", "chass"},
		{"chassant", "chass"},
		{"chasse", "chass"},
		{"châsse", "châss"},
		{"chassé", "chass"},
		{"chassée", "chass"},
		{"chassent", "chassent"},
		{"chasser", "chass"},
		{"chassera", "chass"},
		{"chassés", "chass"},
		{"chasseur", "chasseur"},
		{"chasseurs", "chasseur"},
		{"chassez", "chass"},
		{"châssis", "châss"},
		{"chasuble", "chasubl"},
		{"chasubles", "chasubl"},
		{"chat", "chat"},
		{"châtaignes", "châtaign"},
		{"châtaignier", "châtaigni"},
		{"châtaigniers", "châtaigni"},
		{"châtain", "châtain"},
		{"châtains", "châtain"},
		{"château", "château"},
		{"châteaux", "château"},
		{"châtier", "châti"},
		{"châtiment", "chât"},
		{"chatouillement", "chatouill"},
		{"chatouilleuse", "chatouill"},
		{"chats", "chat"},
		{"chaud", "chaud"},
		{"chaude", "chaud"},
		{"chaudement", "chaud"},
		{"chaudes", "chaud"},
		{"chaudière", "chaudi"},
		{"chaudières", "chaudi"},
		{"chauffage", "chauffag"},
		{"chauffaient", "chauff"},
		{"chauffé", "chauff"},
		{"chauffée", "chauff"},
		{"chauffer", "chauff"},
		{"chauffeur", "chauffeur"},
		{"chauffeurs", "chauffeur"},
		{"chauffez", "chauff"},
		{"chauffons", "chauffon"},
		{"chaulnes", "chauln"},
		{"chaume", "chaum"},
		{"chaumière", "chaumi"},
		{"chaumières", "chaumi"},
		{"chaussa", "chauss"},
		{"chaussé", "chauss"},
		{"chaussée", "chauss"},
		{"chaussés", "chauss"},
		{"chaussettes", "chausset"},
		{"chaussure", "chaussur"},
		{"chaussures", "chaussur"},
		{"chauve", "chauv"},
		{"chaux", "chaux"},
		{"chavira", "chav"},
		{"chavirer", "chavir"},
		{"chazel", "chazel"},
		{"che", "che"},
		{"chef", "chef"},
		{"chefs", "chef"},
		{"chekina", "chekin"},
		{"chékina", "chékin"},
		{"chélan", "chélan"},
		{"chelem", "chelem"},
		{"chelles", "chel"},
		{"chemin", "chemin"},
		{"cheminant", "chemin"},
		{"cheminée", "chemin"},
		{"cheminées", "chemin"},
		{"chemins", "chemin"},
		{"chemise", "chemis"},
		{"chemises", "chemis"},
		{"chêne", "chên"},
		{"chênes", "chên"},
		{"chènevière", "chènevi"},
		{"chenu", "chenu"},
		{"chèque", "chequ"},
		{"chèques", "chequ"},
		{"cher", "cher"},
		{"chercha", "cherch"},
		{"cherchaient", "cherch"},
		{"cherchais", "cherch"},
		{"cherchait", "cherch"},
		{"cherchant", "cherch"},
		{"cherche", "cherch"},
		{"cherché", "cherch"},
		{"cherchée", "cherch"},
		{"cherchent", "cherchent"},
		{"chercher", "cherch"},
		{"cherchera", "cherch"},
		{"chercherai", "cherch"},
		{"chercherait", "cherch"},
		{"cherchèrent", "cherch"},
		{"chercherez", "cherch"},
		{"cherches", "cherch"},
		{"cherchez", "cherch"},
		{"cherchons", "cherchon"},
		{"chère", "cher"},
		{"chèrement", "cher"},
		{"chères", "cher"},
		{"chéri", "cher"},
		{"chérie", "cher"},
		{"chérir", "cher"},
		{"chéris", "cher"},
		{"chers", "cher"},
		{"chester", "chest"},
		{"chétif", "chétif"},
		{"chétifs", "chétif"},
		{"chétive", "chétiv"},
		{"cheval", "cheval"},
		{"chevaleresque", "chevaleresqu"},
		{"chevalerie", "chevaler"},
		{"chevalier", "chevali"},
		{"chevaliers", "chevali"},
		{"chevaux", "cheval"},
		{"chevelure", "chevelur"},
		{"chevet", "chevet"},
		{"cheveu", "cheveu"},
		{"cheveux", "cheveux"},
		{"chèvrefeuilles", "chèvrefeuill"},
		{"chèvres", "chevr"},
		{"chevreuse", "chevreux"},
		{"cheyenne", "cheyen"},
		{"chez", "chez"},
		{"chiaramonti", "chiaramont"},
		{"chicago", "chicago"},
		{"chicorée", "chicor"},
		{"chien", "chien"},
		{"chiens", "chien"},
		{"chiffonna", "chiffon"},
		{"chiffonnait", "chiffon"},
		{"chiffons", "chiffon"},
		{"chiffre", "chiffr"},
		{"chiffrée", "chiffr"},
		{"chiffrées", "chiffr"},
		{"chiffrer", "chiffr"},
		{"chiffres", "chiffr"},
		{"chili", "chil"},
		{"chimère", "chimer"},
		{"chimères", "chimer"},
		{"chimériques", "chimer"},
		{"china", "chin"},
		{"chine", "chin"},
		{"chinois", "chinois"},
		{"chinoise", "chinois"},
		{"chinoises", "chinois"},
		{"chiper", "chip"},
		{"chiperait", "chip"},
		{"chirurgicales", "chirurgical"},
		{"chirurgie", "chirurg"},
		{"chirurgien", "chirurgien"},
		{"chlemm", "chlemm"},
		{"choc", "choc"},
		{"chocolat", "chocolat"},
		{"choeur", "choeur"},
		{"choeurs", "choeur"},
		{"choisi", "chois"},
		{"choisie", "chois"},
		{"choisies", "chois"},
		{"choisir", "chois"},
		{"choisirai", "chois"},
		{"choisirez", "chois"},
		{"choisis", "chois"},
		{"choisissait", "chois"},
		{"choisissez", "chois"},
		{"choisit", "chois"},
		{"choix", "choix"},
		{"cholin", "cholin"},
		{"choqua", "choqu"},
		{"choquait", "choqu"},
		{"choquante", "choqu"},
		{"choque", "choqu"},
		{"choqué", "choqu"},
		{"choquée", "choqu"},
		{"choquer", "choqu"},
		{"choquera", "choqu"},
		{"choqueraient", "choqu"},
		{"choquèrent", "choqu"},
		{"choqués", "choqu"},
		{"chorus", "chorus"},
		{"chose", "chos"},
		{"choses", "chos"},
		{"chouart", "chouart"},
		{"choucroute", "choucrout"},
		{"choyé", "choi"},
		{"chrétien", "chrétien"},
		{"chrétienne", "chrétien"},
		{"chrétiennes", "chrétien"},
		{"chrétiens", "chrétien"},
		{"christ", "christ"},
		{"christianisme", "christian"},
		{"chromatique", "chromat"},
		{"chronicle", "chronicl"},
		{"chronique", "chroniqu"},
		{"chronomètre", "chronometr"},
		{"chronomètres", "chronometr"},
		{"chronométriquement", "chronometr"},
		{"chunar", "chunar"},
		{"chut", "chut"},
		{"chute", "chut"},
		{"chuté", "chut"},
		{"chutes", "chut"},
		{"ci", "ci"},
		{"ciceri", "cicer"},
		{"cicéron", "cicéron"},
		{"cicéronien", "cicéronien"},
		{"ciel", "ciel"},
		{"cierge", "cierg"},
		{"cierges", "cierg"},
		{"cieux", "cieux"},
		{"cigales", "cigal"},
		{"cigare", "cigar"},
		{"cigares", "cigar"},
		{"ciguë", "cigu"},
		{"cils", "cil"},
		{"cimabué", "cimabu"},
		{"cimarosa", "cimaros"},
		{"cime", "cim"},
		{"cimes", "cim"},
		{"cimetière", "cimeti"},
		{"cinnamome", "cinnamom"},
		{"cinq", "cinq"},
		{"cinquantaine", "cinquantain"},
		{"cinquante", "cinqu"},
		{"cinquantième", "cinquantiem"},
		{"cinquième", "cinquiem"},
		{"cipayes", "cipay"},
		{"circé", "circ"},
		{"circonférence", "circonférent"},
		{"circonspection", "circonspect"},
		{"circonstance", "circonst"},
		{"circonstances", "circonst"},
		{"circonstancié", "circonstanci"},
		{"circonvenir", "circonven"},
		{"circula", "circul"},
		{"circulaient", "circul"},
		{"circulaire", "circulair"},
		{"circulairement", "circulair"},
		{"circulaires", "circulair"},
		{"circulant", "circul"},
		{"circulante", "circul"},
		{"circulation", "circul"},
		{"circuler", "circul"},
		{"circulerait", "circul"},
		{"circumnavigation", "circumnavig"},
		{"cire", "cir"},
		{"cirque", "cirqu"},
		{"cirques", "cirqu"},
		{"cisalpin", "cisalpin"},
		{"cisalpine", "cisalpin"},
		{"cisco", "cisco"},
		{"ciseaux", "ciseau"},
		{"cita", "cit"},
		{"citadelle", "citadel"},
		{"citaient", "cit"},
		{"citait", "cit"},
		{"citant", "cit"},
		{"citation", "citat"},
		{"citations", "citat"},
		{"cite", "cit"},
		{"cité", "cit"},
		{"citée", "cit"},
		{"citer", "cit"},
		{"citernes", "citern"},
		{"cités", "cit"},
		{"citoyen", "citoyen"},
		{"citoyennes", "citoyen"},
		{"citoyens", "citoyen"},
		{"citronnier", "citronni"},
		{"city", "city"},
		{"civil", "civil"},
		{"civile", "civil"},
		{"civilisation", "civilis"},
		{"civilisées", "civilis"},
		{"civilisés", "civilis"},
		{"civils", "civil"},
		{"clair", "clair"},
		{"claire", "clair"},
		{"clairement", "clair"},
		{"claires", "clair"},
		{"clairière", "clairi"},
		{"clairs", "clair"},
		{"clairvoyance", "clairvoi"},
		{"clairvoyant", "clairvoi"},
		{"clairvoyante", "clairvoi"},
		{"clairvoyants", "clairvoi"},
		{"clameurs", "clameur"},
		{"clandestin", "clandestin"},
		{"clara", "clar"},
		{"claret", "claret"},
		{"clarté", "clart"},
		{"classait", "class"},
		{"classe", "class"},
		{"classé", "class"},
		{"classes", "class"},
		{"classés", "class"},
		{"clause", "claus"},
		{"clayonnage", "clayonnag"},
		{"clef", "clef"},
		{"clefs", "clef"},
		{"clélia", "cléli"},
		{"clémence", "clémenc"},
		{"clément", "clément"},
		{"clerc", "clerc"},
		{"clercs", "clerc"},
		{"clergé", "clerg"},
		{"clignant", "clign"},
		{"climat", "climat"},
		{"clin", "clin"},
		{"clinquant", "clinqu"},
		{"clippers", "clipper"},
		{"clique", "cliqu"},
		{"cliquettes", "cliquet"},
		{"cloaque", "cloaqu"},
		{"cloche", "cloch"},
		{"clocher", "cloch"},
		{"clochers", "clocher"},
		{"cloches", "cloch"},
		{"clochettes", "clochet"},
		{"cloître", "cloîtr"},
		{"clopant", "clop"},
		{"clopin", "clopin"},
		{"clore", "clor"},
		{"clos", "clos"},
		{"close", "clos"},
		{"closes", "clos"},
		{"clôture", "clôtur"},
		{"clou", "clou"},
		{"cloua", "clou"},
		{"cloud", "cloud"},
		{"cloué", "clou"},
		{"clouée", "clou"},
		{"clous", "clous"},
		{"cloutier", "clouti"},
		{"clown", "clown"},
		{"clowns", "clown"},
		{"club", "club"},
		{"co", "co"},
		{"coach", "coach"},
		{"coaches", "coach"},
		{"coadjuteur", "coadjuteur"},
		{"coblentz", "coblentz"},
		{"cocarde", "cocard"},
		{"cocher", "coch"},
		{"cochère", "cocher"},
		{"cochers", "cocher"},
		{"cochinchine", "cochinchin"},
		{"coconasso", "coconasso"},
		{"cocotiers", "cocoti"},
		{"cocotte", "cocott"},
		{"code", "cod"},
		{"coeur", "coeur"},
		{"coeurs", "coeur"},
		{"coffre", "coffr"},
		{"coffré", "coffr"},
		{"coffres", "coffr"},
		{"cohue", "cohu"},
		{"coidet", "coidet"},
		{"coiffa", "coiff"},
		{"coiffé", "coiff"},
		{"coiffée", "coiff"},
		{"coiffés", "coiff"},
		{"coiffure", "coiffur"},
		{"coin", "coin"},
		{"coïncidence", "coïncident"},
		{"coins", "coin"},
		{"col", "col"},
		{"cola", "col"},
		{"colaba", "colab"},
		{"coldness", "coldness"},
		{"colère", "coler"},
		{"colères", "coler"},
		{"colfax", "colfax"},
		{"colifichet", "colifichet"},
		{"colimaçon", "colimaçon"},
		{"colique", "coliqu"},
		{"colis", "colis"},
		{"collaborateurs", "collabor"},
		{"collait", "coll"},
		{"collants", "coll"},
		{"colle", "coll"},
		{"collé", "coll"},
		{"collecteur", "collecteur"},
		{"collecteurs", "collecteur"},
		{"collection", "collect"},
		{"collège", "colleg"},
		{"collègue", "collègu"},
		{"collègues", "collègu"},
		{"coller", "coll"},
		{"collés", "coll"},
		{"collet", "collet"},
		{"collier", "colli"},
		{"colliers", "colli"},
		{"colline", "collin"},
		{"collines", "collin"},
		{"colloque", "colloqu"},
		{"colmar", "colmar"},
		{"colombes", "colomb"},
		{"colombier", "colombi"},
		{"colonel", "colonel"},
		{"colonels", "colonel"},
		{"colonie", "colon"},
		{"colonisateur", "colonis"},
		{"coloniser", "colonis"},
		{"colonnade", "colonnad"},
		{"colonne", "colon"},
		{"colonnes", "colon"},
		{"colorado", "colorado"},
		{"coloré", "color"},
		{"colorée", "color"},
		{"colorié", "colori"},
		{"coloris", "color"},
		{"colorno", "colorno"},
		{"colossal", "colossal"},
		{"colossale", "colossal"},
		{"colossales", "colossal"},
		{"colpa", "colp"},
		{"colt", "colt"},
		{"columbus", "columbus"},
		{"combat", "combat"},
		{"combats", "combat"},
		{"combattaient", "combatt"},
		{"combattants", "combatt"},
		{"combattis", "combatt"},
		{"combattit", "combatt"},
		{"combattre", "combattr"},
		{"combattu", "combattu"},
		{"combattue", "combattu"},
		{"combien", "combien"},
		{"combina", "combin"},
		{"combinaisons", "combinaison"},
		{"combinant", "combin"},
		{"combiné", "combin"},
		{"combinés", "combin"},
		{"comble", "combl"},
		{"comblé", "combl"},
		{"combler", "combl"},
		{"comblerait", "combl"},
		{"combles", "combl"},
		{"combustible", "combustibl"},
		{"come", "com"},
		{"côme", "côm"},
		{"comédie", "comed"},
		{"comédien", "comédien"},
		{"comédiens", "comédien"},
		{"comédies", "comed"},
		{"comestible", "comestibl"},
		{"comestibles", "comestibl"},
		{"comique", "comiqu"},
		{"comiques", "comiqu"},
		{"comité", "comit"},
		{"commanda", "command"},
		{"commandaient", "command"},
		{"commandait", "command"},
		{"commandant", "command"},
		{"commande", "command"},
		{"commandé", "command"},
		{"commandée", "command"},
		{"commandement", "command"},
		{"commandements", "command"},
		{"commander", "command"},
		{"commandera", "command"},
		{"commanderez", "command"},
		{"commandons", "commandon"},
		{"comme", "comm"},
		{"commença", "commenc"},
		{"commençai", "commenc"},
		{"commençaient", "commenc"},
		{"commençait", "commenc"},
		{"commençant", "commenc"},
		{"commence", "commenc"},
		{"commencé", "commenc"},
		{"commencée", "commenc"},
		{"commencement", "commenc"},
		{"commencements", "commenc"},
		{"commencent", "commencent"},
		{"commencer", "commenc"},
		{"commencera", "commenc"},
		{"commencerai", "commenc"},
		{"commencerait", "commenc"},
		{"commencèrent", "commenc"},
		{"commencés", "commenc"},
		{"commencez", "commenc"},
		{"commençons", "commençon"},
		{"commensal", "commensal"},
		{"comment", "comment"},
		{"commentaire", "commentair"},
		{"commentaires", "commentair"},
		{"commentateurs", "comment"},
		{"commentée", "comment"},
		{"commenter", "comment"},
		{"commentés", "comment"},
		{"commerçante", "commerc"},
		{"commerçantes", "commerc"},
		{"commerçants", "commerc"},
		{"commerce", "commerc"},
		{"commerciale", "commercial"},
		{"commets", "commet"},
		{"commette", "commet"},
		{"commettent", "commettent"},
		{"commettra", "commettr"},
		{"commettrais", "commettr"},
		{"commettre", "commettr"},
		{"commis", "comm"},
		{"commise", "commis"},
		{"commisération", "commiser"},
		{"commissaire", "commissair"},
		{"commission", "commiss"},
		{"commissions", "comm"},
		{"commit", "comm"},
		{"commode", "commod"},
		{"commodément", "commod"},
		{"commotion", "commot"},
		{"commuée", "commu"},
		{"commuer", "commu"},
		{"commuera", "commu"},
		{"commun", "commun"},
		{"communauté", "communaut"},
		{"communaux", "communal"},
		{"commune", "commun"},
		{"communément", "commun"},
		{"communes", "commun"},
		{"communicatif", "commun"},
		{"communication", "commun"},
		{"communications", "commun"},
		{"communicative", "commun"},
		{"communie", "commun"},
		{"communion", "communion"},
		{"communiqua", "communiqu"},
		{"communiquaient", "communiqu"},
		{"communiquait", "communiqu"},
		{"communique", "commun"},
		{"communiqué", "communiqu"},
		{"communiquée", "communiqu"},
		{"communiquer", "communiqu"},
		{"communs", "commun"},
		{"commutation", "commut"},
		{"compagne", "compagn"},
		{"compagnie", "compagn"},
		{"compagnies", "compagn"},
		{"compagnon", "compagnon"},
		{"compagnons", "compagnon"},
		{"compara", "compar"},
		{"comparable", "compar"},
		{"comparaient", "compar"},
		{"comparaison", "comparaison"},
		{"comparait", "compar"},
		{"comparaîtrez", "comparaîtr"},
		{"comparant", "compar"},
		{"comparativement", "compar"},
		{"compare", "compar"},
		{"comparé", "compar"},
		{"comparée", "compar"},
		{"comparer", "compar"},
		{"compartiment", "compart"},
		{"compartiments", "compart"},
		{"compas", "comp"},
		{"compassé", "compass"},
		{"compassées", "compass"},
		{"compatibles", "compatibl"},
		{"compatriote", "compatriot"},
		{"compatriotes", "compatriot"},
		{"compensation", "compens"},
		{"compense", "compens"},
		{"compensé", "compens"},
		{"compenser", "compens"},
		{"compère", "comper"},
		{"complaisait", "complais"},
		{"complaisamment", "complais"},
		{"complaisance", "complais"},
		{"complaisances", "complais"},
		{"complaisant", "complais"},
		{"complaisante", "complais"},
		{"complaisantes", "complais"},
		{"complaisants", "complais"},
		{"complément", "compl"},
		{"complet", "complet"},
		{"complétaient", "complet"},
		{"complétait", "complet"},
		{"complète", "complet"},
		{"complété", "complet"},
		{"complètement", "complet"},
		{"compléter", "complet"},
		{"complètes", "complet"},
		{"complets", "complet"},
		{"complication", "compliqu"},
		{"complice", "complic"},
		{"complices", "complic"},
		{"complicité", "compliqu"},
		{"compliment", "compl"},
		{"complimenté", "compliment"},
		{"complimenteur", "complimenteur"},
		{"complimenteurs", "complimenteur"},
		{"complimenteuse", "compliment"},
		{"compliments", "compl"},
		{"complique", "compliqu"},
		{"compliqué", "compliqu"},
		{"compliquée", "compliqu"},
		{"compliquées", "compliqu"},
		{"compliqués", "compliqu"},
		{"complot", "complot"},
		{"componction", "componct"},
		{"comporta", "comport"},
		{"comportait", "comport"},
		{"comporte", "comport"},
		{"comporté", "comport"},
		{"comportée", "comport"},
		{"composa", "compos"},
		{"composaient", "compos"},
		{"composait", "compos"},
		{"composant", "compos"},
		{"compose", "compos"},
		{"composé", "compos"},
		{"composée", "compos"},
		{"composées", "compos"},
		{"composent", "composent"},
		{"composer", "compos"},
		{"composition", "composit"},
		{"compositions", "composit"},
		{"comprato", "comprato"},
		{"comprenaient", "compren"},
		{"comprenait", "compren"},
		{"comprenant", "compren"},
		{"comprend", "comprend"},
		{"comprendra", "comprendr"},
		{"comprendrai", "comprendr"},
		{"comprendrait", "comprendr"},
		{"comprendre", "comprendr"},
		{"comprendrez", "comprendr"},
		{"comprends", "comprend"},
		{"comprenez", "compren"},
		{"comprenne", "compren"},
		{"comprennent", "comprennent"},
		{"comprimant", "comprim"},
		{"comprimées", "comprim"},
		{"comprimés", "comprim"},
		{"comprirent", "compr"},
		{"compris", "compr"},
		{"comprise", "compris"},
		{"comprit", "compr"},
		{"comprît", "compr"},
		{"compromet", "compromet"},
		{"compromets", "compromet"},
		{"compromettait", "compromet"},
		{"compromettant", "compromet"},
		{"compromettante", "compromet"},
		{"compromettantes", "compromet"},
		{"compromette", "compromet"},
		{"compromettra", "compromettr"},
		{"compromettrais", "compromettr"},
		{"compromettre", "compromettr"},
		{"compromis", "comprom"},
		{"compromise", "compromis"},
		{"compromises", "compromis"},
		{"compta", "compt"},
		{"comptabilité", "comptabl"},
		{"comptaient", "compt"},
		{"comptais", "compt"},
		{"comptait", "compt"},
		{"comptant", "compt"},
		{"comptât", "compt"},
		{"compte", "compt"},
		{"compté", "compt"},
		{"comptées", "compt"},
		{"comptent", "comptent"},
		{"compter", "compt"},
		{"compterai", "compt"},
		{"comptèrent", "compt"},
		{"comptes", "compt"},
		{"comptés", "compt"},
		{"comptez", "compt"},
		{"comptiez", "compt"},
		{"comptoir", "comptoir"},
		{"comptoirs", "comptoir"},
		{"comptons", "compton"},
		{"comte", "comt"},
		{"comté", "comt"},
		{"comtes", "comt"},
		{"comtés", "comt"},
		{"comtesse", "comtess"},
		{"comtois", "comtois"},
		{"comtoise", "comtois"},
		{"concentré", "concentr"},
		{"concentrer", "concentr"},
		{"concentrés", "concentr"},
		{"concernaient", "concern"},
		{"concernant", "concern"},
		{"concerné", "concern"},
		{"concert", "concert"},
		{"concerté", "concert"},
		{"concerts", "concert"},
		{"concession", "concess"},
		{"concessions", "concess"},
		{"concevaient", "concev"},
		{"concevait", "concev"},
		{"concevez", "concev"},
		{"concevoir", "concevoir"},
		{"concierge", "concierg"},
		{"concilié", "concili"},
		{"concilier", "concili"},
		{"concises", "concis"},
		{"concitoyens", "concitoyen"},
		{"conclu", "conclu"},
		{"concluaient", "conclu"},
		{"concluantes", "conclu"},
		{"concluants", "conclu"},
		{"conclue", "conclu"},
		{"conclure", "conclur"},
		{"conclurent", "conclurent"},
		{"conclusion", "conclus"},
		{"conclusions", "conclus"},
		{"conclut", "conclut"},
		{"conçois", "conçois"},
		{"conçoit", "conçoit"},
		{"concordance", "concord"},
		{"concordant", "concord"},
		{"concordat", "concordat"},
		{"concourent", "concourent"},
		{"concours", "concour"},
		{"conçu", "conçu"},
		{"conçue", "conçu"},
		{"concurrent", "concurrent"},
		{"concurrents", "concurrent"},
		{"conçut", "conçut"},
		{"condamnable", "condamn"},
		{"condamnait", "condamn"},
		{"condamnant", "condamn"},
		{"condamnation", "condamn"},
		{"condamne", "condamn"},
		{"condamné", "condamn"},
		{"condamnée", "condamn"},
		{"condamnent", "condamnent"},
		{"condamner", "condamn"},
		{"condamneront", "condamn"},
		{"condamnés", "condamn"},
		{"condé", "cond"},
		{"condescendance", "condescend"},
		{"condillac", "condillac"},
		{"condiments", "cond"},
		{"condition", "condit"},
		{"conditionnée", "condition"},
		{"conditions", "condit"},
		{"condoléance", "condolé"},
		{"conducteur", "conducteur"},
		{"conduira", "conduir"},
		{"conduirait", "conduir"},
		{"conduire", "conduir"},
		{"conduisaient", "conduis"},
		{"conduisait", "conduis"},
		{"conduisant", "conduis"},
		{"conduisent", "conduisent"},
		{"conduisez", "conduis"},
		{"conduisit", "conduis"},
		{"conduit", "conduit"},
		{"conduite", "conduit"},
		{"conduites", "conduit"},
		{"conduits", "conduit"},
		{"confection", "confect"},
		{"confédéré", "conféder"},
		{"conférait", "confer"},
		{"conférence", "conférent"},
		{"conférences", "conférent"},
		{"conférer", "confer"},
		{"confessant", "confess"},
		{"confesser", "confess"},
		{"confesserai", "confess"},
		{"confesseur", "confesseur"},
		{"confesseurs", "confesseur"},
		{"confession", "confess"},
		{"confessionnal", "confessionnal"},
		{"confessionnaux", "confessionnal"},
		{"confessions", "confess"},
		{"confia", "confi"},
		{"confiance", "confianc"},
		{"confidence", "confident"},
		{"confidences", "confident"},
		{"confident", "confident"},
		{"confidente", "confident"},
		{"confié", "confi"},
		{"confiée", "confi"},
		{"confiées", "confi"},
		{"confier", "confi"},
		{"confiés", "confi"},
		{"confiné", "confin"},
		{"confiner", "confin"},
		{"confirma", "confirm"},
		{"confirmait", "confirm"},
		{"confirmée", "confirm"},
		{"confirmer", "confirm"},
		{"confirmèrent", "confirm"},
		{"confiscation", "confisc"},
		{"confisque", "confisqu"},
		{"confisqué", "confisqu"},
		{"confit", "conf"},
		{"confluent", "confluent"},
		{"confondait", "confond"},
		{"confonde", "confond"},
		{"confondent", "confondent"},
		{"confondre", "confondr"},
		{"confondu", "confondu"},
		{"confondus", "confondus"},
		{"conformait", "conform"},
		{"conforme", "conform"},
		{"conformer", "conform"},
		{"confort", "confort"},
		{"confortable", "confort"},
		{"confortablement", "confort"},
		{"confortables", "confort"},
		{"confrère", "confrer"},
		{"confrères", "confrer"},
		{"confrérie", "confrer"},
		{"confucius", "confucius"},
		{"confus", "confus"},
		{"confuse", "confus"},
		{"confusément", "confus"},
		{"confusion", "confus"},
		{"confusions", "confus"},
		{"congé", "cong"},
		{"congédia", "congédi"},
		{"congédiant", "congédi"},
		{"congédié", "congédi"},
		{"congédiés", "congédi"},
		{"congénères", "congéner"},
		{"congestionnée", "congestion"},
		{"congratulation", "congratul"},
		{"congréganiste", "congrégan"},
		{"congréganistes", "congrégan"},
		{"congrégation", "congreg"},
		{"congrès", "congres"},
		{"conjecture", "conjectur"},
		{"conjecturer", "conjectur"},
		{"conjectures", "conjectur"},
		{"conjonctions", "conjonct"},
		{"conjugal", "conjugal"},
		{"conjugale", "conjugal"},
		{"conjuguer", "conjugu"},
		{"conjurait", "conjur"},
		{"conjurant", "conjur"},
		{"conjure", "conjur"},
		{"conjurée", "conjur"},
		{"conjurer", "conjur"},
		{"connais", "con"},
		{"connaissaient", "connaiss"},
		{"connaissais", "connaiss"},
		{"connaissait", "connaiss"},
		{"connaissance", "connaiss"},
		{"connaissances", "connaiss"},
		{"connaissent", "connaissent"},
		{"connaisseurs", "connaisseur"},
		{"connaissez", "connaiss"},
		{"connaissiez", "connaiss"},
		{"connaissons", "connaisson"},
		{"connaît", "connaît"},
		{"connaîtrait", "connaîtr"},
		{"connaitre", "connaitr"},
		{"connaître", "connaîtr"},
		{"connivence", "connivent"},
		{"connu", "connu"},
		{"connue", "connu"},
		{"connues", "connu"},
		{"connus", "connus"},
		{"connut", "connut"},
		{"conquérir", "conquer"},
		{"conquête", "conquêt"},
		{"conquis", "conqu"},
		{"conquise", "conquis"},
		{"conradin", "conradin"},
		{"consacrait", "consacr"},
		{"consacre", "consacr"},
		{"consacré", "consacr"},
		{"consacrée", "consacr"},
		{"consacrées", "consacr"},
		{"consacrer", "consacr"},
		{"consacrerai", "consacr"},
		{"consacrés", "consacr"},
		{"conscience", "conscienc"},
		{"consciencieusement", "conscienci"},
		{"conscription", "conscript"},
		{"conscrit", "conscr"},
		{"conseil", "conseil"},
		{"conseilla", "conseil"},
		{"conseillaient", "conseil"},
		{"conseillait", "conseil"},
		{"conseille", "conseil"},
		{"conseillé", "conseil"},
		{"conseillée", "conseil"},
		{"conseiller", "conseil"},
		{"conseillerai", "conseil"},
		{"conseillerais", "conseil"},
		{"conseillerait", "conseil"},
		{"conseilleriez", "conseil"},
		{"conseillers", "conseiller"},
		{"conseils", "conseil"},
		{"consens", "consen"},
		{"consentait", "consent"},
		{"consente", "consent"},
		{"consentement", "consent"},
		{"consentez", "consent"},
		{"consenti", "consent"},
		{"consentions", "consent"},
		{"consentir", "consent"},
		{"consentirai", "consent"},
		{"consentirais", "consent"},
		{"consentirent", "consent"},
		{"consentit", "consent"},
		{"conséquemment", "conséquent"},
		{"conséquence", "conséquent"},
		{"conséquences", "conséquent"},
		{"conséquent", "conséquent"},
		{"conservait", "conserv"},
		{"conservant", "conserv"},
		{"conservation", "conserv"},
		{"conservatoire", "conservatoir"},
		{"conserve", "conserv"},
		{"conservé", "conserv"},
		{"conservée", "conserv"},
		{"conservées", "conserv"},
		{"conservent", "conservent"},
		{"conserver", "conserv"},
		{"conserverai", "conserv"},
		{"conserves", "conserv"},
		{"considéra", "consider"},
		{"considérable", "consider"},
		{"considérablement", "consider"},
		{"considérables", "consider"},
		{"considérait", "consider"},
		{"considérant", "consider"},
		{"considération", "consider"},
		{"considérations", "consider"},
		{"considère", "consider"},
		{"considéré", "consider"},
		{"considérée", "consider"},
		{"considérées", "consider"},
		{"considérer", "consider"},
		{"considérerais", "consider"},
		{"considérés", "consider"},
		{"consigne", "consign"},
		{"consistait", "consist"},
		{"consistance", "consist"},
		{"consiste", "consist"},
		{"consister", "consist"},
		{"consola", "consol"},
		{"consolaient", "consol"},
		{"consolait", "consol"},
		{"consolation", "consol"},
		{"consolations", "consol"},
		{"console", "consol"},
		{"consolé", "consol"},
		{"consolée", "consol"},
		{"consoler", "consol"},
		{"consomma", "consomm"},
		{"consommateur", "consomm"},
		{"consommateurs", "consomm"},
		{"consommé", "consomm"},
		{"consommée", "consomm"},
		{"consommer", "consomm"},
		{"consommez", "consomm"},
		{"conspirais", "consp"},
		{"conspirante", "conspir"},
		{"conspirateur", "conspir"},
		{"conspirateurs", "conspir"},
		{"conspiration", "conspir"},
		{"conspiré", "conspir"},
		{"conspirer", "conspir"},
		{"conspué", "conspu"},
		{"constamment", "const"},
		{"constance", "constanc"},
		{"constant", "const"},
		{"constante", "const"},
		{"constantin", "constantin"},
		{"constants", "const"},
		{"constata", "constat"},
		{"constaté", "constat"},
		{"constater", "constat"},
		{"consternation", "constern"},
		{"consterné", "constern"},
		{"consternée", "constern"},
		{"constituait", "constitu"},
		{"constitue", "constitu"},
		{"constitué", "constitu"},
		{"constituer", "constitu"},
		{"constituera", "constitu"},
		{"constitution", "constitu"},
		{"constitutionnel", "constitutionnel"},
		{"constitutionnels", "constitutionnel"},
		{"construction", "construct"},
		{"construire", "construir"},
		{"construisait", "construis"},
		{"construit", "construit"},
		{"construite", "construit"},
		{"construites", "construit"},
		{"construits", "construit"},
		{"consul", "consul"},
		{"consulaire", "consulair"},
		{"consulaires", "consulair"},
		{"consulat", "consulat"},
		{"consulta", "consult"},
		{"consultait", "consult"},
		{"consultant", "consult"},
		{"consultative", "consult"},
		{"consulté", "consult"},
		{"consulter", "consult"},
		{"consulterais", "consult"},
		{"consultés", "consult"},
		{"consultons", "consulton"},
		{"consumé", "consum"},
		{"conta", "cont"},
		{"contact", "contact"},
		{"contagieux", "contagi"},
		{"contais", "cont"},
		{"contait", "cont"},
		{"contarini", "contarin"},
		{"conte", "cont"},
		{"conté", "cont"},
		{"contée", "cont"},
		{"contemplait", "contempl"},
		{"contemplant", "contempl"},
		{"contemplation", "contempl"},
		{"contemple", "contempl"},
		{"contempler", "contempl"},
		{"contemplerait", "contempl"},
		{"contemporains", "contemporain"},
		{"contenaient", "conten"},
		{"contenait", "conten"},
		{"contenance", "conten"},
		{"contenant", "conten"},
		{"contenir", "conten"},
		{"content", "content"},
		{"contenta", "content"},
		{"contentait", "content"},
		{"contentant", "content"},
		{"contente", "content"},
		{"contentement", "content"},
		{"contenterait", "content"},
		{"contents", "content"},
		{"contenu", "contenu"},
		{"contenue", "contenu"},
		{"contenues", "contenu"},
		{"contenus", "contenus"},
		{"conter", "cont"},
		{"conterai", "cont"},
		{"conterait", "cont"},
		{"contes", "cont"},
		{"contessina", "contessin"},
		{"contester", "contest"},
		{"conti", "cont"},
		{"contiennent", "contiennent"},
		{"contient", "contient"},
		{"contine", "contin"},
		{"continent", "continent"},
		{"continental", "continental"},
		{"continents", "continent"},
		{"contino", "contino"},
		{"contînt", "contînt"},
		{"continu", "continu"},
		{"continua", "continu"},
		{"continuait", "continu"},
		{"continuant", "continu"},
		{"continuation", "continu"},
		{"continue", "continu"},
		{"continué", "continu"},
		{"continuel", "continuel"},
		{"continuelle", "continuel"},
		{"continuellement", "continuel"},
		{"continuelles", "continuel"},
		{"continuels", "continuel"},
		{"continuer", "continu"},
		{"continuera", "continu"},
		{"continuerait", "continu"},
		{"continueront", "continu"},
		{"continuez", "continu"},
		{"continuité", "continu"},
		{"continuons", "continuon"},
		{"contorsions", "contors"},
		{"contour", "contour"},
		{"contournait", "contourn"},
		{"contournant", "contourn"},
		{"contourne", "contourn"},
		{"contours", "contour"},
		{"contractait", "contract"},
		{"contracté", "contract"},
		{"contractée", "contract"},
		{"contracter", "contract"},
		{"contractés", "contract"},
		{"contractions", "contract"},
		{"contradictoirement", "contradictoir"},
		{"contradictoires", "contradictoir"},
		{"contraindre", "contraindr"},
		{"contraint", "contraint"},
		{"contrainte", "contraint"},
		{"contraintes", "contraint"},
		{"contraire", "contrair"},
		{"contraires", "contrair"},
		{"contraria", "contrari"},
		{"contrariait", "contrari"},
		{"contrariant", "contrari"},
		{"contrarie", "contrar"},
		{"contrarié", "contrari"},
		{"contrariée", "contrari"},
		{"contrarier", "contrari"},
		{"contrariera", "contrari"},
		{"contrariés", "contrari"},
		{"contrariété", "contrariet"},
		{"contrariétés", "contrariet"},
		{"contrariez", "contrar"},
		{"contrastait", "contrast"},
		{"contraste", "contrast"},
		{"contrastes", "contrast"},
		{"contrat", "contrat"},
		{"contravention", "contravent"},
		{"contraventions", "contravent"},
		{"contre", "contr"},
		{"contrebandier", "contrebandi"},
		{"contrebandiers", "contrebandi"},
		{"contrebas", "contreb"},
		{"contrebasse", "contreb"},
		{"contrebasses", "contreb"},
		{"contrecoup", "contrecoup"},
		{"contrecoups", "contrecoup"},
		{"contredanse", "contredans"},
		{"contredanses", "contredans"},
		{"contredirait", "contred"},
		{"contredire", "contredir"},
		{"contredisait", "contredis"},
		{"contredisant", "contredis"},
		{"contredit", "contred"},
		{"contrée", "contr"},
		{"contrées", "contr"},
		{"contrefaite", "contrefait"},
		{"contrefera", "contref"},
		{"contrefort", "contrefort"},
		{"contresens", "contresen"},
		{"contresigner", "contresign"},
		{"contretemps", "contretemp"},
		{"contribua", "contribu"},
		{"contribuai", "contribu"},
		{"contribué", "contribu"},
		{"contribuer", "contribu"},
		{"contribution", "contribu"},
		{"contributions", "contribu"},
		{"contrit", "contr"},
		{"contrition", "contrit"},
		{"contrôle", "contrôl"},
		{"contrôler", "contrôl"},
		{"contrôles", "contrôl"},
		{"contumace", "contumac"},
		{"contusions", "contus"},
		{"convaincre", "convaincr"},
		{"convaincu", "convaincu"},
		{"convaincue", "convaincu"},
		{"convenable", "conven"},
		{"convenablement", "conven"},
		{"convenables", "conven"},
		{"convenaient", "conven"},
		{"convenait", "conven"},
		{"convenance", "conven"},
		{"convenances", "conven"},
		{"convenez", "conven"},
		{"convenir", "conven"},
		{"convenons", "convenon"},
		{"convention", "convent"},
		{"convenu", "convenu"},
		{"convenue", "convenu"},
		{"convenus", "convenus"},
		{"converger", "converg"},
		{"conversation", "convers"},
		{"conversations", "convers"},
		{"conversion", "convers"},
		{"conversions", "convers"},
		{"convertie", "convert"},
		{"convertir", "convert"},
		{"convertis", "convert"},
		{"conviction", "convict"},
		{"convictions", "convict"},
		{"conviendra", "conviendr"},
		{"conviendrait", "conviendr"},
		{"conviendrez", "conviendr"},
		{"conviendrons", "conviendron"},
		{"convienne", "convien"},
		{"conviennent", "conviennent"},
		{"conviens", "convien"},
		{"convient", "convient"},
		{"convier", "convi"},
		{"convint", "convint"},
		{"convînt", "convînt"},
		{"convives", "conviv"},
		{"convocation", "convoc"},
		{"convoi", "convoi"},
		{"convois", "convois"},
		{"convoitise", "convoitis"},
		{"convulsif", "convuls"},
		{"convulsifs", "convuls"},
		{"convulsion", "convuls"},
		{"convulsionnait", "convulsion"},
		{"convulsive", "convuls"},
		{"convulsivement", "convuls"},
		{"coolies", "cool"},
		{"copeaux", "copeau"},
		{"copiait", "copi"},
		{"copiant", "copi"},
		{"copie", "cop"},
		{"copié", "copi"},
		{"copiée", "copi"},
		{"copient", "copient"},
		{"copier", "copi"},
		{"copies", "cop"},
		{"copieusement", "copieux"},
		{"copiez", "cop"},
		{"copiste", "copist"},
		{"coq", "coq"},
		{"coque", "coqu"},
		{"coquets", "coquet"},
		{"coquette", "coquet"},
		{"coquetterie", "coquetter"},
		{"coquetteries", "coquetter"},
		{"coquille", "coquill"},
		{"coquin", "coquin"},
		{"coquine", "coquin"},
		{"coquinerie", "coquiner"},
		{"coquineries", "coquiner"},
		{"coquins", "coquin"},
		{"coran", "coran"},
		{"corbeau", "corbeau"},
		{"corbeaux", "corbeau"},
		{"corbeille", "corbeil"},
		{"corday", "corday"},
		{"corde", "cord"},
		{"cordes", "cord"},
		{"cordon", "cordon"},
		{"cordons", "cordon"},
		{"corea", "cor"},
		{"cormorans", "cormoran"},
		{"cornac", "cornac"},
		{"corne", "corn"},
		{"corneille", "corneil"},
		{"corneilles", "corneil"},
		{"cornelia", "corneli"},
		{"cornelii", "cornelii"},
		{"cornes", "corn"},
		{"corniche", "cornich"},
		{"cornichon", "cornichon"},
		{"cornwallis", "cornwall"},
		{"corporation", "corpor"},
		{"corps", "corp"},
		{"corpus", "corpus"},
		{"corrals", "corral"},
		{"correct", "correct"},
		{"correcte", "correct"},
		{"correctement", "correct"},
		{"correctif", "correct"},
		{"correction", "correct"},
		{"correctionnelle", "correctionnel"},
		{"corrège", "correg"},
		{"correspondait", "correspond"},
		{"correspondance", "correspond"},
		{"correspondances", "correspond"},
		{"correspondant", "correspond"},
		{"correspondants", "correspond"},
		{"correspondre", "correspondr"},
		{"corridor", "corridor"},
		{"corridors", "corridor"},
		{"corrigeait", "corrig"},
		{"corrigée", "corrig"},
		{"corriger", "corrig"},
		{"corrigerez", "corrig"},
		{"corrodant", "corrod"},
		{"corrompre", "corrompr"},
		{"corrompu", "corrompu"},
		{"corrosif", "corros"},
		{"corruption", "corrupt"},
		{"cors", "cor"},
		{"corso", "corso"},
		{"cortège", "corteg"},
		{"cortellate", "cortellat"},
		{"corvée", "corv"},
		{"cosa", "cos"},
		{"cosaques", "cosaqu"},
		{"cosi", "cos"},
		{"cosmopolite", "cosmopolit"},
		{"costume", "costum"},
		{"costumes", "costum"},
		{"cotait", "cot"},
		{"cote", "cot"},
		{"côte", "côt"},
		{"côté", "côt"},
		{"coteaux", "coteau"},
		{"cotée", "cot"},
		{"coterie", "coter"},
		{"coteries", "coter"},
		{"côtes", "côt"},
		{"côtés", "côt"},
		{"coton", "coton"},
		{"cotonnades", "cotonnad"},
		{"cotons", "coton"},
		{"côtoyant", "côtoi"},
		{"côtoyé", "côtoi"},
		{"cotre", "cotr"},
		{"cotte", "cott"},
		{"cou", "cou"},
		{"coucha", "couch"},
		{"couchait", "couch"},
		{"couchant", "couch"},
		{"couche", "couch"},
		{"couché", "couch"},
		{"couchée", "couch"},
		{"couchent", "couchent"},
		{"coucher", "couch"},
		{"coucherai", "couch"},
		{"coucherait", "couch"},
		{"couchèrent", "couch"},
		{"couchés", "couch"},
		{"couchettes", "couchet"},
		{"couchez", "couch"},
		{"coude", "coud"},
		{"coudes", "coud"},
		{"coudre", "coudr"},
		{"coudrier", "coudri"},
		{"couds", "coud"},
		{"coulaient", "coul"},
		{"coulait", "coul"},
		{"coulant", "coul"},
		{"coule", "coul"},
		{"coulé", "coul"},
		{"couler", "coul"},
		{"coulèrent", "coul"},
		{"couleur", "couleur"},
		{"couleurs", "couleur"},
		{"coulisse", "coul"},
		{"coulisses", "coul"},
		{"couloirs", "couloir"},
		{"coulon", "coulon"},
		{"council", "council"},
		{"coup", "coup"},
		{"coupa", "coup"},
		{"coupable", "coupabl"},
		{"coupables", "coupabl"},
		{"coupaient", "coup"},
		{"coupait", "coup"},
		{"coupant", "coup"},
		{"coupât", "coup"},
		{"coupe", "coup"},
		{"coupé", "coup"},
		{"coupée", "coup"},
		{"coupées", "coup"},
		{"coupent", "coupent"},
		{"couper", "coup"},
		{"couperosée", "couperos"},
		{"coupes", "coup"},
		{"coupés", "coup"},
		{"coupez", "coup"},
		{"couple", "coupl"},
		{"couples", "coupl"},
		{"couplet", "couplet"},
		{"couplets", "couplet"},
		{"coups", "coup"},
		{"coupure", "coupur"},
		{"cour", "cour"},
		{"courage", "courag"},
		{"courages", "courag"},
		{"courageuse", "courag"},
		{"courageusement", "courag"},
		{"courageuses", "courag"},
		{"courageux", "courag"},
		{"couraient", "cour"},
		{"courait", "cour"},
		{"courant", "cour"},
		{"courants", "cour"},
		{"courbait", "courb"},
		{"courbe", "courb"},
		{"courbé", "courb"},
		{"courber", "courb"},
		{"courbes", "courb"},
		{"courbés", "courb"},
		{"courbure", "courbur"},
		{"coure", "cour"},
		{"courent", "courent"},
		{"coureur", "coureur"},
		{"coureurs", "coureur"},
		{"courez", "cour"},
		{"courier", "couri"},
		{"courir", "cour"},
		{"couronnaient", "couron"},
		{"couronne", "couron"},
		{"couronné", "couron"},
		{"couronnées", "couron"},
		{"couronnement", "couron"},
		{"couronnent", "couronnent"},
		{"couronner", "couron"},
		{"couronnes", "couron"},
		{"courons", "couron"},
		{"courrais", "courr"},
		{"courrez", "courr"},
		{"courrier", "courri"},
		{"courriers", "courri"},
		{"courroucés", "courrouc"},
		{"courroux", "courroux"},
		{"cours", "cour"},
		{"course", "cours"},
		{"courses", "cours"},
		{"court", "court"},
		{"courte", "court"},
		{"courtes", "court"},
		{"courtier", "courti"},
		{"courtiers", "courti"},
		{"courtine", "courtin"},
		{"courtisan", "courtisan"},
		{"courtisanerie", "courtisaner"},
		{"courtisanesque", "courtisanesqu"},
		{"courtisans", "courtisan"},
		{"courts", "court"},
		{"couru", "couru"},
		{"coururent", "coururent"},
		{"courus", "courus"},
		{"courut", "courut"},
		{"courût", "courût"},
		{"cousait", "cous"},
		{"cousin", "cousin"},
		{"cousine", "cousin"},
		{"cousines", "cousin"},
		{"coussin", "coussin"},
		{"coussins", "coussin"},
		{"cousu", "cousu"},
		{"coûta", "coût"},
		{"coûtaient", "coût"},
		{"coûtait", "coût"},
		{"coûtant", "coût"},
		{"coûte", "coût"},
		{"coûté", "coût"},
		{"couteau", "couteau"},
		{"couteaux", "couteau"},
		{"coûtent", "coûtent"},
		{"couter", "cout"},
		{"coûter", "coût"},
		{"coûtera", "coût"},
		{"coûterait", "coût"},
		{"coûtèrent", "coût"},
		{"coûteront", "coût"},
		{"coutil", "coutil"},
		{"coutume", "coutum"},
		{"coutumes", "coutum"},
		{"couvent", "couvent"},
		{"couvents", "couvent"},
		{"couvert", "couvert"},
		{"couverte", "couvert"},
		{"couvertes", "couvert"},
		{"couverts", "couvert"},
		{"couverture", "couvertur"},
		{"couvertures", "couvertur"},
		{"couvraient", "couvr"},
		{"couvrait", "couvr"},
		{"couvrant", "couvr"},
		{"couvre", "couvr"},
		{"couvrent", "couvrent"},
		{"couvrir", "couvr"},
		{"couvrira", "couvr"},
		{"couvrit", "couvr"},
		{"craché", "crach"},
		{"cracher", "crach"},
		{"craignais", "craign"},
		{"craignait", "craign"},
		{"craignant", "craign"},
		{"craignez", "craign"},
		{"craignions", "craignion"},
		{"craignirent", "craign"},
		{"craignit", "craign"},
		{"craignons", "craignon"},
		{"craindra", "craindr"},
		{"craindrai", "craindr"},
		{"craindre", "craindr"},
		{"crains", "crain"},
		{"craint", "craint"},
		{"crainte", "craint"},
		{"craintes", "craint"},
		{"craintifs", "craintif"},
		{"cramoisi", "cramois"},
		{"cramoisie", "cramois"},
		{"cramoisis", "cramois"},
		{"crampe", "cramp"},
		{"cramponne", "crampon"},
		{"crâne", "crân"},
		{"craquement", "craqu"},
		{"crasseux", "crasseux"},
		{"cravache", "cravach"},
		{"cravate", "cravat"},
		{"crayon", "crayon"},
		{"créance", "créanc"},
		{"créature", "créatur"},
		{"créatures", "créatur"},
		{"credete", "credet"},
		{"crédit", "cred"},
		{"créditeur", "créditeur"},
		{"crédulité", "crédul"},
		{"créé", "cré"},
		{"creek", "creek"},
		{"creeks", "creek"},
		{"crème", "crem"},
		{"créole", "créol"},
		{"crépuscule", "crépuscul"},
		{"crescentini", "crescentin"},
		{"crescentius", "crescentius"},
		{"crescenzi", "crescenz"},
		{"crête", "crêt"},
		{"creusaient", "creus"},
		{"creusait", "creus"},
		{"creusé", "creus"},
		{"creuses", "creus"},
		{"creux", "creux"},
		{"crevé", "crev"},
		{"crèvera", "crev"},
		{"cri", "cri"},
		{"cria", "cri"},
		{"criaient", "cri"},
		{"criait", "cri"},
		{"criant", "cri"},
		{"criante", "cri"},
		{"criards", "criard"},
		{"criblé", "cribl"},
		{"crie", "cri"},
		{"crié", "cri"},
		{"crier", "cri"},
		{"crièrent", "cri"},
		{"criés", "cri"},
		{"criez", "cri"},
		{"crime", "crim"},
		{"crimes", "crim"},
		{"criminel", "criminel"},
		{"criminelle", "criminel"},
		{"criminelles", "criminel"},
		{"criminels", "criminel"},
		{"crin", "crin"},
		{"crinière", "crini"},
		{"crinières", "crini"},
		{"crins", "crin"},
		{"cris", "cris"},
		{"crise", "cris"},
		{"crispe", "crisp"},
		{"cristal", "cristal"},
		{"cristaux", "cristal"},
		{"critique", "critiqu"},
		{"critiqué", "critiqu"},
		{"critiquent", "critiquent"},
		{"critiques", "critiqu"},
		{"crochet", "crochet"},
		{"croie", "croi"},
		{"croient", "croient"},
		{"croira", "croir"},
		{"croirai", "croir"},
		{"croiraient", "croir"},
		{"croirais", "croir"},
		{"croirait", "croir"},
		{"croire", "croir"},
		{"croirez", "croir"},
		{"croiriez", "croir"},
		{"croirons", "croiron"},
		{"croiront", "croiront"},
		{"crois", "crois"},
		{"croisa", "crois"},
		{"croisade", "croisad"},
		{"croisades", "croisad"},
		{"croisaient", "crois"},
		{"croisait", "crois"},
		{"croisé", "crois"},
		{"croisée", "crois"},
		{"croisées", "crois"},
		{"croisenois", "croisenois"},
		{"croisés", "crois"},
		{"croissaient", "croiss"},
		{"croissait", "croiss"},
		{"croissance", "croissanc"},
		{"croissant", "croiss"},
		{"croissante", "croiss"},
		{"croit", "croit"},
		{"croître", "croîtr"},
		{"croix", "croix"},
		{"cromarty", "cromarty"},
		{"cross", "cross"},
		{"crotte", "crott"},
		{"crotté", "crott"},
		{"crottées", "crott"},
		{"crouler", "croul"},
		{"croupe", "croup"},
		{"croupir", "croup"},
		{"croyaient", "croi"},
		{"croyais", "croi"},
		{"croyait", "croi"},
		{"croyance", "croyanc"},
		{"croyant", "croi"},
		{"croyante", "croi"},
		{"croyants", "croi"},
		{"croyez", "croi"},
		{"croyiez", "croi"},
		{"croyons", "croyon"},
		{"cru", "cru"},
		{"cruauté", "cruaut"},
		{"cruautés", "cruaut"},
		{"cruciales", "crucial"},
		{"crucifix", "crucifix"},
		{"crue", "cru"},
		{"cruel", "cruel"},
		{"cruelle", "cruel"},
		{"cruellement", "cruel"},
		{"cruelles", "cruel"},
		{"cruels", "cruel"},
		{"crues", "cru"},
		{"crûment", "crûment"},
		{"crurent", "crurent"},
		{"crus", "crus"},
		{"crut", "crut"},
		{"crût", "crût"},
		{"cueilli", "cueil"},
		{"cuir", "cuir"},
		{"cuirasse", "cuir"},
		{"cuirasses", "cuir"},
		{"cuirassier", "cuirassi"},
		{"cuirassiers", "cuirassi"},
		{"cuisant", "cuis"},
		{"cuisante", "cuis"},
		{"cuisine", "cuisin"},
		{"cuisines", "cuisin"},
		{"cuisinier", "cuisini"},
		{"cuisinière", "cuisini"},
		{"cuisiniers", "cuisini"},
		{"cuisse", "cuiss"},
		{"cuisses", "cuiss"},
		{"cuistre", "cuistr"},
		{"cuistres", "cuistr"},
		{"cuite", "cuit"},
		{"cuivre", "cuivr"},
		{"cuivres", "cuivr"},
		{"culbute", "culbut"},
		{"culotte", "culott"},
		{"culottes", "culott"},
		{"culpa", "culp"},
		{"culpabilité", "culpabl"},
		{"culte", "cult"},
		{"cultivée", "cultiv"},
		{"cultivées", "cultiv"},
		{"cultivent", "cultivent"},
		{"cultiver", "cultiv"},
		{"cunard", "cunard"},
		{"cunctando", "cunctando"},
		{"cupidité", "cupid"},
		{"cure", "cur"},
		{"curé", "cur"},
		{"cures", "cur"},
		{"curés", "cur"},
		{"curieuse", "curieux"},
		{"curieusement", "curieux"},
		{"curieuses", "curieux"},
		{"curieux", "curieux"},
		{"curiosité", "curios"},
		{"custom", "custom"},
		{"cuves", "cuv"},
		{"cuvette", "cuvet"},
		{"cygne", "cygn"},
		{"cymbales", "cymbal"},
		{"cynique", "cyniqu"},
		{"cyr", "cyr"},
		{"d", "d"},
		{"da", "da"},
		{"dague", "dagu"},
		{"daigna", "daign"},
		{"daignaient", "daign"},
		{"daignait", "daign"},
		{"daignant", "daign"},
		{"daignât", "daign"},
		{"daigne", "daign"},
		{"daigné", "daign"},
		{"daignent", "daignent"},
		{"daigner", "daign"},
		{"daignera", "daign"},
		{"daignerait", "daign"},
		{"daignèrent", "daign"},
		{"daignerez", "daign"},
		{"daignez", "daign"},
		{"daily", "daily"},
		{"daim", "daim"},
		{"dais", "dais"},
		{"dakota", "dakot"},
		{"dalles", "dall"},
		{"dalliance", "dallianc"},
		{"dalmate", "dalmat"},
		{"damas", "dam"},
		{"damasquinés", "damasquin"},
		{"dame", "dam"},
		{"dames", "dam"},
		{"damné", "damn"},
		{"damnée", "damn"},
		{"damnent", "damnent"},
		{"damnés", "damn"},
		{"dandin", "dandin"},
		{"dandinant", "dandin"},
		{"dandy", "dandy"},
		{"dandys", "dandy"},
		{"danger", "dang"},
		{"dangereuse", "danger"},
		{"dangereusement", "danger"},
		{"dangereuses", "danger"},
		{"dangereux", "danger"},
		{"dangers", "danger"},
		{"dans", "dan"},
		{"dansa", "dans"},
		{"dansaient", "dans"},
		{"dansait", "dans"},
		{"dansant", "dans"},
		{"danse", "dans"},
		{"dansé", "dans"},
		{"dansent", "dansent"},
		{"danser", "dans"},
		{"dansèrent", "dans"},
		{"danses", "dans"},
		{"danseuses", "danseux"},
		{"dante", "dant"},
		{"danton", "danton"},
		{"darkness", "darkness"},
		{"data", "dat"},
		{"datait", "dat"},
		{"date", "dat"},
		{"daté", "dat"},
		{"datée", "dat"},
		{"datées", "dat"},
		{"dates", "dat"},
		{"dattiers", "datti"},
		{"davantage", "davantag"},
		{"davenport", "davenport"},
		{"davide", "david"},
		{"day", "day"},
		{"days", "day"},
		{"de", "de"},
		{"dé", "dé"},
		{"débarqua", "débarqu"},
		{"débarquaient", "débarqu"},
		{"débarquait", "débarqu"},
		{"débarquant", "débarqu"},
		{"débarque", "débarqu"},
		{"débarqué", "débarqu"},
		{"débarquement", "débarqu"},
		{"débarquer", "débarqu"},
		{"débarquèrent", "débarqu"},
		{"débarrassé", "débarrass"},
		{"débarrassée", "débarrass"},
		{"débarrasser", "débarrass"},
		{"débarrassés", "débarrass"},
		{"débarrassez", "débarrass"},
		{"débat", "débat"},
		{"débats", "débat"},
		{"débattait", "débatt"},
		{"débattant", "débatt"},
		{"débattrais", "débattr"},
		{"débattre", "débattr"},
		{"débattue", "débattu"},
		{"débauche", "débauch"},
		{"debile", "debil"},
		{"débit", "deb"},
		{"débita", "débit"},
		{"débitaient", "débit"},
		{"débitant", "débit"},
		{"débite", "débit"},
		{"débitées", "débit"},
		{"débiter", "débit"},
		{"débiteur", "débiteur"},
		{"débonnaireté", "débonnairet"},
		{"débordait", "débord"},
		{"débordé", "débord"},
		{"déborder", "débord"},
		{"débouchaient", "débouch"},
		{"débouche", "débouch"},
		{"déboucher", "débouch"},
		{"déboursé", "débours"},
		{"debout", "debout"},
		{"débrider", "débrid"},
		{"débris", "debr"},
		{"débrouillée", "débrouill"},
		{"débrouiller", "débrouill"},
		{"début", "début"},
		{"débuta", "début"},
		{"débute", "début"},
		{"débuté", "début"},
		{"débuter", "début"},
		{"décachetait", "décachet"},
		{"décachetée", "décachet"},
		{"décachetées", "décachet"},
		{"décadence", "décadent"},
		{"décampa", "décamp"},
		{"décampe", "décamp"},
		{"décampé", "décamp"},
		{"décamper", "décamp"},
		{"décampons", "décampon"},
		{"décapité", "décap"},
		{"décelât", "décel"},
		{"décembre", "décembr"},
		{"décemment", "décent"},
		{"décence", "décenc"},
		{"décent", "décent"},
		{"décente", "décent"},
		{"décents", "décent"},
		{"déceptions", "décept"},
		{"décès", "déces"},
		{"déchaînait", "déchaîn"},
		{"déchaînât", "déchaîn"},
		{"déchaînement", "déchaîn"},
		{"déchaîner", "déchaîn"},
		{"décharge", "décharg"},
		{"déchargement", "décharg"},
		{"décharger", "décharg"},
		{"décharges", "décharg"},
		{"déchira", "dech"},
		{"déchiraient", "dech"},
		{"déchirait", "dech"},
		{"déchirant", "déchir"},
		{"déchire", "déchir"},
		{"déchiré", "déchir"},
		{"déchirée", "déchir"},
		{"déchirées", "déchir"},
		{"déchirements", "déchir"},
		{"déchirer", "déchir"},
		{"déchirés", "déchir"},
		{"déchirures", "déchirur"},
		{"déchoir", "déchoir"},
		{"déchu", "déchu"},
		{"décida", "décid"},
		{"décidaient", "décid"},
		{"décidait", "décid"},
		{"décidant", "décid"},
		{"décide", "décid"},
		{"décidé", "décid"},
		{"décidée", "décid"},
		{"décidées", "décid"},
		{"décidément", "décid"},
		{"décident", "décident"},
		{"décider", "décid"},
		{"décidera", "décid"},
		{"déciderait", "décid"},
		{"décidèrent", "décid"},
		{"déciderez", "décid"},
		{"décimus", "décimus"},
		{"décisif", "décis"},
		{"décision", "décis"},
		{"décisions", "décis"},
		{"décisive", "décis"},
		{"décius", "décius"},
		{"déclamer", "déclam"},
		{"déclara", "déclar"},
		{"déclarait", "déclar"},
		{"déclarant", "déclar"},
		{"déclaration", "déclar"},
		{"déclarations", "déclar"},
		{"déclare", "déclar"},
		{"déclaré", "déclar"},
		{"déclarée", "déclar"},
		{"déclarer", "déclar"},
		{"déclarerait", "déclar"},
		{"déclarèrent", "déclar"},
		{"déclarés", "déclar"},
		{"déclarez", "déclar"},
		{"déclassés", "déclass"},
		{"décoloré", "décolor"},
		{"décolorée", "décolor"},
		{"décolorées", "décolor"},
		{"décomposition", "décomposit"},
		{"déconcerta", "déconcert"},
		{"déconcertait", "déconcert"},
		{"déconcerté", "déconcert"},
		{"déconcertée", "déconcert"},
		{"déconcerter", "déconcert"},
		{"déconcertés", "déconcert"},
		{"déconfit", "déconf"},
		{"décontenancé", "décontenanc"},
		{"déconvenues", "déconvenu"},
		{"décorateur", "décor"},
		{"décoration", "décor"},
		{"décore", "décor"},
		{"décoré", "décor"},
		{"décorée", "décor"},
		{"décorum", "décorum"},
		{"découchait", "découch"},
		{"découpait", "découp"},
		{"découpant", "découp"},
		{"découpés", "découp"},
		{"découragé", "décourag"},
		{"découragement", "décourag"},
		{"découragements", "décourag"},
		{"décourager", "décourag"},
		{"décousus", "décousus"},
		{"découvert", "découvert"},
		{"découverte", "découvert"},
		{"découvertes", "découvert"},
		{"découverts", "découvert"},
		{"découvrait", "découvr"},
		{"découvrant", "découvr"},
		{"découvre", "découvr"},
		{"découvrir", "découvr"},
		{"découvrira", "découvr"},
		{"découvrirent", "découvr"},
		{"découvrit", "découvr"},
		{"décrépitude", "décrépitud"},
		{"décret", "décret"},
		{"décrets", "décret"},
		{"décrire", "décrir"},
		{"décrit", "decr"},
		{"décrivaient", "décriv"},
		{"décrivait", "décriv"},
		{"décrivant", "décriv"},
		{"décrocha", "décroch"},
		{"dédaignant", "dédaign"},
		{"dédaignée", "dédaign"},
		{"dédaigneuse", "dédaign"},
		{"dédaigneux", "dédaign"},
		{"dédain", "dédain"},
		{"dédains", "dédain"},
		{"dedans", "dedan"},
		{"dédommageait", "dédommag"},
		{"dédommagement", "dédommag"},
		{"dédommagent", "dédommagent"},
		{"dédommagerez", "dédommag"},
		{"déduire", "déduir"},
		{"déduisit", "déduis"},
		{"deem", "deem"},
		{"deeper", "deep"},
		{"déesse", "déess"},
		{"défaillante", "défaill"},
		{"défaillir", "défaill"},
		{"défaire", "défair"},
		{"défais", "def"},
		{"défaisait", "défais"},
		{"défait", "def"},
		{"défaite", "défait"},
		{"défaites", "défait"},
		{"défasse", "def"},
		{"défaut", "défaut"},
		{"défauts", "défaut"},
		{"défaveur", "défaveur"},
		{"défavorable", "défavor"},
		{"défection", "défect"},
		{"défend", "défend"},
		{"défendaient", "défend"},
		{"défendait", "défend"},
		{"défendant", "défend"},
		{"défende", "défend"},
		{"défendent", "défendent"},
		{"défendez", "défend"},
		{"défendit", "défend"},
		{"défendons", "défendon"},
		{"défendre", "défendr"},
		{"défends", "défend"},
		{"défendu", "défendu"},
		{"défendue", "défendu"},
		{"défendues", "défendu"},
		{"défense", "défens"},
		{"défenses", "défens"},
		{"défenseurs", "défenseur"},
		{"défensive", "défens"},
		{"déferai", "def"},
		{"déférence", "déférent"},
		{"défiait", "défi"},
		{"défiance", "défianc"},
		{"défiant", "défi"},
		{"défie", "def"},
		{"défigurait", "défigur"},
		{"défiguré", "défigur"},
		{"défila", "défil"},
		{"défilé", "défil"},
		{"défiler", "défil"},
		{"défilés", "défil"},
		{"définissable", "définiss"},
		{"définitive", "définit"},
		{"définitivement", "définit"},
		{"défit", "def"},
		{"défoncèrent", "défonc"},
		{"défroque", "défroqu"},
		{"défunt", "défunt"},
		{"dégagé", "dégag"},
		{"dégagea", "dégag"},
		{"dégagée", "dégag"},
		{"dégagement", "dégag"},
		{"dégager", "dégag"},
		{"dégât", "deg"},
		{"dégénéré", "dégéner"},
		{"dégoût", "dégoût"},
		{"dégoûtant", "dégoût"},
		{"dégoûtante", "dégoût"},
		{"dégoûté", "dégoût"},
		{"dégoûtée", "dégoût"},
		{"dégradée", "dégrad"},
		{"dégradent", "dégradent"},
		{"dégrader", "dégrad"},
		{"degré", "degr"},
		{"degrés", "degr"},
		{"dégrisa", "dégris"},
		{"déguenillée", "déguenill"},
		{"déguerpir", "déguerp"},
		{"déguisa", "déguis"},
		{"déguisait", "déguis"},
		{"déguise", "déguis"},
		{"déguisé", "déguis"},
		{"déguisée", "déguis"},
		{"déguisement", "déguis"},
		{"déguiser", "déguis"},
		{"déguisèrent", "déguis"},
		{"déguisés", "déguis"},
		{"dehors", "dehor"},
		{"déjà", "déjà"},
		{"déjetées", "déjet"},
		{"déjeuna", "déjeun"},
		{"déjeunait", "déjeun"},
		{"déjeunant", "déjeun"},
		{"déjeuné", "déjeun"},
		{"déjeuner", "déjeun"},
		{"déjeunèrent", "déjeun"},
		{"déjeunons", "déjeunon"},
		{"déjouer", "déjou"},
		{"déjoués", "déjou"},
		{"del", "del"},
		{"delà", "delà"},
		{"délabré", "délabr"},
		{"délabrée", "délabr"},
		{"délabrement", "délabr"},
		{"délai", "del"},
		{"délais", "del"},
		{"délaissé", "délaiss"},
		{"délasser", "délass"},
		{"délations", "délat"},
		{"delavigne", "delavign"},
		{"délayé", "délai"},
		{"délégation", "déleg"},
		{"délibéra", "déliber"},
		{"délibération", "déliber"},
		{"délibérer", "déliber"},
		{"délibérons", "délibéron"},
		{"délicat", "délicat"},
		{"délicate", "délicat"},
		{"délicates", "délicat"},
		{"délicatesse", "délicatess"},
		{"délicats", "délicat"},
		{"délices", "délic"},
		{"délicieuse", "délici"},
		{"délicieusement", "délici"},
		{"délicieuses", "délici"},
		{"délicieux", "délici"},
		{"délier", "déli"},
		{"delille", "delill"},
		{"délire", "délir"},
		{"délit", "del"},
		{"délivra", "délivr"},
		{"délivrance", "délivr"},
		{"délivre", "délivr"},
		{"délivré", "délivr"},
		{"délivrée", "délivr"},
		{"délivrer", "délivr"},
		{"délivrera", "délivr"},
		{"dell", "del"},
		{"della", "del"},
		{"demain", "demain"},
		{"demanda", "demand"},
		{"demandai", "demand"},
		{"demandaient", "demand"},
		{"demandais", "demand"},
		{"demandait", "demand"},
		{"demandant", "demand"},
		{"demande", "demand"},
		{"demandé", "demand"},
		{"demandée", "demand"},
		{"demandées", "demand"},
		{"demander", "demand"},
		{"demandera", "demand"},
		{"demanderai", "demand"},
		{"demanderaient", "demand"},
		{"demanderait", "demand"},
		{"demandèrent", "demand"},
		{"demanderont", "demand"},
		{"demandes", "demand"},
		{"demandés", "demand"},
		{"demandez", "demand"},
		{"demandiez", "demand"},
		{"demandons", "demandon"},
		{"démantelé", "démantel"},
		{"démarche", "démarch"},
		{"démarches", "démarch"},
		{"démarquer", "démarqu"},
		{"démasquer", "démasqu"},
		{"démêloir", "démêloir"},
		{"déménager", "déménag"},
		{"démenaient", "démen"},
		{"démenait", "démen"},
		{"démenti", "dément"},
		{"démentir", "dément"},
		{"démentis", "dément"},
		{"démesurée", "démesur"},
		{"demesurement", "demesur"},
		{"démesurément", "démesur"},
		{"demeura", "demeur"},
		{"demeurait", "demeur"},
		{"demeure", "demeur"},
		{"demeuré", "demeur"},
		{"demeurer", "demeur"},
		{"demeurerait", "demeur"},
		{"demeurèrent", "demeur"},
		{"demeurés", "demeur"},
		{"demi", "dem"},
		{"demie", "dem"},
		{"démission", "démiss"},
		{"démit", "dem"},
		{"demoiselle", "demoisel"},
		{"demoiselles", "demoisel"},
		{"démolir", "démol"},
		{"démolissent", "démol"},
		{"démolition", "démolit"},
		{"démon", "démon"},
		{"démonstration", "démonstr"},
		{"démonté", "démont"},
		{"démontée", "démont"},
		{"démonter", "démont"},
		{"démontés", "démont"},
		{"démontra", "démontr"},
		{"démontrait", "démontr"},
		{"démontré", "démontr"},
		{"dénaturée", "dénatur"},
		{"deniers", "deni"},
		{"dénigrement", "dénigr"},
		{"dénomination", "dénomin"},
		{"dénonçait", "dénonc"},
		{"dénonce", "dénonc"},
		{"dénoncé", "dénonc"},
		{"dénoncer", "dénonc"},
		{"dénoncera", "dénonc"},
		{"dénoncez", "dénonc"},
		{"dénonciateur", "dénonci"},
		{"dénonciation", "dénonci"},
		{"dénonciations", "dénonci"},
		{"dénotaient", "dénot"},
		{"dénotait", "dénot"},
		{"dénoter", "dénot"},
		{"dénouement", "dénou"},
		{"dénoûment", "dénoû"},
		{"densité", "densit"},
		{"dent", "dent"},
		{"dentelle", "dentel"},
		{"dentelles", "dentel"},
		{"dents", "dent"},
		{"dénué", "dénu"},
		{"dénuées", "dénu"},
		{"denver", "denv"},
		{"deo", "deo"},
		{"déparait", "dépar"},
		{"déparant", "dépar"},
		{"déparée", "dépar"},
		{"dépareillé", "dépareil"},
		{"départ", "départ"},
		{"département", "départ"},
		{"dépassaient", "dépass"},
		{"dépassait", "dépass"},
		{"dépasse", "dep"},
		{"dépassé", "dépass"},
		{"dépassées", "dépass"},
		{"dépasser", "dépass"},
		{"dépassés", "dépass"},
		{"dépêcha", "dépêch"},
		{"dépêche", "dépêch"},
		{"dépêché", "dépêch"},
		{"dépêches", "dépêch"},
		{"dépêchez", "dépêch"},
		{"dépêchons", "dépêchon"},
		{"dépeint", "dépeint"},
		{"dépeintes", "dépeint"},
		{"dépenaillé", "dépenaill"},
		{"dépend", "dépend"},
		{"dépendait", "dépend"},
		{"dépendance", "dépend"},
		{"dépendant", "dépend"},
		{"dépendre", "dépendr"},
		{"dépendu", "dépendu"},
		{"dépens", "dépen"},
		{"dépensait", "dépens"},
		{"dépense", "dépens"},
		{"dépensé", "dépens"},
		{"dépensées", "dépens"},
		{"dépenser", "dépens"},
		{"dépenses", "dépens"},
		{"dépérir", "déper"},
		{"dépeuplèrent", "dépeupl"},
		{"dépisté", "dépist"},
		{"dépistées", "dépist"},
		{"dépister", "dépist"},
		{"dépit", "dep"},
		{"dépitée", "dépit"},
		{"déplaçait", "déplac"},
		{"déplace", "déplac"},
		{"déplacé", "déplac"},
		{"déplacées", "déplac"},
		{"déplacement", "déplac"},
		{"déplairais", "déplair"},
		{"déplaire", "déplair"},
		{"déplairez", "déplair"},
		{"déplaisait", "déplais"},
		{"déplaise", "déplais"},
		{"déplaisent", "déplaisent"},
		{"déplaisir", "déplais"},
		{"déplaît", "déplaît"},
		{"dépliage", "dépliag"},
		{"déploie", "déploi"},
		{"déploierais", "déploi"},
		{"déplorable", "déplor"},
		{"déplorables", "déplor"},
		{"déplorait", "déplor"},
		{"déploré", "déplor"},
		{"déploya", "déploi"},
		{"déployaient", "déploi"},
		{"déployé", "déploi"},
		{"déployée", "déploi"},
		{"déployer", "déploi"},
		{"déplu", "déplu"},
		{"déplut", "déplut"},
		{"déporta", "déport"},
		{"déportés", "déport"},
		{"déposa", "dépos"},
		{"déposait", "dépos"},
		{"déposant", "dépos"},
		{"déposât", "dépos"},
		{"dépose", "dépos"},
		{"déposé", "dépos"},
		{"déposée", "dépos"},
		{"déposées", "dépos"},
		{"déposer", "dépos"},
		{"déposèrent", "dépos"},
		{"déposés", "dépos"},
		{"dépositaire", "dépositair"},
		{"déposition", "déposit"},
		{"dépositions", "déposit"},
		{"dépossédé", "dépossed"},
		{"dépôt", "dépôt"},
		{"dépôts", "dépôt"},
		{"dépouillait", "dépouill"},
		{"dépouille", "dépouill"},
		{"dépouillé", "dépouill"},
		{"dépouiller", "dépouill"},
		{"dépourvu", "dépourvu"},
		{"dépression", "dépress"},
		{"dépressions", "dépress"},
		{"déprimée", "déprim"},
		{"depuis", "depuis"},
		{"députation", "déput"},
		{"député", "déput"},
		{"députés", "déput"},
		{"déraillement", "déraill"},
		{"déraillements", "déraill"},
		{"déraison", "déraison"},
		{"déraisonnable", "déraison"},
		{"déraisonnables", "déraison"},
		{"déraisonner", "déraison"},
		{"dérangé", "dérang"},
		{"dérangeait", "dérang"},
		{"dérangements", "dérang"},
		{"déranger", "dérang"},
		{"dérangés", "dérang"},
		{"derechef", "derechef"},
		{"déridait", "dérid"},
		{"dérision", "déris"},
		{"dernier", "derni"},
		{"dernière", "derni"},
		{"dernieres", "dernier"},
		{"dernières", "derni"},
		{"derniers", "derni"},
		{"dérobaient", "dérob"},
		{"dérobé", "dérob"},
		{"dérobée", "dérob"},
		{"dérober", "dérob"},
		{"déroberait", "dérob"},
		{"déroger", "dérog"},
		{"déroula", "déroul"},
		{"déroulant", "déroul"},
		{"déroulèrent", "déroul"},
		{"déroute", "dérout"},
		{"derrière", "derri"},
		{"derville", "dervill"},
		{"des", "de"},
		{"dès", "des"},
		{"désabusé", "désabus"},
		{"désagréable", "désagré"},
		{"désagréablement", "désagré"},
		{"désagréables", "désagré"},
		{"désagrément", "désagr"},
		{"désagréments", "désagr"},
		{"desaix", "desaix"},
		{"désaltéré", "désalter"},
		{"désappointé", "désappoint"},
		{"désappointement", "désappoint"},
		{"désapprouver", "désapprouv"},
		{"désarmé", "désarm"},
		{"désarmer", "désarm"},
		{"désastre", "désastr"},
		{"désastres", "désastr"},
		{"désavantage", "désavantag"},
		{"désavantages", "désavantag"},
		{"désavoue", "désavou"},
		{"desceller", "descel"},
		{"descend", "descend"},
		{"descendaient", "descend"},
		{"descendait", "descend"},
		{"descendant", "descend"},
		{"descendante", "descend"},
		{"descendants", "descend"},
		{"descende", "descend"},
		{"descendent", "descendent"},
		{"descendirent", "descend"},
		{"descendit", "descend"},
		{"descendrait", "descendr"},
		{"descendre", "descendr"},
		{"descendrez", "descendr"},
		{"descends", "descend"},
		{"descendu", "descendu"},
		{"descendue", "descendu"},
		{"descendus", "descendus"},
		{"descente", "descent"},
		{"descoulis", "descoul"},
		{"description", "descript"},
		{"descriptions", "descript"},
		{"désennuyait", "désennui"},
		{"désennuyer", "désennui"},
		{"désert", "désert"},
		{"désertais", "désert"},
		{"déserte", "désert"},
		{"déserté", "désert"},
		{"déserter", "désert"},
		{"désertes", "désert"},
		{"déserteur", "déserteur"},
		{"désertez", "désert"},
		{"désertion", "désert"},
		{"désespérais", "désesper"},
		{"désespérait", "désesper"},
		{"désespérant", "désesper"},
		{"désespérante", "désesper"},
		{"désespérants", "désesper"},
		{"désespère", "désesper"},
		{"désespéré", "désesper"},
		{"désespérée", "désesper"},
		{"désespérer", "désesper"},
		{"désespoir", "désespoir"},
		{"déshabillaient", "déshabill"},
		{"déshabiller", "déshabill"},
		{"déshonneur", "déshonneur"},
		{"déshonorait", "déshonor"},
		{"déshonorant", "déshonor"},
		{"déshonorante", "déshonor"},
		{"déshonore", "déshonor"},
		{"déshonoré", "déshonor"},
		{"déshonorée", "déshonor"},
		{"déshonorent", "déshonorent"},
		{"déshonorer", "déshonor"},
		{"déshonorez", "déshonor"},
		{"désignant", "désign"},
		{"désigne", "désign"},
		{"désigné", "désign"},
		{"désigner", "désign"},
		{"désignèrent", "désign"},
		{"désignés", "désign"},
		{"désignez", "désign"},
		{"désintéresse", "désintéress"},
		{"désintéressé", "désintéress"},
		{"désintéressées", "désintéress"},
		{"désintéressement", "désintéress"},
		{"désintérêt", "désintérêt"},
		{"désir", "des"},
		{"désira", "des"},
		{"désirable", "désir"},
		{"désirait", "des"},
		{"désire", "désir"},
		{"désiré", "désir"},
		{"désirée", "désir"},
		{"désirées", "désir"},
		{"désirent", "des"},
		{"désirer", "désir"},
		{"désirerais", "désir"},
		{"désirez", "des"},
		{"désirs", "désir"},
		{"désobéi", "désobéi"},
		{"désobéissante", "désobéiss"},
		{"désoblige", "désoblig"},
		{"désobligeante", "désoblig"},
		{"désobligeants", "désoblig"},
		{"désoeuvrement", "désoeuvr"},
		{"désolait", "désol"},
		{"désolant", "désol"},
		{"désolation", "désol"},
		{"désole", "désol"},
		{"désolé", "désol"},
		{"désolée", "désol"},
		{"désordonnée", "désordon"},
		{"désordonnés", "désordon"},
		{"désordre", "désordr"},
		{"désorienté", "désorient"},
		{"désormais", "désorm"},
		{"despote", "despot"},
		{"despotes", "despot"},
		{"despotique", "despot"},
		{"despotiquement", "despot"},
		{"despotiques", "despot"},
		{"despotisme", "despot"},
		{"despotismes", "despot"},
		{"desquelles", "desquel"},
		{"desquels", "desquel"},
		{"desséchant", "dessech"},
		{"desséché", "dessech"},
		{"desséchée", "dessech"},
		{"dessein", "dessein"},
		{"desseins", "dessein"},
		{"dessert", "dessert"},
		{"desservait", "desserv"},
		{"desservant", "desserv"},
		{"desservi", "desserv"},
		{"dessiller", "dessill"},
		{"dessillèrent", "dessill"},
		{"dessin", "dessin"},
		{"dessina", "dessin"},
		{"dessinaient", "dessin"},
		{"dessinait", "dessin"},
		{"dessine", "dessin"},
		{"dessiné", "dessin"},
		{"dessinent", "dessinent"},
		{"dessiner", "dessin"},
		{"dessins", "dessin"},
		{"dessous", "dessous"},
		{"dessus", "dessus"},
		{"destin", "destin"},
		{"destinait", "destin"},
		{"destination", "destin"},
		{"destine", "destin"},
		{"destiné", "destin"},
		{"destinée", "destin"},
		{"destinées", "destin"},
		{"destinés", "destin"},
		{"destins", "destin"},
		{"destitua", "destitu"},
		{"destitué", "destitu"},
		{"destituer", "destitu"},
		{"destituera", "destitu"},
		{"destituerai", "destitu"},
		{"destitution", "destitu"},
		{"désunir", "désun"},
		{"détacha", "détach"},
		{"détachaient", "détach"},
		{"détachait", "détach"},
		{"détachant", "détach"},
		{"détache", "détach"},
		{"détaché", "détach"},
		{"détachée", "détach"},
		{"détachées", "détach"},
		{"détachement", "détach"},
		{"détacher", "détach"},
		{"détachèrent", "détach"},
		{"détail", "détail"},
		{"détailla", "détaill"},
		{"détaillant", "détaill"},
		{"détaille", "détaill"},
		{"détaillé", "détaill"},
		{"détaillée", "détaill"},
		{"détaillées", "détaill"},
		{"détailler", "détaill"},
		{"détails", "détail"},
		{"détaler", "détal"},
		{"détective", "détect"},
		{"détectives", "détect"},
		{"détendu", "détendu"},
		{"détendue", "détendu"},
		{"détendus", "détendus"},
		{"détente", "détent"},
		{"détention", "détent"},
		{"détenus", "détenus"},
		{"détermina", "détermin"},
		{"déterminaient", "détermin"},
		{"déterminait", "détermin"},
		{"déterminât", "détermin"},
		{"détermination", "détermin"},
		{"déterminations", "détermin"},
		{"déterminé", "détermin"},
		{"déterminées", "détermin"},
		{"déterminer", "détermin"},
		{"déterminera", "détermin"},
		{"déterrer", "déterr"},
		{"détestable", "détest"},
		{"détestables", "détest"},
		{"détestaient", "détest"},
		{"détestait", "détest"},
		{"déteste", "détest"},
		{"détesté", "détest"},
		{"détestent", "détestent"},
		{"détester", "détest"},
		{"détesterez", "détest"},
		{"détonation", "déton"},
		{"détonations", "déton"},
		{"détour", "détour"},
		{"détourna", "détourn"},
		{"détournaient", "détourn"},
		{"détourné", "détourn"},
		{"détours", "détour"},
		{"détraquer", "détraqu"},
		{"détresse", "détress"},
		{"détriment", "detr"},
		{"détroit", "détroit"},
		{"détrompé", "détromp"},
		{"détruire", "détruir"},
		{"détruisaient", "détruis"},
		{"détruisit", "détruis"},
		{"détruit", "détruit"},
		{"détruites", "détruit"},
		{"détruits", "détruit"},
		{"dette", "det"},
		{"dettes", "det"},
		{"deuil", "deuil"},
		{"deuils", "deuil"},
		{"deum", "deum"},
		{"deux", "deux"},
		{"deuxième", "deuxiem"},
		{"devaient", "dev"},
		{"devais", "dev"},
		{"devait", "dev"},
		{"devançant", "devanc"},
		{"devancer", "devanc"},
		{"devant", "dev"},
		{"devants", "dev"},
		{"développa", "développ"},
		{"développaient", "développ"},
		{"développait", "développ"},
		{"développée", "développ"},
		{"développement", "développ"},
		{"devenaient", "deven"},
		{"devenais", "deven"},
		{"devenait", "deven"},
		{"devenant", "deven"},
		{"devenez", "deven"},
		{"devenir", "deven"},
		{"devenu", "devenu"},
		{"devenue", "devenu"},
		{"devenus", "devenus"},
		{"déverse", "dévers"},
		{"déverser", "dévers"},
		{"devez", "dev"},
		{"deviendra", "deviendr"},
		{"deviendrai", "deviendr"},
		{"deviendraient", "deviendr"},
		{"deviendrais", "deviendr"},
		{"deviendrait", "deviendr"},
		{"deviendrez", "deviendr"},
		{"deviendriez", "deviendr"},
		{"deviennent", "deviennent"},
		{"deviens", "devien"},
		{"devient", "devient"},
		{"deviez", "dev"},
		{"devina", "devin"},
		{"devinait", "devin"},
		{"devinant", "devin"},
		{"devine", "devin"},
		{"deviné", "devin"},
		{"devinée", "devin"},
		{"devinées", "devin"},
		{"deviner", "devin"},
		{"devinera", "devin"},
		{"devinerait", "devin"},
		{"devineront", "devin"},
		{"devines", "devin"},
		{"devinez", "devin"},
		{"devînmes", "devînm"},
		{"devinrent", "devinrent"},
		{"devins", "devin"},
		{"devint", "devint"},
		{"devis", "dev"},
		{"dévisageait", "dévisag"},
		{"dévisager", "dévisag"},
		{"dévisser", "déviss"},
		{"devo", "devo"},
		{"dévoilé", "dévoil"},
		{"devoir", "devoir"},
		{"devoirs", "devoir"},
		{"devons", "devon"},
		{"dévora", "dévor"},
		{"dévoraient", "dévor"},
		{"dévorait", "dévor"},
		{"dévorante", "dévor"},
		{"dévorât", "dévor"},
		{"dévore", "dévor"},
		{"dévoré", "dévor"},
		{"dévorée", "dévor"},
		{"dévorer", "dévor"},
		{"dévoreront", "dévor"},
		{"dévorés", "dévor"},
		{"devoret", "devoret"},
		{"dévorez", "dévor"},
		{"dévot", "dévot"},
		{"dévote", "dévot"},
		{"dévotement", "dévot"},
		{"dévotes", "dévot"},
		{"dévotion", "dévot"},
		{"dévots", "dévot"},
		{"dévoué", "dévou"},
		{"dévouée", "dévou"},
		{"dévouement", "dévou"},
		{"dévouements", "dévou"},
		{"dévouer", "dévou"},
		{"dévoués", "dévou"},
		{"devra", "devr"},
		{"devrai", "devr"},
		{"devraient", "devr"},
		{"devrais", "devr"},
		{"devrait", "devr"},
		{"devriez", "devr"},
		{"devrons", "devron"},
		{"devront", "devront"},
		{"di", "di"},
		{"diable", "diabl"},
		{"diables", "diabl"},
		{"diacre", "diacr"},
		{"dialogue", "dialogu"},
		{"dialogues", "dialogu"},
		{"diamant", "diam"},
		{"diamanté", "diamant"},
		{"diamants", "diam"},
		{"diane", "dian"},
		{"diar", "diar"},
		{"dicta", "dict"},
		{"dictait", "dict"},
		{"dictateur", "dictateur"},
		{"dicté", "dict"},
		{"dictée", "dict"},
		{"dictées", "dict"},
		{"dicter", "dict"},
		{"dictés", "dict"},
		{"diction", "diction"},
		{"dictionnaire", "dictionnair"},
		{"dicton", "dicton"},
		{"diderot", "diderot"},
		{"didon", "didon"},
		{"diego", "diego"},
		{"diègue", "diègu"},
		{"dieu", "dieu"},
		{"dieux", "dieux"},
		{"différence", "différent"},
		{"différences", "différent"},
		{"différend", "différend"},
		{"différends", "différend"},
		{"différent", "différent"},
		{"diffèrent", "diff"},
		{"différente", "différent"},
		{"différentes", "différent"},
		{"différents", "différent"},
		{"différer", "differ"},
		{"difficile", "difficil"},
		{"difficilement", "difficil"},
		{"difficiles", "difficil"},
		{"difficulté", "difficult"},
		{"difficultés", "difficult"},
		{"difficultueux", "difficultu"},
		{"difforme", "difform"},
		{"digère", "diger"},
		{"digne", "dign"},
		{"dignement", "dign"},
		{"dignes", "dign"},
		{"dignité", "dignit"},
		{"dignités", "dignit"},
		{"digue", "digu"},
		{"dijon", "dijon"},
		{"dilettanti", "dilettant"},
		{"diligence", "diligent"},
		{"diligente", "diligent"},
		{"dimanche", "dimanch"},
		{"dimanches", "dimanch"},
		{"dîme", "dîm"},
		{"dimension", "dimens"},
		{"dimensions", "dimens"},
		{"diminua", "diminu"},
		{"diminuaient", "diminu"},
		{"diminuait", "diminu"},
		{"diminuant", "diminu"},
		{"diminue", "diminu"},
		{"diminué", "diminu"},
		{"diminuée", "diminu"},
		{"diminuer", "diminu"},
		{"diminution", "diminu"},
		{"dîna", "dîn"},
		{"dînait", "dîn"},
		{"dînant", "dîn"},
		{"dindon", "dindon"},
		{"dîné", "dîn"},
		{"dînée", "dîn"},
		{"dînées", "dîn"},
		{"dînent", "dînent"},
		{"dîner", "dîn"},
		{"dînerai", "dîn"},
		{"dînerait", "dîn"},
		{"dînèrent", "dîn"},
		{"dîners", "dîner"},
		{"diocèse", "dioces"},
		{"diplomate", "diplomat"},
		{"diplomates", "diplomat"},
		{"diplomatie", "diplomat"},
		{"diplomatique", "diplomat"},
		{"diplomatiquement", "diplomat"},
		{"diplomatiques", "diplomat"},
		{"dira", "dir"},
		{"dirai", "dir"},
		{"diraient", "dir"},
		{"dirais", "dir"},
		{"dirait", "dir"},
		{"diras", "dir"},
		{"dire", "dir"},
		{"direct", "direct"},
		{"directe", "direct"},
		{"directement", "direct"},
		{"directes", "direct"},
		{"directeur", "directeur"},
		{"direction", "direct"},
		{"directions", "direct"},
		{"directoire", "directoir"},
		{"directs", "direct"},
		{"dirent", "dirent"},
		{"direz", "dir"},
		{"dirige", "dirig"},
		{"dirigé", "dirig"},
		{"dirigea", "dirig"},
		{"dirigeait", "dirig"},
		{"dirigeant", "dirig"},
		{"dirigeante", "dirig"},
		{"dirigée", "dirig"},
		{"dirigées", "dirig"},
		{"dirigent", "dirigent"},
		{"diriger", "dirig"},
		{"dirigerait", "dirig"},
		{"dirigèrent", "dirig"},
		{"dirigés", "dirig"},
		{"dirigez", "dirig"},
		{"dirions", "dirion"},
		{"dirons", "diron"},
		{"diront", "diront"},
		{"dis", "dis"},
		{"disaient", "dis"},
		{"disais", "dis"},
		{"disait", "dis"},
		{"disant", "dis"},
		{"disc", "disc"},
		{"discerna", "discern"},
		{"disciple", "discipl"},
		{"disciples", "discipl"},
		{"discontinuaient", "discontinu"},
		{"discontinuer", "discontinu"},
		{"disconvenir", "disconven"},
		{"discordant", "discord"},
		{"discours", "discour"},
		{"discret", "discret"},
		{"discrète", "discret"},
		{"discrétion", "discret"},
		{"discrets", "discret"},
		{"discussion", "discuss"},
		{"discussions", "discuss"},
		{"discuta", "discut"},
		{"discutaient", "discut"},
		{"discutait", "discut"},
		{"discutant", "discut"},
		{"discute", "discut"},
		{"discutée", "discut"},
		{"discuter", "discut"},
		{"discutèrent", "discut"},
		{"discutés", "discut"},
		{"dise", "dis"},
		{"disent", "disent"},
		{"disgrâce", "disgrâc"},
		{"disgracie", "disgrac"},
		{"disgracié", "disgraci"},
		{"disgracierez", "disgraci"},
		{"disinvoltura", "disinvoltur"},
		{"disloqué", "disloqu"},
		{"disparaissaient", "disparaiss"},
		{"disparaissait", "disparaiss"},
		{"disparaissent", "disparaissent"},
		{"disparaît", "disparaît"},
		{"disparaître", "disparaîtr"},
		{"disparition", "disparit"},
		{"disparu", "disparu"},
		{"disparurent", "disparurent"},
		{"disparussent", "disparussent"},
		{"disparut", "disparut"},
		{"dispendieuses", "dispendi"},
		{"dispensé", "dispens"},
		{"dispenser", "dispens"},
		{"dispersaient", "dispers"},
		{"dispersés", "dispers"},
		{"dispos", "dispos"},
		{"disposa", "dispos"},
		{"disposaient", "dispos"},
		{"disposait", "dispos"},
		{"dispose", "dispos"},
		{"disposé", "dispos"},
		{"disposée", "dispos"},
		{"disposées", "dispos"},
		{"disposer", "dispos"},
		{"disposés", "dispos"},
		{"disposez", "dispos"},
		{"disposition", "disposit"},
		{"dispositions", "disposit"},
		{"disposons", "disposon"},
		{"disproportion", "disproport"},
		{"disputa", "disput"},
		{"disputaient", "disput"},
		{"disputait", "disput"},
		{"dispute", "disput"},
		{"disputer", "disput"},
		{"disputèrent", "disput"},
		{"disputons", "disputon"},
		{"disque", "disqu"},
		{"dissembles", "dissembl"},
		{"dissent", "dissent"},
		{"disséquée", "dissequ"},
		{"dissertation", "dissert"},
		{"dissertations", "dissert"},
		{"disserte", "dissert"},
		{"dissimulaient", "dissimul"},
		{"dissimulait", "dissimul"},
		{"dissimulant", "dissimul"},
		{"dissimule", "dissimul"},
		{"dissimulée", "dissimul"},
		{"dissimuler", "dissimul"},
		{"dissimulerai", "dissimul"},
		{"dissimulerons", "dissimul"},
		{"dissimulons", "dissimulon"},
		{"dissipait", "dissip"},
		{"dissipation", "dissip"},
		{"dissipations", "dissip"},
		{"dissipé", "dissip"},
		{"dissipée", "dissip"},
		{"dissipées", "dissip"},
		{"dissiper", "dissip"},
		{"dissipèrent", "dissip"},
		{"dissolution", "dissolu"},
		{"distance", "distanc"},
		{"distancé", "distanc"},
		{"distances", "distanc"},
		{"distant", "dist"},
		{"distillée", "distill"},
		{"distinctement", "distinct"},
		{"distinctes", "distinct"},
		{"distinction", "distinct"},
		{"distinctions", "distinct"},
		{"distincts", "distinct"},
		{"distinguaient", "distingu"},
		{"distinguait", "distingu"},
		{"distinguant", "distingu"},
		{"distingue", "distingu"},
		{"distingué", "distingu"},
		{"distinguée", "distingu"},
		{"distinguées", "distingu"},
		{"distinguer", "distingu"},
		{"distingués", "distingu"},
		{"distinguons", "distinguon"},
		{"distraction", "distract"},
		{"distractions", "distract"},
		{"distrairait", "distrair"},
		{"distraire", "distrair"},
		{"distrait", "distr"},
		{"distraite", "distrait"},
		{"distraites", "distrait"},
		{"distrayaient", "distrai"},
		{"distrayait", "distrai"},
		{"distribua", "distribu"},
		{"distribuait", "distribu"},
		{"distribue", "distribu"},
		{"distribué", "distribu"},
		{"distribuent", "distribuent"},
		{"distribuer", "distribu"},
		{"distribuez", "distribu"},
		{"distribution", "distribu"},
		{"distributions", "distribu"},
		{"distributive", "distribut"},
		{"dit", "dit"},
		{"dît", "dît"},
		{"dite", "dit"},
		{"dites", "dit"},
		{"dithyrambe", "dithyramb"},
		{"dits", "dit"},
		{"diurne", "diurn"},
		{"divan", "divan"},
		{"divers", "diver"},
		{"diverses", "divers"},
		{"diversion", "divers"},
		{"divertir", "divert"},
		{"divertissant", "divert"},
		{"divertissements", "divert"},
		{"divin", "divin"},
		{"divine", "divin"},
		{"divinité", "divin"},
		{"divins", "divin"},
		{"divisa", "divis"},
		{"divisait", "divis"},
		{"divise", "divis"},
		{"divisé", "divis"},
		{"divisée", "divis"},
		{"divisées", "divis"},
		{"divisèrent", "divis"},
		{"divisés", "divis"},
		{"division", "divis"},
		{"divisions", "divis"},
		{"divulguer", "divulgu"},
		{"dix", "dix"},
		{"dixerunt", "dixerunt"},
		{"dixième", "dixiem"},
		{"dizaine", "dizain"},
		{"do", "do"},
		{"docilité", "docil"},
		{"docks", "dock"},
		{"docteur", "docteur"},
		{"doctrine", "doctrin"},
		{"doctrines", "doctrin"},
		{"document", "docu"},
		{"documents", "docu"},
		{"dodge", "dodg"},
		{"doge", "dog"},
		{"dogmatiquement", "dogmat"},
		{"dogme", "dogm"},
		{"doigt", "doigt"},
		{"doigts", "doigt"},
		{"dois", "dois"},
		{"doit", "doit"},
		{"doivent", "doivent"},
		{"dolci", "dolc"},
		{"dôle", "dôl"},
		{"doléances", "doléanc"},
		{"dollar", "dollar"},
		{"dollars", "dollar"},
		{"dolman", "dolman"},
		{"domaine", "domain"},
		{"domaniale", "domanial"},
		{"dôme", "dôm"},
		{"domesticité", "domest"},
		{"domestique", "domest"},
		{"domestiqué", "domestiqu"},
		{"domestiques", "domest"},
		{"domicile", "domicil"},
		{"domina", "domin"},
		{"dominaient", "domin"},
		{"dominait", "domin"},
		{"dominant", "domin"},
		{"dominante", "domin"},
		{"dominateur", "domin"},
		{"domination", "domin"},
		{"domine", "domin"},
		{"dominé", "domin"},
		{"dominée", "domin"},
		{"domingue", "domingu"},
		{"domini", "domin"},
		{"dommage", "dommag"},
		{"dommages", "dommag"},
		{"dompté", "dompt"},
		{"dompter", "dompt"},
		{"don", "don"},
		{"donation", "donat"},
		{"donc", "donc"},
		{"dongo", "dongo"},
		{"donjon", "donjon"},
		{"donna", "don"},
		{"donnai", "don"},
		{"donnaient", "don"},
		{"donnait", "don"},
		{"donnant", "don"},
		{"donnassent", "don"},
		{"donnât", "don"},
		{"donnâtes", "don"},
		{"donne", "don"},
		{"donné", "don"},
		{"donnée", "don"},
		{"données", "don"},
		{"donnent", "donnent"},
		{"donner", "don"},
		{"donnera", "don"},
		{"donnerai", "don"},
		{"donneraient", "don"},
		{"donnerais", "don"},
		{"donnerait", "don"},
		{"donneras", "don"},
		{"donnèrent", "don"},
		{"donnerez", "don"},
		{"donneriez", "don"},
		{"donneront", "don"},
		{"donnes", "don"},
		{"donnés", "don"},
		{"donneur", "donneur"},
		{"donnez", "don"},
		{"donnions", "donnion"},
		{"donnons", "donnon"},
		{"dont", "dont"},
		{"dorat", "dorat"},
		{"doré", "dor"},
		{"dorée", "dor"},
		{"dorées", "dor"},
		{"dorénavant", "dorénav"},
		{"dorés", "dor"},
		{"dormaient", "dorm"},
		{"dormait", "dorm"},
		{"dormantes", "dorm"},
		{"dormeurs", "dormeur"},
		{"dormi", "dorm"},
		{"dormir", "dorm"},
		{"dormirai", "dorm"},
		{"dormit", "dorm"},
		{"dormît", "dorm"},
		{"dort", "dort"},
		{"dortoir", "dortoir"},
		{"dortoirs", "dortoir"},
		{"dorures", "dorur"},
		{"dos", "dos"},
		{"dose", "dos"},
		{"dossier", "dossi"},
		{"dossiers", "dossi"},
		{"dot", "dot"},
		{"douaire", "douair"},
		{"douairière", "douairi"},
		{"douane", "douan"},
		{"douanier", "douani"},
		{"douaniers", "douani"},
		{"doubla", "doubl"},
		{"doublant", "doubl"},
		{"double", "doubl"},
		{"doublé", "doubl"},
		{"doublée", "doubl"},
		{"doublement", "doubl"},
		{"doubler", "doubl"},
		{"doubles", "doubl"},
		{"doublure", "doublur"},
		{"doubs", "doub"},
		{"doubt", "doubt"},
		{"douce", "douc"},
		{"doucement", "douc"},
		{"douces", "douc"},
		{"douceur", "douceur"},
		{"douceurs", "douceur"},
		{"doué", "dou"},
		{"douée", "dou"},
		{"doués", "dou"},
		{"douleur", "douleur"},
		{"douleurs", "douleur"},
		{"douloureuse", "doulour"},
		{"douloureuses", "doulour"},
		{"douloureux", "doulour"},
		{"douta", "dout"},
		{"doutait", "dout"},
		{"doutât", "dout"},
		{"doute", "dout"},
		{"douté", "dout"},
		{"doutée", "dout"},
		{"douter", "dout"},
		{"douterez", "dout"},
		{"doutes", "dout"},
		{"douteuse", "douteux"},
		{"douteuses", "douteux"},
		{"douteux", "douteux"},
		{"doutez", "dout"},
		{"douvres", "douvr"},
		{"doux", "doux"},
		{"douzaine", "douzain"},
		{"douzaines", "douzain"},
		{"douze", "douz"},
		{"doyen", "doyen"},
		{"dragon", "dragon"},
		{"dragons", "dragon"},
		{"dramatique", "dramat"},
		{"dramatiques", "dramat"},
		{"drame", "dram"},
		{"drap", "drap"},
		{"drapeau", "drapeau"},
		{"drapeaux", "drapeau"},
		{"drapée", "drap"},
		{"drapier", "drapi"},
		{"draps", "drap"},
		{"drawing", "drawing"},
		{"dressaient", "dress"},
		{"dressait", "dress"},
		{"dresse", "dress"},
		{"dressé", "dress"},
		{"dressée", "dress"},
		{"dresser", "dress"},
		{"dressés", "dress"},
		{"drisses", "driss"},
		{"drogue", "drogu"},
		{"drogues", "drogu"},
		{"droit", "droit"},
		{"droite", "droit"},
		{"droites", "droit"},
		{"droits", "droit"},
		{"droiture", "droitur"},
		{"drôle", "drôl"},
		{"drôles", "drôl"},
		{"drome", "drom"},
		{"dromes", "drom"},
		{"du", "du"},
		{"dû", "dû"},
		{"dublin", "dublin"},
		{"dubois", "dubois"},
		{"duc", "duc"},
		{"ducats", "ducat"},
		{"duché", "duch"},
		{"duchesse", "duchess"},
		{"duchesses", "duchess"},
		{"ducrest", "ducrest"},
		{"ducros", "ducros"},
		{"ducs", "duc"},
		{"dudit", "dud"},
		{"due", "du"},
		{"duel", "duel"},
		{"duellistes", "duellist"},
		{"duettino", "duettino"},
		{"dugnani", "dugnan"},
		{"dûment", "dûment"},
		{"dunette", "dunet"},
		{"dupe", "dup"},
		{"duperie", "duper"},
		{"dupes", "dup"},
		{"duplicité", "dupliqu"},
		{"duquel", "duquel"},
		{"dur", "dur"},
		{"dura", "dur"},
		{"durable", "durabl"},
		{"durables", "durabl"},
		{"duraient", "dur"},
		{"durait", "dur"},
		{"durand", "durand"},
		{"durant", "dur"},
		{"durât", "dur"},
		{"durati", "durat"},
		{"durcie", "durc"},
		{"dure", "dur"},
		{"duré", "dur"},
		{"durée", "dur"},
		{"durement", "dur"},
		{"durent", "durent"},
		{"durer", "dur"},
		{"durera", "dur"},
		{"dureraient", "dur"},
		{"durerait", "dur"},
		{"dureront", "dur"},
		{"dures", "dur"},
		{"dureté", "duret"},
		{"durini", "durin"},
		{"durs", "dur"},
		{"dus", "dus"},
		{"dussé", "duss"},
		{"dut", "dut"},
		{"dût", "dût"},
		{"duvoisin", "duvoisin"},
		{"e", "e"},
		{"è", "è"},
		{"earnshaw", "earnshaw"},
		{"eastern", "eastern"},
		{"eau", "eau"},
		{"eaux", "eau"},
		{"ébahi", "ébah"},
		{"ébahie", "ébah"},
		{"ébahis", "ébah"},
		{"ébahissement", "ébah"},
		{"ébattre", "ébattr"},
		{"ébaucha", "ébauch"},
		{"ébaudissait", "ébaud"},
		{"ébène", "ében"},
		{"ébloui", "éblou"},
		{"éblouissant", "éblou"},
		{"éblouissante", "éblou"},
		{"éblouissement", "éblou"},
		{"éblouit", "éblou"},
		{"ébouriffé", "ébouriff"},
		{"ébouriffée", "ébouriff"},
		{"ébranla", "ébranl"},
		{"ébranlaient", "ébranl"},
		{"ébranlant", "ébranl"},
		{"ébranlé", "ébranl"},
		{"ébranlée", "ébranl"},
		{"ébranler", "ébranl"},
		{"ébréché", "ébrech"},
		{"ébruité", "ébruit"},
		{"écailles", "écaill"},
		{"écarlate", "écarlat"},
		{"écarquillés", "écarquill"},
		{"écart", "écart"},
		{"écartaient", "écart"},
		{"écartait", "écart"},
		{"écartant", "écart"},
		{"écarte", "écart"},
		{"écarté", "écart"},
		{"écartée", "écart"},
		{"écartées", "écart"},
		{"écartelé", "écartel"},
		{"écarter", "écart"},
		{"eccellenza", "eccellenz"},
		{"ecclesiam", "ecclesiam"},
		{"ecclésiastique", "ecclésiast"},
		{"ecclésiastiques", "ecclésiast"},
		{"écervelés", "écervel"},
		{"échafaud", "échafaud"},
		{"échange", "échang"},
		{"échangé", "échang"},
		{"échangeaient", "échang"},
		{"échangeant", "échang"},
		{"échangée", "échang"},
		{"échangées", "échang"},
		{"échanger", "échang"},
		{"échangés", "échang"},
		{"échantillons", "échantillon"},
		{"échappa", "échapp"},
		{"échappaient", "échapp"},
		{"échappait", "échapp"},
		{"échappe", "échapp"},
		{"échappé", "échapp"},
		{"échappée", "échapp"},
		{"échappées", "échapp"},
		{"échappement", "échapp"},
		{"échapper", "échapp"},
		{"échapperait", "échapp"},
		{"échappés", "échapp"},
		{"échappons", "échappon"},
		{"écharpe", "écharp"},
		{"échauffant", "échauff"},
		{"échauffer", "échauff"},
		{"échéant", "éché"},
		{"échec", "échec"},
		{"échecs", "échec"},
		{"échelle", "échel"},
		{"échelles", "échel"},
		{"échelon", "échelon"},
		{"échelons", "échelon"},
		{"échevelé", "échevel"},
		{"echiquier", "echiqui"},
		{"échiquiers", "échiqui"},
		{"échoiront", "échoiront"},
		{"échoppe", "échopp"},
		{"échouait", "échou"},
		{"échoue", "échou"},
		{"échoué", "échou"},
		{"échouer", "échou"},
		{"échoueraient", "échou"},
		{"échus", "échus"},
		{"échut", "échut"},
		{"éclair", "éclair"},
		{"éclairaient", "éclair"},
		{"éclairait", "éclair"},
		{"éclairant", "éclair"},
		{"éclaircies", "éclairc"},
		{"éclaircir", "éclairc"},
		{"éclaircirent", "éclairc"},
		{"éclaircissant", "éclairc"},
		{"éclaircissement", "éclairc"},
		{"éclaircissent", "éclairc"},
		{"éclaircit", "éclairc"},
		{"éclairé", "éclair"},
		{"éclairée", "éclair"},
		{"éclairer", "éclair"},
		{"éclairèrent", "éclair"},
		{"éclairés", "éclair"},
		{"éclairs", "éclair"},
		{"éclat", "éclat"},
		{"éclata", "éclat"},
		{"éclataient", "éclat"},
		{"éclatait", "éclat"},
		{"éclatant", "éclat"},
		{"éclatante", "éclat"},
		{"éclatantes", "éclat"},
		{"éclatants", "éclat"},
		{"éclate", "éclat"},
		{"éclaté", "éclat"},
		{"éclatent", "éclatent"},
		{"éclater", "éclat"},
		{"éclatèrent", "éclat"},
		{"éclats", "éclat"},
		{"éclipsait", "éclips"},
		{"éclipse", "éclips"},
		{"éclipses", "éclips"},
		{"éclipsés", "éclips"},
		{"école", "écol"},
		{"écolier", "écoli"},
		{"éconduire", "éconduir"},
		{"éconduit", "éconduit"},
		{"économe", "économ"},
		{"économes", "économ"},
		{"économie", "économ"},
		{"économies", "économ"},
		{"économise", "économis"},
		{"économisées", "économis"},
		{"économiser", "économis"},
		{"économisés", "économis"},
		{"écorcha", "écorch"},
		{"écorchaient", "écorch"},
		{"écorchait", "écorch"},
		{"écorché", "écorch"},
		{"écorchée", "écorch"},
		{"écorcher", "écorch"},
		{"écorchures", "écorchur"},
		{"écorné", "écorn"},
		{"écossaise", "écossais"},
		{"écoula", "écoul"},
		{"écoulaient", "écoul"},
		{"écoulant", "écoul"},
		{"écoulé", "écoul"},
		{"écoulée", "écoul"},
		{"écoulement", "écoul"},
		{"écoulent", "écoulent"},
		{"écouler", "écoul"},
		{"écoulèrent", "écoul"},
		{"écoulés", "écoul"},
		{"écouta", "écout"},
		{"écoutaient", "écout"},
		{"écoutais", "écout"},
		{"écoutait", "écout"},
		{"écoutant", "écout"},
		{"écoute", "écout"},
		{"écouté", "écout"},
		{"écoutée", "écout"},
		{"écoutent", "écoutent"},
		{"écouter", "écout"},
		{"écouterai", "écout"},
		{"écouteront", "écout"},
		{"écoutes", "écout"},
		{"écoutés", "écout"},
		{"ecoutez", "ecout"},
		{"écoutez", "écout"},
		{"écoutiez", "écout"},
		{"écrasaient", "écras"},
		{"écrasé", "écras"},
		{"écrasée", "écras"},
		{"écrasées", "écras"},
		{"écraser", "écras"},
		{"écria", "écri"},
		{"écriaient", "écri"},
		{"écriait", "écri"},
		{"écriant", "écri"},
		{"écrie", "écri"},
		{"écrié", "écri"},
		{"écriée", "écri"},
		{"écrient", "écrient"},
		{"écrier", "écri"},
		{"écrièrent", "écri"},
		{"écrin", "écrin"},
		{"écrira", "écrir"},
		{"écrirai", "écrir"},
		{"écrirais", "écrir"},
		{"écrirait", "écrir"},
		{"écrire", "écrir"},
		{"écrirez", "écrir"},
		{"écris", "écris"},
		{"écrit", "écrit"},
		{"écrite", "écrit"},
		{"écriteau", "écriteau"},
		{"écrites", "écrit"},
		{"écritoire", "écritoir"},
		{"écrits", "écrit"},
		{"écriture", "écritur"},
		{"écritures", "écritur"},
		{"écrivailleur", "écrivailleur"},
		{"écrivailleurs", "écrivailleur"},
		{"écrivain", "écrivain"},
		{"écrivains", "écrivain"},
		{"écrivais", "écriv"},
		{"écrivait", "écriv"},
		{"écrivant", "écriv"},
		{"écrivante", "écriv"},
		{"écrive", "écriv"},
		{"écrivez", "écriv"},
		{"écrivirent", "écriv"},
		{"écrivit", "écriv"},
		{"écrivît", "écriv"},
		{"écrivîtes", "écriv"},
		{"écrou", "écrou"},
		{"écroué", "écrou"},
		{"écroula", "écroul"},
		{"écroulaient", "écroul"},
		{"écu", "écu"},
		{"écueil", "écueil"},
		{"écuelle", "écuel"},
		{"écumait", "écum"},
		{"ecumeur", "ecumeur"},
		{"écumeuses", "écum"},
		{"écureuil", "écureuil"},
		{"écurie", "écur"},
		{"écuries", "écur"},
		{"écus", "écus"},
		{"écuyer", "écui"},
		{"édification", "édif"},
		{"édifice", "édific"},
		{"edimbourg", "edimbourg"},
		{"édimbourg", "édimbourg"},
		{"edinburgh", "edinburgh"},
		{"éditer", "édit"},
		{"éditeur", "éditeur"},
		{"édition", "édit"},
		{"éditions", "édit"},
		{"edrisi", "edris"},
		{"éducation", "éduc"},
		{"éduquer", "éduqu"},
		{"effaça", "effac"},
		{"efface", "effac"},
		{"effacé", "effac"},
		{"effacée", "effac"},
		{"effacées", "effac"},
		{"effacer", "effac"},
		{"effaré", "effar"},
		{"effarouchant", "effarouch"},
		{"effarouche", "effarouch"},
		{"effarouché", "effarouch"},
		{"effaroucher", "effarouch"},
		{"effarouches", "effarouch"},
		{"effarouchés", "effarouch"},
		{"effectivement", "effect"},
		{"effectives", "effect"},
		{"efféminées", "effémin"},
		{"effervescence", "effervescent"},
		{"effet", "effet"},
		{"effets", "effet"},
		{"effeuilla", "effeuill"},
		{"effigie", "effig"},
		{"effleurer", "effleur"},
		{"efforçait", "efforc"},
		{"effort", "effort"},
		{"efforts", "effort"},
		{"effraie", "effrai"},
		{"effraya", "effrai"},
		{"effrayaient", "effrai"},
		{"effrayait", "effrai"},
		{"effrayant", "effrai"},
		{"effrayante", "effrai"},
		{"effraye", "effray"},
		{"effrayé", "effrai"},
		{"effrayée", "effrai"},
		{"effrayées", "effrai"},
		{"effrayent", "effrayent"},
		{"effrayer", "effrai"},
		{"effrayèrent", "effrai"},
		{"effrayés", "effrai"},
		{"effréné", "effren"},
		{"effrénée", "effren"},
		{"effroi", "effroi"},
		{"effronté", "effront"},
		{"effrontément", "effront"},
		{"effronterie", "effronter"},
		{"effrontés", "effront"},
		{"effroyable", "effroi"},
		{"effroyablement", "effroi"},
		{"effusion", "effus"},
		{"égaiera", "égai"},
		{"égal", "égal"},
		{"égalait", "égal"},
		{"égale", "égal"},
		{"également", "égal"},
		{"égalent", "égalent"},
		{"égales", "égal"},
		{"égalité", "égal"},
		{"égalités", "égal"},
		{"égara", "égar"},
		{"égarait", "égar"},
		{"égard", "égard"},
		{"égards", "égard"},
		{"égare", "égar"},
		{"égaré", "égar"},
		{"égarée", "égar"},
		{"égarement", "égar"},
		{"égarements", "égar"},
		{"égarer", "égar"},
		{"égares", "égar"},
		{"égarés", "égar"},
		{"égaux", "égal"},
		{"égaya", "égai"},
		{"égayée", "égai"},
		{"égayer", "égai"},
		{"égayés", "égai"},
		{"église", "églis"},
		{"églises", "églis"},
		{"ego", "ego"},
		{"égoïsme", "égoïsm"},
		{"égoïste", "égoïst"},
		{"égoïstes", "égoïst"},
		{"égorgent", "égorgent"},
		{"égorger", "égorg"},
		{"égorgés", "égorg"},
		{"égout", "égout"},
		{"égratignure", "égratignur"},
		{"égratignures", "égratignur"},
		{"egypte", "egypt"},
		{"égypte", "égypt"},
		{"egyptiens", "egyptien"},
		{"égyptiens", "égyptien"},
		{"eh", "eh"},
		{"el", "el"},
		{"élan", "élan"},
		{"élança", "élanc"},
		{"élançait", "élanc"},
		{"élançant", "élanc"},
		{"élance", "élanc"},
		{"élancé", "élanc"},
		{"élancer", "élanc"},
		{"élancera", "élanc"},
		{"élans", "élan"},
		{"élargi", "élarg"},
		{"élastiques", "élast"},
		{"elder", "elder"},
		{"électeur", "électeur"},
		{"électeurs", "électeur"},
		{"élection", "élect"},
		{"élections", "élect"},
		{"électives", "élect"},
		{"électrique", "électr"},
		{"électriques", "électr"},
		{"électrisait", "électris"},
		{"élégamment", "éleg"},
		{"élégance", "éleg"},
		{"élégant", "éleg"},
		{"élégante", "éleg"},
		{"élégantes", "éleg"},
		{"élégants", "éleg"},
		{"élémentaire", "élémentair"},
		{"éléments", "élément"},
		{"éléphant", "éleph"},
		{"eléphanta", "eléphant"},
		{"éléphants", "éleph"},
		{"éleva", "élev"},
		{"élevaient", "élev"},
		{"élevait", "élev"},
		{"élevant", "élev"},
		{"élévation", "élev"},
		{"élévations", "élev"},
		{"élève", "élev"},
		{"élevé", "élev"},
		{"élevée", "élev"},
		{"élevées", "élev"},
		{"élèvent", "élèvent"},
		{"élever", "élev"},
		{"élèvera", "élev"},
		{"élèverai", "élev"},
		{"élèverez", "élev"},
		{"élèves", "élev"},
		{"élevés", "élev"},
		{"éleveurs", "éleveur"},
		{"élisa", "élis"},
		{"élite", "élit"},
		{"elko", "elko"},
		{"elle", "elle"},
		{"elles", "elle"},
		{"ellora", "ellor"},
		{"élocution", "élocu"},
		{"éloge", "élog"},
		{"éloges", "élog"},
		{"éloigna", "éloign"},
		{"éloignaient", "éloign"},
		{"éloignais", "éloign"},
		{"éloignait", "éloign"},
		{"éloignant", "éloign"},
		{"éloignât", "éloign"},
		{"éloigne", "éloign"},
		{"éloigné", "éloign"},
		{"éloignée", "éloign"},
		{"éloignées", "éloign"},
		{"éloignement", "éloign"},
		{"éloigner", "éloign"},
		{"éloignerai", "éloign"},
		{"éloignerait", "éloign"},
		{"éloignèrent", "éloign"},
		{"éloignés", "éloign"},
		{"éloignez", "éloign"},
		{"éloigniez", "éloign"},
		{"éloquence", "éloquent"},
		{"éloquent", "éloquent"},
		{"éloquente", "éloquent"},
		{"élu", "élu"},
		{"éluder", "élud"},
		{"elysées", "elys"},
		{"élysées", "élys"},
		{"émaillée", "émaill"},
		{"emballée", "emball"},
		{"emballer", "emball"},
		{"embarcation", "embarc"},
		{"embarcations", "embarc"},
		{"embardée", "embard"},
		{"embardées", "embard"},
		{"embarqua", "embarqu"},
		{"embarquait", "embarqu"},
		{"embarquant", "embarqu"},
		{"embarque", "embarqu"},
		{"embarqué", "embarqu"},
		{"embarquement", "embarqu"},
		{"embarquer", "embarqu"},
		{"embarqués", "embarqu"},
		{"embarras", "embarr"},
		{"embarrassa", "embarrass"},
		{"embarrassait", "embarrass"},
		{"embarrasse", "embarr"},
		{"embarrassé", "embarrass"},
		{"embarrassée", "embarrass"},
		{"embarrasser", "embarrass"},
		{"embarrassés", "embarrass"},
		{"embaumé", "embaum"},
		{"embaumée", "embaum"},
		{"embelli", "embel"},
		{"embellie", "embel"},
		{"embellir", "embel"},
		{"emblée", "emblé"},
		{"emblème", "emblem"},
		{"embonpoint", "embonpoint"},
		{"embossent", "embossent"},
		{"embouchure", "embouchur"},
		{"embouquait", "embouqu"},
		{"embourbant", "embourb"},
		{"embourser", "embours"},
		{"embranchement", "embranch"},
		{"embranchements", "embranch"},
		{"embrasé", "embras"},
		{"embrasée", "embras"},
		{"embrassa", "embrass"},
		{"embrassaient", "embrass"},
		{"embrassait", "embrass"},
		{"embrassant", "embrass"},
		{"embrasse", "embrass"},
		{"embrassé", "embrass"},
		{"embrassements", "embrass"},
		{"embrasser", "embrass"},
		{"embrasserais", "embrass"},
		{"embrassèrent", "embrass"},
		{"embrouilla", "embrouill"},
		{"embrouillant", "embrouill"},
		{"embrouille", "embrouill"},
		{"embrouiller", "embrouill"},
		{"embrumé", "embrum"},
		{"embrunir", "embrun"},
		{"embruns", "embrun"},
		{"embûche", "embûch"},
		{"embûches", "embûch"},
		{"embuscade", "embuscad"},
		{"émergeait", "émerg"},
		{"émerveillé", "émerveil"},
		{"émerveillée", "émerveil"},
		{"émettre", "émettr"},
		{"émeute", "émeut"},
		{"émeuvent", "émeuvent"},
		{"émigrants", "émigr"},
		{"émigration", "émigr"},
		{"émigrera", "émigr"},
		{"éminemment", "éminent"},
		{"émissaire", "émissair"},
		{"emmancher", "emmanch"},
		{"emmêlés", "emmêl"},
		{"emmena", "emmen"},
		{"emmenaient", "emmen"},
		{"emmenait", "emmen"},
		{"emmène", "emmen"},
		{"emmené", "emmen"},
		{"emmener", "emmen"},
		{"emmènerait", "emmen"},
		{"emmènes", "emmen"},
		{"émoi", "émoi"},
		{"émotion", "émot"},
		{"émotions", "émot"},
		{"émoussés", "émouss"},
		{"émoustiller", "émoustill"},
		{"émouvoir", "émouvoir"},
		{"empaquetée", "empaquet"},
		{"empara", "empar"},
		{"emparaient", "empar"},
		{"emparait", "empar"},
		{"emparant", "empar"},
		{"emparé", "empar"},
		{"emparer", "empar"},
		{"emparèrent", "empar"},
		{"emparés", "empar"},
		{"empaumer", "empaum"},
		{"empêcha", "empêch"},
		{"empêchaient", "empêch"},
		{"empêchait", "empêch"},
		{"empêchât", "empêch"},
		{"empêche", "empêch"},
		{"empêché", "empêch"},
		{"empêchée", "empêch"},
		{"empêchent", "empêchent"},
		{"empêcher", "empêch"},
		{"empêchera", "empêch"},
		{"empêcherai", "empêch"},
		{"empêcheraient", "empêch"},
		{"empêcherait", "empêch"},
		{"empêchèrent", "empêch"},
		{"empêcheront", "empêch"},
		{"empêches", "empêch"},
		{"empêchez", "empêch"},
		{"empereur", "empereur"},
		{"empereurs", "empereur"},
		{"empesé", "empes"},
		{"empestée", "empest"},
		{"empêtrés", "empêtr"},
		{"emphase", "emphas"},
		{"emphatique", "emphat"},
		{"emphatiques", "emphat"},
		{"empilé", "empil"},
		{"empir", "empir"},
		{"empire", "empir"},
		{"empires", "empir"},
		{"emplacement", "emplac"},
		{"emplettes", "emplet"},
		{"emplissent", "emplissent"},
		{"emploi", "emploi"},
		{"emploie", "emploi"},
		{"emploient", "emploient"},
		{"emploierai", "emploi"},
		{"emploieraient", "emploi"},
		{"emploierais", "emploi"},
		{"emploierait", "emploi"},
		{"emplois", "emplois"},
		{"employa", "emploi"},
		{"employaient", "emploi"},
		{"employait", "emploi"},
		{"employant", "emploi"},
		{"employât", "emploi"},
		{"employâtes", "emploi"},
		{"employé", "emploi"},
		{"employée", "emploi"},
		{"employées", "emploi"},
		{"employer", "emploi"},
		{"employés", "emploi"},
		{"employez", "emploi"},
		{"emplumés", "emplum"},
		{"empoché", "empoch"},
		{"empoigné", "empoign"},
		{"empoignera", "empoign"},
		{"empoisonna", "empoison"},
		{"empoisonnaient", "empoison"},
		{"empoisonnait", "empoison"},
		{"empoisonne", "empoison"},
		{"empoisonné", "empoison"},
		{"empoisonnée", "empoison"},
		{"empoisonnement", "empoison"},
		{"empoisonnent", "empoisonnent"},
		{"empoisonner", "empoison"},
		{"empoisonnerai", "empoison"},
		{"empoisonnerait", "empoison"},
		{"empoisonnés", "empoison"},
		{"empoisonneur", "empoisonneur"},
		{"empoisonneurs", "empoisonneur"},
		{"emporta", "emport"},
		{"emportaient", "emport"},
		{"emportait", "emport"},
		{"emportant", "emport"},
		{"emporte", "emport"},
		{"emporté", "emport"},
		{"emportée", "emport"},
		{"emportées", "emport"},
		{"emportement", "emport"},
		{"emportent", "emportent"},
		{"emporter", "emport"},
		{"emporterait", "emport"},
		{"emporterez", "emport"},
		{"emporteront", "emport"},
		{"emportés", "emport"},
		{"empreinte", "empreint"},
		{"empressa", "empress"},
		{"empressé", "empress"},
		{"empressée", "empress"},
		{"empressées", "empress"},
		{"empressement", "empress"},
		{"empressements", "empress"},
		{"empressèrent", "empress"},
		{"empressés", "empress"},
		{"emprisonne", "emprison"},
		{"emprisonné", "emprison"},
		{"emprisonnée", "emprison"},
		{"emprisonnement", "emprison"},
		{"emprunt", "emprunt"},
		{"emprunta", "emprunt"},
		{"emprunté", "emprunt"},
		{"emprunter", "emprunt"},
		{"empruntés", "emprunt"},
		{"ému", "ému"},
		{"émue", "ému"},
		{"émule", "émul"},
		{"émurent", "émurent"},
		{"émus", "émus"},
		{"émut", "émut"},
		{"en", "en"},
		{"encablure", "encablur"},
		{"encadre", "encadr"},
		{"encadré", "encadr"},
		{"encadrées", "encadr"},
		{"enceinte", "enceint"},
		{"encens", "encen"},
		{"enchaîné", "enchaîn"},
		{"enchaînée", "enchaîn"},
		{"enchaînés", "enchaîn"},
		{"enchantait", "enchant"},
		{"enchanté", "enchant"},
		{"enchantée", "enchant"},
		{"enchantement", "enchant"},
		{"enchanter", "enchant"},
		{"enchanteresses", "enchanteress"},
		{"enchantés", "enchant"},
		{"enchanteur", "enchanteur"},
		{"enchanteurs", "enchanteur"},
		{"enchâsser", "enchâss"},
		{"enchère", "encher"},
		{"enchères", "encher"},
		{"enclos", "enclos"},
		{"enclume", "enclum"},
		{"encolure", "encolur"},
		{"encombraient", "encombr"},
		{"encombrant", "encombr"},
		{"encombrante", "encombr"},
		{"encombre", "encombr"},
		{"encombré", "encombr"},
		{"encombrée", "encombr"},
		{"encombrées", "encombr"},
		{"encombrés", "encombr"},
		{"encore", "encor"},
		{"encouragé", "encourag"},
		{"encourageante", "encourag"},
		{"encouragement", "encourag"},
		{"encourager", "encourag"},
		{"encouragés", "encourag"},
		{"encourir", "encour"},
		{"encouru", "encouru"},
		{"encrassaient", "encrass"},
		{"encre", "encre"},
		{"encroûtées", "encroût"},
		{"encyclopédie", "encycloped"},
		{"endoctriné", "endoctrin"},
		{"endommagé", "endommag"},
		{"endormi", "endorm"},
		{"endormir", "endorm"},
		{"endormis", "endorm"},
		{"endormit", "endorm"},
		{"endors", "endor"},
		{"endossé", "endoss"},
		{"endosser", "endoss"},
		{"endroit", "endroit"},
		{"endroits", "endroit"},
		{"enduisirent", "enduis"},
		{"enduraient", "endur"},
		{"endurant", "endur"},
		{"endurci", "endurc"},
		{"endurcie", "endurc"},
		{"endurées", "endur"},
		{"énergie", "énerg"},
		{"énergique", "énerg"},
		{"énergumène", "énergumen"},
		{"énergumènes", "énergumen"},
		{"énervé", "énerv"},
		{"énervée", "énerv"},
		{"énerver", "énerv"},
		{"enfance", "enfanc"},
		{"enfant", "enfant"},
		{"enfantillage", "enfantillag"},
		{"enfantillages", "enfantillag"},
		{"enfantine", "enfantin"},
		{"enfantines", "enfantin"},
		{"enfants", "enfant"},
		{"enfer", "enfer"},
		{"enferma", "enferm"},
		{"enfermais", "enferm"},
		{"enfermait", "enferm"},
		{"enfermant", "enferm"},
		{"enferme", "enferm"},
		{"enfermé", "enferm"},
		{"enfermée", "enferm"},
		{"enfermer", "enferm"},
		{"enfermèrent", "enferm"},
		{"enfermés", "enferm"},
		{"enferrer", "enferr"},
		{"enfers", "enfer"},
		{"enfield", "enfield"},
		{"enfin", "enfin"},
		{"enflamma", "enflamm"},
		{"enflammé", "enflamm"},
		{"enflammée", "enflamm"},
		{"enflammées", "enflamm"},
		{"enflammer", "enflamm"},
		{"enflammèrent", "enflamm"},
		{"enflammés", "enflamm"},
		{"enflée", "enflé"},
		{"enflées", "enflé"},
		{"enfonça", "enfonc"},
		{"enfonçaient", "enfonc"},
		{"enfonçait", "enfonc"},
		{"enfonçât", "enfonc"},
		{"enfonce", "enfonc"},
		{"enfoncer", "enfonc"},
		{"enfonceront", "enfonc"},
		{"enfouis", "enfou"},
		{"enfourcher", "enfourch"},
		{"enfui", "enfui"},
		{"enfuie", "enfui"},
		{"enfuir", "enfuir"},
		{"enfuirai", "enfuir"},
		{"enfuis", "enfuis"},
		{"enfuit", "enfuit"},
		{"enfumé", "enfum"},
		{"enfuyaient", "enfui"},
		{"enfuyait", "enfui"},
		{"engage", "engag"},
		{"engagé", "engag"},
		{"engagea", "engag"},
		{"engageaient", "engag"},
		{"engageait", "engag"},
		{"engageant", "engag"},
		{"engagée", "engag"},
		{"engagées", "engag"},
		{"engagement", "engag"},
		{"engagements", "engag"},
		{"engagent", "engagent"},
		{"engager", "engag"},
		{"engagerai", "engag"},
		{"engeance", "engeanc"},
		{"engendre", "engendr"},
		{"engine", "engin"},
		{"england", "england"},
		{"englishman", "englishman"},
		{"engloutir", "englout"},
		{"engoués", "engou"},
		{"engouffrées", "engouffr"},
		{"engourdis", "engourd"},
		{"engourdissement", "engourd"},
		{"énigmatique", "énigmat"},
		{"énigme", "énigm"},
		{"enivraient", "enivr"},
		{"enivré", "enivr"},
		{"enivrée", "enivr"},
		{"enivrées", "enivr"},
		{"enivrer", "enivr"},
		{"enjambée", "enjamb"},
		{"enjeu", "enjeu"},
		{"enjouement", "enjou"},
		{"enleva", "enlev"},
		{"enlevaient", "enlev"},
		{"enlevait", "enlev"},
		{"enlevât", "enlev"},
		{"enlève", "enlev"},
		{"enlevé", "enlev"},
		{"enlevée", "enlev"},
		{"enlevées", "enlev"},
		{"enlèvement", "enlev"},
		{"enlèvent", "enlèvent"},
		{"enlever", "enlev"},
		{"enlèverais", "enlev"},
		{"enlevèrent", "enlev"},
		{"enlèverez", "enlev"},
		{"enlevez", "enlev"},
		{"enluminée", "enlumin"},
		{"ennemi", "ennem"},
		{"ennemie", "ennem"},
		{"ennemis", "ennem"},
		{"ennius", "ennius"},
		{"ennnui", "ennnui"},
		{"ennui", "ennui"},
		{"ennuie", "ennui"},
		{"ennuient", "ennuient"},
		{"ennuiera", "ennui"},
		{"ennuierais", "ennui"},
		{"ennuierait", "ennui"},
		{"ennuieront", "ennui"},
		{"ennuies", "ennui"},
		{"ennuis", "ennuis"},
		{"ennuya", "ennui"},
		{"ennuyaient", "ennui"},
		{"ennuyais", "ennui"},
		{"ennuyait", "ennui"},
		{"ennuyant", "ennui"},
		{"ennuyé", "ennui"},
		{"ennuyée", "ennui"},
		{"ennuyer", "ennui"},
		{"ennuyés", "ennui"},
		{"ennuyeuse", "ennui"},
		{"ennuyeusement", "ennui"},
		{"ennuyeuses", "ennui"},
		{"ennuyeux", "ennui"},
		{"ennuyez", "ennui"},
		{"énonçait", "énonc"},
		{"énoncé", "énonc"},
		{"énoncer", "énonc"},
		{"enorgueilli", "enorgueil"},
		{"énorme", "énorm"},
		{"énormes", "énorm"},
		{"énormité", "énorm"},
		{"enquérait", "enquer"},
		{"enquête", "enquêt"},
		{"enraciner", "enracin"},
		{"enragé", "enrag"},
		{"enragés", "enrag"},
		{"enrayer", "enrai"},
		{"enrayés", "enrai"},
		{"enrégimentés", "enrégiment"},
		{"enregistrement", "enregistr"},
		{"enregistrer", "enregistr"},
		{"enrhumaient", "enrhum"},
		{"enrhumé", "enrhum"},
		{"enrichi", "enrich"},
		{"enrichir", "enrich"},
		{"enrichis", "enrich"},
		{"enrichissait", "enrich"},
		{"enrichissent", "enrich"},
		{"enrichit", "enrich"},
		{"enrouaient", "enrou"},
		{"enroulait", "enroul"},
		{"enroulant", "enroul"},
		{"ensachés", "ensach"},
		{"ensanglanté", "ensanglant"},
		{"enseignait", "enseign"},
		{"enseigne", "enseign"},
		{"enseigné", "enseign"},
		{"enseignées", "enseign"},
		{"enseignements", "enseign"},
		{"enseignent", "enseignent"},
		{"enseigner", "enseign"},
		{"enseignera", "enseign"},
		{"ensemble", "ensembl"},
		{"enserrer", "enserr"},
		{"ensevelir", "ensevel"},
		{"ensorcela", "ensorcel"},
		{"ensuit", "ensuit"},
		{"ensuite", "ensuit"},
		{"ensuivit", "ensuiv"},
		{"entachés", "entach"},
		{"entama", "entam"},
		{"entame", "entam"},
		{"entamée", "entam"},
		{"entamer", "entam"},
		{"entasse", "entass"},
		{"entassée", "entass"},
		{"entassent", "entassent"},
		{"entend", "entend"},
		{"entendaient", "entend"},
		{"entendais", "entend"},
		{"entendait", "entend"},
		{"entendant", "entend"},
		{"entendent", "entendent"},
		{"entendez", "entend"},
		{"entendirent", "entend"},
		{"entendit", "entend"},
		{"entendît", "entend"},
		{"entendra", "entendr"},
		{"entendrait", "entendr"},
		{"entendre", "entendr"},
		{"entendrez", "entendr"},
		{"entendrons", "entendron"},
		{"entends", "entend"},
		{"entendu", "entendu"},
		{"entendue", "entendu"},
		{"entendus", "entendus"},
		{"enterrer", "enterr"},
		{"enterrez", "enterr"},
		{"entêté", "entêt"},
		{"enthousiasmait", "enthousiasm"},
		{"enthousiasme", "enthousiasm"},
		{"enthousiasmé", "enthousiasm"},
		{"enthousiasmées", "enthousiasm"},
		{"enthousiaste", "enthousiast"},
		{"enthousiastes", "enthousiast"},
		{"entiché", "entich"},
		{"entier", "enti"},
		{"entière", "entier"},
		{"entièrement", "entier"},
		{"entières", "entier"},
		{"entiers", "entier"},
		{"entomologique", "entomolog"},
		{"entonnant", "enton"},
		{"entonné", "enton"},
		{"entonner", "enton"},
		{"entortiller", "entortill"},
		{"entortillés", "entortill"},
		{"entour", "entour"},
		{"entoura", "entour"},
		{"entouraient", "entour"},
		{"entourait", "entour"},
		{"entoure", "entour"},
		{"entouré", "entour"},
		{"entourée", "entour"},
		{"entourées", "entour"},
		{"entourent", "entourent"},
		{"entourés", "entour"},
		{"entra", "entra"},
		{"entrai", "entrai"},
		{"entraient", "entraient"},
		{"entrailles", "entraill"},
		{"entraîna", "entraîn"},
		{"entraînait", "entraîn"},
		{"entraînant", "entraîn"},
		{"entraîne", "entraîn"},
		{"entraîné", "entraîn"},
		{"entraînée", "entraîn"},
		{"entraînement", "entraîn"},
		{"entraîner", "entraîn"},
		{"entraînera", "entraîn"},
		{"entraînés", "entraîn"},
		{"entrait", "entrait"},
		{"entrant", "entrant"},
		{"entraves", "entrav"},
		{"entre", "entre"},
		{"entré", "entré"},
		{"entrecoupée", "entrecoup"},
		{"entrecoupées", "entrecoup"},
		{"entrée", "entré"},
		{"entrées", "entré"},
		{"entrefaites", "entrefait"},
		{"entrelacement", "entrelac"},
		{"entremêlées", "entremêl"},
		{"entremêlés", "entremêl"},
		{"entremets", "entremet"},
		{"entremettre", "entremettr"},
		{"entremise", "entremis"},
		{"entrent", "entrent"},
		{"entrepôts", "entrepôt"},
		{"entreprenait", "entrepren"},
		{"entreprenant", "entrepren"},
		{"entreprend", "entreprend"},
		{"entreprendre", "entreprendr"},
		{"entrepreneur", "entrepreneur"},
		{"entrepris", "entrepr"},
		{"entreprise", "entrepris"},
		{"entreprises", "entrepris"},
		{"entreprit", "entrepr"},
		{"entrer", "entrer"},
		{"entreraient", "entrer"},
		{"entreras", "entrer"},
		{"entrèrent", "entrèrent"},
		{"entrés", "entré"},
		{"entresol", "entresol"},
		{"entretenait", "entreten"},
		{"entretenir", "entreten"},
		{"entretenu", "entretenu"},
		{"entretenus", "entretenus"},
		{"entretien", "entretien"},
		{"entretiens", "entretien"},
		{"entretient", "entretient"},
		{"entretinrent", "entretinrent"},
		{"entretint", "entretint"},
		{"entrevit", "entrev"},
		{"entrevoir", "entrevoir"},
		{"entrevois", "entrevois"},
		{"entrevoit", "entrevoit"},
		{"entrevoyaient", "entrevoi"},
		{"entrevoyait", "entrevoi"},
		{"entrevoyant", "entrevoi"},
		{"entrevu", "entrevu"},
		{"entrevue", "entrevu"},
		{"entrevues", "entrevu"},
		{"entrez", "entrez"},
		{"entrons", "entron"},
		{"entrouverte", "entrouvert"},
		{"entrouvrant", "entrouvr"},
		{"énumération", "énumer"},
		{"envahi", "envah"},
		{"envahie", "envah"},
		{"envahir", "envah"},
		{"envahissaient", "envah"},
		{"envahit", "envah"},
		{"enveloppaient", "envelopp"},
		{"enveloppait", "envelopp"},
		{"enveloppe", "envelopp"},
		{"enveloppé", "envelopp"},
		{"enveloppée", "envelopp"},
		{"envelopper", "envelopp"},
		{"enveloppes", "envelopp"},
		{"envenimée", "envenim"},
		{"enverguait", "envergu"},
		{"envergure", "envergur"},
		{"enverra", "enverr"},
		{"enverrai", "enverr"},
		{"enverrais", "enverr"},
		{"enverras", "enverr"},
		{"enverrez", "enverr"},
		{"envers", "enver"},
		{"envi", "envi"},
		{"enviable", "enviabl"},
		{"enviait", "envi"},
		{"enviant", "envi"},
		{"envie", "envi"},
		{"envié", "envi"},
		{"enviée", "envi"},
		{"envieux", "envieux"},
		{"environ", "environ"},
		{"environnaient", "environ"},
		{"environnait", "environ"},
		{"environnantes", "environ"},
		{"environne", "environ"},
		{"environné", "environ"},
		{"environnée", "environ"},
		{"environnent", "environnent"},
		{"environner", "environ"},
		{"environnés", "environ"},
		{"environs", "environ"},
		{"envisageons", "envisageon"},
		{"envisager", "envisag"},
		{"envisagez", "envisag"},
		{"envoi", "envoi"},
		{"envoie", "envoi"},
		{"envola", "envol"},
		{"envolant", "envol"},
		{"envole", "envol"},
		{"envolées", "envol"},
		{"envoler", "envol"},
		{"envolés", "envol"},
		{"envoya", "envoi"},
		{"envoyaient", "envoi"},
		{"envoyait", "envoi"},
		{"envoyant", "envoi"},
		{"envoyât", "envoi"},
		{"envoyâtes", "envoi"},
		{"envoyé", "envoi"},
		{"envoyée", "envoi"},
		{"envoyées", "envoi"},
		{"envoyer", "envoi"},
		{"envoyés", "envoi"},
		{"envoyez", "envoi"},
		{"éolienne", "éolien"},
		{"épagneul", "épagneul"},
		{"épais", "épais"},
		{"épaisse", "épaiss"},
		{"épaisseur", "épaisseur"},
		{"épanchement", "épanch"},
		{"épanouir", "épanou"},
		{"épanouissaient", "épanou"},
		{"épanouissait", "épanou"},
		{"épanouissement", "épanou"},
		{"épanouit", "épanou"},
		{"épargna", "épargn"},
		{"épargnait", "épargn"},
		{"épargne", "épargn"},
		{"épargné", "épargn"},
		{"épargnée", "épargn"},
		{"épargner", "épargn"},
		{"épargnera", "épargn"},
		{"épargnerait", "épargn"},
		{"épargnez", "épargn"},
		{"épargniez", "épargn"},
		{"éparses", "épars"},
		{"éparvérés", "éparver"},
		{"épaule", "épaul"},
		{"épaules", "épaul"},
		{"épaulés", "épaul"},
		{"épaulettes", "épaulet"},
		{"épée", "épé"},
		{"épées", "épé"},
		{"éperdu", "éperdu"},
		{"éperdue", "éperdu"},
		{"éperdument", "éperdu"},
		{"éperon", "éperon"},
		{"éperons", "éperon"},
		{"épervier", "épervi"},
		{"éperviers", "épervi"},
		{"éphémère", "éphémer"},
		{"épia", "épi"},
		{"épiait", "épi"},
		{"épicée", "épic"},
		{"épiciers", "épici"},
		{"épiderme", "épiderm"},
		{"épie", "épi"},
		{"épient", "épient"},
		{"épier", "épi"},
		{"épigrammatique", "épigrammat"},
		{"épigramme", "épigramm"},
		{"épigrammes", "épigramm"},
		{"épine", "épin"},
		{"épines", "épin"},
		{"épineuses", "épin"},
		{"épingle", "épingl"},
		{"épingles", "épingl"},
		{"épique", "épiqu"},
		{"épis", "épis"},
		{"épiscopal", "épiscopal"},
		{"épiscopat", "épiscopat"},
		{"épisode", "épisod"},
		{"épitaphe", "épitaph"},
		{"épithète", "épithet"},
		{"épître", "épîtr"},
		{"époque", "époqu"},
		{"époques", "époqu"},
		{"épousa", "épous"},
		{"épousait", "épous"},
		{"épousant", "épous"},
		{"épouse", "épous"},
		{"épousé", "épous"},
		{"épousée", "épous"},
		{"épousées", "épous"},
		{"épouser", "épous"},
		{"épousera", "épous"},
		{"épouserai", "épous"},
		{"épouserez", "épous"},
		{"épouseur", "épouseur"},
		{"épousez", "épous"},
		{"épousseter", "épousset"},
		{"épouvantable", "épouvant"},
		{"épouvantablement", "épouvant"},
		{"épouvantables", "épouvant"},
		{"épouvantaient", "épouvant"},
		{"épouvante", "épouv"},
		{"épouvantée", "épouvant"},
		{"épouvantements", "épouvant"},
		{"époux", "époux"},
		{"épreuve", "épreuv"},
		{"épreuves", "épreuv"},
		{"épris", "épris"},
		{"éprise", "épris"},
		{"éprouva", "éprouv"},
		{"éprouvai", "éprouv"},
		{"éprouvait", "éprouv"},
		{"éprouvant", "éprouv"},
		{"éprouve", "éprouv"},
		{"éprouvé", "éprouv"},
		{"éprouvée", "éprouv"},
		{"éprouvées", "éprouv"},
		{"éprouvent", "éprouvent"},
		{"éprouver", "éprouv"},
		{"éprouverais", "éprouv"},
		{"éprouverait", "éprouv"},
		{"éprouvés", "éprouv"},
		{"épuisait", "épuis"},
		{"épuisé", "épuis"},
		{"épuisée", "épuis"},
		{"épuisement", "épuis"},
		{"épuisèrent", "épuis"},
		{"équarrir", "équarr"},
		{"équarrissaient", "équarr"},
		{"équilibre", "équilibr"},
		{"équilibré", "équilibr"},
		{"équilibristes", "équilibr"},
		{"equinoctial", "equinoctial"},
		{"équinoxe", "équinox"},
		{"équinoxes", "équinox"},
		{"équipage", "équipag"},
		{"équipé", "équip"},
		{"équipée", "équip"},
		{"équité", "équit"},
		{"équivalait", "équival"},
		{"équivoques", "équivoqu"},
		{"equus", "equus"},
		{"er", "er"},
		{"éreinté", "éreint"},
		{"ériger", "érig"},
		{"erit", "erit"},
		{"ermitages", "ermitag"},
		{"ernest", "ernest"},
		{"erra", "erra"},
		{"erraient", "erraient"},
		{"errait", "errait"},
		{"errant", "errant"},
		{"errante", "errant"},
		{"erré", "erré"},
		{"errer", "errer"},
		{"erreur", "erreur"},
		{"erreurs", "erreur"},
		{"escadron", "escadron"},
		{"escaladé", "escalad"},
		{"escalader", "escalad"},
		{"escale", "escal"},
		{"escalier", "escali"},
		{"escaliers", "escali"},
		{"escamotée", "escamot"},
		{"escamoter", "escamot"},
		{"escapade", "escapad"},
		{"escarpés", "escarp"},
		{"escarpins", "escarpin"},
		{"escaut", "escaut"},
		{"esclandre", "esclandr"},
		{"esclave", "esclav"},
		{"esclaves", "esclav"},
		{"escofié", "escofi"},
		{"escogriffe", "escogriff"},
		{"escortaient", "escort"},
		{"escorte", "escort"},
		{"escorté", "escort"},
		{"escortée", "escort"},
		{"escouade", "escouad"},
		{"escrime", "escrim"},
		{"escrit", "escrit"},
		{"espace", "espac"},
		{"espaces", "espac"},
		{"espagne", "espagn"},
		{"espagnol", "espagnol"},
		{"espagnole", "espagnol"},
		{"espagnoles", "espagnol"},
		{"espagnolette", "espagnolet"},
		{"espagnols", "espagnol"},
		{"esparres", "esparr"},
		{"espèce", "espec"},
		{"espèces", "espec"},
		{"espéra", "esper"},
		{"espérais", "esper"},
		{"espérait", "esper"},
		{"espérance", "esper"},
		{"espérances", "esper"},
		{"espérant", "esper"},
		{"espère", "esper"},
		{"espéré", "esper"},
		{"espérée", "esper"},
		{"espérer", "esper"},
		{"espères", "esper"},
		{"espérez", "esper"},
		{"espérons", "espéron"},
		{"espiègle", "espiegl"},
		{"espion", "espion"},
		{"espionnage", "espionnag"},
		{"espionné", "espion"},
		{"espionnes", "espion"},
		{"espions", "espion"},
		{"esplanade", "esplanad"},
		{"espoir", "espoir"},
		{"esprit", "esprit"},
		{"esprits", "esprit"},
		{"esq", "esq"},
		{"esquire", "esquir"},
		{"esquisse", "esquiss"},
		{"esquisser", "esquiss"},
		{"esquiva", "esquiv"},
		{"esquiveras", "esquiv"},
		{"essai", "essai"},
		{"essaiera", "essai"},
		{"essaierai", "essai"},
		{"essaierait", "essai"},
		{"essais", "essais"},
		{"essaya", "essai"},
		{"essayaient", "essai"},
		{"essayait", "essai"},
		{"essayant", "essai"},
		{"essaye", "essay"},
		{"essayé", "essai"},
		{"essayées", "essai"},
		{"essayer", "essai"},
		{"essayons", "essayon"},
		{"essence", "essenc"},
		{"essences", "essenc"},
		{"essentiel", "essentiel"},
		{"essentielle", "essentiel"},
		{"essentiellement", "essentiel"},
		{"essieux", "essieux"},
		{"essoufflé", "essouffl"},
		{"essoufflée", "essouffl"},
		{"essuie", "essui"},
		{"essuyait", "essui"},
		{"essuyé", "essui"},
		{"essuyer", "essui"},
		{"est", "est"},
		{"estafette", "estafet"},
		{"estafier", "estafi"},
		{"estafiers", "estafi"},
		{"estampe", "estamp"},
		{"estampes", "estamp"},
		{"estimable", "estim"},
		{"estimables", "estim"},
		{"estimaient", "estim"},
		{"estimait", "estim"},
		{"estimant", "estim"},
		{"estime", "estim"},
		{"estimé", "estim"},
		{"estimée", "estim"},
		{"estimer", "estim"},
		{"estimeraient", "estim"},
		{"estimez", "estim"},
		{"estomac", "estomac"},
		{"estomacs", "estomac"},
		{"et", "et"},
		{"établi", "établ"},
		{"établie", "établ"},
		{"établies", "établ"},
		{"établir", "établ"},
		{"établira", "établ"},
		{"établirent", "établ"},
		{"établirez", "établ"},
		{"établis", "établ"},
		{"établissaient", "établ"},
		{"établissait", "établ"},
		{"établissant", "établ"},
		{"établissement", "établ"},
		{"établissements", "établ"},
		{"établit", "établ"},
		{"étage", "étag"},
		{"étagea", "étag"},
		{"étageaient", "étag"},
		{"étages", "étag"},
		{"étai", "étai"},
		{"étaie", "étai"},
		{"etaient", "etaient"},
		{"étaient", "étaient"},
		{"étain", "étain"},
		{"étais", "étais"},
		{"etait", "etait"},
		{"était", "était"},
		{"étalage", "étalag"},
		{"étalages", "étalag"},
		{"étalait", "étal"},
		{"étalant", "étal"},
		{"étale", "étal"},
		{"étalé", "étal"},
		{"étalée", "étal"},
		{"étaler", "étal"},
		{"étalés", "étal"},
		{"étancher", "étanch"},
		{"étang", "étang"},
		{"étangs", "étang"},
		{"etant", "etant"},
		{"étant", "étant"},
		{"étarquée", "étarqu"},
		{"etat", "etat"},
		{"état", "état"},
		{"etats", "etat"},
		{"états", "état"},
		{"étaux", "étal"},
		{"étayé", "étai"},
		{"etc", "etc"},
		{"été", "été"},
		{"éteignaient", "éteign"},
		{"éteignait", "éteign"},
		{"éteignirent", "éteign"},
		{"éteignit", "éteign"},
		{"éteindra", "éteindr"},
		{"éteindre", "éteindr"},
		{"éteindrons", "éteindron"},
		{"éteint", "éteint"},
		{"éteinte", "éteint"},
		{"éteints", "éteint"},
		{"étend", "étend"},
		{"étendaient", "étend"},
		{"étendait", "étend"},
		{"étendant", "étend"},
		{"étendent", "étendent"},
		{"étendirent", "étend"},
		{"étendit", "étend"},
		{"étendre", "étendr"},
		{"étendu", "étendu"},
		{"étendue", "étendu"},
		{"étendues", "étendu"},
		{"étendus", "étendus"},
		{"éternel", "éternel"},
		{"éternelle", "éternel"},
		{"éternelles", "éternel"},
		{"éternels", "éternel"},
		{"éternité", "étern"},
		{"êtes", "ête"},
		{"ethnographiques", "ethnograph"},
		{"étienne", "étien"},
		{"étiez", "éti"},
		{"étincelante", "étincel"},
		{"étincelants", "étincel"},
		{"étincelle", "étincel"},
		{"étincelles", "étincel"},
		{"étiole", "étiol"},
		{"étiolés", "étiol"},
		{"étions", "étion"},
		{"étiquette", "étiquet"},
		{"étoffe", "étoff"},
		{"étoffes", "étoff"},
		{"étoile", "étoil"},
		{"étoilé", "étoil"},
		{"étoiles", "étoil"},
		{"étonna", "éton"},
		{"étonnaient", "éton"},
		{"étonnait", "éton"},
		{"étonnant", "éton"},
		{"étonnante", "éton"},
		{"étonnantes", "éton"},
		{"étonnants", "éton"},
		{"étonne", "éton"},
		{"étonné", "éton"},
		{"étonnée", "éton"},
		{"étonnées", "éton"},
		{"étonnement", "éton"},
		{"étonnent", "étonnent"},
		{"étonner", "éton"},
		{"étonnera", "éton"},
		{"étonnerait", "éton"},
		{"étonnèrent", "éton"},
		{"étonnés", "éton"},
		{"étouffaient", "étouff"},
		{"étouffait", "étouff"},
		{"étouffe", "étouff"},
		{"étouffé", "étouff"},
		{"étouffée", "étouff"},
		{"étouffer", "étouff"},
		{"étourderie", "étourder"},
		{"étourderies", "étourder"},
		{"étourdi", "étourd"},
		{"étourdie", "étourd"},
		{"étourdiment", "étourd"},
		{"étourdir", "étourd"},
		{"étourdissait", "étourd"},
		{"étourdissement", "étourd"},
		{"étourdit", "étourd"},
		{"étrange", "étrang"},
		{"étrangement", "étrang"},
		{"étranger", "étrang"},
		{"étrangère", "étranger"},
		{"étrangères", "étranger"},
		{"étrangers", "étranger"},
		{"étranges", "étrang"},
		{"étrangeté", "étranget"},
		{"étranglaient", "étrangl"},
		{"étrangler", "étrangl"},
		{"etrangleurs", "etrangleur"},
		{"étrave", "étrav"},
		{"etre", "etre"},
		{"être", "être"},
		{"étreignit", "étreign"},
		{"étreinte", "étreint"},
		{"êtres", "être"},
		{"étrier", "étri"},
		{"étriers", "étrier"},
		{"étrille", "étrill"},
		{"étroit", "étroit"},
		{"étroite", "étroit"},
		{"étroitement", "étroit"},
		{"étroites", "étroit"},
		{"étroits", "étroit"},
		{"étude", "étud"},
		{"études", "étud"},
		{"étudia", "étudi"},
		{"étudiait", "étudi"},
		{"étudiant", "étudi"},
		{"étudie", "étud"},
		{"étudié", "étudi"},
		{"étudiée", "étudi"},
		{"étudier", "étudi"},
		{"étudierai", "étudi"},
		{"étudiez", "étud"},
		{"étui", "étui"},
		{"étuis", "étuis"},
		{"eu", "eu"},
		{"eue", "eue"},
		{"eues", "eue"},
		{"eugène", "eugen"},
		{"eunuque", "eunuqu"},
		{"euphorbes", "euphorb"},
		{"eurent", "eurent"},
		{"europe", "europ"},
		{"européen", "européen"},
		{"européenne", "européen"},
		{"européennes", "européen"},
		{"européens", "européen"},
		{"eus", "eus"},
		{"eusse", "euss"},
		{"eussent", "eussent"},
		{"eussiez", "euss"},
		{"eut", "eut"},
		{"eût", "eût"},
		{"eux", "eux"},
		{"évacuer", "évacu"},
		{"évader", "évad"},
		{"évaluait", "évalu"},
		{"évangélique", "évangel"},
		{"évangile", "évangil"},
		{"évanoui", "évanou"},
		{"évanouie", "évanou"},
		{"évanouir", "évanou"},
		{"évanouirent", "évanou"},
		{"évanouis", "évanou"},
		{"évanouissait", "évanou"},
		{"évanouissement", "évanou"},
		{"évanouit", "évanou"},
		{"evans", "evan"},
		{"évaporatoire", "évaporatoir"},
		{"évasion", "évas"},
		{"évêché", "évêch"},
		{"éveil", "éveil"},
		{"éveilla", "éveil"},
		{"éveillant", "éveil"},
		{"éveillé", "éveil"},
		{"éveillée", "éveil"},
		{"éveillées", "éveil"},
		{"éveiller", "éveil"},
		{"éveillera", "éveil"},
		{"éveillés", "éveil"},
		{"even", "even"},
		{"événement", "éven"},
		{"événements", "éven"},
		{"evening", "evening"},
		{"éventail", "éventail"},
		{"éventée", "évent"},
		{"éventualité", "éventual"},
		{"éventualités", "éventual"},
		{"évêque", "évêqu"},
		{"évêques", "évêqu"},
		{"évidaient", "évid"},
		{"evidemment", "evident"},
		{"évidemment", "évident"},
		{"évidence", "évident"},
		{"évident", "évident"},
		{"évidente", "évident"},
		{"évidentes", "évident"},
		{"évidents", "évident"},
		{"évita", "évit"},
		{"évitait", "évit"},
		{"évitant", "évit"},
		{"évité", "évit"},
		{"éviter", "évit"},
		{"évitèrent", "évit"},
		{"évitons", "éviton"},
		{"ex", "ex"},
		{"exact", "exact"},
		{"exacte", "exact"},
		{"exactement", "exact"},
		{"exactes", "exact"},
		{"exactitude", "exactitud"},
		{"exacts", "exact"},
		{"exagéra", "exager"},
		{"exagérait", "exager"},
		{"exagérant", "exager"},
		{"exagération", "exager"},
		{"exagérations", "exager"},
		{"exagère", "exager"},
		{"exagéré", "exager"},
		{"exagérée", "exager"},
		{"exagérées", "exager"},
		{"exagèrent", "exag"},
		{"exagérer", "exager"},
		{"exagérés", "exager"},
		{"exalta", "exalt"},
		{"exaltait", "exalt"},
		{"exaltation", "exalt"},
		{"exalte", "exalt"},
		{"exalté", "exalt"},
		{"exaltée", "exalt"},
		{"exaltées", "exalt"},
		{"exalter", "exalt"},
		{"examen", "examen"},
		{"examens", "examen"},
		{"examina", "examin"},
		{"examinait", "examin"},
		{"examinant", "examin"},
		{"examinateur", "examin"},
		{"examinateurs", "examin"},
		{"examine", "examin"},
		{"examiné", "examin"},
		{"examinée", "examin"},
		{"examiner", "examin"},
		{"examinez", "examin"},
		{"exaspérait", "exasper"},
		{"exaspéré", "exasper"},
		{"exaspérée", "exasper"},
		{"exaspérer", "exasper"},
		{"excédé", "exced"},
		{"excédée", "exced"},
		{"excéder", "exced"},
		{"excellence", "excellent"},
		{"excellent", "excellent"},
		{"excellente", "excellent"},
		{"excellentes", "excellent"},
		{"excellents", "excellent"},
		{"excentricité", "excentr"},
		{"excentrique", "excentr"},
		{"excepté", "except"},
		{"exception", "except"},
		{"excès", "exces"},
		{"excessif", "excess"},
		{"excessifs", "excess"},
		{"excessive", "excess"},
		{"excessivement", "excess"},
		{"excessives", "excess"},
		{"excita", "excit"},
		{"excitait", "excit"},
		{"excitation", "excit"},
		{"excité", "excit"},
		{"excitée", "excit"},
		{"excitées", "excit"},
		{"excitent", "excitent"},
		{"exciter", "excit"},
		{"exciterai", "excit"},
		{"exclama", "exclam"},
		{"exclamation", "exclam"},
		{"exclamations", "exclam"},
		{"exclu", "exclu"},
		{"exclusive", "exclus"},
		{"exclusivement", "exclus"},
		{"excusa", "excus"},
		{"excusable", "excus"},
		{"excusait", "excus"},
		{"excusant", "excus"},
		{"excuse", "excus"},
		{"excusée", "excus"},
		{"excuser", "excus"},
		{"excuserez", "excus"},
		{"excuses", "excus"},
		{"excusez", "excus"},
		{"exécrable", "execr"},
		{"exécrait", "execr"},
		{"exécration", "execr"},
		{"exècre", "execr"},
		{"exécrer", "execr"},
		{"exécuta", "exécut"},
		{"exécutait", "exécut"},
		{"exécutant", "exécut"},
		{"exécute", "exécut"},
		{"exécuté", "exécut"},
		{"exécutée", "exécut"},
		{"exécutées", "exécut"},
		{"exécuter", "exécut"},
		{"exécuterai", "exécut"},
		{"exécuterait", "exécut"},
		{"exécuteras", "exécut"},
		{"exécutés", "exécut"},
		{"exécuteur", "exécuteur"},
		{"exécution", "exécu"},
		{"exécutions", "exécu"},
		{"exécutoire", "exécutoir"},
		{"exemplaire", "exemplair"},
		{"exemplaires", "exemplair"},
		{"exemple", "exempl"},
		{"exemples", "exempl"},
		{"exempte", "exempt"},
		{"exempté", "exempt"},
		{"exerçait", "exerc"},
		{"exerçant", "exerc"},
		{"exerce", "exerc"},
		{"exercée", "exerc"},
		{"exercent", "exercent"},
		{"exercer", "exerc"},
		{"exercerait", "exerc"},
		{"exercés", "exerc"},
		{"exercice", "exercic"},
		{"exercices", "exercic"},
		{"exhibant", "exhib"},
		{"exhibe", "exhib"},
		{"exhibition", "exhibit"},
		{"exhibitions", "exhibit"},
		{"exhortait", "exhort"},
		{"exige", "exig"},
		{"exigé", "exig"},
		{"exigea", "exig"},
		{"exigeait", "exig"},
		{"exigences", "exigent"},
		{"exigeons", "exigeon"},
		{"exiger", "exig"},
		{"exigez", "exig"},
		{"exiguë", "exigu"},
		{"exigus", "exigus"},
		{"exil", "exil"},
		{"exila", "exil"},
		{"exilait", "exil"},
		{"exile", "exil"},
		{"exilé", "exil"},
		{"exilées", "exil"},
		{"exiler", "exil"},
		{"exista", "exist"},
		{"existaient", "exist"},
		{"existait", "exist"},
		{"existât", "exist"},
		{"existe", "exist"},
		{"existé", "exist"},
		{"existence", "existent"},
		{"existent", "existent"},
		{"exister", "exist"},
		{"exorbitante", "exorbit"},
		{"exotiques", "exot"},
		{"expansif", "expans"},
		{"expansion", "expans"},
		{"expédia", "expédi"},
		{"expédié", "expédi"},
		{"expédiée", "expédi"},
		{"expédier", "expédi"},
		{"expédition", "expédit"},
		{"expéditions", "expédit"},
		{"expérience", "expérient"},
		{"expiation", "expiat"},
		{"expira", "expir"},
		{"expiré", "expir"},
		{"expirer", "expir"},
		{"explicable", "explic"},
		{"explication", "expliqu"},
		{"explications", "expliqu"},
		{"explicite", "explicit"},
		{"expliqua", "expliqu"},
		{"expliquait", "expliqu"},
		{"expliquant", "expliqu"},
		{"explique", "expliqu"},
		{"expliqué", "expliqu"},
		{"expliquée", "expliqu"},
		{"expliquer", "expliqu"},
		{"expliqués", "expliqu"},
		{"expliquez", "expliqu"},
		{"exploit", "exploit"},
		{"exploits", "exploit"},
		{"explosion", "explos"},
		{"exposa", "expos"},
		{"exposais", "expos"},
		{"exposait", "expos"},
		{"exposant", "expos"},
		{"expose", "expos"},
		{"exposé", "expos"},
		{"exposée", "expos"},
		{"exposées", "expos"},
		{"exposer", "expos"},
		{"exposerai", "expos"},
		{"exposerais", "expos"},
		{"exposerait", "expos"},
		{"exposés", "expos"},
		{"exposez", "expos"},
		{"exprès", "expres"},
		{"express", "express"},
		{"expresse", "express"},
		{"expressément", "express"},
		{"expressifs", "express"},
		{"expression", "express"},
		{"expressions", "express"},
		{"exprima", "exprim"},
		{"exprimaient", "exprim"},
		{"exprimait", "exprim"},
		{"exprimant", "exprim"},
		{"exprime", "exprim"},
		{"exprimé", "exprim"},
		{"exprimée", "exprim"},
		{"exprimer", "exprim"},
		{"exprimèrent", "exprim"},
		{"exprimeront", "exprim"},
		{"exquise", "exquis"},
		{"extase", "extas"},
		{"extases", "extas"},
		{"extatique", "extat"},
		{"extension", "extens"},
		{"extérieur", "extérieur"},
		{"extérieure", "extérieur"},
		{"extérieures", "extérieur"},
		{"extérieurs", "extérieur"},
		{"exterminés", "extermin"},
		{"extinction", "extinct"},
		{"extorqué", "extorqu"},
		{"extradition", "extradit"},
		{"extrait", "extrait"},
		{"extraites", "extrait"},
		{"extraordinaire", "extraordinair"},
		{"extraordinairement", "extraordinair"},
		{"extraordinaires", "extraordinair"},
		{"extravagant", "extravag"},
		{"extravagante", "extravag"},
		{"extravagantes", "extravag"},
		{"extrême", "extrêm"},
		{"extrêmement", "extrêm"},
		{"extrêmes", "extrêm"},
		{"extrémité", "extrem"},
		{"extrémités", "extrem"},
		{"f", "f"},
		{"fa", "fa"},
		{"fabio", "fabio"},
		{"fable", "fabl"},
		{"fables", "fabl"},
		{"fabricant", "fabric"},
		{"fabricants", "fabric"},
		{"fabrice", "fabric"},
		{"fabriquaient", "fabriqu"},
		{"fabrique", "fabriqu"},
		{"fabriquée", "fabriqu"},
		{"fabriquées", "fabriqu"},
		{"fabriquer", "fabriqu"},
		{"fabriques", "fabriqu"},
		{"fabuleux", "fabul"},
		{"fabuliste", "fabul"},
		{"façade", "façad"},
		{"façades", "façad"},
		{"face", "fac"},
		{"fâcha", "fâch"},
		{"fâchait", "fâch"},
		{"fâchât", "fâch"},
		{"fâche", "fâch"},
		{"fâché", "fâch"},
		{"fâchée", "fâch"},
		{"fâchent", "fâchent"},
		{"fâcher", "fâch"},
		{"fâchèrent", "fâch"},
		{"fâchés", "fâch"},
		{"fâcheuse", "fâcheux"},
		{"fâcheuses", "fâcheux"},
		{"fâcheux", "fâcheux"},
		{"fâchez", "fâch"},
		{"facile", "facil"},
		{"facilement", "facil"},
		{"facilitait", "facilit"},
		{"facilité", "facil"},
		{"facilitent", "facilitent"},
		{"faciliter", "facilit"},
		{"facio", "facio"},
		{"façon", "façon"},
		{"faconde", "facond"},
		{"façons", "façon"},
		{"facteur", "facteur"},
		{"faction", "faction"},
		{"factotum", "factotum"},
		{"faculté", "facult"},
		{"facultés", "facult"},
		{"fade", "fad"},
		{"fagots", "fagot"},
		{"fahrenheit", "fahrenheit"},
		{"faible", "faibl"},
		{"faibles", "faibl"},
		{"faiblesse", "faibless"},
		{"faiblesses", "faibless"},
		{"faiblir", "faibl"},
		{"faiblirent", "faibl"},
		{"faict", "faict"},
		{"failli", "faill"},
		{"faillir", "faill"},
		{"faillirent", "faill"},
		{"faillit", "faill"},
		{"faim", "faim"},
		{"fainéant", "fainé"},
		{"faire", "fair"},
		{"fais", "fais"},
		{"faisable", "faisabl"},
		{"faisaient", "fais"},
		{"faisais", "fais"},
		{"faisait", "fais"},
		{"faisant", "fais"},
		{"faisceaux", "faisceau"},
		{"faisiez", "fais"},
		{"faisons", "faison"},
		{"fait", "fait"},
		{"faite", "fait"},
		{"faîte", "faît"},
		{"faites", "fait"},
		{"faits", "fait"},
		{"fakirs", "fakir"},
		{"falcoz", "falcoz"},
		{"faliero", "faliero"},
		{"fallait", "fall"},
		{"fallentin", "fallentin"},
		{"falloir", "falloir"},
		{"fallu", "fallu"},
		{"fallut", "fallut"},
		{"fameuse", "fameux"},
		{"fameusement", "fameux"},
		{"fameux", "fameux"},
		{"familiarisé", "familiaris"},
		{"familiarité", "familiar"},
		{"familier", "famili"},
		{"familièrement", "famili"},
		{"famille", "famill"},
		{"familles", "famill"},
		{"fanal", "fanal"},
		{"fanatique", "fanat"},
		{"fanatiques", "fanat"},
		{"fanatisme", "fanat"},
		{"fanfarons", "fanfaron"},
		{"fange", "fang"},
		{"fangeuses", "fangeux"},
		{"fangeux", "fangeux"},
		{"fantaisie", "fantais"},
		{"fantaisies", "fantais"},
		{"fantaisistes", "fantais"},
		{"fantasque", "fantasqu"},
		{"fantastique", "fantast"},
		{"fantastiques", "fantast"},
		{"fantôme", "fantôm"},
		{"faquins", "faquin"},
		{"far", "far"},
		{"faraud", "faraud"},
		{"farceur", "farceur"},
		{"farceurs", "farceur"},
		{"farci", "farc"},
		{"fardeau", "fardeau"},
		{"fare", "far"},
		{"farine", "farin"},
		{"farmer", "farm"},
		{"farnèse", "farnes"},
		{"farouche", "farouch"},
		{"farouches", "farouch"},
		{"farthing", "farthing"},
		{"fascinant", "fascin"},
		{"fascination", "fascin"},
		{"fasciné", "fascin"},
		{"fasciner", "fascin"},
		{"fasse", "fass"},
		{"fassent", "fassent"},
		{"fassiez", "fass"},
		{"faste", "fast"},
		{"fastenet", "fastenet"},
		{"fat", "fat"},
		{"fata", "fat"},
		{"fatal", "fatal"},
		{"fatale", "fatal"},
		{"fatales", "fatal"},
		{"fatalité", "fatal"},
		{"fatigant", "fatig"},
		{"fatigante", "fatig"},
		{"fatigua", "fatigu"},
		{"fatiguaient", "fatigu"},
		{"fatigue", "fatigu"},
		{"fatigué", "fatigu"},
		{"fatiguée", "fatigu"},
		{"fatiguées", "fatigu"},
		{"fatiguer", "fatigu"},
		{"fatiguerai", "fatigu"},
		{"fatigues", "fatigu"},
		{"fatuité", "fatuit"},
		{"faublas", "faubl"},
		{"faubourg", "faubourg"},
		{"faudra", "faudr"},
		{"faudrait", "faudr"},
		{"faufilant", "faufil"},
		{"fausse", "fauss"},
		{"fausses", "fauss"},
		{"fausseté", "fausset"},
		{"faussetés", "fausset"},
		{"fausta", "faust"},
		{"faut", "faut"},
		{"faute", "faut"},
		{"fautes", "faut"},
		{"fauteuil", "fauteuil"},
		{"fauteuils", "fauteuil"},
		{"fauve", "fauv"},
		{"fauves", "fauv"},
		{"faux", "faux"},
		{"faveur", "faveur"},
		{"faveurs", "faveur"},
		{"favorable", "favor"},
		{"favorablement", "favor"},
		{"favorables", "favor"},
		{"favori", "favor"},
		{"favoris", "favor"},
		{"favorisa", "favoris"},
		{"favorisait", "favoris"},
		{"favoriser", "favoris"},
		{"favorite", "favorit"},
		{"fay", "fay"},
		{"fayette", "fayet"},
		{"fé", "fé"},
		{"fébrile", "fébril"},
		{"fébrilement", "fébril"},
		{"feci", "fec"},
		{"fécond", "fécond"},
		{"féconde", "fécond"},
		{"feignait", "feign"},
		{"feignant", "feign"},
		{"feignent", "feignent"},
		{"feignit", "feign"},
		{"feindre", "feindr"},
		{"feins", "fein"},
		{"feint", "feint"},
		{"feinte", "feint"},
		{"felice", "felic"},
		{"félicita", "félicit"},
		{"félicitait", "félicit"},
		{"félicite", "félicit"},
		{"félicité", "féliqu"},
		{"félicités", "féliqu"},
		{"felino", "felino"},
		{"félix", "félix"},
		{"fellah", "fellah"},
		{"fellahs", "fellah"},
		{"féminin", "féminin"},
		{"féminine", "féminin"},
		{"femme", "femm"},
		{"femmelette", "femmelet"},
		{"femmes", "femm"},
		{"fendaient", "fend"},
		{"fendait", "fend"},
		{"fendues", "fendu"},
		{"fénelon", "fénelon"},
		{"fenestrelles", "fenestrel"},
		{"fenêtre", "fenêtr"},
		{"fenêtres", "fenêtr"},
		{"fente", "fent"},
		{"féodal", "féodal"},
		{"fer", "fer"},
		{"fera", "fer"},
		{"ferai", "fer"},
		{"feraient", "fer"},
		{"ferais", "fer"},
		{"ferait", "fer"},
		{"feras", "fer"},
		{"feretrius", "feretrius"},
		{"ferez", "fer"},
		{"feriez", "fer"},
		{"feringhea", "feringh"},
		{"ferions", "ferion"},
		{"ferma", "ferm"},
		{"fermage", "fermag"},
		{"fermaient", "ferm"},
		{"fermait", "ferm"},
		{"fermant", "ferm"},
		{"fermât", "ferm"},
		{"ferme", "ferm"},
		{"fermé", "ferm"},
		{"fermée", "ferm"},
		{"fermées", "ferm"},
		{"fermement", "ferm"},
		{"fermentation", "ferment"},
		{"fermer", "ferm"},
		{"fermerait", "ferm"},
		{"fermèrent", "ferm"},
		{"fermerons", "ferm"},
		{"fermes", "ferm"},
		{"fermés", "ferm"},
		{"fermeté", "fermet"},
		{"fermeture", "fermetur"},
		{"fermier", "fermi"},
		{"fermiers", "fermi"},
		{"féroce", "féroc"},
		{"féroces", "féroc"},
		{"férocité", "féroc"},
		{"ferons", "feron"},
		{"feront", "feront"},
		{"ferrante", "ferr"},
		{"ferrare", "ferrar"},
		{"ferrée", "ferr"},
		{"ferrures", "ferrur"},
		{"ferry", "ferry"},
		{"fers", "fer"},
		{"fertile", "fertil"},
		{"fertiles", "fertil"},
		{"fervaques", "fervaqu"},
		{"fervent", "fervent"},
		{"fervente", "fervent"},
		{"fervents", "fervent"},
		{"ferveur", "ferveur"},
		{"fête", "fêt"},
		{"fêté", "fêt"},
		{"fêtes", "fêt"},
		{"feu", "feu"},
		{"feuillage", "feuillag"},
		{"feuille", "feuill"},
		{"feuilles", "feuill"},
		{"feuilleta", "feuillet"},
		{"feuilletait", "feuillet"},
		{"feuilletant", "feuillet"},
		{"feuillets", "feuillet"},
		{"feutre", "feutr"},
		{"feux", "feux"},
		{"few", "few"},
		{"fi", "fi"},
		{"fiacre", "fiacr"},
		{"ficelé", "ficel"},
		{"ficelles", "ficel"},
		{"fichée", "fich"},
		{"fiches", "fich"},
		{"fidèle", "fidel"},
		{"fidèlement", "fidel"},
		{"fidèles", "fidel"},
		{"fidélité", "fidel"},
		{"fie", "fi"},
		{"fier", "fi"},
		{"fierai", "fi"},
		{"fière", "fier"},
		{"fièrement", "fier"},
		{"fières", "fier"},
		{"fiers", "fier"},
		{"fierté", "fiert"},
		{"fièvre", "fievr"},
		{"fiévreux", "fiévreux"},
		{"figaro", "figaro"},
		{"figura", "figur"},
		{"figuraient", "figur"},
		{"figurais", "figur"},
		{"figurait", "figur"},
		{"figurât", "figur"},
		{"figure", "figur"},
		{"figuré", "figur"},
		{"figurer", "figur"},
		{"figurerai", "figur"},
		{"figureriez", "figur"},
		{"figures", "figur"},
		{"figurés", "figur"},
		{"figurez", "figur"},
		{"fil", "fil"},
		{"fila", "fil"},
		{"filait", "fil"},
		{"filant", "fil"},
		{"filasse", "fil"},
		{"file", "fil"},
		{"filé", "fil"},
		{"filent", "filent"},
		{"filer", "fil"},
		{"filerai", "fil"},
		{"files", "fil"},
		{"filet", "filet"},
		{"filez", "fil"},
		{"fili", "fil"},
		{"filiale", "filial"},
		{"filigrane", "filigran"},
		{"filins", "filin"},
		{"fille", "fill"},
		{"filles", "fill"},
		{"filons", "filon"},
		{"fils", "fil"},
		{"filtrait", "filtr"},
		{"fin", "fin"},
		{"finance", "financ"},
		{"finances", "financ"},
		{"financier", "financi"},
		{"financièrement", "financi"},
		{"finds", "find"},
		{"fine", "fin"},
		{"fines", "fin"},
		{"finesse", "finess"},
		{"finesses", "finess"},
		{"fini", "fin"},
		{"finie", "fin"},
		{"finies", "fin"},
		{"finir", "fin"},
		{"finira", "fin"},
		{"finiraient", "fin"},
		{"finirait", "fin"},
		{"finirent", "fin"},
		{"finis", "fin"},
		{"finissait", "fin"},
		{"finissant", "fin"},
		{"finisse", "fin"},
		{"finissent", "fin"},
		{"finissons", "fin"},
		{"finit", "fin"},
		{"fins", "fin"},
		{"fiole", "fiol"},
		{"fir", "fir"},
		{"fire", "fir"},
		{"firent", "firent"},
		{"fis", "fis"},
		{"fiscal", "fiscal"},
		{"fisse", "fiss"},
		{"fit", "fit"},
		{"fît", "fît"},
		{"fitz", "fitz"},
		{"fix", "fix"},
		{"fixa", "fix"},
		{"fixaient", "fix"},
		{"fixait", "fix"},
		{"fixant", "fix"},
		{"fixe", "fix"},
		{"fixé", "fix"},
		{"fixée", "fix"},
		{"fixement", "fix"},
		{"fixer", "fix"},
		{"fixèrent", "fix"},
		{"fixés", "fix"},
		{"fixité", "fixit"},
		{"flacon", "flacon"},
		{"flagrant", "flagr"},
		{"flair", "flair"},
		{"flairait", "flair"},
		{"flamand", "flamand"},
		{"flamande", "flamand"},
		{"flambé", "flamb"},
		{"flambeaux", "flambeau"},
		{"flambés", "flamb"},
		{"flamboyants", "flamboi"},
		{"flamme", "flamm"},
		{"flammes", "flamm"},
		{"flâna", "flân"},
		{"flanagan", "flanagan"},
		{"flânait", "flân"},
		{"flânant", "flân"},
		{"flanc", "flanc"},
		{"flancs", "flanc"},
		{"flatta", "flatt"},
		{"flattait", "flatt"},
		{"flatte", "flatt"},
		{"flatté", "flatt"},
		{"flattée", "flatt"},
		{"flatter", "flatt"},
		{"flatteries", "flatter"},
		{"flatteur", "flatteur"},
		{"flatteurs", "flatteur"},
		{"flatteuse", "flatteux"},
		{"flatteuses", "flatteux"},
		{"flèche", "flech"},
		{"flèches", "flech"},
		{"fléchir", "flech"},
		{"flegmatique", "flegmat"},
		{"flegme", "flegm"},
		{"flétri", "fletr"},
		{"flétrirait", "fletr"},
		{"fleur", "fleur"},
		{"fleurs", "fleur"},
		{"fleury", "fleury"},
		{"fleuve", "fleuv"},
		{"fleuves", "fleuv"},
		{"flexible", "flexibl"},
		{"flexibles", "flexibl"},
		{"flocons", "flocon"},
		{"florence", "florenc"},
		{"florissante", "flor"},
		{"flot", "flot"},
		{"flots", "flot"},
		{"flottaient", "flott"},
		{"flottant", "flott"},
		{"flottantes", "flott"},
		{"flottants", "flott"},
		{"flotter", "flott"},
		{"flottille", "flottill"},
		{"flûtes", "flût"},
		{"fo", "fo"},
		{"foc", "foc"},
		{"focs", "foc"},
		{"fogg", "fogg"},
		{"foi", "foi"},
		{"foin", "foin"},
		{"fois", "fois"},
		{"foison", "foison"},
		{"foisonnait", "foison"},
		{"foisonne", "foison"},
		{"fokelberg", "fokelberg"},
		{"fol", "fol"},
		{"folie", "fol"},
		{"folies", "fol"},
		{"folio", "folio"},
		{"folke", "folk"},
		{"folle", "foll"},
		{"follement", "foll"},
		{"folles", "foll"},
		{"foncé", "fonc"},
		{"foncièrement", "fonci"},
		{"fonctionnaire", "fonctionnair"},
		{"fonctionnaires", "fonctionnair"},
		{"fonctionne", "fonction"},
		{"fonctions", "fonction"},
		{"fond", "fond"},
		{"fonda", "fond"},
		{"fondaient", "fond"},
		{"fondait", "fond"},
		{"fondant", "fond"},
		{"fondateur", "fondateur"},
		{"fondations", "fondat"},
		{"fondé", "fond"},
		{"fondée", "fond"},
		{"fondées", "fond"},
		{"fondements", "fond"},
		{"fondent", "fondent"},
		{"fonder", "fond"},
		{"fonderies", "fonder"},
		{"fondés", "fond"},
		{"fondirent", "fond"},
		{"fondit", "fond"},
		{"fondre", "fondr"},
		{"fonds", "fond"},
		{"fondu", "fondu"},
		{"fondue", "fondu"},
		{"font", "font"},
		{"fontaine", "fontain"},
		{"fontaines", "fontain"},
		{"fontan", "fontan"},
		{"fontana", "fontan"},
		{"fontenoy", "fontenoy"},
		{"for", "for"},
		{"força", "forc"},
		{"forçaient", "forc"},
		{"forçais", "forc"},
		{"forçait", "forc"},
		{"forçant", "forc"},
		{"force", "forc"},
		{"forcé", "forc"},
		{"forcée", "forc"},
		{"forcer", "forc"},
		{"forcera", "forc"},
		{"forcerai", "forc"},
		{"forcèrent", "forc"},
		{"forces", "forc"},
		{"forcés", "forc"},
		{"forcez", "forc"},
		{"forêt", "forêt"},
		{"foretells", "foretel"},
		{"forets", "foret"},
		{"forêts", "forêt"},
		{"forfaitz", "forfaitz"},
		{"forgé", "forg"},
		{"forma", "form"},
		{"formaient", "form"},
		{"formait", "form"},
		{"formalisait", "formalis"},
		{"formaliste", "formal"},
		{"formalistes", "formal"},
		{"formalité", "formal"},
		{"formalités", "formal"},
		{"formant", "form"},
		{"formation", "format"},
		{"forme", "form"},
		{"formé", "form"},
		{"formée", "form"},
		{"formées", "form"},
		{"formel", "formel"},
		{"formellement", "formel"},
		{"forment", "forment"},
		{"former", "form"},
		{"formera", "form"},
		{"formèrent", "form"},
		{"formes", "form"},
		{"formés", "form"},
		{"formez", "form"},
		{"formidable", "formid"},
		{"formidables", "formid"},
		{"formose", "formos"},
		{"formula", "formul"},
		{"formule", "formul"},
		{"formulée", "formul"},
		{"formuler", "formul"},
		{"formules", "formul"},
		{"forster", "forst"},
		{"fort", "fort"},
		{"forte", "fort"},
		{"fortement", "fort"},
		{"forteresse", "forteress"},
		{"fortes", "fort"},
		{"forth", "forth"},
		{"fortifications", "fortif"},
		{"fortifié", "fortifi"},
		{"fortifiée", "fortifi"},
		{"fortifiées", "fortifi"},
		{"fortifier", "fortifi"},
		{"fortifions", "fortif"},
		{"forts", "fort"},
		{"fortune", "fortun"},
		{"fortuné", "fortun"},
		{"fortunées", "fortun"},
		{"fortunes", "fortun"},
		{"fortunés", "fortun"},
		{"foscarini", "foscarin"},
		{"fosse", "foss"},
		{"fossé", "foss"},
		{"fosses", "foss"},
		{"fossés", "foss"},
		{"fou", "fou"},
		{"foudre", "foudr"},
		{"foudroyer", "foudroi"},
		{"fouet", "fouet"},
		{"fouettait", "fouet"},
		{"fouetté", "fouet"},
		{"fouetter", "fouet"},
		{"fouetterait", "fouet"},
		{"fougères", "fouger"},
		{"fougueuse", "fougueux"},
		{"fougueux", "fougueux"},
		{"fouillait", "fouill"},
		{"fouillât", "fouill"},
		{"fouille", "fouill"},
		{"fouillé", "fouill"},
		{"fouiller", "fouill"},
		{"fouilles", "fouill"},
		{"fouillés", "fouill"},
		{"fouillez", "fouill"},
		{"foule", "foul"},
		{"foulé", "foul"},
		{"foules", "foul"},
		{"fouqué", "fouqu"},
		{"fourbes", "fourb"},
		{"fourgons", "fourgon"},
		{"fourmilière", "fourmili"},
		{"fourmillait", "fourmill"},
		{"fourmillement", "fourmill"},
		{"fourmis", "fourm"},
		{"fourmont", "fourmont"},
		{"fourneaux", "fourneau"},
		{"fourni", "fourn"},
		{"fournie", "fourn"},
		{"fournir", "fourn"},
		{"fourniront", "fourn"},
		{"fournis", "fourn"},
		{"fournissaient", "fourn"},
		{"fournissait", "fourn"},
		{"fournissant", "fourn"},
		{"fournisseurs", "fournisseur"},
		{"fournit", "fourn"},
		{"fourniture", "fournitur"},
		{"fournitures", "fournitur"},
		{"fourraient", "fourr"},
		{"fourré", "fourr"},
		{"fourreau", "fourreau"},
		{"fourrer", "fourr"},
		{"fourrier", "fourri"},
		{"fourriers", "fourri"},
		{"fourrures", "fourrur"},
		{"fous", "fous"},
		{"fox", "fox"},
		{"foyer", "foi"},
		{"frac", "frac"},
		{"fracas", "frac"},
		{"fracassée", "fracass"},
		{"fracassés", "fracass"},
		{"fragile", "fragil"},
		{"fragments", "fragment"},
		{"fraîche", "fraîch"},
		{"fraîches", "fraîch"},
		{"fraîcheur", "fraîcheur"},
		{"fraîchir", "fraîch"},
		{"fraîchissait", "fraîch"},
		{"fraîchit", "fraîch"},
		{"frailty", "frailty"},
		{"frais", "frais"},
		{"franc", "franc"},
		{"français", "franc"},
		{"française", "français"},
		{"françaises", "français"},
		{"france", "franc"},
		{"francese", "frances"},
		{"francfort", "francfort"},
		{"franche", "franch"},
		{"franchement", "franch"},
		{"franchi", "franch"},
		{"franchie", "franch"},
		{"franchies", "franch"},
		{"franchir", "franch"},
		{"franchis", "franch"},
		{"franchise", "franchis"},
		{"franchissait", "franch"},
		{"franchissant", "franch"},
		{"franchissent", "franch"},
		{"franchit", "franch"},
		{"francis", "franc"},
		{"francisco", "francisco"},
		{"françois", "françois"},
		{"francs", "franc"},
		{"frappa", "frapp"},
		{"frappaient", "frapp"},
		{"frappait", "frapp"},
		{"frappant", "frapp"},
		{"frappante", "frapp"},
		{"frappantes", "frapp"},
		{"frappât", "frapp"},
		{"frappe", "frapp"},
		{"frappé", "frapp"},
		{"frappée", "frapp"},
		{"frappées", "frapp"},
		{"frappent", "frappent"},
		{"frapper", "frapp"},
		{"frapperai", "frapp"},
		{"frappés", "frapp"},
		{"fraternelle", "fraternel"},
		{"fraternisaient", "fraternis"},
		{"fraude", "fraud"},
		{"frayée", "frai"},
		{"frayeur", "frayeur"},
		{"frédéric", "frédéric"},
		{"freins", "frein"},
		{"frêle", "frêl"},
		{"freluquets", "freluquet"},
		{"frémi", "frem"},
		{"frémir", "frem"},
		{"frémis", "frem"},
		{"frémit", "frem"},
		{"fremont", "fremont"},
		{"frênes", "frên"},
		{"frénétique", "frénet"},
		{"fréquemment", "fréquent"},
		{"fréquence", "fréquenc"},
		{"fréquent", "fréquent"},
		{"fréquente", "fréquent"},
		{"fréquenté", "fréquent"},
		{"fréquentée", "fréquent"},
		{"fréquentées", "fréquent"},
		{"fréquenter", "fréquent"},
		{"fréquentes", "fréquent"},
		{"fréquents", "fréquent"},
		{"frère", "frer"},
		{"frères", "frer"},
		{"fresque", "fresqu"},
		{"fret", "fret"},
		{"fréter", "fret"},
		{"frilair", "frilair"},
		{"fringante", "fring"},
		{"friperie", "friper"},
		{"friperies", "friper"},
		{"fripon", "fripon"},
		{"friponner", "fripon"},
		{"friponnerie", "friponner"},
		{"friponneries", "friponner"},
		{"fripons", "fripon"},
		{"frisaient", "fris"},
		{"frisait", "fris"},
		{"friser", "fris"},
		{"frises", "fris"},
		{"frisés", "fris"},
		{"frisson", "frisson"},
		{"frivole", "frivol"},
		{"froc", "froc"},
		{"froid", "froid"},
		{"froide", "froid"},
		{"froidement", "froid"},
		{"froides", "froid"},
		{"froideur", "froideur"},
		{"froids", "froid"},
		{"froissa", "froiss"},
		{"froissant", "froiss"},
		{"froissants", "froiss"},
		{"froissé", "froiss"},
		{"from", "from"},
		{"fromage", "fromag"},
		{"froment", "froment"},
		{"fronçait", "fronc"},
		{"fronçant", "fronc"},
		{"froncé", "fronc"},
		{"froncement", "fronc"},
		{"froncer", "fronc"},
		{"froncera", "fronc"},
		{"fronde", "frond"},
		{"frondeur", "frondeur"},
		{"front", "front"},
		{"frontière", "fronti"},
		{"frontières", "fronti"},
		{"frontins", "frontin"},
		{"fronts", "front"},
		{"frotta", "frott"},
		{"frottait", "frott"},
		{"frottant", "frott"},
		{"frotté", "frott"},
		{"frottements", "frott"},
		{"fruit", "fruit"},
		{"fruitières", "fruiti"},
		{"fruits", "fruit"},
		{"fugitif", "fugit"},
		{"fugitifs", "fugit"},
		{"fui", "fui"},
		{"fuient", "fuient"},
		{"fuir", "fuir"},
		{"fuirais", "fuir"},
		{"fuis", "fuis"},
		{"fuite", "fuit"},
		{"fulgence", "fulgenc"},
		{"fuligineuses", "fuligin"},
		{"fulminante", "fulmin"},
		{"fumaient", "fum"},
		{"fumant", "fum"},
		{"fume", "fum"},
		{"fumée", "fum"},
		{"fumées", "fum"},
		{"fumer", "fum"},
		{"fumeur", "fumeur"},
		{"fumeurs", "fumeur"},
		{"funèbre", "funebr"},
		{"funèbres", "funebr"},
		{"funeste", "funest"},
		{"funestes", "funest"},
		{"fur", "fur"},
		{"furent", "furent"},
		{"fureur", "fureur"},
		{"furia", "furi"},
		{"furibond", "furibond"},
		{"furibonde", "furibond"},
		{"furibondes", "furibond"},
		{"furibonds", "furibond"},
		{"furieuse", "furieux"},
		{"furieusement", "furieux"},
		{"furieuses", "furieux"},
		{"furieux", "furieux"},
		{"furtivement", "furtiv"},
		{"fus", "fus"},
		{"fusait", "fus"},
		{"fuseau", "fuseau"},
		{"fuseaux", "fuseau"},
		{"fusil", "fusil"},
		{"fusiliers", "fusili"},
		{"fusillé", "fusill"},
		{"fusiller", "fusill"},
		{"fusillés", "fusill"},
		{"fusils", "fusil"},
		{"fusse", "fuss"},
		{"fussent", "fussent"},
		{"fussiez", "fuss"},
		{"fut", "fut"},
		{"fût", "fût"},
		{"futé", "fut"},
		{"futilité", "futil"},
		{"futilités", "futil"},
		{"futur", "futur"},
		{"futura", "futur"},
		{"future", "futur"},
		{"futures", "futur"},
		{"futurs", "futur"},
		{"fuyaient", "fui"},
		{"fuyait", "fui"},
		{"fuyant", "fui"},
		{"fuyards", "fuyard"},
		{"fuyez", "fui"},
		{"g", "g"},
		{"gabarit", "gabar"},
		{"gabrielle", "gabriel"},
		{"gâchis", "gâch"},
		{"gage", "gag"},
		{"gages", "gag"},
		{"gageure", "gageur"},
		{"gagna", "gagn"},
		{"gagnaient", "gagn"},
		{"gagnais", "gagn"},
		{"gagnait", "gagn"},
		{"gagnant", "gagn"},
		{"gagne", "gagn"},
		{"gagné", "gagn"},
		{"gagnée", "gagn"},
		{"gagnées", "gagn"},
		{"gagnent", "gagnent"},
		{"gagner", "gagn"},
		{"gagnerai", "gagn"},
		{"gagneras", "gagn"},
		{"gagnèrent", "gagn"},
		{"gagnés", "gagn"},
		{"gagnez", "gagn"},
		{"gagnons", "gagnon"},
		{"gai", "gai"},
		{"gaie", "gai"},
		{"gaiement", "gai"},
		{"gaies", "gai"},
		{"gaieté", "gaiet"},
		{"gaillard", "gaillard"},
		{"gaillardement", "gaillard"},
		{"gaîment", "gaî"},
		{"gain", "gain"},
		{"gains", "gain"},
		{"gais", "gais"},
		{"gaîté", "gaît"},
		{"gala", "gal"},
		{"galamment", "gal"},
		{"galant", "gal"},
		{"galante", "gal"},
		{"galanterie", "galanter"},
		{"galantes", "gal"},
		{"galéas", "galé"},
		{"galère", "galer"},
		{"galères", "galer"},
		{"galerie", "galer"},
		{"galériens", "galérien"},
		{"galeries", "galer"},
		{"galeuse", "galeux"},
		{"galeux", "galeux"},
		{"gallicane", "gallican"},
		{"galon", "galon"},
		{"galonnées", "galon"},
		{"galons", "galon"},
		{"galop", "galop"},
		{"galopa", "galop"},
		{"galopaient", "galop"},
		{"galopait", "galop"},
		{"galopant", "galop"},
		{"galope", "galop"},
		{"galoper", "galop"},
		{"galopera", "galop"},
		{"galoperai", "galop"},
		{"galopez", "galop"},
		{"galvanisées", "galvanis"},
		{"gambadant", "gambad"},
		{"gamins", "gamin"},
		{"gamme", "gamm"},
		{"ganaches", "ganach"},
		{"gange", "gang"},
		{"gangrène", "gangren"},
		{"gangrenée", "gangren"},
		{"gangrenées", "gangren"},
		{"gant", "gant"},
		{"gants", "gant"},
		{"garantie", "garant"},
		{"garanties", "garant"},
		{"garantissait", "garant"},
		{"garçon", "garçon"},
		{"garçons", "garçon"},
		{"garda", "gard"},
		{"gardai", "gard"},
		{"gardaient", "gard"},
		{"gardait", "gard"},
		{"gardant", "gard"},
		{"garde", "gard"},
		{"gardé", "gard"},
		{"gardée", "gard"},
		{"gardées", "gard"},
		{"gardens", "garden"},
		{"garder", "gard"},
		{"gardera", "gard"},
		{"garderai", "gard"},
		{"garderais", "gard"},
		{"garderait", "gard"},
		{"garderas", "gard"},
		{"gardes", "gard"},
		{"gardés", "gard"},
		{"gardez", "gard"},
		{"gardien", "gardien"},
		{"gardiens", "gardien"},
		{"gare", "gar"},
		{"garnement", "garn"},
		{"garni", "garn"},
		{"garnie", "garn"},
		{"garnies", "garn"},
		{"garnis", "garn"},
		{"garnison", "garnison"},
		{"garnit", "garn"},
		{"garrot", "garrot"},
		{"garrotté", "garrott"},
		{"gascons", "gascon"},
		{"gaspilla", "gaspill"},
		{"gâtaient", "gât"},
		{"gâtait", "gât"},
		{"gâte", "gât"},
		{"gâté", "gât"},
		{"gâteau", "gâteau"},
		{"gâtée", "gât"},
		{"gâtées", "gât"},
		{"gâter", "gât"},
		{"gâterait", "gât"},
		{"gâtés", "gât"},
		{"gauche", "gauch"},
		{"gaucherie", "gaucher"},
		{"gaucheries", "gaucher"},
		{"gauches", "gauch"},
		{"gauthier", "gauthi"},
		{"gay", "gay"},
		{"gaz", "gaz"},
		{"gaze", "gaz"},
		{"gazes", "gaz"},
		{"gazette", "gazet"},
		{"gazettes", "gazet"},
		{"gazon", "gazon"},
		{"gazouillements", "gazouill"},
		{"géant", "gé"},
		{"géants", "gé"},
		{"gémi", "gem"},
		{"gémissait", "gem"},
		{"gémissant", "gem"},
		{"gémissements", "gem"},
		{"gênaient", "gên"},
		{"gênait", "gên"},
		{"gênant", "gên"},
		{"gênante", "gên"},
		{"gendarme", "gendarm"},
		{"gendarmer", "gendarm"},
		{"gendarmerie", "gendarmer"},
		{"gendarmes", "gendarm"},
		{"gendre", "gendr"},
		{"gêne", "gên"},
		{"gêné", "gên"},
		{"généalogie", "généalog"},
		{"généalogique", "généalog"},
		{"gênent", "gênent"},
		{"gêner", "gên"},
		{"gênerait", "gên"},
		{"general", "general"},
		{"général", "général"},
		{"générale", "général"},
		{"généralement", "général"},
		{"générales", "général"},
		{"généraux", "général"},
		{"généreuse", "géner"},
		{"généreusement", "géner"},
		{"généreuses", "géner"},
		{"généreux", "géner"},
		{"générosité", "généros"},
		{"gênes", "gên"},
		{"genèse", "genes"},
		{"genève", "genev"},
		{"genevois", "genevois"},
		{"génie", "gen"},
		{"génies", "gen"},
		{"genlis", "genl"},
		{"génoise", "génois"},
		{"genoux", "genoux"},
		{"genre", "genr"},
		{"genres", "genr"},
		{"gens", "gen"},
		{"gentil", "gentil"},
		{"gentilhomme", "gentilhomm"},
		{"gentille", "gentill"},
		{"gentils", "gentil"},
		{"gentilshommes", "gentilshomm"},
		{"gentle", "gentl"},
		{"gentleman", "gentleman"},
		{"gentlemen", "gentlemen"},
		{"géographie", "géograph"},
		{"geôlier", "geôli"},
		{"geôlière", "geôli"},
		{"geôliers", "geôli"},
		{"géomètre", "géometr"},
		{"géométrie", "géometr"},
		{"george", "georg"},
		{"géorgiques", "géorgiqu"},
		{"gerbes", "gerb"},
		{"germain", "germain"},
		{"géronimo", "géronimo"},
		{"geste", "gest"},
		{"gestes", "gest"},
		{"gesticulant", "gesticul"},
		{"ghari", "ghar"},
		{"ghâtes", "ghât"},
		{"ghazepour", "ghazepour"},
		{"gherardi", "gherard"},
		{"ghisleri", "ghisler"},
		{"ghisolfi", "ghisolf"},
		{"ghita", "ghit"},
		{"gia", "gi"},
		{"giacomo", "giacomo"},
		{"gibelotte", "gibelott"},
		{"giberne", "gibern"},
		{"gibier", "gibi"},
		{"gibraltar", "gibraltar"},
		{"gigantesque", "gigantesqu"},
		{"gigantesques", "gigantesqu"},
		{"gilet", "gilet"},
		{"gilets", "gilet"},
		{"giletti", "gilet"},
		{"gin", "gin"},
		{"gina", "gin"},
		{"giovanni", "giovann"},
		{"giovannone", "giovannon"},
		{"giovita", "giovit"},
		{"girafe", "giraf"},
		{"giration", "girat"},
		{"giraud", "giraud"},
		{"girodet", "girodet"},
		{"girofliers", "girofli"},
		{"girondin", "girondin"},
		{"gisaient", "gis"},
		{"gisant", "gis"},
		{"gît", "gît"},
		{"giulia", "giuli"},
		{"giuseppe", "giusepp"},
		{"give", "giv"},
		{"glaçaient", "glac"},
		{"glaçait", "glac"},
		{"glace", "glac"},
		{"glacé", "glac"},
		{"glacée", "glac"},
		{"glacées", "glac"},
		{"glacer", "glac"},
		{"glaces", "glac"},
		{"glacial", "glacial"},
		{"glaciale", "glacial"},
		{"glances", "glanc"},
		{"glapissante", "glap"},
		{"glasgow", "glasgow"},
		{"glissa", "gliss"},
		{"glissaient", "gliss"},
		{"glissait", "gliss"},
		{"glissant", "gliss"},
		{"glisse", "gliss"},
		{"glissé", "gliss"},
		{"glissent", "glissent"},
		{"glisser", "gliss"},
		{"glissera", "gliss"},
		{"glissons", "glisson"},
		{"globe", "glob"},
		{"gloire", "gloir"},
		{"glorieux", "glorieux"},
		{"glory", "glory"},
		{"gloutons", "glouton"},
		{"glu", "glu"},
		{"goa", "go"},
		{"gobelet", "gobelet"},
		{"godart", "godart"},
		{"godavery", "godavery"},
		{"godille", "godill"},
		{"goélette", "goélet"},
		{"goélettes", "goélet"},
		{"goethe", "goeth"},
		{"goguenard", "goguenard"},
		{"goguenards", "goguenard"},
		{"golconde", "golcond"},
		{"goldoni", "goldon"},
		{"golfe", "golf"},
		{"golfes", "golf"},
		{"golgonda", "golgond"},
		{"golgonde", "golgond"},
		{"gonds", "gond"},
		{"gongs", "gong"},
		{"gonzo", "gonzo"},
		{"gorge", "gorg"},
		{"gorgées", "gorg"},
		{"gorges", "gorg"},
		{"gothard", "gothard"},
		{"gothique", "gothiqu"},
		{"gothiques", "gothiqu"},
		{"gouailler", "gouaill"},
		{"goudron", "goudron"},
		{"goulée", "goul"},
		{"gour", "gour"},
		{"gourmande", "gourmand"},
		{"gourmandise", "gourmandis"},
		{"gourmets", "gourmet"},
		{"gousset", "gousset"},
		{"goût", "goût"},
		{"goûta", "goût"},
		{"goûtant", "goût"},
		{"goûte", "goût"},
		{"goûté", "goût"},
		{"goûtées", "goût"},
		{"goûter", "goût"},
		{"goûterait", "goût"},
		{"goûtés", "goût"},
		{"goûts", "goût"},
		{"goutte", "goutt"},
		{"gouttes", "goutt"},
		{"gouvernail", "gouvernail"},
		{"gouvernait", "gouvern"},
		{"gouverne", "gouvern"},
		{"gouverné", "gouvern"},
		{"gouvernement", "gouvern"},
		{"gouvernements", "gouvern"},
		{"gouvernent", "gouvernent"},
		{"gouverner", "gouvern"},
		{"gouverneriez", "gouvern"},
		{"gouverneur", "gouverneur"},
		{"gouverneurs", "gouverneur"},
		{"gouvion", "gouvion"},
		{"government", "government"},
		{"grâce", "grâc"},
		{"grâces", "grâc"},
		{"gracieuse", "gracieux"},
		{"gracieusement", "gracieux"},
		{"gracieuses", "gracieux"},
		{"gracieux", "gracieux"},
		{"grade", "grad"},
		{"graduée", "gradu"},
		{"graduellement", "graduel"},
		{"grain", "grain"},
		{"grains", "grain"},
		{"graisse", "graiss"},
		{"graisser", "graiss"},
		{"grand", "grand"},
		{"grande", "grand"},
		{"grandement", "grand"},
		{"grandes", "grand"},
		{"grandeur", "grandeur"},
		{"grandeurs", "grandeur"},
		{"grandi", "grand"},
		{"grandie", "grand"},
		{"grandiose", "grandios"},
		{"grandis", "grand"},
		{"grandit", "grand"},
		{"grands", "grand"},
		{"granit", "gran"},
		{"grant", "grant"},
		{"granvelle", "granvel"},
		{"grappe", "grapp"},
		{"gras", "gras"},
		{"grasse", "grass"},
		{"gratification", "gratif"},
		{"gratifié", "gratifi"},
		{"gratis", "grat"},
		{"gratius", "gratius"},
		{"gratter", "gratt"},
		{"gratuite", "gratuit"},
		{"gratuitement", "gratuit"},
		{"grave", "grav"},
		{"gravé", "grav"},
		{"gravement", "grav"},
		{"graver", "grav"},
		{"graves", "grav"},
		{"graveur", "graveur"},
		{"gravir", "grav"},
		{"gravissaient", "grav"},
		{"gravitaient", "gravit"},
		{"gravité", "gravit"},
		{"gravures", "gravur"},
		{"gray", "gray"},
		{"gré", "gré"},
		{"great", "great"},
		{"greatly", "greatly"},
		{"grec", "grec"},
		{"grèce", "grec"},
		{"grecque", "grecqu"},
		{"grecques", "grecqu"},
		{"gredin", "gredin"},
		{"gréé", "gré"},
		{"green", "green"},
		{"greenwich", "greenwich"},
		{"gréer", "gré"},
		{"greffier", "greffi"},
		{"grégoire", "grégoir"},
		{"grêles", "grêl"},
		{"grenade", "grenad"},
		{"grenadier", "grenadi"},
		{"grenadiers", "grenadi"},
		{"grenier", "greni"},
		{"grève", "grev"},
		{"grianta", "griant"},
		{"griefs", "grief"},
		{"grièvement", "griev"},
		{"griffe", "griff"},
		{"griffonnages", "griffonnag"},
		{"griffonnait", "griffon"},
		{"grillages", "grillag"},
		{"grille", "grill"},
		{"grillé", "grill"},
		{"grillée", "grill"},
		{"grillées", "grill"},
		{"grilles", "grill"},
		{"grillo", "grillo"},
		{"grimaçant", "grimac"},
		{"grimaçante", "grimac"},
		{"grimaçantes", "grimac"},
		{"grimaçants", "grimac"},
		{"grimace", "grimac"},
		{"grimaces", "grimac"},
		{"grimer", "grim"},
		{"grimpait", "grimp"},
		{"grimpant", "grimp"},
		{"gringalet", "gringalet"},
		{"gris", "gris"},
		{"grisâtre", "grisâtr"},
		{"grise", "gris"},
		{"grisé", "gris"},
		{"grisonnants", "grison"},
		{"grivelé", "grivel"},
		{"grogeot", "grogeot"},
		{"grognement", "grogn"},
		{"grognon", "grognon"},
		{"gronda", "grond"},
		{"grondait", "grond"},
		{"grondant", "grond"},
		{"gronde", "grond"},
		{"grondé", "grond"},
		{"grondée", "grond"},
		{"gronder", "grond"},
		{"gronderez", "grond"},
		{"grondez", "grond"},
		{"groom", "groom"},
		{"gros", "gros"},
		{"groseilles", "groseil"},
		{"grosse", "gross"},
		{"grosses", "gross"},
		{"grossesse", "grossess"},
		{"grosseur", "grosseur"},
		{"grossie", "gross"},
		{"grossier", "grossi"},
		{"grossière", "grossi"},
		{"grossièrement", "grossi"},
		{"grossières", "grossi"},
		{"grossièreté", "grossièret"},
		{"grossièretés", "grossièret"},
		{"grossiers", "grossi"},
		{"grotesque", "grotesqu"},
		{"grotesques", "grotesqu"},
		{"grotte", "grott"},
		{"grottes", "grott"},
		{"grouillait", "grouill"},
		{"groupe", "group"},
		{"groupes", "group"},
		{"grues", "gru"},
		{"guardate", "guardat"},
		{"guèbres", "guebr"},
		{"guenilles", "guenill"},
		{"guépards", "guépard"},
		{"guerchin", "guerchin"},
		{"guère", "guer"},
		{"guéri", "guer"},
		{"guérin", "guérin"},
		{"guérir", "guer"},
		{"guérison", "guérison"},
		{"guérissez", "guer"},
		{"guérit", "guer"},
		{"guérite", "guérit"},
		{"guérites", "guérit"},
		{"guerre", "guerr"},
		{"guerres", "guerr"},
		{"guerriers", "guerri"},
		{"guess", "guess"},
		{"guet", "guet"},
		{"guetta", "guet"},
		{"guettait", "guet"},
		{"guette", "guet"},
		{"guettent", "guettent"},
		{"guetteurs", "guetteur"},
		{"gueule", "gueul"},
		{"gueux", "gueux"},
		{"gui", "gui"},
		{"guichet", "guichet"},
		{"guichetier", "guicheti"},
		{"guidant", "guid"},
		{"guide", "guid"},
		{"guidé", "guid"},
		{"guidée", "guid"},
		{"guider", "guid"},
		{"guidés", "guid"},
		{"guidon", "guidon"},
		{"guignon", "guignon"},
		{"guillotinaient", "guillotin"},
		{"guillotine", "guillotin"},
		{"guillotiné", "guillotin"},
		{"guillotiner", "guillotin"},
		{"guillotinés", "guillotin"},
		{"guindé", "guind"},
		{"guinder", "guind"},
		{"guinée", "guin"},
		{"guinées", "guin"},
		{"guise", "guis"},
		{"guitare", "guitar"},
		{"gustave", "gustav"},
		{"guy", "guy"},
		{"gymnase", "gymnas"},
		{"gymnaste", "gymnast"},
		{"gymnastes", "gymnast"},
		{"gymnastique", "gymnast"},
		{"h", "h"},
		{"ha", "ha"},
		{"habile", "habil"},
		{"habilement", "habil"},
		{"habiles", "habil"},
		{"habileté", "habilet"},
		{"habilla", "habill"},
		{"habillait", "habill"},
		{"habillant", "habill"},
		{"habille", "habill"},
		{"habillé", "habill"},
		{"habillée", "habill"},
		{"habillement", "habill"},
		{"habiller", "habill"},
		{"habillés", "habill"},
		{"habit", "hab"},
		{"habitais", "habit"},
		{"habitait", "habit"},
		{"habitant", "habit"},
		{"habitants", "habit"},
		{"habitation", "habit"},
		{"habite", "habit"},
		{"habité", "habit"},
		{"habitée", "habit"},
		{"habiter", "habit"},
		{"habiterai", "habit"},
		{"habiteras", "habit"},
		{"habités", "habit"},
		{"habitez", "habit"},
		{"habits", "habit"},
		{"habitude", "habitud"},
		{"habitudes", "habitud"},
		{"habituée", "habitu"},
		{"habituel", "habituel"},
		{"habituelle", "habituel"},
		{"habituellement", "habituel"},
		{"habituelles", "habituel"},
		{"habituels", "habituel"},
		{"habitués", "habitu"},
		{"hache", "hach"},
		{"haches", "hach"},
		{"hagard", "hagard"},
		{"hagards", "hagard"},
		{"haï", "haï"},
		{"haie", "hai"},
		{"haies", "hai"},
		{"haillons", "haillon"},
		{"haine", "hain"},
		{"haines", "hain"},
		{"haineuse", "haineux"},
		{"haineuses", "haineux"},
		{"hair", "hair"},
		{"haïr", "haïr"},
		{"haïraient", "haïr"},
		{"hais", "hais"},
		{"haïssaient", "haïss"},
		{"haïssait", "haïss"},
		{"haïssent", "haïssent"},
		{"hait", "hait"},
		{"hâla", "hâl"},
		{"halage", "halag"},
		{"hâle", "hâl"},
		{"haleine", "halein"},
		{"haletant", "halet"},
		{"halleck", "halleck"},
		{"halte", "halt"},
		{"haltes", "halt"},
		{"hambourg", "hambourg"},
		{"hambourgeois", "hambourgeois"},
		{"hambourgeoise", "hambourgeois"},
		{"hameau", "hameau"},
		{"hampe", "hamp"},
		{"hanche", "hanch"},
		{"hand", "hand"},
		{"hang", "hang"},
		{"hangar", "hangar"},
		{"hantée", "hant"},
		{"hantées", "hant"},
		{"happy", "happy"},
		{"harangua", "harangu"},
		{"haranguer", "harangu"},
		{"harassés", "harass"},
		{"hardes", "hard"},
		{"hardi", "hard"},
		{"hardie", "hard"},
		{"hardies", "hard"},
		{"hardiesse", "hardiess"},
		{"hardiment", "hard"},
		{"hardis", "hard"},
		{"harmonie", "harmon"},
		{"harmonieusement", "harmoni"},
		{"harmonieux", "harmoni"},
		{"harmoniques", "harmon"},
		{"harpe", "harp"},
		{"hasard", "hasard"},
		{"hasarda", "hasard"},
		{"hasardaient", "hasard"},
		{"hasarde", "hasard"},
		{"hasardé", "hasard"},
		{"hasarder", "hasard"},
		{"hasardera", "hasard"},
		{"hasardeuse", "hasard"},
		{"hasardeuses", "hasard"},
		{"hasardeux", "hasard"},
		{"hasards", "hasard"},
		{"hâta", "hât"},
		{"hâtait", "hât"},
		{"hâte", "hât"},
		{"hâté", "hât"},
		{"hâter", "hât"},
		{"hâtèrent", "hât"},
		{"hâtez", "hât"},
		{"hâtons", "hâton"},
		{"haubans", "hauban"},
		{"haubert", "haubert"},
		{"haus", "haus"},
		{"haussant", "hauss"},
		{"haussé", "hauss"},
		{"haussement", "hauss"},
		{"haut", "haut"},
		{"hautain", "hautain"},
		{"hautaine", "hautain"},
		{"haute", "haut"},
		{"hautement", "haut"},
		{"hautes", "haut"},
		{"hauteur", "hauteur"},
		{"hauteurs", "hauteur"},
		{"hauts", "haut"},
		{"havre", "havr"},
		{"hay", "hay"},
		{"hayez", "hai"},
		{"he", "he"},
		{"hé", "hé"},
		{"heaviest", "heaviest"},
		{"hébété", "hébet"},
		{"hébétés", "hébet"},
		{"hein", "hein"},
		{"héla", "hel"},
		{"hélas", "hel"},
		{"hélène", "hélen"},
		{"hélice", "hélic"},
		{"héloïse", "héloïs"},
		{"helvétie", "helvet"},
		{"hémisphère", "hémispher"},
		{"henné", "hen"},
		{"hennissant", "hen"},
		{"hennissement", "hen"},
		{"hennissements", "hen"},
		{"hennit", "hen"},
		{"henri", "henr"},
		{"henriet", "henriet"},
		{"henrietta", "henriet"},
		{"henry", "henry"},
		{"her", "her"},
		{"héraut", "héraut"},
		{"hérauts", "héraut"},
		{"herbe", "herb"},
		{"herbes", "herb"},
		{"hercule", "hercul"},
		{"herculéenne", "herculéen"},
		{"hère", "her"},
		{"héréditaire", "héréditair"},
		{"hérédité", "héred"},
		{"hères", "her"},
		{"hérésie", "héres"},
		{"hérésies", "héres"},
		{"hérisse", "her"},
		{"hérissé", "hériss"},
		{"hérissée", "hériss"},
		{"hérissées", "hériss"},
		{"hérissés", "hériss"},
		{"héritage", "héritag"},
		{"hérité", "hérit"},
		{"héritier", "hériti"},
		{"héritière", "hériti"},
		{"hermétiquement", "hermet"},
		{"hernani", "hernan"},
		{"hérodiade", "hérodiad"},
		{"hérodiades", "hérodiad"},
		{"héroïne", "héroïn"},
		{"héroïque", "héroïqu"},
		{"héroïquement", "héroïqu"},
		{"héroïques", "héroïqu"},
		{"héroïsme", "héroïsm"},
		{"héron", "héron"},
		{"héros", "héros"},
		{"hers", "her"},
		{"hésita", "hésit"},
		{"hésitait", "hésit"},
		{"hésitant", "hésit"},
		{"hésitation", "hésit"},
		{"hésite", "hésit"},
		{"hésité", "hésit"},
		{"hésiter", "hésit"},
		{"hésiterais", "hésit"},
		{"hésiteront", "hésit"},
		{"hésitiez", "hésit"},
		{"hêtre", "hêtr"},
		{"hêtres", "hêtr"},
		{"heure", "heur"},
		{"heures", "heur"},
		{"heureuse", "heureux"},
		{"heureusement", "heureux"},
		{"heureuses", "heureux"},
		{"heureux", "heureux"},
		{"heurtaient", "heurt"},
		{"heurte", "heurt"},
		{"heurter", "heurt"},
		{"hic", "hic"},
		{"hideuse", "hideux"},
		{"hideuses", "hideux"},
		{"hideux", "hideux"},
		{"hier", "hi"},
		{"hiérarchie", "hiérarch"},
		{"hill", "hill"},
		{"himalaya", "himalai"},
		{"hindous", "hindous"},
		{"hip", "hip"},
		{"hippolyte", "hippolyt"},
		{"his", "his"},
		{"hissa", "hiss"},
		{"hissé", "hiss"},
		{"hissées", "hiss"},
		{"histoire", "histoir"},
		{"histoires", "histoir"},
		{"historien", "historien"},
		{"historiens", "historien"},
		{"historique", "histor"},
		{"historiques", "histor"},
		{"histrion", "histrion"},
		{"hitch", "hitch"},
		{"hiver", "hiv"},
		{"ho", "ho"},
		{"hobbes", "hobb"},
		{"hobereau", "hobereau"},
		{"hobereaux", "hobereau"},
		{"hoc", "hoc"},
		{"hochant", "hoch"},
		{"hoche", "hoch"},
		{"holà", "holà"},
		{"holland", "holland"},
		{"hollandais", "holland"},
		{"hollandaises", "hollandais"},
		{"hollande", "holland"},
		{"homélie", "homel"},
		{"homélies", "homel"},
		{"homère", "homer"},
		{"hommage", "hommag"},
		{"hommages", "hommag"},
		{"homme", "homm"},
		{"hommes", "homm"},
		{"hong", "hong"},
		{"hongrois", "hongrois"},
		{"honnête", "honnêt"},
		{"honnêtes", "honnêt"},
		{"honnêteté", "honnêtet"},
		{"honnêtetés", "honnêtet"},
		{"honneur", "honneur"},
		{"honneurs", "honneur"},
		{"honorabilité", "honor"},
		{"honorable", "honor"},
		{"honorables", "honor"},
		{"honore", "honor"},
		{"honoré", "honor"},
		{"honorée", "honor"},
		{"honorent", "honorent"},
		{"honorer", "honor"},
		{"honorera", "honor"},
		{"honorés", "honor"},
		{"honte", "hont"},
		{"honteuse", "honteux"},
		{"honteusement", "honteux"},
		{"honteux", "honteux"},
		{"honur", "honur"},
		{"hook", "hook"},
		{"hôpital", "hôpital"},
		{"hôpitaux", "hôpital"},
		{"horace", "horac"},
		{"horde", "hord"},
		{"horions", "horion"},
		{"horizon", "horizon"},
		{"horloge", "horlog"},
		{"horlogerie", "horloger"},
		{"horloges", "horlog"},
		{"horreur", "horreur"},
		{"horreurs", "horreur"},
		{"horrible", "horribl"},
		{"horriblement", "horribl"},
		{"horribles", "horribl"},
		{"hors", "hor"},
		{"hospice", "hospic"},
		{"hospitalité", "hospital"},
		{"hostie", "host"},
		{"hostile", "hostil"},
		{"hostiles", "hostil"},
		{"hostilité", "hostil"},
		{"hôte", "hôt"},
		{"hôtel", "hôtel"},
		{"hôtels", "hôtel"},
		{"hôtes", "hôt"},
		{"hôtesse", "hôtess"},
		{"hôtesses", "hôtess"},
		{"hotte", "hott"},
		{"hottentots", "hottentot"},
		{"hougly", "hougly"},
		{"houille", "houill"},
		{"houle", "houl"},
		{"houles", "houl"},
		{"houleuse", "houleux"},
		{"housards", "housard"},
		{"house", "hous"},
		{"housse", "houss"},
		{"how", "how"},
		{"hudson", "hudson"},
		{"hugo", "hugo"},
		{"hui", "hui"},
		{"huile", "huil"},
		{"huilé", "huil"},
		{"huissier", "huissi"},
		{"huit", "huit"},
		{"huitième", "huitiem"},
		{"huîtres", "huîtr"},
		{"humain", "humain"},
		{"humaine", "humain"},
		{"humainement", "humain"},
		{"humaines", "humain"},
		{"humains", "humain"},
		{"humait", "hum"},
		{"humaniste", "human"},
		{"humanité", "human"},
		{"humanités", "human"},
		{"humble", "humbl"},
		{"humblement", "humbl"},
		{"humbles", "humbl"},
		{"humboldt", "humboldt"},
		{"hume", "hum"},
		{"humer", "hum"},
		{"humeur", "humeur"},
		{"humide", "humid"},
		{"humides", "humid"},
		{"humidité", "humid"},
		{"humiliant", "humili"},
		{"humiliante", "humili"},
		{"humiliantes", "humili"},
		{"humiliants", "humili"},
		{"humiliation", "humili"},
		{"humiliations", "humili"},
		{"humilie", "humil"},
		{"humilié", "humili"},
		{"humiliée", "humili"},
		{"humilient", "humilient"},
		{"humilier", "humili"},
		{"humiliez", "humil"},
		{"humilité", "humil"},
		{"hune", "hun"},
		{"huniers", "huni"},
		{"hurlante", "hurl"},
		{"hurlements", "hurl"},
		{"hurlent", "hurlent"},
		{"hurler", "hurl"},
		{"hurrah", "hurrah"},
		{"hurrahs", "hurrah"},
		{"hussard", "hussard"},
		{"hussarde", "hussard"},
		{"hussards", "hussard"},
		{"hutte", "hutt"},
		{"huttes", "hutt"},
		{"hvram", "hvram"},
		{"hydrogène", "hydrogen"},
		{"hydrographique", "hydrograph"},
		{"hyères", "hyer"},
		{"hypocrisie", "hypocris"},
		{"hypocrisies", "hypocris"},
		{"hypocrite", "hypocrit"},
		{"hypocrites", "hypocrit"},
		{"hypogées", "hypog"},
		{"hypothèse", "hypothes"},
		{"i", "i"},
		{"iago", "iago"},
		{"ici", "ici"},
		{"ideal", "ideal"},
		{"idéal", "idéal"},
		{"idéale", "idéal"},
		{"idée", "idé"},
		{"idées", "idé"},
		{"identique", "ident"},
		{"identité", "ident"},
		{"idiots", "idiot"},
		{"idolâtrie", "idolâtr"},
		{"idole", "idol"},
		{"ignares", "ignar"},
		{"ignes", "igne"},
		{"ignoble", "ignobl"},
		{"ignobles", "ignobl"},
		{"ignominie", "ignomin"},
		{"ignoraient", "ignor"},
		{"ignorais", "ignor"},
		{"ignorait", "ignor"},
		{"ignorance", "ignor"},
		{"ignorant", "ignor"},
		{"ignorante", "ignor"},
		{"ignore", "ignor"},
		{"ignorent", "ignorent"},
		{"ignorer", "ignor"},
		{"ignores", "ignor"},
		{"ignoriez", "ignor"},
		{"ii", "ii"},
		{"iii", "iii"},
		{"il", "il"},
		{"île", "île"},
		{"îles", "île"},
		{"illicites", "illicit"},
		{"illimité", "illim"},
		{"illimitée", "illimit"},
		{"illinois", "illinois"},
		{"illisible", "illisibl"},
		{"illisibles", "illisibl"},
		{"illumina", "illumin"},
		{"illumination", "illumin"},
		{"illuminé", "illumin"},
		{"illuminée", "illumin"},
		{"illuminer", "illumin"},
		{"illusion", "illus"},
		{"illusions", "illus"},
		{"illustrated", "illustrated"},
		{"illustration", "illustr"},
		{"illustre", "illustr"},
		{"illustrer", "illustr"},
		{"illustrissime", "illustrissim"},
		{"îlot", "îlot"},
		{"îlots", "îlot"},
		{"ils", "il"},
		{"image", "imag"},
		{"images", "imag"},
		{"imagina", "imagin"},
		{"imaginaient", "imagin"},
		{"imaginaire", "imaginair"},
		{"imaginaires", "imaginair"},
		{"imaginais", "imagin"},
		{"imaginait", "imagin"},
		{"imaginant", "imagin"},
		{"imagination", "imagin"},
		{"imaginations", "imagin"},
		{"imaginative", "imagin"},
		{"imagine", "imagin"},
		{"imaginé", "imagin"},
		{"imaginer", "imagin"},
		{"imbécile", "imbécil"},
		{"imbéciles", "imbécil"},
		{"imita", "imit"},
		{"imitait", "imit"},
		{"imitant", "imit"},
		{"imitation", "imit"},
		{"imité", "imit"},
		{"imitées", "imit"},
		{"imiter", "imit"},
		{"imman", "imman"},
		{"immanquable", "immanqu"},
		{"immanquablement", "immanqu"},
		{"immédiat", "immédiat"},
		{"immédiate", "immédiat"},
		{"immédiatement", "immédiat"},
		{"immédiates", "immédiat"},
		{"immédiats", "immédiat"},
		{"immense", "immens"},
		{"immensément", "immens"},
		{"immenses", "immens"},
		{"immensité", "immens"},
		{"imminence", "imminent"},
		{"imminent", "imminent"},
		{"immobile", "immobil"},
		{"immobiles", "immobil"},
		{"immolé", "immol"},
		{"immoler", "immol"},
		{"immonde", "immond"},
		{"immoral", "immoral"},
		{"immorales", "immoral"},
		{"immoralité", "immoral"},
		{"immortalisé", "immortalis"},
		{"immortaliser", "immortalis"},
		{"immortel", "immortel"},
		{"immuable", "immuabl"},
		{"imola", "imol"},
		{"impardonnable", "impardon"},
		{"impardonnables", "impardon"},
		{"imparfait", "imparf"},
		{"imparfaite", "imparfait"},
		{"imparfaitement", "imparfait"},
		{"impassibilité", "impassibil"},
		{"impassible", "impassibl"},
		{"impatience", "impatient"},
		{"impatient", "impatient"},
		{"impatienta", "impatient"},
		{"impatientait", "impatient"},
		{"impatientant", "impatient"},
		{"impatientante", "impatient"},
		{"impatiente", "impatient"},
		{"impatienté", "impatient"},
		{"impatientée", "impatient"},
		{"impatienter", "impatient"},
		{"impatientera", "impatient"},
		{"impayable", "impai"},
		{"impénétrable", "impénetr"},
		{"impératif", "imper"},
		{"impératrice", "imper"},
		{"imperceptibles", "imperceptibl"},
		{"imperfection", "imperfect"},
		{"impérial", "impérial"},
		{"impériale", "impérial"},
		{"impériales", "impérial"},
		{"impérieuse", "impéri"},
		{"impérieusement", "impéri"},
		{"impérieux", "impéri"},
		{"impertinence", "impertinent"},
		{"impertinences", "impertinent"},
		{"impertinent", "impertinent"},
		{"impertinente", "impertinent"},
		{"impertinentes", "impertinent"},
		{"imperturbable", "imperturb"},
		{"imperturbablement", "imperturb"},
		{"impétueusement", "impétu"},
		{"impie", "impi"},
		{"impies", "impi"},
		{"impiété", "impiet"},
		{"impitoyable", "impitoi"},
		{"impitoyablement", "impitoi"},
		{"implacable", "implac"},
		{"implicite", "implicit"},
		{"implora", "implor"},
		{"implorer", "implor"},
		{"impoli", "impol"},
		{"impolie", "impol"},
		{"impoliment", "impol"},
		{"importait", "import"},
		{"importance", "import"},
		{"important", "import"},
		{"importante", "import"},
		{"importantes", "import"},
		{"importants", "import"},
		{"importe", "import"},
		{"importée", "import"},
		{"importent", "importent"},
		{"importés", "import"},
		{"importun", "importun"},
		{"importuna", "importun"},
		{"importune", "importun"},
		{"importunités", "importun"},
		{"importuns", "importun"},
		{"imposa", "impos"},
		{"imposaient", "impos"},
		{"imposait", "impos"},
		{"imposant", "impos"},
		{"imposante", "impos"},
		{"impose", "impos"},
		{"imposé", "impos"},
		{"imposée", "impos"},
		{"imposées", "impos"},
		{"imposent", "imposent"},
		{"imposer", "impos"},
		{"imposera", "impos"},
		{"imposèrent", "impos"},
		{"imposés", "impos"},
		{"impositions", "imposit"},
		{"impossibilité", "impossibil"},
		{"impossible", "impossibl"},
		{"impossibles", "impossibl"},
		{"impraticable", "impratic"},
		{"imprécations", "imprec"},
		{"imprégnait", "impregn"},
		{"imprégné", "impregn"},
		{"impresario", "impresario"},
		{"imprescriptibles", "imprescriptibl"},
		{"impression", "impress"},
		{"impressionna", "impression"},
		{"impressionnait", "impression"},
		{"impressions", "impress"},
		{"imprévu", "imprévu"},
		{"imprévue", "imprévu"},
		{"imprévues", "imprévu"},
		{"imprima", "imprim"},
		{"imprimait", "imprim"},
		{"imprime", "imprim"},
		{"imprimé", "imprim"},
		{"imprimée", "imprim"},
		{"imprimées", "imprim"},
		{"imprimer", "imprim"},
		{"imprimerie", "imprimer"},
		{"imprimés", "imprim"},
		{"imprimeur", "imprimeur"},
		{"imprimeurs", "imprimeur"},
		{"improbable", "improb"},
		{"improbables", "improb"},
		{"impromptu", "impromptu"},
		{"impromptue", "impromptu"},
		{"impropre", "impropr"},
		{"improprement", "impropr"},
		{"improvisais", "improvis"},
		{"improvisait", "improvis"},
		{"improvisée", "improvis"},
		{"improvisées", "improvis"},
		{"improvises", "improvis"},
		{"improvisés", "improvis"},
		{"improviste", "improv"},
		{"imprudemment", "imprudent"},
		{"imprudence", "imprudent"},
		{"imprudences", "imprudent"},
		{"imprudent", "imprudent"},
		{"imprudente", "imprudent"},
		{"imprudentes", "imprudent"},
		{"imprudents", "imprudent"},
		{"impudence", "impudent"},
		{"impudent", "impudent"},
		{"impudents", "impudent"},
		{"impuissante", "impuiss"},
		{"impuissants", "impuiss"},
		{"impulsion", "impuls"},
		{"impunément", "impun"},
		{"impunité", "impun"},
		{"imputant", "imput"},
		{"imputations", "imput"},
		{"imputées", "imput"},
		{"imputer", "imput"},
		{"in", "in"},
		{"ina", "ina"},
		{"inaccessible", "inaccessibl"},
		{"inaccessibles", "inaccessibl"},
		{"inaccoutumé", "inaccoutum"},
		{"inaccoutumée", "inaccoutum"},
		{"inaccoutumées", "inaccoutum"},
		{"inaction", "inact"},
		{"inadmissible", "inadmissibl"},
		{"inadmissibles", "inadmissibl"},
		{"inaltérable", "inalter"},
		{"inamovible", "inamovibl"},
		{"inanimé", "inanim"},
		{"inanimée", "inanim"},
		{"inanimés", "inanim"},
		{"inaperçu", "inaperçu"},
		{"inaperçue", "inaperçu"},
		{"inaperçues", "inaperçu"},
		{"inarticulés", "inarticul"},
		{"inattaquable", "inattaqu"},
		{"inattaquables", "inattaqu"},
		{"inattendu", "inattendu"},
		{"inattendue", "inattendu"},
		{"inattention", "inattent"},
		{"inauguration", "inaugur"},
		{"inaugurer", "inaugur"},
		{"incapable", "incap"},
		{"incarne", "incarn"},
		{"incarnée", "incarn"},
		{"incartade", "incartad"},
		{"incedo", "incedo"},
		{"incendiaires", "incendiair"},
		{"incendie", "incend"},
		{"incendies", "incend"},
		{"incertain", "incertain"},
		{"incertaine", "incertain"},
		{"incertains", "incertain"},
		{"incertitude", "incertitud"},
		{"incertitudes", "incertitud"},
		{"incessamment", "incess"},
		{"incessant", "incess"},
		{"incessants", "incess"},
		{"inceste", "incest"},
		{"incident", "incident"},
		{"incidenter", "incident"},
		{"incidents", "incident"},
		{"incisif", "incis"},
		{"incisions", "incis"},
		{"incisive", "incis"},
		{"incisives", "incis"},
		{"inclina", "inclin"},
		{"inclinaient", "inclin"},
		{"inclination", "inclin"},
		{"incliné", "inclin"},
		{"inclinée", "inclin"},
		{"inclinera", "inclin"},
		{"inclus", "inclus"},
		{"incluse", "inclus"},
		{"incognito", "incognito"},
		{"incommensurable", "incommensur"},
		{"incommode", "incommod"},
		{"incommodée", "incommod"},
		{"incommodes", "incommod"},
		{"incommodité", "incommod"},
		{"incomodo", "incomodo"},
		{"incompatible", "incompatibl"},
		{"incompatibles", "incompatibl"},
		{"incomplètes", "incomplet"},
		{"incongruité", "incongru"},
		{"inconnu", "inconnu"},
		{"inconnue", "inconnu"},
		{"inconnus", "inconnus"},
		{"inconsciemment", "inconscient"},
		{"inconsciente", "inconscient"},
		{"inconsidérée", "inconsider"},
		{"inconsolable", "inconsol"},
		{"inconstance", "inconst"},
		{"inconstant", "inconst"},
		{"inconstante", "inconst"},
		{"incontestable", "incontest"},
		{"incontestablement", "incontest"},
		{"inconvenance", "inconven"},
		{"inconvenant", "inconven"},
		{"inconvenante", "inconven"},
		{"inconvenantes", "inconven"},
		{"inconvénient", "inconvénient"},
		{"inconvénients", "inconvénient"},
		{"incorporer", "incorpor"},
		{"incrédule", "incrédul"},
		{"incrédulité", "incrédul"},
		{"incriminait", "incrimin"},
		{"incroyable", "incroi"},
		{"incroyables", "incroi"},
		{"incrusta", "incrust"},
		{"incruster", "incrust"},
		{"incrustés", "incrust"},
		{"incurie", "incur"},
		{"inde", "inde"},
		{"indécence", "indécent"},
		{"indécent", "indécent"},
		{"indécentes", "indécent"},
		{"indécents", "indécent"},
		{"indéchiffrable", "indéchiffr"},
		{"indécis", "indec"},
		{"indécise", "indécis"},
		{"indéfini", "indéfin"},
		{"indéfinissable", "indéfiniss"},
		{"indélicatesse", "indélicatess"},
		{"indemne", "indemn"},
		{"indépendamment", "indépend"},
		{"independance", "independ"},
		{"indépendance", "indépend"},
		{"indépendant", "indépend"},
		{"indépendante", "indépend"},
		{"indépendantes", "indépend"},
		{"indépendants", "indépend"},
		{"indes", "inde"},
		{"indian", "indian"},
		{"indiana", "indian"},
		{"indication", "indiqu"},
		{"indications", "indiqu"},
		{"indices", "indic"},
		{"indicible", "indicibl"},
		{"indicibles", "indicibl"},
		{"indien", "indien"},
		{"indienne", "indien"},
		{"indiens", "indien"},
		{"indifférence", "indifférent"},
		{"indifférent", "indifférent"},
		{"indifférents", "indifférent"},
		{"indigène", "indigen"},
		{"indigènes", "indigen"},
		{"indigna", "indign"},
		{"indignation", "indign"},
		{"indignations", "indign"},
		{"indigne", "indign"},
		{"indigné", "indign"},
		{"indignée", "indign"},
		{"indignement", "indign"},
		{"indigner", "indign"},
		{"indignes", "indign"},
		{"indignés", "indign"},
		{"indigo", "indigo"},
		{"indiqua", "indiqu"},
		{"indiquaient", "indiqu"},
		{"indiquait", "indiqu"},
		{"indiquant", "indiqu"},
		{"indique", "indiqu"},
		{"indiqué", "indiqu"},
		{"indiquée", "indiqu"},
		{"indiquent", "indiquent"},
		{"indiquer", "indiqu"},
		{"indiquera", "indiqu"},
		{"indiquerait", "indiqu"},
		{"indiqués", "indiqu"},
		{"indiquez", "indiqu"},
		{"indirect", "indirect"},
		{"indirecte", "indirect"},
		{"indirectement", "indirect"},
		{"indirectes", "indirect"},
		{"indiscret", "indiscret"},
		{"indiscrète", "indiscret"},
		{"indiscrètes", "indiscret"},
		{"indiscrétion", "indiscret"},
		{"indiscrétions", "indiscret"},
		{"indiscrets", "indiscret"},
		{"indispensable", "indispens"},
		{"indispensables", "indispens"},
		{"indisposé", "indispos"},
		{"indisposée", "indispos"},
		{"indisposition", "indisposit"},
		{"indistincte", "indistinct"},
		{"individu", "individu"},
		{"individuelle", "individuel"},
		{"individus", "individus"},
		{"indou", "indou"},
		{"indoue", "indou"},
		{"indoues", "indou"},
		{"indous", "indous"},
		{"indubitable", "indubit"},
		{"indue", "indu"},
		{"indulgence", "indulgent"},
		{"indulgent", "indulgent"},
		{"indulgente", "indulgent"},
		{"indulgents", "indulgent"},
		{"indus", "indus"},
		{"industrie", "industr"},
		{"industriel", "industriel"},
		{"industrielle", "industriel"},
		{"industriels", "industriel"},
		{"industrieux", "industri"},
		{"inébranlable", "inébranl"},
		{"inedit", "ined"},
		{"inédit", "ined"},
		{"ineffaçable", "ineffac"},
		{"inégal", "inégal"},
		{"inégalement", "inégal"},
		{"inégales", "inégal"},
		{"inélégantes", "inéleg"},
		{"inéprouvé", "inéprouv"},
		{"inepte", "inept"},
		{"ineptes", "inept"},
		{"inépuisable", "inépuis"},
		{"inerte", "inert"},
		{"inespéré", "inesper"},
		{"inévitablement", "inévit"},
		{"inévitables", "inévit"},
		{"inexactitude", "inexactitud"},
		{"inexécutable", "inexécut"},
		{"inexorable", "inexor"},
		{"inexpérimenté", "inexpériment"},
		{"inexplicable", "inexplic"},
		{"inexprimable", "inexprim"},
		{"inexprimables", "inexprim"},
		{"inextricable", "inextric"},
		{"infâme", "infâm"},
		{"infâmes", "infâm"},
		{"infamie", "infam"},
		{"infamies", "infam"},
		{"infanterie", "infanter"},
		{"infatigable", "infatig"},
		{"inférieur", "inférieur"},
		{"inférieure", "inférieur"},
		{"inférieures", "inférieur"},
		{"inférieurs", "inférieur"},
		{"infériorité", "inférior"},
		{"infernale", "infernal"},
		{"infidèle", "infidel"},
		{"infidélité", "infidel"},
		{"infini", "infin"},
		{"infinie", "infin"},
		{"infinies", "infin"},
		{"infiniment", "infin"},
		{"infinis", "infin"},
		{"infinité", "infin"},
		{"infirmerie", "infirmer"},
		{"infirmités", "infirm"},
		{"inflammation", "inflamm"},
		{"infléchit", "inflech"},
		{"infligé", "inflig"},
		{"infligée", "inflig"},
		{"infliger", "inflig"},
		{"influence", "influenc"},
		{"influencer", "influenc"},
		{"influent", "influent"},
		{"influents", "influent"},
		{"influer", "influ"},
		{"informa", "inform"},
		{"informait", "inform"},
		{"informations", "inform"},
		{"informée", "inform"},
		{"informes", "inform"},
		{"informés", "inform"},
		{"infortune", "infortun"},
		{"infortuné", "infortun"},
		{"infortunée", "infortun"},
		{"infortunes", "infortun"},
		{"infortunés", "infortun"},
		{"infraction", "infract"},
		{"infusion", "infus"},
		{"ingénieur", "ingénieur"},
		{"ingénieurs", "ingénieur"},
		{"ingénieuse", "ingéni"},
		{"ingénieusement", "ingéni"},
		{"ingénieuses", "ingéni"},
		{"ingénieux", "ingéni"},
		{"ingénu", "ingénu"},
		{"ingrat", "ingrat"},
		{"ingrate", "ingrat"},
		{"ingratitude", "ingratitud"},
		{"inhabileté", "inhabilet"},
		{"inhabitée", "inhabit"},
		{"inhalation", "inhal"},
		{"inhérent", "inhérent"},
		{"inhérente", "inhérent"},
		{"inhumaine", "inhumain"},
		{"inhumé", "inhum"},
		{"inintelligible", "inintelligibl"},
		{"iniques", "iniqu"},
		{"iniquité", "iniqu"},
		{"initiative", "initi"},
		{"initièrent", "initi"},
		{"injure", "injur"},
		{"injures", "injur"},
		{"injurié", "injuri"},
		{"injurieuse", "injuri"},
		{"injurieuses", "injuri"},
		{"injurieux", "injuri"},
		{"injuste", "injust"},
		{"injustement", "injust"},
		{"injustes", "injust"},
		{"injustice", "injustic"},
		{"injustices", "injustic"},
		{"inn", "inn"},
		{"inné", "inné"},
		{"innée", "inné"},
		{"innocemment", "innocent"},
		{"innocence", "innocent"},
		{"innocent", "innocent"},
		{"innocente", "innocent"},
		{"innocentes", "innocent"},
		{"innombrable", "innombr"},
		{"innombrables", "innombr"},
		{"innovation", "innov"},
		{"inoccupées", "inoccup"},
		{"inoculé", "inocul"},
		{"inoffensif", "inoffens"},
		{"inondaient", "inond"},
		{"inondait", "inond"},
		{"inondation", "inond"},
		{"inondé", "inond"},
		{"inondée", "inond"},
		{"inondées", "inond"},
		{"inonder", "inond"},
		{"inondèrent", "inond"},
		{"inondés", "inond"},
		{"inopinément", "inopin"},
		{"inopportunes", "inopportun"},
		{"inopportuns", "inopportun"},
		{"inouï", "inouï"},
		{"inouïs", "inouï"},
		{"inqualifiable", "inqualifi"},
		{"inquiet", "inquiet"},
		{"inquiéta", "inquiet"},
		{"inquiétait", "inquiet"},
		{"inquiétant", "inquiet"},
		{"inquiétantes", "inquiet"},
		{"inquiétants", "inquiet"},
		{"inquiète", "inquiet"},
		{"inquiéter", "inquiet"},
		{"inquiets", "inquiet"},
		{"inquiétude", "inquiétud"},
		{"inquiétudes", "inquiétud"},
		{"inquisiteurs", "inquisiteur"},
		{"inquisition", "inquisit"},
		{"insaisissable", "insaisiss"},
		{"inscription", "inscript"},
		{"inscriptions", "inscript"},
		{"inscrire", "inscrir"},
		{"inscrit", "inscrit"},
		{"inscrits", "inscrit"},
		{"inscrivit", "inscriv"},
		{"insectes", "insect"},
		{"insensé", "insens"},
		{"insensée", "insens"},
		{"insensés", "insens"},
		{"insensibilité", "insensibil"},
		{"insensible", "insensibl"},
		{"insensiblement", "insensibl"},
		{"insensibles", "insensibl"},
		{"inséparable", "insépar"},
		{"insérer", "inser"},
		{"insigne", "insign"},
		{"insignes", "insign"},
		{"insignifiance", "insignifi"},
		{"insignifiant", "insignifi"},
		{"insignifiante", "insignifi"},
		{"insignifiants", "insignifi"},
		{"insinuera", "insinu"},
		{"insipide", "insipid"},
		{"insipides", "insipid"},
		{"insista", "insist"},
		{"insistait", "insist"},
		{"insistance", "insist"},
		{"insistant", "insist"},
		{"insiste", "insist"},
		{"insisté", "insist"},
		{"insister", "insist"},
		{"insociable", "insoci"},
		{"insolemment", "insolent"},
		{"insolence", "insolent"},
		{"insolences", "insolent"},
		{"insolent", "insolent"},
		{"insolente", "insolent"},
		{"insolents", "insolent"},
		{"insolite", "insolit"},
		{"insolites", "insolit"},
		{"insouciance", "insouci"},
		{"insouciant", "insouci"},
		{"inspecteur", "inspecteur"},
		{"inspecteurs", "inspecteur"},
		{"inspection", "inspect"},
		{"inspira", "inspir"},
		{"inspiraient", "inspir"},
		{"inspirait", "inspir"},
		{"inspirant", "inspir"},
		{"inspiration", "inspir"},
		{"inspirations", "inspir"},
		{"inspire", "inspir"},
		{"inspiré", "inspir"},
		{"inspirée", "inspir"},
		{"inspirées", "inspir"},
		{"inspirent", "inspirent"},
		{"inspirer", "inspir"},
		{"inspirerait", "inspir"},
		{"inspirèrent", "inspir"},
		{"inspires", "inspir"},
		{"inspirés", "inspir"},
		{"inspirez", "inspir"},
		{"instable", "instabl"},
		{"installation", "install"},
		{"installé", "install"},
		{"installée", "install"},
		{"installer", "install"},
		{"installèrent", "install"},
		{"installés", "install"},
		{"instances", "instanc"},
		{"instant", "instant"},
		{"instantané", "instantan"},
		{"instantanément", "instantan"},
		{"instante", "instant"},
		{"instants", "instant"},
		{"instar", "instar"},
		{"instinct", "instinct"},
		{"instinctif", "instinct"},
		{"instinctivement", "instinct"},
		{"institution", "institu"},
		{"institutions", "institu"},
		{"instructif", "instruct"},
		{"instruction", "instruct"},
		{"instructions", "instruct"},
		{"instruire", "instruir"},
		{"instruisait", "instruis"},
		{"instruit", "instruit"},
		{"instrument", "instrument"},
		{"instruments", "instrument"},
		{"insu", "insu"},
		{"insuccès", "insucces"},
		{"insuffisant", "insuffis"},
		{"insuffisante", "insuffis"},
		{"insuffisantes", "insuffis"},
		{"insulta", "insult"},
		{"insultant", "insult"},
		{"insultante", "insult"},
		{"insulte", "insult"},
		{"insulté", "insult"},
		{"insulter", "insult"},
		{"insultes", "insult"},
		{"insulteur", "insulteur"},
		{"insupportable", "insupport"},
		{"insupportables", "insupport"},
		{"insurmontable", "insurmont"},
		{"insurmontables", "insurmont"},
		{"insurrection", "insurrect"},
		{"intact", "intact"},
		{"intègres", "integr"},
		{"intellectuel", "intellectuel"},
		{"intellectuelle", "intellectuel"},
		{"intelligence", "intelligent"},
		{"intelligences", "intelligent"},
		{"intelligent", "intelligent"},
		{"intelligente", "intelligent"},
		{"intelligenti", "intelligent"},
		{"intelligents", "intelligent"},
		{"intelligible", "intelligibl"},
		{"intelligibles", "intelligibl"},
		{"intempestive", "intempest"},
		{"intendant", "intend"},
		{"intense", "intens"},
		{"intenses", "intens"},
		{"intensité", "intens"},
		{"intention", "intent"},
		{"intentionné", "intention"},
		{"intentions", "intent"},
		{"interceptait", "intercept"},
		{"intercepté", "intercept"},
		{"interceptée", "intercept"},
		{"interceptées", "intercept"},
		{"intercepter", "intercept"},
		{"intercepterait", "intercept"},
		{"intercession", "intercess"},
		{"interdire", "interdir"},
		{"interdis", "interd"},
		{"interdit", "interd"},
		{"interdite", "interdit"},
		{"intéressa", "intéress"},
		{"intéressaient", "intéress"},
		{"intéressait", "intéress"},
		{"intéressant", "intéress"},
		{"intéressante", "intéress"},
		{"intéressantes", "intéress"},
		{"intéressants", "intéress"},
		{"intéresse", "intéress"},
		{"intéressé", "intéress"},
		{"intéressée", "intéress"},
		{"intéressent", "intéressent"},
		{"intéresser", "intéress"},
		{"intéresseront", "intéress"},
		{"intéressés", "intéress"},
		{"interet", "interet"},
		{"intérêt", "intérêt"},
		{"interets", "interet"},
		{"intérêts", "intérêt"},
		{"intérieur", "intérieur"},
		{"intérieure", "intérieur"},
		{"intérieurement", "intérieur"},
		{"intérieurs", "intérieur"},
		{"interjections", "interject"},
		{"interlocuteur", "interlocuteur"},
		{"interlocuteurs", "interlocuteur"},
		{"interlope", "interlop"},
		{"intermédiaire", "intermédiair"},
		{"interminable", "intermin"},
		{"interminables", "intermin"},
		{"international", "international"},
		{"interpellé", "interpel"},
		{"interpeller", "interpel"},
		{"interposé", "interpos"},
		{"interpréta", "interpret"},
		{"interprétation", "interpret"},
		{"interprétations", "interpret"},
		{"interprété", "interpret"},
		{"interprétée", "interpret"},
		{"interprètes", "interpret"},
		{"interrogatif", "interrog"},
		{"interrogation", "interrog"},
		{"interrogatoire", "interrogatoir"},
		{"interrogatoires", "interrogatoir"},
		{"interroge", "interrog"},
		{"interrogé", "interrog"},
		{"interrogea", "interrog"},
		{"interrogeant", "interrog"},
		{"interroger", "interrog"},
		{"interrogerait", "interrog"},
		{"interrogez", "interrog"},
		{"interrompait", "interromp"},
		{"interrompant", "interromp"},
		{"interrompirent", "interromp"},
		{"interrompit", "interromp"},
		{"interromprait", "interrompr"},
		{"interrompre", "interrompr"},
		{"interrompt", "interrompt"},
		{"interrompu", "interrompu"},
		{"interrompue", "interrompu"},
		{"interrompues", "interrompu"},
		{"interrompus", "interrompus"},
		{"interrupteur", "interrupteur"},
		{"interruption", "interrupt"},
		{"intervalle", "intervall"},
		{"intervalles", "intervall"},
		{"intervention", "intervent"},
		{"interviendrais", "interviendr"},
		{"intima", "intim"},
		{"intime", "intim"},
		{"intimement", "intim"},
		{"intimes", "intim"},
		{"intimida", "intimid"},
		{"intimidé", "intimid"},
		{"intimident", "intimident"},
		{"intimider", "intimid"},
		{"intimité", "intim"},
		{"intitulés", "intitul"},
		{"intolérable", "intoler"},
		{"intolérables", "intoler"},
		{"intonation", "inton"},
		{"intraitable", "intrait"},
		{"intrépide", "intrépid"},
		{"intrépidement", "intrépid"},
		{"intrépidité", "intrépid"},
		{"intrigant", "intrig"},
		{"intrigante", "intrig"},
		{"intrigants", "intrig"},
		{"intriguait", "intrigu"},
		{"intrigue", "intrigu"},
		{"intrigué", "intrigu"},
		{"intrigues", "intrigu"},
		{"introduction", "introduct"},
		{"introduire", "introduir"},
		{"introduis", "introduis"},
		{"introduisit", "introduis"},
		{"introduit", "introduit"},
		{"introduite", "introduit"},
		{"introduits", "introduit"},
		{"intrus", "intrus"},
		{"inusitée", "inusit"},
		{"inutile", "inutil"},
		{"inutilement", "inutil"},
		{"inutiles", "inutil"},
		{"inutilite", "inutilit"},
		{"inutilité", "inutil"},
		{"invalides", "invalid"},
		{"invariable", "invari"},
		{"invariablement", "invari"},
		{"invariables", "invari"},
		{"invasion", "invas"},
		{"inventaire", "inventair"},
		{"inventait", "invent"},
		{"inventant", "invent"},
		{"invente", "invent"},
		{"inventé", "invent"},
		{"inventée", "invent"},
		{"inventées", "invent"},
		{"inventer", "invent"},
		{"inventèrent", "invent"},
		{"inventeur", "inventeur"},
		{"inventif", "invent"},
		{"invention", "invent"},
		{"inventions", "invent"},
		{"inverse", "invers"},
		{"invincible", "invincibl"},
		{"invinciblement", "invincibl"},
		{"invisible", "invisibl"},
		{"invisibles", "invisibl"},
		{"invita", "invit"},
		{"invitait", "invit"},
		{"invitant", "invit"},
		{"invitation", "invit"},
		{"invitations", "invit"},
		{"invite", "invit"},
		{"invité", "invit"},
		{"inviter", "invit"},
		{"inviterai", "invit"},
		{"invités", "invit"},
		{"inviti", "invit"},
		{"invocation", "invoc"},
		{"involontaire", "involontair"},
		{"involontairement", "involontair"},
		{"invraisemblable", "invraisembl"},
		{"invraisemblables", "invraisembl"},
		{"invraisemblance", "invraisembl"},
		{"invulnérable", "invulner"},
		{"ioniques", "ioniqu"},
		{"iowa", "iow"},
		{"ira", "ira"},
		{"irai", "irai"},
		{"iraient", "iraient"},
		{"irais", "irais"},
		{"irait", "irait"},
		{"iras", "iras"},
		{"irascible", "irascibl"},
		{"irez", "irez"},
		{"iriez", "iri"},
		{"irlande", "irland"},
		{"ironie", "iron"},
		{"ironique", "iron"},
		{"irons", "iron"},
		{"iront", "iront"},
		{"irrégulière", "irréguli"},
		{"irrégulièrement", "irréguli"},
		{"irréguliers", "irréguli"},
		{"irrémédiable", "irrémédi"},
		{"irrémissiblement", "irrémissibl"},
		{"irréparable", "irrépar"},
		{"irréparablement", "irrépar"},
		{"irréprochable", "irréproch"},
		{"irréprochables", "irréproch"},
		{"irrésistible", "irrésistibl"},
		{"irrésistiblement", "irrésistibl"},
		{"irrésistibles", "irrésistibl"},
		{"irrésolution", "irrésolu"},
		{"irrésolutions", "irrésolu"},
		{"irresponsable", "irrespons"},
		{"irréussites", "irréussit"},
		{"irrévocablement", "irrévoc"},
		{"irriguaient", "irrigu"},
		{"irrita", "irrit"},
		{"irritable", "irrit"},
		{"irritaient", "irrit"},
		{"irritait", "irrit"},
		{"irritant", "irrit"},
		{"irritation", "irrit"},
		{"irrite", "irrit"},
		{"irrité", "irrit"},
		{"irritée", "irrit"},
		{"irriter", "irrit"},
		{"irritèrent", "irrit"},
		{"irruption", "irrupt"},
		{"is", "is"},
		{"island", "island"},
		{"isole", "isol"},
		{"isolé", "isol"},
		{"isolée", "isol"},
		{"isolement", "isol"},
		{"isolés", "isol"},
		{"isota", "isot"},
		{"israël", "israël"},
		{"issu", "issu"},
		{"issue", "issu"},
		{"issues", "issu"},
		{"it", "it"},
		{"ita", "ita"},
		{"italie", "ital"},
		{"italien", "italien"},
		{"italienne", "italien"},
		{"italiennes", "italien"},
		{"italiens", "italien"},
		{"itinéraire", "itinérair"},
		{"itou", "itou"},
		{"its", "it"},
		{"itself", "itself"},
		{"iv", "iv"},
		{"ivoire", "ivoir"},
		{"ivre", "ivre"},
		{"ivres", "ivre"},
		{"ivresse", "ivress"},
		{"ivrogne", "ivrogn"},
		{"ivrognes", "ivrogn"},
		{"ix", "ix"},
		{"j", "j"},
		{"jacobin", "jacobin"},
		{"jacobine", "jacobin"},
		{"jacobinisme", "jacobin"},
		{"jacobins", "jacobin"},
		{"jacopo", "jacopo"},
		{"jacques", "jacqu"},
		{"jadis", "jad"},
		{"jaggernaut", "jaggernaut"},
		{"jailli", "jaill"},
		{"jaillissaient", "jaill"},
		{"jais", "jais"},
		{"jalousait", "jalous"},
		{"jalouse", "jalous"},
		{"jalousé", "jalous"},
		{"jalouses", "jalous"},
		{"jalousie", "jalous"},
		{"jalousies", "jalous"},
		{"jaloux", "jaloux"},
		{"jamais", "jam"},
		{"jambe", "jamb"},
		{"jambes", "jamb"},
		{"james", "jam"},
		{"jansénisme", "jansen"},
		{"janséniste", "jansen"},
		{"jansénistes", "jansen"},
		{"jante", "jant"},
		{"janvier", "janvi"},
		{"japon", "japon"},
		{"japonais", "japon"},
		{"japonaise", "japonais"},
		{"japonaisé", "japonais"},
		{"japonaises", "japonais"},
		{"jaquette", "jaquet"},
		{"jardin", "jardin"},
		{"jardinage", "jardinag"},
		{"jardinier", "jardini"},
		{"jardins", "jardin"},
		{"jarnac", "jarnac"},
		{"jarrets", "jarret"},
		{"jasant", "jas"},
		{"jasent", "jasent"},
		{"jaser", "jas"},
		{"jasmin", "jasmin"},
		{"jaugeant", "jaug"},
		{"jaunâtres", "jaunâtr"},
		{"jaune", "jaun"},
		{"jaunes", "jaun"},
		{"jaunets", "jaunet"},
		{"jaunies", "jaun"},
		{"je", "je"},
		{"jean", "jean"},
		{"jeannes", "jeann"},
		{"jejeebhoy", "jejeebhoy"},
		{"jejeeh", "jejeeh"},
		{"jenrel", "jenrel"},
		{"jérémiades", "jérémiad"},
		{"jérôme", "jérôm"},
		{"jersey", "jersey"},
		{"jérusalem", "jérusalem"},
		{"jest", "jest"},
		{"jésuite", "jésuit"},
		{"jésuites", "jésuit"},
		{"jésuitique", "jésuit"},
		{"jésuitisme", "jésuit"},
		{"jésus", "jésus"},
		{"jeta", "jet"},
		{"jetaient", "jet"},
		{"jetait", "jet"},
		{"jetant", "jet"},
		{"jetât", "jet"},
		{"jeté", "jet"},
		{"jetée", "jet"},
		{"jetées", "jet"},
		{"jeter", "jet"},
		{"jetèrent", "jet"},
		{"jetés", "jet"},
		{"jetez", "jet"},
		{"jetons", "jeton"},
		{"jette", "jet"},
		{"jettent", "jettent"},
		{"jetterais", "jet"},
		{"jetterait", "jet"},
		{"jetterez", "jet"},
		{"jeu", "jeu"},
		{"jeudi", "jeud"},
		{"jeudis", "jeud"},
		{"jeun", "jeun"},
		{"jeûnant", "jeûn"},
		{"jeune", "jeun"},
		{"jeûne", "jeûn"},
		{"jeunes", "jeun"},
		{"jeûnes", "jeûn"},
		{"jeunesse", "jeuness"},
		{"jeux", "jeux"},
		{"joe", "jo"},
		{"john", "john"},
		{"joie", "joi"},
		{"joies", "joi"},
		{"joignait", "joign"},
		{"joignant", "joign"},
		{"joignit", "joign"},
		{"joindre", "joindr"},
		{"joint", "joint"},
		{"jointe", "joint"},
		{"jointes", "joint"},
		{"joli", "jol"},
		{"jolie", "jol"},
		{"jolies", "jol"},
		{"joliment", "jol"},
		{"jolis", "jol"},
		{"jonc", "jonc"},
		{"jonché", "jonch"},
		{"jonglaient", "jongl"},
		{"jonglait", "jongl"},
		{"jonglerie", "jongler"},
		{"jongleurs", "jongleur"},
		{"jonques", "jonqu"},
		{"jonquille", "jonquill"},
		{"joseph", "joseph"},
		{"joua", "jou"},
		{"jouaient", "jou"},
		{"jouait", "jou"},
		{"jouant", "jou"},
		{"jouât", "jou"},
		{"joue", "jou"},
		{"joué", "jou"},
		{"jouée", "jou"},
		{"jouer", "jou"},
		{"jouera", "jou"},
		{"jouerai", "jou"},
		{"joueraient", "jou"},
		{"jouerais", "jou"},
		{"jouerait", "jou"},
		{"joues", "jou"},
		{"joueur", "joueur"},
		{"joueurs", "joueur"},
		{"jouez", "jou"},
		{"joui", "jou"},
		{"jouir", "jou"},
		{"jouira", "jou"},
		{"jouirait", "jou"},
		{"jouiras", "jou"},
		{"jouis", "jou"},
		{"jouissaient", "jou"},
		{"jouissait", "jou"},
		{"jouissance", "jouiss"},
		{"jouissances", "jouiss"},
		{"jouissant", "jou"},
		{"jouisse", "jou"},
		{"jouissent", "jou"},
		{"jouissons", "jou"},
		{"jouit", "jou"},
		{"joujoux", "joujoux"},
		{"joumate", "joumat"},
		{"jouons", "jouon"},
		{"jour", "jour"},
		{"jourdain", "jourdain"},
		{"jourdan", "jourdan"},
		{"journal", "journal"},
		{"journalière", "journali"},
		{"journalistes", "journal"},
		{"journaux", "journal"},
		{"journée", "journ"},
		{"journées", "journ"},
		{"journellement", "journel"},
		{"jours", "jour"},
		{"joyeuse", "joyeux"},
		{"joyeusement", "joyeux"},
		{"joyeux", "joyeux"},
		{"juan", "juan"},
		{"jucha", "juch"},
		{"juché", "juch"},
		{"judiciaires", "judiciair"},
		{"judicieux", "judici"},
		{"judith", "judith"},
		{"juge", "jug"},
		{"jugé", "jug"},
		{"jugea", "jug"},
		{"jugeait", "jug"},
		{"jugeant", "jug"},
		{"jugement", "jug"},
		{"jugements", "jug"},
		{"jugent", "jugent"},
		{"juger", "jug"},
		{"jugera", "jug"},
		{"jugerai", "jug"},
		{"jugeront", "jug"},
		{"juges", "jug"},
		{"jugés", "jug"},
		{"jugez", "jug"},
		{"juif", "juif"},
		{"juifs", "juif"},
		{"juillet", "juillet"},
		{"juin", "juin"},
		{"jules", "jul"},
		{"julesburgh", "julesburgh"},
		{"julia", "juli"},
		{"julien", "julien"},
		{"jumna", "jumn"},
		{"junction", "junction"},
		{"jungles", "jungl"},
		{"junior", "junior"},
		{"jupon", "jupon"},
		{"jura", "jur"},
		{"juraient", "jur"},
		{"jurait", "jur"},
		{"jurant", "jur"},
		{"jure", "jur"},
		{"juré", "jur"},
		{"jurée", "jur"},
		{"jurement", "jur"},
		{"jurements", "jur"},
		{"jurer", "jur"},
		{"jurerai", "jur"},
		{"jurerais", "jur"},
		{"jures", "jur"},
		{"jurés", "jur"},
		{"jurez", "jur"},
		{"jurisconsulte", "jurisconsult"},
		{"jurisconsultes", "jurisconsult"},
		{"juron", "juron"},
		{"jurons", "juron"},
		{"jury", "jury"},
		{"jusqu", "jusqu"},
		{"jusque", "jusqu"},
		{"jusques", "jusqu"},
		{"just", "just"},
		{"juste", "just"},
		{"justement", "just"},
		{"justes", "just"},
		{"justesse", "justess"},
		{"justice", "justic"},
		{"justifia", "justifi"},
		{"justifiait", "justifi"},
		{"justificatif", "justif"},
		{"justification", "justif"},
		{"justifications", "justif"},
		{"justifie", "justif"},
		{"justifié", "justifi"},
		{"justifiée", "justifi"},
		{"justifier", "justifi"},
		{"kâli", "kâl"},
		{"kalisky", "kalisky"},
		{"kallenger", "kalleng"},
		{"kama", "kam"},
		{"kamerfield", "kamerfield"},
		{"kandallah", "kandallah"},
		{"kanhérie", "kanher"},
		{"kansas", "kans"},
		{"kant", "kant"},
		{"kauffmann", "kauffmann"},
		{"kearney", "kearney"},
		{"kehl", "kehl"},
		{"kent", "kent"},
		{"khajours", "khajour"},
		{"khandeish", "khandeish"},
		{"kholby", "kholby"},
		{"kien", "kien"},
		{"kilomètres", "kilometr"},
		{"kind", "kind"},
		{"kiouni", "kioun"},
		{"kirimon", "kirimon"},
		{"kirkland", "kirkland"},
		{"kléber", "kleb"},
		{"kong", "kong"},
		{"korasoff", "korasoff"},
		{"l", "l"},
		{"la", "la"},
		{"là", "là"},
		{"laborieusement", "labori"},
		{"laborieux", "labori"},
		{"labourée", "labour"},
		{"lac", "lac"},
		{"lâcha", "lâch"},
		{"lachaise", "lachais"},
		{"lâchait", "lâch"},
		{"lâche", "lâch"},
		{"lâchement", "lâch"},
		{"lâcher", "lâch"},
		{"lâches", "lâch"},
		{"lâcheté", "lâchet"},
		{"lâchez", "lâch"},
		{"laconiquement", "lacon"},
		{"lacs", "lac"},
		{"ladislas", "ladisl"},
		{"ladri", "ladr"},
		{"lafayette", "lafayet"},
		{"lago", "lago"},
		{"laïc", "laïc"},
		{"laïcs", "laïc"},
		{"laid", "laid"},
		{"laide", "laid"},
		{"laides", "laid"},
		{"laideur", "laideur"},
		{"laideurs", "laideur"},
		{"laine", "lain"},
		{"laïque", "laïqu"},
		{"laissa", "laiss"},
		{"laissaient", "laiss"},
		{"laissais", "laiss"},
		{"laissait", "laiss"},
		{"laissant", "laiss"},
		{"laissât", "laiss"},
		{"laisse", "laiss"},
		{"laissé", "laiss"},
		{"laissée", "laiss"},
		{"laissées", "laiss"},
		{"laissent", "laissent"},
		{"laisser", "laiss"},
		{"laissera", "laiss"},
		{"laisserai", "laiss"},
		{"laisserais", "laiss"},
		{"laisserait", "laiss"},
		{"laissèrent", "laiss"},
		{"laisserez", "laiss"},
		{"laissés", "laiss"},
		{"laissez", "laiss"},
		{"laissions", "laission"},
		{"laissons", "laisson"},
		{"lait", "lait"},
		{"laiterie", "laiter"},
		{"laiton", "laiton"},
		{"laitue", "laitu"},
		{"lake", "lak"},
		{"lambeau", "lambeau"},
		{"lambeaux", "lambeau"},
		{"lambris", "lambr"},
		{"lambrissée", "lambriss"},
		{"lame", "lam"},
		{"lamée", "lam"},
		{"lamentable", "lament"},
		{"lamentables", "lament"},
		{"lames", "lam"},
		{"lampe", "lamp"},
		{"lampes", "lamp"},
		{"lança", "lanc"},
		{"lançaient", "lanc"},
		{"lançait", "lanc"},
		{"lançant", "lanc"},
		{"lance", "lanc"},
		{"lancé", "lanc"},
		{"lancée", "lanc"},
		{"lancer", "lanc"},
		{"lancés", "lanc"},
		{"landau", "landau"},
		{"landes", "land"},
		{"landriani", "landrian"},
		{"langage", "langag"},
		{"lange", "lang"},
		{"langue", "langu"},
		{"languedoc", "languedoc"},
		{"langues", "langu"},
		{"langueur", "langueur"},
		{"languir", "languir"},
		{"languis", "languis"},
		{"languissait", "languiss"},
		{"languissante", "languiss"},
		{"lanterne", "lantern"},
		{"lanternes", "lantern"},
		{"lapidaire", "lapidair"},
		{"lapin", "lapin"},
		{"laps", "lap"},
		{"laquais", "laqu"},
		{"laque", "laqu"},
		{"laquelle", "laquel"},
		{"laramie", "laram"},
		{"lard", "lard"},
		{"larga", "larg"},
		{"large", "larg"},
		{"largement", "larg"},
		{"larges", "larg"},
		{"largeur", "largeur"},
		{"larguait", "largu"},
		{"largue", "largu"},
		{"larme", "larm"},
		{"larmes", "larm"},
		{"las", "las"},
		{"lascia", "lasci"},
		{"lassa", "lass"},
		{"lasse", "lass"},
		{"lasser", "lass"},
		{"lasseront", "lass"},
		{"last", "last"},
		{"lataniers", "latani"},
		{"latérale", "latéral"},
		{"latérales", "latéral"},
		{"latin", "latin"},
		{"latinam", "latinam"},
		{"latine", "latin"},
		{"latines", "latin"},
		{"latiniste", "latin"},
		{"latinité", "latin"},
		{"latins", "latin"},
		{"laudanum", "laudanum"},
		{"laughter", "laught"},
		{"lauréat", "lauréat"},
		{"laurent", "laurent"},
		{"laurier", "lauri"},
		{"lauriers", "lauri"},
		{"lausanne", "lausann"},
		{"lavalette", "lavalet"},
		{"lave", "lav"},
		{"laveno", "laveno"},
		{"laver", "lav"},
		{"laye", "lay"},
		{"le", "le"},
		{"least", "least"},
		{"lecco", "lecco"},
		{"leçon", "leçon"},
		{"leçons", "leçon"},
		{"lecteur", "lecteur"},
		{"lecteurs", "lecteur"},
		{"lectrice", "lectric"},
		{"lecture", "lectur"},
		{"ledit", "led"},
		{"left", "left"},
		{"légale", "légal"},
		{"légalement", "légal"},
		{"légalité", "légal"},
		{"légation", "légat"},
		{"légendaire", "légendair"},
		{"légendes", "légend"},
		{"leger", "leg"},
		{"léger", "leg"},
		{"légère", "léger"},
		{"légèrement", "léger"},
		{"légères", "léger"},
		{"légèreté", "légèret"},
		{"légers", "léger"},
		{"légion", "légion"},
		{"législateur", "législ"},
		{"législateurs", "législ"},
		{"législature", "législatur"},
		{"legitime", "legitim"},
		{"légitime", "légitim"},
		{"légitimes", "légitim"},
		{"légitimité", "légitim"},
		{"legs", "leg"},
		{"légua", "légu"},
		{"lègue", "lègu"},
		{"légué", "légu"},
		{"leipsick", "leipsick"},
		{"lek", "lek"},
		{"lélio", "lélio"},
		{"lendemain", "lendemain"},
		{"lent", "lent"},
		{"lente", "lent"},
		{"lentement", "lent"},
		{"lentes", "lent"},
		{"lenteur", "lenteur"},
		{"lenteurs", "lenteur"},
		{"lents", "lent"},
		{"leo", "leo"},
		{"léonard", "léonard"},
		{"léontine", "léontin"},
		{"léotard", "léotard"},
		{"lépidoptères", "lépidopter"},
		{"lequel", "lequel"},
		{"leroy", "leroy"},
		{"les", "le"},
		{"lés", "lé"},
		{"lescaut", "lescaut"},
		{"lesdits", "lesdit"},
		{"lèse", "les"},
		{"lésineries", "lésiner"},
		{"lesquelles", "lesquel"},
		{"lesquels", "lesquel"},
		{"less", "less"},
		{"lesseps", "lessep"},
		{"lest", "lest"},
		{"leste", "lest"},
		{"lestement", "lest"},
		{"lester", "lest"},
		{"léthargique", "létharg"},
		{"lettre", "lettr"},
		{"lettres", "lettr"},
		{"leur", "leur"},
		{"leurs", "leur"},
		{"leva", "lev"},
		{"levait", "lev"},
		{"levant", "lev"},
		{"lève", "lev"},
		{"levé", "lev"},
		{"levée", "lev"},
		{"levées", "lev"},
		{"lèvent", "lèvent"},
		{"lever", "lev"},
		{"lèvera", "lev"},
		{"levèrent", "lev"},
		{"levez", "lev"},
		{"levier", "levi"},
		{"leviers", "levi"},
		{"levis", "lev"},
		{"lévite", "lévit"},
		{"lèvre", "levr"},
		{"lèvres", "levr"},
		{"lia", "li"},
		{"liaison", "liaison"},
		{"liaisons", "liaison"},
		{"liant", "li"},
		{"liard", "liard"},
		{"liasse", "li"},
		{"liasses", "li"},
		{"liât", "li"},
		{"libellé", "libel"},
		{"libellée", "libel"},
		{"libéral", "libéral"},
		{"libérales", "libéral"},
		{"libéralisme", "libéral"},
		{"libérateur", "liber"},
		{"libéraux", "libéral"},
		{"liberté", "libert"},
		{"libertin", "libertin"},
		{"libertinage", "libertinag"},
		{"libraire", "librair"},
		{"libraires", "librair"},
		{"libre", "libr"},
		{"librement", "libr"},
		{"libres", "libr"},
		{"lichtenberg", "lichtenberg"},
		{"lie", "li"},
		{"lié", "li"},
		{"liée", "li"},
		{"liées", "li"},
		{"liège", "lieg"},
		{"lien", "lien"},
		{"liens", "lien"},
		{"lier", "li"},
		{"lierre", "lierr"},
		{"liés", "li"},
		{"lieu", "lieu"},
		{"lieue", "lieu"},
		{"lieues", "lieu"},
		{"lieutenance", "lieuten"},
		{"lieutenant", "lieuten"},
		{"lieutenants", "lieuten"},
		{"lieux", "lieux"},
		{"liéven", "liéven"},
		{"lièvre", "lievr"},
		{"lièvres", "lievr"},
		{"light", "light"},
		{"ligne", "lign"},
		{"lignes", "lign"},
		{"ligny", "ligny"},
		{"ligorio", "ligorio"},
		{"ligue", "ligu"},
		{"ligués", "ligu"},
		{"lilliputiens", "lilliputien"},
		{"limercati", "limercat"},
		{"limite", "limit"},
		{"limites", "limit"},
		{"limpide", "limpid"},
		{"limpides", "limpid"},
		{"lincoln", "lincoln"},
		{"line", "lin"},
		{"linge", "ling"},
		{"lingère", "linger"},
		{"lingot", "lingot"},
		{"linguam", "linguam"},
		{"lion", "lion"},
		{"lip", "lip"},
		{"liqueur", "liqueur"},
		{"liqueurs", "liqueur"},
		{"liquide", "liquid"},
		{"liquidée", "liquid"},
		{"lira", "lir"},
		{"lirai", "lir"},
		{"liras", "lir"},
		{"lire", "lir"},
		{"lis", "lis"},
		{"lisaient", "lis"},
		{"lisait", "lis"},
		{"lisant", "lis"},
		{"lisard", "lisard"},
		{"lisent", "lisent"},
		{"lisez", "lis"},
		{"lisible", "lisibl"},
		{"lisière", "lisi"},
		{"lisse", "liss"},
		{"lisses", "liss"},
		{"liste", "list"},
		{"lit", "lit"},
		{"litanies", "litan"},
		{"lithographiée", "lithographi"},
		{"lithographies", "lithograph"},
		{"litière", "liti"},
		{"litières", "liti"},
		{"lits", "lit"},
		{"littéraire", "littérair"},
		{"littéraires", "littérair"},
		{"littéralement", "littéral"},
		{"littérature", "littératur"},
		{"little", "littl"},
		{"live", "liv"},
		{"liverpool", "liverpool"},
		{"liveru", "liveru"},
		{"livra", "livr"},
		{"livraient", "livr"},
		{"livrais", "livr"},
		{"livrait", "livr"},
		{"livre", "livr"},
		{"livré", "livr"},
		{"livrée", "livr"},
		{"livrées", "livr"},
		{"livrer", "livr"},
		{"livres", "livr"},
		{"livrés", "livr"},
		{"livret", "livret"},
		{"ll", "ll"},
		{"locale", "local"},
		{"locarno", "locarno"},
		{"locataire", "locatair"},
		{"location", "locat"},
		{"loch", "loch"},
		{"locke", "lock"},
		{"locomotion", "locomot"},
		{"locomotive", "locomot"},
		{"locomotives", "locomot"},
		{"lodge", "lodg"},
		{"lodi", "lod"},
		{"loge", "log"},
		{"logé", "log"},
		{"logeable", "logeabl"},
		{"logement", "log"},
		{"logements", "log"},
		{"loger", "log"},
		{"loges", "log"},
		{"logés", "log"},
		{"logions", "logion"},
		{"logique", "logiqu"},
		{"logiques", "logiqu"},
		{"logis", "log"},
		{"loi", "loi"},
		{"loin", "loin"},
		{"lointain", "lointain"},
		{"lointaine", "lointain"},
		{"lointaines", "lointain"},
		{"lointains", "lointain"},
		{"lois", "lois"},
		{"loisir", "lois"},
		{"lombarde", "lombard"},
		{"lombardie", "lombard"},
		{"lombardo", "lombardo"},
		{"lombards", "lombard"},
		{"lonato", "lonato"},
		{"london", "london"},
		{"londonner", "london"},
		{"londres", "londr"},
		{"long", "long"},
		{"longanimité", "longanim"},
		{"longe", "long"},
		{"longeant", "long"},
		{"longerons", "long"},
		{"longévité", "longev"},
		{"longs", "long"},
		{"longsferry", "longsferry"},
		{"longtemps", "longtemp"},
		{"longue", "longu"},
		{"longuement", "longu"},
		{"longues", "longu"},
		{"longueur", "longueur"},
		{"longueville", "longuevill"},
		{"lope", "lop"},
		{"loquerisne", "loquerisn"},
		{"loques", "loqu"},
		{"loquet", "loquet"},
		{"lord", "lord"},
		{"lords", "lord"},
		{"lorgnette", "lorgnet"},
		{"lorgnettes", "lorgnet"},
		{"lors", "lor"},
		{"lorsqu", "lorsqu"},
		{"lorsque", "lorsqu"},
		{"lot", "lot"},
		{"loterie", "loter"},
		{"lotus", "lotus"},
		{"loua", "lou"},
		{"louables", "louabl"},
		{"louage", "louag"},
		{"louaient", "lou"},
		{"louange", "louang"},
		{"louanges", "louang"},
		{"louant", "lou"},
		{"louche", "louch"},
		{"loue", "lou"},
		{"loué", "lou"},
		{"louée", "lou"},
		{"louées", "lou"},
		{"louer", "lou"},
		{"louera", "lou"},
		{"loueras", "lou"},
		{"louèrent", "lou"},
		{"loueur", "loueur"},
		{"loueuse", "loueux"},
		{"louez", "lou"},
		{"louiez", "lou"},
		{"louis", "lou"},
		{"louise", "louis"},
		{"louons", "louon"},
		{"loup", "loup"},
		{"loups", "loup"},
		{"lourd", "lourd"},
		{"lourde", "lourd"},
		{"lourdement", "lourd"},
		{"lourdes", "lourd"},
		{"lourdeur", "lourdeur"},
		{"loutre", "loutr"},
		{"louvre", "louvr"},
		{"love", "lov"},
		{"lowe", "low"},
		{"loyal", "loyal"},
		{"loyalement", "loyal"},
		{"loyauté", "loyaut"},
		{"loyaux", "loyal"},
		{"lu", "lu"},
		{"lucides", "lucid"},
		{"lucrative", "lucrat"},
		{"ludovic", "ludovic"},
		{"lue", "lu"},
		{"lues", "lu"},
		{"lueur", "lueur"},
		{"lueurs", "lueur"},
		{"lugano", "lugano"},
		{"lugubre", "lugubr"},
		{"lugubrement", "lugubr"},
		{"lugubres", "lugubr"},
		{"lui", "lui"},
		{"luisante", "luis"},
		{"lumière", "lumi"},
		{"lumières", "lumi"},
		{"lumineuse", "lumin"},
		{"luna", "lun"},
		{"lunch", "lunch"},
		{"lundi", "lund"},
		{"lune", "lun"},
		{"lunette", "lunet"},
		{"luoghi", "luogh"},
		{"lustre", "lustr"},
		{"lustres", "lustr"},
		{"lut", "lut"},
		{"luther", "luth"},
		{"luttaient", "lutt"},
		{"luttait", "lutt"},
		{"lutte", "lutt"},
		{"luttent", "luttent"},
		{"lutter", "lutt"},
		{"luttera", "lutt"},
		{"luttes", "lutt"},
		{"luxe", "lux"},
		{"luxée", "lux"},
		{"luxembourg", "luxembourg"},
		{"luxuriantes", "luxuri"},
		{"luz", "luz"},
		{"lycéens", "lycéen"},
		{"lyon", "lyon"},
		{"lyre", "lyr"},
		{"m", "m"},
		{"ma", "ma"},
		{"mac", "mac"},
		{"macadamisées", "macadamis"},
		{"macao", "macao"},
		{"mâchais", "mâch"},
		{"mâchait", "mâch"},
		{"machiavel", "machiavel"},
		{"machiavélisme", "machiavel"},
		{"machiavelli", "machiavel"},
		{"machinalement", "machinal"},
		{"machine", "machin"},
		{"machines", "machin"},
		{"mackintosh", "mackintosh"},
		{"macon", "macon"},
		{"maçon", "maçon"},
		{"maçonnerie", "maçonner"},
		{"maçons", "maçon"},
		{"madame", "madam"},
		{"made", "mad"},
		{"madeleine", "madelein"},
		{"mademoiselle", "mademoisel"},
		{"madone", "madon"},
		{"madras", "madr"},
		{"madriers", "madri"},
		{"maestro", "maestro"},
		{"magalon", "magalon"},
		{"magasin", "magasin"},
		{"magasins", "magasin"},
		{"maggiore", "maggior"},
		{"magicien", "magicien"},
		{"magie", "mag"},
		{"magique", "magiqu"},
		{"magistral", "magistral"},
		{"magistrat", "magistrat"},
		{"magistrats", "magistrat"},
		{"magistrature", "magistratur"},
		{"magnanime", "magnanim"},
		{"magnificence", "magnificent"},
		{"magnifique", "magnif"},
		{"magnifiquement", "magnif"},
		{"magnifiques", "magnif"},
		{"magot", "magot"},
		{"mahomet", "mahomet"},
		{"mahout", "mahout"},
		{"mai", "mai"},
		{"maigre", "maigr"},
		{"maigres", "maigr"},
		{"maigreur", "maigreur"},
		{"maigri", "maigr"},
		{"maigrie", "maigr"},
		{"mail", "mail"},
		{"mailles", "maill"},
		{"main", "main"},
		{"mains", "main"},
		{"maintenaient", "mainten"},
		{"maintenait", "mainten"},
		{"maintenant", "mainten"},
		{"maintenir", "mainten"},
		{"maintenue", "maintenu"},
		{"maintien", "maintien"},
		{"maintint", "maintint"},
		{"maire", "mair"},
		{"maires", "mair"},
		{"mairie", "mair"},
		{"mais", "mais"},
		{"maïs", "maï"},
		{"maison", "maison"},
		{"maisons", "maison"},
		{"maistre", "maistr"},
		{"maitre", "maitr"},
		{"maître", "maîtr"},
		{"maîtres", "maîtr"},
		{"maîtresse", "maîtress"},
		{"maîtresses", "maîtress"},
		{"majesté", "majest"},
		{"majestueuse", "majestu"},
		{"majestueusement", "majestu"},
		{"majestueux", "majestu"},
		{"majeur", "majeur"},
		{"majeure", "majeur"},
		{"major", "major"},
		{"majordome", "majordom"},
		{"majordomes", "majordom"},
		{"majorité", "major"},
		{"majorités", "major"},
		{"mal", "mal"},
		{"malacca", "malacc"},
		{"malade", "malad"},
		{"malades", "malad"},
		{"maladie", "malad"},
		{"maladies", "malad"},
		{"maladive", "malad"},
		{"maladresse", "maladress"},
		{"maladresses", "maladress"},
		{"maladroit", "maladroit"},
		{"maladroite", "maladroit"},
		{"maladroitement", "maladroit"},
		{"malaga", "malag"},
		{"malagrida", "malagrid"},
		{"malais", "mal"},
		{"malaise", "malais"},
		{"malaisie", "malais"},
		{"maldonne", "maldon"},
		{"mâle", "mâl"},
		{"malebar", "malebar"},
		{"malédiction", "malédict"},
		{"malencontreuse", "malencontr"},
		{"malencontreusement", "malencontr"},
		{"malencontreux", "malencontr"},
		{"malentendu", "malentendu"},
		{"mâles", "mâl"},
		{"malgré", "malgr"},
		{"malhabile", "malhabil"},
		{"malheur", "malheur"},
		{"malheureuse", "malheur"},
		{"malheureusement", "malheur"},
		{"malheureuses", "malheur"},
		{"malheureux", "malheur"},
		{"malheurs", "malheur"},
		{"malhonnête", "malhonnêt"},
		{"malhonnêteté", "malhonnêtet"},
		{"malice", "malic"},
		{"malicieux", "malici"},
		{"maligne", "malign"},
		{"malignité", "malign"},
		{"malin", "malin"},
		{"mall", "mall"},
		{"malle", "mall"},
		{"malles", "mall"},
		{"malligaum", "malligaum"},
		{"malmaison", "malmaison"},
		{"malmener", "malmen"},
		{"maltraite", "maltrait"},
		{"maltraité", "maltrait"},
		{"maltraitées", "maltrait"},
		{"maltraiter", "maltrait"},
		{"maman", "maman"},
		{"mameluks", "mameluk"},
		{"mammacia", "mammaci"},
		{"manant", "man"},
		{"mancar", "mancar"},
		{"manche", "manch"},
		{"manches", "manch"},
		{"manchester", "manchest"},
		{"manda", "mand"},
		{"mandarin", "mandarin"},
		{"mandat", "mandat"},
		{"mandé", "mand"},
		{"mandeb", "mandeb"},
		{"mandement", "mand"},
		{"mander", "mand"},
		{"mandiboy", "mandiboy"},
		{"manège", "maneg"},
		{"mânes", "mân"},
		{"manette", "manet"},
		{"mange", "mang"},
		{"mangé", "mang"},
		{"mangea", "mang"},
		{"mangeaient", "mang"},
		{"mangeais", "mang"},
		{"mangeait", "mang"},
		{"mangeant", "mang"},
		{"mangent", "mangent"},
		{"mangeoire", "mangeoir"},
		{"manger", "mang"},
		{"mangerai", "mang"},
		{"mangerait", "mang"},
		{"mangèrent", "mang"},
		{"mangeur", "mangeur"},
		{"mangoustes", "mangoust"},
		{"maniable", "maniabl"},
		{"maniait", "mani"},
		{"maniant", "mani"},
		{"maniaque", "maniaqu"},
		{"manie", "man"},
		{"manié", "mani"},
		{"manier", "mani"},
		{"manière", "mani"},
		{"manières", "mani"},
		{"manifestation", "manifest"},
		{"manifesté", "manifest"},
		{"manifester", "manifest"},
		{"mannequins", "mannequin"},
		{"manner", "mann"},
		{"manoeuvraient", "manoeuvr"},
		{"manoeuvrant", "manoeuvr"},
		{"manoeuvre", "manoeuvr"},
		{"manoeuvré", "manoeuvr"},
		{"manoeuvrer", "manoeuvr"},
		{"manoeuvres", "manoeuvr"},
		{"manoeuvrés", "manoeuvr"},
		{"manon", "manon"},
		{"manouvriers", "manouvri"},
		{"manqua", "manqu"},
		{"manquaient", "manqu"},
		{"manquais", "manqu"},
		{"manquait", "manqu"},
		{"manquant", "manqu"},
		{"manquât", "manqu"},
		{"manque", "manqu"},
		{"manqué", "manqu"},
		{"manquée", "manqu"},
		{"manquent", "manquent"},
		{"manquer", "manqu"},
		{"manquera", "manqu"},
		{"manquerai", "manqu"},
		{"manqueraient", "manqu"},
		{"manquerais", "manqu"},
		{"manquerait", "manqu"},
		{"manquèrent", "manqu"},
		{"manquerez", "manqu"},
		{"manqueront", "manqu"},
		{"manques", "manqu"},
		{"manquez", "manqu"},
		{"manquons", "manquon"},
		{"mansarde", "mansard"},
		{"manteau", "manteau"},
		{"manteaux", "manteau"},
		{"mantoue", "mantou"},
		{"manuel", "manuel"},
		{"manufacture", "manufactur"},
		{"manufactures", "manufactur"},
		{"manufacturières", "manufacturi"},
		{"manufacturiers", "manufacturi"},
		{"manuscrit", "manuscr"},
		{"manuscrites", "manuscrit"},
		{"manuscrits", "manuscrit"},
		{"maquignon", "maquignon"},
		{"marais", "mar"},
		{"marâtre", "marâtr"},
		{"marbre", "marbr"},
		{"marbres", "marbr"},
		{"marcha", "march"},
		{"marchaient", "march"},
		{"marchait", "march"},
		{"marchand", "marchand"},
		{"marchande", "marchand"},
		{"marchandise", "marchandis"},
		{"marchandises", "marchandis"},
		{"marchands", "marchand"},
		{"marchant", "march"},
		{"marche", "march"},
		{"marché", "march"},
		{"marchent", "marchent"},
		{"marchepied", "marchepied"},
		{"marchepieds", "marchepied"},
		{"marcher", "march"},
		{"marcherez", "march"},
		{"marcherions", "march"},
		{"marcherons", "march"},
		{"marches", "march"},
		{"marchés", "march"},
		{"marchesi", "marches"},
		{"marchesine", "marchesin"},
		{"marchesino", "marchesino"},
		{"marcheur", "marcheur"},
		{"marcheurs", "marcheur"},
		{"marchez", "march"},
		{"marchiez", "march"},
		{"marchions", "marchion"},
		{"marchons", "marchon"},
		{"mardi", "mard"},
		{"mare", "mar"},
		{"marécages", "marécag"},
		{"marécageuse", "marécag"},
		{"maréchal", "maréchal"},
		{"maréchale", "maréchal"},
		{"maréchaux", "maréchal"},
		{"marée", "mar"},
		{"marengo", "marengo"},
		{"marge", "marg"},
		{"marges", "marg"},
		{"margot", "margot"},
		{"marguerite", "marguerit"},
		{"marguillier", "marguilli"},
		{"mari", "mar"},
		{"maria", "mari"},
		{"mariage", "mariag"},
		{"mariages", "mariag"},
		{"mariait", "mari"},
		{"marie", "mar"},
		{"marié", "mari"},
		{"mariée", "mari"},
		{"mariées", "mari"},
		{"marier", "mari"},
		{"mariera", "mari"},
		{"marietta", "mariet"},
		{"mariettina", "mariettin"},
		{"marin", "marin"},
		{"marini", "marin"},
		{"mariniers", "marini"},
		{"marino", "marino"},
		{"marins", "marin"},
		{"maris", "mar"},
		{"maritime", "maritim"},
		{"maritimes", "maritim"},
		{"markésine", "markésin"},
		{"market", "market"},
		{"marmande", "marmand"},
		{"marmots", "marmot"},
		{"maroquin", "maroquin"},
		{"marote", "marot"},
		{"marotte", "marott"},
		{"marqua", "marqu"},
		{"marquaient", "marqu"},
		{"marquait", "marqu"},
		{"marquant", "marqu"},
		{"marque", "marqu"},
		{"marqué", "marqu"},
		{"marquée", "marqu"},
		{"marquées", "marqu"},
		{"marquent", "marquent"},
		{"marquer", "marqu"},
		{"marquerai", "marqu"},
		{"marquèrent", "marqu"},
		{"marques", "marqu"},
		{"marqués", "marqu"},
		{"marqueterie", "marqueter"},
		{"marquinot", "marquinot"},
		{"marquis", "marqu"},
		{"marquise", "marquis"},
		{"marronnier", "marronni"},
		{"marronniers", "marronni"},
		{"marrons", "marron"},
		{"mars", "mar"},
		{"marseille", "marseil"},
		{"marteau", "marteau"},
		{"marteaux", "marteau"},
		{"martial", "martial"},
		{"martiale", "martial"},
		{"martin", "martin"},
		{"martyr", "martyr"},
		{"martyre", "martyr"},
		{"mary", "mary"},
		{"mascarilles", "mascarill"},
		{"masculin", "masculin"},
		{"masculine", "masculin"},
		{"maslon", "maslon"},
		{"masquait", "masqu"},
		{"masque", "masqu"},
		{"masqué", "masqu"},
		{"masquée", "masqu"},
		{"masquer", "masqu"},
		{"masques", "masqu"},
		{"masqués", "masqu"},
		{"massacre", "massacr"},
		{"massacrer", "massacr"},
		{"massacrés", "massacr"},
		{"massant", "mass"},
		{"masse", "mass"},
		{"masses", "mass"},
		{"massif", "massif"},
		{"massifs", "massif"},
		{"massillon", "massillon"},
		{"massinger", "massing"},
		{"mat", "mat"},
		{"mât", "mât"},
		{"matches", "match"},
		{"matelas", "matel"},
		{"matelots", "matelot"},
		{"matériel", "matériel"},
		{"matérielle", "matériel"},
		{"matériellement", "matériel"},
		{"matérielles", "matériel"},
		{"matériels", "matériel"},
		{"maternel", "maternel"},
		{"mathématique", "mathémat"},
		{"mathématiquement", "mathémat"},
		{"mathématiques", "mathémat"},
		{"mathilde", "mathild"},
		{"matiere", "matier"},
		{"matière", "mati"},
		{"matieres", "matier"},
		{"matières", "mati"},
		{"matin", "matin"},
		{"matinale", "matinal"},
		{"matinée", "matin"},
		{"matins", "matin"},
		{"matrimoniale", "matrimonial"},
		{"matrimonio", "matrimonio"},
		{"mâts", "mât"},
		{"mâture", "mâtur"},
		{"mâtures", "mâtur"},
		{"maubeuge", "maubeug"},
		{"maudire", "maudir"},
		{"maudis", "maud"},
		{"maudissait", "maud"},
		{"maudissant", "maud"},
		{"maudissez", "maud"},
		{"maudit", "maud"},
		{"maudite", "maudit"},
		{"maudits", "maudit"},
		{"maugiron", "maugiron"},
		{"maugirons", "maug"},
		{"maugréait", "maugré"},
		{"maury", "maury"},
		{"mausolée", "mausol"},
		{"maussade", "maussad"},
		{"mauvais", "mauv"},
		{"mauvaise", "mauvais"},
		{"mauvaises", "mauvais"},
		{"mauve", "mauv"},
		{"maux", "maux"},
		{"maxime", "maxim"},
		{"maximes", "maxim"},
		{"maximum", "maximum"},
		{"mayence", "mayenc"},
		{"me", "me"},
		{"meako", "meako"},
		{"mean", "mean"},
		{"mécanicien", "mécanicien"},
		{"mécanique", "mécan"},
		{"mécaniquement", "mécan"},
		{"mécanisme", "mécan"},
		{"mécène", "mécen"},
		{"méchanceté", "méchancet"},
		{"méchancetés", "méchancet"},
		{"méchant", "mech"},
		{"méchante", "mech"},
		{"méchantes", "mech"},
		{"méchants", "mech"},
		{"mèche", "mech"},
		{"mécomptes", "mécompt"},
		{"méconnaissez", "méconnaiss"},
		{"mécontent", "mécontent"},
		{"mécontente", "mécontent"},
		{"mécontentement", "mécontent"},
		{"mécontents", "mécontent"},
		{"médaille", "médaill"},
		{"médecin", "médecin"},
		{"médecine", "médecin"},
		{"médecins", "médecin"},
		{"médée", "med"},
		{"medicine", "medicin"},
		{"médicis", "médic"},
		{"médiocre", "médiocr"},
		{"médiocrement", "médiocr"},
		{"médiocres", "médiocr"},
		{"médiocrité", "médiocr"},
		{"médire", "médir"},
		{"médisance", "médis"},
		{"médît", "med"},
		{"médita", "médit"},
		{"méditait", "médit"},
		{"méditation", "médit"},
		{"méditations", "médit"},
		{"méditée", "médit"},
		{"méditer", "médit"},
		{"meeting", "meeting"},
		{"méfiance", "méfianc"},
		{"méfiant", "méfi"},
		{"méfie", "mef"},
		{"méfier", "méfi"},
		{"meilleur", "meilleur"},
		{"meilleure", "meilleur"},
		{"meilleures", "meilleur"},
		{"meilleurs", "meilleur"},
		{"mêla", "mêl"},
		{"mêlaient", "mêl"},
		{"mêlait", "mêl"},
		{"mélancolie", "mélancol"},
		{"mélancolique", "mélancol"},
		{"mélancoliquement", "mélancol"},
		{"mélange", "mélang"},
		{"mélangé", "mélang"},
		{"mélangée", "mélang"},
		{"mélangées", "mélang"},
		{"mêlant", "mêl"},
		{"mêle", "mêl"},
		{"mêlé", "mêl"},
		{"mêlée", "mêl"},
		{"mêlent", "mêlent"},
		{"mêler", "mêl"},
		{"mêlés", "mêl"},
		{"mêlez", "mêl"},
		{"mélodie", "mélod"},
		{"mélodieuse", "mélodi"},
		{"mélodrame", "mélodram"},
		{"melzi", "melz"},
		{"membre", "membr"},
		{"membres", "membr"},
		{"membrure", "membrur"},
		{"meme", "mem"},
		{"même", "mêm"},
		{"mêmes", "mêm"},
		{"mémoire", "mémoir"},
		{"mémoires", "mémoir"},
		{"mémorial", "mémorial"},
		{"menaça", "menac"},
		{"menaçaient", "menac"},
		{"menaçais", "menac"},
		{"menaçait", "menac"},
		{"menaçant", "menac"},
		{"menace", "menac"},
		{"menacé", "menac"},
		{"menacée", "menac"},
		{"menacer", "menac"},
		{"menaces", "menac"},
		{"ménage", "ménag"},
		{"ménagé", "ménag"},
		{"ménagement", "ménag"},
		{"ménagements", "ménag"},
		{"ménager", "ménag"},
		{"ménagera", "ménag"},
		{"ménages", "ménag"},
		{"ménagez", "ménag"},
		{"menagio", "menagio"},
		{"menait", "men"},
		{"mendiant", "mendi"},
		{"mendiante", "mendi"},
		{"mendicité", "mendiqu"},
		{"mène", "men"},
		{"mené", "men"},
		{"menée", "men"},
		{"menées", "men"},
		{"mener", "men"},
		{"mènera", "men"},
		{"mènerai", "men"},
		{"mènerais", "men"},
		{"mènerait", "men"},
		{"menèrent", "men"},
		{"meneurs", "meneur"},
		{"menez", "men"},
		{"menottes", "menott"},
		{"mens", "men"},
		{"mensonge", "mensong"},
		{"mensonges", "mensong"},
		{"mentait", "ment"},
		{"mentales", "mental"},
		{"mente", "ment"},
		{"menteurs", "menteur"},
		{"menteuse", "menteux"},
		{"mentez", "ment"},
		{"menti", "ment"},
		{"mention", "mention"},
		{"mentionne", "mention"},
		{"mentionner", "mention"},
		{"mentir", "ment"},
		{"mentirais", "ment"},
		{"menton", "menton"},
		{"menues", "menu"},
		{"menuiserie", "menuiser"},
		{"menuisier", "menuisi"},
		{"menuisiers", "menuisi"},
		{"méphistophélès", "méphistophéles"},
		{"méprenait", "mépren"},
		{"méprendre", "méprendr"},
		{"mépris", "mepr"},
		{"méprisa", "mépris"},
		{"méprisable", "mépris"},
		{"méprisais", "mépris"},
		{"méprisait", "mépris"},
		{"méprisant", "mépris"},
		{"méprise", "mépris"},
		{"méprisé", "mépris"},
		{"méprisée", "mépris"},
		{"méprisent", "méprisent"},
		{"mépriser", "mépris"},
		{"méprisera", "mépris"},
		{"mépriserai", "mépris"},
		{"mépriseraient", "mépris"},
		{"mépriserait", "mépris"},
		{"méprisez", "mépris"},
		{"mer", "mer"},
		{"mercadante", "mercad"},
		{"mercantile", "mercantil"},
		{"merci", "merc"},
		{"mercredi", "mercred"},
		{"mercure", "mercur"},
		{"mère", "mer"},
		{"méridien", "méridien"},
		{"méridiens", "méridien"},
		{"méridional", "méridional"},
		{"méridionale", "méridional"},
		{"méridionaux", "méridional"},
		{"mérimée", "mérim"},
		{"méritait", "mérit"},
		{"méritant", "mérit"},
		{"mérite", "mérit"},
		{"mérité", "mérit"},
		{"méritée", "mérit"},
		{"méritées", "mérit"},
		{"mériter", "mérit"},
		{"méritera", "mérit"},
		{"mériterais", "mérit"},
		{"méritèrent", "mérit"},
		{"mérites", "mérit"},
		{"mérités", "mérit"},
		{"méritez", "mérit"},
		{"méritoire", "méritoir"},
		{"mers", "mer"},
		{"merveille", "merveil"},
		{"merveilles", "merveil"},
		{"merveilleuse", "merveil"},
		{"merveilleusement", "merveil"},
		{"merveilleuses", "merveil"},
		{"merveilleux", "merveil"},
		{"méry", "méry"},
		{"mes", "me"},
		{"mésalliance", "mésalli"},
		{"mésaventure", "mésaventur"},
		{"mesi", "mes"},
		{"mesquin", "mesquin"},
		{"mesquine", "mesquin"},
		{"mesquins", "mesquin"},
		{"message", "messag"},
		{"messager", "messag"},
		{"messageries", "messager"},
		{"messe", "mess"},
		{"messes", "mess"},
		{"messieurs", "messieur"},
		{"messire", "messir"},
		{"mesure", "mesur"},
		{"mesuré", "mesur"},
		{"mesurée", "mesur"},
		{"mesurées", "mesur"},
		{"mesurer", "mesur"},
		{"mesures", "mesur"},
		{"mesurés", "mesur"},
		{"met", "met"},
		{"métal", "métal"},
		{"métalliques", "métall"},
		{"métaphysiques", "métaphys"},
		{"météorologiste", "météorolog"},
		{"méthodes", "méthod"},
		{"méthodique", "méthod"},
		{"méthodiquement", "méthod"},
		{"méthodisme", "méthod"},
		{"méticuleuse", "méticul"},
		{"méticuleux", "méticul"},
		{"métier", "méti"},
		{"métiers", "méti"},
		{"mètres", "metr"},
		{"métropole", "métropol"},
		{"métropolitaine", "métropolitain"},
		{"mets", "met"},
		{"mettaient", "met"},
		{"mettais", "met"},
		{"mettait", "met"},
		{"mettant", "met"},
		{"mette", "met"},
		{"mettent", "mettent"},
		{"mettez", "met"},
		{"mettions", "mettion"},
		{"mettons", "metton"},
		{"mettra", "mettr"},
		{"mettrai", "mettr"},
		{"mettraient", "mettr"},
		{"mettrais", "mettr"},
		{"mettrait", "mettr"},
		{"mettras", "mettr"},
		{"mettre", "mettr"},
		{"mettrez", "mettr"},
		{"mettriez", "mettr"},
		{"mettrons", "mettron"},
		{"mettront", "mettront"},
		{"metz", "metz"},
		{"meuble", "meubl"},
		{"meublée", "meubl"},
		{"meublées", "meubl"},
		{"meubler", "meubl"},
		{"meubles", "meubl"},
		{"meudon", "meudon"},
		{"meunier", "meuni"},
		{"meure", "meur"},
		{"meures", "meur"},
		{"meurs", "meur"},
		{"meurt", "meurt"},
		{"meurtre", "meurtr"},
		{"meurtres", "meurtr"},
		{"meurtrier", "meurtri"},
		{"meurtrière", "meurtri"},
		{"meurtris", "meurtr"},
		{"meurtrissure", "meurtrissur"},
		{"meus", "meus"},
		{"mexicain", "mexicain"},
		{"mexique", "mexiqu"},
		{"mezzo", "mezzo"},
		{"mi", "mi"},
		{"mia", "mi"},
		{"miaulé", "miaul"},
		{"michaud", "michaud"},
		{"michel", "michel"},
		{"michele", "michel"},
		{"michelet", "michelet"},
		{"micheli", "michel"},
		{"michigan", "michigan"},
		{"microscope", "microscop"},
		{"microscopique", "microscop"},
		{"microscopiques", "microscop"},
		{"midi", "mid"},
		{"mielleuse", "mielleux"},
		{"mielleuses", "mielleux"},
		{"mien", "mien"},
		{"mienne", "mien"},
		{"miens", "mien"},
		{"miettes", "miet"},
		{"mieux", "mieux"},
		{"mignonnes", "mignon"},
		{"migraine", "migrain"},
		{"migration", "migrat"},
		{"mikado", "mikado"},
		{"milan", "milan"},
		{"milanais", "milan"},
		{"milanaise", "milanais"},
		{"milieu", "milieu"},
		{"militaire", "militair"},
		{"militaires", "militair"},
		{"mille", "mill"},
		{"milles", "mill"},
		{"milliard", "milliard"},
		{"milliards", "milliard"},
		{"milliers", "milli"},
		{"million", "million"},
		{"millionnaire", "millionnair"},
		{"millionnaires", "millionnair"},
		{"millions", "million"},
		{"mimosées", "mimos"},
		{"minaret", "minaret"},
		{"minarets", "minaret"},
		{"mince", "minc"},
		{"minces", "minc"},
		{"mind", "mind"},
		{"mine", "min"},
		{"minéralogie", "minéralog"},
		{"minéraux", "minéral"},
		{"minerve", "minerv"},
		{"mines", "min"},
		{"mineurs", "mineur"},
		{"miniature", "miniatur"},
		{"minimum", "minimum"},
		{"ministère", "minister"},
		{"ministères", "minister"},
		{"ministériel", "ministériel"},
		{"ministérielle", "ministériel"},
		{"ministre", "ministr"},
		{"ministres", "ministr"},
		{"minuit", "minuit"},
		{"minute", "minut"},
		{"minutes", "minut"},
		{"mirabeau", "mirabeau"},
		{"miracle", "miracl"},
		{"miracles", "miracl"},
		{"miraculeuse", "miracul"},
		{"mirari", "mirar"},
		{"miraut", "miraut"},
		{"mire", "mir"},
		{"mirent", "mirent"},
		{"miroir", "miroir"},
		{"mis", "mis"},
		{"misaine", "misain"},
		{"mise", "mis"},
		{"misène", "misen"},
		{"misérable", "miser"},
		{"misérables", "miser"},
		{"misère", "miser"},
		{"misères", "miser"},
		{"miséricorde", "miséricord"},
		{"mises", "mis"},
		{"missel", "missel"},
		{"mission", "mission"},
		{"missionnaire", "missionnair"},
		{"missions", "mission"},
		{"mississippi", "mississipp"},
		{"missive", "missiv"},
		{"missouri", "missour"},
		{"mit", "mit"},
		{"mît", "mît"},
		{"mitiger", "mitig"},
		{"mitre", "mitr"},
		{"mitres", "mitr"},
		{"mm", "mm"},
		{"mme", "mme"},
		{"mobile", "mobil"},
		{"mobiles", "mobil"},
		{"mobilité", "mobil"},
		{"mode", "mod"},
		{"modèle", "model"},
		{"modelée", "model"},
		{"modèles", "model"},
		{"modène", "moden"},
		{"modéra", "moder"},
		{"modération", "moder"},
		{"modéré", "moder"},
		{"modérée", "moder"},
		{"modérés", "moder"},
		{"modérez", "moder"},
		{"moderne", "modern"},
		{"modernes", "modern"},
		{"modes", "mod"},
		{"modeste", "modest"},
		{"modestie", "modest"},
		{"modifia", "modifi"},
		{"modification", "modif"},
		{"modifiée", "modifi"},
		{"modifier", "modifi"},
		{"modifierait", "modifi"},
		{"moelleux", "moelleux"},
		{"moeurs", "moeur"},
		{"moi", "moi"},
		{"moindre", "moindr"},
		{"moindres", "moindr"},
		{"moine", "moin"},
		{"moineaux", "moineau"},
		{"moines", "moin"},
		{"moins", "moin"},
		{"moirod", "moirod"},
		{"mois", "mois"},
		{"moisson", "moisson"},
		{"moitié", "moiti"},
		{"moka", "mok"},
		{"mole", "mol"},
		{"molière", "moli"},
		{"molle", "moll"},
		{"mollement", "moll"},
		{"molleton", "molleton"},
		{"mollissait", "moll"},
		{"mollit", "moll"},
		{"moltiplico", "moltiplico"},
		{"moment", "moment"},
		{"momentanée", "momentan"},
		{"moments", "moment"},
		{"momie", "mom"},
		{"momies", "mom"},
		{"mon", "mon"},
		{"monaca", "monac"},
		{"monarchie", "monarch"},
		{"monarchies", "monarch"},
		{"monarchique", "monarch"},
		{"monarchiques", "monarch"},
		{"monastères", "monaster"},
		{"moncade", "moncad"},
		{"monceau", "monceau"},
		{"moncontour", "moncontour"},
		{"mondain", "mondain"},
		{"mondaine", "mondain"},
		{"mondaines", "mondain"},
		{"mondains", "mondain"},
		{"monde", "mond"},
		{"monférine", "monférin"},
		{"monfleury", "monfleury"},
		{"monghir", "mongh"},
		{"mongolia", "mongoli"},
		{"moniteur", "moniteur"},
		{"monnaie", "monnai"},
		{"monologue", "monologu"},
		{"monomanes", "monoman"},
		{"monopole", "monopol"},
		{"monosyllabes", "monosyllab"},
		{"monotone", "monoton"},
		{"monotones", "monoton"},
		{"monseigneur", "monseigneur"},
		{"monsieur", "monsieur"},
		{"monsignor", "monsignor"},
		{"monsignore", "monsignor"},
		{"monstre", "monstr"},
		{"monstres", "monstr"},
		{"monstrueuses", "monstrueux"},
		{"mont", "mont"},
		{"monta", "mont"},
		{"montagne", "montagn"},
		{"montagnes", "montagn"},
		{"montagneuse", "montagn"},
		{"montagneux", "montagn"},
		{"montagnola", "montagnol"},
		{"montaient", "mont"},
		{"montais", "mont"},
		{"montait", "mont"},
		{"montant", "mont"},
		{"montât", "mont"},
		{"monte", "mont"},
		{"monté", "mont"},
		{"montée", "mont"},
		{"montent", "montent"},
		{"monter", "mont"},
		{"montera", "mont"},
		{"monterai", "mont"},
		{"montèrent", "mont"},
		{"montés", "mont"},
		{"montesson", "montesson"},
		{"montez", "mont"},
		{"montgommery", "montgommery"},
		{"monti", "mont"},
		{"montmartre", "montmartr"},
		{"montmirail", "montmirail"},
		{"montmorency", "montmorency"},
		{"montoir", "montoir"},
		{"montons", "monton"},
		{"montra", "montr"},
		{"montraient", "montr"},
		{"montrait", "montr"},
		{"montrant", "montr"},
		{"montrât", "montr"},
		{"montre", "montr"},
		{"montré", "montr"},
		{"montrée", "montr"},
		{"montrent", "montrent"},
		{"montrer", "montr"},
		{"montrera", "montr"},
		{"montrerai", "montr"},
		{"montrerait", "montr"},
		{"montrèrent", "montr"},
		{"montreriez", "montr"},
		{"montres", "montr"},
		{"montrés", "montr"},
		{"montreur", "montreur"},
		{"montrez", "montr"},
		{"monts", "mont"},
		{"monture", "montur"},
		{"monument", "monu"},
		{"monumental", "monumental"},
		{"monuments", "monu"},
		{"moqua", "moqu"},
		{"moquai", "moqu"},
		{"moquaient", "moqu"},
		{"moquais", "moqu"},
		{"moquait", "moqu"},
		{"moquant", "moqu"},
		{"moque", "moqu"},
		{"moqué", "moqu"},
		{"moquent", "moquent"},
		{"moquer", "moqu"},
		{"moquera", "moqu"},
		{"moquerais", "moqu"},
		{"moquerait", "moqu"},
		{"moquerie", "moquer"},
		{"moqueur", "moqueur"},
		{"moqueurs", "moqueur"},
		{"moqueuse", "moqueux"},
		{"moral", "moral"},
		{"morale", "moral"},
		{"morales", "moral"},
		{"moralité", "moral"},
		{"moralités", "moral"},
		{"moraux", "moral"},
		{"moravie", "morav"},
		{"morbleu", "morbleu"},
		{"morceau", "morceau"},
		{"morceaux", "morceau"},
		{"mord", "mord"},
		{"mordait", "mord"},
		{"mordant", "mord"},
		{"mordantes", "mord"},
		{"moreri", "morer"},
		{"morfondre", "morfondr"},
		{"mormon", "mormon"},
		{"mormone", "mormon"},
		{"mormones", "mormon"},
		{"mormonisme", "mormon"},
		{"mormons", "mormon"},
		{"morne", "morn"},
		{"mornes", "morn"},
		{"morning", "morning"},
		{"morom", "morom"},
		{"morose", "moros"},
		{"moroses", "moros"},
		{"mort", "mort"},
		{"mortaretti", "mortaret"},
		{"morte", "mort"},
		{"mortel", "mortel"},
		{"mortelle", "mortel"},
		{"mortellement", "mortel"},
		{"mortelles", "mortel"},
		{"mortels", "mortel"},
		{"mortes", "mort"},
		{"mortier", "morti"},
		{"mortiers", "morti"},
		{"mortifiantes", "mortifi"},
		{"mortifié", "mortifi"},
		{"mortifier", "mortifi"},
		{"morts", "mort"},
		{"morveux", "morveux"},
		{"mosca", "mosc"},
		{"moscou", "moscou"},
		{"moskova", "moskov"},
		{"mosquées", "mosqu"},
		{"most", "most"},
		{"mot", "mot"},
		{"moteur", "moteur"},
		{"motif", "motif"},
		{"motifs", "motif"},
		{"motivât", "motiv"},
		{"motivé", "motiv"},
		{"motrices", "motric"},
		{"mots", "mot"},
		{"motu", "motu"},
		{"mouche", "mouch"},
		{"mouchent", "mouchent"},
		{"moucher", "mouch"},
		{"moucheurs", "moucheur"},
		{"mouchoir", "mouchoir"},
		{"mouchoirs", "mouchoir"},
		{"mouettes", "mouet"},
		{"mouilla", "mouill"},
		{"mouillé", "mouill"},
		{"mouiller", "mouill"},
		{"mouillèrent", "mouill"},
		{"mouillés", "mouill"},
		{"moule", "moul"},
		{"moulin", "moulin"},
		{"moulus", "moulus"},
		{"mountains", "mountain"},
		{"mourais", "mour"},
		{"mourait", "mour"},
		{"mourant", "mour"},
		{"mourante", "mour"},
		{"mourants", "mour"},
		{"mourions", "mourion"},
		{"mourir", "mour"},
		{"mourons", "mouron"},
		{"mourra", "mourr"},
		{"mourrai", "mourr"},
		{"mourrais", "mourr"},
		{"mourrait", "mourr"},
		{"mourras", "mourr"},
		{"mourrez", "mourr"},
		{"mourrons", "mourron"},
		{"mourront", "mourront"},
		{"mourshedabad", "mourshedabad"},
		{"mourusse", "mouruss"},
		{"mourut", "mourut"},
		{"mourût", "mourût"},
		{"mousqueterie", "mousqueter"},
		{"mousseline", "mousselin"},
		{"mousseux", "mousseux"},
		{"moustache", "moustach"},
		{"moustaches", "moustach"},
		{"mouton", "mouton"},
		{"moutonnaient", "mouton"},
		{"moutons", "mouton"},
		{"mouvement", "mouv"},
		{"mouvements", "mouv"},
		{"mouvoir", "mouvoir"},
		{"moyen", "moyen"},
		{"moyenne", "moyen"},
		{"moyennes", "moyen"},
		{"moyens", "moyen"},
		{"moyeux", "moyeux"},
		{"mozart", "mozart"},
		{"mr", "mr"},
		{"mrs", "mr"},
		{"mû", "mû"},
		{"much", "much"},
		{"muddy", "muddy"},
		{"mudge", "mudg"},
		{"muet", "muet"},
		{"muets", "muet"},
		{"mugissements", "mug"},
		{"mulet", "mulet"},
		{"muletiers", "muleti"},
		{"mulhouse", "mulhous"},
		{"multicolores", "multicolor"},
		{"multiplications", "multipl"},
		{"multiplicité", "multipl"},
		{"multiplie", "multipl"},
		{"multiplier", "multipli"},
		{"multipliés", "multipli"},
		{"muni", "mun"},
		{"munich", "munich"},
		{"municipal", "municipal"},
		{"munir", "mun"},
		{"munis", "mun"},
		{"munster", "munst"},
		{"mur", "mur"},
		{"mûr", "mûr"},
		{"muraille", "muraill"},
		{"murailles", "muraill"},
		{"mural", "mural"},
		{"murat", "murat"},
		{"murées", "mur"},
		{"mûrement", "mûr"},
		{"mûrir", "mûr"},
		{"murmura", "murmur"},
		{"murmurait", "murmur"},
		{"murmure", "murmur"},
		{"murmures", "murmur"},
		{"murs", "mur"},
		{"muscadiers", "muscadi"},
		{"muscles", "muscl"},
		{"musculaire", "musculair"},
		{"musculature", "musculatur"},
		{"musée", "mus"},
		{"mushroom", "mushroom"},
		{"musicien", "musicien"},
		{"musiciens", "musicien"},
		{"musique", "musiqu"},
		{"musset", "musset"},
		{"mutilation", "mutil"},
		{"mutilé", "mutil"},
		{"mutsh", "mutsh"},
		{"mutuel", "mutuel"},
		{"mylord", "mylord"},
		{"mystère", "myster"},
		{"mystères", "myster"},
		{"mystérieuse", "mystéri"},
		{"mystérieusement", "mystéri"},
		{"mystérieux", "mystéri"},
		{"mysticité", "mystiqu"},
		{"mystification", "mystif"},
		{"mystifie", "mystif"},
		{"mystifié", "mystifi"},
		{"mystifiée", "mystifi"},
		{"mystifier", "mystifi"},
		{"mystique", "mystiqu"},
		{"n", "n"},
		{"nadir", "nad"},
		{"nagasaki", "nagasak"},
		{"nage", "nag"},
		{"nagent", "nagent"},
		{"nager", "nag"},
		{"naguère", "naguer"},
		{"naïf", "naïf"},
		{"naïfs", "naïf"},
		{"nains", "nain"},
		{"naissance", "naissanc"},
		{"naissant", "naiss"},
		{"naissante", "naiss"},
		{"naissent", "naissent"},
		{"naît", "naît"},
		{"naître", "naîtr"},
		{"naïve", "naïv"},
		{"naïvement", "naïv"},
		{"naïves", "naïv"},
		{"naïveté", "naïvet"},
		{"nanan", "nanan"},
		{"nani", "nan"},
		{"nankin", "nankin"},
		{"nanking", "nanking"},
		{"napier", "napi"},
		{"naples", "napl"},
		{"napoléon", "napoléon"},
		{"napoléons", "napoléon"},
		{"napolitain", "napolitain"},
		{"nappe", "napp"},
		{"narcotique", "narcot"},
		{"narra", "narr"},
		{"narrant", "narr"},
		{"narrateur", "narrateur"},
		{"narration", "narrat"},
		{"narrer", "narr"},
		{"nasillard", "nasillard"},
		{"nassik", "nassik"},
		{"natal", "natal"},
		{"natif", "natif"},
		{"nation", "nation"},
		{"national", "national"},
		{"nationale", "national"},
		{"nationales", "national"},
		{"nationalité", "national"},
		{"nationalités", "national"},
		{"nations", "nation"},
		{"nature", "natur"},
		{"naturel", "naturel"},
		{"naturelle", "naturel"},
		{"naturellement", "naturel"},
		{"naturelles", "naturel"},
		{"naturels", "naturel"},
		{"naufrages", "naufrag"},
		{"nauséabondes", "nauséabond"},
		{"nausées", "naus"},
		{"nautique", "nautiqu"},
		{"nauvoo", "nauvoo"},
		{"navarre", "navarr"},
		{"navigateur", "navig"},
		{"navigateurs", "navig"},
		{"navigation", "navig"},
		{"navigua", "navigu"},
		{"naviguant", "navigu"},
		{"naviguent", "naviguent"},
		{"navire", "navir"},
		{"navires", "navir"},
		{"navré", "navr"},
		{"navrée", "navr"},
		{"nazaro", "nazaro"},
		{"ne", "ne"},
		{"né", "né"},
		{"néanmoins", "néanmoin"},
		{"néant", "né"},
		{"nébieu", "nébieu"},
		{"nebraska", "nebrask"},
		{"nécessaire", "nécessair"},
		{"nécessairement", "nécessair"},
		{"nécessaires", "nécessair"},
		{"nécessitant", "nécessit"},
		{"nécessité", "nécess"},
		{"née", "né"},
		{"nées", "né"},
		{"nef", "nef"},
		{"nefs", "nef"},
		{"négatif", "négat"},
		{"néglige", "néglig"},
		{"négligé", "néglig"},
		{"négligeait", "néglig"},
		{"négligemment", "négligent"},
		{"négligence", "négligent"},
		{"négligents", "négligent"},
		{"négliger", "néglig"},
		{"négligerait", "néglig"},
		{"négociant", "négoci"},
		{"négociants", "négoci"},
		{"négociateur", "négoci"},
		{"négociation", "négoci"},
		{"négociations", "négoci"},
		{"nègre", "negr"},
		{"nègres", "negr"},
		{"neige", "neig"},
		{"neigeait", "neig"},
		{"neiges", "neig"},
		{"neptune", "neptun"},
		{"nerfs", "nerf"},
		{"nerval", "nerval"},
		{"nerveuse", "nerveux"},
		{"nerveux", "nerveux"},
		{"nés", "né"},
		{"net", "net"},
		{"nets", "net"},
		{"nette", "net"},
		{"nettement", "net"},
		{"nettes", "net"},
		{"netteté", "nettet"},
		{"nettoya", "nettoi"},
		{"nettoyer", "nettoi"},
		{"neuf", "neuf"},
		{"neufs", "neuf"},
		{"neutralité", "neutral"},
		{"neutre", "neutr"},
		{"neuvaine", "neuvain"},
		{"neuve", "neuv"},
		{"neuves", "neuv"},
		{"neuvième", "neuviem"},
		{"nevada", "nevad"},
		{"never", "nev"},
		{"neveu", "neveu"},
		{"neveux", "neveux"},
		{"new", "new"},
		{"news", "new"},
		{"ney", "ney"},
		{"nez", "nez"},
		{"ni", "ni"},
		{"niais", "ni"},
		{"niaiserie", "niaiser"},
		{"niaiseries", "niaiser"},
		{"niaises", "niais"},
		{"nice", "nic"},
		{"niche", "nich"},
		{"nicolas", "nicol"},
		{"nids", "nid"},
		{"nie", "ni"},
		{"nié", "ni"},
		{"nièce", "niec"},
		{"nièces", "niec"},
		{"nier", "ni"},
		{"nierais", "ni"},
		{"nierons", "ni"},
		{"nigaud", "nigaud"},
		{"nigauderie", "nigauder"},
		{"nigauds", "nigaud"},
		{"night", "night"},
		{"nil", "nil"},
		{"niveau", "niveau"},
		{"nivelées", "nivel"},
		{"nizam", "nizam"},
		{"no", "no"},
		{"nobiliaires", "nobiliair"},
		{"noble", "nobl"},
		{"nobles", "nobl"},
		{"noblesse", "nobless"},
		{"noce", "noc"},
		{"noces", "noc"},
		{"nocturne", "nocturn"},
		{"nocturnes", "nocturn"},
		{"noeud", "noeud"},
		{"noeuds", "noeud"},
		{"noie", "noi"},
		{"noir", "noir"},
		{"noirci", "noirc"},
		{"noircie", "noirc"},
		{"noircies", "noirc"},
		{"noircis", "noirc"},
		{"noire", "noir"},
		{"noires", "noir"},
		{"noiroud", "noiroud"},
		{"noirs", "noir"},
		{"noix", "noix"},
		{"nom", "nom"},
		{"nombre", "nombr"},
		{"nombreuse", "nombreux"},
		{"nombreuses", "nombreux"},
		{"nombreux", "nombreux"},
		{"nominal", "nominal"},
		{"nominale", "nominal"},
		{"nomination", "nomin"},
		{"nommait", "nomm"},
		{"nommant", "nomm"},
		{"nomme", "nomm"},
		{"nommé", "nomm"},
		{"nommée", "nomm"},
		{"nommer", "nomm"},
		{"nommés", "nomm"},
		{"nommez", "nomm"},
		{"noms", "nom"},
		{"non", "non"},
		{"nonante", "non"},
		{"nonchalance", "nonchal"},
		{"nonchalant", "nonchal"},
		{"norbert", "norbert"},
		{"nord", "nord"},
		{"norimons", "norimon"},
		{"normale", "normal"},
		{"normandie", "normand"},
		{"normands", "normand"},
		{"north", "north"},
		{"nos", "nos"},
		{"not", "not"},
		{"notable", "notabl"},
		{"notablement", "notabl"},
		{"notables", "notabl"},
		{"notaire", "notair"},
		{"notaires", "notair"},
		{"note", "not"},
		{"noté", "not"},
		{"noter", "not"},
		{"notes", "not"},
		{"notez", "not"},
		{"nothing", "nothing"},
		{"notice", "notic"},
		{"notices", "notic"},
		{"notions", "notion"},
		{"notoire", "notoir"},
		{"notoirement", "notoir"},
		{"notre", "notr"},
		{"nôtre", "nôtr"},
		{"nôtres", "nôtr"},
		{"nouant", "nou"},
		{"nouée", "nou"},
		{"nourri", "nourr"},
		{"nourrice", "nourric"},
		{"nourrie", "nourr"},
		{"nourrir", "nourr"},
		{"nourrirait", "nourr"},
		{"nourris", "nourr"},
		{"nourrissait", "nourr"},
		{"nourrissant", "nourr"},
		{"nourrisse", "nourr"},
		{"nourrissent", "nourr"},
		{"nourrit", "nourr"},
		{"nourriture", "nourritur"},
		{"nous", "nous"},
		{"nouveau", "nouveau"},
		{"nouveauté", "nouveaut"},
		{"nouveautés", "nouveaut"},
		{"nouveaux", "nouveau"},
		{"nouvel", "nouvel"},
		{"nouvelle", "nouvel"},
		{"nouvellement", "nouvel"},
		{"nouvelles", "nouvel"},
		{"novare", "novar"},
		{"novateurs", "novateur"},
		{"novembre", "novembr"},
		{"novi", "nov"},
		{"novice", "novic"},
		{"now", "now"},
		{"noyée", "noi"},
		{"noyer", "noi"},
		{"noyers", "noyer"},
		{"noyés", "noi"},
		{"nu", "nu"},
		{"nuage", "nuag"},
		{"nuages", "nuag"},
		{"nuance", "nuanc"},
		{"nuances", "nuanc"},
		{"nudité", "nudit"},
		{"nue", "nu"},
		{"nuées", "nu"},
		{"nues", "nu"},
		{"nui", "nui"},
		{"nuira", "nuir"},
		{"nuiraient", "nuir"},
		{"nuirait", "nuir"},
		{"nuire", "nuir"},
		{"nuisait", "nuis"},
		{"nuisant", "nuis"},
		{"nuisible", "nuisibl"},
		{"nuisibles", "nuisibl"},
		{"nuit", "nuit"},
		{"nuits", "nuit"},
		{"nul", "nul"},
		{"nulle", "null"},
		{"nullement", "null"},
		{"nullité", "nullit"},
		{"nuls", "nul"},
		{"numéro", "numéro"},
		{"numéros", "numéros"},
		{"numérotées", "numérot"},
		{"nunc", "nunc"},
		{"nuremberg", "nuremberg"},
		{"nus", "nus"},
		{"o", "o"},
		{"ô", "ô"},
		{"oakland", "oakland"},
		{"oaths", "oath"},
		{"obadiah", "obadiah"},
		{"obéi", "obéi"},
		{"obéie", "obei"},
		{"obéir", "obéir"},
		{"obéira", "obéir"},
		{"obéirai", "obéir"},
		{"obéirais", "obéir"},
		{"obéirez", "obéir"},
		{"obéis", "obéis"},
		{"obéissaient", "obéiss"},
		{"obéissait", "obéiss"},
		{"obéissance", "obéiss"},
		{"obéissant", "obéiss"},
		{"obéissants", "obéiss"},
		{"obéisse", "obéiss"},
		{"obéissez", "obéiss"},
		{"obéit", "obéit"},
		{"objecter", "object"},
		{"objection", "object"},
		{"objections", "object"},
		{"objet", "objet"},
		{"objets", "objet"},
		{"obligation", "oblig"},
		{"obligations", "oblig"},
		{"obligatoire", "obligatoir"},
		{"oblige", "oblig"},
		{"obligé", "oblig"},
		{"obligea", "oblig"},
		{"obligeait", "oblig"},
		{"obligeamment", "oblig"},
		{"obligeances", "oblige"},
		{"obligeant", "oblig"},
		{"obligeante", "oblig"},
		{"obligeantes", "oblig"},
		{"obligeants", "oblig"},
		{"obligeât", "oblig"},
		{"obligée", "oblig"},
		{"obligent", "obligent"},
		{"obliger", "oblig"},
		{"obligera", "oblig"},
		{"obligeraient", "oblig"},
		{"obligerez", "oblig"},
		{"obligés", "oblig"},
		{"obligez", "oblig"},
		{"obliqua", "obliqu"},
		{"obole", "obol"},
		{"obscur", "obscur"},
		{"obscurcit", "obscurc"},
		{"obscure", "obscur"},
		{"obscurément", "obscur"},
		{"obscures", "obscur"},
		{"obscurité", "obscur"},
		{"obséda", "obsed"},
		{"obsédé", "obsed"},
		{"observa", "observ"},
		{"observaient", "observ"},
		{"observait", "observ"},
		{"observance", "observ"},
		{"observant", "observ"},
		{"observateur", "observ"},
		{"observateurs", "observ"},
		{"observation", "observ"},
		{"observations", "observ"},
		{"observatoire", "observatoir"},
		{"observe", "observ"},
		{"observé", "observ"},
		{"observer", "observ"},
		{"observèrent", "observ"},
		{"observés", "observ"},
		{"obstacle", "obstacl"},
		{"obstacles", "obstacl"},
		{"obstina", "obstin"},
		{"obstinait", "obstin"},
		{"obstinant", "obstin"},
		{"obstination", "obstin"},
		{"obstiné", "obstin"},
		{"obstinément", "obstin"},
		{"obstiner", "obstin"},
		{"obstruaient", "obstru"},
		{"obtenait", "obten"},
		{"obtenez", "obten"},
		{"obtenir", "obten"},
		{"obtenu", "obtenu"},
		{"obtenue", "obtenu"},
		{"obtenus", "obtenus"},
		{"obtiendrai", "obtiendr"},
		{"obtiendrais", "obtiendr"},
		{"obtiendront", "obtiendront"},
		{"obtienne", "obtien"},
		{"obtiennent", "obtiennent"},
		{"obtiens", "obtien"},
		{"obtient", "obtient"},
		{"obtinrent", "obtinrent"},
		{"obtint", "obtint"},
		{"occasion", "occas"},
		{"occasionner", "occasion"},
		{"occasionnés", "occasion"},
		{"occasions", "occas"},
		{"occidentales", "occidental"},
		{"occulte", "occult"},
		{"occultes", "occult"},
		{"occupa", "occup"},
		{"occupaient", "occup"},
		{"occupait", "occup"},
		{"occupant", "occup"},
		{"occupante", "occup"},
		{"occupât", "occup"},
		{"occupation", "occup"},
		{"occupations", "occup"},
		{"occupe", "occup"},
		{"occupé", "occup"},
		{"occupée", "occup"},
		{"occupées", "occup"},
		{"occupent", "occupent"},
		{"occuper", "occup"},
		{"occupera", "occup"},
		{"occuperai", "occup"},
		{"occupèrent", "occup"},
		{"occupés", "occup"},
		{"occupiez", "occup"},
		{"occupons", "occupon"},
		{"occurrence", "occurrent"},
		{"occurrences", "occurrent"},
		{"ocean", "ocean"},
		{"océan", "océan"},
		{"océans", "océan"},
		{"ocre", "ocre"},
		{"octave", "octav"},
		{"octobre", "octobr"},
		{"octroi", "octroi"},
		{"odalisque", "odalisqu"},
		{"ode", "ode"},
		{"odes", "ode"},
		{"odeur", "odeur"},
		{"odieuse", "odieux"},
		{"odieux", "odieux"},
		{"odorante", "odor"},
		{"odorat", "odorat"},
		{"oeil", "oeil"},
		{"oeuf", "oeuf"},
		{"oeufs", "oeuf"},
		{"oeuvre", "oeuvr"},
		{"oeuvres", "oeuvr"},
		{"of", "of"},
		{"offensa", "offens"},
		{"offensait", "offens"},
		{"offensant", "offens"},
		{"offensante", "offens"},
		{"offensantes", "offens"},
		{"offensât", "offens"},
		{"offense", "offens"},
		{"offensé", "offens"},
		{"offensée", "offens"},
		{"offenser", "offens"},
		{"offenserait", "offens"},
		{"offenses", "offens"},
		{"offensés", "offens"},
		{"offensive", "offens"},
		{"offert", "offert"},
		{"offerte", "offert"},
		{"offerts", "offert"},
		{"office", "offic"},
		{"offices", "offic"},
		{"officiant", "offici"},
		{"officiel", "officiel"},
		{"officielle", "officiel"},
		{"officiellement", "officiel"},
		{"officielles", "officiel"},
		{"officiels", "officiel"},
		{"officier", "offici"},
		{"officiers", "offici"},
		{"offraient", "offraient"},
		{"offrais", "offrais"},
		{"offrait", "offrait"},
		{"offrande", "offrand"},
		{"offrandes", "offrand"},
		{"offrant", "offrant"},
		{"offre", "offre"},
		{"offrent", "offrent"},
		{"offres", "offre"},
		{"offrir", "offrir"},
		{"offrira", "offrir"},
		{"offrirai", "offrir"},
		{"offriraient", "offrir"},
		{"offrirait", "offrir"},
		{"offris", "offris"},
		{"offrit", "offrit"},
		{"offusqué", "offusqu"},
		{"ogden", "ogden"},
		{"ogive", "ogiv"},
		{"ogre", "ogre"},
		{"oh", "oh"},
		{"ohio", "ohio"},
		{"oies", "oie"},
		{"oiseau", "oiseau"},
		{"oiseaux", "oiseau"},
		{"oisifs", "oisif"},
		{"oisive", "oisiv"},
		{"olivier", "olivi"},
		{"olo", "olo"},
		{"omaha", "omah"},
		{"ombragée", "ombrag"},
		{"ombrages", "ombrag"},
		{"ombre", "ombre"},
		{"ombres", "ombre"},
		{"ombreuses", "ombreux"},
		{"omelette", "omelet"},
		{"omis", "omis"},
		{"omission", "omiss"},
		{"omit", "omit"},
		{"omnia", "omni"},
		{"omnibus", "omnibus"},
		{"on", "on"},
		{"onces", "once"},
		{"oncle", "oncle"},
		{"oncles", "oncle"},
		{"onction", "onction"},
		{"ondulation", "ondul"},
		{"ondulations", "ondul"},
		{"ont", "ont"},
		{"onze", "onze"},
		{"opera", "oper"},
		{"opéra", "oper"},
		{"opéraient", "oper"},
		{"opérait", "oper"},
		{"opération", "oper"},
		{"opérations", "oper"},
		{"opère", "oper"},
		{"opérée", "oper"},
		{"opérer", "oper"},
		{"opiner", "opin"},
		{"opiniâtrement", "opiniâtr"},
		{"opinion", "opinion"},
		{"opinions", "opin"},
		{"opium", "opium"},
		{"opportun", "opportun"},
		{"opportune", "opportun"},
		{"opportunité", "opportun"},
		{"opposaient", "oppos"},
		{"opposant", "oppos"},
		{"oppose", "oppos"},
		{"opposé", "oppos"},
		{"opposée", "oppos"},
		{"opposées", "oppos"},
		{"opposent", "opposent"},
		{"opposer", "oppos"},
		{"opposerai", "oppos"},
		{"opposés", "oppos"},
		{"opposition", "opposit"},
		{"opprimait", "opprim"},
		{"opprimés", "opprim"},
		{"opprobre", "opprobr"},
		{"opter", "opter"},
		{"optime", "optim"},
		{"opulence", "opulent"},
		{"opulentes", "opulent"},
		{"opulents", "opulent"},
		{"or", "or"},
		{"ora", "ora"},
		{"orage", "orag"},
		{"orages", "orag"},
		{"orange", "orang"},
		{"orangé", "orang"},
		{"oranger", "orang"},
		{"orangerie", "oranger"},
		{"orangers", "oranger"},
		{"oranges", "orang"},
		{"orateur", "orateur"},
		{"orateurs", "orateur"},
		{"orbite", "orbit"},
		{"orchestre", "orchestr"},
		{"ordinaire", "ordinair"},
		{"ordinairement", "ordinair"},
		{"ordinaires", "ordinair"},
		{"ordonna", "ordon"},
		{"ordonnait", "ordon"},
		{"ordonnance", "ordon"},
		{"ordonnant", "ordon"},
		{"ordonne", "ordon"},
		{"ordonné", "ordon"},
		{"ordonnée", "ordon"},
		{"ordonner", "ordon"},
		{"ordonnera", "ordon"},
		{"ordonnes", "ordon"},
		{"ordonnés", "ordon"},
		{"ordre", "ordre"},
		{"ordres", "ordre"},
		{"oregon", "oregon"},
		{"oreille", "oreil"},
		{"oreiller", "oreil"},
		{"oreillers", "oreiller"},
		{"oreilles", "oreil"},
		{"orfèvre", "orfevr"},
		{"orfèvrerie", "orfèvrer"},
		{"orfèvres", "orfevr"},
		{"orfraie", "orfrai"},
		{"organes", "organ"},
		{"organisa", "organis"},
		{"organisait", "organis"},
		{"organisation", "organis"},
		{"organisé", "organis"},
		{"organisée", "organis"},
		{"organiser", "organis"},
		{"organisèrent", "organis"},
		{"orge", "orge"},
		{"orgie", "orgi"},
		{"orgue", "orgu"},
		{"orgueil", "orgueil"},
		{"orgueilleuse", "orgueil"},
		{"orgueilleusement", "orgueil"},
		{"orgueilleux", "orgueil"},
		{"orient", "orient"},
		{"oriental", "oriental"},
		{"orientale", "oriental"},
		{"orientalistes", "oriental"},
		{"orienter", "orient"},
		{"original", "original"},
		{"originale", "original"},
		{"originalité", "original"},
		{"originaux", "original"},
		{"origine", "origin"},
		{"orléans", "orléan"},
		{"ornaient", "ornaient"},
		{"ornait", "ornait"},
		{"orne", "orne"},
		{"orné", "orné"},
		{"ornée", "orné"},
		{"ornées", "orné"},
		{"ornement", "ornement"},
		{"ornementation", "ornement"},
		{"ornements", "ornement"},
		{"ornent", "ornent"},
		{"orner", "orner"},
		{"ornèrent", "ornèrent"},
		{"ornés", "orné"},
		{"ornière", "ornier"},
		{"orpheline", "orphelin"},
		{"orphelins", "orphelin"},
		{"orta", "orta"},
		{"orteils", "orteil"},
		{"orthographe", "orthograph"},
		{"orties", "orti"},
		{"ory", "ory"},
		{"os", "os"},
		{"osa", "osa"},
		{"osai", "osai"},
		{"osaient", "osaient"},
		{"osais", "osais"},
		{"osait", "osait"},
		{"osant", "osant"},
		{"osât", "osât"},
		{"oscillaient", "oscill"},
		{"oscillait", "oscill"},
		{"oscuro", "oscuro"},
		{"ose", "ose"},
		{"osé", "osé"},
		{"osées", "osé"},
		{"oseille", "oseil"},
		{"osent", "osent"},
		{"oser", "oser"},
		{"osera", "oser"},
		{"oserai", "oser"},
		{"oseraient", "oser"},
		{"oserais", "oser"},
		{"oserait", "oser"},
		{"osèrent", "osèrent"},
		{"oserons", "oseron"},
		{"osez", "osez"},
		{"ostensiblement", "ostensibl"},
		{"ôta", "ôta"},
		{"ôtaient", "ôtaient"},
		{"ôtait", "ôtait"},
		{"ôte", "ôte"},
		{"ôté", "ôté"},
		{"ôtées", "ôté"},
		{"ôter", "ôter"},
		{"ôtera", "ôter"},
		{"ôterai", "ôter"},
		{"ôterait", "ôter"},
		{"ôtèrent", "ôtèrent"},
		{"ôterez", "ôter"},
		{"ôtez", "ôtez"},
		{"othello", "othello"},
		{"ottomane", "ottoman"},
		{"ou", "ou"},
		{"où", "où"},
		{"ouailles", "ouaill"},
		{"oubli", "oubl"},
		{"oublia", "oubli"},
		{"oubliai", "oubli"},
		{"oubliais", "oubli"},
		{"oubliait", "oubli"},
		{"oubliant", "oubli"},
		{"oublie", "oubl"},
		{"oublié", "oubli"},
		{"oubliée", "oubli"},
		{"oubliées", "oubli"},
		{"oublient", "oublient"},
		{"oublier", "oubli"},
		{"oubliera", "oubli"},
		{"oublierai", "oubli"},
		{"oublierais", "oubli"},
		{"oublierait", "oubli"},
		{"oublierez", "oubli"},
		{"oublies", "oubl"},
		{"oubliez", "oubl"},
		{"oublions", "oublion"},
		{"ouest", "ouest"},
		{"oui", "oui"},
		{"ouï", "ouï"},
		{"ouïe", "ouï"},
		{"our", "our"},
		{"ouragan", "ouragan"},
		{"ouragans", "ouragan"},
		{"ourdies", "ourd"},
		{"ours", "our"},
		{"outils", "outil"},
		{"outrage", "outrag"},
		{"outrageait", "outrag"},
		{"outrageantes", "outrag"},
		{"outrageants", "outrag"},
		{"outragée", "outrag"},
		{"outrager", "outrag"},
		{"outrages", "outrag"},
		{"outragés", "outrag"},
		{"outrageusement", "outrag"},
		{"outrait", "outr"},
		{"outrance", "outranc"},
		{"outrant", "outr"},
		{"outre", "outr"},
		{"outré", "outr"},
		{"outrée", "outr"},
		{"outrepasser", "outrepass"},
		{"outrez", "outr"},
		{"ouvert", "ouvert"},
		{"ouverte", "ouvert"},
		{"ouvertement", "ouvert"},
		{"ouvertes", "ouvert"},
		{"ouverts", "ouvert"},
		{"ouverture", "ouvertur"},
		{"ouvrage", "ouvrag"},
		{"ouvragé", "ouvrag"},
		{"ouvrages", "ouvrag"},
		{"ouvraient", "ouvr"},
		{"ouvrait", "ouvr"},
		{"ouvrant", "ouvr"},
		{"ouvrante", "ouvr"},
		{"ouvre", "ouvr"},
		{"ouvrez", "ouvr"},
		{"ouvrier", "ouvri"},
		{"ouvrière", "ouvri"},
		{"ouvriers", "ouvri"},
		{"ouvrir", "ouvr"},
		{"ouvrira", "ouvr"},
		{"ouvrirait", "ouvr"},
		{"ouvrirent", "ouvr"},
		{"ouvrit", "ouvr"},
		{"ouvrît", "ouvr"},
		{"ovale", "oval"},
		{"oxydé", "oxyd"},
		{"oysterpuf", "oysterpuf"},
		{"oysters", "oyster"},
		{"p", "p"},
		{"pablo", "pablo"},
		{"pace", "pac"},
		{"pacha", "pach"},
		{"pacific", "pacific"},
		{"pacifique", "pacif"},
		{"pacifiques", "pacif"},
		{"packet", "packet"},
		{"padoue", "padou"},
		{"paganisme", "pagan"},
		{"pagato", "pagato"},
		{"page", "pag"},
		{"pages", "pag"},
		{"pagina", "pagin"},
		{"pagode", "pagod"},
		{"pagodes", "pagod"},
		{"paie", "pai"},
		{"paiement", "pai"},
		{"paiements", "pai"},
		{"païen", "païen"},
		{"paient", "paient"},
		{"paierai", "pai"},
		{"paierait", "pai"},
		{"paieras", "pai"},
		{"paieriez", "pai"},
		{"paillasse", "paill"},
		{"paillasses", "paill"},
		{"paille", "paill"},
		{"pain", "pain"},
		{"pains", "pain"},
		{"pair", "pair"},
		{"paire", "pair"},
		{"paires", "pair"},
		{"pairie", "pair"},
		{"pairs", "pair"},
		{"paisible", "paisibl"},
		{"paisiblement", "paisibl"},
		{"paix", "paix"},
		{"pal", "pal"},
		{"palais", "pal"},
		{"palanquin", "palanquin"},
		{"palanquins", "palanquin"},
		{"palanza", "palanz"},
		{"palazzeto", "palazzeto"},
		{"pale", "pal"},
		{"pâle", "pâl"},
		{"palefrenier", "palefreni"},
		{"pâles", "pâl"},
		{"palestine", "palestin"},
		{"paletot", "paletot"},
		{"pâleur", "pâleur"},
		{"pâli", "pâl"},
		{"palier", "pali"},
		{"pâlir", "pâl"},
		{"palissades", "palissad"},
		{"pâlissaient", "pâl"},
		{"pâlissant", "pâl"},
		{"pâlissantes", "pâl"},
		{"pâlissent", "pâl"},
		{"pâlit", "pâl"},
		{"palki", "palk"},
		{"palkigharis", "palkighar"},
		{"pall", "pall"},
		{"palla", "pall"},
		{"pallagi", "pallag"},
		{"pallida", "pallid"},
		{"palmiers", "palmi"},
		{"palpitait", "palpit"},
		{"palpitant", "palpit"},
		{"palpitante", "palpit"},
		{"palpitants", "palpit"},
		{"palpiter", "palpit"},
		{"pamphlet", "pamphlet"},
		{"pamphlets", "pamphlet"},
		{"pan", "pan"},
		{"panache", "panach"},
		{"pança", "panc"},
		{"panier", "pani"},
		{"panneaux", "panneau"},
		{"panorama", "panoram"},
		{"panoramique", "panoram"},
		{"pansa", "pans"},
		{"pansaient", "pans"},
		{"pansement", "pans"},
		{"panser", "pans"},
		{"pansez", "pans"},
		{"pantalon", "pantalon"},
		{"pantalons", "pantalon"},
		{"panthères", "panther"},
		{"pantoufles", "pantoufl"},
		{"paolina", "paolin"},
		{"papa", "pap"},
		{"pape", "pap"},
		{"paperasses", "paper"},
		{"paperassière", "paperassi"},
		{"papier", "papi"},
		{"papiers", "papi"},
		{"papillon", "papillon"},
		{"papillons", "papillon"},
		{"papillotes", "papillot"},
		{"papouas", "papou"},
		{"papyrus", "papyrus"},
		{"paquebot", "paquebot"},
		{"paquebots", "paquebot"},
		{"paquet", "paquet"},
		{"paquetées", "paquet"},
		{"paquets", "paquet"},
		{"par", "par"},
		{"para", "par"},
		{"parade", "parad"},
		{"parader", "parad"},
		{"paradis", "parad"},
		{"parages", "parag"},
		{"paragraphe", "paragraph"},
		{"parais", "par"},
		{"paraissaient", "paraiss"},
		{"paraissais", "paraiss"},
		{"paraissait", "paraiss"},
		{"paraisse", "paraiss"},
		{"paraissent", "paraissent"},
		{"parait", "par"},
		{"paraît", "paraît"},
		{"paraîtra", "paraîtr"},
		{"paraîtrai", "paraîtr"},
		{"paraîtraient", "paraîtr"},
		{"paraître", "paraîtr"},
		{"paraîtrons", "paraîtron"},
		{"parallèle", "parallel"},
		{"parallèlement", "parallel"},
		{"paralysaient", "paralys"},
		{"paralyse", "paralys"},
		{"paralysé", "paralys"},
		{"paralyser", "paralys"},
		{"paralytique", "paralyt"},
		{"parant", "par"},
		{"parapet", "parapet"},
		{"paraphait", "paraph"},
		{"paraphe", "paraph"},
		{"paraphrasa", "paraphras"},
		{"parapluie", "paraplui"},
		{"paratonnerre", "paratonnerr"},
		{"paratonnerres", "paratonnerr"},
		{"paravent", "paravent"},
		{"parbleu", "parbleu"},
		{"parc", "parc"},
		{"parce", "parc"},
		{"parchemin", "parchemin"},
		{"parcouraient", "parcour"},
		{"parcourait", "parcour"},
		{"parcourant", "parcour"},
		{"parcourent", "parcourent"},
		{"parcourir", "parcour"},
		{"parcourrai", "parcourr"},
		{"parcours", "parcour"},
		{"parcourt", "parcourt"},
		{"parcouru", "parcouru"},
		{"parcourue", "parcouru"},
		{"parcoururent", "parcoururent"},
		{"parcourus", "parcourus"},
		{"parcourut", "parcourut"},
		{"pardessus", "pardessus"},
		{"pardi", "pard"},
		{"pardieu", "pardieu"},
		{"pardon", "pardon"},
		{"pardonna", "pardon"},
		{"pardonnable", "pardon"},
		{"pardonnables", "pardon"},
		{"pardonnait", "pardon"},
		{"pardonne", "pardon"},
		{"pardonné", "pardon"},
		{"pardonnées", "pardon"},
		{"pardonnent", "pardonnent"},
		{"pardonner", "pardon"},
		{"pardonnera", "pardon"},
		{"pardonnerai", "pardon"},
		{"pardonnerais", "pardon"},
		{"pardonnerait", "pardon"},
		{"pardonnerez", "pardon"},
		{"pardonnes", "pardon"},
		{"pardonnez", "pardon"},
		{"pardonniez", "pardon"},
		{"pardons", "pardon"},
		{"pare", "par"},
		{"paré", "par"},
		{"parée", "par"},
		{"pareil", "pareil"},
		{"pareille", "pareil"},
		{"pareilles", "pareil"},
		{"pareils", "pareil"},
		{"parent", "parent"},
		{"parente", "parent"},
		{"parentes", "parent"},
		{"parenthèse", "parenthes"},
		{"parents", "parent"},
		{"parer", "par"},
		{"paresse", "paress"},
		{"paresseuse", "paress"},
		{"paresseux", "paress"},
		{"parfait", "parf"},
		{"parfaite", "parfait"},
		{"parfaitement", "parfait"},
		{"parfaites", "parfait"},
		{"parfaits", "parfait"},
		{"parfois", "parfois"},
		{"parfum", "parfum"},
		{"parfumée", "parfum"},
		{"parfums", "parfum"},
		{"pari", "pari"},
		{"paria", "pari"},
		{"parie", "pari"},
		{"parié", "pari"},
		{"parient", "parient"},
		{"parier", "pari"},
		{"parierais", "pari"},
		{"parièrent", "pari"},
		{"pariétaires", "pariétair"},
		{"parieurs", "parieur"},
		{"paris", "paris"},
		{"parisien", "parisien"},
		{"parisienne", "parisien"},
		{"parisiennes", "parisien"},
		{"parisiens", "parisien"},
		{"parla", "parl"},
		{"parlaient", "parl"},
		{"parlais", "parl"},
		{"parlait", "parl"},
		{"parlant", "parl"},
		{"parlantes", "parl"},
		{"parlants", "parl"},
		{"parlassent", "parl"},
		{"parlât", "parl"},
		{"parle", "parl"},
		{"parlé", "parl"},
		{"parlement", "parl"},
		{"parlent", "parlent"},
		{"parler", "parl"},
		{"parlera", "parl"},
		{"parlerai", "parl"},
		{"parlerais", "parl"},
		{"parlerait", "parl"},
		{"parlèrent", "parl"},
		{"parlerez", "parl"},
		{"parlerons", "parl"},
		{"parleront", "parl"},
		{"parles", "parl"},
		{"parleur", "parleur"},
		{"parleurs", "parleur"},
		{"parlez", "parl"},
		{"parliez", "parl"},
		{"parlions", "parlion"},
		{"parloir", "parloir"},
		{"parlons", "parlon"},
		{"parma", "parm"},
		{"parme", "parm"},
		{"parmesan", "parmesan"},
		{"parmesans", "parmesan"},
		{"parmi", "parm"},
		{"parois", "parois"},
		{"paroisse", "paroiss"},
		{"paroisses", "paroiss"},
		{"paroissiale", "paroissial"},
		{"paroissiens", "paroissien"},
		{"parole", "parol"},
		{"paroles", "parol"},
		{"paroxysme", "paroxysm"},
		{"parquet", "parquet"},
		{"parquetée", "parquet"},
		{"parrain", "parrain"},
		{"pars", "par"},
		{"parsemée", "parsem"},
		{"parsi", "pars"},
		{"parsie", "pars"},
		{"parsis", "pars"},
		{"part", "part"},
		{"partage", "partag"},
		{"partagé", "partag"},
		{"partagea", "partag"},
		{"partageait", "partag"},
		{"partagée", "partag"},
		{"partager", "partag"},
		{"partagerai", "partag"},
		{"partagerais", "partag"},
		{"partaient", "part"},
		{"partait", "part"},
		{"partana", "partan"},
		{"partance", "partanc"},
		{"partant", "part"},
		{"parte", "part"},
		{"partenaire", "partenair"},
		{"partenaires", "partenair"},
		{"partent", "partent"},
		{"parterre", "parterr"},
		{"parterres", "parterr"},
		{"partes", "part"},
		{"partez", "part"},
		{"parti", "part"},
		{"particularité", "particular"},
		{"particularités", "particular"},
		{"particulier", "particuli"},
		{"particulière", "particuli"},
		{"particulièrement", "particuli"},
		{"particuliers", "particuli"},
		{"partie", "part"},
		{"parties", "part"},
		{"partiez", "part"},
		{"partir", "part"},
		{"partira", "part"},
		{"partirai", "part"},
		{"partirais", "part"},
		{"partirait", "part"},
		{"partirent", "part"},
		{"partirez", "part"},
		{"partirons", "part"},
		{"partis", "part"},
		{"partisan", "partisan"},
		{"partisans", "partisan"},
		{"partit", "part"},
		{"partît", "part"},
		{"partner", "partn"},
		{"partons", "parton"},
		{"partout", "partout"},
		{"parts", "part"},
		{"paru", "paru"},
		{"parurent", "parurent"},
		{"parut", "parut"},
		{"parût", "parût"},
		{"parvenaient", "parven"},
		{"parvenait", "parven"},
		{"parvenir", "parven"},
		{"parvenons", "parvenon"},
		{"parvenu", "parvenu"},
		{"parvenue", "parvenu"},
		{"parviendrai", "parviendr"},
		{"parviendraient", "parviendr"},
		{"parviendrait", "parviendr"},
		{"parviendras", "parviendr"},
		{"parviendrons", "parviendron"},
		{"parviendront", "parviendront"},
		{"parviens", "parvien"},
		{"parvient", "parvient"},
		{"parvinrent", "parvinrent"},
		{"parvint", "parvint"},
		{"pas", "pas"},
		{"pascal", "pascal"},
		{"pass", "pass"},
		{"passa", "pass"},
		{"passable", "passabl"},
		{"passablement", "passabl"},
		{"passables", "passabl"},
		{"passage", "passag"},
		{"passager", "passag"},
		{"passagère", "passager"},
		{"passagères", "passager"},
		{"passagers", "passager"},
		{"passages", "passag"},
		{"passaient", "pass"},
		{"passais", "pass"},
		{"passait", "pass"},
		{"passant", "pass"},
		{"passants", "pass"},
		{"passar", "passar"},
		{"passe", "pass"},
		{"passé", "pass"},
		{"passée", "pass"},
		{"passées", "pass"},
		{"passementier", "passementi"},
		{"passent", "passent"},
		{"passepartout", "passepartout"},
		{"passeport", "passeport"},
		{"passeports", "passeport"},
		{"passer", "pass"},
		{"passera", "pass"},
		{"passerai", "pass"},
		{"passerais", "pass"},
		{"passerait", "pass"},
		{"passerelle", "passerel"},
		{"passerelles", "passerel"},
		{"passèrent", "pass"},
		{"passerons", "pass"},
		{"passeront", "pass"},
		{"passes", "pass"},
		{"passés", "pass"},
		{"passez", "pass"},
		{"passif", "passif"},
		{"passion", "passion"},
		{"passionnait", "passion"},
		{"passionnant", "passion"},
		{"passionne", "passion"},
		{"passionné", "passion"},
		{"passionnée", "passion"},
		{"passionnées", "passion"},
		{"passionnément", "passion"},
		{"passionner", "passion"},
		{"passionnés", "passion"},
		{"passions", "passion"},
		{"passive", "passiv"},
		{"passons", "passon"},
		{"pastoral", "pastoral"},
		{"pastorale", "pastoral"},
		{"pataugeant", "pataug"},
		{"pâté", "pât"},
		{"patelin", "patelin"},
		{"patente", "patent"},
		{"pater", "pat"},
		{"paterne", "patern"},
		{"paternel", "paternel"},
		{"paternelle", "paternel"},
		{"pathétique", "pathet"},
		{"pathétiques", "pathet"},
		{"pathos", "pathos"},
		{"patiemment", "patient"},
		{"patience", "patienc"},
		{"patinant", "patin"},
		{"patna", "patn"},
		{"patois", "patois"},
		{"patrician", "patrician"},
		{"patricien", "patricien"},
		{"patricienne", "patricien"},
		{"patriciens", "patricien"},
		{"patrie", "patr"},
		{"patriotes", "patriot"},
		{"patriotisme", "patriot"},
		{"patron", "patron"},
		{"patronage", "patronag"},
		{"patronne", "patron"},
		{"patronnes", "patron"},
		{"patrouille", "patrouill"},
		{"patte", "patt"},
		{"pâture", "pâtur"},
		{"pauca", "pauc"},
		{"paul", "paul"},
		{"paupière", "paupi"},
		{"paupières", "paupi"},
		{"pauvre", "pauvr"},
		{"pauvrement", "pauvr"},
		{"pauvres", "pauvr"},
		{"pauvreté", "pauvret"},
		{"pauwell", "pauwel"},
		{"pavé", "pav"},
		{"pavée", "pav"},
		{"pavie", "pav"},
		{"pavillon", "pavillon"},
		{"pavillons", "pavillon"},
		{"pavois", "pavois"},
		{"pawnies", "pawn"},
		{"paya", "pai"},
		{"payables", "payabl"},
		{"payait", "pai"},
		{"payant", "pai"},
		{"paye", "pay"},
		{"payé", "pai"},
		{"payée", "pai"},
		{"payent", "payent"},
		{"payer", "pai"},
		{"payera", "pai"},
		{"payés", "pai"},
		{"payiez", "pai"},
		{"payons", "payon"},
		{"pays", "pay"},
		{"paysage", "paysag"},
		{"paysages", "paysag"},
		{"paysan", "paysan"},
		{"paysanne", "paysann"},
		{"paysans", "paysan"},
		{"peak", "peak"},
		{"peau", "peau"},
		{"peccadille", "peccadill"},
		{"peccavi", "peccav"},
		{"péchais", "pech"},
		{"pêche", "pêch"},
		{"péché", "pech"},
		{"pécher", "pech"},
		{"péchés", "pech"},
		{"pécheur", "pécheur"},
		{"pêcheurs", "pêcheur"},
		{"pectorale", "pectoral"},
		{"pécule", "pécul"},
		{"pécuniaire", "pécuniair"},
		{"pédant", "ped"},
		{"pédanterie", "pédanter"},
		{"pédantesque", "pédantesqu"},
		{"pedroti", "pedrot"},
		{"peignaient", "peign"},
		{"peignait", "peign"},
		{"peignant", "peign"},
		{"peigné", "peign"},
		{"peignez", "peign"},
		{"peignit", "peign"},
		{"peindra", "peindr"},
		{"peindre", "peindr"},
		{"peine", "pein"},
		{"peiné", "pein"},
		{"peines", "pein"},
		{"peinés", "pein"},
		{"peint", "peint"},
		{"peinte", "peint"},
		{"peintes", "peint"},
		{"peintre", "peintr"},
		{"peintres", "peintr"},
		{"peints", "peint"},
		{"peinture", "peintur"},
		{"peintures", "peintur"},
		{"pékin", "pékin"},
		{"pêle", "pêl"},
		{"pelegrino", "pelegrino"},
		{"pèlerinage", "pèlerinag"},
		{"pèlerinages", "pèlerinag"},
		{"pèlerine", "pèlerin"},
		{"pèlerins", "pèlerin"},
		{"pélicans", "pélican"},
		{"pelisse", "pel"},
		{"pelisses", "pel"},
		{"pelletées", "pellet"},
		{"pellico", "pellico"},
		{"peloton", "peloton"},
		{"pénal", "pénal"},
		{"penaud", "penaud"},
		{"penauds", "penaud"},
		{"pence", "penc"},
		{"pencha", "pench"},
		{"penchaient", "pench"},
		{"penchant", "pench"},
		{"penché", "pench"},
		{"pencher", "pench"},
		{"pend", "pend"},
		{"pendable", "pendabl"},
		{"pendables", "pendabl"},
		{"pendaient", "pend"},
		{"pendait", "pend"},
		{"pendant", "pend"},
		{"pendante", "pend"},
		{"pendantes", "pend"},
		{"pendre", "pendr"},
		{"pendu", "pendu"},
		{"pendue", "pendu"},
		{"pendule", "pendul"},
		{"pendules", "pendul"},
		{"pendus", "pendus"},
		{"pénétra", "pénetr"},
		{"pénétrait", "pénetr"},
		{"pénétrant", "pénetr"},
		{"pénétration", "pénetr"},
		{"pénètre", "pénetr"},
		{"pénétré", "pénetr"},
		{"pénétrée", "pénetr"},
		{"pénétrer", "pénetr"},
		{"pénétrera", "pénetr"},
		{"pénétrèrent", "pénetr"},
		{"pénible", "pénibl"},
		{"péniblement", "pénibl"},
		{"pénibles", "pénibl"},
		{"péninsulaire", "péninsulair"},
		{"peninsular", "peninsular"},
		{"péninsule", "péninsul"},
		{"pénitence", "pénitent"},
		{"pennsylvanie", "pennsylvan"},
		{"penny", "penny"},
		{"pensa", "pens"},
		{"pensai", "pens"},
		{"pensaient", "pens"},
		{"pensais", "pens"},
		{"pensait", "pens"},
		{"pensant", "pens"},
		{"pensante", "pens"},
		{"pensants", "pens"},
		{"pensât", "pens"},
		{"pense", "pens"},
		{"pensé", "pens"},
		{"pensée", "pens"},
		{"pensées", "pens"},
		{"pensent", "pensent"},
		{"penser", "pens"},
		{"pensera", "pens"},
		{"penserais", "pens"},
		{"penserait", "pens"},
		{"pensèrent", "pens"},
		{"penseur", "penseur"},
		{"pensez", "pens"},
		{"pensif", "pensif"},
		{"pension", "pension"},
		{"pensionnaire", "pensionnair"},
		{"pensive", "pensiv"},
		{"pensons", "penson"},
		{"pentagone", "pentagon"},
		{"pentateuque", "pentateuqu"},
		{"pente", "pent"},
		{"pentes", "pent"},
		{"pénurie", "pénur"},
		{"pépé", "pep"},
		{"pépites", "pépit"},
		{"pequigny", "pequigny"},
		{"per", "per"},
		{"perçaient", "perc"},
		{"perçait", "perc"},
		{"perçant", "perc"},
		{"perçante", "perc"},
		{"perce", "perc"},
		{"percé", "perc"},
		{"percepteur", "percepteur"},
		{"percer", "perc"},
		{"percèrent", "perc"},
		{"perche", "perch"},
		{"perché", "perch"},
		{"percussion", "percuss"},
		{"perd", "perd"},
		{"perdaient", "perd"},
		{"perdais", "perd"},
		{"perdait", "perd"},
		{"perdant", "perd"},
		{"perde", "perd"},
		{"perdent", "perdent"},
		{"perdez", "perd"},
		{"perdirent", "perd"},
		{"perdit", "perd"},
		{"perdition", "perdit"},
		{"perdons", "perdon"},
		{"perdra", "perdr"},
		{"perdrai", "perdr"},
		{"perdrais", "perdr"},
		{"perdrait", "perdr"},
		{"perdras", "perdr"},
		{"perdre", "perdr"},
		{"perdreau", "perdreau"},
		{"perdriez", "perdr"},
		{"perdrix", "perdrix"},
		{"perds", "perd"},
		{"perdu", "perdu"},
		{"perdue", "perdu"},
		{"perdus", "perdus"},
		{"père", "per"},
		{"pereire", "pereir"},
		{"pères", "per"},
		{"perfection", "perfect"},
		{"perfectionner", "perfection"},
		{"pergolèse", "pergoles"},
		{"péri", "per"},
		{"péril", "péril"},
		{"périlleux", "périll"},
		{"périls", "péril"},
		{"périmètre", "périmetr"},
		{"période", "périod"},
		{"périodique", "périod"},
		{"périr", "per"},
		{"périraient", "per"},
		{"péris", "per"},
		{"périsse", "per"},
		{"périssent", "per"},
		{"périssez", "per"},
		{"péristyle", "péristyl"},
		{"péristyles", "péristyl"},
		{"périt", "per"},
		{"perles", "perl"},
		{"permanence", "permanent"},
		{"permanente", "permanent"},
		{"permet", "permet"},
		{"permets", "permet"},
		{"permettaient", "permet"},
		{"permettait", "permet"},
		{"permettant", "permet"},
		{"permette", "permet"},
		{"permettent", "permettent"},
		{"permettez", "permet"},
		{"permettra", "permettr"},
		{"permettrai", "permettr"},
		{"permettrait", "permettr"},
		{"permettre", "permettr"},
		{"permettrez", "permettr"},
		{"permettront", "permettront"},
		{"permirent", "perm"},
		{"permis", "perm"},
		{"permise", "permis"},
		{"permission", "permiss"},
		{"permit", "perm"},
		{"permît", "perm"},
		{"pernice", "pernic"},
		{"pérorait", "péror"},
		{"péroré", "péror"},
		{"pérou", "pérou"},
		{"pérouse", "pérous"},
		{"perpendiculairement", "perpendiculair"},
		{"perpétuelle", "perpétuel"},
		{"perpétuels", "perpétuel"},
		{"perplexités", "perplex"},
		{"perqué", "perqu"},
		{"perquisition", "perquisit"},
		{"perron", "perron"},
		{"perruque", "perruqu"},
		{"perruques", "perruqu"},
		{"persans", "persan"},
		{"persécutait", "persécut"},
		{"persécute", "persécut"},
		{"persécuté", "persécut"},
		{"persécuterait", "persécut"},
		{"persécutés", "persécut"},
		{"persécutions", "persécu"},
		{"perses", "pers"},
		{"persienne", "persien"},
		{"persiennes", "persien"},
		{"persiflage", "persiflag"},
		{"persiflèrent", "persifl"},
		{"persista", "persist"},
		{"persistance", "persist"},
		{"personnage", "personnag"},
		{"personnages", "personnag"},
		{"personnalité", "personnal"},
		{"personne", "person"},
		{"personnel", "personnel"},
		{"personnelle", "personnel"},
		{"personnellement", "personnel"},
		{"personnelles", "personnel"},
		{"personnels", "personnel"},
		{"personnes", "person"},
		{"personnification", "personnif"},
		{"personnifiée", "personnifi"},
		{"perspective", "perspect"},
		{"perspectives", "perspect"},
		{"perspicacité", "perspicac"},
		{"persuadé", "persuad"},
		{"persuadée", "persuad"},
		{"persuader", "persuad"},
		{"persuasion", "persuas"},
		{"perte", "pert"},
		{"pertes", "pert"},
		{"pertuis", "pertuis"},
		{"perturbations", "perturb"},
		{"péruvien", "péruvien"},
		{"pervers", "perver"},
		{"perversité", "pervers"},
		{"pervertisse", "pervert"},
		{"pesait", "pes"},
		{"pesant", "pes"},
		{"pesante", "pes"},
		{"pesanteur", "pesanteur"},
		{"pesants", "pes"},
		{"pèse", "pes"},
		{"pesé", "pes"},
		{"peser", "pes"},
		{"pesés", "pes"},
		{"pesez", "pes"},
		{"pessimisme", "pessim"},
		{"peste", "pest"},
		{"pester", "pest"},
		{"pétaudière", "pétaudi"},
		{"pétillante", "pétill"},
		{"petit", "pet"},
		{"petite", "petit"},
		{"petites", "petit"},
		{"petitesse", "petitess"},
		{"petitesses", "petitess"},
		{"pétition", "pétit"},
		{"pétitions", "pétit"},
		{"petits", "petit"},
		{"pétrarque", "pétrarqu"},
		{"pétrifié", "pétrifi"},
		{"pétrifiée", "pétrifi"},
		{"pétrole", "pétrol"},
		{"pétrone", "pétron"},
		{"petto", "petto"},
		{"pétulance", "pétul"},
		{"peu", "peu"},
		{"peuple", "peupl"},
		{"peuplé", "peupl"},
		{"peuplée", "peupl"},
		{"peuplent", "peuplent"},
		{"peuples", "peupl"},
		{"peuplés", "peupl"},
		{"peuplier", "peupli"},
		{"peur", "peur"},
		{"peureux", "peureux"},
		{"peurs", "peur"},
		{"peut", "peut"},
		{"peuvent", "peuvent"},
		{"peux", "peux"},
		{"pharsale", "pharsal"},
		{"pherson", "pherson"},
		{"phileas", "phil"},
		{"philippe", "philipp"},
		{"philosophant", "philosoph"},
		{"philosophe", "philosoph"},
		{"philosophes", "philosoph"},
		{"philosophie", "philosoph"},
		{"philosophique", "philosoph"},
		{"philosophiquement", "philosoph"},
		{"philosophiques", "philosoph"},
		{"phosphorescentes", "phosphorescent"},
		{"photographie", "photograph"},
		{"phrase", "phras"},
		{"phrases", "phras"},
		{"physiognomoniques", "physiognomon"},
		{"physionomie", "physionom"},
		{"physionomies", "physionom"},
		{"physionomistes", "physionom"},
		{"physique", "physiqu"},
		{"physiquement", "physiqu"},
		{"physiques", "physiqu"},
		{"piacere", "piacer"},
		{"pianiste", "pianist"},
		{"piano", "piano"},
		{"pianos", "pianos"},
		{"pic", "pic"},
		{"pichegru", "pichegru"},
		{"pics", "pic"},
		{"pie", "pi"},
		{"pièce", "piec"},
		{"pièces", "piec"},
		{"piécettes", "piécet"},
		{"pied", "pied"},
		{"pieds", "pied"},
		{"piège", "pieg"},
		{"pieges", "pieg"},
		{"pièges", "pieg"},
		{"piémont", "piémont"},
		{"piémontais", "piémont"},
		{"piémontaise", "piémontais"},
		{"pier", "pi"},
		{"pierre", "pierr"},
		{"pierres", "pierr"},
		{"piété", "piet"},
		{"piéton", "piéton"},
		{"piétons", "piéton"},
		{"pietragrua", "pietragru"},
		{"pietranera", "pietran"},
		{"piètre", "pietr"},
		{"pieuse", "pieus"},
		{"pieusement", "pieus"},
		{"pieuses", "pieus"},
		{"pieux", "pieux"},
		{"pigeon", "pigeon"},
		{"pigeons", "pigeon"},
		{"pilastre", "pilastr"},
		{"pile", "pil"},
		{"pilé", "pil"},
		{"piler", "pil"},
		{"pilier", "pili"},
		{"piliers", "pili"},
		{"pillage", "pillag"},
		{"pillages", "pillag"},
		{"pillaient", "pill"},
		{"pillaji", "pillaj"},
		{"pille", "pill"},
		{"pillé", "pill"},
		{"pillée", "pill"},
		{"pillent", "pillent"},
		{"pillés", "pill"},
		{"pilori", "pilor"},
		{"pilote", "pilot"},
		{"pilotes", "pilot"},
		{"pimentel", "pimentel"},
		{"pinceau", "pinceau"},
		{"pincée", "pinc"},
		{"pincette", "pincet"},
		{"pintes", "pint"},
		{"piochant", "pioch"},
		{"pioneer", "pione"},
		{"pipe", "pip"},
		{"pipes", "pip"},
		{"piqua", "piqu"},
		{"piquait", "piqu"},
		{"piquant", "piqu"},
		{"piquante", "piqu"},
		{"piquantes", "piqu"},
		{"pique", "piqu"},
		{"piqué", "piqu"},
		{"piquée", "piqu"},
		{"piquer", "piqu"},
		{"piquerait", "piqu"},
		{"piqués", "piqu"},
		{"piquez", "piqu"},
		{"pirard", "pirard"},
		{"pirate", "pirat"},
		{"pire", "pir"},
		{"pis", "pis"},
		{"piste", "pist"},
		{"pistolet", "pistolet"},
		{"pistolets", "pistolet"},
		{"piston", "piston"},
		{"pistons", "piston"},
		{"pitance", "pitanc"},
		{"piteuse", "piteux"},
		{"piteusement", "piteux"},
		{"piteux", "piteux"},
		{"pitié", "piti"},
		{"pitoyable", "pitoi"},
		{"pitt", "pitt"},
		{"pittoresque", "pittoresqu"},
		{"pittoresques", "pittoresqu"},
		{"pittsburg", "pittsburg"},
		{"più", "più"},
		{"plaça", "plac"},
		{"plaçait", "plac"},
		{"plaçant", "plac"},
		{"placard", "placard"},
		{"place", "plac"},
		{"placé", "plac"},
		{"placée", "plac"},
		{"placées", "plac"},
		{"placer", "plac"},
		{"placera", "plac"},
		{"placerai", "plac"},
		{"placerait", "plac"},
		{"placèrent", "plac"},
		{"placerez", "plac"},
		{"placers", "placer"},
		{"places", "plac"},
		{"placés", "plac"},
		{"placez", "plac"},
		{"placidité", "placid"},
		{"plafond", "plafond"},
		{"plafonds", "plafond"},
		{"plaida", "plaid"},
		{"plaidait", "plaid"},
		{"plaider", "plaid"},
		{"plaidoirie", "plaidoir"},
		{"plaie", "plai"},
		{"plaignait", "plaign"},
		{"plaignant", "plaign"},
		{"plaignants", "plaign"},
		{"plaignez", "plaign"},
		{"plaignirent", "plaign"},
		{"plaignit", "plaign"},
		{"plain", "plain"},
		{"plaindrai", "plaindr"},
		{"plaindrais", "plaindr"},
		{"plaindre", "plaindr"},
		{"plaine", "plain"},
		{"plaines", "plain"},
		{"plains", "plain"},
		{"plaint", "plaint"},
		{"plainte", "plaint"},
		{"plaintes", "plaint"},
		{"plaintive", "plaintiv"},
		{"plaints", "plaint"},
		{"plaira", "plair"},
		{"plairait", "plair"},
		{"plaire", "plair"},
		{"plais", "plais"},
		{"plaisaient", "plais"},
		{"plaisait", "plais"},
		{"plaisamment", "plais"},
		{"plaisance", "plaisanc"},
		{"plaisant", "plais"},
		{"plaisanta", "plaisant"},
		{"plaisantaient", "plaisant"},
		{"plaisantait", "plaisant"},
		{"plaisantant", "plaisant"},
		{"plaisantât", "plaisant"},
		{"plaisante", "plais"},
		{"plaisanté", "plaisant"},
		{"plaisantent", "plaisantent"},
		{"plaisanter", "plaisant"},
		{"plaisanterie", "plaisanter"},
		{"plaisanteries", "plaisanter"},
		{"plaisantes", "plais"},
		{"plaisants", "plais"},
		{"plaise", "plais"},
		{"plaisent", "plaisent"},
		{"plaisir", "plais"},
		{"plaisirs", "plaisir"},
		{"plaît", "plaît"},
		{"plan", "plan"},
		{"planait", "plan"},
		{"planche", "planch"},
		{"plancher", "planch"},
		{"planches", "planch"},
		{"planchette", "planchet"},
		{"planisphère", "planispher"},
		{"plans", "plan"},
		{"plant", "plant"},
		{"planta", "plant"},
		{"plantation", "plantat"},
		{"plantations", "plantat"},
		{"plante", "plant"},
		{"planté", "plant"},
		{"planter", "plant"},
		{"planterez", "plant"},
		{"planterons", "plant"},
		{"plantes", "plant"},
		{"plantés", "plant"},
		{"plantiez", "plant"},
		{"plaque", "plaqu"},
		{"plaques", "plaqu"},
		{"plastron", "plastron"},
		{"plat", "plat"},
		{"platanes", "platan"},
		{"plate", "plat"},
		{"plateau", "plateau"},
		{"platement", "plat"},
		{"plates", "plat"},
		{"platitude", "platitud"},
		{"platitudes", "platitud"},
		{"platonique", "platon"},
		{"plâtre", "plâtr"},
		{"plats", "plat"},
		{"platte", "platt"},
		{"plausible", "plausibl"},
		{"please", "pleas"},
		{"plébéien", "plébéien"},
		{"plébéiennes", "plébéien"},
		{"plébéiens", "plébéien"},
		{"plein", "plein"},
		{"pleine", "plein"},
		{"pleinement", "plein"},
		{"pleines", "plein"},
		{"pleins", "plein"},
		{"pleura", "pleur"},
		{"pleurait", "pleur"},
		{"pleurant", "pleur"},
		{"pleure", "pleur"},
		{"pleuré", "pleur"},
		{"pleurer", "pleur"},
		{"pleurera", "pleur"},
		{"pleurerait", "pleur"},
		{"pleurèrent", "pleur"},
		{"pleureur", "pleureur"},
		{"pleurez", "pleur"},
		{"pleurs", "pleur"},
		{"pleutre", "pleutr"},
		{"pleuvaient", "pleuv"},
		{"pleuvait", "pleuv"},
		{"pleuvent", "pleuvent"},
		{"pli", "pli"},
		{"plia", "pli"},
		{"pliaient", "pli"},
		{"plié", "pli"},
		{"pliée", "pli"},
		{"plier", "pli"},
		{"pliniana", "plinian"},
		{"plis", "plis"},
		{"plissa", "pliss"},
		{"plomb", "plomb"},
		{"plombées", "plomb"},
		{"plonge", "plong"},
		{"plongé", "plong"},
		{"plongea", "plong"},
		{"plongeaient", "plong"},
		{"plongeait", "plong"},
		{"plongeant", "plong"},
		{"plongée", "plong"},
		{"plonger", "plong"},
		{"plongerais", "plong"},
		{"plongés", "plong"},
		{"plours", "plour"},
		{"plu", "plu"},
		{"pluie", "plui"},
		{"pluies", "plui"},
		{"plum", "plum"},
		{"plume", "plum"},
		{"plumes", "plum"},
		{"plupart", "plupart"},
		{"plus", "plus"},
		{"plusieurs", "plusieur"},
		{"plut", "plut"},
		{"plût", "plût"},
		{"plutôt", "plutôt"},
		{"pô", "pô"},
		{"poche", "poch"},
		{"poches", "poch"},
		{"poco", "poco"},
		{"podestat", "podestat"},
		{"podestats", "podestat"},
		{"poêle", "poêl"},
		{"poème", "poem"},
		{"poemi", "poem"},
		{"poésie", "poes"},
		{"poésies", "poes"},
		{"poète", "poet"},
		{"poètes", "poet"},
		{"poétique", "poétiqu"},
		{"poids", "poid"},
		{"poignante", "poign"},
		{"poignantes", "poign"},
		{"poignard", "poignard"},
		{"poignarda", "poignard"},
		{"poignardé", "poignard"},
		{"poignardée", "poignard"},
		{"poignarder", "poignard"},
		{"poignards", "poignard"},
		{"poigne", "poign"},
		{"poignée", "poign"},
		{"poignées", "poign"},
		{"poils", "poil"},
		{"poindre", "poindr"},
		{"poing", "poing"},
		{"poings", "poing"},
		{"point", "point"},
		{"pointe", "point"},
		{"pointes", "point"},
		{"points", "point"},
		{"pointus", "pointus"},
		{"poireaux", "poireau"},
		{"poison", "poison"},
		{"poisons", "poison"},
		{"poisson", "poisson"},
		{"poissonnerie", "poissonner"},
		{"poissons", "poisson"},
		{"poissy", "poissy"},
		{"poitrail", "poitrail"},
		{"poitrine", "poitrin"},
		{"poivriers", "poivri"},
		{"pole", "pol"},
		{"poli", "pol"},
		{"police", "polic"},
		{"policeman", "policeman"},
		{"policemen", "policemen"},
		{"polices", "polic"},
		{"polichinelle", "polichinel"},
		{"polidori", "polidor"},
		{"polie", "pol"},
		{"polies", "pol"},
		{"poliment", "pol"},
		{"polis", "pol"},
		{"polish", "polish"},
		{"polisson", "polisson"},
		{"politesse", "politess"},
		{"politesses", "politess"},
		{"politique", "polit"},
		{"politiques", "polit"},
		{"polonais", "polon"},
		{"polyeucte", "polyeuct"},
		{"polygames", "polygam"},
		{"polygamie", "polygam"},
		{"polygones", "polygon"},
		{"pommades", "pommad"},
		{"pomme", "pomm"},
		{"pommeau", "pommeau"},
		{"pommelé", "pommel"},
		{"pommes", "pomm"},
		{"pommettes", "pommet"},
		{"pommiers", "pommi"},
		{"pompe", "pomp"},
		{"pompes", "pomp"},
		{"pompiers", "pompi"},
		{"pompon", "pompon"},
		{"ponceaux", "ponceau"},
		{"poncet", "poncet"},
		{"ponctualité", "ponctual"},
		{"ponctuel", "ponctuel"},
		{"ponctuelle", "ponctuel"},
		{"ponctuellement", "ponctuel"},
		{"pondéré", "ponder"},
		{"poney", "poney"},
		{"poneys", "poney"},
		{"pont", "pont"},
		{"pontarlier", "pontarli"},
		{"ponte", "pont"},
		{"ponton", "ponton"},
		{"ponts", "pont"},
		{"populace", "populac"},
		{"populaire", "populair"},
		{"population", "popul"},
		{"populations", "popul"},
		{"porcelaine", "porcelain"},
		{"porco", "porco"},
		{"porcs", "porc"},
		{"porphyre", "porphyr"},
		{"port", "port"},
		{"porta", "port"},
		{"portaient", "port"},
		{"portait", "port"},
		{"portant", "port"},
		{"portât", "port"},
		{"portative", "portat"},
		{"porte", "port"},
		{"porté", "port"},
		{"portée", "port"},
		{"portées", "port"},
		{"portefaix", "portefaix"},
		{"portefeuille", "portefeuill"},
		{"portefeuilles", "portefeuill"},
		{"portemanteau", "portemanteau"},
		{"portent", "portent"},
		{"porter", "port"},
		{"portera", "port"},
		{"porteras", "port"},
		{"portèrent", "port"},
		{"porterez", "port"},
		{"porteront", "port"},
		{"portes", "port"},
		{"portés", "port"},
		{"porteur", "porteur"},
		{"porteurs", "porteur"},
		{"portez", "port"},
		{"portier", "porti"},
		{"portière", "porti"},
		{"portières", "porti"},
		{"portiers", "porti"},
		{"portion", "portion"},
		{"portique", "portiqu"},
		{"porto", "porto"},
		{"portons", "porton"},
		{"portrait", "portr"},
		{"portraits", "portrait"},
		{"ports", "port"},
		{"portugais", "portug"},
		{"portugaise", "portugais"},
		{"posa", "pos"},
		{"posant", "pos"},
		{"posé", "pos"},
		{"posée", "pos"},
		{"posément", "pos"},
		{"poser", "pos"},
		{"posés", "pos"},
		{"positif", "posit"},
		{"position", "posit"},
		{"positions", "posit"},
		{"positive", "posit"},
		{"positivement", "posit"},
		{"possédait", "possed"},
		{"possédant", "possed"},
		{"possède", "possed"},
		{"possédé", "possed"},
		{"posséder", "possed"},
		{"possédés", "possed"},
		{"possesseur", "possesseur"},
		{"possession", "possess"},
		{"possessions", "possess"},
		{"possibilité", "possibil"},
		{"possibilités", "possibil"},
		{"possible", "possibl"},
		{"possibles", "possibl"},
		{"poste", "post"},
		{"posté", "post"},
		{"postérité", "poster"},
		{"postes", "post"},
		{"posthume", "posthum"},
		{"postillon", "postillon"},
		{"potage", "potag"},
		{"potager", "potag"},
		{"potagères", "potager"},
		{"potentats", "potentat"},
		{"pouce", "pouc"},
		{"pouces", "pouc"},
		{"poudrait", "poudr"},
		{"poudre", "poudr"},
		{"poudrées", "poudr"},
		{"poudrés", "poudr"},
		{"pouffer", "pouff"},
		{"poule", "poul"},
		{"poulet", "poulet"},
		{"poulets", "poulet"},
		{"poulies", "poul"},
		{"poumons", "poumon"},
		{"pounah", "pounah"},
		{"poupée", "poup"},
		{"poupées", "poup"},
		{"pour", "pour"},
		{"pourceaux", "pourceau"},
		{"pourparlers", "pourparler"},
		{"pourpensées", "pourpens"},
		{"pourpoint", "pourpoint"},
		{"pourpres", "pourpr"},
		{"pourquoi", "pourquoi"},
		{"pourra", "pourr"},
		{"pourrai", "pourr"},
		{"pourraient", "pourr"},
		{"pourrais", "pourr"},
		{"pourrait", "pourr"},
		{"pourras", "pourr"},
		{"pourrez", "pourr"},
		{"pourriez", "pourr"},
		{"pourrions", "pourrion"},
		{"pourriture", "pourritur"},
		{"pourrons", "pourron"},
		{"pourront", "pourront"},
		{"poursuit", "poursuit"},
		{"poursuite", "poursuit"},
		{"poursuites", "poursuit"},
		{"poursuivaient", "poursuiv"},
		{"poursuivait", "poursuiv"},
		{"poursuivant", "poursuiv"},
		{"poursuivez", "poursuiv"},
		{"poursuivi", "poursuiv"},
		{"poursuivie", "poursuiv"},
		{"poursuivis", "poursuiv"},
		{"poursuivit", "poursuiv"},
		{"poursuivrai", "poursuivr"},
		{"poursuivraient", "poursuivr"},
		{"poursuivrait", "poursuivr"},
		{"poursuivre", "poursuivr"},
		{"poursuivront", "poursuivront"},
		{"pourtant", "pourt"},
		{"pourvoir", "pourvoir"},
		{"pourvu", "pourvu"},
		{"poussa", "pouss"},
		{"poussaient", "pouss"},
		{"poussait", "pouss"},
		{"poussant", "pouss"},
		{"pousse", "pouss"},
		{"poussé", "pouss"},
		{"poussée", "pouss"},
		{"poussent", "poussent"},
		{"pousser", "pouss"},
		{"poussera", "pouss"},
		{"poussés", "pouss"},
		{"poussez", "pouss"},
		{"poussière", "poussi"},
		{"poutre", "poutr"},
		{"poutres", "poutr"},
		{"pouvaient", "pouv"},
		{"pouvais", "pouv"},
		{"pouvait", "pouv"},
		{"pouvant", "pouv"},
		{"pouvez", "pouv"},
		{"pouviez", "pouv"},
		{"pouvions", "pouvion"},
		{"pouvoir", "pouvoir"},
		{"pouvons", "pouvon"},
		{"pozzo", "pozzo"},
		{"pradt", "pradt"},
		{"prairie", "prair"},
		{"prairies", "prair"},
		{"praticable", "pratic"},
		{"pratiquais", "pratiqu"},
		{"pratique", "pratiqu"},
		{"pratiqué", "pratiqu"},
		{"pratiquée", "pratiqu"},
		{"pratiquées", "pratiqu"},
		{"pratiquer", "pratiqu"},
		{"pratiques", "pratiqu"},
		{"pré", "pré"},
		{"préalable", "préalabl"},
		{"préalablement", "préalabl"},
		{"préambule", "préambul"},
		{"précaution", "précaut"},
		{"précautions", "précaut"},
		{"précéda", "préced"},
		{"précédait", "préced"},
		{"précédé", "préced"},
		{"précédée", "préced"},
		{"précédent", "précédent"},
		{"précédente", "précédent"},
		{"précédents", "précédent"},
		{"précédèrent", "préced"},
		{"précédés", "préced"},
		{"précepte", "précept"},
		{"préceptes", "précept"},
		{"précepteur", "précepteur"},
		{"précepteurs", "précepteur"},
		{"prêcha", "prêch"},
		{"prêchaient", "prêch"},
		{"prêchait", "prêch"},
		{"prêchant", "prêch"},
		{"prêché", "prêch"},
		{"prêcher", "prêch"},
		{"prêchera", "prêch"},
		{"prêcherai", "prêch"},
		{"prêcherait", "prêch"},
		{"prêches", "prêch"},
		{"précieuse", "précieux"},
		{"précieusement", "précieux"},
		{"précieuses", "précieux"},
		{"précieux", "précieux"},
		{"précipices", "précipic"},
		{"précipita", "précipit"},
		{"précipitaient", "précipit"},
		{"précipitait", "précipit"},
		{"précipitamment", "précipit"},
		{"précipitant", "précipit"},
		{"précipitation", "précipit"},
		{"précipite", "précipit"},
		{"précipité", "précip"},
		{"précipitent", "précipitent"},
		{"précipiter", "précipit"},
		{"précipitèrent", "précipit"},
		{"précipités", "précip"},
		{"précis", "prec"},
		{"précise", "précis"},
		{"précisément", "précis"},
		{"précises", "précis"},
		{"précision", "précis"},
		{"précoces", "précoc"},
		{"précurseur", "précurseur"},
		{"précurseurs", "précurseur"},
		{"prédécesseur", "prédécesseur"},
		{"prédestiné", "prédestin"},
		{"prédestinée", "prédestin"},
		{"prédicateur", "prédiqu"},
		{"prédication", "prédiqu"},
		{"prédications", "prédiqu"},
		{"prédiction", "prédict"},
		{"prédictions", "prédict"},
		{"prédire", "prédir"},
		{"prédispose", "prédispos"},
		{"prédit", "pred"},
		{"préface", "préfac"},
		{"préfecture", "préfectur"},
		{"préféra", "préfer"},
		{"préférable", "préfer"},
		{"préférables", "préfer"},
		{"préférait", "préfer"},
		{"préfère", "préfer"},
		{"préféré", "préfer"},
		{"préférence", "préférent"},
		{"préférences", "préférent"},
		{"préférer", "préfer"},
		{"préférerez", "préfer"},
		{"préférez", "préfer"},
		{"préfériez", "préfer"},
		{"préfet", "préfet"},
		{"préfète", "préfet"},
		{"préfets", "préfet"},
		{"préjudice", "préjudic"},
		{"préjudiciable", "préjudici"},
		{"préjudicierait", "préjudici"},
		{"préjugé", "préjug"},
		{"préjugés", "préjug"},
		{"prélat", "prélat"},
		{"prélature", "prélatur"},
		{"prélevés", "prélev"},
		{"préliminaires", "préliminair"},
		{"préludé", "prélud"},
		{"prématurément", "prématur"},
		{"préméditation", "prémédit"},
		{"prémédité", "prémed"},
		{"premier", "premi"},
		{"première", "premi"},
		{"premièrement", "premi"},
		{"premières", "premi"},
		{"premiers", "premi"},
		{"prenaient", "pren"},
		{"prenais", "pren"},
		{"prenait", "pren"},
		{"prenant", "pren"},
		{"prend", "prend"},
		{"prendra", "prendr"},
		{"prendrai", "prendr"},
		{"prendrais", "prendr"},
		{"prendrait", "prendr"},
		{"prendras", "prendr"},
		{"prendre", "prendr"},
		{"prendrez", "prendr"},
		{"prendront", "prendront"},
		{"prends", "prend"},
		{"prenez", "pren"},
		{"prenne", "pren"},
		{"prennent", "prennent"},
		{"prenons", "prenon"},
		{"préoccupa", "préoccup"},
		{"préoccupait", "préoccup"},
		{"préoccupation", "préoccup"},
		{"préoccupations", "préoccup"},
		{"préoccupé", "préoccup"},
		{"préoccupée", "préoccup"},
		{"préoccuper", "préoccup"},
		{"prépara", "prépar"},
		{"préparaient", "prépar"},
		{"préparais", "prépar"},
		{"préparait", "prépar"},
		{"préparant", "prépar"},
		{"préparât", "prépar"},
		{"préparatifs", "prépar"},
		{"préparation", "prépar"},
		{"préparatoire", "préparatoir"},
		{"préparatoires", "préparatoir"},
		{"prépare", "prépar"},
		{"préparé", "prépar"},
		{"préparer", "prépar"},
		{"préparera", "prépar"},
		{"préparèrent", "prépar"},
		{"préparés", "prépar"},
		{"préparez", "prépar"},
		{"préposé", "prépos"},
		{"prepotenze", "prepotenz"},
		{"prérogative", "prérog"},
		{"près", "pres"},
		{"présage", "présag"},
		{"présageait", "présag"},
		{"présages", "présag"},
		{"presbytère", "presbyter"},
		{"prescrira", "prescr"},
		{"prescrire", "prescrir"},
		{"prescrit", "prescr"},
		{"prescrite", "prescrit"},
		{"prescrivaient", "prescriv"},
		{"prescrivait", "prescriv"},
		{"préséance", "préséanc"},
		{"préséances", "préséanc"},
		{"présence", "présenc"},
		{"présent", "présent"},
		{"présenta", "présent"},
		{"présentable", "présent"},
		{"présentaient", "présent"},
		{"présentait", "présent"},
		{"présentant", "présent"},
		{"présentation", "présent"},
		{"présentations", "présent"},
		{"présente", "présent"},
		{"présenté", "présent"},
		{"présentée", "présent"},
		{"présentées", "présent"},
		{"présentement", "présent"},
		{"présentent", "présentent"},
		{"présenter", "présent"},
		{"présentera", "présent"},
		{"présenterait", "présent"},
		{"présentèrent", "présent"},
		{"présenterez", "présent"},
		{"présentes", "présent"},
		{"présentés", "présent"},
		{"présentez", "présent"},
		{"présents", "présent"},
		{"préservant", "préserv"},
		{"préserve", "préserv"},
		{"préservé", "préserv"},
		{"préservée", "préserv"},
		{"préserver", "préserv"},
		{"présidait", "présid"},
		{"présidence", "président"},
		{"président", "président"},
		{"présidente", "président"},
		{"présidents", "président"},
		{"présider", "présid"},
		{"présomption", "présompt"},
		{"présomptions", "présompt"},
		{"presqu", "presqu"},
		{"presque", "presqu"},
		{"pressa", "press"},
		{"pressaient", "press"},
		{"pressais", "press"},
		{"pressait", "press"},
		{"pressant", "press"},
		{"pressante", "press"},
		{"presse", "press"},
		{"pressé", "press"},
		{"pressée", "press"},
		{"pressent", "pressent"},
		{"pressentiment", "pressent"},
		{"pressentiments", "pressent"},
		{"pressentir", "pressent"},
		{"presser", "press"},
		{"pressés", "press"},
		{"pression", "pression"},
		{"pressoir", "pressoir"},
		{"pressure", "pressur"},
		{"prestige", "prestig"},
		{"prestigieuse", "prestigi"},
		{"prestigieux", "prestigi"},
		{"présume", "présum"},
		{"présumé", "présum"},
		{"prêt", "prêt"},
		{"prêta", "prêt"},
		{"prêtait", "prêt"},
		{"prêtant", "prêt"},
		{"prête", "prêt"},
		{"prêté", "prêt"},
		{"prêtée", "prêt"},
		{"prétend", "prétend"},
		{"prétendaient", "prétend"},
		{"prétendait", "prétend"},
		{"prétendant", "prétend"},
		{"prétendent", "prétendent"},
		{"prétendez", "prétend"},
		{"prétendit", "prétend"},
		{"prétendons", "prétendon"},
		{"prétendrait", "prétendr"},
		{"prétendre", "prétendr"},
		{"prétends", "prétend"},
		{"prétendu", "prétendu"},
		{"prétendue", "prétendu"},
		{"prétendues", "prétendu"},
		{"prétendus", "prétendus"},
		{"prétention", "prétent"},
		{"prétentions", "prétent"},
		{"prêter", "prêt"},
		{"prêtes", "prêt"},
		{"prêtés", "prêt"},
		{"prétexta", "prétext"},
		{"prétextant", "prétext"},
		{"prétexte", "prétext"},
		{"prétextes", "prétext"},
		{"prêtez", "prêt"},
		{"prétoire", "prétoir"},
		{"prêtre", "prêtr"},
		{"prêtres", "prêtr"},
		{"prêts", "prêt"},
		{"preuve", "preuv"},
		{"preuves", "preuv"},
		{"preux", "preux"},
		{"prévalant", "préval"},
		{"prévaloir", "prévaloir"},
		{"prévenait", "préven"},
		{"prévenances", "préven"},
		{"prévenez", "préven"},
		{"prévenir", "préven"},
		{"préventions", "prévent"},
		{"prévenu", "prévenu"},
		{"prévenus", "prévenus"},
		{"prévienne", "prévien"},
		{"préviens", "prévien"},
		{"prévision", "prévis"},
		{"prévisions", "prévis"},
		{"prévoir", "prévoir"},
		{"prévois", "prévois"},
		{"prévoit", "prévoit"},
		{"prévost", "prévost"},
		{"prévoyait", "prévoi"},
		{"prévoyant", "prévoi"},
		{"prévu", "prévu"},
		{"prévue", "prévu"},
		{"pria", "pri"},
		{"priais", "pri"},
		{"priait", "pri"},
		{"priant", "pri"},
		{"prie", "pri"},
		{"prié", "pri"},
		{"prier", "pri"},
		{"prierai", "pri"},
		{"prière", "prier"},
		{"prièrent", "pri"},
		{"prières", "prier"},
		{"prieur", "prieur"},
		{"priez", "pri"},
		{"prime", "prim"},
		{"primé", "prim"},
		{"primer", "prim"},
		{"primes", "prim"},
		{"primitif", "primit"},
		{"primitives", "primit"},
		{"primo", "primo"},
		{"prina", "prin"},
		{"prince", "princ"},
		{"princes", "princ"},
		{"princesse", "princess"},
		{"princesses", "princess"},
		{"principal", "principal"},
		{"principale", "principal"},
		{"principalement", "principal"},
		{"principales", "principal"},
		{"principaux", "principal"},
		{"principe", "princip"},
		{"principes", "princip"},
		{"printemps", "printemp"},
		{"prirent", "prirent"},
		{"pris", "pris"},
		{"prise", "pris"},
		{"prises", "pris"},
		{"prison", "prison"},
		{"prisonnier", "prisonni"},
		{"prisonnière", "prisonni"},
		{"prisonniers", "prisonni"},
		{"prisons", "prison"},
		{"prit", "prit"},
		{"prît", "prît"},
		{"priva", "priv"},
		{"privation", "privat"},
		{"prive", "priv"},
		{"privé", "priv"},
		{"privée", "priv"},
		{"privées", "priv"},
		{"priverai", "priv"},
		{"privés", "priv"},
		{"privilège", "privileg"},
		{"privilèges", "privileg"},
		{"privilégié", "privilégi"},
		{"privilégiée", "privilégi"},
		{"prix", "prix"},
		{"pro", "pro"},
		{"probabilité", "probabl"},
		{"probabilités", "probabl"},
		{"probable", "probabl"},
		{"probablement", "probabl"},
		{"probantes", "prob"},
		{"probe", "prob"},
		{"probes", "prob"},
		{"probité", "probit"},
		{"problème", "problem"},
		{"problèmes", "problem"},
		{"procédé", "proced"},
		{"procédés", "proced"},
		{"procédure", "procédur"},
		{"procès", "proces"},
		{"procession", "process"},
		{"processionnellement", "processionnel"},
		{"processions", "process"},
		{"prochain", "prochain"},
		{"prochaine", "prochain"},
		{"proche", "proch"},
		{"proches", "proch"},
		{"proclamation", "proclam"},
		{"proclamations", "proclam"},
		{"proclamée", "proclam"},
		{"proclamer", "proclam"},
		{"proctor", "proctor"},
		{"procuration", "procur"},
		{"procure", "procur"},
		{"procuré", "procur"},
		{"procurée", "procur"},
		{"procurer", "procur"},
		{"procurerai", "procur"},
		{"procureur", "procureur"},
		{"procureurs", "procureur"},
		{"prodige", "prodig"},
		{"prodiges", "prodig"},
		{"prodigieuse", "prodigi"},
		{"prodigieusement", "prodigi"},
		{"prodigieux", "prodigi"},
		{"prodiguait", "prodigu"},
		{"prodigue", "prodigu"},
		{"prodiguer", "prodigu"},
		{"prodiguera", "prodigu"},
		{"prodiguerait", "prodigu"},
		{"prodiguèrent", "prodigu"},
		{"production", "product"},
		{"produirait", "produir"},
		{"produire", "produir"},
		{"produis", "produis"},
		{"produisaient", "produis"},
		{"produisait", "produis"},
		{"produisant", "produis"},
		{"produise", "produis"},
		{"produisit", "produis"},
		{"produit", "produit"},
		{"produite", "produit"},
		{"produites", "produit"},
		{"produits", "produit"},
		{"profanateur", "profan"},
		{"profane", "profan"},
		{"profanes", "profan"},
		{"proférait", "profer"},
		{"proférant", "profer"},
		{"proférées", "profer"},
		{"proférer", "profer"},
		{"proférés", "profer"},
		{"professait", "profess"},
		{"professeur", "professeur"},
		{"professeurs", "professeur"},
		{"profession", "profess"},
		{"professions", "profess"},
		{"profil", "profil"},
		{"profilait", "profil"},
		{"profilant", "profil"},
		{"profilée", "profil"},
		{"profils", "profil"},
		{"profit", "prof"},
		{"profita", "profit"},
		{"profitable", "profit"},
		{"profitait", "profit"},
		{"profitant", "profit"},
		{"profité", "profit"},
		{"profiter", "profit"},
		{"profiterait", "profit"},
		{"profitons", "profiton"},
		{"profits", "profit"},
		{"profond", "profond"},
		{"profonde", "profond"},
		{"profondément", "profond"},
		{"profondes", "profond"},
		{"profondeur", "profondeur"},
		{"profondeurs", "profondeur"},
		{"profonds", "profond"},
		{"programme", "programm"},
		{"progrès", "progres"},
		{"prohibée", "prohib"},
		{"proie", "proi"},
		{"projectiles", "projectil"},
		{"projet", "projet"},
		{"projeté", "projet"},
		{"projetée", "projet"},
		{"projets", "projet"},
		{"prolonge", "prolong"},
		{"prolongé", "prolong"},
		{"prolongea", "prolong"},
		{"prolongeaient", "prolong"},
		{"prolongeant", "prolong"},
		{"prolongée", "prolong"},
		{"prolongent", "prolongent"},
		{"prolonger", "prolong"},
		{"prolongés", "prolong"},
		{"promena", "promen"},
		{"promenade", "promenad"},
		{"promenades", "promenad"},
		{"promenaient", "promen"},
		{"promenait", "promen"},
		{"promenant", "promen"},
		{"promène", "promen"},
		{"promené", "promen"},
		{"promènent", "promènent"},
		{"promener", "promen"},
		{"promenèrent", "promen"},
		{"promenés", "promen"},
		{"promeneur", "promeneur"},
		{"promeneurs", "promeneur"},
		{"promenez", "promen"},
		{"promenions", "promen"},
		{"promesse", "promess"},
		{"promesses", "promess"},
		{"promet", "promet"},
		{"promets", "promet"},
		{"promettait", "promet"},
		{"promette", "promet"},
		{"promettent", "promettent"},
		{"promettez", "promet"},
		{"promettons", "prometton"},
		{"promettre", "promettr"},
		{"promis", "prom"},
		{"promise", "promis"},
		{"promit", "prom"},
		{"promontoire", "promontoir"},
		{"promotion", "promot"},
		{"prompte", "prompt"},
		{"promptement", "prompt"},
		{"prôner", "prôn"},
		{"prononça", "prononc"},
		{"prononçait", "prononc"},
		{"prononçant", "prononc"},
		{"prononce", "prononc"},
		{"prononcé", "prononc"},
		{"prononcée", "prononc"},
		{"prononcées", "prononc"},
		{"prononcer", "prononc"},
		{"prononcera", "prononc"},
		{"prononcerait", "prononc"},
		{"prononcèrent", "prononc"},
		{"prononcés", "prononc"},
		{"prononciation", "prononci"},
		{"prononçons", "prononçon"},
		{"pronostics", "pronostic"},
		{"propageaient", "propag"},
		{"propagèrent", "propag"},
		{"propension", "propens"},
		{"prophète", "prophet"},
		{"prophètes", "prophet"},
		{"prophétie", "prophet"},
		{"prophéties", "prophet"},
		{"propice", "propic"},
		{"propices", "propic"},
		{"propitiatoires", "propitiatoir"},
		{"proportion", "proport"},
		{"proportionné", "proportion"},
		{"proportions", "proport"},
		{"propos", "propos"},
		{"proposa", "propos"},
		{"proposait", "propos"},
		{"propose", "propos"},
		{"proposé", "propos"},
		{"proposer", "propos"},
		{"proposerait", "propos"},
		{"proposés", "propos"},
		{"proposez", "propos"},
		{"proposition", "proposit"},
		{"propositions", "proposit"},
		{"propre", "propr"},
		{"proprement", "propr"},
		{"propres", "propr"},
		{"propreté", "propret"},
		{"propriétaire", "propriétair"},
		{"propriétaires", "propriétair"},
		{"propriété", "propriet"},
		{"propriétés", "propriet"},
		{"proprio", "proprio"},
		{"prosaïquement", "prosaïqu"},
		{"prosaïques", "prosaïqu"},
		{"proscrivent", "proscrivent"},
		{"prose", "pros"},
		{"prosélytisme", "prosélyt"},
		{"prospèrent", "prosp"},
		{"prospérité", "prosper"},
		{"prospérités", "prosper"},
		{"prosterna", "prostern"},
		{"prosternait", "prostern"},
		{"prosternant", "prostern"},
		{"prostration", "prostrat"},
		{"protecteur", "protecteur"},
		{"protecteurs", "protecteur"},
		{"protection", "protect"},
		{"protections", "protect"},
		{"protège", "proteg"},
		{"protégé", "proteg"},
		{"protégeaient", "proteg"},
		{"protégeait", "proteg"},
		{"protéger", "proteg"},
		{"protégera", "proteg"},
		{"protégés", "proteg"},
		{"protégez", "proteg"},
		{"protesta", "protest"},
		{"protestait", "protest"},
		{"protestant", "protest"},
		{"protestante", "protest"},
		{"protestantisme", "protestant"},
		{"protestants", "protest"},
		{"protestations", "protest"},
		{"proteste", "protest"},
		{"protesté", "protest"},
		{"protester", "protest"},
		{"prouesse", "prouess"},
		{"prouva", "prouv"},
		{"prouvaient", "prouv"},
		{"prouvait", "prouv"},
		{"prouvant", "prouv"},
		{"prouve", "prouv"},
		{"prouvé", "prouv"},
		{"prouvée", "prouv"},
		{"prouvées", "prouv"},
		{"prouvent", "prouvent"},
		{"prouver", "prouv"},
		{"prouveraient", "prouv"},
		{"prouvez", "prouv"},
		{"provenaient", "proven"},
		{"provenait", "proven"},
		{"provence", "provenc"},
		{"proverbe", "proverb"},
		{"proverbiale", "proverbial"},
		{"providence", "provident"},
		{"provient", "provient"},
		{"province", "provinc"},
		{"provinces", "provinc"},
		{"provincial", "provincial"},
		{"provinciale", "provincial"},
		{"provinciaux", "provincial"},
		{"provision", "provis"},
		{"provisions", "provis"},
		{"provocante", "provoc"},
		{"provocatrice", "provoc"},
		{"provoqua", "provoqu"},
		{"provoquait", "provoqu"},
		{"provoqué", "provoqu"},
		{"provoquée", "provoqu"},
		{"provoquer", "provoqu"},
		{"prude", "prud"},
		{"prudemment", "prudent"},
		{"prudence", "prudenc"},
		{"prudent", "prudent"},
		{"prudente", "prudent"},
		{"prudentes", "prudent"},
		{"prudents", "prudent"},
		{"pruderie", "pruder"},
		{"prunelle", "prunel"},
		{"pruniers", "pruni"},
		{"prusse", "pruss"},
		{"prussien", "prussien"},
		{"prussienne", "prussien"},
		{"prussiens", "prussien"},
		{"psalmodie", "psalmod"},
		{"psaume", "psaum"},
		{"psaumes", "psaum"},
		{"pseudonyme", "pseudonym"},
		{"psyché", "psych"},
		{"pu", "pu"},
		{"publia", "publi"},
		{"public", "public"},
		{"publicité", "publiqu"},
		{"publics", "public"},
		{"publie", "publ"},
		{"publié", "publi"},
		{"publiée", "publi"},
		{"publiées", "publi"},
		{"publièrent", "publi"},
		{"publique", "publiqu"},
		{"publiquement", "publiqu"},
		{"publiques", "publiqu"},
		{"pudeur", "pudeur"},
		{"puériles", "puéril"},
		{"puis", "puis"},
		{"puisait", "puis"},
		{"puisant", "puis"},
		{"puisée", "puis"},
		{"puiser", "puis"},
		{"puisés", "puis"},
		{"puisqu", "puisqu"},
		{"puisque", "puisqu"},
		{"puissamment", "puiss"},
		{"puissance", "puissanc"},
		{"puissances", "puissanc"},
		{"puissant", "puiss"},
		{"puissante", "puiss"},
		{"puissantes", "puiss"},
		{"puissants", "puiss"},
		{"puisse", "puiss"},
		{"puissent", "puissent"},
		{"puissiez", "puiss"},
		{"puissions", "puission"},
		{"puits", "puit"},
		{"pullulaient", "pullul"},
		{"pullulent", "pullulent"},
		{"punch", "punch"},
		{"puni", "pun"},
		{"punie", "pun"},
		{"punir", "pun"},
		{"punira", "pun"},
		{"punirait", "pun"},
		{"punirmi", "punirm"},
		{"punis", "pun"},
		{"punisse", "pun"},
		{"punit", "pun"},
		{"punîtes", "pun"},
		{"punition", "punit"},
		{"punitions", "punit"},
		{"puntiglio", "puntiglio"},
		{"pupille", "pupill"},
		{"pupitre", "pupitr"},
		{"pur", "pur"},
		{"pure", "pur"},
		{"purement", "pur"},
		{"purent", "purent"},
		{"pureté", "puret"},
		{"purgeait", "purg"},
		{"purger", "purg"},
		{"puritaine", "puritain"},
		{"purs", "pur"},
		{"purser", "purs"},
		{"pusillanime", "pusillanim"},
		{"pusillanimes", "pusillanim"},
		{"pusillanimité", "pusillanim"},
		{"pusse", "puss"},
		{"pussent", "pussent"},
		{"put", "put"},
		{"pût", "pût"},
		{"putiphar", "putiphar"},
		{"putréfaction", "putréfact"},
		{"pyramide", "pyramid"},
		{"pyrénées", "pyren"},
		{"qu", "qu"},
		{"qua", "qua"},
		{"quadrille", "quadrill"},
		{"quadrupède", "quadruped"},
		{"quadruple", "quadrupl"},
		{"quai", "quai"},
		{"quais", "quais"},
		{"quakeresse", "quakeress"},
		{"qualifiait", "qualifi"},
		{"qualificatif", "qualif"},
		{"qualification", "qualif"},
		{"qualité", "qualit"},
		{"qualités", "qualit"},
		{"quand", "quand"},
		{"quando", "quando"},
		{"quant", "quant"},
		{"quanti", "quant"},
		{"quantième", "quantiem"},
		{"quantièmes", "quantiem"},
		{"quantité", "quantit"},
		{"quarantaine", "quarantain"},
		{"quarante", "quar"},
		{"quarantième", "quarantiem"},
		{"quart", "quart"},
		{"quarti", "quart"},
		{"quartier", "quarti"},
		{"quartiers", "quarti"},
		{"quarto", "quarto"},
		{"quarts", "quart"},
		{"quatorze", "quatorz"},
		{"quatre", "quatr"},
		{"quatrième", "quatriem"},
		{"quatrièmes", "quatriem"},
		{"que", "que"},
		{"queenstown", "queenstown"},
		{"quel", "quel"},
		{"quelconque", "quelconqu"},
		{"quelle", "quel"},
		{"quelles", "quel"},
		{"quelqu", "quelqu"},
		{"quelque", "quelqu"},
		{"quelquefois", "quelquefois"},
		{"quelques", "quelqu"},
		{"quels", "quel"},
		{"quem", "quem"},
		{"querelle", "querel"},
		{"querelles", "querel"},
		{"question", "question"},
		{"questionner", "question"},
		{"questions", "question"},
		{"quête", "quêt"},
		{"quêtes", "quêt"},
		{"quêteurs", "quêteur"},
		{"queue", "queu"},
		{"qui", "qui"},
		{"quibus", "quibus"},
		{"quichotte", "quichott"},
		{"quiconque", "quiconqu"},
		{"quid", "quid"},
		{"quille", "quill"},
		{"quint", "quint"},
		{"quintaux", "quintal"},
		{"quinte", "quint"},
		{"quinzaine", "quinzain"},
		{"quinze", "quinz"},
		{"quinzième", "quinziem"},
		{"quitta", "quitt"},
		{"quittai", "quitt"},
		{"quittais", "quitt"},
		{"quittait", "quitt"},
		{"quittant", "quitt"},
		{"quittât", "quitt"},
		{"quitte", "quitt"},
		{"quitté", "quitt"},
		{"quittent", "quittent"},
		{"quitter", "quitt"},
		{"quittera", "quitt"},
		{"quitterai", "quitt"},
		{"quitterait", "quitt"},
		{"quittèrent", "quitt"},
		{"quitterez", "quitt"},
		{"quitterons", "quitt"},
		{"quittes", "quitt"},
		{"quittés", "quitt"},
		{"quittez", "quitt"},
		{"quittons", "quitton"},
		{"quoerens", "quoeren"},
		{"quoi", "quoi"},
		{"quoiqu", "quoiqu"},
		{"quoique", "quoiqu"},
		{"quotidien", "quotidien"},
		{"quotidienne", "quotidien"},
		{"quotidiennement", "quotidien"},
		{"quotité", "quotit"},
		{"r", "r"},
		{"rabâchage", "rabâchag"},
		{"rabaisser", "rabaiss"},
		{"raccommodait", "raccommod"},
		{"raccommodant", "raccommod"},
		{"raccommodé", "raccommod"},
		{"raccommodements", "raccommod"},
		{"raccommoder", "raccommod"},
		{"raccordent", "raccordent"},
		{"raccourcis", "raccourc"},
		{"race", "rac"},
		{"races", "rac"},
		{"rachète", "rachet"},
		{"racheter", "rachet"},
		{"racine", "racin"},
		{"raconta", "racont"},
		{"racontait", "racont"},
		{"racontant", "racont"},
		{"racontât", "racont"},
		{"raconte", "racont"},
		{"raconté", "racont"},
		{"racontent", "racontent"},
		{"raconter", "racont"},
		{"raconterai", "racont"},
		{"raconteraient", "racont"},
		{"raconterait", "racont"},
		{"racontèrent", "racont"},
		{"raconterons", "racont"},
		{"racontés", "racont"},
		{"racontez", "racont"},
		{"rade", "rad"},
		{"radieux", "radieux"},
		{"radouci", "radouc"},
		{"rafale", "rafal"},
		{"rafales", "rafal"},
		{"raffinée", "raffin"},
		{"rafraîchie", "rafraîch"},
		{"rafraîchir", "rafraîch"},
		{"rafraîchissements", "rafraîch"},
		{"rafraîchit", "rafraîch"},
		{"rage", "rag"},
		{"rageant", "rag"},
		{"rages", "rag"},
		{"rageurs", "rageur"},
		{"raide", "raid"},
		{"raidi", "raid"},
		{"raidie", "raid"},
		{"raidirent", "raid"},
		{"raies", "rai"},
		{"rail", "rail"},
		{"raillerie", "railler"},
		{"railroad", "railroad"},
		{"rails", "rail"},
		{"railway", "railway"},
		{"railways", "railway"},
		{"raison", "raison"},
		{"raisonna", "raison"},
		{"raisonnable", "raison"},
		{"raisonnablement", "raison"},
		{"raisonnables", "raison"},
		{"raisonnais", "raison"},
		{"raisonnait", "raison"},
		{"raisonnante", "raison"},
		{"raisonne", "raison"},
		{"raisonnement", "raison"},
		{"raisonnements", "raison"},
		{"raisonner", "raison"},
		{"raisonneur", "raisonneur"},
		{"raisons", "raison"},
		{"rajah", "rajah"},
		{"rajahs", "rajah"},
		{"rajeuni", "rajeun"},
		{"râlaient", "râl"},
		{"ralentie", "ralent"},
		{"ralentir", "ralent"},
		{"ralentissant", "ralent"},
		{"ralentit", "ralent"},
		{"ralliait", "ralli"},
		{"rallier", "ralli"},
		{"ralluma", "rallum"},
		{"ralph", "ralph"},
		{"ramassa", "ramass"},
		{"ramassaient", "ramass"},
		{"ramassait", "ramass"},
		{"ramasse", "ram"},
		{"ramassé", "ramass"},
		{"ramasser", "ramass"},
		{"ramasseraient", "ramass"},
		{"ramassés", "ramass"},
		{"ramassis", "ramass"},
		{"ramayana", "ramayan"},
		{"rambarde", "rambard"},
		{"rambo", "rambo"},
		{"rame", "ram"},
		{"ramena", "ramen"},
		{"ramenait", "ramen"},
		{"ramenant", "ramen"},
		{"ramener", "ramen"},
		{"rames", "ram"},
		{"rameurs", "rameur"},
		{"ramifiant", "ramifi"},
		{"ramifications", "ramif"},
		{"ramifiées", "ramifi"},
		{"rampant", "ramp"},
		{"rampe", "ramp"},
		{"rampes", "ramp"},
		{"ramure", "ramur"},
		{"ramures", "ramur"},
		{"ranchos", "ranchos"},
		{"rancune", "rancun"},
		{"rancunier", "rancuni"},
		{"rang", "rang"},
		{"rangé", "rang"},
		{"rangea", "rang"},
		{"rangeait", "rang"},
		{"rangée", "rang"},
		{"rangées", "rang"},
		{"ranger", "rang"},
		{"rangerait", "rang"},
		{"rangèrent", "rang"},
		{"ranges", "rang"},
		{"rangés", "rang"},
		{"rangoon", "rangoon"},
		{"rangs", "rang"},
		{"ranima", "ranim"},
		{"ranimé", "ranim"},
		{"ranimer", "ranim"},
		{"ranimèrent", "ranim"},
		{"ranuce", "ranuc"},
		{"rapacité", "rapac"},
		{"rapatrier", "rapatri"},
		{"râpé", "râp"},
		{"râpés", "râp"},
		{"rapide", "rapid"},
		{"rapidement", "rapid"},
		{"rapides", "rapid"},
		{"rapidité", "rapid"},
		{"rappela", "rappel"},
		{"rappelaient", "rappel"},
		{"rappelait", "rappel"},
		{"rappelant", "rappel"},
		{"rappelât", "rappel"},
		{"rappelé", "rappel"},
		{"rappelée", "rappel"},
		{"rappeler", "rappel"},
		{"rappelèrent", "rappel"},
		{"rappelés", "rappel"},
		{"rappelez", "rappel"},
		{"rappelle", "rappel"},
		{"rappellent", "rappellent"},
		{"rappellera", "rappel"},
		{"rappellerai", "rappel"},
		{"rappellerais", "rappel"},
		{"rappellerait", "rappel"},
		{"rappellerez", "rappel"},
		{"rapport", "rapport"},
		{"rapporta", "rapport"},
		{"rapportaient", "rapport"},
		{"rapportait", "rapport"},
		{"rapportant", "rapport"},
		{"rapportât", "rapport"},
		{"rapporte", "rapport"},
		{"rapporté", "rapport"},
		{"rapportée", "rapport"},
		{"rapportées", "rapport"},
		{"rapporter", "rapport"},
		{"rapporterait", "rapport"},
		{"rapportèrent", "rapport"},
		{"rapporterons", "rapport"},
		{"rapportes", "rapport"},
		{"rapporteur", "rapporteur"},
		{"rapports", "rapport"},
		{"rapprocha", "rapproch"},
		{"rapprochaient", "rapproch"},
		{"rapprochait", "rapproch"},
		{"rapprochant", "rapproch"},
		{"rapproche", "rapproch"},
		{"rapproché", "rapproch"},
		{"rapprochées", "rapproch"},
		{"rapprocher", "rapproch"},
		{"rapprocherait", "rapproch"},
		{"rapprochèrent", "rapproch"},
		{"rapprochés", "rapproch"},
		{"raquettes", "raquet"},
		{"rare", "rar"},
		{"raréfiaient", "raréfi"},
		{"rarement", "rar"},
		{"rares", "rar"},
		{"ras", "ras"},
		{"rasant", "ras"},
		{"rasé", "ras"},
		{"rasée", "ras"},
		{"rasées", "ras"},
		{"raser", "ras"},
		{"raserait", "ras"},
		{"rassasié", "rassasi"},
		{"rassasiée", "rassasi"},
		{"rassasier", "rassasi"},
		{"rassemblait", "rassembl"},
		{"rassemblé", "rassembl"},
		{"rassembler", "rassembl"},
		{"rassemblés", "rassembl"},
		{"rasseoir", "rasseoir"},
		{"rasséréna", "rasséren"},
		{"rasséréner", "rasséren"},
		{"rassi", "rass"},
		{"rassit", "rass"},
		{"rassura", "rassur"},
		{"rassuraient", "rassur"},
		{"rassurait", "rassur"},
		{"rassure", "rassur"},
		{"rassuré", "rassur"},
		{"rassurée", "rassur"},
		{"rassurer", "rassur"},
		{"rassurez", "rassur"},
		{"ratine", "ratin"},
		{"rationnelle", "rationnel"},
		{"rationnellement", "rationnel"},
		{"rats", "rat"},
		{"rattachait", "rattach"},
		{"ravages", "ravag"},
		{"ravaler", "raval"},
		{"ravenne", "raven"},
		{"raversi", "ravers"},
		{"ravi", "rav"},
		{"ravie", "rav"},
		{"ravins", "ravin"},
		{"ravir", "rav"},
		{"ravis", "rav"},
		{"raviser", "ravis"},
		{"ravissait", "rav"},
		{"ravissant", "rav"},
		{"ravissante", "rav"},
		{"ravissantes", "rav"},
		{"ravissants", "rav"},
		{"ravissement", "rav"},
		{"ravisseur", "ravisseur"},
		{"ravisseurs", "ravisseur"},
		{"ravit", "rav"},
		{"ravoir", "ravoir"},
		{"rayées", "rai"},
		{"raynal", "raynal"},
		{"rayon", "rayon"},
		{"rayonna", "rayon"},
		{"rayons", "rayon"},
		{"razori", "razor"},
		{"réaction", "réaction"},
		{"reading", "reading"},
		{"réal", "réal"},
		{"réalisable", "réalis"},
		{"réaliser", "réalis"},
		{"réaliste", "réalist"},
		{"réalité", "réalit"},
		{"réapprovisionner", "réapprovision"},
		{"rébarbatif", "rébarb"},
		{"rébarbative", "rébarb"},
		{"rebâtir", "rebât"},
		{"rebelle", "rebel"},
		{"rebelles", "rebel"},
		{"rébellion", "rébellion"},
		{"récapitulant", "récapitul"},
		{"récemment", "récent"},
		{"récente", "récent"},
		{"réception", "récept"},
		{"réceptions", "récept"},
		{"recette", "recet"},
		{"recettes", "recet"},
		{"recevaient", "recev"},
		{"recevait", "recev"},
		{"recevant", "recev"},
		{"recevez", "recev"},
		{"receviez", "recev"},
		{"recevoir", "recevoir"},
		{"recevra", "recevr"},
		{"recevrai", "recevr"},
		{"recevrait", "recevr"},
		{"recevrez", "recevr"},
		{"recharger", "recharg"},
		{"rechargez", "recharg"},
		{"réchauffé", "réchauff"},
		{"rechercha", "recherch"},
		{"recherchait", "recherch"},
		{"recherche", "recherch"},
		{"recherché", "recherch"},
		{"rechercher", "recherch"},
		{"rechercheront", "recherch"},
		{"recherches", "recherch"},
		{"recherchés", "recherch"},
		{"rechignant", "rechign"},
		{"rechigné", "rechign"},
		{"rechute", "rechut"},
		{"récidive", "récid"},
		{"reciproquement", "reciproqu"},
		{"réciproquement", "réciproqu"},
		{"recit", "rec"},
		{"récit", "rec"},
		{"récita", "récit"},
		{"récitaient", "récit"},
		{"récitait", "récit"},
		{"récitant", "récit"},
		{"récitatif", "récit"},
		{"récitation", "récit"},
		{"récite", "récit"},
		{"récité", "récit"},
		{"réciter", "récit"},
		{"réciterai", "récit"},
		{"réciterez", "récit"},
		{"récits", "récit"},
		{"réclamait", "réclam"},
		{"réclame", "réclam"},
		{"réclamée", "réclam"},
		{"réclamer", "réclam"},
		{"réclamés", "réclam"},
		{"reclus", "reclus"},
		{"réclusion", "réclus"},
		{"reçois", "reçois"},
		{"reçoit", "reçoit"},
		{"reçoive", "reçoiv"},
		{"reçoivent", "reçoivent"},
		{"récolte", "récolt"},
		{"recommanda", "recommand"},
		{"recommandait", "recommand"},
		{"recommandant", "recommand"},
		{"recommandation", "recommand"},
		{"recommandations", "recommand"},
		{"recommande", "recommand"},
		{"recommandé", "recommand"},
		{"recommander", "recommand"},
		{"recommandés", "recommand"},
		{"recommandez", "recommand"},
		{"recommença", "recommenc"},
		{"recommençaient", "recommenc"},
		{"recommençait", "recommenc"},
		{"recommençât", "recommenc"},
		{"recommence", "recomment"},
		{"recommencé", "recommenc"},
		{"recommencer", "recommenc"},
		{"recommencerai", "recommenc"},
		{"recommenceraient", "recommenc"},
		{"recommencerait", "recommenc"},
		{"recommencèrent", "recommenc"},
		{"recommencez", "recommenc"},
		{"récompense", "récompens"},
		{"récompensé", "récompens"},
		{"récompensée", "récompens"},
		{"recomptait", "recompt"},
		{"réconciliation", "réconcili"},
		{"réconcilie", "réconcil"},
		{"reconduire", "reconduir"},
		{"reconduisait", "reconduis"},
		{"reconduisant", "reconduis"},
		{"reconduisit", "reconduis"},
		{"reconduit", "reconduit"},
		{"reconnais", "recon"},
		{"reconnaissable", "reconnaiss"},
		{"reconnaissaient", "reconnaiss"},
		{"reconnaissait", "reconnaiss"},
		{"reconnaissance", "reconnaiss"},
		{"reconnaissant", "reconnaiss"},
		{"reconnaisse", "reconnaiss"},
		{"reconnaissent", "reconnaissent"},
		{"reconnaissez", "reconnaiss"},
		{"reconnaît", "reconnaît"},
		{"reconnaîtra", "reconnaîtr"},
		{"reconnaître", "reconnaîtr"},
		{"reconnu", "reconnu"},
		{"reconnue", "reconnu"},
		{"reconnurent", "reconnurent"},
		{"reconnus", "reconnus"},
		{"reconnut", "reconnut"},
		{"reconquérir", "reconquer"},
		{"reconquerrait", "reconquerr"},
		{"reconquis", "reconqu"},
		{"recoucher", "recouch"},
		{"recoudre", "recoudr"},
		{"recourir", "recour"},
		{"recours", "recour"},
		{"recouvert", "recouvert"},
		{"recouverte", "recouvert"},
		{"recouverts", "recouvert"},
		{"recouvra", "recouvr"},
		{"recouvrait", "recouvr"},
		{"recouvrée", "recouvr"},
		{"recouvrer", "recouvr"},
		{"recouvrir", "recouvr"},
		{"recouvrît", "recouvr"},
		{"récréation", "récréat"},
		{"récréations", "récréat"},
		{"récriait", "récri"},
		{"récrier", "récri"},
		{"récrièrent", "récri"},
		{"recrues", "recru"},
		{"recrute", "recrut"},
		{"recruter", "recrut"},
		{"rectifia", "rectifi"},
		{"rectifiait", "rectifi"},
		{"rectifier", "rectifi"},
		{"rectiligne", "rectilign"},
		{"reçu", "reçu"},
		{"reçue", "reçu"},
		{"recueil", "recueil"},
		{"recueillent", "recueillent"},
		{"recueilli", "recueil"},
		{"recueillir", "recueil"},
		{"recueillirent", "recueil"},
		{"reçues", "reçu"},
		{"recula", "recul"},
		{"reculant", "recul"},
		{"recule", "recul"},
		{"reculé", "recul"},
		{"reculer", "recul"},
		{"reçurent", "reçurent"},
		{"reçus", "reçus"},
		{"reçut", "reçut"},
		{"reçût", "reçût"},
		{"rédacteur", "rédacteur"},
		{"rédacteurs", "rédacteur"},
		{"rédaction", "rédact"},
		{"redemande", "redemand"},
		{"redescend", "redescend"},
		{"redescendant", "redescend"},
		{"redescendit", "redescend"},
		{"redescendre", "redescendr"},
		{"redescendue", "redescendu"},
		{"redevenaient", "redeven"},
		{"redevenait", "redeven"},
		{"redevenant", "redeven"},
		{"redevenu", "redevenu"},
		{"redevenue", "redevenu"},
		{"redevint", "redevint"},
		{"rédigea", "rédig"},
		{"rédigée", "rédig"},
		{"rédigerait", "rédig"},
		{"redingote", "redingot"},
		{"redingotes", "redingot"},
		{"redira", "red"},
		{"redits", "redit"},
		{"redonner", "redon"},
		{"redoubla", "redoubl"},
		{"redoublaient", "redoubl"},
		{"redoublait", "redoubl"},
		{"redoublant", "redoubl"},
		{"redouble", "redoubl"},
		{"redoublé", "redoubl"},
		{"redoublement", "redoubl"},
		{"redoubler", "redoubl"},
		{"redoublerait", "redoubl"},
		{"redoublèrent", "redoubl"},
		{"redoutables", "redout"},
		{"redoutaient", "redout"},
		{"redoutait", "redout"},
		{"redoute", "redout"},
		{"redouté", "redout"},
		{"redouter", "redout"},
		{"redoutons", "redouton"},
		{"redressait", "redress"},
		{"redresser", "redress"},
		{"réduiras", "réduir"},
		{"réduire", "réduir"},
		{"réduirons", "réduiron"},
		{"réduisait", "réduis"},
		{"réduisent", "réduisent"},
		{"réduisit", "réduis"},
		{"réduit", "réduit"},
		{"réduite", "réduit"},
		{"réduites", "réduit"},
		{"réduits", "réduit"},
		{"réel", "réel"},
		{"réélection", "réélect"},
		{"réelle", "réel"},
		{"réellement", "réel"},
		{"réelles", "réel"},
		{"réels", "réel"},
		{"réexpédié", "réexpédi"},
		{"refaire", "refair"},
		{"refaisait", "refais"},
		{"refaites", "refait"},
		{"réfectoire", "réfectoir"},
		{"références", "référent"},
		{"referma", "referm"},
		{"refermé", "referm"},
		{"refermée", "referm"},
		{"refermer", "referm"},
		{"réfléchi", "réflech"},
		{"réfléchir", "réflech"},
		{"réfléchirez", "réflech"},
		{"réfléchissait", "réflech"},
		{"réfléchissant", "réflech"},
		{"réfléchissez", "réflech"},
		{"réfléchit", "réflech"},
		{"réfléchît", "réflech"},
		{"reflet", "reflet"},
		{"reflétait", "reflet"},
		{"reflète", "reflet"},
		{"reflets", "reflet"},
		{"réflexion", "réflexion"},
		{"réflexions", "réflex"},
		{"reflua", "reflu"},
		{"refluait", "reflu"},
		{"reform", "reform"},
		{"refouler", "refoul"},
		{"réfractaire", "réfractair"},
		{"réfractaires", "réfractair"},
		{"refrain", "refrain"},
		{"refrains", "refrain"},
		{"refroidies", "refroid"},
		{"refroidir", "refroid"},
		{"refuge", "refug"},
		{"réfugia", "réfugi"},
		{"réfugiait", "réfugi"},
		{"réfugiant", "réfugi"},
		{"réfugie", "réfug"},
		{"réfugié", "réfugi"},
		{"réfugiée", "réfugi"},
		{"réfugier", "réfugi"},
		{"réfugieront", "réfugi"},
		{"réfugiés", "réfugi"},
		{"réfugiez", "réfug"},
		{"refus", "refus"},
		{"refusa", "refus"},
		{"refusait", "refus"},
		{"refusant", "refus"},
		{"refuse", "refus"},
		{"refusé", "refus"},
		{"refusée", "refus"},
		{"refusées", "refus"},
		{"refusent", "refusent"},
		{"refuser", "refus"},
		{"refuserait", "refus"},
		{"refuseriez", "refus"},
		{"refusez", "refus"},
		{"refusiez", "refus"},
		{"réfutés", "réfut"},
		{"regagna", "regagn"},
		{"regagnaient", "regagn"},
		{"regagnait", "regagn"},
		{"regagné", "regagn"},
		{"regagner", "regagn"},
		{"régalait", "régal"},
		{"regard", "regard"},
		{"regarda", "regard"},
		{"regardaient", "regard"},
		{"regardais", "regard"},
		{"regardait", "regard"},
		{"regardant", "regard"},
		{"regarde", "regard"},
		{"regardé", "regard"},
		{"regardée", "regard"},
		{"regardent", "regardent"},
		{"regarder", "regard"},
		{"regardera", "regard"},
		{"regarderai", "regard"},
		{"regardèrent", "regard"},
		{"regarderez", "regard"},
		{"regarderont", "regard"},
		{"regardés", "regard"},
		{"regardez", "regard"},
		{"regards", "regard"},
		{"régate", "régat"},
		{"régénérateur", "régéner"},
		{"régénérer", "régéner"},
		{"regent", "regent"},
		{"régent", "régent"},
		{"regimbait", "regimb"},
		{"régime", "régim"},
		{"régiment", "reg"},
		{"régiments", "reg"},
		{"régimes", "régim"},
		{"région", "région"},
		{"régions", "région"},
		{"registre", "registr"},
		{"registres", "registr"},
		{"régla", "regl"},
		{"réglant", "regl"},
		{"règle", "regl"},
		{"réglé", "regl"},
		{"réglée", "regl"},
		{"règlement", "regl"},
		{"réglementaire", "réglementair"},
		{"réglementaires", "réglementair"},
		{"réglementation", "réglement"},
		{"réglementée", "réglement"},
		{"règlements", "regl"},
		{"régler", "regl"},
		{"réglera", "regl"},
		{"règles", "regl"},
		{"réglés", "regl"},
		{"régna", "regn"},
		{"régnaient", "regn"},
		{"régnait", "regn"},
		{"régnant", "regn"},
		{"régnante", "regn"},
		{"règne", "regn"},
		{"régné", "regn"},
		{"règnent", "règnent"},
		{"régner", "regn"},
		{"régnera", "regn"},
		{"régnez", "regn"},
		{"régnons", "régnon"},
		{"regorgeant", "regorg"},
		{"regorgent", "regorgent"},
		{"regret", "regret"},
		{"regrets", "regret"},
		{"regrettaient", "regret"},
		{"regrettait", "regret"},
		{"regrette", "regret"},
		{"regrettée", "regret"},
		{"regrettées", "regret"},
		{"regretter", "regret"},
		{"regretterais", "regret"},
		{"regretteriez", "regret"},
		{"regrettez", "regret"},
		{"régularisé", "régularis"},
		{"régularité", "régular"},
		{"régulateur", "régul"},
		{"régulier", "réguli"},
		{"régulière", "réguli"},
		{"régulièrement", "réguli"},
		{"rehausse", "rehauss"},
		{"rein", "rein"},
		{"reina", "rein"},
		{"reine", "rein"},
		{"reins", "rein"},
		{"réintégré", "réintegr"},
		{"réintégrés", "réintegr"},
		{"réitérée", "réiter"},
		{"rejaillir", "rejaill"},
		{"rejeta", "rejet"},
		{"rejeté", "rejet"},
		{"rejettera", "rejet"},
		{"rejoignait", "rejoign"},
		{"rejoignant", "rejoign"},
		{"rejoignent", "rejoignent"},
		{"rejoignirent", "rejoign"},
		{"rejoignit", "rejoign"},
		{"rejoindra", "rejoindr"},
		{"rejoindre", "rejoindr"},
		{"rejoint", "rejoint"},
		{"réjouie", "réjou"},
		{"réjouir", "réjou"},
		{"réjouirez", "réjou"},
		{"réjouissait", "réjou"},
		{"réjouissances", "réjouiss"},
		{"réjouit", "réjou"},
		{"relâchait", "relâch"},
		{"relâche", "relâch"},
		{"relâché", "relâch"},
		{"relâchées", "relâch"},
		{"relâcher", "relâch"},
		{"relais", "rel"},
		{"relancer", "relanc"},
		{"relatif", "relat"},
		{"relatifs", "relat"},
		{"relation", "relat"},
		{"relations", "relat"},
		{"relative", "relat"},
		{"relativement", "relat"},
		{"relatives", "relat"},
		{"relégué", "relégu"},
		{"reléguée", "relégu"},
		{"reléguer", "relégu"},
		{"releva", "relev"},
		{"relevaient", "relev"},
		{"relevait", "relev"},
		{"relevant", "relev"},
		{"relevé", "relev"},
		{"relevée", "relev"},
		{"relevées", "relev"},
		{"relèvent", "relèvent"},
		{"relever", "relev"},
		{"relèveras", "relev"},
		{"relié", "reli"},
		{"reliée", "reli"},
		{"reliées", "reli"},
		{"relief", "relief"},
		{"reliefs", "relief"},
		{"relier", "reli"},
		{"reliés", "reli"},
		{"religieuse", "religi"},
		{"religieusement", "religi"},
		{"religieuses", "religi"},
		{"religieux", "religi"},
		{"religion", "religion"},
		{"religions", "relig"},
		{"religiosité", "religios"},
		{"reliquat", "reliquat"},
		{"relique", "reliqu"},
		{"relire", "relir"},
		{"relis", "rel"},
		{"relisais", "relis"},
		{"relisait", "relis"},
		{"relise", "relis"},
		{"relu", "relu"},
		{"relut", "relut"},
		{"rem", "rem"},
		{"remariais", "remari"},
		{"remarqua", "remarqu"},
		{"remarquable", "remarqu"},
		{"remarquablement", "remarqu"},
		{"remarquables", "remarqu"},
		{"remarquaient", "remarqu"},
		{"remarquait", "remarqu"},
		{"remarquant", "remarqu"},
		{"remarquâtes", "remarqu"},
		{"remarque", "remarqu"},
		{"remarqué", "remarqu"},
		{"remarquée", "remarqu"},
		{"remarquer", "remarqu"},
		{"remarquera", "remarqu"},
		{"remarquèrent", "remarqu"},
		{"remarquerez", "remarqu"},
		{"remarques", "remarqu"},
		{"remarqués", "remarqu"},
		{"remarquez", "remarqu"},
		{"rembourse", "rembours"},
		{"remboursement", "rembours"},
		{"rembourser", "rembours"},
		{"rembrunit", "rembrun"},
		{"remède", "remed"},
		{"remèdes", "remed"},
		{"remédier", "remédi"},
		{"remercia", "remerci"},
		{"remerciait", "remerci"},
		{"remerciant", "remerci"},
		{"remercie", "remerc"},
		{"remercié", "remerci"},
		{"remerciée", "remerci"},
		{"remerciements", "remerci"},
		{"remercier", "remerci"},
		{"remercierais", "remerci"},
		{"remercierait", "remerci"},
		{"remercîment", "remercî"},
		{"remercîments", "remercî"},
		{"remet", "remet"},
		{"remets", "remet"},
		{"remettait", "remet"},
		{"remettant", "remet"},
		{"remette", "remet"},
		{"remettent", "remettent"},
		{"remettra", "remettr"},
		{"remettrai", "remettr"},
		{"remettrais", "remettr"},
		{"remettrait", "remettr"},
		{"remettras", "remettr"},
		{"remettre", "remettr"},
		{"remettrez", "remettr"},
		{"remiers", "remi"},
		{"remirent", "rem"},
		{"remis", "rem"},
		{"remise", "remis"},
		{"remises", "remis"},
		{"rémission", "rémiss"},
		{"remit", "rem"},
		{"remonta", "remont"},
		{"remontait", "remont"},
		{"remontant", "remont"},
		{"remonte", "remont"},
		{"remonté", "remont"},
		{"remontent", "remontent"},
		{"remonter", "remont"},
		{"remonterait", "remont"},
		{"remontez", "remont"},
		{"remontrance", "remontr"},
		{"remontrances", "remontr"},
		{"remontrer", "remontr"},
		{"remords", "remord"},
		{"remorquant", "remorqu"},
		{"remous", "remous"},
		{"rempart", "rempart"},
		{"remparts", "rempart"},
		{"remplaça", "remplac"},
		{"remplaçaient", "remplac"},
		{"remplaçait", "remplac"},
		{"remplace", "remplac"},
		{"remplacé", "remplac"},
		{"remplacée", "remplac"},
		{"remplacement", "remplac"},
		{"remplacer", "remplac"},
		{"remplacera", "remplac"},
		{"remplacés", "remplac"},
		{"rempli", "rempl"},
		{"remplie", "rempl"},
		{"remplies", "rempl"},
		{"remplir", "rempl"},
		{"remplirent", "rempl"},
		{"remplis", "rempl"},
		{"remplissaient", "rempl"},
		{"remplissait", "rempl"},
		{"remplissant", "rempl"},
		{"remplissent", "rempl"},
		{"remplit", "rempl"},
		{"remplumai", "remplum"},
		{"remportant", "remport"},
		{"remporter", "remport"},
		{"remua", "remu"},
		{"remuant", "remu"},
		{"remue", "remu"},
		{"remuée", "remu"},
		{"remuer", "remu"},
		{"remues", "remu"},
		{"rémunération", "rémuner"},
		{"renaissant", "renaiss"},
		{"renaissantes", "renaiss"},
		{"renaître", "renaîtr"},
		{"rênal", "rênal"},
		{"renard", "renard"},
		{"rencontra", "rencontr"},
		{"rencontrai", "rencontr"},
		{"rencontrait", "rencontr"},
		{"rencontrâmes", "rencontr"},
		{"rencontrant", "rencontr"},
		{"rencontrassent", "rencontr"},
		{"rencontrât", "rencontr"},
		{"rencontre", "rencontr"},
		{"rencontré", "rencontr"},
		{"rencontrée", "rencontr"},
		{"rencontrées", "rencontr"},
		{"rencontrent", "rencontrent"},
		{"rencontrer", "rencontr"},
		{"rencontrerai", "rencontr"},
		{"rencontreraient", "rencontr"},
		{"rencontrèrent", "rencontr"},
		{"rencontres", "rencontr"},
		{"rencontrés", "rencontr"},
		{"rend", "rend"},
		{"rendaient", "rend"},
		{"rendait", "rend"},
		{"rendant", "rend"},
		{"rende", "rend"},
		{"rendent", "rendent"},
		{"rendez", "rend"},
		{"rendirent", "rend"},
		{"rendit", "rend"},
		{"rendît", "rend"},
		{"rendra", "rendr"},
		{"rendrai", "rendr"},
		{"rendrait", "rendr"},
		{"rendre", "rendr"},
		{"rendrez", "rendr"},
		{"rendriez", "rendr"},
		{"rendrons", "rendron"},
		{"rendront", "rendront"},
		{"rends", "rend"},
		{"rendu", "rendu"},
		{"rendue", "rendu"},
		{"rendus", "rendus"},
		{"rêne", "rên"},
		{"rené", "ren"},
		{"renfermait", "renferm"},
		{"renfermant", "renferm"},
		{"renferme", "renferm"},
		{"renfermé", "renferm"},
		{"renfermée", "renferm"},
		{"renfermer", "renferm"},
		{"renforcé", "renforc"},
		{"renfort", "renfort"},
		{"renfrogné", "renfrogn"},
		{"reniera", "reni"},
		{"reno", "reno"},
		{"renom", "renom"},
		{"renommé", "renomm"},
		{"renommée", "renomm"},
		{"renonça", "renonc"},
		{"renonçait", "renonc"},
		{"renonce", "renonc"},
		{"renoncé", "renonc"},
		{"renoncer", "renonc"},
		{"renoncera", "renonc"},
		{"renoncerais", "renonc"},
		{"renoncez", "renonc"},
		{"renoué", "renou"},
		{"renouer", "renou"},
		{"renouvela", "renouvel"},
		{"renouvelait", "renouvel"},
		{"renouvelant", "renouvel"},
		{"renouvelé", "renouvel"},
		{"renouvelées", "renouvel"},
		{"renouveler", "renouvel"},
		{"renouvelèrent", "renouvel"},
		{"renouvellement", "renouvel"},
		{"renouvellent", "renouvellent"},
		{"renseignement", "renseign"},
		{"renseignements", "renseign"},
		{"rente", "rent"},
		{"rentes", "rent"},
		{"rentra", "rentr"},
		{"rentrait", "rentr"},
		{"rentrant", "rentr"},
		{"rentrât", "rentr"},
		{"rentre", "rentr"},
		{"rentré", "rentr"},
		{"rentrée", "rentr"},
		{"rentrent", "rentrent"},
		{"rentrer", "rentr"},
		{"rentrera", "rentr"},
		{"rentrerai", "rentr"},
		{"rentrèrent", "rentr"},
		{"rentrerez", "rentr"},
		{"rentrés", "rentr"},
		{"rentrons", "rentron"},
		{"renverra", "renverr"},
		{"renverraient", "renverr"},
		{"renverrez", "renverr"},
		{"renverrons", "renverron"},
		{"renversa", "renvers"},
		{"renversant", "renvers"},
		{"renverse", "renvers"},
		{"renversé", "renvers"},
		{"renversée", "renvers"},
		{"renversement", "renvers"},
		{"renverser", "renvers"},
		{"renverseront", "renvers"},
		{"renvoie", "renvoi"},
		{"renvoya", "renvoi"},
		{"renvoyait", "renvoi"},
		{"renvoyant", "renvoi"},
		{"renvoyé", "renvoi"},
		{"renvoyée", "renvoi"},
		{"renvoyer", "renvoi"},
		{"renvoyèrent", "renvoi"},
		{"renvoyez", "renvoi"},
		{"renvoyiez", "renvoi"},
		{"renvoyons", "renvoyon"},
		{"renza", "renz"},
		{"repaire", "repair"},
		{"répandaient", "répand"},
		{"répandait", "répand"},
		{"répandant", "répand"},
		{"répandent", "répandent"},
		{"répandit", "répand"},
		{"répandît", "répand"},
		{"répandrait", "répandr"},
		{"répandre", "répandr"},
		{"répandu", "répandu"},
		{"répandue", "répandu"},
		{"répandues", "répandu"},
		{"reparaissaient", "reparaiss"},
		{"reparaissait", "reparaiss"},
		{"reparaîtra", "reparaîtr"},
		{"reparaîtrai", "reparaîtr"},
		{"reparaîtrait", "reparaîtr"},
		{"reparaître", "reparaîtr"},
		{"reparaîtrons", "reparaîtron"},
		{"réparation", "répar"},
		{"réparations", "répar"},
		{"réparée", "répar"},
		{"réparent", "réparent"},
		{"réparer", "répar"},
		{"répareraient", "répar"},
		{"réparerait", "répar"},
		{"répareras", "répar"},
		{"reparlait", "reparl"},
		{"repartaient", "repart"},
		{"reparti", "repart"},
		{"repartie", "repart"},
		{"reparties", "repart"},
		{"repartir", "repart"},
		{"repartirent", "repart"},
		{"repartit", "repart"},
		{"répartition", "répartit"},
		{"repartons", "reparton"},
		{"reparu", "reparu"},
		{"reparurent", "reparurent"},
		{"reparut", "reparut"},
		{"repas", "rep"},
		{"repassa", "repass"},
		{"repassai", "repass"},
		{"repassait", "repass"},
		{"repassant", "repass"},
		{"repasser", "repass"},
		{"repasserait", "repass"},
		{"repens", "repen"},
		{"repensant", "repens"},
		{"repenser", "repens"},
		{"repentie", "repent"},
		{"repentir", "repent"},
		{"reperdre", "reperdr"},
		{"répéta", "répet"},
		{"répétaient", "répet"},
		{"répétait", "répet"},
		{"répétant", "répet"},
		{"répétât", "répet"},
		{"répète", "répet"},
		{"répété", "répet"},
		{"répétée", "répet"},
		{"répétées", "répet"},
		{"répètent", "répètent"},
		{"repeter", "repet"},
		{"répéter", "répet"},
		{"répétera", "répet"},
		{"répéterai", "répet"},
		{"répéterait", "répet"},
		{"répéterons", "répet"},
		{"répéteront", "répet"},
		{"répétez", "répet"},
		{"répétions", "répet"},
		{"répétiteur", "répétiteur"},
		{"répit", "rep"},
		{"replaça", "replac"},
		{"replaçant", "replac"},
		{"replacer", "replac"},
		{"repliée", "repli"},
		{"replièrent", "repli"},
		{"répliqua", "répliqu"},
		{"répliquait", "répliqu"},
		{"réplique", "répliqu"},
		{"répliquer", "répliqu"},
		{"répliquerai", "répliqu"},
		{"répliques", "répliqu"},
		{"replis", "repl"},
		{"répond", "répond"},
		{"répondaient", "répond"},
		{"répondais", "répond"},
		{"répondait", "répond"},
		{"répondant", "répond"},
		{"répondez", "répond"},
		{"répondiez", "répond"},
		{"répondirent", "répond"},
		{"répondit", "répond"},
		{"répondît", "répond"},
		{"répondra", "répondr"},
		{"répondrai", "répondr"},
		{"répondrais", "répondr"},
		{"répondrait", "répondr"},
		{"répondras", "répondr"},
		{"répondre", "répondr"},
		{"répondrez", "répondr"},
		{"réponds", "répond"},
		{"répondu", "répondu"},
		{"réponse", "répons"},
		{"réponses", "répons"},
		{"reportant", "report"},
		{"reporté", "report"},
		{"reporters", "reporter"},
		{"repos", "repos"},
		{"reposa", "repos"},
		{"reposait", "repos"},
		{"repose", "repos"},
		{"reposé", "repos"},
		{"reposer", "repos"},
		{"reposera", "repos"},
		{"reposoir", "reposoir"},
		{"reposoirs", "reposoir"},
		{"repoussa", "repouss"},
		{"repoussais", "repouss"},
		{"repoussait", "repouss"},
		{"repoussant", "repouss"},
		{"repousse", "repouss"},
		{"repoussé", "repouss"},
		{"repoussée", "repouss"},
		{"repoussées", "repouss"},
		{"repousser", "repouss"},
		{"repousserait", "repouss"},
		{"repoussés", "repouss"},
		{"repoussez", "repouss"},
		{"reprenaient", "repren"},
		{"reprenait", "repren"},
		{"reprenant", "repren"},
		{"reprend", "reprend"},
		{"reprendra", "reprendr"},
		{"reprendrai", "reprendr"},
		{"reprendrait", "reprendr"},
		{"reprendre", "reprendr"},
		{"reprendrons", "reprendron"},
		{"reprends", "reprend"},
		{"reprenez", "repren"},
		{"représenta", "représent"},
		{"représentaient", "représent"},
		{"représentait", "représent"},
		{"représentant", "représent"},
		{"représentation", "représent"},
		{"representations", "represent"},
		{"représentations", "représent"},
		{"représente", "représent"},
		{"représenté", "représent"},
		{"représenter", "représent"},
		{"réprimande", "réprimand"},
		{"réprimer", "réprim"},
		{"reprirent", "repr"},
		{"repris", "repr"},
		{"reprise", "repris"},
		{"reprises", "repris"},
		{"reprit", "repr"},
		{"reprît", "repr"},
		{"reprocha", "reproch"},
		{"reprochaient", "reproch"},
		{"reprochait", "reproch"},
		{"reprochant", "reproch"},
		{"reproche", "reproch"},
		{"reproché", "reproch"},
		{"reprochée", "reproch"},
		{"reprocher", "reproch"},
		{"reprochera", "reproch"},
		{"reprocherai", "reproch"},
		{"reprocherais", "reproch"},
		{"reproches", "reproch"},
		{"reprochons", "reprochon"},
		{"reproduire", "reproduir"},
		{"reproduisait", "reproduis"},
		{"reproduisent", "reproduisent"},
		{"reproduisirent", "reproduis"},
		{"reproduisit", "reproduis"},
		{"reproduit", "reproduit"},
		{"reprouvez", "reprouv"},
		{"reptation", "reptat"},
		{"reptiles", "reptil"},
		{"républicain", "républicain"},
		{"républicains", "républicain"},
		{"republican", "republican"},
		{"république", "républ"},
		{"républiques", "républ"},
		{"répudié", "répudi"},
		{"répugnait", "répugn"},
		{"répugnance", "répugn"},
		{"réputation", "réput"},
		{"réputations", "réput"},
		{"requiers", "requi"},
		{"requise", "requis"},
		{"réquisitionnaire", "réquisitionnair"},
		{"rescousse", "rescouss"},
		{"rescrit", "rescr"},
		{"réseau", "réseau"},
		{"resegon", "resegon"},
		{"resembleth", "resembleth"},
		{"réserva", "réserv"},
		{"réservait", "réserv"},
		{"réserve", "réserv"},
		{"réservé", "réserv"},
		{"réservée", "réserv"},
		{"réservées", "réserv"},
		{"réserver", "réserv"},
		{"réserves", "réserv"},
		{"réservés", "réserv"},
		{"réservoir", "réservoir"},
		{"résidais", "résid"},
		{"résidait", "résid"},
		{"résidence", "résident"},
		{"résigna", "résign"},
		{"résignation", "résign"},
		{"résigné", "résign"},
		{"résignée", "résign"},
		{"résigner", "résign"},
		{"résines", "résin"},
		{"résista", "résist"},
		{"résistaient", "résist"},
		{"résistait", "résist"},
		{"résistance", "résist"},
		{"résistant", "résist"},
		{"résiste", "résist"},
		{"résister", "résist"},
		{"résisterez", "résist"},
		{"résistes", "résist"},
		{"résolu", "résolu"},
		{"résolue", "résolu"},
		{"résolues", "résolu"},
		{"résolument", "résolu"},
		{"résolurent", "résolurent"},
		{"résolut", "résolut"},
		{"résolution", "résolu"},
		{"résolutions", "résolu"},
		{"résonnaient", "réson"},
		{"résoudre", "résoudr"},
		{"respect", "respect"},
		{"respecta", "respect"},
		{"respectable", "respect"},
		{"respectables", "respect"},
		{"respectais", "respect"},
		{"respectait", "respect"},
		{"respectant", "respect"},
		{"respectante", "respect"},
		{"respecte", "respect"},
		{"respecté", "respect"},
		{"respecter", "respect"},
		{"respecterais", "respect"},
		{"respectés", "respect"},
		{"respective", "respect"},
		{"respects", "respect"},
		{"respectueuse", "respectu"},
		{"respectueusement", "respectu"},
		{"respectueuses", "respectu"},
		{"respectueux", "respectu"},
		{"respira", "resp"},
		{"respiraient", "resp"},
		{"respirait", "resp"},
		{"respirant", "respir"},
		{"respiration", "respir"},
		{"respire", "respir"},
		{"respirer", "respir"},
		{"respirerai", "respir"},
		{"respirons", "resp"},
		{"resplendissant", "resplend"},
		{"resplendissante", "resplend"},
		{"resplendissent", "resplend"},
		{"responsable", "respons"},
		{"ressemblaient", "ressembl"},
		{"ressemblait", "ressembl"},
		{"ressemblance", "ressembl"},
		{"ressemblant", "ressembl"},
		{"ressemblât", "ressembl"},
		{"ressemble", "ressembl"},
		{"ressemblé", "ressembl"},
		{"ressemblent", "ressemblent"},
		{"ressembler", "ressembl"},
		{"ressemblera", "ressembl"},
		{"ressemblez", "ressembl"},
		{"ressens", "ressen"},
		{"ressentait", "ressent"},
		{"ressentir", "ressent"},
		{"ressentit", "ressent"},
		{"resserré", "resserr"},
		{"ressort", "ressort"},
		{"ressorti", "ressort"},
		{"ressortir", "ressort"},
		{"ressorts", "ressort"},
		{"ressource", "ressourc"},
		{"ressources", "ressourc"},
		{"ressuscité", "ressusc"},
		{"ressuscitèrent", "ressuscit"},
		{"resta", "rest"},
		{"restai", "rest"},
		{"restaient", "rest"},
		{"restait", "rest"},
		{"restant", "rest"},
		{"restassent", "rest"},
		{"restât", "rest"},
		{"restaurant", "restaur"},
		{"restaurants", "restaur"},
		{"restauration", "restaur"},
		{"restaurations", "restaur"},
		{"restauré", "restaur"},
		{"reste", "rest"},
		{"resté", "rest"},
		{"restée", "rest"},
		{"restées", "rest"},
		{"restent", "restent"},
		{"rester", "rest"},
		{"restera", "rest"},
		{"resterai", "rest"},
		{"resteraient", "rest"},
		{"resterais", "rest"},
		{"resterait", "rest"},
		{"resteras", "rest"},
		{"restèrent", "rest"},
		{"resterez", "rest"},
		{"resteront", "rest"},
		{"restes", "rest"},
		{"restés", "rest"},
		{"restez", "rest"},
		{"restiez", "rest"},
		{"restitué", "restitu"},
		{"restituée", "restitu"},
		{"restituit", "restituit"},
		{"restitution", "restitu"},
		{"restriction", "restrict"},
		{"résultait", "résult"},
		{"résultat", "résultat"},
		{"résultats", "résultat"},
		{"résumait", "résum"},
		{"résumé", "résum"},
		{"résumer", "résum"},
		{"résumons", "résumon"},
		{"résurrection", "résurrect"},
		{"rétabli", "rétabl"},
		{"rétablie", "rétabl"},
		{"rétablir", "rétabl"},
		{"rétabliras", "rétabl"},
		{"rétablissait", "rétabl"},
		{"rétablissement", "rétabl"},
		{"rétablit", "rétabl"},
		{"retard", "retard"},
		{"retardait", "retard"},
		{"retardant", "retard"},
		{"retardataire", "retardatair"},
		{"retarde", "retard"},
		{"retardé", "retard"},
		{"retardée", "retard"},
		{"retardent", "retardent"},
		{"retarder", "retard"},
		{"retardez", "retard"},
		{"retards", "retard"},
		{"retenaient", "reten"},
		{"retenait", "reten"},
		{"retenant", "reten"},
		{"retenez", "reten"},
		{"retenir", "reten"},
		{"retenti", "retent"},
		{"retentir", "retent"},
		{"retentiraient", "retent"},
		{"retentirait", "retent"},
		{"retentirent", "retent"},
		{"retentissaient", "retent"},
		{"retentissait", "retent"},
		{"retentissant", "retent"},
		{"retentissement", "retent"},
		{"retentit", "retent"},
		{"retenu", "retenu"},
		{"retenue", "retenu"},
		{"retenues", "retenu"},
		{"retenus", "retenus"},
		{"retiendrai", "retiendr"},
		{"retienne", "retien"},
		{"retiennent", "retiennent"},
		{"retient", "retient"},
		{"retinrent", "retinrent"},
		{"retint", "retint"},
		{"retira", "ret"},
		{"retiraient", "ret"},
		{"retirait", "ret"},
		{"retirant", "retir"},
		{"retirât", "retir"},
		{"retire", "retir"},
		{"retiré", "retir"},
		{"retirée", "retir"},
		{"retirées", "retir"},
		{"retirer", "retir"},
		{"retirerais", "retir"},
		{"retirerait", "retir"},
		{"retirèrent", "retir"},
		{"retirerez", "retir"},
		{"retirés", "retir"},
		{"retirez", "ret"},
		{"retirons", "ret"},
		{"retomba", "retomb"},
		{"retombait", "retomb"},
		{"retombant", "retomb"},
		{"retombe", "retomb"},
		{"retombée", "retomb"},
		{"retomber", "retomb"},
		{"retomberait", "retomb"},
		{"retombes", "retomb"},
		{"retour", "retour"},
		{"retourna", "retourn"},
		{"retournaient", "retourn"},
		{"retournait", "retourn"},
		{"retournant", "retourn"},
		{"retourne", "retourn"},
		{"retourné", "retourn"},
		{"retourner", "retourn"},
		{"retournèrent", "retourn"},
		{"retournez", "retourn"},
		{"retracer", "retrac"},
		{"rétracte", "rétract"},
		{"retraite", "retrait"},
		{"retraites", "retrait"},
		{"retrancher", "retranch"},
		{"retranchés", "retranch"},
		{"rétréci", "rétrec"},
		{"rétrécir", "rétrec"},
		{"rétrécissait", "rétrec"},
		{"retrempait", "retremp"},
		{"retrempé", "retremp"},
		{"rétribution", "rétribu"},
		{"rétrospectif", "rétrospect"},
		{"retrouva", "retrouv"},
		{"retrouvaient", "retrouv"},
		{"retrouvait", "retrouv"},
		{"retrouvant", "retrouv"},
		{"retrouvât", "retrouv"},
		{"retrouve", "retrouv"},
		{"retrouvé", "retrouv"},
		{"retrouvée", "retrouv"},
		{"retrouver", "retrouv"},
		{"retrouvera", "retrouv"},
		{"retrouverai", "retrouv"},
		{"retrouvèrent", "retrouv"},
		{"retrouverez", "retrouv"},
		{"retrouverons", "retrouv"},
		{"retz", "retz"},
		{"réuni", "réun"},
		{"réunie", "réun"},
		{"réunies", "réun"},
		{"réunion", "réunion"},
		{"réunir", "réun"},
		{"réunirent", "réun"},
		{"réuniront", "réun"},
		{"réunis", "réun"},
		{"réunissaient", "réun"},
		{"réunissait", "réun"},
		{"réunissant", "réun"},
		{"réunissent", "réun"},
		{"réunissez", "réun"},
		{"réunit", "réun"},
		{"réussi", "réuss"},
		{"réussir", "réuss"},
		{"réussirais", "réuss"},
		{"réussis", "réuss"},
		{"réussissait", "réuss"},
		{"réussissez", "réuss"},
		{"réussit", "réuss"},
		{"réussite", "réussit"},
		{"rêva", "rêv"},
		{"rêvait", "rêv"},
		{"revanche", "revanch"},
		{"rêvant", "rêv"},
		{"revaudrai", "revaudr"},
		{"rêve", "rêv"},
		{"rêvé", "rêv"},
		{"réveil", "réveil"},
		{"réveilla", "réveil"},
		{"réveillait", "réveil"},
		{"réveillant", "réveil"},
		{"réveillât", "réveil"},
		{"réveille", "réveil"},
		{"réveillé", "réveil"},
		{"réveillée", "réveil"},
		{"réveiller", "réveil"},
		{"réveillèrent", "réveil"},
		{"réveillés", "réveil"},
		{"révéla", "rével"},
		{"révélait", "rével"},
		{"révélation", "rével"},
		{"révélé", "rével"},
		{"révéler", "rével"},
		{"revenaient", "reven"},
		{"revenait", "reven"},
		{"revenant", "reven"},
		{"revendeur", "revendeur"},
		{"revendre", "revendr"},
		{"revenez", "reven"},
		{"revenir", "reven"},
		{"revenons", "revenon"},
		{"revenu", "revenu"},
		{"revenue", "revenu"},
		{"revenus", "revenus"},
		{"rêver", "rêv"},
		{"révéré", "réver"},
		{"révérence", "révérent"},
		{"révérences", "révérent"},
		{"révérencieuses", "révérenci"},
		{"révérend", "révérend"},
		{"révérendissime", "révérendissim"},
		{"rêverie", "rêver"},
		{"rêveries", "rêver"},
		{"reverrai", "reverr"},
		{"reverrais", "reverr"},
		{"reverrait", "reverr"},
		{"reverrez", "reverr"},
		{"reverrons", "reverron"},
		{"reverront", "reverront"},
		{"revers", "rever"},
		{"rêves", "rêv"},
		{"revêtir", "revêt"},
		{"revêtu", "revêtu"},
		{"revêtus", "revêtus"},
		{"rêveur", "rêveur"},
		{"rêveuse", "rêveux"},
		{"rêvez", "rêv"},
		{"reviendra", "reviendr"},
		{"reviendrai", "reviendr"},
		{"reviendrais", "reviendr"},
		{"reviendrait", "reviendr"},
		{"reviendrez", "reviendr"},
		{"reviendront", "reviendront"},
		{"revienne", "revien"},
		{"reviennent", "reviennent"},
		{"reviens", "revien"},
		{"revient", "revient"},
		{"review", "review"},
		{"revinrent", "revinrent"},
		{"revinssent", "revinssent"},
		{"revint", "revint"},
		{"revirement", "revir"},
		{"revirent", "rev"},
		{"revissent", "rev"},
		{"revit", "rev"},
		{"revivaient", "reviv"},
		{"revoie", "revoi"},
		{"revoies", "revoi"},
		{"revoir", "revoir"},
		{"revois", "revois"},
		{"révolta", "révolt"},
		{"révoltante", "révolt"},
		{"révoltantes", "révolt"},
		{"révoltât", "révolt"},
		{"révolte", "révolt"},
		{"révolté", "révolt"},
		{"révoltée", "révolt"},
		{"révolter", "révolt"},
		{"révolues", "révolu"},
		{"révolus", "révolus"},
		{"révolution", "révolu"},
		{"révolutionnaire", "révolutionnair"},
		{"révolutionnaires", "révolutionnair"},
		{"révolutions", "révolu"},
		{"revolver", "revolv"},
		{"revolvers", "revolver"},
		{"révoquée", "révoqu"},
		{"révoquerai", "révoqu"},
		{"révoquerez", "révoqu"},
		{"revoyait", "revoi"},
		{"revoyant", "revoi"},
		{"revu", "revu"},
		{"revue", "revu"},
		{"revues", "revu"},
		{"rez", "rez"},
		{"rhin", "rhin"},
		{"rhône", "rhôn"},
		{"rhubarbe", "rhubarb"},
		{"rhume", "rhum"},
		{"ri", "ri"},
		{"riaient", "ri"},
		{"riais", "ri"},
		{"riait", "ri"},
		{"riant", "ri"},
		{"riante", "ri"},
		{"riants", "ri"},
		{"ricanant", "rican"},
		{"ricanement", "rican"},
		{"ricciarda", "ricciard"},
		{"richards", "richard"},
		{"riche", "rich"},
		{"richelieu", "richelieu"},
		{"richement", "rich"},
		{"richemond", "richemond"},
		{"riches", "rich"},
		{"richesse", "richess"},
		{"richesses", "richess"},
		{"richissime", "richissim"},
		{"rideau", "rideau"},
		{"rideaux", "rideau"},
		{"rides", "rid"},
		{"ridicule", "ridicul"},
		{"ridicules", "ridicul"},
		{"rie", "ri"},
		{"rien", "rien"},
		{"riens", "rien"},
		{"rient", "rient"},
		{"rienzi", "rienz"},
		{"rigides", "rigid"},
		{"rigoureuse", "rigour"},
		{"rigoureusement", "rigour"},
		{"rigoureux", "rigour"},
		{"rigueur", "rigueur"},
		{"rigueurs", "rigueur"},
		{"rimailleur", "rimailleur"},
		{"rimes", "rim"},
		{"rios", "rios"},
		{"riposta", "ripost"},
		{"ripostaient", "ripost"},
		{"rire", "rir"},
		{"rires", "rir"},
		{"ris", "ris"},
		{"riscara", "riscar"},
		{"risqua", "risqu"},
		{"risquaient", "risqu"},
		{"risque", "risqu"},
		{"risqué", "risqu"},
		{"risquer", "risqu"},
		{"risquerai", "risqu"},
		{"risques", "risqu"},
		{"risquiez", "risqu"},
		{"risquons", "risquon"},
		{"rit", "rit"},
		{"rît", "rît"},
		{"ritournelle", "ritournel"},
		{"riva", "riv"},
		{"rivage", "rivag"},
		{"rival", "rival"},
		{"rivale", "rival"},
		{"rivales", "rival"},
		{"rivalité", "rival"},
		{"rivarol", "rivarol"},
		{"rivaux", "rival"},
		{"rive", "riv"},
		{"river", "riv"},
		{"riverisco", "riverisco"},
		{"rives", "riv"},
		{"rivière", "rivi"},
		{"rivières", "rivi"},
		{"rivoli", "rivol"},
		{"riz", "riz"},
		{"rizières", "rizi"},
		{"road", "road"},
		{"roads", "road"},
		{"roastbeef", "roastbeef"},
		{"robe", "rob"},
		{"robert", "robert"},
		{"robes", "rob"},
		{"robespierre", "robespierr"},
		{"robre", "robr"},
		{"robres", "robr"},
		{"robuste", "robust"},
		{"robustes", "robust"},
		{"roc", "roc"},
		{"roch", "roch"},
		{"rocher", "roch"},
		{"rochers", "rocher"},
		{"roches", "roch"},
		{"rocheuses", "rocheux"},
		{"rock", "rock"},
		{"rocky", "rocky"},
		{"roclin", "roclin"},
		{"rôdait", "rôd"},
		{"rôdant", "rôd"},
		{"rôder", "rôd"},
		{"rogations", "rogat"},
		{"rogné", "rogn"},
		{"rognure", "rognur"},
		{"rogue", "rogu"},
		{"roguerie", "roguer"},
		{"roi", "roi"},
		{"rois", "rois"},
		{"roiville", "roivill"},
		{"roland", "roland"},
		{"rôle", "rôl"},
		{"rôles", "rôl"},
		{"romagnan", "romagnan"},
		{"romagnano", "romagnano"},
		{"romagne", "romagn"},
		{"romain", "romain"},
		{"romaine", "romain"},
		{"romains", "romain"},
		{"roman", "roman"},
		{"romanesque", "romanesqu"},
		{"romanesques", "romanesqu"},
		{"romans", "roman"},
		{"rome", "rom"},
		{"rompe", "romp"},
		{"rompez", "romp"},
		{"rompit", "romp"},
		{"rompre", "rompr"},
		{"rompt", "rompt"},
		{"rompu", "rompu"},
		{"rompus", "rompus"},
		{"ronces", "ronc"},
		{"ronceux", "ronceux"},
		{"rond", "rond"},
		{"ronde", "rond"},
		{"rondement", "rond"},
		{"rondes", "rond"},
		{"ronds", "rond"},
		{"ronflait", "ronfl"},
		{"ronflantes", "ronfl"},
		{"ronfle", "ronfl"},
		{"ronflement", "ronfl"},
		{"ronflements", "ronfl"},
		{"rongeait", "rong"},
		{"ronsard", "ronsard"},
		{"room", "room"},
		{"rooms", "room"},
		{"rose", "ros"},
		{"roseau", "roseau"},
		{"roseaux", "roseau"},
		{"rosée", "ros"},
		{"roses", "ros"},
		{"rosier", "rosi"},
		{"rosiers", "rosi"},
		{"rosse", "ross"},
		{"rossé", "ross"},
		{"rosses", "ross"},
		{"rossini", "rossin"},
		{"rothal", "rothal"},
		{"rothschild", "rothschild"},
		{"rôti", "rôt"},
		{"rôtie", "rôt"},
		{"rôties", "rôt"},
		{"rotrou", "rotrou"},
		{"roue", "rou"},
		{"roué", "rou"},
		{"rouer", "rou"},
		{"roueries", "rouer"},
		{"roues", "rou"},
		{"rouffles", "rouffl"},
		{"rouge", "roug"},
		{"rougeâtre", "rougeâtr"},
		{"rouges", "roug"},
		{"rougeur", "rougeur"},
		{"rougi", "roug"},
		{"rougir", "roug"},
		{"rougira", "roug"},
		{"rougirait", "roug"},
		{"rougissait", "roug"},
		{"rougissant", "roug"},
		{"rougit", "roug"},
		{"rouille", "rouill"},
		{"rouillé", "rouill"},
		{"roula", "roul"},
		{"roulaient", "roul"},
		{"roulait", "roul"},
		{"roulant", "roul"},
		{"roule", "roul"},
		{"rouleau", "rouleau"},
		{"roulement", "roul"},
		{"roulèrent", "roul"},
		{"roulis", "roul"},
		{"rousseau", "rousseau"},
		{"route", "rout"},
		{"routes", "rout"},
		{"rouvert", "rouvert"},
		{"rouvray", "rouvray"},
		{"rouvrit", "rouvr"},
		{"rovere", "rover"},
		{"rovère", "rover"},
		{"row", "row"},
		{"rowan", "rowan"},
		{"royal", "royal"},
		{"royale", "royal"},
		{"royaliste", "royal"},
		{"royalistes", "royal"},
		{"royaume", "royaum"},
		{"ruban", "ruban"},
		{"rubempré", "rubempr"},
		{"rubigneau", "rubigneau"},
		{"rude", "rud"},
		{"rudement", "rud"},
		{"rudes", "rud"},
		{"rudesse", "rudess"},
		{"rue", "ru"},
		{"rues", "ru"},
		{"ruga", "rug"},
		{"rugissements", "rug"},
		{"ruina", "ruin"},
		{"ruinait", "ruin"},
		{"ruine", "ruin"},
		{"ruiné", "ruin"},
		{"ruinée", "ruin"},
		{"ruinées", "ruin"},
		{"ruinent", "ruinent"},
		{"ruiner", "ruin"},
		{"ruines", "ruin"},
		{"ruinés", "ruin"},
		{"ruineux", "ruineux"},
		{"ruinez", "ruin"},
		{"ruisseau", "ruisseau"},
		{"ruisseaux", "ruisseau"},
		{"ruminait", "rumin"},
		{"ruminants", "rumin"},
		{"rupture", "ruptur"},
		{"rus", "rus"},
		{"rusca", "rusc"},
		{"ruse", "rus"},
		{"rusé", "rus"},
		{"russe", "russ"},
		{"russell", "russel"},
		{"russes", "russ"},
		{"russie", "russ"},
		{"s", "s"},
		{"sa", "sa"},
		{"sablé", "sabl"},
		{"sablées", "sabl"},
		{"sabre", "sabr"},
		{"sabré", "sabr"},
		{"sabrer", "sabr"},
		{"sabres", "sabr"},
		{"sabrés", "sabr"},
		{"sac", "sac"},
		{"sacca", "sacc"},
		{"saccadés", "saccad"},
		{"sachant", "sach"},
		{"sache", "sach"},
		{"saches", "sach"},
		{"sachez", "sach"},
		{"sachiez", "sach"},
		{"sachons", "sachon"},
		{"sacramentelle", "sacramentel"},
		{"sacramento", "sacramento"},
		{"sacre", "sacr"},
		{"sacré", "sacr"},
		{"sacrédié", "sacrédi"},
		{"sacrée", "sacr"},
		{"sacrées", "sacr"},
		{"sacrement", "sacr"},
		{"sacrements", "sacr"},
		{"sacrés", "sacr"},
		{"sacrifiait", "sacrifi"},
		{"sacrifiant", "sacrifi"},
		{"sacrifice", "sacrific"},
		{"sacrifices", "sacrific"},
		{"sacrifie", "sacrif"},
		{"sacrifié", "sacrifi"},
		{"sacrifiées", "sacrifi"},
		{"sacrifier", "sacrifi"},
		{"sacrifierait", "sacrifi"},
		{"sacrifiés", "sacrifi"},
		{"sacrifiez", "sacrif"},
		{"sacrilège", "sacrileg"},
		{"sacristie", "sacrist"},
		{"saddle", "saddl"},
		{"sagacité", "sagac"},
		{"sage", "sag"},
		{"sagement", "sag"},
		{"sages", "sag"},
		{"sagesse", "sagess"},
		{"sagoutiers", "sagouti"},
		{"saïd", "saïd"},
		{"saigna", "saign"},
		{"saignait", "saign"},
		{"saigné", "saign"},
		{"saignées", "saign"},
		{"saigner", "saign"},
		{"saillant", "saill"},
		{"saillante", "saill"},
		{"saillantes", "saill"},
		{"saillants", "saill"},
		{"saillie", "saill"},
		{"saillies", "saill"},
		{"sain", "sain"},
		{"sainclair", "sainclair"},
		{"saine", "sain"},
		{"sains", "sain"},
		{"saint", "saint"},
		{"sainte", "saint"},
		{"saintes", "saint"},
		{"sainteté", "saintet"},
		{"saints", "saint"},
		{"sais", "sais"},
		{"saisi", "sais"},
		{"saisie", "sais"},
		{"saisines", "saisin"},
		{"saisir", "sais"},
		{"saisirais", "sais"},
		{"saisirent", "sais"},
		{"saisis", "sais"},
		{"saisissait", "sais"},
		{"saisissant", "sais"},
		{"saisissante", "sais"},
		{"saisissement", "sais"},
		{"saisissez", "sais"},
		{"saisit", "sais"},
		{"saisîtes", "sais"},
		{"saison", "saison"},
		{"sait", "sait"},
		{"saki", "sak"},
		{"salaire", "salair"},
		{"salanganes", "salangan"},
		{"salcette", "salcet"},
		{"sale", "sal"},
		{"salé", "sal"},
		{"salement", "sal"},
		{"sales", "sal"},
		{"saleté", "salet"},
		{"salis", "sal"},
		{"salle", "sall"},
		{"salles", "sall"},
		{"salomon", "salomon"},
		{"salon", "salon"},
		{"salons", "salon"},
		{"salt", "salt"},
		{"saltimbanques", "saltimbanqu"},
		{"salua", "salu"},
		{"saluaient", "salu"},
		{"saluait", "salu"},
		{"saluant", "salu"},
		{"salue", "salu"},
		{"salué", "salu"},
		{"saluer", "salu"},
		{"saluèrent", "salu"},
		{"salure", "salur"},
		{"salut", "salut"},
		{"salutaire", "salutair"},
		{"samedi", "samed"},
		{"samedis", "samed"},
		{"samuel", "samuel"},
		{"san", "san"},
		{"sancho", "sancho"},
		{"sanctifie", "sanctif"},
		{"sanctuaire", "sanctuair"},
		{"sanctus", "sanctus"},
		{"sandales", "sandal"},
		{"sandolaro", "sandolaro"},
		{"sandrino", "sandrino"},
		{"sandy", "sandy"},
		{"sang", "sang"},
		{"sanglant", "sangl"},
		{"sanglante", "sangl"},
		{"sanglier", "sangli"},
		{"sanglot", "sanglot"},
		{"sanglotaient", "sanglot"},
		{"sanglotant", "sanglot"},
		{"sanglots", "sanglot"},
		{"sangsues", "sangsu"},
		{"sanguigna", "sanguign"},
		{"sanguinaire", "sanguinair"},
		{"sannazaro", "sannazaro"},
		{"sans", "san"},
		{"sanseverina", "sanseverin"},
		{"santal", "santal"},
		{"santé", "sant"},
		{"saper", "sap"},
		{"sapin", "sapin"},
		{"sapins", "sapin"},
		{"sarà", "sarà"},
		{"saragosse", "saragoss"},
		{"sarasine", "sarasin"},
		{"sarcasmes", "sarcasm"},
		{"sardonique", "sardon"},
		{"sarono", "sarono"},
		{"sat", "sat"},
		{"satiété", "satiet"},
		{"satin", "satin"},
		{"satirique", "satir"},
		{"satiriques", "satir"},
		{"satisfaction", "satisfact"},
		{"satisfaire", "satisfair"},
		{"satisfaisant", "satisfais"},
		{"satisfait", "satisf"},
		{"satisfera", "satisf"},
		{"saturaient", "satur"},
		{"saturée", "satur"},
		{"sauce", "sauc"},
		{"saucisses", "sauc"},
		{"sauders", "sauder"},
		{"sauf", "sauf"},
		{"saugrenues", "saugrenu"},
		{"saule", "saul"},
		{"saules", "saul"},
		{"saura", "saur"},
		{"saurai", "saur"},
		{"sauraient", "saur"},
		{"saurais", "saur"},
		{"saurait", "saur"},
		{"saurez", "saur"},
		{"sauriez", "saur"},
		{"saurons", "sauron"},
		{"sauront", "sauront"},
		{"saut", "saut"},
		{"sauta", "saut"},
		{"sautait", "saut"},
		{"sautant", "saut"},
		{"saute", "saut"},
		{"sauté", "saut"},
		{"sauter", "saut"},
		{"sauterai", "saut"},
		{"sauterait", "saut"},
		{"sautèrent", "saut"},
		{"sautés", "saut"},
		{"sauteur", "sauteur"},
		{"sauteuse", "sauteux"},
		{"sautillant", "sautill"},
		{"sautoir", "sautoir"},
		{"sauts", "saut"},
		{"sauva", "sauv"},
		{"sauvage", "sauvag"},
		{"sauvages", "sauvag"},
		{"sauvait", "sauv"},
		{"sauvant", "sauv"},
		{"sauve", "sauv"},
		{"sauvé", "sauv"},
		{"sauvée", "sauv"},
		{"sauvegarder", "sauvegard"},
		{"sauvent", "sauvent"},
		{"sauver", "sauv"},
		{"sauvera", "sauv"},
		{"sauverai", "sauv"},
		{"sauveraient", "sauv"},
		{"sauveront", "sauv"},
		{"sauvés", "sauv"},
		{"sauveur", "sauveur"},
		{"sauveurs", "sauveur"},
		{"sauvez", "sauv"},
		{"sauvions", "sauvion"},
		{"sauvons", "sauvon"},
		{"savaient", "sav"},
		{"savais", "sav"},
		{"savait", "sav"},
		{"savamment", "sav"},
		{"savant", "sav"},
		{"savante", "sav"},
		{"savantes", "sav"},
		{"savants", "sav"},
		{"savent", "savent"},
		{"saveur", "saveur"},
		{"savez", "sav"},
		{"saviez", "sav"},
		{"saville", "savill"},
		{"savoir", "savoir"},
		{"savons", "savon"},
		{"saxe", "sax"},
		{"saxon", "saxon"},
		{"saxons", "saxon"},
		{"sbires", "sbir"},
		{"scabreuses", "scabreux"},
		{"scabreux", "scabreux"},
		{"scagliola", "scagliol"},
		{"scala", "scal"},
		{"scalpent", "scalpent"},
		{"scandale", "scandal"},
		{"scandaleuse", "scandal"},
		{"scandalisaient", "scandalis"},
		{"scandalisé", "scandalis"},
		{"scandalisée", "scandalis"},
		{"scandalisera", "scandalis"},
		{"scandalisés", "scandalis"},
		{"scandinavie", "scandinav"},
		{"sceau", "sceau"},
		{"sceaux", "sceau"},
		{"scélérat", "scélérat"},
		{"scélératesse", "scélératess"},
		{"scélérats", "scélérat"},
		{"scellé", "scel"},
		{"scène", "scen"},
		{"scènes", "scen"},
		{"schiller", "schill"},
		{"schuyler", "schuyl"},
		{"sciant", "sci"},
		{"scie", "sci"},
		{"scié", "sci"},
		{"sciemment", "scient"},
		{"science", "scienc"},
		{"sciences", "scienc"},
		{"scier", "sci"},
		{"scierie", "scier"},
		{"scies", "sci"},
		{"scieur", "scieur"},
		{"scintillants", "scintill"},
		{"scintillations", "scintill"},
		{"scotland", "scotland"},
		{"scott", "scott"},
		{"scotti", "scott"},
		{"scrupule", "scrupul"},
		{"scrupules", "scrupul"},
		{"scrupuleuse", "scrupul"},
		{"scrupuleusement", "scrupul"},
		{"scrutateur", "scrutateur"},
		{"sculptée", "sculpt"},
		{"sculptées", "sculpt"},
		{"sculptés", "sculpt"},
		{"sculpteur", "sculpteur"},
		{"sculpteurs", "sculpteur"},
		{"se", "se"},
		{"séance", "séanc"},
		{"sec", "sec"},
		{"secatore", "secator"},
		{"sèche", "sech"},
		{"séché", "sech"},
		{"séchée", "sech"},
		{"sèchement", "sech"},
		{"sécher", "sech"},
		{"sécheresse", "sécheress"},
		{"sèches", "sech"},
		{"second", "second"},
		{"secondaient", "second"},
		{"seconde", "second"},
		{"secondé", "second"},
		{"secondent", "secondent"},
		{"seconderait", "second"},
		{"secondes", "second"},
		{"secoua", "secou"},
		{"secouaient", "secou"},
		{"secouait", "secou"},
		{"secouant", "secou"},
		{"secoue", "secou"},
		{"secouée", "secou"},
		{"secoués", "secou"},
		{"secourable", "secour"},
		{"secourez", "secour"},
		{"secourir", "secour"},
		{"secours", "secour"},
		{"secouru", "secouru"},
		{"secousse", "secouss"},
		{"secousses", "secouss"},
		{"secret", "secret"},
		{"secrétaire", "secrétair"},
		{"secrétaires", "secrétair"},
		{"secrète", "secret"},
		{"secrètement", "secret"},
		{"secrètes", "secret"},
		{"secrets", "secret"},
		{"secs", "sec"},
		{"sectaire", "sectair"},
		{"sectateur", "sectateur"},
		{"sectateurs", "sectateur"},
		{"secte", "sect"},
		{"section", "section"},
		{"séculaires", "séculair"},
		{"secundo", "secundo"},
		{"sécurité", "sécur"},
		{"sédentaire", "sédentair"},
		{"sedgwick", "sedgwick"},
		{"sediola", "sediol"},
		{"séditieuse", "séditi"},
		{"séditieux", "séditi"},
		{"sédition", "sédit"},
		{"séducteur", "séducteur"},
		{"séduction", "séduct"},
		{"séductions", "séduct"},
		{"séduire", "séduir"},
		{"séduisait", "séduis"},
		{"séduisant", "séduis"},
		{"séduisante", "séduis"},
		{"séduisants", "séduis"},
		{"séduit", "séduit"},
		{"séduite", "séduit"},
		{"séduits", "séduit"},
		{"segreto", "segreto"},
		{"seigneur", "seigneur"},
		{"seigneuries", "seigneur"},
		{"seigneurs", "seigneur"},
		{"sein", "sein"},
		{"seine", "sein"},
		{"seize", "seiz"},
		{"séjour", "séjour"},
		{"séjournait", "séjourn"},
		{"séjournât", "séjourn"},
		{"séjourner", "séjourn"},
		{"sel", "sel"},
		{"sella", "sel"},
		{"selle", "sel"},
		{"sellé", "sel"},
		{"sellette", "sellet"},
		{"selon", "selon"},
		{"sema", "sem"},
		{"semaine", "semain"},
		{"semaines", "semain"},
		{"sembla", "sembl"},
		{"semblable", "semblabl"},
		{"semblables", "semblabl"},
		{"semblaient", "sembl"},
		{"semblait", "sembl"},
		{"semblant", "sembl"},
		{"semblât", "sembl"},
		{"semble", "sembl"},
		{"semblé", "sembl"},
		{"semblent", "semblent"},
		{"sembler", "sembl"},
		{"semblera", "sembl"},
		{"semblerait", "sembl"},
		{"semblèrent", "sembl"},
		{"semblez", "sembl"},
		{"sème", "sem"},
		{"semé", "sem"},
		{"semée", "sem"},
		{"semées", "sem"},
		{"semelle", "semel"},
		{"semelles", "semel"},
		{"semer", "sem"},
		{"semés", "sem"},
		{"semez", "sem"},
		{"semi", "sem"},
		{"sémillant", "sémill"},
		{"sémillante", "sémill"},
		{"séminaire", "séminair"},
		{"séminariste", "séminar"},
		{"séminaristes", "séminar"},
		{"sempans", "sempan"},
		{"sens", "sen"},
		{"sensation", "sensat"},
		{"sensations", "sensat"},
		{"sensée", "sens"},
		{"sensés", "sens"},
		{"sensibilité", "sensibil"},
		{"sensible", "sensibl"},
		{"sensiblement", "sensibl"},
		{"sensibles", "sensibl"},
		{"sent", "sent"},
		{"sentaient", "sent"},
		{"sentais", "sent"},
		{"sentait", "sent"},
		{"sentant", "sent"},
		{"sentence", "sentenc"},
		{"sentences", "sentenc"},
		{"sentent", "sentent"},
		{"sentez", "sent"},
		{"senti", "sent"},
		{"sentie", "sent"},
		{"sentier", "senti"},
		{"sentiers", "senti"},
		{"senties", "sent"},
		{"sentiment", "sent"},
		{"sentimentale", "sentimental"},
		{"sentimentales", "sentimental"},
		{"sentiments", "sent"},
		{"sentinelle", "sentinel"},
		{"sentinelles", "sentinel"},
		{"sentir", "sent"},
		{"sentira", "sent"},
		{"sentirez", "sent"},
		{"sentisses", "sent"},
		{"sentit", "sent"},
		{"sépara", "sépar"},
		{"séparaient", "sépar"},
		{"séparait", "sépar"},
		{"séparant", "sépar"},
		{"séparation", "sépar"},
		{"sépare", "sépar"},
		{"séparé", "sépar"},
		{"séparée", "sépar"},
		{"séparées", "sépar"},
		{"séparent", "séparent"},
		{"séparer", "sépar"},
		{"séparera", "sépar"},
		{"séparés", "sépar"},
		{"séparons", "séparon"},
		{"sept", "sept"},
		{"septembre", "septembr"},
		{"septentrional", "septentrional"},
		{"septentrionale", "septentrional"},
		{"septentrionales", "septentrional"},
		{"septième", "septiem"},
		{"sequin", "sequin"},
		{"sequins", "sequin"},
		{"sera", "ser"},
		{"serai", "ser"},
		{"seraient", "ser"},
		{"sérail", "sérail"},
		{"serais", "ser"},
		{"serait", "ser"},
		{"seras", "ser"},
		{"sérénade", "sérénad"},
		{"sérénades", "sérénad"},
		{"sérénissime", "sérénissim"},
		{"sérénité", "séren"},
		{"serez", "ser"},
		{"sergent", "sergent"},
		{"série", "ser"},
		{"sérieuse", "sérieux"},
		{"serieusement", "serieux"},
		{"sérieusement", "sérieux"},
		{"sérieuses", "sérieux"},
		{"sérieux", "sérieux"},
		{"seriez", "ser"},
		{"serions", "serion"},
		{"serious", "serious"},
		{"serment", "serment"},
		{"serments", "serment"},
		{"sermon", "sermon"},
		{"sermons", "sermon"},
		{"serons", "seron"},
		{"seront", "seront"},
		{"serpent", "serpent"},
		{"serpentent", "serpentent"},
		{"serpents", "serpent"},
		{"serpolet", "serpolet"},
		{"serra", "serr"},
		{"serraient", "serr"},
		{"serrait", "serr"},
		{"serrant", "serr"},
		{"serre", "serr"},
		{"serré", "serr"},
		{"serrée", "serr"},
		{"serrées", "serr"},
		{"serrements", "serr"},
		{"serrent", "serrent"},
		{"serrer", "serr"},
		{"serrèrent", "serr"},
		{"serres", "serr"},
		{"serrés", "serr"},
		{"serrure", "serrur"},
		{"serrures", "serrur"},
		{"serrurier", "serruri"},
		{"sert", "sert"},
		{"servaient", "serv"},
		{"servais", "serv"},
		{"servait", "serv"},
		{"servant", "serv"},
		{"servante", "serv"},
		{"serve", "serv"},
		{"servent", "servent"},
		{"servi", "serv"},
		{"serviable", "serviabl"},
		{"service", "servic"},
		{"services", "servic"},
		{"servie", "serv"},
		{"serviettes", "serviet"},
		{"servilement", "servil"},
		{"servir", "serv"},
		{"servira", "serv"},
		{"servirai", "serv"},
		{"serviraient", "serv"},
		{"servirait", "serv"},
		{"servirez", "serv"},
		{"servis", "serv"},
		{"servit", "serv"},
		{"serviteur", "serviteur"},
		{"serviteurs", "serviteur"},
		{"ses", "se"},
		{"session", "session"},
		{"seuil", "seuil"},
		{"seul", "seul"},
		{"seule", "seul"},
		{"seulement", "seul"},
		{"seules", "seul"},
		{"seuls", "seul"},
		{"sévère", "séver"},
		{"sévèrement", "séver"},
		{"sévères", "séver"},
		{"sévérité", "séver"},
		{"sévérités", "séver"},
		{"sexagésimales", "sexagésimal"},
		{"sexe", "sex"},
		{"sexes", "sex"},
		{"sfondrata", "sfondrat"},
		{"sforce", "sforc"},
		{"sforze", "sforz"},
		{"shako", "shako"},
		{"shakos", "shakos"},
		{"shakspeare", "shakspear"},
		{"shangaï", "shangaï"},
		{"sheppard", "sheppard"},
		{"sheridan", "sheridan"},
		{"sherry", "sherry"},
		{"shilling", "shilling"},
		{"shillings", "shilling"},
		{"shipping", "shipping"},
		{"shiva", "shiv"},
		{"shows", "show"},
		{"si", "si"},
		{"siamo", "siamo"},
		{"siècle", "siecl"},
		{"siècles", "siecl"},
		{"siège", "sieg"},
		{"sièges", "sieg"},
		{"sien", "sien"},
		{"sienne", "sien"},
		{"siennes", "sien"},
		{"siens", "sien"},
		{"sierra", "sierr"},
		{"sieste", "siest"},
		{"sieur", "sieur"},
		{"sieyès", "sieyes"},
		{"siffla", "siffl"},
		{"sifflaient", "siffl"},
		{"sifflait", "siffl"},
		{"sifflant", "siffl"},
		{"sifflement", "siffl"},
		{"sifflements", "siffl"},
		{"siffler", "siffl"},
		{"sifflet", "sifflet"},
		{"sifflets", "sifflet"},
		{"sighs", "sigh"},
		{"sigisbée", "sigisb"},
		{"sigismond", "sigismond"},
		{"signa", "sign"},
		{"signait", "sign"},
		{"signal", "signal"},
		{"signala", "signal"},
		{"signale", "signal"},
		{"signalé", "signal"},
		{"signalée", "signal"},
		{"signalement", "signal"},
		{"signalements", "signal"},
		{"signaler", "signal"},
		{"signalerait", "signal"},
		{"signant", "sign"},
		{"signature", "signatur"},
		{"signaux", "signal"},
		{"signe", "sign"},
		{"signé", "sign"},
		{"signée", "sign"},
		{"signées", "sign"},
		{"signer", "sign"},
		{"signera", "sign"},
		{"signerai", "sign"},
		{"signes", "sign"},
		{"signets", "signet"},
		{"signez", "sign"},
		{"signifiait", "signifi"},
		{"significatif", "signif"},
		{"signification", "signif"},
		{"significative", "signif"},
		{"significatives", "signif"},
		{"signifie", "signif"},
		{"signifier", "signifi"},
		{"signor", "signor"},
		{"signora", "signor"},
		{"signorino", "signorino"},
		{"silence", "silenc"},
		{"silencieuse", "silenci"},
		{"silencieusement", "silenci"},
		{"silencieux", "silenci"},
		{"silhouette", "silhouet"},
		{"sillonnaient", "sillon"},
		{"sillonne", "sillon"},
		{"sillonné", "sillon"},
		{"sillonnés", "sillon"},
		{"sillons", "sillon"},
		{"simagrées", "simagr"},
		{"simon", "simon"},
		{"simonie", "simon"},
		{"simple", "simpl"},
		{"simplement", "simpl"},
		{"simples", "simpl"},
		{"simplicité", "simpliqu"},
		{"simplifiée", "simplifi"},
		{"simule", "simul"},
		{"sinant", "sin"},
		{"since", "sinc"},
		{"sincère", "sincer"},
		{"sincèrement", "sincer"},
		{"sincères", "sincer"},
		{"sincérité", "sincer"},
		{"sindes", "sind"},
		{"sine", "sin"},
		{"sinécures", "sinécur"},
		{"singapore", "singapor"},
		{"singe", "sing"},
		{"singeries", "singer"},
		{"singes", "sing"},
		{"singularité", "singular"},
		{"singularités", "singular"},
		{"singulier", "singuli"},
		{"singulière", "singuli"},
		{"singulièrement", "singuli"},
		{"singulières", "singuli"},
		{"singuliers", "singuli"},
		{"sinistre", "sinistr"},
		{"sinistres", "sinistr"},
		{"sinon", "sinon"},
		{"sinuosités", "sinuos"},
		{"sioux", "sioux"},
		{"sir", "sir"},
		{"sire", "sir"},
		{"site", "sit"},
		{"sites", "sit"},
		{"sitôt", "sitôt"},
		{"situation", "situat"},
		{"situé", "situ"},
		{"située", "situ"},
		{"situés", "situ"},
		{"six", "six"},
		{"sixième", "sixiem"},
		{"sixte", "sixt"},
		{"sixtine", "sixtin"},
		{"sky", "sky"},
		{"sleeping", "sleeping"},
		{"slight", "slight"},
		{"sloop", "sloop"},
		{"small", "small"},
		{"smith", "smith"},
		{"smollett", "smollet"},
		{"smyth", "smyth"},
		{"so", "so"},
		{"social", "social"},
		{"sociale", "social"},
		{"sociales", "social"},
		{"sociaux", "social"},
		{"société", "societ"},
		{"sociétés", "societ"},
		{"socques", "socqu"},
		{"soeur", "soeur"},
		{"soeurs", "soeur"},
		{"soi", "soi"},
		{"soie", "soi"},
		{"soient", "soient"},
		{"soif", "soif"},
		{"soigna", "soign"},
		{"soignait", "soign"},
		{"soignant", "soign"},
		{"soigne", "soign"},
		{"soigné", "soign"},
		{"soignée", "soign"},
		{"soigner", "soign"},
		{"soignés", "soign"},
		{"soigneusement", "soigneux"},
		{"soigneux", "soigneux"},
		{"soin", "soin"},
		{"soins", "soin"},
		{"soir", "soir"},
		{"soirée", "soir"},
		{"soirées", "soir"},
		{"soirs", "soir"},
		{"sois", "sois"},
		{"soissons", "soisson"},
		{"soit", "soit"},
		{"soixante", "soix"},
		{"soixantième", "soixantiem"},
		{"sol", "sol"},
		{"solaire", "solair"},
		{"soldant", "sold"},
		{"soldat", "soldat"},
		{"soldats", "soldat"},
		{"solde", "sold"},
		{"soldé", "sold"},
		{"soldés", "sold"},
		{"soleil", "soleil"},
		{"solennel", "solennel"},
		{"solennelle", "solennel"},
		{"solennels", "solennel"},
		{"solide", "solid"},
		{"solidement", "solid"},
		{"solides", "solid"},
		{"solidité", "solid"},
		{"solitaire", "solitair"},
		{"solitaires", "solitair"},
		{"solitude", "solitud"},
		{"sollicita", "sollicit"},
		{"sollicitait", "sollicit"},
		{"sollicitations", "sollicit"},
		{"sollicite", "sollicit"},
		{"sollicité", "solliqu"},
		{"solliciter", "sollicit"},
		{"solliciteur", "solliciteur"},
		{"sollicitez", "sollicit"},
		{"sols", "sol"},
		{"solution", "solut"},
		{"somanlis", "somanl"},
		{"sombre", "sombr"},
		{"sombrer", "sombr"},
		{"sombreros", "sombreros"},
		{"sombres", "sombr"},
		{"sommaire", "sommair"},
		{"sommariva", "sommariv"},
		{"sommation", "sommat"},
		{"somme", "somm"},
		{"sommeil", "sommeil"},
		{"sommeillait", "sommeil"},
		{"sommer", "somm"},
		{"sommes", "somm"},
		{"sommet", "sommet"},
		{"sommets", "sommet"},
		{"sommités", "sommit"},
		{"somptueuse", "somptueux"},
		{"somptueusement", "somptueux"},
		{"somptueux", "somptueux"},
		{"somptuosité", "somptuos"},
		{"son", "son"},
		{"sonder", "sond"},
		{"songe", "song"},
		{"songé", "song"},
		{"songea", "song"},
		{"songeaient", "song"},
		{"songeais", "song"},
		{"songeait", "song"},
		{"songeant", "song"},
		{"songeât", "song"},
		{"songent", "songent"},
		{"songeons", "songeon"},
		{"songer", "song"},
		{"songera", "song"},
		{"songerai", "song"},
		{"songerait", "song"},
		{"songèrent", "song"},
		{"songes", "song"},
		{"songez", "song"},
		{"sonna", "son"},
		{"sonnaient", "son"},
		{"sonnait", "son"},
		{"sonnant", "son"},
		{"sonnât", "son"},
		{"sonne", "son"},
		{"sonné", "son"},
		{"sonnée", "son"},
		{"sonnées", "son"},
		{"sonner", "son"},
		{"sonnera", "son"},
		{"sonnèrent", "son"},
		{"sonneries", "sonner"},
		{"sonneront", "son"},
		{"sonnés", "son"},
		{"sonnet", "sonnet"},
		{"sonnets", "sonnet"},
		{"sonnette", "sonnet"},
		{"sonnettes", "sonnet"},
		{"sonneurs", "sonneur"},
		{"sonore", "sonor"},
		{"sonores", "sonor"},
		{"sons", "son"},
		{"sont", "sont"},
		{"sorbonne", "sorbon"},
		{"sorcier", "sorci"},
		{"sordide", "sordid"},
		{"sorel", "sorel"},
		{"sorezana", "sorezan"},
		{"sorgho", "sorgho"},
		{"sors", "sor"},
		{"sort", "sort"},
		{"sortaient", "sort"},
		{"sortais", "sort"},
		{"sortait", "sort"},
		{"sortant", "sort"},
		{"sorte", "sort"},
		{"sortent", "sortent"},
		{"sortes", "sort"},
		{"sortez", "sort"},
		{"sorti", "sort"},
		{"sortie", "sort"},
		{"sortiez", "sort"},
		{"sortir", "sort"},
		{"sortira", "sort"},
		{"sortirai", "sort"},
		{"sortirais", "sort"},
		{"sortirait", "sort"},
		{"sortiras", "sort"},
		{"sortirent", "sort"},
		{"sortirez", "sort"},
		{"sortirions", "sort"},
		{"sortirons", "sort"},
		{"sortis", "sort"},
		{"sortit", "sort"},
		{"sortît", "sort"},
		{"sorts", "sort"},
		{"sot", "sot"},
		{"sots", "sot"},
		{"sotte", "sott"},
		{"sottement", "sott"},
		{"sottes", "sott"},
		{"sottise", "sottis"},
		{"sottises", "sottis"},
		{"sou", "sou"},
		{"soubrette", "soubret"},
		{"souci", "souc"},
		{"soucia", "souci"},
		{"souciait", "souci"},
		{"soucier", "souci"},
		{"soucieuse", "soucieux"},
		{"soucieux", "soucieux"},
		{"soucis", "souc"},
		{"soudain", "soudain"},
		{"soudaine", "soudain"},
		{"soudainement", "soudain"},
		{"souffert", "souffert"},
		{"souffla", "souffl"},
		{"soufflait", "souffl"},
		{"soufflant", "souffl"},
		{"souffle", "souffl"},
		{"soufflent", "soufflent"},
		{"souffler", "souffl"},
		{"soufflet", "soufflet"},
		{"soufflets", "soufflet"},
		{"souffleur", "souffleur"},
		{"souffrais", "souffr"},
		{"souffrait", "souffr"},
		{"souffrance", "souffranc"},
		{"souffrances", "souffranc"},
		{"souffrant", "souffr"},
		{"souffrante", "souffr"},
		{"souffre", "souffr"},
		{"souffrent", "souffrent"},
		{"souffres", "souffr"},
		{"souffrez", "souffr"},
		{"souffrir", "souffr"},
		{"souffrira", "souffr"},
		{"souffrirai", "souffr"},
		{"souffrirais", "souffr"},
		{"souffriras", "souffr"},
		{"souffrissent", "souffr"},
		{"souffrit", "souffr"},
		{"souhait", "souh"},
		{"souhaitant", "souhait"},
		{"souhaite", "souhait"},
		{"souhaité", "souhait"},
		{"souillé", "souill"},
		{"souiller", "souill"},
		{"soulagé", "soulag"},
		{"soulagée", "soulag"},
		{"soulagement", "soulag"},
		{"soulcy", "soulcy"},
		{"souleva", "soulev"},
		{"soulevait", "soulev"},
		{"soulevant", "soulev"},
		{"soulevée", "soulev"},
		{"soulevées", "soulev"},
		{"soulèvent", "soulèvent"},
		{"soulever", "soulev"},
		{"soulèverai", "soulev"},
		{"soulier", "souli"},
		{"souliers", "souli"},
		{"soulte", "soult"},
		{"soumettait", "soumet"},
		{"soumettre", "soumettr"},
		{"soumis", "soum"},
		{"soumise", "soumis"},
		{"soumises", "soumis"},
		{"soumission", "soumiss"},
		{"soumit", "soum"},
		{"soupait", "soup"},
		{"soupapes", "soupap"},
		{"soupçon", "soupçon"},
		{"soupçonna", "soupçon"},
		{"soupçonnaient", "soupçon"},
		{"soupçonnait", "soupçon"},
		{"soupçonne", "soupçon"},
		{"soupçonné", "soupçon"},
		{"soupçonnée", "soupçon"},
		{"soupçonner", "soupçon"},
		{"soupçonnés", "soupçon"},
		{"soupçonneux", "soupçon"},
		{"soupçons", "soupçon"},
		{"soupe", "soup"},
		{"soupé", "soup"},
		{"souper", "soup"},
		{"soupers", "souper"},
		{"soupez", "soup"},
		{"soupir", "soup"},
		{"soupirant", "soupir"},
		{"soupire", "soupir"},
		{"soupirer", "soupir"},
		{"souple", "soupl"},
		{"souplesse", "soupless"},
		{"source", "sourc"},
		{"sources", "sourc"},
		{"sourcil", "sourcil"},
		{"sourciliers", "sourcili"},
		{"sourcilla", "sourcill"},
		{"sourcillé", "sourcill"},
		{"sourciller", "sourcill"},
		{"sourcils", "sourcil"},
		{"sourd", "sourd"},
		{"sourde", "sourd"},
		{"souri", "sour"},
		{"souriait", "souri"},
		{"souriant", "souri"},
		{"souriante", "souri"},
		{"souriantes", "souri"},
		{"sourire", "sourir"},
		{"sourires", "sourir"},
		{"sourit", "sour"},
		{"sournois", "sournois"},
		{"sournoise", "sournois"},
		{"sous", "sous"},
		{"soustrairais", "soustrair"},
		{"soustraire", "soustrair"},
		{"soustrait", "soustr"},
		{"soutane", "soutan"},
		{"soutenaient", "souten"},
		{"soutenait", "souten"},
		{"soutenant", "souten"},
		{"soutènement", "souten"},
		{"soutenir", "souten"},
		{"soutenu", "soutenu"},
		{"soutenue", "soutenu"},
		{"soutenues", "soutenu"},
		{"souterrain", "souterrain"},
		{"souterraines", "souterrain"},
		{"soutes", "sout"},
		{"south", "south"},
		{"southampton", "southampton"},
		{"southey", "southey"},
		{"soutiennent", "soutiennent"},
		{"soutiens", "soutien"},
		{"soutient", "soutient"},
		{"soutint", "soutint"},
		{"souvenaient", "souven"},
		{"souvenait", "souven"},
		{"souvenant", "souven"},
		{"souvenez", "souven"},
		{"souvenir", "souven"},
		{"souvenirs", "souvenir"},
		{"souvent", "souvent"},
		{"souvenus", "souvenus"},
		{"souverain", "souverain"},
		{"souveraine", "souverain"},
		{"souverainement", "souverain"},
		{"souveraineté", "souverainet"},
		{"souverains", "souverain"},
		{"souviendra", "souviendr"},
		{"souviendrez", "souviendr"},
		{"souvienne", "souvien"},
		{"souviennent", "souviennent"},
		{"souviens", "souvien"},
		{"souvient", "souvient"},
		{"souvint", "souvint"},
		{"soyeux", "soyeux"},
		{"soyez", "soi"},
		{"soyons", "soyon"},
		{"spardeck", "spardeck"},
		{"speak", "speak"},
		{"spécial", "spécial"},
		{"spéciale", "spécial"},
		{"spécialement", "spécial"},
		{"spécialité", "spécial"},
		{"spécifier", "spécifi"},
		{"spécifique", "spécif"},
		{"spectacle", "spectacl"},
		{"spectacles", "spectacl"},
		{"spectateur", "spectateur"},
		{"spectateurs", "spectateur"},
		{"spectrale", "spectral"},
		{"spectre", "spectr"},
		{"spéculation", "spécul"},
		{"spéculations", "spécul"},
		{"speedy", "speedy"},
		{"sphère", "spher"},
		{"sphéroïde", "sphéroïd"},
		{"spielberg", "spielberg"},
		{"spirale", "spiral"},
		{"spirales", "spiral"},
		{"spirituel", "spirituel"},
		{"spirituelle", "spirituel"},
		{"spirituelles", "spirituel"},
		{"splendide", "splendid"},
		{"splendides", "splendid"},
		{"spring", "spring"},
		{"squares", "squar"},
		{"squelette", "squelet"},
		{"st", "st"},
		{"staël", "staël"},
		{"stalles", "stall"},
		{"stamp", "stamp"},
		{"standard", "standard"},
		{"stanislas", "stanisl"},
		{"star", "star"},
		{"starke", "stark"},
		{"station", "station"},
		{"stationnera", "station"},
		{"stationnés", "station"},
		{"stations", "station"},
		{"statuaire", "statuair"},
		{"statue", "statu"},
		{"statues", "statu"},
		{"stature", "statur"},
		{"staub", "staub"},
		{"steam", "steam"},
		{"steamboats", "steamboat"},
		{"steamer", "steam"},
		{"steamers", "steamer"},
		{"steccata", "steccat"},
		{"stefano", "stefano"},
		{"stentor", "stentor"},
		{"stephenson", "stephenson"},
		{"sterling", "sterling"},
		{"sterne", "stern"},
		{"steward", "steward"},
		{"stigmates", "stigmat"},
		{"stigmatiser", "stigmatis"},
		{"still", "still"},
		{"stimuler", "stimul"},
		{"stolen", "stolen"},
		{"strabon", "strabon"},
		{"stramonium", "stramonium"},
		{"strand", "strand"},
		{"strasbourg", "strasbourg"},
		{"stratagème", "stratagem"},
		{"straw", "straw"},
		{"street", "street"},
		{"strict", "strict"},
		{"stricte", "strict"},
		{"strictement", "strict"},
		{"strombeck", "strombeck"},
		{"strongest", "strongest"},
		{"strychnine", "strychnin"},
		{"stuart", "stuart"},
		{"studbook", "studbook"},
		{"stupéfaction", "stupéfact"},
		{"stupéfait", "stupef"},
		{"stupéfiante", "stupéfi"},
		{"stupefie", "stupef"},
		{"stupéfie", "stupef"},
		{"stupéfié", "stupéfi"},
		{"stupéfier", "stupéfi"},
		{"stupeur", "stupeur"},
		{"stupide", "stupid"},
		{"stupides", "stupid"},
		{"style", "styl"},
		{"stylé", "styl"},
		{"stylés", "styl"},
		{"su", "su"},
		{"suave", "suav"},
		{"subalterne", "subaltern"},
		{"subalternes", "subaltern"},
		{"subi", "sub"},
		{"subir", "sub"},
		{"subira", "sub"},
		{"subirait", "sub"},
		{"subis", "sub"},
		{"subissant", "sub"},
		{"subit", "sub"},
		{"subite", "subit"},
		{"subitement", "subit"},
		{"subjugue", "subjugu"},
		{"subjugué", "subjugu"},
		{"subjuguer", "subjugu"},
		{"sublime", "sublim"},
		{"sublimes", "sublim"},
		{"sublimité", "sublim"},
		{"submerge", "submerg"},
		{"submergé", "submerg"},
		{"submergèrent", "submerg"},
		{"subordonnés", "subordon"},
		{"suborner", "suborn"},
		{"subsistance", "subsist"},
		{"subsiste", "subsist"},
		{"subsistent", "subsistent"},
		{"substance", "substanc"},
		{"substitué", "substitu"},
		{"substituée", "substitu"},
		{"substituées", "substitu"},
		{"substituer", "substitu"},
		{"substitut", "substitut"},
		{"subtilité", "subtil"},
		{"subvention", "subvent"},
		{"succéda", "succed"},
		{"succédaient", "succed"},
		{"succédait", "succed"},
		{"succède", "succed"},
		{"succédé", "succed"},
		{"succéder", "succed"},
		{"succédèrent", "succed"},
		{"succès", "succes"},
		{"successeur", "successeur"},
		{"successeurs", "successeur"},
		{"successifs", "success"},
		{"succession", "success"},
		{"successive", "success"},
		{"successivement", "success"},
		{"successives", "success"},
		{"successor", "successor"},
		{"succombant", "succomb"},
		{"succombé", "succomb"},
		{"succombent", "succombent"},
		{"succomber", "succomb"},
		{"succulentes", "succulent"},
		{"succulents", "succulent"},
		{"such", "such"},
		{"suchet", "suchet"},
		{"sucre", "sucr"},
		{"sucreries", "sucrer"},
		{"sud", "sud"},
		{"suédois", "suédois"},
		{"sueur", "sueur"},
		{"suez", "su"},
		{"suffers", "suffer"},
		{"suffi", "suff"},
		{"suffira", "suff"},
		{"suffiraient", "suff"},
		{"suffirait", "suff"},
		{"suffire", "suffir"},
		{"suffirent", "suff"},
		{"suffiront", "suff"},
		{"suffisaient", "suffis"},
		{"suffisait", "suffis"},
		{"suffisamment", "suffis"},
		{"suffisance", "suffis"},
		{"suffisant", "suffis"},
		{"suffisante", "suffis"},
		{"suffisantes", "suffis"},
		{"suffisent", "suffisent"},
		{"suffit", "suff"},
		{"suffocations", "suffoc"},
		{"suffoqua", "suffoqu"},
		{"suffoqué", "suffoqu"},
		{"suffoquerait", "suffoqu"},
		{"suggéra", "sugger"},
		{"suggéré", "sugger"},
		{"suggérée", "sugger"},
		{"suggérer", "sugger"},
		{"suicide", "suicid"},
		{"suie", "sui"},
		{"suis", "suis"},
		{"suisse", "suiss"},
		{"suisses", "suiss"},
		{"suit", "suit"},
		{"suite", "suit"},
		{"suites", "suit"},
		{"suivaient", "suiv"},
		{"suivais", "suiv"},
		{"suivait", "suiv"},
		{"suivant", "suiv"},
		{"suivante", "suiv"},
		{"suivantes", "suiv"},
		{"suivants", "suiv"},
		{"suivent", "suivent"},
		{"suivez", "suiv"},
		{"suivi", "suiv"},
		{"suivie", "suiv"},
		{"suivies", "suiv"},
		{"suiviez", "suiv"},
		{"suivirent", "suiv"},
		{"suivis", "suiv"},
		{"suivit", "suiv"},
		{"suivons", "suivon"},
		{"suivra", "suivr"},
		{"suivrai", "suivr"},
		{"suivraient", "suivr"},
		{"suivras", "suivr"},
		{"suivre", "suivr"},
		{"suivront", "suivront"},
		{"sujet", "sujet"},
		{"sujets", "sujet"},
		{"sujette", "sujet"},
		{"sujettes", "sujet"},
		{"sullivan", "sullivan"},
		{"sultan", "sultan"},
		{"sumatra", "sumatr"},
		{"sun", "sun"},
		{"superbe", "superb"},
		{"superbes", "superb"},
		{"superficie", "superfic"},
		{"superficiel", "superficiel"},
		{"superflu", "superflu"},
		{"supérieur", "supérieur"},
		{"supérieure", "supérieur"},
		{"supérieurement", "supérieur"},
		{"supérieures", "supérieur"},
		{"supérieurs", "supérieur"},
		{"supériorité", "supérior"},
		{"suppléé", "supplé"},
		{"supplément", "suppl"},
		{"supplémentaire", "supplémentair"},
		{"supplia", "suppli"},
		{"suppliaient", "suppli"},
		{"suppliait", "suppli"},
		{"suppliant", "suppli"},
		{"suppliante", "suppli"},
		{"supplications", "suppliqu"},
		{"supplice", "supplic"},
		{"supplices", "supplic"},
		{"supplie", "suppl"},
		{"supplier", "suppli"},
		{"suppliez", "suppl"},
		{"supportable", "support"},
		{"supportait", "support"},
		{"supportent", "supportent"},
		{"supporter", "support"},
		{"supporterait", "support"},
		{"supposait", "suppos"},
		{"supposant", "suppos"},
		{"suppose", "suppos"},
		{"supposé", "suppos"},
		{"supposée", "suppos"},
		{"supposer", "suppos"},
		{"supposés", "suppos"},
		{"supposez", "suppos"},
		{"supposition", "supposit"},
		{"suppositions", "supposit"},
		{"supposons", "supposon"},
		{"suppression", "suppress"},
		{"supprimant", "supprim"},
		{"supprime", "supprim"},
		{"supprimé", "supprim"},
		{"supprimée", "supprim"},
		{"supprimer", "supprim"},
		{"supprimons", "supprimon"},
		{"supputait", "supput"},
		{"suprême", "suprêm"},
		{"sur", "sur"},
		{"sûr", "sûr"},
		{"surannée", "surann"},
		{"surate", "surat"},
		{"surchargés", "surcharg"},
		{"surchauffée", "surchauff"},
		{"surcroît", "surcroît"},
		{"sûre", "sûr"},
		{"surélevés", "surélev"},
		{"sûrement", "sûr"},
		{"surenchère", "surencher"},
		{"surent", "surent"},
		{"suresnes", "suresn"},
		{"sûreté", "sûret"},
		{"surexcité", "surexc"},
		{"surface", "surfac"},
		{"surgir", "surg"},
		{"surgirait", "surg"},
		{"surgirent", "surg"},
		{"surhumain", "surhumain"},
		{"surhumaine", "surhumain"},
		{"surlendemain", "surlendemain"},
		{"surmontant", "surmont"},
		{"surmonte", "surmont"},
		{"surmontée", "surmont"},
		{"surmontées", "surmont"},
		{"surmonter", "surmont"},
		{"surnage", "surnag"},
		{"surnagea", "surnag"},
		{"surnageait", "surnag"},
		{"surnaturelle", "surnaturel"},
		{"surnom", "surnom"},
		{"surnommèrent", "surnomm"},
		{"surpassa", "surpass"},
		{"surpasse", "surp"},
		{"surpassé", "surpass"},
		{"surpassent", "surp"},
		{"surplis", "surpl"},
		{"surplus", "surplus"},
		{"surprenait", "surpren"},
		{"surprenant", "surpren"},
		{"surprend", "surprend"},
		{"surprendre", "surprendr"},
		{"surprirent", "surpr"},
		{"surpris", "surpr"},
		{"surprise", "surpris"},
		{"surprises", "surpris"},
		{"surprit", "surpr"},
		{"surrey", "surrey"},
		{"sûrs", "sûr"},
		{"sursaut", "sursaut"},
		{"surtout", "surtout"},
		{"survécut", "survécut"},
		{"surveillait", "surveil"},
		{"surveillance", "surveil"},
		{"surveillants", "surveil"},
		{"surveille", "surveil"},
		{"surveillé", "surveil"},
		{"surveillée", "surveil"},
		{"surveiller", "surveil"},
		{"surveillera", "surveil"},
		{"survenait", "surven"},
		{"survenir", "surven"},
		{"survenue", "survenu"},
		{"survinrent", "survinrent"},
		{"survint", "survint"},
		{"survit", "surv"},
		{"survivant", "surviv"},
		{"survivrai", "survivr"},
		{"survivrais", "survivr"},
		{"survivre", "survivr"},
		{"survivrez", "survivr"},
		{"sus", "sus"},
		{"susceptible", "susceptibl"},
		{"susceptibles", "susceptibl"},
		{"susciter", "suscit"},
		{"suscitèrent", "suscit"},
		{"susdit", "susd"},
		{"suspect", "suspect"},
		{"suspecte", "suspect"},
		{"suspecter", "suspect"},
		{"suspects", "suspect"},
		{"suspendre", "suspendr"},
		{"suspendu", "suspendu"},
		{"suspendue", "suspendu"},
		{"suspendus", "suspendus"},
		{"suspens", "suspen"},
		{"suspicion", "suspicion"},
		{"sut", "sut"},
		{"sût", "sût"},
		{"sutpour", "sutpour"},
		{"sutty", "sutty"},
		{"svelte", "svelt"},
		{"sweeter", "sweet"},
		{"sydenham", "sydenham"},
		{"syénites", "syénit"},
		{"syllabe", "syllab"},
		{"syllabes", "syllab"},
		{"syllogisme", "syllog"},
		{"symbole", "symbol"},
		{"symboles", "symbol"},
		{"symbolisent", "symbolisent"},
		{"symétrie", "symetr"},
		{"symétriques", "symetr"},
		{"sympathie", "sympath"},
		{"sympathisait", "sympathis"},
		{"sympathise", "sympathis"},
		{"symphonie", "symphon"},
		{"symptôme", "symptôm"},
		{"symptômes", "symptôm"},
		{"synagogues", "synagogu"},
		{"synonyme", "synonym"},
		{"synonymes", "synonym"},
		{"système", "system"},
		{"t", "t"},
		{"ta", "ta"},
		{"tabac", "tabac"},
		{"tabagie", "tabag"},
		{"tabagies", "tabag"},
		{"tabatière", "tabati"},
		{"tabernacle", "tabernacl"},
		{"table", "tabl"},
		{"tableau", "tableau"},
		{"tableaux", "tableau"},
		{"tables", "tabl"},
		{"tablette", "tablet"},
		{"tablier", "tabli"},
		{"tabouret", "tabouret"},
		{"tâchait", "tâch"},
		{"tâchant", "tâch"},
		{"tache", "tach"},
		{"tâche", "tâch"},
		{"taché", "tach"},
		{"tachée", "tach"},
		{"tachées", "tach"},
		{"tâcher", "tâch"},
		{"tâcherai", "tâch"},
		{"taches", "tach"},
		{"tachetée", "tachet"},
		{"tâchez", "tâch"},
		{"tâchons", "tâchon"},
		{"tacite", "tacit"},
		{"tacites", "tacit"},
		{"taciti", "tacit"},
		{"taciturnes", "taciturn"},
		{"tact", "tact"},
		{"taffetas", "taffet"},
		{"taïkoun", "taïkoun"},
		{"tailla", "taill"},
		{"taillanderie", "taillander"},
		{"taillant", "taill"},
		{"taille", "taill"},
		{"taillé", "taill"},
		{"taillées", "taill"},
		{"tailler", "taill"},
		{"taillés", "taill"},
		{"tailleur", "tailleur"},
		{"taillis", "taill"},
		{"taire", "tair"},
		{"tais", "tais"},
		{"taisaient", "tais"},
		{"taisait", "tais"},
		{"taisant", "tais"},
		{"taisez", "tais"},
		{"tait", "tait"},
		{"takes", "tak"},
		{"talent", "talent"},
		{"talents", "talent"},
		{"talleyrand", "talleyrand"},
		{"talma", "talm"},
		{"taloches", "taloch"},
		{"talons", "talon"},
		{"talus", "talus"},
		{"tam", "tam"},
		{"tamarins", "tamarin"},
		{"tambour", "tambour"},
		{"tambourins", "tambourin"},
		{"tambours", "tambour"},
		{"tams", "tam"},
		{"tanari", "tanar"},
		{"tanbeau", "tanbeau"},
		{"tancrède", "tancred"},
		{"tandis", "tand"},
		{"tangage", "tangag"},
		{"tangles", "tangl"},
		{"tankadère", "tankader"},
		{"tankardere", "tankarder"},
		{"tankardère", "tankarder"},
		{"tankas", "tank"},
		{"tannah", "tannah"},
		{"tanquam", "tanquam"},
		{"tant", "tant"},
		{"tante", "tant"},
		{"tantes", "tant"},
		{"tantôt", "tantôt"},
		{"tanzi", "tanz"},
		{"tapage", "tapag"},
		{"tapé", "tap"},
		{"tapis", "tapis"},
		{"tapissée", "tapiss"},
		{"tapisserie", "tapisser"},
		{"tapissier", "tapissi"},
		{"tapissiers", "tapissi"},
		{"tapty", "tapty"},
		{"tard", "tard"},
		{"tarda", "tard"},
		{"tardaient", "tard"},
		{"tardait", "tard"},
		{"tarde", "tard"},
		{"tardé", "tard"},
		{"tarder", "tard"},
		{"tarderait", "tard"},
		{"tardèrent", "tard"},
		{"tardive", "tardiv"},
		{"tarés", "tar"},
		{"taries", "tar"},
		{"tarirent", "tar"},
		{"tarissait", "tar"},
		{"tarit", "tar"},
		{"tarots", "tarot"},
		{"tarragone", "tarragon"},
		{"tartines", "tartin"},
		{"tartufe", "tartuf"},
		{"tas", "tas"},
		{"tasse", "tass"},
		{"tasses", "tass"},
		{"tâtonnent", "tâtonnent"},
		{"taudis", "taud"},
		{"taureaux", "taureau"},
		{"taverne", "tavern"},
		{"taxes", "tax"},
		{"taxis", "tax"},
		{"te", "te"},
		{"tea", "te"},
		{"tecks", "teck"},
		{"teindre", "teindr"},
		{"teint", "teint"},
		{"teinte", "teint"},
		{"teintes", "teint"},
		{"tel", "tel"},
		{"telegraph", "telegraph"},
		{"télégraphe", "télégraph"},
		{"télégraphié", "télégraphi"},
		{"télégraphique", "télégraph"},
		{"télégraphiques", "télégraph"},
		{"télémaque", "télémaqu"},
		{"télescope", "télescop"},
		{"télescopes", "télescop"},
		{"telle", "tel"},
		{"tellement", "tel"},
		{"telles", "tel"},
		{"tels", "tel"},
		{"téméraire", "témérair"},
		{"témérité", "témer"},
		{"témoignage", "témoignag"},
		{"témoignaient", "témoign"},
		{"témoignait", "témoign"},
		{"temoigne", "temoign"},
		{"témoigne", "témoign"},
		{"témoigné", "témoign"},
		{"témoigner", "témoign"},
		{"témoigneront", "témoign"},
		{"témoigniez", "témoign"},
		{"témoin", "témoin"},
		{"témoins", "témoin"},
		{"tempe", "temp"},
		{"tempérait", "temper"},
		{"tempérament", "temper"},
		{"température", "températur"},
		{"tempérée", "temper"},
		{"tempes", "temp"},
		{"tempest", "tempest"},
		{"tempête", "tempêt"},
		{"temple", "templ"},
		{"temples", "templ"},
		{"temps", "temp"},
		{"tenable", "tenabl"},
		{"ténacité", "ténac"},
		{"tenaient", "ten"},
		{"tenais", "ten"},
		{"tenait", "ten"},
		{"tenant", "ten"},
		{"tenante", "ten"},
		{"tend", "tend"},
		{"tendaient", "tend"},
		{"tendait", "tend"},
		{"tendance", "tendanc"},
		{"tendent", "tendent"},
		{"tender", "tend"},
		{"tendit", "tend"},
		{"tendre", "tendr"},
		{"tendrement", "tendr"},
		{"tendres", "tendr"},
		{"tendresse", "tendress"},
		{"tendresses", "tendress"},
		{"tendu", "tendu"},
		{"tendue", "tendu"},
		{"tendues", "tendu"},
		{"tendus", "tendus"},
		{"tenerani", "teneran"},
		{"tenere", "tener"},
		{"teneur", "teneur"},
		{"tenez", "ten"},
		{"tenir", "ten"},
		{"tenons", "tenon"},
		{"ténor", "ténor"},
		{"tenta", "tent"},
		{"tentait", "tent"},
		{"tentation", "tentat"},
		{"tentations", "tentat"},
		{"tentative", "tentat"},
		{"tentatives", "tentat"},
		{"tente", "tent"},
		{"tenté", "tent"},
		{"tenter", "tent"},
		{"tenteras", "tent"},
		{"tentez", "tent"},
		{"tentures", "tentur"},
		{"tenu", "tenu"},
		{"tenue", "tenu"},
		{"tenues", "tenu"},
		{"tenus", "tenus"},
		{"terme", "term"},
		{"termes", "term"},
		{"termina", "termin"},
		{"terminaient", "termin"},
		{"terminait", "termin"},
		{"termine", "termin"},
		{"terminé", "termin"},
		{"terminée", "termin"},
		{"terminées", "termin"},
		{"terminer", "termin"},
		{"terminera", "termin"},
		{"terminèrent", "termin"},
		{"ternir", "tern"},
		{"terrain", "terrain"},
		{"terrains", "terrain"},
		{"terrasse", "terr"},
		{"terrassé", "terrass"},
		{"terrasses", "terr"},
		{"terrassiers", "terrassi"},
		{"terre", "terr"},
		{"terres", "terr"},
		{"terrestre", "terrestr"},
		{"terrestres", "terrestr"},
		{"terreur", "terreur"},
		{"terreurs", "terreur"},
		{"terrible", "terribl"},
		{"terriblement", "terribl"},
		{"terribles", "terribl"},
		{"territoire", "territoir"},
		{"tertio", "tertio"},
		{"tertre", "tertr"},
		{"terzo", "terzo"},
		{"tes", "te"},
		{"testa", "test"},
		{"testament", "test"},
		{"tête", "têt"},
		{"têtes", "têt"},
		{"teulier", "teuli"},
		{"texte", "text"},
		{"textuellement", "textuel"},
		{"thaler", "thal"},
		{"that", "that"},
		{"the", "the"},
		{"thé", "thé"},
		{"théâtral", "théâtral"},
		{"théâtre", "théâtr"},
		{"théâtres", "théâtr"},
		{"theft", "theft"},
		{"thème", "them"},
		{"thèmes", "them"},
		{"thémistocle", "thémistocl"},
		{"then", "then"},
		{"théodolinde", "théodolind"},
		{"théologie", "théolog"},
		{"théologien", "théologien"},
		{"théorie", "théor"},
		{"théories", "théor"},
		{"théoriquement", "théoriqu"},
		{"there", "ther"},
		{"thérèse", "théres"},
		{"thermomètre", "thermometr"},
		{"thésée", "thes"},
		{"thing", "thing"},
		{"this", "this"},
		{"thomas", "thom"},
		{"though", "though"},
		{"thousands", "thousand"},
		{"thrilling", "thrilling"},
		{"thugs", "thug"},
		{"tibère", "tiber"},
		{"tibi", "tib"},
		{"tien", "tien"},
		{"tiendra", "tiendr"},
		{"tiendrait", "tiendr"},
		{"tiendras", "tiendr"},
		{"tiendrez", "tiendr"},
		{"tiendront", "tiendront"},
		{"tienne", "tien"},
		{"tiennent", "tiennent"},
		{"tiens", "tien"},
		{"tient", "tient"},
		{"tierce", "tierc"},
		{"tiers", "tier"},
		{"tige", "tig"},
		{"tiges", "tig"},
		{"tigre", "tigr"},
		{"tigres", "tigr"},
		{"tilbury", "tilbury"},
		{"tilburys", "tilbury"},
		{"tilleul", "tilleul"},
		{"tilleuls", "tilleul"},
		{"timbre", "timbr"},
		{"timbrée", "timbr"},
		{"timbrées", "timbr"},
		{"timbres", "timbr"},
		{"time", "tim"},
		{"times", "tim"},
		{"timide", "timid"},
		{"timidement", "timid"},
		{"timides", "timid"},
		{"timidité", "timid"},
		{"timorée", "timor"},
		{"tingou", "tingou"},
		{"tinrent", "tinrent"},
		{"tinssent", "tinssent"},
		{"tint", "tint"},
		{"tînt", "tînt"},
		{"tintamarre", "tintamarr"},
		{"tinte", "tint"},
		{"tira", "tir"},
		{"tirade", "tirad"},
		{"tirades", "tirad"},
		{"tirage", "tirag"},
		{"tiraient", "tir"},
		{"tiraillé", "tiraill"},
		{"tirailler", "tiraill"},
		{"tirait", "tir"},
		{"tirant", "tir"},
		{"tirât", "tir"},
		{"tire", "tir"},
		{"tiré", "tir"},
		{"tirée", "tir"},
		{"tirent", "tirent"},
		{"tirer", "tir"},
		{"tirera", "tir"},
		{"tireraient", "tir"},
		{"tirerais", "tir"},
		{"tirerait", "tir"},
		{"tirèrent", "tir"},
		{"tirerez", "tir"},
		{"tirés", "tir"},
		{"tirez", "tir"},
		{"tiroir", "tiroir"},
		{"tissu", "tissu"},
		{"tissue", "tissu"},
		{"tite", "tit"},
		{"titre", "titr"},
		{"titres", "titr"},
		{"titubant", "titub"},
		{"titulaire", "titulair"},
		{"to", "to"},
		{"together", "togeth"},
		{"toi", "toi"},
		{"toile", "toil"},
		{"toiles", "toil"},
		{"toilette", "toilet"},
		{"toilettes", "toilet"},
		{"toisait", "tois"},
		{"toisant", "tois"},
		{"toises", "tois"},
		{"toison", "toison"},
		{"toit", "toit"},
		{"toits", "toit"},
		{"toiture", "toitur"},
		{"tolède", "toled"},
		{"tolèrent", "tol"},
		{"tolérer", "toler"},
		{"tolly", "tolly"},
		{"tomba", "tomb"},
		{"tombaient", "tomb"},
		{"tombait", "tomb"},
		{"tombant", "tomb"},
		{"tombante", "tomb"},
		{"tombassent", "tomb"},
		{"tombât", "tomb"},
		{"tombe", "tomb"},
		{"tombé", "tomb"},
		{"tombeau", "tombeau"},
		{"tombeaux", "tombeau"},
		{"tombée", "tomb"},
		{"tombent", "tombent"},
		{"tomber", "tomb"},
		{"tomberai", "tomb"},
		{"tomberaient", "tomb"},
		{"tomberais", "tomb"},
		{"tombèrent", "tomb"},
		{"tomberont", "tomb"},
		{"tombés", "tomb"},
		{"tombez", "tomb"},
		{"tombone", "tombon"},
		{"tome", "tom"},
		{"tomes", "tom"},
		{"ton", "ton"},
		{"tonalités", "tonal"},
		{"tondre", "tondr"},
		{"tonnage", "tonnag"},
		{"tonnaient", "ton"},
		{"tonnante", "ton"},
		{"tonne", "ton"},
		{"tonneaux", "tonneau"},
		{"tonnelier", "tonneli"},
		{"tonnerre", "tonnerr"},
		{"tonnerres", "tonnerr"},
		{"tonnes", "ton"},
		{"tonte", "tont"},
		{"too", "too"},
		{"toque", "toqu"},
		{"torche", "torch"},
		{"torches", "torch"},
		{"torchon", "torchon"},
		{"tordait", "tord"},
		{"tordant", "tord"},
		{"tordit", "tord"},
		{"tordre", "tordr"},
		{"tords", "tord"},
		{"torpeur", "torpeur"},
		{"torrent", "torrent"},
		{"torrents", "torrent"},
		{"torsade", "torsad"},
		{"torses", "tors"},
		{"tort", "tort"},
		{"torto", "torto"},
		{"tortoni", "torton"},
		{"torts", "tort"},
		{"tortue", "tortu"},
		{"tortues", "tortu"},
		{"tortura", "tortur"},
		{"torturait", "tortur"},
		{"torture", "tortur"},
		{"torturé", "tortur"},
		{"torturer", "tortur"},
		{"tortures", "tortur"},
		{"toscane", "toscan"},
		{"tôt", "tôt"},
		{"total", "total"},
		{"totale", "total"},
		{"totalement", "total"},
		{"toto", "toto"},
		{"toucha", "touch"},
		{"touchaient", "touch"},
		{"touchais", "touch"},
		{"touchait", "touch"},
		{"touchant", "touch"},
		{"touchante", "touch"},
		{"touchantes", "touch"},
		{"touchants", "touch"},
		{"touchât", "touch"},
		{"touche", "touch"},
		{"touché", "touch"},
		{"touchée", "touch"},
		{"toucher", "touch"},
		{"touchera", "touch"},
		{"toucherez", "touch"},
		{"toucheront", "touch"},
		{"touches", "touch"},
		{"touchés", "touch"},
		{"touffe", "touff"},
		{"touffes", "touff"},
		{"touffu", "touffu"},
		{"touffus", "touffus"},
		{"toujours", "toujour"},
		{"toulouse", "toulous"},
		{"toupie", "toup"},
		{"toupies", "toup"},
		{"tour", "tour"},
		{"tourbillon", "tourbillon"},
		{"tourbillonnait", "tourbillon"},
		{"tourbillons", "tourbillon"},
		{"touriste", "tourist"},
		{"tourment", "tourment"},
		{"tourmentait", "tourment"},
		{"tourmente", "tourment"},
		{"tourmenté", "tourment"},
		{"tourmentée", "tourment"},
		{"tourmentés", "tourment"},
		{"tourmentin", "tourmentin"},
		{"tourments", "tourment"},
		{"tourna", "tourn"},
		{"tournai", "tourn"},
		{"tournaient", "tourn"},
		{"tournait", "tourn"},
		{"tournant", "tourn"},
		{"tournante", "tourn"},
		{"tournantes", "tourn"},
		{"tourne", "tourn"},
		{"tourné", "tourn"},
		{"tournée", "tourn"},
		{"tournées", "tourn"},
		{"tournent", "tournent"},
		{"tourner", "tourn"},
		{"tournera", "tourn"},
		{"tournèrent", "tourn"},
		{"tourniquets", "tourniquet"},
		{"tournoyant", "tournoi"},
		{"tournure", "tournur"},
		{"tournures", "tournur"},
		{"tours", "tour"},
		{"tous", "tous"},
		{"toussa", "touss"},
		{"tousser", "touss"},
		{"tout", "tout"},
		{"toute", "tout"},
		{"toutefois", "toutefois"},
		{"toutes", "tout"},
		{"tr", "tr"},
		{"trac", "trac"},
		{"traçait", "trac"},
		{"tracasse", "trac"},
		{"tracasserie", "tracasser"},
		{"tracasseries", "tracasser"},
		{"trace", "trac"},
		{"tracé", "trac"},
		{"tracée", "trac"},
		{"tracées", "trac"},
		{"tracer", "trac"},
		{"traces", "trac"},
		{"tracés", "trac"},
		{"traction", "traction"},
		{"tradition", "tradit"},
		{"traduction", "traduct"},
		{"traduire", "traduir"},
		{"traduis", "traduis"},
		{"traduit", "traduit"},
		{"traduite", "traduit"},
		{"traduites", "traduit"},
		{"tragédie", "traged"},
		{"tragique", "tragiqu"},
		{"tragiques", "tragiqu"},
		{"trahi", "trah"},
		{"trahie", "trah"},
		{"trahies", "trah"},
		{"trahir", "trah"},
		{"trahira", "trah"},
		{"trahirais", "trah"},
		{"trahirait", "trah"},
		{"trahiriez", "trah"},
		{"trahiront", "trah"},
		{"trahis", "trah"},
		{"trahison", "trahison"},
		{"trahissaient", "trah"},
		{"trahissait", "trah"},
		{"trahissant", "trah"},
		{"trahisse", "trah"},
		{"trahissez", "trah"},
		{"trahit", "trah"},
		{"trahît", "trah"},
		{"train", "train"},
		{"traînaient", "traîn"},
		{"traînait", "traîn"},
		{"traînante", "traîn"},
		{"traînard", "traînard"},
		{"traîné", "traîn"},
		{"traîneau", "traîneau"},
		{"traîneaux", "traîneau"},
		{"traînée", "traîn"},
		{"traînées", "traîn"},
		{"traîner", "traîn"},
		{"trains", "train"},
		{"trait", "trait"},
		{"traita", "trait"},
		{"traitait", "trait"},
		{"traitant", "trait"},
		{"traitât", "trait"},
		{"traite", "trait"},
		{"traité", "trait"},
		{"traitée", "trait"},
		{"traitement", "trait"},
		{"traitent", "traitent"},
		{"traiter", "trait"},
		{"traitera", "trait"},
		{"traitèrent", "trait"},
		{"traités", "trait"},
		{"traitez", "trait"},
		{"traître", "traîtr"},
		{"traîtreusement", "traîtreux"},
		{"traits", "trait"},
		{"trajectoires", "trajectoir"},
		{"trajet", "trajet"},
		{"trame", "tram"},
		{"tramezzina", "tramezzin"},
		{"tramways", "tramway"},
		{"tranchait", "tranch"},
		{"tranchant", "tranch"},
		{"tranchants", "tranch"},
		{"tranche", "tranch"},
		{"tranchée", "tranch"},
		{"tranchées", "tranch"},
		{"tranches", "tranch"},
		{"tranchés", "tranch"},
		{"tranquille", "tranquill"},
		{"tranquillement", "tranquill"},
		{"tranquilles", "tranquill"},
		{"tranquillise", "tranquillis"},
		{"tranquillisée", "tranquillis"},
		{"tranquilliserait", "tranquillis"},
		{"tranquillisez", "tranquillis"},
		{"tranquillité", "tranquill"},
		{"transaction", "transact"},
		{"transactions", "transact"},
		{"transatlantique", "transatlant"},
		{"transatlantiques", "transatlant"},
		{"transcrire", "transcrir"},
		{"transcrit", "transcr"},
		{"transcrivait", "transcriv"},
		{"transcrivant", "transcriv"},
		{"transcrivit", "transcriv"},
		{"transes", "trans"},
		{"transfèrement", "transfer"},
		{"transférer", "transfer"},
		{"transfigure", "transfigur"},
		{"transforma", "transform"},
		{"transformant", "transform"},
		{"transformé", "transform"},
		{"transformée", "transform"},
		{"transforment", "transforment"},
		{"transformer", "transform"},
		{"transformés", "transform"},
		{"transfuge", "transfug"},
		{"transgression", "transgress"},
		{"transie", "trans"},
		{"transiger", "transig"},
		{"transire", "transir"},
		{"transit", "trans"},
		{"transmettre", "transmettr"},
		{"transmise", "transmis"},
		{"transmissible", "transmissibl"},
		{"transmit", "transm"},
		{"transocéanienne", "transocéanien"},
		{"transocéaniennes", "transocéanien"},
		{"transparents", "transparent"},
		{"transport", "transport"},
		{"transporta", "transport"},
		{"transportait", "transport"},
		{"transportât", "transport"},
		{"transporté", "transport"},
		{"transportée", "transport"},
		{"transporter", "transport"},
		{"transports", "transport"},
		{"transversale", "transversal"},
		{"trapp", "trapp"},
		{"trappe", "trapp"},
		{"trappes", "trapp"},
		{"trappiste", "trappist"},
		{"trapues", "trapu"},
		{"traqué", "traqu"},
		{"trattamento", "trattamento"},
		{"trattoria", "trattori"},
		{"travail", "travail"},
		{"travailla", "travaill"},
		{"travaillaient", "travaill"},
		{"travaillait", "travaill"},
		{"travaille", "travaill"},
		{"travaillé", "travaill"},
		{"travailler", "travaill"},
		{"travailleurs", "travailleur"},
		{"travaux", "traval"},
		{"travers", "traver"},
		{"traversa", "travers"},
		{"traversaient", "travers"},
		{"traversait", "travers"},
		{"traversant", "travers"},
		{"traverse", "travers"},
		{"traversé", "travers"},
		{"traversee", "traverse"},
		{"traversée", "travers"},
		{"traversées", "travers"},
		{"traversent", "traversent"},
		{"traverser", "travers"},
		{"traverserai", "travers"},
		{"traversèrent", "travers"},
		{"traversés", "travers"},
		{"traversez", "travers"},
		{"trébuchant", "trébuch"},
		{"tredici", "tredic"},
		{"treize", "treiz"},
		{"treizième", "treiziem"},
		{"trembla", "trembl"},
		{"tremblaient", "trembl"},
		{"tremblais", "trembl"},
		{"tremblait", "trembl"},
		{"tremblant", "trembl"},
		{"tremblante", "trembl"},
		{"tremblantes", "trembl"},
		{"tremble", "trembl"},
		{"tremblé", "trembl"},
		{"tremblement", "trembl"},
		{"trembler", "trembl"},
		{"tremblera", "trembl"},
		{"tremblerai", "trembl"},
		{"trembleront", "trembl"},
		{"tremblez", "trembl"},
		{"trempe", "tremp"},
		{"trempé", "tremp"},
		{"tremper", "tremp"},
		{"tremplin", "tremplin"},
		{"tremulously", "tremulously"},
		{"trentaine", "trentain"},
		{"trente", "trent"},
		{"trépas", "trep"},
		{"tres", "tre"},
		{"très", "tres"},
		{"trésor", "trésor"},
		{"trésors", "trésor"},
		{"tressailli", "tressaill"},
		{"tressaillir", "tressaill"},
		{"tressaillit", "tressaill"},
		{"tresse", "tress"},
		{"tressé", "tress"},
		{"trêve", "trêv"},
		{"trévise", "trévis"},
		{"triangle", "triangl"},
		{"triangulaire", "triangulair"},
		{"tribu", "tribu"},
		{"tribun", "tribun"},
		{"tribunal", "tribunal"},
		{"tribunaux", "tribunal"},
		{"tribune", "tribun"},
		{"tributaires", "tributair"},
		{"tric", "tric"},
		{"tricolore", "tricolor"},
		{"trinquette", "trinquet"},
		{"trio", "trio"},
		{"triompha", "triomph"},
		{"triomphait", "triomph"},
		{"triomphale", "triomphal"},
		{"triomphant", "triomph"},
		{"triomphante", "triomph"},
		{"triomphe", "triomph"},
		{"triomphé", "triomph"},
		{"triomphent", "triomphent"},
		{"triompher", "triomph"},
		{"triomphera", "triomph"},
		{"triompherait", "triomph"},
		{"triompheront", "triomph"},
		{"triomphes", "triomph"},
		{"triple", "tripl"},
		{"triplé", "tripl"},
		{"triste", "trist"},
		{"tristement", "trist"},
		{"tristes", "trist"},
		{"tristesse", "tristess"},
		{"tristesses", "tristess"},
		{"triumvirat", "triumvirat"},
		{"triviale", "trivial"},
		{"trois", "trois"},
		{"troisième", "troisiem"},
		{"troisièmes", "troisiem"},
		{"trombe", "tromb"},
		{"trompa", "tromp"},
		{"trompaient", "tromp"},
		{"trompait", "tromp"},
		{"trompant", "tromp"},
		{"trompe", "tromp"},
		{"trompé", "tromp"},
		{"trompée", "tromp"},
		{"tromper", "tromp"},
		{"tromperai", "tromp"},
		{"tromperais", "tromp"},
		{"trompes", "tromp"},
		{"trompés", "tromp"},
		{"trompez", "tromp"},
		{"tronc", "tronc"},
		{"tronçon", "tronçon"},
		{"troncs", "tronc"},
		{"trône", "trôn"},
		{"trop", "trop"},
		{"trophée", "troph"},
		{"tropicale", "tropical"},
		{"tropicales", "tropical"},
		{"tropique", "tropiqu"},
		{"tropiques", "tropiqu"},
		{"troppo", "troppo"},
		{"trot", "trot"},
		{"trottait", "trott"},
		{"trotter", "trott"},
		{"trottoir", "trottoir"},
		{"trottoirs", "trottoir"},
		{"trou", "trou"},
		{"troubla", "troubl"},
		{"troublaient", "troubl"},
		{"troublait", "troubl"},
		{"troublant", "troubl"},
		{"trouble", "troubl"},
		{"troublé", "troubl"},
		{"troublée", "troubl"},
		{"troublées", "troubl"},
		{"troubler", "troubl"},
		{"troublera", "troubl"},
		{"troublèrent", "troubl"},
		{"troubleront", "troubl"},
		{"troubles", "troubl"},
		{"troublés", "troubl"},
		{"troublez", "troubl"},
		{"troupe", "troup"},
		{"troupeau", "troupeau"},
		{"troupes", "troup"},
		{"troupiers", "troupi"},
		{"trous", "trous"},
		{"trousses", "trouss"},
		{"trouva", "trouv"},
		{"trouvai", "trouv"},
		{"trouvaient", "trouv"},
		{"trouvais", "trouv"},
		{"trouvait", "trouv"},
		{"trouvant", "trouv"},
		{"trouvât", "trouv"},
		{"trouve", "trouv"},
		{"trouvé", "trouv"},
		{"trouvée", "trouv"},
		{"trouvées", "trouv"},
		{"trouvent", "trouvent"},
		{"trouver", "trouv"},
		{"trouvera", "trouv"},
		{"trouverai", "trouv"},
		{"trouveraient", "trouv"},
		{"trouverais", "trouv"},
		{"trouverait", "trouv"},
		{"trouveras", "trouv"},
		{"trouvèrent", "trouv"},
		{"trouverez", "trouv"},
		{"trouveriez", "trouv"},
		{"trouverions", "trouv"},
		{"trouverons", "trouv"},
		{"trouveront", "trouv"},
		{"trouves", "trouv"},
		{"trouvés", "trouv"},
		{"trouvez", "trouv"},
		{"trouvons", "trouvon"},
		{"truites", "truit"},
		{"trunk", "trunk"},
		{"tu", "tu"},
		{"tua", "tu"},
		{"tuais", "tu"},
		{"tuait", "tu"},
		{"tuant", "tu"},
		{"tudieu", "tudieu"},
		{"tue", "tu"},
		{"tué", "tu"},
		{"tuée", "tu"},
		{"tuent", "tuent"},
		{"tuer", "tu"},
		{"tuera", "tu"},
		{"tuerai", "tu"},
		{"tuerais", "tu"},
		{"tuerait", "tu"},
		{"tués", "tu"},
		{"tuez", "tu"},
		{"tuileries", "tuiler"},
		{"tuiles", "tuil"},
		{"tuilla", "tuill"},
		{"tumulte", "tumult"},
		{"tumultueuse", "tumultu"},
		{"tunique", "tuniqu"},
		{"tunnel", "tunnel"},
		{"tunnels", "tunnel"},
		{"turban", "turban"},
		{"turbans", "turban"},
		{"turin", "turin"},
		{"turn", "turn"},
		{"turpitude", "turpitud"},
		{"tussaud", "tussaud"},
		{"tut", "tut"},
		{"tutoie", "tutoi"},
		{"tutoiement", "tutoi"},
		{"tutoyait", "tutoi"},
		{"tutoyer", "tutoi"},
		{"tutti", "tutt"},
		{"tutto", "tutto"},
		{"tuyau", "tuyau"},
		{"tuyaux", "tuyal"},
		{"twelfth", "twelfth"},
		{"two", "two"},
		{"tympaniser", "tympanis"},
		{"type", "typ"},
		{"types", "typ"},
		{"typhon", "typhon"},
		{"tyran", "tyran"},
		{"tyrannie", "tyrann"},
		{"tyrannisait", "tyrannis"},
		{"tyrans", "tyran"},
		{"uçaf", "uçaf"},
		{"uddaul", "uddaul"},
		{"ultimatum", "ultimatum"},
		{"ultra", "ultra"},
		{"ultras", "ultras"},
		{"ulysse", "ulyss"},
		{"un", "un"},
		{"unam", "unam"},
		{"unanime", "unanim"},
		{"unanimement", "unanim"},
		{"uncertain", "uncertain"},
		{"une", "une"},
		{"unes", "une"},
		{"uni", "uni"},
		{"unie", "uni"},
		{"unième", "uniem"},
		{"unies", "uni"},
		{"uniforme", "uniform"},
		{"uniformes", "uniform"},
		{"union", "union"},
		{"unique", "uniqu"},
		{"uniquement", "uniqu"},
		{"unis", "unis"},
		{"unissaient", "uniss"},
		{"unissait", "uniss"},
		{"unissez", "uniss"},
		{"univers", "univer"},
		{"universel", "universel"},
		{"universelle", "universel"},
		{"uns", "un"},
		{"uranus", "uranus"},
		{"urbanité", "urban"},
		{"urne", "urne"},
		{"usa", "usa"},
		{"usage", "usag"},
		{"usages", "usag"},
		{"usant", "usant"},
		{"use", "use"},
		{"usé", "usé"},
		{"user", "user"},
		{"usine", "usin"},
		{"ustensile", "ustensil"},
		{"usure", "usur"},
		{"usuriers", "usuri"},
		{"usurpateur", "usurp"},
		{"usurpations", "usurp"},
		{"usurpe", "usurp"},
		{"usurpé", "usurp"},
		{"usurpées", "usurp"},
		{"usurpés", "usurp"},
		{"utah", "utah"},
		{"utile", "util"},
		{"utiles", "util"},
		{"utilisaient", "utilis"},
		{"utiliser", "utilis"},
		{"utilité", "util"},
		{"uzeri", "uzer"},
		{"uzès", "uzes"},
		{"v", "v"},
		{"va", "va"},
		{"vacances", "vacanc"},
		{"vacant", "vac"},
		{"vacante", "vac"},
		{"vacarme", "vacarm"},
		{"vache", "vach"},
		{"vacillais", "vacill"},
		{"vagabond", "vagabond"},
		{"vagabonde", "vagabond"},
		{"vague", "vagu"},
		{"vaguement", "vagu"},
		{"vagues", "vagu"},
		{"vaille", "vaill"},
		{"vain", "vain"},
		{"vaincre", "vaincr"},
		{"vaincu", "vaincu"},
		{"vaincue", "vaincu"},
		{"vaincus", "vaincus"},
		{"vaine", "vain"},
		{"vainement", "vain"},
		{"vaines", "vain"},
		{"vainqueur", "vainqueur"},
		{"vains", "vain"},
		{"vais", "vais"},
		{"vaisseau", "vaisseau"},
		{"valable", "valabl"},
		{"valables", "valabl"},
		{"valaient", "val"},
		{"valait", "val"},
		{"valant", "val"},
		{"vale", "val"},
		{"valenod", "valenod"},
		{"valent", "valent"},
		{"valère", "valer"},
		{"valet", "valet"},
		{"valets", "valet"},
		{"valeur", "valeur"},
		{"vallee", "valle"},
		{"vallée", "vall"},
		{"vallées", "vall"},
		{"valoir", "valoir"},
		{"valois", "valois"},
		{"valserra", "valserr"},
		{"valu", "valu"},
		{"valurent", "valurent"},
		{"valut", "valut"},
		{"vandalisme", "vandal"},
		{"vane", "van"},
		{"vaneau", "vaneau"},
		{"vanité", "vanit"},
		{"vaniteuse", "vanit"},
		{"vaniteux", "vanit"},
		{"vanta", "vant"},
		{"vantait", "vant"},
		{"vanté", "vant"},
		{"vantent", "vantent"},
		{"vanter", "vant"},
		{"vanvitelli", "vanvitel"},
		{"vapeur", "vapeur"},
		{"vapeurs", "vapeur"},
		{"varèse", "vares"},
		{"variaient", "vari"},
		{"varie", "var"},
		{"varié", "vari"},
		{"variétés", "variet"},
		{"vas", "vas"},
		{"vase", "vas"},
		{"vases", "vas"},
		{"vasi", "vas"},
		{"vasistas", "vasist"},
		{"vaste", "vast"},
		{"vastes", "vast"},
		{"vau", "vau"},
		{"vaudeville", "vaudevill"},
		{"vaudra", "vaudr"},
		{"vaudrait", "vaudr"},
		{"vaudront", "vaudront"},
		{"vaurien", "vaurien"},
		{"vaut", "vaut"},
		{"vaux", "vaux"},
		{"vécu", "vécu"},
		{"vécurent", "vécurent"},
		{"vécut", "vécut"},
		{"vécût", "vécût"},
		{"vedette", "vedet"},
		{"vega", "veg"},
		{"végétaient", "véget"},
		{"végéter", "véget"},
		{"véhémence", "véhément"},
		{"véhément", "veh"},
		{"véhicule", "véhicul"},
		{"véhiculé", "véhicul"},
		{"véhicules", "véhicul"},
		{"veilla", "veil"},
		{"veillaient", "veil"},
		{"veillait", "veil"},
		{"veillât", "veil"},
		{"veille", "veil"},
		{"veillé", "veil"},
		{"veillent", "veillent"},
		{"veiller", "veil"},
		{"veillerons", "veil"},
		{"veilles", "veil"},
		{"veilleuse", "veilleux"},
		{"veine", "vein"},
		{"veines", "vein"},
		{"velléité", "velléit"},
		{"velléités", "velléit"},
		{"velleja", "vellej"},
		{"velours", "velour"},
		{"vély", "vély"},
		{"venaient", "ven"},
		{"venait", "ven"},
		{"vénales", "vénal"},
		{"vénalité", "vénal"},
		{"venant", "ven"},
		{"venceslas", "vencesl"},
		{"vend", "vend"},
		{"vendait", "vend"},
		{"vendant", "vend"},
		{"vendée", "vend"},
		{"vendent", "vendent"},
		{"vendeur", "vendeur"},
		{"vendeurs", "vendeur"},
		{"vendit", "vend"},
		{"vendrai", "vendr"},
		{"vendrais", "vendr"},
		{"vendrait", "vendr"},
		{"vendre", "vendr"},
		{"vendredi", "vendred"},
		{"vendredis", "vendred"},
		{"vendront", "vendront"},
		{"vends", "vend"},
		{"vendu", "vendu"},
		{"vendue", "vendu"},
		{"vendus", "vendus"},
		{"vénérable", "véner"},
		{"vénérables", "véner"},
		{"vénération", "véner"},
		{"vénère", "véner"},
		{"vénérées", "véner"},
		{"vénérer", "véner"},
		{"venette", "venet"},
		{"venez", "ven"},
		{"venge", "veng"},
		{"vengé", "veng"},
		{"vengea", "veng"},
		{"vengeait", "veng"},
		{"vengeance", "vengeanc"},
		{"vengée", "veng"},
		{"vengent", "vengent"},
		{"venger", "veng"},
		{"vengera", "veng"},
		{"vengerai", "veng"},
		{"veniez", "ven"},
		{"venin", "venin"},
		{"venir", "ven"},
		{"venise", "venis"},
		{"vénitien", "vénitien"},
		{"vénitiens", "vénitien"},
		{"venons", "venon"},
		{"vent", "vent"},
		{"vente", "vent"},
		{"ventes", "vent"},
		{"ventre", "ventr"},
		{"vents", "vent"},
		{"venu", "venu"},
		{"venue", "venu"},
		{"venues", "venu"},
		{"venus", "venus"},
		{"vérandas", "vérand"},
		{"verbal", "verbal"},
		{"verbe", "verb"},
		{"verdâtres", "verdâtr"},
		{"verdict", "verdict"},
		{"verdoyantes", "verdoi"},
		{"verdoyants", "verdoi"},
		{"verdure", "verdur"},
		{"verger", "verg"},
		{"vergogne", "vergogn"},
		{"vergues", "vergu"},
		{"vergy", "vergy"},
		{"véridiques", "vérid"},
		{"vérifia", "vérifi"},
		{"vérification", "vérif"},
		{"vérifié", "vérifi"},
		{"vérifier", "vérifi"},
		{"véritable", "vérit"},
		{"véritablement", "vérit"},
		{"véritables", "vérit"},
		{"vérité", "vérit"},
		{"vérités", "vérit"},
		{"vermeilles", "vermeil"},
		{"vermont", "vermont"},
		{"vermoulu", "vermoulu"},
		{"vermoulue", "vermoulu"},
		{"verna", "vern"},
		{"vernaye", "vernay"},
		{"verne", "vern"},
		{"vernes", "vern"},
		{"verni", "vern"},
		{"vernis", "vern"},
		{"vérole", "vérol"},
		{"verona", "veron"},
		{"vérone", "véron"},
		{"verra", "verr"},
		{"verrai", "verr"},
		{"verraient", "verr"},
		{"verrais", "verr"},
		{"verrait", "verr"},
		{"verras", "verr"},
		{"verre", "verr"},
		{"verres", "verr"},
		{"verrez", "verr"},
		{"verrières", "verri"},
		{"verrons", "verron"},
		{"verront", "verront"},
		{"verrou", "verrou"},
		{"verrous", "verrous"},
		{"verruqueux", "verruqu"},
		{"vers", "ver"},
		{"versa", "vers"},
		{"versailles", "versaill"},
		{"versait", "vers"},
		{"versant", "vers"},
		{"versants", "vers"},
		{"verse", "vers"},
		{"verser", "vers"},
		{"versets", "verset"},
		{"vert", "vert"},
		{"verte", "vert"},
		{"vertement", "vert"},
		{"vertes", "vert"},
		{"vertical", "vertical"},
		{"verticale", "vertical"},
		{"verticalement", "vertical"},
		{"verts", "vert"},
		{"vertu", "vertu"},
		{"vertueuse", "vertueux"},
		{"vertueuses", "vertueux"},
		{"vertueux", "vertueux"},
		{"vertus", "vertus"},
		{"verve", "verv"},
		{"very", "very"},
		{"vespasien", "vespasien"},
		{"vesper", "vesp"},
		{"veste", "vest"},
		{"vestes", "vest"},
		{"vestibule", "vestibul"},
		{"vestige", "vestig"},
		{"vésuve", "vésuv"},
		{"vêtement", "vêt"},
		{"vêtements", "vêt"},
		{"vêtent", "vêtent"},
		{"vétille", "vétill"},
		{"vetturini", "vetturin"},
		{"vetturino", "vetturino"},
		{"vêtu", "vêtu"},
		{"vêtue", "vêtu"},
		{"vêtues", "vêtu"},
		{"vêtus", "vêtus"},
		{"veuf", "veuf"},
		{"veuille", "veuill"},
		{"veuillez", "veuill"},
		{"veuilliez", "veuill"},
		{"veulent", "veulent"},
		{"veut", "veut"},
		{"veuvage", "veuvag"},
		{"veuve", "veuv"},
		{"veuves", "veuv"},
		{"veux", "veux"},
		{"vexait", "vex"},
		{"vexantes", "vex"},
		{"vexations", "vexat"},
		{"vexé", "vex"},
		{"vexer", "vex"},
		{"vexés", "vex"},
		{"vi", "vi"},
		{"via", "vi"},
		{"viaducs", "viaduc"},
		{"viagère", "viager"},
		{"viande", "viand"},
		{"viandes", "viand"},
		{"viatique", "viatiqu"},
		{"vibrations", "vibrat"},
		{"vicaire", "vicair"},
		{"vicaires", "vicair"},
		{"vice", "vic"},
		{"vices", "vic"},
		{"vicieux", "vicieux"},
		{"vicissitudes", "vicissitud"},
		{"vico", "vico"},
		{"vicolo", "vicolo"},
		{"vicomte", "vicomt"},
		{"vicomtes", "vicomt"},
		{"victime", "victim"},
		{"victimes", "victim"},
		{"victoire", "victoir"},
		{"victoires", "victoir"},
		{"victor", "victor"},
		{"victoria", "victori"},
		{"victorieuse", "victori"},
		{"vicvacarma", "vicvacarm"},
		{"vida", "vid"},
		{"vidaient", "vid"},
		{"vidant", "vid"},
		{"vide", "vid"},
		{"vidées", "vid"},
		{"vider", "vid"},
		{"vides", "vid"},
		{"videz", "vid"},
		{"vie", "vi"},
		{"vieil", "vieil"},
		{"vieillard", "vieillard"},
		{"vieillards", "vieillard"},
		{"vieille", "vieil"},
		{"vieilles", "vieil"},
		{"vieillesse", "vieilless"},
		{"vieillie", "vieil"},
		{"vieillir", "vieil"},
		{"vieillit", "vieil"},
		{"viendra", "viendr"},
		{"viendrai", "viendr"},
		{"viendraient", "viendr"},
		{"viendrait", "viendr"},
		{"viendras", "viendr"},
		{"vienne", "vien"},
		{"viennent", "viennent"},
		{"viennes", "vien"},
		{"viens", "vien"},
		{"vient", "vient"},
		{"vierge", "vierg"},
		{"vies", "vi"},
		{"vieux", "vieux"},
		{"views", "view"},
		{"vif", "vif"},
		{"vifs", "vif"},
		{"vigano", "vigano"},
		{"vigilante", "vigil"},
		{"vignano", "vignano"},
		{"vigne", "vign"},
		{"vigoureuse", "vigour"},
		{"vigoureusement", "vigour"},
		{"vigoureuses", "vigour"},
		{"vigoureux", "vigour"},
		{"vigueur", "vigueur"},
		{"viharis", "vihar"},
		{"vii", "vii"},
		{"viii", "vii"},
		{"vil", "vil"},
		{"vilain", "vilain"},
		{"vilaine", "vilain"},
		{"vilaines", "vilain"},
		{"vilains", "vilain"},
		{"vile", "vil"},
		{"vilenie", "vilen"},
		{"vilipendé", "vilipend"},
		{"vilipendée", "vilipend"},
		{"villa", "vill"},
		{"village", "villag"},
		{"villages", "villag"},
		{"ville", "vill"},
		{"villequier", "villequi"},
		{"villes", "vill"},
		{"vils", "vil"},
		{"vîmes", "vîm"},
		{"vin", "vin"},
		{"vinci", "vinc"},
		{"vindhias", "vindhi"},
		{"vindicatif", "vindiqu"},
		{"vindicative", "vindiqu"},
		{"vingt", "vingt"},
		{"vingtaine", "vingtain"},
		{"vingtième", "vingtiem"},
		{"vingts", "vingt"},
		{"vinrent", "vinrent"},
		{"vins", "vin"},
		{"vinsse", "vinss"},
		{"vint", "vint"},
		{"vînt", "vînt"},
		{"viole", "viol"},
		{"violé", "viol"},
		{"violemment", "violent"},
		{"violence", "violenc"},
		{"violent", "violent"},
		{"violentant", "violent"},
		{"violente", "violent"},
		{"violenté", "violent"},
		{"violentes", "violent"},
		{"violents", "violent"},
		{"violer", "viol"},
		{"violes", "viol"},
		{"violets", "violet"},
		{"violette", "violet"},
		{"violettes", "violet"},
		{"violiers", "violi"},
		{"violon", "violon"},
		{"vipère", "viper"},
		{"virago", "virago"},
		{"virent", "virent"},
		{"virgile", "virgil"},
		{"virtue", "virtu"},
		{"virtuose", "virtuos"},
		{"vis", "vis"},
		{"visa", "vis"},
		{"visage", "visag"},
		{"visages", "visag"},
		{"visas", "vis"},
		{"visconti", "viscont"},
		{"visé", "vis"},
		{"visent", "visent"},
		{"viser", "vis"},
		{"viserez", "vis"},
		{"visible", "visibl"},
		{"visiblement", "visibl"},
		{"visibles", "visibl"},
		{"vision", "vision"},
		{"visions", "vision"},
		{"visita", "visit"},
		{"visitation", "visit"},
		{"visite", "visit"},
		{"visité", "visit"},
		{"visitées", "visit"},
		{"visiter", "visit"},
		{"visitera", "visit"},
		{"visites", "visit"},
		{"viso", "viso"},
		{"visons", "vison"},
		{"visu", "visu"},
		{"vit", "vit"},
		{"vît", "vît"},
		{"vite", "vit"},
		{"vitesse", "vitess"},
		{"vitesses", "vitess"},
		{"vitraux", "vitral"},
		{"vitre", "vitr"},
		{"vitres", "vitr"},
		{"viva", "viv"},
		{"vivacité", "vivac"},
		{"vivaient", "viv"},
		{"vivais", "viv"},
		{"vivait", "viv"},
		{"vivandière", "vivandi"},
		{"vivandières", "vivandi"},
		{"vivant", "viv"},
		{"vivante", "viv"},
		{"vivants", "viv"},
		{"vive", "viv"},
		{"vivement", "viv"},
		{"vivent", "vivent"},
		{"vives", "viv"},
		{"vivez", "viv"},
		{"vivons", "vivon"},
		{"vivoter", "vivot"},
		{"vivra", "vivr"},
		{"vivrai", "vivr"},
		{"vivrait", "vivr"},
		{"vivre", "vivr"},
		{"vivres", "vivr"},
		{"vivrez", "vivr"},
		{"vivrions", "vivrion"},
		{"vivrons", "vivron"},
		{"vivront", "vivront"},
		{"vocation", "vocat"},
		{"vociférations", "vocifer"},
		{"voeu", "voeu"},
		{"voeux", "voeux"},
		{"voguer", "vogu"},
		{"voi", "voi"},
		{"voici", "voic"},
		{"voie", "voi"},
		{"voient", "voient"},
		{"voies", "voi"},
		{"voilà", "voilà"},
		{"voile", "voil"},
		{"voilé", "voil"},
		{"voilée", "voil"},
		{"voilées", "voil"},
		{"voiles", "voil"},
		{"voilés", "voil"},
		{"voilure", "voilur"},
		{"voir", "voir"},
		{"vois", "vois"},
		{"voisin", "voisin"},
		{"voisinage", "voisinag"},
		{"voisine", "voisin"},
		{"voisines", "voisin"},
		{"voisins", "voisin"},
		{"voit", "voit"},
		{"voiture", "voitur"},
		{"voitures", "voitur"},
		{"voix", "voix"},
		{"vol", "vol"},
		{"volaient", "vol"},
		{"volaille", "volaill"},
		{"volait", "vol"},
		{"volant", "vol"},
		{"volants", "vol"},
		{"volatiles", "volatil"},
		{"volcan", "volcan"},
		{"vole", "vol"},
		{"volé", "vol"},
		{"volée", "vol"},
		{"volées", "vol"},
		{"volent", "volent"},
		{"voler", "vol"},
		{"volera", "vol"},
		{"volerai", "vol"},
		{"volerait", "vol"},
		{"volés", "vol"},
		{"volet", "volet"},
		{"volets", "volet"},
		{"voleur", "voleur"},
		{"voleurs", "voleur"},
		{"volez", "vol"},
		{"volière", "voli"},
		{"volontaire", "volontair"},
		{"volontairement", "volontair"},
		{"volontaires", "volontair"},
		{"volonté", "volont"},
		{"volontés", "volont"},
		{"volontiers", "volonti"},
		{"vols", "vol"},
		{"voltaire", "voltair"},
		{"voltige", "voltig"},
		{"voltigeant", "voltig"},
		{"voltiger", "voltig"},
		{"voltigeur", "voltigeur"},
		{"volubilité", "volubil"},
		{"volume", "volum"},
		{"volumes", "volum"},
		{"volupté", "volupt"},
		{"voluptés", "volupt"},
		{"voluptueuse", "voluptu"},
		{"voluptueusement", "voluptu"},
		{"voluptueuses", "voluptu"},
		{"voluptueux", "voluptu"},
		{"volutes", "volut"},
		{"vomero", "vomero"},
		{"vomir", "vom"},
		{"vomissait", "vom"},
		{"vont", "vont"},
		{"voraces", "vorac"},
		{"vos", "vos"},
		{"votaient", "vot"},
		{"votant", "vot"},
		{"vote", "vot"},
		{"voté", "vot"},
		{"votée", "vot"},
		{"voter", "vot"},
		{"votes", "vot"},
		{"votre", "votr"},
		{"vôtre", "vôtr"},
		{"vôtres", "vôtr"},
		{"voudra", "voudr"},
		{"voudraient", "voudr"},
		{"voudrais", "voudr"},
		{"voudrait", "voudr"},
		{"voudras", "voudr"},
		{"voudrez", "voudr"},
		{"voudriez", "voudr"},
		{"voudrions", "voudrion"},
		{"voudront", "voudront"},
		{"voué", "vou"},
		{"vouent", "vouent"},
		{"vouer", "vou"},
		{"vouèrent", "vou"},
		{"voulaient", "voul"},
		{"voulais", "voul"},
		{"voulait", "voul"},
		{"voulant", "voul"},
		{"voulez", "voul"},
		{"vouliez", "voul"},
		{"vouloir", "vouloir"},
		{"voulons", "voulon"},
		{"voulu", "voulu"},
		{"voulues", "voulu"},
		{"voulurent", "voulurent"},
		{"voulus", "voulus"},
		{"voulussent", "voulussent"},
		{"voulut", "voulut"},
		{"voulût", "voulût"},
		{"vous", "vous"},
		{"voûte", "voût"},
		{"voûtes", "voût"},
		{"voyage", "voyag"},
		{"voyagé", "voyag"},
		{"voyageait", "voyag"},
		{"voyageant", "voyag"},
		{"voyagent", "voyagent"},
		{"voyageons", "voyageon"},
		{"voyager", "voyag"},
		{"voyages", "voyag"},
		{"voyageur", "voyageur"},
		{"voyageurs", "voyageur"},
		{"voyageuse", "voyag"},
		{"voyagez", "voyag"},
		{"voyaient", "voi"},
		{"voyais", "voi"},
		{"voyait", "voi"},
		{"voyant", "voi"},
		{"voyez", "voi"},
		{"voyiez", "voi"},
		{"voyons", "voyon"},
		{"vrai", "vrai"},
		{"vraie", "vrai"},
		{"vraies", "vrai"},
		{"vraiment", "vrai"},
		{"vrais", "vrais"},
		{"vraisemblable", "vraisembl"},
		{"vraisemblablement", "vraisembl"},
		{"vu", "vu"},
		{"vue", "vu"},
		{"vues", "vu"},
		{"vulgaire", "vulgair"},
		{"vulgairement", "vulgair"},
		{"vulgaires", "vulgair"},
		{"vulgarité", "vulgar"},
		{"vus", "vus"},
		{"w", "w"},
		{"wagon", "wagon"},
		{"wagons", "wagon"},
		{"wagram", "wagram"},
		{"wahsatch", "wahsatch"},
		{"walbah", "walbah"},
		{"walter", "walt"},
		{"wandering", "wandering"},
		{"was", "was"},
		{"washington", "washington"},
		{"waterloo", "waterloo"},
		{"wayne", "wayn"},
		{"we", "we"},
		{"weber", "web"},
		{"wellington", "wellington"},
		{"were", "wer"},
		{"west", "west"},
		{"wharf", "wharf"},
		{"wharfs", "wharf"},
		{"which", "which"},
		{"whisky", "whisky"},
		{"whisnou", "whisnou"},
		{"whist", "whist"},
		{"white", "whit"},
		{"will", "will"},
		{"william", "william"},
		{"wilson", "wilson"},
		{"with", "with"},
		{"withdrew", "withdrew"},
		{"word", "word"},
		{"would", "would"},
		{"wyoming", "wyoming"},
		{"x", "x"},
		{"xavier", "xavi"},
		{"xérès", "xéres"},
		{"xi", "xi"},
		{"xii", "xii"},
		{"xiii", "xii"},
		{"xiv", "xiv"},
		{"xix", "xix"},
		{"xv", "xv"},
		{"xvi", "xvi"},
		{"xvii", "xvii"},
		{"xviii", "xvii"},
		{"xx", "xx"},
		{"xxi", "xxi"},
		{"xxii", "xxii"},
		{"xxiii", "xxii"},
		{"xxiv", "xxiv"},
		{"xxix", "xxix"},
		{"xxv", "xxv"},
		{"xxvi", "xxvi"},
		{"xxvii", "xxvii"},
		{"xxviii", "xxvii"},
		{"xxx", "xxx"},
		{"xxxi", "xxxi"},
		{"xxxii", "xxxii"},
		{"xxxiii", "xxxii"},
		{"xxxiv", "xxxiv"},
		{"xxxv", "xxxv"},
		{"xxxvi", "xxxvi"},
		{"xxxvii", "xxxvii"},
		{"y", "y"},
		{"yacht", "yacht"},
		{"yachts", "yacht"},
		{"yakounines", "yakounin"},
		{"yankee", "yanke"},
		{"yeddo", "yeddo"},
		{"yert", "yert"},
		{"yet", "yet"},
		{"yeux", "yeux"},
		{"yokohama", "yokoham"},
		{"york", "york"},
		{"young", "young"},
		{"zambajon", "zambajon"},
		{"zeb", "zeb"},
		{"zébrés", "zebr"},
		{"zébus", "zébus"},
		{"zèle", "zel"},
		{"zélés", "zel"},
		{"zénith", "zénith"},
		{"zigzags", "zigzag"},
		{"zingarelli", "zingarel"},
		{"zonders", "zonder"},
		{"zoroastre", "zoroastr"},
		{"zurla", "zurl"},
	}
	for _, testCase := range testCases {
		result := french.Stem(testCase.in, true)
		if result != testCase.out {
			t.Errorf("Expected %v -> %v, but got %v", testCase.in, testCase.out, result)
		}
	}
}
