// Copyright 2024 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package merchantapi provides access to the Merchant API.
//
// For product documentation, see: https://developers.devsite.corp.google.com/merchant/api
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/merchantapi/reviews_v1beta"
//	...
//	ctx := context.Background()
//	merchantapiService, err := merchantapi.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	merchantapiService, err := merchantapi.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	merchantapiService, err := merchantapi.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package merchantapi // import "google.golang.org/api/merchantapi/reviews_v1beta"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version

const apiId = "merchantapi:reviews_v1beta"
const apiName = "merchantapi"
const apiVersion = "reviews_v1beta"
const basePath = "https://merchantapi.googleapis.com/"
const basePathTemplate = "https://merchantapi.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://merchantapi.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// Manage your product listings and accounts for Google Shopping
	ContentScope = "https://www.googleapis.com/auth/content"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/content",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s, err := New(client)
	if err != nil {
		return nil, err
	}
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	s := &Service{client: client, BasePath: basePath}
	s.Accounts = NewAccountsService(s)
	return s, nil
}

type Service struct {
	client    *http.Client
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Accounts *AccountsService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewAccountsService(s *Service) *AccountsService {
	rs := &AccountsService{s: s}
	rs.MerchantReviews = NewAccountsMerchantReviewsService(s)
	rs.ProductReviews = NewAccountsProductReviewsService(s)
	return rs
}

type AccountsService struct {
	s *Service

	MerchantReviews *AccountsMerchantReviewsService

	ProductReviews *AccountsProductReviewsService
}

func NewAccountsMerchantReviewsService(s *Service) *AccountsMerchantReviewsService {
	rs := &AccountsMerchantReviewsService{s: s}
	return rs
}

type AccountsMerchantReviewsService struct {
	s *Service
}

func NewAccountsProductReviewsService(s *Service) *AccountsProductReviewsService {
	rs := &AccountsProductReviewsService{s: s}
	return rs
}

type AccountsProductReviewsService struct {
	s *Service
}

// CustomAttribute: A message that represents custom attributes. Exactly one of
// `value` or `group_values` must not be empty.
type CustomAttribute struct {
	// GroupValues: Subattributes within this attribute group. If `group_values` is
	// not empty, `value` must be empty.
	GroupValues []*CustomAttribute `json:"groupValues,omitempty"`
	// Name: The name of the attribute.
	Name string `json:"name,omitempty"`
	// Value: The value of the attribute. If `value` is not empty, `group_values`
	// must be empty.
	Value string `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "GroupValues") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "GroupValues") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CustomAttribute) MarshalJSON() ([]byte, error) {
	type NoMethod CustomAttribute
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Empty: A generic empty message that you can re-use to avoid defining
// duplicated empty messages in your APIs. A typical example is to use it as
// the request or the response type of an API method. For instance: service Foo
// { rpc Bar(google.protobuf.Empty) returns (google.protobuf.Empty); }
type Empty struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// ListMerchantReviewsResponse: Response message for the `ListMerchantsReview`
// method.
type ListMerchantReviewsResponse struct {
	// MerchantReviews: The merchant review.
	MerchantReviews []*MerchantReview `json:"merchantReviews,omitempty"`
	// NextPageToken: The token to retrieve the next page of results.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "MerchantReviews") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MerchantReviews") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListMerchantReviewsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListMerchantReviewsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListProductReviewsResponse: response message for the ListProductReviews
// method.
type ListProductReviewsResponse struct {
	// NextPageToken: A token, which can be sent as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// ProductReviews: The product review.
	ProductReviews []*ProductReview `json:"productReviews,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListProductReviewsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListProductReviewsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MerchantReview: A review for a merchant. For more information, see
// Introduction to Merchant Review Feeds
// (https://developers.google.com/merchant-review-feeds)
type MerchantReview struct {
	// Attributes: Optional. A list of merchant review attributes.
	Attributes *MerchantReviewAttributes `json:"attributes,omitempty"`
	// CustomAttributes: Required. A list of custom (merchant-provided) attributes.
	// It can also be used for submitting any attribute of the data specification
	// in its generic form (for example, `{ "name": "size type", "value": "regular"
	// }`). This is useful for submitting attributes not explicitly exposed by the
	// API, such as experimental attributes. Maximum allowed number of characters
	// for each custom attribute is 10240 (represents sum of characters for name
	// and value). Maximum 2500 custom attributes can be set per product, with
	// total size of 102.4kB. Underscores in custom attribute names are replaced by
	// spaces upon insertion.
	CustomAttributes []*CustomAttribute `json:"customAttributes,omitempty"`
	// DataSource: Output only. The primary data source of the merchant review.
	DataSource string `json:"dataSource,omitempty"`
	// MerchantReviewId: Required. The user provided merchant review ID to uniquely
	// identify the merchant review.
	MerchantReviewId string `json:"merchantReviewId,omitempty"`
	// MerchantReviewStatus: Output only. The status of a merchant review, data
	// validation issues, that is, information about a merchant review computed
	// asynchronously.
	MerchantReviewStatus *MerchantReviewStatus `json:"merchantReviewStatus,omitempty"`
	// Name: Identifier. The name of the merchant review. Format:
	// "{merchantreview.name=accounts/{account}/merchantReviews/{merchantReview}}"
	Name string `json:"name,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Attributes") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Attributes") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MerchantReview) MarshalJSON() ([]byte, error) {
	type NoMethod MerchantReview
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MerchantReviewAttributes: Attributes.
type MerchantReviewAttributes struct {
	// CollectionMethod: Optional. The method used to collect the review.
	//
	// Possible values:
	//   "COLLECTION_METHOD_UNSPECIFIED" - Collection method unspecified.
	//   "MERCHANT_UNSOLICITED" - The user was not responding to a specific
	// solicitation when they submitted the review.
	//   "POINT_OF_SALE" - The user submitted the review in response to a
	// solicitation when the user placed an order.
	//   "AFTER_FULFILLMENT" - The user submitted the review in response to a
	// solicitation after fulfillment of the user's order.
	CollectionMethod string `json:"collectionMethod,omitempty"`
	// Content: Required. This should be any freeform text provided by the user and
	// should not be truncated. If multiple responses to different questions are
	// provided, all responses should be included, with the minimal context for the
	// responses to make sense. Context should not be provided if questions were
	// left unanswered.
	Content string `json:"content,omitempty"`
	// IsAnonymous: Optional. Set to true if the reviewer should remain anonymous.
	IsAnonymous bool `json:"isAnonymous,omitempty"`
	// MaxRating: Optional. The maximum possible number for the rating. The value
	// of the max rating must be greater than the value of the min rating.
	MaxRating int64 `json:"maxRating,omitempty,string"`
	// MerchantDisplayName: Optional. Human-readable display name for the merchant.
	MerchantDisplayName string `json:"merchantDisplayName,omitempty"`
	// MerchantId: Required. Must be unique and stable across all requests. In
	// other words, if a request today and another 90 days ago refer to the same
	// merchant, they must have the same id.
	MerchantId string `json:"merchantId,omitempty"`
	// MerchantLink: Optional. URL to the merchant's main website. Do not use a
	// redirect URL for this value. In other words, the value should point directly
	// to the merchant's site.
	MerchantLink string `json:"merchantLink,omitempty"`
	// MerchantRatingLink: Optional. URL to the landing page that hosts the reviews
	// for this merchant. Do not use a redirect URL.
	MerchantRatingLink string `json:"merchantRatingLink,omitempty"`
	// MinRating: Optional. The minimum possible number for the rating. This should
	// be the worst possible rating and should not be a value for no rating.
	MinRating int64 `json:"minRating,omitempty,string"`
	// Rating: Optional. The reviewer's overall rating of the merchant.
	Rating float64 `json:"rating,omitempty"`
	// ReviewCountry: Optional. The country where the reviewer made the order
	// defined by ISO 3166-1 Alpha-2 Country Code.
	ReviewCountry string `json:"reviewCountry,omitempty"`
	// ReviewLanguage: Required. The language of the review defined by BCP-47
	// language code.
	ReviewLanguage string `json:"reviewLanguage,omitempty"`
	// ReviewTime: Required. The timestamp indicating when the review was written.
	ReviewTime string `json:"reviewTime,omitempty"`
	// ReviewerId: Optional. A permanent, unique identifier for the author of the
	// review in the publisher's system.
	ReviewerId string `json:"reviewerId,omitempty"`
	// ReviewerUsername: Optional. Display name of the review author.
	ReviewerUsername string `json:"reviewerUsername,omitempty"`
	// Title: Optional. The title of the review.
	Title string `json:"title,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CollectionMethod") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CollectionMethod") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MerchantReviewAttributes) MarshalJSON() ([]byte, error) {
	type NoMethod MerchantReviewAttributes
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *MerchantReviewAttributes) UnmarshalJSON(data []byte) error {
	type NoMethod MerchantReviewAttributes
	var s1 struct {
		Rating gensupport.JSONFloat64 `json:"rating"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Rating = float64(s1.Rating)
	return nil
}

// MerchantReviewDestinationStatus: The destination status of the merchant
// review status.
type MerchantReviewDestinationStatus struct {
	// ReportingContext: Output only. The name of the reporting context.
	//
	// Possible values:
	//   "REPORTING_CONTEXT_ENUM_UNSPECIFIED" - Not specified.
	//   "SHOPPING_ADS" - [Shopping
	// ads](https://support.google.com/merchants/answer/6149970).
	//   "DISCOVERY_ADS" - Deprecated: Use `DEMAND_GEN_ADS` instead. [Discovery and
	// Demand Gen ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS" - [Demand Gen
	// ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS_DISCOVER_SURFACE" - [Demand Gen ads on Discover
	// surface](https://support.google.com/merchants/answer/13389785).
	//   "VIDEO_ADS" - [Video
	// ads](https://support.google.com/google-ads/answer/6340491).
	//   "DISPLAY_ADS" - [Display
	// ads](https://support.google.com/merchants/answer/6069387).
	//   "LOCAL_INVENTORY_ADS" - [Local inventory
	// ads](https://support.google.com/merchants/answer/3271956).
	//   "VEHICLE_INVENTORY_ADS" - [Vehicle inventory
	// ads](https://support.google.com/merchants/answer/11544533).
	//   "FREE_LISTINGS" - [Free product
	// listings](https://support.google.com/merchants/answer/9199328).
	//   "FREE_LOCAL_LISTINGS" - [Free local product
	// listings](https://support.google.com/merchants/answer/9825611).
	//   "FREE_LOCAL_VEHICLE_LISTINGS" - [Free local vehicle
	// listings](https://support.google.com/merchants/answer/11544533).
	//   "YOUTUBE_SHOPPING" - [YouTube
	// Shopping](https://support.google.com/merchants/answer/13478370).
	//   "CLOUD_RETAIL" - [Cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "LOCAL_CLOUD_RETAIL" - [Local cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "PRODUCT_REVIEWS" - [Product
	// Reviews](https://support.google.com/merchants/answer/14620732).
	//   "MERCHANT_REVIEWS" - [Merchant
	// Reviews](https://developers.google.com/merchant-review-feeds).
	//   "YOUTUBE_CHECKOUT" - [YouTube Checkout](
	ReportingContext string `json:"reportingContext,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ReportingContext") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ReportingContext") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MerchantReviewDestinationStatus) MarshalJSON() ([]byte, error) {
	type NoMethod MerchantReviewDestinationStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MerchantReviewItemLevelIssue: The ItemLevelIssue of the merchant review
// status.
type MerchantReviewItemLevelIssue struct {
	// Attribute: Output only. The attribute's name, if the issue is caused by a
	// single attribute.
	Attribute string `json:"attribute,omitempty"`
	// Code: Output only. The error code of the issue.
	Code string `json:"code,omitempty"`
	// Description: Output only. A short issue description in English.
	Description string `json:"description,omitempty"`
	// Detail: Output only. A detailed issue description in English.
	Detail string `json:"detail,omitempty"`
	// Documentation: Output only. The URL of a web page to help with resolving
	// this issue.
	Documentation string `json:"documentation,omitempty"`
	// ReportingContext: Output only. The reporting context the issue applies to.
	//
	// Possible values:
	//   "REPORTING_CONTEXT_ENUM_UNSPECIFIED" - Not specified.
	//   "SHOPPING_ADS" - [Shopping
	// ads](https://support.google.com/merchants/answer/6149970).
	//   "DISCOVERY_ADS" - Deprecated: Use `DEMAND_GEN_ADS` instead. [Discovery and
	// Demand Gen ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS" - [Demand Gen
	// ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS_DISCOVER_SURFACE" - [Demand Gen ads on Discover
	// surface](https://support.google.com/merchants/answer/13389785).
	//   "VIDEO_ADS" - [Video
	// ads](https://support.google.com/google-ads/answer/6340491).
	//   "DISPLAY_ADS" - [Display
	// ads](https://support.google.com/merchants/answer/6069387).
	//   "LOCAL_INVENTORY_ADS" - [Local inventory
	// ads](https://support.google.com/merchants/answer/3271956).
	//   "VEHICLE_INVENTORY_ADS" - [Vehicle inventory
	// ads](https://support.google.com/merchants/answer/11544533).
	//   "FREE_LISTINGS" - [Free product
	// listings](https://support.google.com/merchants/answer/9199328).
	//   "FREE_LOCAL_LISTINGS" - [Free local product
	// listings](https://support.google.com/merchants/answer/9825611).
	//   "FREE_LOCAL_VEHICLE_LISTINGS" - [Free local vehicle
	// listings](https://support.google.com/merchants/answer/11544533).
	//   "YOUTUBE_SHOPPING" - [YouTube
	// Shopping](https://support.google.com/merchants/answer/13478370).
	//   "CLOUD_RETAIL" - [Cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "LOCAL_CLOUD_RETAIL" - [Local cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "PRODUCT_REVIEWS" - [Product
	// Reviews](https://support.google.com/merchants/answer/14620732).
	//   "MERCHANT_REVIEWS" - [Merchant
	// Reviews](https://developers.google.com/merchant-review-feeds).
	//   "YOUTUBE_CHECKOUT" - [YouTube Checkout](
	ReportingContext string `json:"reportingContext,omitempty"`
	// Resolution: Output only. Whether the issue can be resolved by the merchant.
	Resolution string `json:"resolution,omitempty"`
	// Severity: Output only. How this issue affects serving of the merchant
	// review.
	//
	// Possible values:
	//   "SEVERITY_UNSPECIFIED" - Not specified.
	//   "NOT_IMPACTED" - This issue represents a warning and does not have a
	// direct affect on the merchant review.
	//   "DISAPPROVED" - Issue disapproves the merchant review.
	Severity string `json:"severity,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Attribute") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Attribute") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MerchantReviewItemLevelIssue) MarshalJSON() ([]byte, error) {
	type NoMethod MerchantReviewItemLevelIssue
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MerchantReviewStatus: The status of a merchant review, data validation
// issues, that is, information about a merchant review computed
// asynchronously.
type MerchantReviewStatus struct {
	// CreateTime: Output only. Date on which the item has been created, in ISO
	// 8601 (http://en.wikipedia.org/wiki/ISO_8601) format.
	CreateTime string `json:"createTime,omitempty"`
	// DestinationStatuses: Output only. The intended destinations for the merchant
	// review.
	DestinationStatuses []*MerchantReviewDestinationStatus `json:"destinationStatuses,omitempty"`
	// ItemLevelIssues: Output only. A list of all issues associated with the
	// merchant review.
	ItemLevelIssues []*MerchantReviewItemLevelIssue `json:"itemLevelIssues,omitempty"`
	// LastUpdateTime: Output only. Date on which the item has been last updated,
	// in ISO 8601 (http://en.wikipedia.org/wiki/ISO_8601) format.
	LastUpdateTime string `json:"lastUpdateTime,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CreateTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CreateTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MerchantReviewStatus) MarshalJSON() ([]byte, error) {
	type NoMethod MerchantReviewStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductChange: The change that happened to the product including old value,
// new value, country code as the region code and reporting context.
type ProductChange struct {
	// NewValue: The new value of the changed resource or attribute.
	NewValue string `json:"newValue,omitempty"`
	// OldValue: The old value of the changed resource or attribute.
	OldValue string `json:"oldValue,omitempty"`
	// RegionCode: Countries that have the change (if applicable)
	RegionCode string `json:"regionCode,omitempty"`
	// ReportingContext: Reporting contexts that have the change (if applicable)
	//
	// Possible values:
	//   "REPORTING_CONTEXT_ENUM_UNSPECIFIED" - Not specified.
	//   "SHOPPING_ADS" - [Shopping
	// ads](https://support.google.com/merchants/answer/6149970).
	//   "DISCOVERY_ADS" - Deprecated: Use `DEMAND_GEN_ADS` instead. [Discovery and
	// Demand Gen ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS" - [Demand Gen
	// ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS_DISCOVER_SURFACE" - [Demand Gen ads on Discover
	// surface](https://support.google.com/merchants/answer/13389785).
	//   "VIDEO_ADS" - [Video
	// ads](https://support.google.com/google-ads/answer/6340491).
	//   "DISPLAY_ADS" - [Display
	// ads](https://support.google.com/merchants/answer/6069387).
	//   "LOCAL_INVENTORY_ADS" - [Local inventory
	// ads](https://support.google.com/merchants/answer/3271956).
	//   "VEHICLE_INVENTORY_ADS" - [Vehicle inventory
	// ads](https://support.google.com/merchants/answer/11544533).
	//   "FREE_LISTINGS" - [Free product
	// listings](https://support.google.com/merchants/answer/9199328).
	//   "FREE_LOCAL_LISTINGS" - [Free local product
	// listings](https://support.google.com/merchants/answer/9825611).
	//   "FREE_LOCAL_VEHICLE_LISTINGS" - [Free local vehicle
	// listings](https://support.google.com/merchants/answer/11544533).
	//   "YOUTUBE_SHOPPING" - [YouTube
	// Shopping](https://support.google.com/merchants/answer/13478370).
	//   "CLOUD_RETAIL" - [Cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "LOCAL_CLOUD_RETAIL" - [Local cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "PRODUCT_REVIEWS" - [Product
	// Reviews](https://support.google.com/merchants/answer/14620732).
	//   "MERCHANT_REVIEWS" - [Merchant
	// Reviews](https://developers.google.com/merchant-review-feeds).
	//   "YOUTUBE_CHECKOUT" - [YouTube Checkout](
	ReportingContext string `json:"reportingContext,omitempty"`
	// ForceSendFields is a list of field names (e.g. "NewValue") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NewValue") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductChange) MarshalJSON() ([]byte, error) {
	type NoMethod ProductChange
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductReview: A review for a product. For more information, see
// Introduction to Product Review Feeds
// (https://developers.google.com/product-review-feeds)
type ProductReview struct {
	// Attributes: Optional. A list of product review attributes.
	Attributes *ProductReviewAttributes `json:"attributes,omitempty"`
	// CustomAttributes: Optional. A list of custom (merchant-provided) attributes.
	CustomAttributes []*CustomAttribute `json:"customAttributes,omitempty"`
	// DataSource: Output only. The primary data source of the product review.
	DataSource string `json:"dataSource,omitempty"`
	// Name: Identifier. The name of the product review. Format:
	// "{productreview.name=accounts/{account}/productReviews/{productReview}}"
	Name string `json:"name,omitempty"`
	// ProductReviewId: Required. The permanent, unique identifier for the product
	// review in the publisher’s system.
	ProductReviewId string `json:"productReviewId,omitempty"`
	// ProductReviewStatus: Output only. The status of a product review, data
	// validation issues, that is, information about a product review computed
	// asynchronously.
	ProductReviewStatus *ProductReviewStatus `json:"productReviewStatus,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Attributes") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Attributes") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductReview) MarshalJSON() ([]byte, error) {
	type NoMethod ProductReview
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductReviewAttributes: Attributes.
type ProductReviewAttributes struct {
	// AggregatorName: Optional. The name of the aggregator of the product reviews.
	// A publisher may use a reviews aggregator to manage reviews and provide the
	// feeds. This element indicates the use of an aggregator and contains
	// information about the aggregator.
	AggregatorName string `json:"aggregatorName,omitempty"`
	// Asins: Optional. Contains ASINs (Amazon Standard Identification Numbers)
	// associated with a product.
	Asins []string `json:"asins,omitempty"`
	// Brands: Optional. Contains brand names associated with a product.
	Brands []string `json:"brands,omitempty"`
	// CollectionMethod: Optional. The method used to collect the review.
	//
	// Possible values:
	//   "COLLECTION_METHOD_UNSPECIFIED" - Collection method unspecified.
	//   "UNSOLICITED" - The user was not responding to a specific solicitation
	// when they submitted the review.
	//   "POST_FULFILLMENT" - The user submitted the review in response to a
	// solicitation after fulfillment of the user's order.
	CollectionMethod string `json:"collectionMethod,omitempty"`
	// Cons: Optional. Contains the disadvantages based on the opinion of the
	// reviewer. Omit boilerplate text like "con:" unless it was written by the
	// reviewer.
	Cons []string `json:"cons,omitempty"`
	// Content: Required. The content of the review.
	Content string `json:"content,omitempty"`
	// Gtins: Optional. Contains GTINs (global trade item numbers) associated with
	// a product. Sub-types of GTINs (e.g. UPC, EAN, ISBN, JAN) are supported.
	Gtins []string `json:"gtins,omitempty"`
	// IsSpam: Optional. Indicates whether the review is marked as spam in the
	// publisher's system.
	IsSpam bool `json:"isSpam,omitempty"`
	// MaxRating: Optional. The maximum possible number for the rating. The value
	// of the max rating must be greater than the value of the min attribute.
	MaxRating int64 `json:"maxRating,omitempty,string"`
	// MinRating: Optional. Contains the ratings associated with the review. The
	// minimum possible number for the rating. This should be the worst possible
	// rating and should not be a value for no rating.
	MinRating int64 `json:"minRating,omitempty,string"`
	// Mpns: Optional. Contains MPNs (manufacturer part numbers) associated with a
	// product.
	Mpns []string `json:"mpns,omitempty"`
	// ProductLinks: Optional. The URI of the product. This URI can have the same
	// value as the `review_link` element, if the review URI and the product URI
	// are the same.
	ProductLinks []string `json:"productLinks,omitempty"`
	// ProductNames: Optional. Descriptive name of a product.
	ProductNames []string `json:"productNames,omitempty"`
	// Pros: Optional. Contains the advantages based on the opinion of the
	// reviewer. Omit boilerplate text like "pro:" unless it was written by the
	// reviewer.
	Pros []string `json:"pros,omitempty"`
	// PublisherFavicon: Optional. A link to the company favicon of the publisher.
	// The image dimensions should be favicon size: 16x16 pixels. The image format
	// should be GIF, JPG or PNG.
	PublisherFavicon string `json:"publisherFavicon,omitempty"`
	// PublisherName: Optional. The name of the publisher of the product reviews.
	// The information about the publisher, which may be a retailer, manufacturer,
	// reviews service company, or any entity that publishes product reviews.
	PublisherName string `json:"publisherName,omitempty"`
	// Rating: Optional. The reviewer's overall rating of the product.
	Rating float64 `json:"rating,omitempty"`
	// ReviewCountry: Optional. The country of the review defined by ISO 3166-1
	// Alpha-2 Country Code.
	ReviewCountry string `json:"reviewCountry,omitempty"`
	// ReviewLanguage: Optional. The language of the review defined by BCP-47
	// language code.
	ReviewLanguage string `json:"reviewLanguage,omitempty"`
	// ReviewLink: Optional. The URI of the review landing page.
	ReviewLink *ReviewLink `json:"reviewLink,omitempty"`
	// ReviewTime: Required. The timestamp indicating when the review was written.
	ReviewTime string `json:"reviewTime,omitempty"`
	// ReviewerId: Optional. The author of the product review. A permanent, unique
	// identifier for the author of the review in the publisher's system.
	ReviewerId string `json:"reviewerId,omitempty"`
	// ReviewerImageLinks: Optional. A URI to an image of the reviewed product
	// created by the review author. The URI does not have to end with an image
	// file extension.
	ReviewerImageLinks []string `json:"reviewerImageLinks,omitempty"`
	// ReviewerIsAnonymous: Optional. Set to true if the reviewer should remain
	// anonymous.
	ReviewerIsAnonymous bool `json:"reviewerIsAnonymous,omitempty"`
	// ReviewerUsername: Optional. The name of the reviewer of the product review.
	ReviewerUsername string `json:"reviewerUsername,omitempty"`
	// Skus: Optional. Contains SKUs (stock keeping units) associated with a
	// product. Often this matches the product Offer Id in the product feed.
	Skus []string `json:"skus,omitempty"`
	// SubclientName: Optional. The name of the subclient of the product reviews.
	// The subclient is an identifier of the product review source. It should be
	// equivalent to the directory provided in the file data source path.
	SubclientName string `json:"subclientName,omitempty"`
	// Title: Optional. The title of the review.
	Title string `json:"title,omitempty"`
	// TransactionId: Optional. A permanent, unique identifier for the transaction
	// associated with the review in the publisher's system. This ID can be used to
	// indicate that multiple reviews are associated with the same transaction.
	TransactionId string `json:"transactionId,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AggregatorName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AggregatorName") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductReviewAttributes) MarshalJSON() ([]byte, error) {
	type NoMethod ProductReviewAttributes
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *ProductReviewAttributes) UnmarshalJSON(data []byte) error {
	type NoMethod ProductReviewAttributes
	var s1 struct {
		Rating gensupport.JSONFloat64 `json:"rating"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Rating = float64(s1.Rating)
	return nil
}

// ProductReviewDestinationStatus: The destination status of the product review
// status.
type ProductReviewDestinationStatus struct {
	// ReportingContext: Output only. The name of the reporting context.
	//
	// Possible values:
	//   "REPORTING_CONTEXT_ENUM_UNSPECIFIED" - Not specified.
	//   "SHOPPING_ADS" - [Shopping
	// ads](https://support.google.com/merchants/answer/6149970).
	//   "DISCOVERY_ADS" - Deprecated: Use `DEMAND_GEN_ADS` instead. [Discovery and
	// Demand Gen ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS" - [Demand Gen
	// ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS_DISCOVER_SURFACE" - [Demand Gen ads on Discover
	// surface](https://support.google.com/merchants/answer/13389785).
	//   "VIDEO_ADS" - [Video
	// ads](https://support.google.com/google-ads/answer/6340491).
	//   "DISPLAY_ADS" - [Display
	// ads](https://support.google.com/merchants/answer/6069387).
	//   "LOCAL_INVENTORY_ADS" - [Local inventory
	// ads](https://support.google.com/merchants/answer/3271956).
	//   "VEHICLE_INVENTORY_ADS" - [Vehicle inventory
	// ads](https://support.google.com/merchants/answer/11544533).
	//   "FREE_LISTINGS" - [Free product
	// listings](https://support.google.com/merchants/answer/9199328).
	//   "FREE_LOCAL_LISTINGS" - [Free local product
	// listings](https://support.google.com/merchants/answer/9825611).
	//   "FREE_LOCAL_VEHICLE_LISTINGS" - [Free local vehicle
	// listings](https://support.google.com/merchants/answer/11544533).
	//   "YOUTUBE_SHOPPING" - [YouTube
	// Shopping](https://support.google.com/merchants/answer/13478370).
	//   "CLOUD_RETAIL" - [Cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "LOCAL_CLOUD_RETAIL" - [Local cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "PRODUCT_REVIEWS" - [Product
	// Reviews](https://support.google.com/merchants/answer/14620732).
	//   "MERCHANT_REVIEWS" - [Merchant
	// Reviews](https://developers.google.com/merchant-review-feeds).
	//   "YOUTUBE_CHECKOUT" - [YouTube Checkout](
	ReportingContext string `json:"reportingContext,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ReportingContext") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ReportingContext") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductReviewDestinationStatus) MarshalJSON() ([]byte, error) {
	type NoMethod ProductReviewDestinationStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductReviewItemLevelIssue: The ItemLevelIssue of the product review
// status.
type ProductReviewItemLevelIssue struct {
	// Attribute: Output only. The attribute's name, if the issue is caused by a
	// single attribute.
	Attribute string `json:"attribute,omitempty"`
	// Code: Output only. The error code of the issue.
	Code string `json:"code,omitempty"`
	// Description: Output only. A short issue description in English.
	Description string `json:"description,omitempty"`
	// Detail: Output only. A detailed issue description in English.
	Detail string `json:"detail,omitempty"`
	// Documentation: Output only. The URL of a web page to help with resolving
	// this issue.
	Documentation string `json:"documentation,omitempty"`
	// ReportingContext: Output only. The reporting context the issue applies to.
	//
	// Possible values:
	//   "REPORTING_CONTEXT_ENUM_UNSPECIFIED" - Not specified.
	//   "SHOPPING_ADS" - [Shopping
	// ads](https://support.google.com/merchants/answer/6149970).
	//   "DISCOVERY_ADS" - Deprecated: Use `DEMAND_GEN_ADS` instead. [Discovery and
	// Demand Gen ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS" - [Demand Gen
	// ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS_DISCOVER_SURFACE" - [Demand Gen ads on Discover
	// surface](https://support.google.com/merchants/answer/13389785).
	//   "VIDEO_ADS" - [Video
	// ads](https://support.google.com/google-ads/answer/6340491).
	//   "DISPLAY_ADS" - [Display
	// ads](https://support.google.com/merchants/answer/6069387).
	//   "LOCAL_INVENTORY_ADS" - [Local inventory
	// ads](https://support.google.com/merchants/answer/3271956).
	//   "VEHICLE_INVENTORY_ADS" - [Vehicle inventory
	// ads](https://support.google.com/merchants/answer/11544533).
	//   "FREE_LISTINGS" - [Free product
	// listings](https://support.google.com/merchants/answer/9199328).
	//   "FREE_LOCAL_LISTINGS" - [Free local product
	// listings](https://support.google.com/merchants/answer/9825611).
	//   "FREE_LOCAL_VEHICLE_LISTINGS" - [Free local vehicle
	// listings](https://support.google.com/merchants/answer/11544533).
	//   "YOUTUBE_SHOPPING" - [YouTube
	// Shopping](https://support.google.com/merchants/answer/13478370).
	//   "CLOUD_RETAIL" - [Cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "LOCAL_CLOUD_RETAIL" - [Local cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "PRODUCT_REVIEWS" - [Product
	// Reviews](https://support.google.com/merchants/answer/14620732).
	//   "MERCHANT_REVIEWS" - [Merchant
	// Reviews](https://developers.google.com/merchant-review-feeds).
	//   "YOUTUBE_CHECKOUT" - [YouTube Checkout](
	ReportingContext string `json:"reportingContext,omitempty"`
	// Resolution: Output only. Whether the issue can be resolved by the merchant.
	Resolution string `json:"resolution,omitempty"`
	// Severity: Output only. How this issue affects serving of the product review.
	//
	// Possible values:
	//   "SEVERITY_UNSPECIFIED" - Not specified.
	//   "NOT_IMPACTED" - This issue represents a warning and does not have a
	// direct affect on the product review.
	//   "DISAPPROVED" - Issue disapproves the product review.
	Severity string `json:"severity,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Attribute") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Attribute") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductReviewItemLevelIssue) MarshalJSON() ([]byte, error) {
	type NoMethod ProductReviewItemLevelIssue
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductReviewStatus: Product review status.
type ProductReviewStatus struct {
	// CreateTime: Output only. Date on which the item has been created, in ISO
	// 8601 (http://en.wikipedia.org/wiki/ISO_8601) format.
	CreateTime string `json:"createTime,omitempty"`
	// DestinationStatuses: Output only. The intended destinations for the product
	// review.
	DestinationStatuses []*ProductReviewDestinationStatus `json:"destinationStatuses,omitempty"`
	// ItemLevelIssues: Output only. A list of all issues associated with the
	// product review.
	ItemLevelIssues []*ProductReviewItemLevelIssue `json:"itemLevelIssues,omitempty"`
	// LastUpdateTime: Output only. Date on which the item has been last updated,
	// in ISO 8601 (http://en.wikipedia.org/wiki/ISO_8601) format.
	LastUpdateTime string `json:"lastUpdateTime,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CreateTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CreateTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductReviewStatus) MarshalJSON() ([]byte, error) {
	type NoMethod ProductReviewStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductStatusChangeMessage: The message that the merchant will receive to
// notify about product status change event
type ProductStatusChangeMessage struct {
	// Account: The target account that owns the entity that changed. Format :
	// `accounts/{merchant_id}`
	Account string `json:"account,omitempty"`
	// Attribute: The attribute in the resource that changed, in this case it will
	// be always `Status`.
	//
	// Possible values:
	//   "ATTRIBUTE_UNSPECIFIED" - Unspecified attribute
	//   "STATUS" - Status of the changed entity
	Attribute string `json:"attribute,omitempty"`
	// Changes: A message to describe the change that happened to the product
	Changes []*ProductChange `json:"changes,omitempty"`
	// ExpirationTime: The product expiration time.
	ExpirationTime string `json:"expirationTime,omitempty"`
	// ManagingAccount: The account that manages the merchant's account. can be the
	// same as merchant id if it is standalone account. Format :
	// `accounts/{service_provider_id}`
	ManagingAccount string `json:"managingAccount,omitempty"`
	// Resource: The product name. Format:
	// `{product.name=accounts/{account}/products/{product}}`
	Resource string `json:"resource,omitempty"`
	// ResourceId: The product id.
	ResourceId string `json:"resourceId,omitempty"`
	// ResourceType: The resource that changed, in this case it will always be
	// `Product`.
	//
	// Possible values:
	//   "RESOURCE_UNSPECIFIED" - Unspecified resource
	//   "PRODUCT" - Resource type : product
	ResourceType string `json:"resourceType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Account") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Account") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductStatusChangeMessage) MarshalJSON() ([]byte, error) {
	type NoMethod ProductStatusChangeMessage
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ReviewLink: The URI of the review landing page.
type ReviewLink struct {
	// Link: Optional. The URI of the review landing page. For example:
	// `http://www.example.com/review_5.html`.
	Link string `json:"link,omitempty"`
	// Type: Optional. Type of the review URI.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Type unspecified.
	//   "SINGLETON" - The review page contains only this single review.
	//   "GROUP" - The review page contains a group of reviews including this
	// review.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Link") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Link") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ReviewLink) MarshalJSON() ([]byte, error) {
	type NoMethod ReviewLink
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AccountsMerchantReviewsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes merchant review.
//
//   - name: The ID of the merchant review. Format:
//     accounts/{account}/merchantReviews/{merchantReview}.
func (r *AccountsMerchantReviewsService) Delete(name string) *AccountsMerchantReviewsDeleteCall {
	c := &AccountsMerchantReviewsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsMerchantReviewsDeleteCall) Fields(s ...googleapi.Field) *AccountsMerchantReviewsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsMerchantReviewsDeleteCall) Context(ctx context.Context) *AccountsMerchantReviewsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsMerchantReviewsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsMerchantReviewsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "reviews/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.merchantReviews.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsMerchantReviewsDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccountsMerchantReviewsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a merchant review.
//
//   - name: The ID of the merchant review. Format:
//     accounts/{account}/merchantReviews/{merchantReview}.
func (r *AccountsMerchantReviewsService) Get(name string) *AccountsMerchantReviewsGetCall {
	c := &AccountsMerchantReviewsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsMerchantReviewsGetCall) Fields(s ...googleapi.Field) *AccountsMerchantReviewsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsMerchantReviewsGetCall) IfNoneMatch(entityTag string) *AccountsMerchantReviewsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsMerchantReviewsGetCall) Context(ctx context.Context) *AccountsMerchantReviewsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsMerchantReviewsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsMerchantReviewsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "reviews/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.merchantReviews.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *MerchantReview.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsMerchantReviewsGetCall) Do(opts ...googleapi.CallOption) (*MerchantReview, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &MerchantReview{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccountsMerchantReviewsInsertCall struct {
	s              *Service
	parent         string
	merchantreview *MerchantReview
	urlParams_     gensupport.URLParams
	ctx_           context.Context
	header_        http.Header
}

// Insert: Inserts a review for your Merchant Center account. If the review
// already exists, then the review is replaced with the new instance.
//
//   - parent: The account where the merchant review will be inserted. Format:
//     accounts/{account}.
func (r *AccountsMerchantReviewsService) Insert(parent string, merchantreview *MerchantReview) *AccountsMerchantReviewsInsertCall {
	c := &AccountsMerchantReviewsInsertCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.merchantreview = merchantreview
	return c
}

// DataSource sets the optional parameter "dataSource": Required. The data
// source of the merchantreview
// (https://support.google.com/merchants/answer/7045996?sjid=5253581244217581976-EU)
// Format: `accounts/{account}/dataSources/{datasource}`.
func (c *AccountsMerchantReviewsInsertCall) DataSource(dataSource string) *AccountsMerchantReviewsInsertCall {
	c.urlParams_.Set("dataSource", dataSource)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsMerchantReviewsInsertCall) Fields(s ...googleapi.Field) *AccountsMerchantReviewsInsertCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsMerchantReviewsInsertCall) Context(ctx context.Context) *AccountsMerchantReviewsInsertCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsMerchantReviewsInsertCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsMerchantReviewsInsertCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.merchantreview)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "reviews/v1beta/{+parent}/merchantReviews:insert")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.merchantReviews.insert" call.
// Any non-2xx status code is an error. Response headers are in either
// *MerchantReview.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsMerchantReviewsInsertCall) Do(opts ...googleapi.CallOption) (*MerchantReview, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &MerchantReview{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccountsMerchantReviewsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists merchant reviews.
//
//   - parent: The account to list merchant reviews for. Format:
//     accounts/{account}.
func (r *AccountsMerchantReviewsService) List(parent string) *AccountsMerchantReviewsListCall {
	c := &AccountsMerchantReviewsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// merchant reviews to return. The service can return fewer than this value.
// The maximum value is 1000; values above 1000 are coerced to 1000. If
// unspecified, the maximum number of reviews is returned.
func (c *AccountsMerchantReviewsListCall) PageSize(pageSize int64) *AccountsMerchantReviewsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListMerchantReviews` call. Provide this to retrieve the
// subsequent page. When paginating, all other parameters provided to
// `ListMerchantReviews` must match the call that provided the page token.
func (c *AccountsMerchantReviewsListCall) PageToken(pageToken string) *AccountsMerchantReviewsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsMerchantReviewsListCall) Fields(s ...googleapi.Field) *AccountsMerchantReviewsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsMerchantReviewsListCall) IfNoneMatch(entityTag string) *AccountsMerchantReviewsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsMerchantReviewsListCall) Context(ctx context.Context) *AccountsMerchantReviewsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsMerchantReviewsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsMerchantReviewsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "reviews/v1beta/{+parent}/merchantReviews")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.merchantReviews.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListMerchantReviewsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsMerchantReviewsListCall) Do(opts ...googleapi.CallOption) (*ListMerchantReviewsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListMerchantReviewsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsMerchantReviewsListCall) Pages(ctx context.Context, f func(*ListMerchantReviewsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsProductReviewsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a product review.
//
//   - name: The ID of the Product review. Format:
//     accounts/{account}/productReviews/{productReview}.
func (r *AccountsProductReviewsService) Delete(name string) *AccountsProductReviewsDeleteCall {
	c := &AccountsProductReviewsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsProductReviewsDeleteCall) Fields(s ...googleapi.Field) *AccountsProductReviewsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsProductReviewsDeleteCall) Context(ctx context.Context) *AccountsProductReviewsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsProductReviewsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsProductReviewsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "reviews/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.productReviews.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsProductReviewsDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccountsProductReviewsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a product review.
//
//   - name: The ID of the merchant review. Format:
//     accounts/{account}/productReviews/{productReview}.
func (r *AccountsProductReviewsService) Get(name string) *AccountsProductReviewsGetCall {
	c := &AccountsProductReviewsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsProductReviewsGetCall) Fields(s ...googleapi.Field) *AccountsProductReviewsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsProductReviewsGetCall) IfNoneMatch(entityTag string) *AccountsProductReviewsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsProductReviewsGetCall) Context(ctx context.Context) *AccountsProductReviewsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsProductReviewsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsProductReviewsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "reviews/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.productReviews.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *ProductReview.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsProductReviewsGetCall) Do(opts ...googleapi.CallOption) (*ProductReview, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ProductReview{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccountsProductReviewsInsertCall struct {
	s             *Service
	parent        string
	productreview *ProductReview
	urlParams_    gensupport.URLParams
	ctx_          context.Context
	header_       http.Header
}

// Insert: Inserts a product review.
//
//   - parent: The account where the product review will be inserted. Format:
//     accounts/{account}.
func (r *AccountsProductReviewsService) Insert(parent string, productreview *ProductReview) *AccountsProductReviewsInsertCall {
	c := &AccountsProductReviewsInsertCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.productreview = productreview
	return c
}

// DataSource sets the optional parameter "dataSource": Required. Format:
// `accounts/{account}/dataSources/{datasource}`.
func (c *AccountsProductReviewsInsertCall) DataSource(dataSource string) *AccountsProductReviewsInsertCall {
	c.urlParams_.Set("dataSource", dataSource)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsProductReviewsInsertCall) Fields(s ...googleapi.Field) *AccountsProductReviewsInsertCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsProductReviewsInsertCall) Context(ctx context.Context) *AccountsProductReviewsInsertCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsProductReviewsInsertCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsProductReviewsInsertCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.productreview)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "reviews/v1beta/{+parent}/productReviews:insert")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.productReviews.insert" call.
// Any non-2xx status code is an error. Response headers are in either
// *ProductReview.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsProductReviewsInsertCall) Do(opts ...googleapi.CallOption) (*ProductReview, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ProductReview{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccountsProductReviewsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists product reviews.
//
//   - parent: The account to list product reviews for. Format:
//     accounts/{account}.
func (r *AccountsProductReviewsService) List(parent string) *AccountsProductReviewsListCall {
	c := &AccountsProductReviewsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// products to return. The service may return fewer than this value.
func (c *AccountsProductReviewsListCall) PageSize(pageSize int64) *AccountsProductReviewsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListProductReviews` call. Provide this to retrieve the
// subsequent page. When paginating, all other parameters provided to
// `ListProductReviews` must match the call that provided the page token.
func (c *AccountsProductReviewsListCall) PageToken(pageToken string) *AccountsProductReviewsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsProductReviewsListCall) Fields(s ...googleapi.Field) *AccountsProductReviewsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsProductReviewsListCall) IfNoneMatch(entityTag string) *AccountsProductReviewsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsProductReviewsListCall) Context(ctx context.Context) *AccountsProductReviewsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsProductReviewsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsProductReviewsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "reviews/v1beta/{+parent}/productReviews")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.productReviews.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListProductReviewsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsProductReviewsListCall) Do(opts ...googleapi.CallOption) (*ListProductReviewsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListProductReviewsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsProductReviewsListCall) Pages(ctx context.Context, f func(*ListProductReviewsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}
