// Copyright 2024 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package merchantapi provides access to the Merchant API.
//
// For product documentation, see: https://developers.devsite.corp.google.com/merchant/api
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/merchantapi/datasources_v1beta"
//	...
//	ctx := context.Background()
//	merchantapiService, err := merchantapi.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	merchantapiService, err := merchantapi.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	merchantapiService, err := merchantapi.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package merchantapi // import "google.golang.org/api/merchantapi/datasources_v1beta"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version

const apiId = "merchantapi:datasources_v1beta"
const apiName = "merchantapi"
const apiVersion = "datasources_v1beta"
const basePath = "https://merchantapi.googleapis.com/"
const basePathTemplate = "https://merchantapi.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://merchantapi.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// Manage your product listings and accounts for Google Shopping
	ContentScope = "https://www.googleapis.com/auth/content"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/content",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s, err := New(client)
	if err != nil {
		return nil, err
	}
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	s := &Service{client: client, BasePath: basePath}
	s.Accounts = NewAccountsService(s)
	return s, nil
}

type Service struct {
	client    *http.Client
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Accounts *AccountsService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewAccountsService(s *Service) *AccountsService {
	rs := &AccountsService{s: s}
	rs.DataSources = NewAccountsDataSourcesService(s)
	return rs
}

type AccountsService struct {
	s *Service

	DataSources *AccountsDataSourcesService
}

func NewAccountsDataSourcesService(s *Service) *AccountsDataSourcesService {
	rs := &AccountsDataSourcesService{s: s}
	rs.FileUploads = NewAccountsDataSourcesFileUploadsService(s)
	return rs
}

type AccountsDataSourcesService struct {
	s *Service

	FileUploads *AccountsDataSourcesFileUploadsService
}

func NewAccountsDataSourcesFileUploadsService(s *Service) *AccountsDataSourcesFileUploadsService {
	rs := &AccountsDataSourcesFileUploadsService{s: s}
	return rs
}

type AccountsDataSourcesFileUploadsService struct {
	s *Service
}

// DataSource: The data source
// (https://support.google.com/merchants/answer/7439058) for the Merchant
// Center account.
type DataSource struct {
	// DataSourceId: Output only. The data source id.
	DataSourceId int64 `json:"dataSourceId,omitempty,string"`
	// DisplayName: Required. The displayed data source name in the Merchant Center
	// UI.
	DisplayName string `json:"displayName,omitempty"`
	// FileInput: Optional. The field is used only when data is managed through a
	// file.
	FileInput *FileInput `json:"fileInput,omitempty"`
	// Input: Output only. Determines the type of input to the data source. Based
	// on the input some settings might not work. Only generic data sources can be
	// created through the API.
	//
	// Possible values:
	//   "INPUT_UNSPECIFIED" - Input unspecified.
	//   "API" - Represents data sources for which the data is primarily provided
	// through the API.
	//   "FILE" - Represents data sources for which the data is primarily provided
	// through file input. Data can still be provided through the API.
	//   "UI" - The data source for products added directly in Merchant Center.
	// This type of data source can not be created or updated through this API,
	// only by Merchant Center UI. This type of data source is read only.
	//   "AUTOFEED" - This is also known as [Automated
	// feeds](https://support.google.com/merchants/answer/12158480) used to
	// automatically build your product data. This type of data source can be
	// enabled or disabled through the Accounts bundle.
	Input string `json:"input,omitempty"`
	// LocalInventoryDataSource: Required. The local inventory
	// (https://support.google.com/merchants/answer/7023001) data source.
	LocalInventoryDataSource *LocalInventoryDataSource `json:"localInventoryDataSource,omitempty"`
	// MerchantReviewDataSource: Required. The merchant review
	// (https://support.google.com/merchants/answer/7045996) data source.
	MerchantReviewDataSource *MerchantReviewDataSource `json:"merchantReviewDataSource,omitempty"`
	// Name: Identifier. The name of the data source. Format:
	// `{datasource.name=accounts/{account}/dataSources/{datasource}}`
	Name string `json:"name,omitempty"`
	// PrimaryProductDataSource: Required. The primary data source
	// (https://support.google.com/merchants/answer/7439058) for local and online
	// products.
	PrimaryProductDataSource *PrimaryProductDataSource `json:"primaryProductDataSource,omitempty"`
	// ProductReviewDataSource: Required. The product review
	// (https://support.google.com/merchants/answer/7045996) data source.
	ProductReviewDataSource *ProductReviewDataSource `json:"productReviewDataSource,omitempty"`
	// PromotionDataSource: Required. The promotion
	// (https://support.google.com/merchants/answer/2906014) data source.
	PromotionDataSource *PromotionDataSource `json:"promotionDataSource,omitempty"`
	// RegionalInventoryDataSource: Required. The regional inventory
	// (https://support.google.com/merchants/answer/7439058) data source.
	RegionalInventoryDataSource *RegionalInventoryDataSource `json:"regionalInventoryDataSource,omitempty"`
	// SupplementalProductDataSource: Required. The supplemental data source
	// (https://support.google.com/merchants/answer/7439058) for local and online
	// products.
	SupplementalProductDataSource *SupplementalProductDataSource `json:"supplementalProductDataSource,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "DataSourceId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DataSourceId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DataSource) MarshalJSON() ([]byte, error) {
	type NoMethod DataSource
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DataSourceReference: Data source reference can be used to manage related
// data sources within the data source service.
type DataSourceReference struct {
	// PrimaryDataSourceName: Optional. The name of the primary data source.
	// Format: `accounts/{account}/dataSources/{datasource}`
	PrimaryDataSourceName string `json:"primaryDataSourceName,omitempty"`
	// Self: Self should be used to reference the primary data source itself.
	Self bool `json:"self,omitempty"`
	// SupplementalDataSourceName: Optional. The name of the supplemental data
	// source. Format: `accounts/{account}/dataSources/{datasource}`
	SupplementalDataSourceName string `json:"supplementalDataSourceName,omitempty"`
	// ForceSendFields is a list of field names (e.g. "PrimaryDataSourceName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "PrimaryDataSourceName") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DataSourceReference) MarshalJSON() ([]byte, error) {
	type NoMethod DataSourceReference
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DefaultRule: Default rule management of the data source.
type DefaultRule struct {
	// TakeFromDataSources: Required. The list of data sources linked in the
	// default rule (https://support.google.com/merchants/answer/7450276). This
	// list is ordered by the default rule priority of joining the data. It might
	// include none or multiple references to `self` and supplemental data sources.
	// The list must not be empty. To link the data source to the default rule, you
	// need to add a new reference to this list (in sequential order). To unlink
	// the data source from the default rule, you need to remove the given
	// reference from this list. Changing the order of this list will result in
	// changing the priority of data sources in the default rule. For example,
	// providing the following list: [`1001`, `self`] will take attribute values
	// from supplemental data source `1001`, and fallback to `self` if the
	// attribute is not set in `1001`.
	TakeFromDataSources []*DataSourceReference `json:"takeFromDataSources,omitempty"`
	// ForceSendFields is a list of field names (e.g. "TakeFromDataSources") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "TakeFromDataSources") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DefaultRule) MarshalJSON() ([]byte, error) {
	type NoMethod DefaultRule
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Empty: A generic empty message that you can re-use to avoid defining
// duplicated empty messages in your APIs. A typical example is to use it as
// the request or the response type of an API method. For instance: service Foo
// { rpc Bar(google.protobuf.Empty) returns (google.protobuf.Empty); }
type Empty struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// FetchDataSourceRequest: Request message for the FetchDataSource method.
type FetchDataSourceRequest struct {
}

// FetchSettings: Fetch details to deliver the data source.
type FetchSettings struct {
	// DayOfMonth: Optional. The day of the month when the data source file should
	// be fetched (1-31). This field can only be set for monthly frequency.
	DayOfMonth int64 `json:"dayOfMonth,omitempty"`
	// DayOfWeek: Optional. The day of the week when the data source file should be
	// fetched. This field can only be set for weekly frequency.
	//
	// Possible values:
	//   "DAY_OF_WEEK_UNSPECIFIED" - The day of the week is unspecified.
	//   "MONDAY" - Monday
	//   "TUESDAY" - Tuesday
	//   "WEDNESDAY" - Wednesday
	//   "THURSDAY" - Thursday
	//   "FRIDAY" - Friday
	//   "SATURDAY" - Saturday
	//   "SUNDAY" - Sunday
	DayOfWeek string `json:"dayOfWeek,omitempty"`
	// Enabled: Optional. Enables or pauses the fetch schedule.
	Enabled bool `json:"enabled,omitempty"`
	// FetchUri: Optional. The URL where the data source file can be fetched.
	// Google Merchant Center supports automatic scheduled uploads using the HTTP,
	// HTTPS or SFTP protocols, so the value will need to be a valid link using one
	// of those three protocols. Immutable for Google Sheets files.
	FetchUri string `json:"fetchUri,omitempty"`
	// Frequency: Required. The frequency describing fetch schedule.
	//
	// Possible values:
	//   "FREQUENCY_UNSPECIFIED" - Frequency unspecified.
	//   "FREQUENCY_DAILY" - The fetch happens every day.
	//   "FREQUENCY_WEEKLY" - The fetch happens every week.
	//   "FREQUENCY_MONTHLY" - The fetch happens every month.
	Frequency string `json:"frequency,omitempty"`
	// Password: Optional. An optional password for fetch_uri. Used for submitting
	// data sources through SFTP
	// (https://support.google.com/merchants/answer/13813117).
	Password string `json:"password,omitempty"`
	// TimeOfDay: Optional. The hour of the day when the data source file should be
	// fetched. Minutes and seconds are not supported and will be ignored.
	TimeOfDay *TimeOfDay `json:"timeOfDay,omitempty"`
	// TimeZone: Optional. Time zone (https://cldr.unicode.org) used for schedule.
	// UTC by default. For example, "America/Los_Angeles".
	TimeZone string `json:"timeZone,omitempty"`
	// Username: Optional. An optional user name for fetch_uri. Used for submitting
	// data sources through SFTP
	// (https://support.google.com/merchants/answer/13813117).
	Username string `json:"username,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DayOfMonth") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DayOfMonth") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FetchSettings) MarshalJSON() ([]byte, error) {
	type NoMethod FetchSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileInput: The data specific for file data sources. This field is empty for
// other data source inputs.
type FileInput struct {
	// FetchSettings: Optional. Fetch details to deliver the data source. It
	// contains settings for `FETCH` and `GOOGLE_SHEETS` file input types. The
	// required fields vary based on the frequency of fetching.
	FetchSettings *FetchSettings `json:"fetchSettings,omitempty"`
	// FileInputType: Output only. The type of file input.
	//
	// Possible values:
	//   "FILE_INPUT_TYPE_UNSPECIFIED" - File input type unspecified.
	//   "UPLOAD" - The file is uploaded through SFTP, Google Cloud Storage or
	// manually in the Merchant Center.
	//   "FETCH" - The file is fetched from the configured fetch_uri.
	//   "GOOGLE_SHEETS" - The file is fetched from Google Sheets specified in the
	// fetch_uri.
	FileInputType string `json:"fileInputType,omitempty"`
	// FileName: Optional. The file name of the data source. Required for `UPLOAD`
	// file input type.
	FileName string `json:"fileName,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FetchSettings") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FetchSettings") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileInput) MarshalJSON() ([]byte, error) {
	type NoMethod FileInput
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileUpload: The file upload of a specific data source, that is, the result
// of the retrieval of the data source at a certain timestamp computed
// asynchronously when the data source processing is finished. Only applicable
// to file data sources.
type FileUpload struct {
	// DataSourceId: Output only. The data source id.
	DataSourceId int64 `json:"dataSourceId,omitempty,string"`
	// Issues: Output only. The list of issues occurring in the data source.
	Issues []*Issue `json:"issues,omitempty"`
	// ItemsCreated: Output only. The number of items in the data source that were
	// created.
	ItemsCreated int64 `json:"itemsCreated,omitempty,string"`
	// ItemsTotal: Output only. The number of items in the data source that were
	// processed.
	ItemsTotal int64 `json:"itemsTotal,omitempty,string"`
	// ItemsUpdated: Output only. The number of items in the data source that were
	// updated.
	ItemsUpdated int64 `json:"itemsUpdated,omitempty,string"`
	// Name: Identifier. The name of the data source file upload. Format:
	// `{datasource.name=accounts/{account}/dataSources/{datasource}/fileUploads/{fi
	// leupload}}`
	Name string `json:"name,omitempty"`
	// ProcessingState: Output only. The processing state of the data source.
	//
	// Possible values:
	//   "PROCESSING_STATE_UNSPECIFIED" - Processing state unspecified.
	//   "FAILED" - The data source could not be processed or all the items had
	// errors.
	//   "IN_PROGRESS" - The data source is being processed.
	//   "SUCCEEDED" - The data source was processed successfully, though some
	// items might have had errors.
	ProcessingState string `json:"processingState,omitempty"`
	// UploadTime: Output only. The date at which the file of the data source was
	// uploaded.
	UploadTime string `json:"uploadTime,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "DataSourceId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DataSourceId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileUpload) MarshalJSON() ([]byte, error) {
	type NoMethod FileUpload
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Issue: An error occurring in the data source, like "invalid price".
type Issue struct {
	// Code: Output only. The code of the error, for example,
	// "validation/invalid_value". Returns "?" if the code is unknown.
	Code string `json:"code,omitempty"`
	// Count: Output only. The number of occurrences of the error in the file
	// upload.
	Count int64 `json:"count,omitempty,string"`
	// Description: Output only. The error description, for example, "Your data
	// source contains items which have too many attributes, or are too big. These
	// items will be dropped".
	Description string `json:"description,omitempty"`
	// DocumentationUri: Output only. Link to the documentation explaining the
	// issue in more details, if available.
	DocumentationUri string `json:"documentationUri,omitempty"`
	// Severity: Output only. The severity of the issue.
	//
	// Possible values:
	//   "SEVERITY_UNSPECIFIED" - Severity unspecified.
	//   "WARNING" - The issue is the warning.
	//   "ERROR" - The issue is an error.
	Severity string `json:"severity,omitempty"`
	// Title: Output only. The title of the issue, for example, "Item too big".
	Title string `json:"title,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Code") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Code") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Issue) MarshalJSON() ([]byte, error) {
	type NoMethod Issue
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListDataSourcesResponse: Response message for the ListDataSources method.
type ListDataSourcesResponse struct {
	// DataSources: The data sources from the specified account.
	DataSources []*DataSource `json:"dataSources,omitempty"`
	// NextPageToken: A token, which can be sent as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "DataSources") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DataSources") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListDataSourcesResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListDataSourcesResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LocalInventoryDataSource: The local inventory data source.
type LocalInventoryDataSource struct {
	// ContentLanguage: Required. Immutable. The two-letter ISO 639-1 language of
	// the items to which the local inventory is provided.
	ContentLanguage string `json:"contentLanguage,omitempty"`
	// FeedLabel: Required. Immutable. The feed label of the offers to which the
	// local inventory is provided. Must be less than or equal to 20 uppercase
	// letters (A-Z), numbers (0-9), and dashes (-). See also migration to feed
	// labels
	// (https://developers.google.com/shopping-content/guides/products/feed-labels).
	FeedLabel string `json:"feedLabel,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ContentLanguage") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ContentLanguage") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LocalInventoryDataSource) MarshalJSON() ([]byte, error) {
	type NoMethod LocalInventoryDataSource
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MerchantReviewDataSource: The merchant review data source.
type MerchantReviewDataSource struct {
}

// PrimaryProductDataSource: The primary data source for local and online
// products.
type PrimaryProductDataSource struct {
	// Channel: Required. Immutable. Specifies the type of data source channel.
	//
	// Possible values:
	//   "CHANNEL_UNSPECIFIED" - Not specified.
	//   "ONLINE_PRODUCTS" - Online product.
	//   "LOCAL_PRODUCTS" - Local product.
	//   "PRODUCTS" - Unified data source for both local and online products. Note:
	// Products management through the API is not possible for this channel.
	Channel string `json:"channel,omitempty"`
	// ContentLanguage: Optional. Immutable. The two-letter ISO 639-1 language of
	// the items in the data source. `feedLabel` and `contentLanguage` must be
	// either both set or unset. The fields can only be unset for data sources
	// without file input. If set, the data source will only accept products
	// matching this combination. If unset, the data source will accept products
	// without that restriction.
	ContentLanguage string `json:"contentLanguage,omitempty"`
	// Countries: Optional. The countries where the items may be displayed.
	// Represented as a CLDR territory code
	// (https://github.com/unicode-org/cldr/blob/latest/common/main/en.xml).
	Countries []string `json:"countries,omitempty"`
	// DefaultRule: Optional. Default rule management of the data source. If set,
	// the linked data sources will be replaced.
	DefaultRule *DefaultRule `json:"defaultRule,omitempty"`
	// FeedLabel: Optional. Immutable. The feed label that is specified on the data
	// source level. Must be less than or equal to 20 uppercase letters (A-Z),
	// numbers (0-9), and dashes (-). See also migration to feed labels
	// (https://developers.google.com/shopping-content/guides/products/feed-labels).
	// `feedLabel` and `contentLanguage` must be either both set or unset for data
	// sources with product content type. They must be set for data sources with a
	// file input. If set, the data source will only accept products matching this
	// combination. If unset, the data source will accept products without that
	// restriction.
	FeedLabel string `json:"feedLabel,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Channel") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Channel") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PrimaryProductDataSource) MarshalJSON() ([]byte, error) {
	type NoMethod PrimaryProductDataSource
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductChange: The change that happened to the product including old value,
// new value, country code as the region code and reporting context.
type ProductChange struct {
	// NewValue: The new value of the changed resource or attribute.
	NewValue string `json:"newValue,omitempty"`
	// OldValue: The old value of the changed resource or attribute.
	OldValue string `json:"oldValue,omitempty"`
	// RegionCode: Countries that have the change (if applicable)
	RegionCode string `json:"regionCode,omitempty"`
	// ReportingContext: Reporting contexts that have the change (if applicable)
	//
	// Possible values:
	//   "REPORTING_CONTEXT_ENUM_UNSPECIFIED" - Not specified.
	//   "SHOPPING_ADS" - [Shopping
	// ads](https://support.google.com/merchants/answer/6149970).
	//   "DISCOVERY_ADS" - Deprecated: Use `DEMAND_GEN_ADS` instead. [Discovery and
	// Demand Gen ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS" - [Demand Gen
	// ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS_DISCOVER_SURFACE" - [Demand Gen ads on Discover
	// surface](https://support.google.com/merchants/answer/13389785).
	//   "VIDEO_ADS" - [Video
	// ads](https://support.google.com/google-ads/answer/6340491).
	//   "DISPLAY_ADS" - [Display
	// ads](https://support.google.com/merchants/answer/6069387).
	//   "LOCAL_INVENTORY_ADS" - [Local inventory
	// ads](https://support.google.com/merchants/answer/3271956).
	//   "VEHICLE_INVENTORY_ADS" - [Vehicle inventory
	// ads](https://support.google.com/merchants/answer/11544533).
	//   "FREE_LISTINGS" - [Free product
	// listings](https://support.google.com/merchants/answer/9199328).
	//   "FREE_LOCAL_LISTINGS" - [Free local product
	// listings](https://support.google.com/merchants/answer/9825611).
	//   "FREE_LOCAL_VEHICLE_LISTINGS" - [Free local vehicle
	// listings](https://support.google.com/merchants/answer/11544533).
	//   "YOUTUBE_SHOPPING" - [YouTube
	// Shopping](https://support.google.com/merchants/answer/13478370).
	//   "CLOUD_RETAIL" - [Cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "LOCAL_CLOUD_RETAIL" - [Local cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "PRODUCT_REVIEWS" - [Product
	// Reviews](https://support.google.com/merchants/answer/14620732).
	//   "MERCHANT_REVIEWS" - [Merchant
	// Reviews](https://developers.google.com/merchant-review-feeds).
	//   "YOUTUBE_CHECKOUT" - [YouTube Checkout](
	ReportingContext string `json:"reportingContext,omitempty"`
	// ForceSendFields is a list of field names (e.g. "NewValue") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NewValue") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductChange) MarshalJSON() ([]byte, error) {
	type NoMethod ProductChange
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductReviewDataSource: The product review data source.
type ProductReviewDataSource struct {
}

// ProductStatusChangeMessage: The message that the merchant will receive to
// notify about product status change event
type ProductStatusChangeMessage struct {
	// Account: The target account that owns the entity that changed. Format :
	// `accounts/{merchant_id}`
	Account string `json:"account,omitempty"`
	// Attribute: The attribute in the resource that changed, in this case it will
	// be always `Status`.
	//
	// Possible values:
	//   "ATTRIBUTE_UNSPECIFIED" - Unspecified attribute
	//   "STATUS" - Status of the changed entity
	Attribute string `json:"attribute,omitempty"`
	// Changes: A message to describe the change that happened to the product
	Changes []*ProductChange `json:"changes,omitempty"`
	// ExpirationTime: The product expiration time.
	ExpirationTime string `json:"expirationTime,omitempty"`
	// ManagingAccount: The account that manages the merchant's account. can be the
	// same as merchant id if it is standalone account. Format :
	// `accounts/{service_provider_id}`
	ManagingAccount string `json:"managingAccount,omitempty"`
	// Resource: The product name. Format:
	// `{product.name=accounts/{account}/products/{product}}`
	Resource string `json:"resource,omitempty"`
	// ResourceId: The product id.
	ResourceId string `json:"resourceId,omitempty"`
	// ResourceType: The resource that changed, in this case it will always be
	// `Product`.
	//
	// Possible values:
	//   "RESOURCE_UNSPECIFIED" - Unspecified resource
	//   "PRODUCT" - Resource type : product
	ResourceType string `json:"resourceType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Account") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Account") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductStatusChangeMessage) MarshalJSON() ([]byte, error) {
	type NoMethod ProductStatusChangeMessage
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PromotionDataSource: The promotion data source.
type PromotionDataSource struct {
	// ContentLanguage: Required. Immutable. The two-letter ISO 639-1 language of
	// the items in the data source.
	ContentLanguage string `json:"contentLanguage,omitempty"`
	// TargetCountry: Required. Immutable. The target country used as part of the
	// unique identifier. Represented as a CLDR territory code
	// (https://github.com/unicode-org/cldr/blob/latest/common/main/en.xml).
	// Promotions are only available in selected countries
	// (https://support.google.com/merchants/answer/4588460).
	TargetCountry string `json:"targetCountry,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ContentLanguage") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ContentLanguage") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PromotionDataSource) MarshalJSON() ([]byte, error) {
	type NoMethod PromotionDataSource
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RegionalInventoryDataSource: The regional inventory data source.
type RegionalInventoryDataSource struct {
	// ContentLanguage: Required. Immutable. The two-letter ISO 639-1 language of
	// the items to which the regional inventory is provided.
	ContentLanguage string `json:"contentLanguage,omitempty"`
	// FeedLabel: Required. Immutable. The feed label of the offers to which the
	// regional inventory is provided. Must be less than or equal to 20 uppercase
	// letters (A-Z), numbers (0-9), and dashes (-). See also migration to feed
	// labels
	// (https://developers.google.com/shopping-content/guides/products/feed-labels).
	FeedLabel string `json:"feedLabel,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ContentLanguage") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ContentLanguage") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RegionalInventoryDataSource) MarshalJSON() ([]byte, error) {
	type NoMethod RegionalInventoryDataSource
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SupplementalProductDataSource: The supplemental data source for local and
// online products. After creation, you should make sure to link the
// supplemental product data source into one or more primary product data
// sources.
type SupplementalProductDataSource struct {
	// ContentLanguage: Optional. Immutable. The two-letter ISO 639-1 language of
	// the items in the data source. `feedLabel` and `contentLanguage` must be
	// either both set or unset. The fields can only be unset for data sources
	// without file input. If set, the data source will only accept products
	// matching this combination. If unset, the data source will accept produts
	// without that restriction.
	ContentLanguage string `json:"contentLanguage,omitempty"`
	// FeedLabel: Optional. Immutable. The feed label that is specified on the data
	// source level. Must be less than or equal to 20 uppercase letters (A-Z),
	// numbers (0-9), and dashes (-). See also migration to feed labels
	// (https://developers.google.com/shopping-content/guides/products/feed-labels).
	// `feedLabel` and `contentLanguage` must be either both set or unset for data
	// sources with product content type. They must be set for data sources with a
	// file input. The fields must be unset for data sources without file input. If
	// set, the data source will only accept products matching this combination. If
	// unset, the data source will accept produts without that restriction.
	FeedLabel string `json:"feedLabel,omitempty"`
	// ReferencingPrimaryDataSources: Output only. The (unordered and deduplicated)
	// list of all primary data sources linked to this data source in either
	// default or custom rules. Supplemental data source cannot be deleted before
	// all links are removed.
	ReferencingPrimaryDataSources []*DataSourceReference `json:"referencingPrimaryDataSources,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ContentLanguage") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ContentLanguage") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SupplementalProductDataSource) MarshalJSON() ([]byte, error) {
	type NoMethod SupplementalProductDataSource
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TimeOfDay: Represents a time of day. The date and time zone are either not
// significant or are specified elsewhere. An API may choose to allow leap
// seconds. Related types are google.type.Date and `google.protobuf.Timestamp`.
type TimeOfDay struct {
	// Hours: Hours of a day in 24 hour format. Must be greater than or equal to 0
	// and typically must be less than or equal to 23. An API may choose to allow
	// the value "24:00:00" for scenarios like business closing time.
	Hours int64 `json:"hours,omitempty"`
	// Minutes: Minutes of an hour. Must be greater than or equal to 0 and less
	// than or equal to 59.
	Minutes int64 `json:"minutes,omitempty"`
	// Nanos: Fractions of seconds, in nanoseconds. Must be greater than or equal
	// to 0 and less than or equal to 999,999,999.
	Nanos int64 `json:"nanos,omitempty"`
	// Seconds: Seconds of a minute. Must be greater than or equal to 0 and
	// typically must be less than or equal to 59. An API may allow the value 60 if
	// it allows leap-seconds.
	Seconds int64 `json:"seconds,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Hours") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Hours") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TimeOfDay) MarshalJSON() ([]byte, error) {
	type NoMethod TimeOfDay
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AccountsDataSourcesCreateCall struct {
	s          *Service
	parent     string
	datasource *DataSource
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates the new data source configuration for the given account.
//
//   - parent: The account where this data source will be created. Format:
//     `accounts/{account}`.
func (r *AccountsDataSourcesService) Create(parent string, datasource *DataSource) *AccountsDataSourcesCreateCall {
	c := &AccountsDataSourcesCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.datasource = datasource
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsDataSourcesCreateCall) Fields(s ...googleapi.Field) *AccountsDataSourcesCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsDataSourcesCreateCall) Context(ctx context.Context) *AccountsDataSourcesCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsDataSourcesCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsDataSourcesCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.datasource)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "datasources/v1beta/{+parent}/dataSources")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.dataSources.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *DataSource.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsDataSourcesCreateCall) Do(opts ...googleapi.CallOption) (*DataSource, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &DataSource{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccountsDataSourcesDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a data source from your Merchant Center account.
//
//   - name: The name of the data source to delete. Format:
//     `accounts/{account}/dataSources/{datasource}`.
func (r *AccountsDataSourcesService) Delete(name string) *AccountsDataSourcesDeleteCall {
	c := &AccountsDataSourcesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsDataSourcesDeleteCall) Fields(s ...googleapi.Field) *AccountsDataSourcesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsDataSourcesDeleteCall) Context(ctx context.Context) *AccountsDataSourcesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsDataSourcesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsDataSourcesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "datasources/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.dataSources.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsDataSourcesDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccountsDataSourcesFetchCall struct {
	s                      *Service
	name                   string
	fetchdatasourcerequest *FetchDataSourceRequest
	urlParams_             gensupport.URLParams
	ctx_                   context.Context
	header_                http.Header
}

// Fetch: Performs the data fetch immediately (even outside fetch schedule) on
// a data source from your Merchant Center Account. If you need to call this
// method more than once per day, you should use the Products service to update
// your product data instead. This method only works on data sources with a
// file input set.
//
//   - name: The name of the data source resource to fetch. Format:
//     `accounts/{account}/dataSources/{datasource}`.
func (r *AccountsDataSourcesService) Fetch(name string, fetchdatasourcerequest *FetchDataSourceRequest) *AccountsDataSourcesFetchCall {
	c := &AccountsDataSourcesFetchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.fetchdatasourcerequest = fetchdatasourcerequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsDataSourcesFetchCall) Fields(s ...googleapi.Field) *AccountsDataSourcesFetchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsDataSourcesFetchCall) Context(ctx context.Context) *AccountsDataSourcesFetchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsDataSourcesFetchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsDataSourcesFetchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.fetchdatasourcerequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "datasources/v1beta/{+name}:fetch")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.dataSources.fetch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsDataSourcesFetchCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccountsDataSourcesGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves the data source configuration for the given account.
//
//   - name: The name of the data source to retrieve. Format:
//     `accounts/{account}/dataSources/{datasource}`.
func (r *AccountsDataSourcesService) Get(name string) *AccountsDataSourcesGetCall {
	c := &AccountsDataSourcesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsDataSourcesGetCall) Fields(s ...googleapi.Field) *AccountsDataSourcesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsDataSourcesGetCall) IfNoneMatch(entityTag string) *AccountsDataSourcesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsDataSourcesGetCall) Context(ctx context.Context) *AccountsDataSourcesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsDataSourcesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsDataSourcesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "datasources/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.dataSources.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *DataSource.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsDataSourcesGetCall) Do(opts ...googleapi.CallOption) (*DataSource, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &DataSource{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccountsDataSourcesListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the configurations for data sources for the given account.
//
// - parent: The account to list data sources for. Format: `accounts/{account}`.
func (r *AccountsDataSourcesService) List(parent string) *AccountsDataSourcesListCall {
	c := &AccountsDataSourcesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of data
// sources to return. The service may return fewer than this value. The maximum
// value is 1000; values above 1000 will be coerced to 1000. If unspecified,
// the maximum number of data sources will be returned.
func (c *AccountsDataSourcesListCall) PageSize(pageSize int64) *AccountsDataSourcesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListDataSources` call. Provide this to retrieve the
// subsequent page. When paginating, all other parameters provided to
// `ListDataSources` must match the call that provided the page token.
func (c *AccountsDataSourcesListCall) PageToken(pageToken string) *AccountsDataSourcesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsDataSourcesListCall) Fields(s ...googleapi.Field) *AccountsDataSourcesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsDataSourcesListCall) IfNoneMatch(entityTag string) *AccountsDataSourcesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsDataSourcesListCall) Context(ctx context.Context) *AccountsDataSourcesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsDataSourcesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsDataSourcesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "datasources/v1beta/{+parent}/dataSources")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.dataSources.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListDataSourcesResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsDataSourcesListCall) Do(opts ...googleapi.CallOption) (*ListDataSourcesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListDataSourcesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsDataSourcesListCall) Pages(ctx context.Context, f func(*ListDataSourcesResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsDataSourcesPatchCall struct {
	s          *Service
	name       string
	datasource *DataSource
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Patch: Updates the existing data source configuration. The fields that are
// set in the update mask but not provided in the resource will be deleted.
//
//   - name: Identifier. The name of the data source. Format:
//     `{datasource.name=accounts/{account}/dataSources/{datasource}}`.
func (r *AccountsDataSourcesService) Patch(name string, datasource *DataSource) *AccountsDataSourcesPatchCall {
	c := &AccountsDataSourcesPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.datasource = datasource
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. The list of
// data source fields to be updated. Fields specified in the update mask
// without a value specified in the body will be deleted from the data source.
// Providing special "*" value for full data source replacement is not
// supported.
func (c *AccountsDataSourcesPatchCall) UpdateMask(updateMask string) *AccountsDataSourcesPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsDataSourcesPatchCall) Fields(s ...googleapi.Field) *AccountsDataSourcesPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsDataSourcesPatchCall) Context(ctx context.Context) *AccountsDataSourcesPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsDataSourcesPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsDataSourcesPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.datasource)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "datasources/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.dataSources.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *DataSource.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsDataSourcesPatchCall) Do(opts ...googleapi.CallOption) (*DataSource, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &DataSource{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccountsDataSourcesFileUploadsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets the latest data source file upload. Only the `latest` alias is
// accepted for a file upload.
//
//   - name: The name of the data source file upload to retrieve. Format:
//     `accounts/{account}/dataSources/{datasource}/fileUploads/latest`.
func (r *AccountsDataSourcesFileUploadsService) Get(name string) *AccountsDataSourcesFileUploadsGetCall {
	c := &AccountsDataSourcesFileUploadsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsDataSourcesFileUploadsGetCall) Fields(s ...googleapi.Field) *AccountsDataSourcesFileUploadsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsDataSourcesFileUploadsGetCall) IfNoneMatch(entityTag string) *AccountsDataSourcesFileUploadsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsDataSourcesFileUploadsGetCall) Context(ctx context.Context) *AccountsDataSourcesFileUploadsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsDataSourcesFileUploadsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsDataSourcesFileUploadsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "datasources/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.dataSources.fileUploads.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *FileUpload.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsDataSourcesFileUploadsGetCall) Do(opts ...googleapi.CallOption) (*FileUpload, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &FileUpload{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}
