// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief  C++ API module wrapper
//!\author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <Python.h>
#include "api_datetime.h"
#include "api_file.h"
#include "api_reader.h"
#include "api_writer.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: prototypes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void api_file_tp_dealloc (api_file_o *);
static PyObject *api_file_tp_new (PyTypeObject *, PyObject *, PyObject *);
static PyObject *api_file_get_is_valid (api_file_o *);
static PyObject *api_file_get_url (api_file_o *);
static PyObject *api_file_get_is_file (api_file_o *);
static PyObject *api_file_get_is_folder (api_file_o *);
static PyObject *api_file_get_exists (api_file_o *);
static PyObject *api_file_get_size (api_file_o *);
static PyObject *api_file_get_user_id (api_file_o *);
static PyObject *api_file_get_user_name (api_file_o *);
static PyObject *api_file_get_group_id (api_file_o *);
static PyObject *api_file_get_group_name (api_file_o *);
static PyObject *api_file_get_permissions (api_file_o *);
static PyObject *api_file_get_last_access_time (api_file_o *);
static PyObject *api_file_get_last_modification_time (api_file_o *);
static PyObject *api_file_get_last_metadata_time (api_file_o *);
static PyObject *api_file_f_new_reader (api_file_o *, PyObject *);
static PyObject *api_file_f_new_writer (api_file_o *, PyObject *);

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: getters and setters structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyGetSetDef api_file_getsetters[] =
{
  {
    (char *) "is_valid",
    (getter) api_file_get_is_valid,
    (setter) 0,
    (char *) "check if file is valid", NULL
  },
  {
    (char *) "url",
    (getter) api_file_get_url,
    (setter) 0,
    (char *) "URL as string", NULL
  },
  {
    (char *) "is_file",
    (getter) api_file_get_is_file,
    (setter) 0,
    (char *) "check if entry is a file", NULL
  },
  {
    (char *) "is_folder",
    (getter) api_file_get_is_folder,
    (setter) 0,
    (char *) "check if entry is a folder", NULL
  },
  {
    (char *) "exists",
    (getter) api_file_get_exists,
    (setter) 0,
    (char *) "check if file exists", NULL
  },
  {
    (char *) "size",
    (getter) api_file_get_size,
    (setter) 0,
    (char *) "file size in bytes", NULL
  },
  {
    (char *) "user_id",
    (getter) api_file_get_user_id,
    (setter) 0,
    (char *) "owner's user ID", NULL
  },
  {
    (char *) "user_name",
    (getter) api_file_get_user_name,
    (setter) 0,
    (char *) "owner's user name", NULL
  },
  {
    (char *) "group_id",
    (getter) api_file_get_group_id,
    (setter) 0,
    (char *) "group ID", NULL
  },
  {
    (char *) "group_name",
    (getter) api_file_get_group_name,
    (setter) 0,
    (char *) "group name", NULL
  },
  {
    (char *) "permissions",
    (getter) api_file_get_permissions,
    (setter) 0,
    (char *) "access permission mask", NULL
  },
  {
    (char *) "last_access_time",
    (getter) api_file_get_last_access_time,
    (setter) 0,
    (char *) "last access time", NULL
  },
  {
    (char *) "last_modification_time",
    (getter) api_file_get_last_modification_time,
    (setter) 0,
    (char *) "last data modification time", NULL
  },
  {
    (char *) "last_metadata_time",
    (getter) api_file_get_last_metadata_time,
    (setter) 0,
    (char *) "last metadata modification time", NULL
  },
  {NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: methods structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef api_file_methods[] =
{
  {
    (char *) "new_reader",
    (PyCFunction) api_file_f_new_reader,
    METH_VARARGS,
    "create new reader"
  },
  {
    (char *) "new_writer",
    (PyCFunction) api_file_f_new_writer,
    METH_VARARGS,
    "create new writer"
  },
  {NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: type structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyTypeObject api_file_t =
{
  PyObject_HEAD_INIT (0)
  0,                                       		// ob_size
  "api.file",                               		// tp_name
  sizeof (api_file_o),                      		// tp_basicsize
  0,                                       		// tp_itemsize
  (destructor) api_file_tp_dealloc,         		// tp_dealloc
  0,                                       		// tp_print
  0,                                       		// tp_getattr
  0,                                       		// tp_setattr
  0,                                       		// tp_compare
  0,                                       		// tp_repr
  0,                                       		// tp_as_number
  0,                                       		// tp_as_sequence
  0,                                       		// tp_as_mapping
  0,                                       		// tp_hash
  0,                                       		// tp_call
  0,                                       		// tp_str
  0,                                       		// tp_getattro
  0,                                       		// tp_setattro
  0,                                       		// tp_as_buffer
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,		// tp_flags
  "file class with support for URL's",     		// tp_doc
  0,                                       		// tp_traverse
  0,                                       		// tp_clear
  0,                                       		// tp_richcompare
  0,                                       		// tp_weaklistoffset
  0,                                       		// tp_iter
  0,                                       		// tp_iternext
  api_file_methods,                         		// tp_methods
  0,                                       		// tp_members
  api_file_getsetters,                      		// tp_getset
  0,                                       		// tp_base
  0,                                       		// tp_dict
  0,                                       		// tp_descr_get
  0,                                       		// tp_descr_set
  0,                                       		// tp_dictoffset
  0,                                       		// tp_init
  0,                                       		// tp_alloc
  api_file_tp_new                           		// tp_new
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: tp_new
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_file_tp_new (PyTypeObject *type, PyObject *args, PyObject *kwds)
{
  const char *arg_uri = nullptr;

  if (!PyArg_ParseTuple (args, "|s", &arg_uri))
    return NULL;

  api_file_o *self = (api_file_o *) type->tp_alloc (type, 0);
  if (self != NULL)
    {
      try
        {
          if (arg_uri)
            self->obj = new mobius::io::file (arg_uri);
          else
            self->obj = new mobius::io::file ();
        }
      catch (const std::runtime_error& e)
        {
          Py_DECREF (self);
          PyErr_SetString (PyExc_IOError, e.what ());
          self = NULL;
        }
    }

  return (PyObject *) self;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: tp_dealloc
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
api_file_tp_dealloc (api_file_o *self)
{
  delete self->obj;
  self->ob_type->tp_free ((PyObject*) self);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: is_valid getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_file_get_is_valid (api_file_o *self)
{
  return PyBool_FromLong (self->obj->operator bool ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: url getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_file_get_url (api_file_o *self)
{
  return PyString_FromString (self->obj->get_url ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: is_file getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_file_get_is_file (api_file_o *self)
{
  return PyBool_FromLong (self->obj->is_file ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: is_folder getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_file_get_is_folder (api_file_o *self)
{
  return PyBool_FromLong (self->obj->is_folder ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: exists getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_file_get_exists (api_file_o *self)
{
  return PyBool_FromLong (self->obj->exists ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: size getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_file_get_size (api_file_o *self)
{
  return PyInt_FromLong (self->obj->get_size ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: user_id getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_file_get_user_id (api_file_o *self)
{
  return PyInt_FromLong (self->obj->get_user_id ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: user_name getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_file_get_user_name (api_file_o *self)
{
  return PyString_FromString (self->obj->get_user_name ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: group_id getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_file_get_group_id (api_file_o *self)
{
  return PyInt_FromLong (self->obj->get_group_id ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: group_name getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_file_get_group_name (api_file_o *self)
{
  return PyString_FromString (self->obj->get_group_name ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: permissions getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_file_get_permissions (api_file_o *self)
{
  return PyInt_FromLong (self->obj->get_permissions ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: last_access_time getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_file_get_last_access_time (api_file_o *self)
{
  return PyDateTime_from_datetime (self->obj->get_last_access_time ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: last_modification_time getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_file_get_last_modification_time (api_file_o *self)
{
 return PyDateTime_from_datetime (self->obj->get_last_modification_time ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: last_metadata_time getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_file_get_last_metadata_time (api_file_o *self)
{
 return PyDateTime_from_datetime (self->obj->get_last_metadata_time ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: new_reader
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_file_f_new_reader (api_file_o *self, PyObject *args)
{
  api_reader_o *ret = nullptr;

  try
    {
      ret = (api_reader_o *) _PyObject_New (&api_reader_t);
      
      if (ret != nullptr)
        ret->obj = new mobius::io::reader (self->obj->new_reader ());
    }
  catch (const std::runtime_error& e)
    {
      PyErr_SetString (PyExc_IOError, e.what ());
    }
    
  return (PyObject *) ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.file: new_writer
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_file_f_new_writer (api_file_o *self, PyObject *args)
{
  // parse input args
  int arg_overwrite = 1;

  if (!PyArg_ParseTuple (args, "|i", &arg_overwrite))
    return nullptr;

  // build and return value
  api_writer_o *ret = nullptr;

  try
    {
      ret = (api_writer_o *) _PyObject_New (&api_writer_t);

      if (ret != nullptr)
        ret->obj = new mobius::io::writer (self->obj->new_writer (arg_overwrite));
    }
  catch (const std::runtime_error& e)
    {
      PyErr_SetString (PyExc_IOError, e.what ());
    }
    
  return (PyObject *) ret;
}
