// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief  C++ API module wrapper
//!\author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <Python.h>
#include <stdexcept>
#include "api_device.h"
#include "api_reader.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.device: prototypes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void api_device_tp_dealloc (api_device_o *);
static PyObject *api_device_get_is_initialized (api_device_o *);
static PyObject *api_device_get_type (api_device_o *);
static PyObject *api_device_get_subsystem (api_device_o *);
static PyObject *api_device_get_node (api_device_o *);
static PyObject *api_device_f_get_property (api_device_o *, PyObject *);
static PyObject *api_device_f_get_sysattr (api_device_o *, PyObject *);
static PyObject *api_device_f_new_reader (api_device_o *, PyObject *);

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.device: getters and setters structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyGetSetDef api_device_getsetters[] =
{
  {
    (char *) "is_initialized",
    (getter) api_device_get_is_initialized,
    (setter) 0,
    (char *) "check if device is initialized", NULL
  },
  {
    (char *) "type",
    (getter) api_device_get_type,
    (setter) 0,
    (char *) "device type", NULL
  },
  {
    (char *) "subsystem",
    (getter) api_device_get_subsystem,
    (setter) 0,
    (char *) "device subsystem", NULL
  },
  {
    (char *) "node",
    (getter) api_device_get_node,
    (setter) 0,
    (char *) "device node", NULL
  },
  {NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.device: methods structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef api_device_methods[] =
{
  {
    (char *) "get_property",
    (PyCFunction) api_device_f_get_property,
    METH_VARARGS,
    "get property value"
  },
  {
    (char *) "get_sysattr",
    (PyCFunction) api_device_f_get_sysattr,
    METH_VARARGS,
    "get system attribute"
  },
  {
    (char *) "new_reader",
    (PyCFunction) api_device_f_new_reader,
    METH_VARARGS,
    "create new reader"
  },
  {NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.device: type structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyTypeObject api_device_t =
{
  PyObject_HEAD_INIT (0)
  0,                                       		// ob_size
  "api.device",                            		// tp_name
  sizeof (api_device_o),                   		// tp_basicsize
  0,                                       		// tp_itemsize
  (destructor) api_device_tp_dealloc,      		// tp_dealloc
  0,                                       		// tp_print
  0,                                       		// tp_getattr
  0,                                       		// tp_setattr
  0,                                       		// tp_compare
  0,                                       		// tp_repr
  0,                                       		// tp_as_number
  0,                                       		// tp_as_sequence
  0,                                       		// tp_as_mapping
  0,                                       		// tp_hash
  0,                                       		// tp_call
  0,                                       		// tp_str
  0,                                       		// tp_getattro
  0,                                       		// tp_setattro
  0,                                       		// tp_as_buffer
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,		// tp_flags
  "system device class",                   		// tp_doc
  0,                                       		// tp_traverse
  0,                                       		// tp_clear
  0,                                       		// tp_richcompare
  0,                                       		// tp_weaklistoffset
  0,                                       		// tp_iter
  0,                                       		// tp_iternext
  api_device_methods,                      		// tp_methods
  0,                                       		// tp_members
  api_device_getsetters,                   		// tp_getset
  0,                                       		// tp_base
  0,                                       		// tp_dict
  0,                                       		// tp_descr_get
  0,                                       		// tp_descr_set
  0,                                       		// tp_dictoffset
  0,                                       		// tp_init
  0,                                       		// tp_alloc
  0                                        		// tp_new
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.device: tp_dealloc
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
api_device_tp_dealloc (api_device_o *self)
{
  delete self->obj;
  self->ob_type->tp_free ((PyObject*) self);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.device: is_initialized getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_device_get_is_initialized (api_device_o *self)
{
  return PyBool_FromLong (self->obj->is_initialized ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.device: type getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_device_get_type (api_device_o *self)
{
  return PyString_FromString (self->obj->get_type ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.device: subsystem getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_device_get_subsystem (api_device_o *self)
{
  return PyString_FromString (self->obj->get_subsystem ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.device: node getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_device_get_node (api_device_o *self)
{
  return PyString_FromString (self->obj->get_node ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.device: get_property
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_device_f_get_property (api_device_o *self, PyObject *args)
{
  const char * arg_name;

  if (!PyArg_ParseTuple (args, "s", &arg_name))
    return nullptr;

  // build and return value
  return PyString_FromString (self->obj->get_property (arg_name).c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.device: get_sysattr
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_device_f_get_sysattr (api_device_o *self, PyObject *args)
{
  const char * arg_name;

  if (!PyArg_ParseTuple (args, "s", &arg_name))
    return nullptr;

  // build and return value
  return PyString_FromString (self->obj->get_sysattr (arg_name).c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.device: new_reader
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_device_f_new_reader (api_device_o *self, PyObject *args)
{
  try
    {
      api_reader_o *ret = (api_reader_o *) _PyObject_New (&api_reader_t);
      if (ret)
        ret->obj = new mobius::io::reader (self->obj->new_reader ());

      return (PyObject *) ret;
    }
  catch (const std::runtime_error& e)
    {
      PyErr_SetString (PyExc_IOError, e.what ());
      return nullptr;
    }
}
