/*
 * Copyright 2008 Sony Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the names of the copyright holders nor the names of their
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "e_cell.h"
#include "e_cell_err.h"

#include <string.h>

#define GET_CELL_CIPHER_CTX(evp_cipher_ctx) \
	((cell_cipher_ctx_t *)((evp_cipher_ctx)->cipher_data))

static void spe_cipher_request_init(spe_cipher_request_t *cmd, EVP_CIPHER_CTX *ctx)
	{
	cell_cipher_ctx_t *cell_cipher = GET_CELL_CIPHER_CTX(ctx);

	cmd->request = 0;
	cmd->nid = ctx->cipher->nid;
	cmd->encrypt = ctx->encrypt;
	cmd->key_size = ctx->key_len;
	cmd->iv_size = ctx->cipher->iv_len;
	cmd->block_size = ctx->cipher->block_size;
	cmd->flags = ctx->cipher->flags;
	cmd->size = 0;

	cell_cipher->in_buffer_ptr %= (cell_cipher->queue_depth + 2);
	cmd->in_ea =
		(uintptr_t)cell_cipher->in_buffer +
		cell_cipher->in_buffer_ptr * cell_cipher->unit_size;
	cell_cipher->in_buffer_ptr++;

	cell_cipher->out_buffer_ptr %= (cell_cipher->queue_depth + 2);
	cmd->out_ea =
		(uintptr_t)cell_cipher->out_buffer +
		cell_cipher->out_buffer_ptr * cell_cipher->unit_size;
	cell_cipher->out_buffer_ptr++;
	}

static int cell_cipher_cleanup(EVP_CIPHER_CTX *ctx)
	{
	cell_cipher_ctx_t *cell_cipher = GET_CELL_CIPHER_CTX(ctx);

	DPRINTF("\n");

	if (cell_cipher->spe)
		{
		/* send 'QUIT' command */
		spe_cipher_request_t cmd;
		spe_cipher_request_init(&cmd, ctx);
		cmd.request = SPE_CIPHER_COMMAND_QUIT;
		cell_queue_push(cell_cipher->q_in, &cmd);

		/* wait for exiting */
		spe_task_wait(cell_cipher->spe);
		}

	if (cell_cipher->q_in)
		{
		cell_queue_destroy(cell_cipher->q_in);
		cell_cipher->q_in = 0;
		}
	if (cell_cipher->q_out)
		{
		cell_queue_destroy(cell_cipher->q_out);
		cell_cipher->q_out = 0;
		}
	if (cell_cipher->in_buffer)
		{
		OPENSSL_cleanse(cell_cipher->in_buffer,
			cell_cipher->unit_size * (cell_cipher->queue_depth + 2));
		aligned_free(cell_cipher->in_buffer);
		cell_cipher->in_buffer = NULL;
		}
	if (cell_cipher->out_buffer)
		{
		OPENSSL_cleanse(cell_cipher->out_buffer,
			cell_cipher->unit_size * (cell_cipher->queue_depth + 2));
		aligned_free(cell_cipher->out_buffer);
		cell_cipher->out_buffer = NULL;
		}
	if (cell_cipher->spe)
		{
		spe_task_destroy(cell_cipher->spe);
		cell_cipher->spe = NULL;
		}

	return 1;
	}

static int cell_cipher_init(EVP_CIPHER_CTX *ctx, const unsigned char *key,
	const unsigned char *iv, int enc,
	spe_program_handle_t *prog)
	{
	cell_cipher_ctx_t *cell_cipher = GET_CELL_CIPHER_CTX(ctx);
	spe_cipher_request_t cmd;

	DPRINTF("\n");

	/* clear */
	memset(cell_cipher, 0, sizeof(*cell_cipher));

	/* copy global parameters */
	CRYPTO_r_lock(CRYPTO_LOCK_ENGINE);
	cell_cipher->queue_depth = cell_queue_depth;
	/* calculate efficient size of unit */
	cell_cipher->unit_size =
		FLOOR(cell_unit_size, ctx->cipher->block_size * SHARED_DATA_ALIGN);
	if (!cell_cipher->unit_size)
		{
		cell_cipher->unit_size = ctx->cipher->block_size * SHARED_DATA_ALIGN;
		}
	CRYPTO_r_unlock(CRYPTO_LOCK_ENGINE);

	/* create task object */
	cell_cipher->spe = spe_task_create(prog);
	if (!cell_cipher->spe)
		{
		cell_cipher_cleanup(ctx);
		return 0;
		}

	/* allocate buffers */
	cell_cipher->in_buffer =
		aligned_malloc(SHARED_DATA_ALIGN,
			cell_cipher->unit_size * (cell_cipher->queue_depth + 2));
	if (!cell_cipher->in_buffer)
		{
		cell_cipher_cleanup(ctx);
		CELLerr(CELL_F_CELL_CIPHER_INIT, CELL_R_MEMORY_ALLOCATION_FAILED);
		return 0;
		}
	cell_cipher->out_buffer =
		aligned_malloc(SHARED_DATA_ALIGN,
			cell_cipher->unit_size * (cell_cipher->queue_depth + 2));
	if (!cell_cipher->out_buffer)
		{
		cell_cipher_cleanup(ctx);
		CELLerr(CELL_F_CELL_CIPHER_INIT, CELL_R_MEMORY_ALLOCATION_FAILED);
		return 0;
		}

	/* create queues */
	cell_cipher->q_in =
		cell_queue_create(cell_cipher->spe,
			sizeof(spe_cipher_request_t), cell_cipher->queue_depth, 0);
	if (!cell_cipher->q_in)
		{
		cell_cipher_cleanup(ctx);
		return 0;
		}

	cell_cipher->q_out =
		cell_queue_create(cell_cipher->spe,
			sizeof(spe_cipher_result_t), cell_cipher->queue_depth, 1);
	if (!cell_cipher->q_out)
		{
		cell_cipher_cleanup(ctx);
		return 0;
		}

	/* run SPE program */
	if (!spe_task_run(cell_cipher->spe, cell_cipher->q_in, cell_cipher->q_out))
		{
		cell_cipher_cleanup(ctx);
		return 0;
		}

	/* send 'init' command */
	spe_cipher_request_init(&cmd, ctx);
	cmd.request = SPE_CIPHER_COMMAND_INIT;
	cmd.size = cmd.iv_size + cmd.key_size;
	ASSERT(cmd.size < cell_cipher->unit_size);
	memcpy((void*)(uintptr_t)cmd.in_ea, key, cmd.key_size);
	memcpy((void*)(uintptr_t)cmd.in_ea + cmd.key_size, iv ? iv : ctx->oiv, cmd.iv_size);
	cell_queue_push(cell_cipher->q_in, &cmd);

	return 1;
	}

static int cell_cipher_do(EVP_CIPHER_CTX *ctx, unsigned char *out,
	const unsigned char *in, unsigned int inl)
	{
	cell_cipher_ctx_t *cell_cipher = GET_CELL_CIPHER_CTX(ctx);
	spe_cipher_result_t result;
	spe_cipher_request_t cmd;

	DPRINTF("%u bytes\n", inl);

	if (IS_ALIGNED(in, SHARED_DATA_ALIGN) &&
		IS_ALIGNED(out, SHARED_DATA_ALIGN) &&
		IS_ALIGNED(inl, SHARED_DATA_ALIGN))
		{
		/* avoid copying data if alined */
		DPRINTF("directly\n");

		/* send request */
		spe_cipher_request_init(&cmd, ctx);
		cmd.request = SPE_CIPHER_COMMAND_DO;
		cmd.size = inl;
		cmd.in_ea = (uintptr_t)in;
		cmd.out_ea = (uintptr_t)out;
		cell_queue_push(cell_cipher->q_in, &cmd);

		/* receive result */
		cell_queue_pop(cell_cipher->q_out, &result);
		/* do nothing */
		}
	else
		{
		/* copy data if unalinged */
		unsigned char *out_req = out;
		unsigned int req_count = 0;
		int resend_cmd = 0;
		while (inl > 0)
			{
			int first = 1;
			/* send request as many as possible (at least one) */
			while (inl > 0)
				{
				if (!resend_cmd)
					{
					unsigned int chunk =
						(inl <= cell_cipher->unit_size) ?
						inl : cell_cipher->unit_size;

					spe_cipher_request_init(&cmd, ctx);
					cmd.request = SPE_CIPHER_COMMAND_DO;
					cmd.size = chunk;
					/* pass input area directly if aligned */
					if (IS_ALIGNED(in, SHARED_DATA_ALIGN) &&
						IS_ALIGNED(chunk, SHARED_DATA_ALIGN))
						cmd.in_ea = (uintptr_t)in;
					else
						memcpy((void*)(uintptr_t)cmd.in_ea, in, chunk);
					/* put result directly on the output area if aligned */
					if (IS_ALIGNED(out_req, SHARED_DATA_ALIGN) &&
						IS_ALIGNED(chunk, SHARED_DATA_ALIGN))
						cmd.out_ea = (uintptr_t)out_req;
					}

				DPRINTF("push: %u chunk: %u\n", req_count, cmd.size);

				if (first)
					cell_queue_push(cell_cipher->q_in, &cmd);
				else if (!cell_queue_try_push(cell_cipher->q_in, &cmd))
					{
					/* queue is full. retry in the next iteration */
					resend_cmd = 1;
					break;
					}

				first = 0;
				resend_cmd = 0;

				inl -= cmd.size;
				in += cmd.size;
				out_req += cmd.size;
				req_count++;
				}

			/* receive result as many as possible (at least one) */
			first = 1;
			while (req_count > 0)
				{
				DPRINTF("pop: %u\n", req_count);
				if (first)
					cell_queue_pop(cell_cipher->q_out, &result);
				else if (!cell_queue_try_pop(cell_cipher->q_out, &result))
					break; /* queue is empty */

				if ((uintptr_t)out != result.out_ea)
					memcpy(out, (void*)(uintptr_t)result.out_ea, result.size);

				first = 0;

				out += result.size;
				req_count--;
				}
			}

		/* receive the rest of result */
		while (req_count > 0)
			{
			DPRINTF("pop: %u\n", req_count);
			cell_queue_pop(cell_cipher->q_out, &result);

			if ((uintptr_t)out != result.out_ea)
				memcpy(out, (void*)(uintptr_t)result.out_ea, result.size);

			out += result.size;
			req_count--;
			}
		}

	return 1;
	}

static int cell_cipher_ctrl(EVP_CIPHER_CTX *ctx, int type, int arg, void *ptr)
	{
	CELLerr(CELL_F_CELL_CIPHER_CTRL,CELL_R_NOT_IMPLEMENTED);

	return 0;
	}

/*** cipher algorithm specific ***/

/* Helper macros */
#define CELL_CIPHER_MODE_IMPLEMENT(name, block_size, key_size, iv_size,	\
	flags, init_proc, do_proc, cleanup_proc,			\
	set_asn1_proc, get_asn1_proc, ctrl_proc)			\
	static const EVP_CIPHER cell_ ## name =				\
		{							\
		NID_ ## name, /* NID */					\
		block_size, key_size, iv_size,				\
		flags,							\
		init_proc, do_proc, cleanup_proc,			\
		sizeof(cell_cipher_ctx_t),				\
		set_asn1_proc, get_asn1_proc, ctrl_proc,		\
		NULL, /* App data */					\
		}

#define CELL_CIPHER_IMPLEMENT(name, block_size, key_size, iv_size,	\
	flags, init_proc, do_proc, cleanup_proc,			\
	set_asn1_proc, get_asn1_proc, ctrl_proc)			\
	CELL_CIPHER_MODE_IMPLEMENT(name ## _cbc, block_size, key_size, iv_size, \
		flags | EVP_CIPH_CBC_MODE,				\
		init_proc, do_proc, cleanup_proc,			\
		set_asn1_proc, get_asn1_proc, ctrl_proc);		\
	CELL_CIPHER_MODE_IMPLEMENT(name ## _ecb, block_size, key_size, iv_size, \
		flags | EVP_CIPH_ECB_MODE,				\
		init_proc, do_proc, cleanup_proc,			\
		set_asn1_proc, get_asn1_proc, ctrl_proc);		\
	CELL_CIPHER_MODE_IMPLEMENT(name ## _ofb128, 1, key_size, iv_size, \
		flags | EVP_CIPH_OFB_MODE,				\
		init_proc, do_proc, cleanup_proc,			\
		set_asn1_proc, get_asn1_proc, ctrl_proc);		\
	CELL_CIPHER_MODE_IMPLEMENT(name ## _cfb128, 1, key_size, iv_size, \
		flags | EVP_CIPH_CFB_MODE,				\
		init_proc, do_proc, cleanup_proc,			\
		set_asn1_proc, get_asn1_proc, ctrl_proc);		\
	CELL_CIPHER_MODE_IMPLEMENT(name ## _cfb1, 1, key_size, iv_size,	\
		flags | EVP_CIPH_CFB_MODE,				\
		init_proc, do_proc, cleanup_proc,			\
		set_asn1_proc, get_asn1_proc, ctrl_proc);		\
	CELL_CIPHER_MODE_IMPLEMENT(name ## _cfb8, 1, key_size, iv_size,	\
		flags | EVP_CIPH_CFB_MODE,				\
		init_proc, do_proc, cleanup_proc,			\
		set_asn1_proc, get_asn1_proc, ctrl_proc)

#define CELL_CIPHER_OBJECT(name) (& cell_ ## name)

#define CELL_CIPHER_OBJECTS(name)			\
	CELL_CIPHER_OBJECT(name ## _cbc),		\
		CELL_CIPHER_OBJECT(name ## _ecb),	\
		CELL_CIPHER_OBJECT(name ## _ofb128),	\
		CELL_CIPHER_OBJECT(name ## _cfb128),	\
		CELL_CIPHER_OBJECT(name ## _cfb1),	\
		CELL_CIPHER_OBJECT(name ## _cfb8)


/* AES */
#ifndef OPENSSL_NO_AES
#include <openssl/aes.h>

extern spe_program_handle_t spe_aes_elf;

static int cell_aes_init(EVP_CIPHER_CTX *ctx, const unsigned char *key,
	const unsigned char *iv, int enc)
	{
	DPRINTF("\n");

	return cell_cipher_init(ctx, key, iv, enc, &spe_aes_elf);
	}

CELL_CIPHER_IMPLEMENT(aes_128,
	AES_BLOCK_SIZE,
	16, /* key size */
	16, /* IV size */
	0, /* flags */
	cell_aes_init,
	cell_cipher_do,
	cell_cipher_cleanup,
	EVP_CIPHER_set_asn1_iv,
	EVP_CIPHER_get_asn1_iv,
	cell_cipher_ctrl);

CELL_CIPHER_IMPLEMENT(aes_192,
	AES_BLOCK_SIZE,
	24, /* key size */
	16, /* IV size */
	0, /* flags */
	cell_aes_init,
	cell_cipher_do,
	cell_cipher_cleanup,
	EVP_CIPHER_set_asn1_iv,
	EVP_CIPHER_get_asn1_iv,
	cell_cipher_ctrl);

CELL_CIPHER_IMPLEMENT(aes_256,
	AES_BLOCK_SIZE,
	32, /* key size */
	16, /* IV size */
	0, /* flags */
	cell_aes_init,
	cell_cipher_do,
	cell_cipher_cleanup,
	EVP_CIPHER_set_asn1_iv,
	EVP_CIPHER_get_asn1_iv,
	cell_cipher_ctrl);
#endif /* !OPENSSL_NO_AES */

/*** API function ***/

#define NUM_CIPHERS (sizeof(cell_ciphers_table) / sizeof(cell_ciphers_table[0]))

static const EVP_CIPHER *cell_ciphers_table[] =
	{
#ifndef OPENSSL_NO_AES
	CELL_CIPHER_OBJECTS(aes_128),
	CELL_CIPHER_OBJECTS(aes_192),
	CELL_CIPHER_OBJECTS(aes_256),
#endif /* !OPENSSL_NO_AES */
	};

static int cell_ciphers_nids[NUM_CIPHERS];

static void cell_ciphers_global_init(void)
	{
	int i;
	static int init_p = 0;

	CRYPTO_w_lock(CRYPTO_LOCK_ENGINE);

	if (!init_p)
		{
		for (i = 0; i < NUM_CIPHERS; i++)
			{
			cell_ciphers_nids[i] = cell_ciphers_table[i]->nid;
			}
		init_p = 1;
		}

	CRYPTO_w_unlock(CRYPTO_LOCK_ENGINE);
	}

int cell_ciphers(ENGINE *e, const EVP_CIPHER **cipher, const int **nids, int nid)
	{
	DPRINTF("EVP_CIPHER: %p\n", cipher);
	DPRINTF("NIDS: %p\n", nids);
	DPRINTF("NID: %d (%s)\n", nid, OBJ_nid2sn(nid));

	if (nid == 0)
		{
		cell_ciphers_global_init();

		*nids = cell_ciphers_nids;
		return NUM_CIPHERS;
		}
	else
		{
		int i;
		for (i = 0; i < NUM_CIPHERS; i++)
			{
			if (nid == cell_ciphers_table[i]->nid)
				{
				*cipher = cell_ciphers_table[i];
				return 1;
				}
			}
		CELLerr(CELL_F_CELL_CIPHERS,CELL_R_CIPHER_NOT_IMPLEMENTED);
		return 0;
		}
	}
