/*
 * Copyright 2008 Sony Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the names of the copyright holders nor the names of their
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef __powerpc__
#ifndef OPENSSL_NO_HW
#ifndef OPENSSL_NO_HW_CELL

#include <stdio.h>
#include <stdint.h>
#include <string.h>

#include <openssl/engine.h>

#include "e_cell.h"
#include "e_cell_err.h"

/* Global variables */
int cell_queue_depth = DEFAULT_QUEUE_DEPTH;
int cell_unit_size = DEFAULT_UNIT_SIZE;
int cell_task_priority = -1;

static int cell_init(ENGINE *e);
static int cell_finish(ENGINE *e);
static int cell_ctrl(ENGINE *e, int cmd, long i, void *p, void (*f)(void));
static int cell_destroy(ENGINE *e);

/* The definitions for control commands specific to this engine */
#define CELL_CMD_QUEUE_DEPTH		(ENGINE_CMD_BASE + 0)
#define CELL_CMD_UNIT_SIZE		(ENGINE_CMD_BASE + 1)
#define CELL_CMD_TASK_PRIORITY		(ENGINE_CMD_BASE + 2)
static const ENGINE_CMD_DEFN cell_cmd_defns[] =
	{
		{
		CELL_CMD_QUEUE_DEPTH,
		"QUEUE_DEPTH",
		"Specifies the depth of queues to pass data between PPE and SPE",
		ENGINE_CMD_FLAG_NUMERIC,
		},
		{
		CELL_CMD_UNIT_SIZE,
		"UNIT_SIZE",
		"Specifies the size of unit to pass data between PPE and SPE",
		ENGINE_CMD_FLAG_NUMERIC,
		},
		{
		CELL_CMD_TASK_PRIORITY,
		"TASK_PRIORITY",
		"Specifies the priority of tasks",
		ENGINE_CMD_FLAG_NUMERIC,
		},
		{
		0, NULL, NULL, 0,
		}
	};

/* Constants used when creating the engine */
static const char *engine_cell_id = "cell";
static const char *engine_cell_name = "Cell Broadband Engine processor support";

/* This internal function is used by engine_cell() and possibly by the
 * dynamic engine support too */
static int bind_cell(ENGINE *e)
	{
	if(!ENGINE_set_id(e, engine_cell_id) ||
		!ENGINE_set_name(e, engine_cell_name) ||
		!ENGINE_set_ciphers(e, cell_ciphers) ||
		!ENGINE_set_digests(e, cell_digests) ||
		!ENGINE_set_init_function(e, cell_init) ||
		!ENGINE_set_destroy_function(e, cell_destroy) ||
		!ENGINE_set_finish_function(e, cell_finish) ||
		!ENGINE_set_ctrl_function(e, cell_ctrl) ||
		!ENGINE_set_cmd_defns(e, cell_cmd_defns))
		return 0;

	/* Ensure the Cell/B.E. error handling is set up */
	ERR_load_CELL_strings();

	return 1;
	}

/* Constructors */
#ifndef OPENSSL_NO_DYNAMIC_ENGINE
static int bind_helper(ENGINE *e, const char *id)
	{
	DPRINTF("\n");

	if(id && (strcmp(id, engine_cell_id) != 0))
		return 0;
	if(!bind_cell(e))
		return 0;
	return 1;
	}
IMPLEMENT_DYNAMIC_CHECK_FN()
IMPLEMENT_DYNAMIC_BIND_FN(bind_helper)
#else
static ENGINE *engine_cell(void)
	{
	ENGINE *ret = ENGINE_new();
	if(!ret)
		return NULL;
	if(!bind_cell(ret))
		{
		ENGINE_free(ret);
		return NULL;
		}
	return ret;
	}

void ENGINE_load_cell(void)
	{
	ENGINE *toadd;

	DPRINTF("\n");

	toadd = engine_cell();
	if(!toadd)
		return;

	ENGINE_add(toadd);
	ENGINE_free(toadd);
	ERR_clear_error();
	}
#endif

/* Destructor */
static int cell_destroy(ENGINE *e)
	{
	DPRINTF("\n");

	ERR_unload_CELL_strings();

	return 1;
	}

/* Initialization */
static int cell_init(ENGINE *e)
	{
	DPRINTF("\n");

	return 1;
	}

/* Finalization */
static int cell_finish(ENGINE *e)
	{
	DPRINTF("\n");

	return 1;
	}

static int cell_ctrl(ENGINE *e, int cmd, long i, void *p, void (*f)(void))
	{
	DPRINTF("\n");

	switch(cmd)
		{
		case CELL_CMD_QUEUE_DEPTH:
			if (i < MIN_QUEUE_DEPTH || i > MAX_QUEUE_DEPTH)
				{
				CELLerr(CELL_F_CELL_CTRL,CELL_R_PARAMETER_OUT_OF_RANGE);
				return 0;
				}
			CRYPTO_w_lock(CRYPTO_LOCK_ENGINE);
			cell_queue_depth = i;
			CRYPTO_w_unlock(CRYPTO_LOCK_ENGINE);
			return 1;
		case CELL_CMD_UNIT_SIZE:
			if (i < MIN_UNIT_SIZE || i > MAX_UNIT_SIZE)
				{
				CELLerr(CELL_F_CELL_CTRL,CELL_R_PARAMETER_OUT_OF_RANGE);
				return 0;
				}
			CRYPTO_w_lock(CRYPTO_LOCK_ENGINE);
			cell_unit_size = i;
			CRYPTO_w_unlock(CRYPTO_LOCK_ENGINE);
			return 1;
		case CELL_CMD_TASK_PRIORITY:
			CRYPTO_w_lock(CRYPTO_LOCK_ENGINE);
			cell_task_priority = i;
			CRYPTO_w_unlock(CRYPTO_LOCK_ENGINE);
			return 1;
		default:
			CELLerr(CELL_F_CELL_CTRL,CELL_R_CTRL_COMMAND_NOT_IMPLEMENTED);
			return 0;
		}
	}

#endif /* !OPENSSL_NO_HW_CELL */
#endif /* !OPENSSL_NO_HW */
#endif /* __powerpc__ */
