/*
 * Decompiled with CFR 0.152.
 */
package com.unboundid.util.parallel;

import com.unboundid.util.Debug;
import com.unboundid.util.InternalUseOnly;
import com.unboundid.util.LDAPSDKThreadFactory;
import com.unboundid.util.NotNull;
import com.unboundid.util.Nullable;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.Validator;
import com.unboundid.util.parallel.Processor;
import com.unboundid.util.parallel.Result;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Semaphore;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

@InternalUseOnly
@ThreadSafety(level=ThreadSafetyLevel.COMPLETELY_THREADSAFE)
public final class ParallelProcessor<I, O> {
    @NotNull
    private final Processor<I, O> processor;
    @NotNull
    private final List<Thread> workers;
    private final int minPerThread;
    @NotNull
    private final Semaphore workerSemaphore = new Semaphore(0);
    @NotNull
    private final AtomicReference<List<? extends I>> inputItems = new AtomicReference();
    @NotNull
    private final AtomicReference<List<Result<I, O>>> outputItems = new AtomicReference();
    @NotNull
    private final AtomicInteger nextToProcess = new AtomicInteger();
    @Nullable
    private volatile CountDownLatch processingCompleteSignal;
    @NotNull
    private final AtomicBoolean shutdown = new AtomicBoolean();

    public ParallelProcessor(@NotNull Processor<I, O> processor, int totalThreads, int minPerThread) {
        this(processor, null, totalThreads, minPerThread);
    }

    public ParallelProcessor(@NotNull Processor<I, O> processor, @Nullable ThreadFactory threadFactory, int totalThreads, int minPerThread) {
        Validator.ensureNotNull(processor);
        Validator.ensureTrue(totalThreads >= 1, "ParallelProcessor.totalThreads must be at least 1.");
        Validator.ensureTrue(totalThreads <= 1000, "ParallelProcessor.totalThreads must not be greater than 1000.");
        Validator.ensureTrue(minPerThread >= 1, "ParallelProcessor.minPerThread must be at least 1.");
        this.processor = processor;
        this.minPerThread = minPerThread;
        ThreadFactory tf = threadFactory == null ? new LDAPSDKThreadFactory("ParallelProcessor-Worker", true) : threadFactory;
        int numExtraThreads = totalThreads - 1;
        ArrayList<Thread> workerList = new ArrayList<Thread>(numExtraThreads);
        for (int i = 0; i < numExtraThreads; ++i) {
            Thread worker = tf.newThread(new Worker());
            workerList.add(worker);
            worker.start();
        }
        this.workers = workerList;
    }

    @NotNull
    public synchronized ArrayList<Result<I, O>> processAll(@NotNull List<? extends I> items) throws InterruptedException, IllegalStateException {
        if (this.shutdown.get()) {
            throw new IllegalStateException("cannot call processAll() after shutdown()");
        }
        Validator.ensureNotNull(items);
        int extraThreads = Math.min(items.size() / this.minPerThread - 1, this.workers.size());
        if (extraThreads <= 0) {
            ArrayList<Result<I, O>> output = new ArrayList<Result<I, O>>(items.size());
            for (I item : items) {
                output.add(this.process(item));
            }
            return output;
        }
        this.processingCompleteSignal = new CountDownLatch(extraThreads);
        this.inputItems.set(items);
        ArrayList<Result<I, O>> output = new ArrayList<Result<I, O>>(items.size());
        for (int i = 0; i < items.size(); ++i) {
            output.add(null);
        }
        this.outputItems.set(output);
        this.nextToProcess.set(0);
        this.workerSemaphore.release(extraThreads);
        this.processInParallel();
        this.processingCompleteSignal.await();
        return output;
    }

    public synchronized void shutdown() throws InterruptedException {
        if (this.shutdown.getAndSet(true)) {
            return;
        }
        this.workerSemaphore.release(this.workers.size());
        for (Thread worker : this.workers) {
            worker.join();
        }
    }

    private void processInParallel() {
        try {
            int next;
            List<I> items = this.inputItems.get();
            List<Result<I, O>> outputs = this.outputItems.get();
            int size = items.size();
            while ((next = this.nextToProcess.getAndIncrement()) < size) {
                I input = items.get(next);
                outputs.set(next, this.process(input));
            }
        }
        catch (Exception e) {
            Debug.debugException(e);
        }
    }

    @NotNull
    private ProcessResult process(@NotNull I input) {
        Object output = null;
        Throwable failureCause = null;
        try {
            output = this.processor.process(input);
        }
        catch (Throwable e) {
            failureCause = e;
        }
        return new ProcessResult(input, output, failureCause);
    }

    private final class ProcessResult
    implements Result<I, O> {
        @NotNull
        private final I inputItem;
        @Nullable
        private final O outputItem;
        @Nullable
        private final Throwable failureCause;

        private ProcessResult(@Nullable I inputItem, @Nullable O outputItem, Throwable failureCause) {
            this.inputItem = inputItem;
            this.outputItem = outputItem;
            this.failureCause = failureCause;
        }

        @Override
        @NotNull
        public I getInput() {
            return this.inputItem;
        }

        @Override
        @Nullable
        public O getOutput() {
            return this.outputItem;
        }

        @Override
        @Nullable
        public Throwable getFailureCause() {
            return this.failureCause;
        }
    }

    private final class Worker
    implements Runnable {
        private Worker() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            while (true) {
                try {
                    ParallelProcessor.this.workerSemaphore.acquire();
                }
                catch (InterruptedException e) {
                    Debug.debugException(e);
                    Thread.currentThread().interrupt();
                }
                if (ParallelProcessor.this.shutdown.get()) {
                    return;
                }
                try {
                    ParallelProcessor.this.processInParallel();
                    continue;
                }
                finally {
                    ParallelProcessor.this.processingCompleteSignal.countDown();
                    continue;
                }
                break;
            }
        }
    }
}

