/*
 * Decompiled with CFR 0.152.
 */
package org.apache.rocketmq.tieredstore.provider;

import com.google.common.base.Stopwatch;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.rocketmq.common.message.MessageQueue;
import org.apache.rocketmq.logging.org.slf4j.Logger;
import org.apache.rocketmq.logging.org.slf4j.LoggerFactory;
import org.apache.rocketmq.tieredstore.common.AppendResult;
import org.apache.rocketmq.tieredstore.common.TieredMessageStoreConfig;
import org.apache.rocketmq.tieredstore.exception.TieredStoreErrorCode;
import org.apache.rocketmq.tieredstore.exception.TieredStoreException;
import org.apache.rocketmq.tieredstore.provider.TieredStoreProvider;
import org.apache.rocketmq.tieredstore.util.MessageBufferUtil;

public abstract class TieredFileSegment
implements Comparable<TieredFileSegment>,
TieredStoreProvider {
    private static final Logger logger = LoggerFactory.getLogger((String)"RocketmqTieredStore");
    private volatile boolean closed = false;
    private final ReentrantLock bufferLock = new ReentrantLock();
    private final Semaphore commitLock = new Semaphore(1);
    private List<ByteBuffer> uploadBufferList = new ArrayList<ByteBuffer>();
    private boolean full;
    protected final FileSegmentType fileType;
    protected final MessageQueue messageQueue;
    protected final TieredMessageStoreConfig storeConfig;
    protected final long baseOffset;
    private volatile long commitPosition;
    private volatile long appendPosition;
    private final long maxSize;
    private long beginTimestamp = Long.MAX_VALUE;
    private long endTimestamp = Long.MAX_VALUE;
    private long commitMsgQueueOffset = 0L;
    private ByteBuffer codaBuffer;
    private CompletableFuture<Boolean> inflightCommitRequest = CompletableFuture.completedFuture(false);

    public TieredFileSegment(FileSegmentType fileType, MessageQueue messageQueue, long baseOffset, TieredMessageStoreConfig storeConfig) {
        this.fileType = fileType;
        this.messageQueue = messageQueue;
        this.storeConfig = storeConfig;
        this.baseOffset = baseOffset;
        this.commitPosition = 0L;
        this.appendPosition = 0L;
        switch (fileType) {
            case COMMIT_LOG: {
                this.maxSize = storeConfig.getTieredStoreCommitLogMaxSize();
                break;
            }
            case CONSUME_QUEUE: {
                this.maxSize = storeConfig.getTieredStoreConsumeQueueMaxSize();
                break;
            }
            case INDEX: {
                this.maxSize = Long.MAX_VALUE;
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported file type: " + (Object)((Object)fileType));
            }
        }
    }

    @Override
    public int compareTo(TieredFileSegment o) {
        return Long.compare(this.baseOffset, o.baseOffset);
    }

    public long getBaseOffset() {
        return this.baseOffset;
    }

    public long getCommitOffset() {
        return this.baseOffset + this.commitPosition;
    }

    public long getCommitPosition() {
        return this.commitPosition;
    }

    public long getCommitMsgQueueOffset() {
        return this.commitMsgQueueOffset;
    }

    public long getMaxOffset() {
        return this.baseOffset + this.appendPosition;
    }

    public long getMaxSize() {
        return this.maxSize;
    }

    public long getBeginTimestamp() {
        return this.beginTimestamp;
    }

    public void setBeginTimestamp(long beginTimestamp) {
        this.beginTimestamp = beginTimestamp;
    }

    public long getEndTimestamp() {
        return this.endTimestamp;
    }

    public void setEndTimestamp(long endTimestamp) {
        this.endTimestamp = endTimestamp;
    }

    public boolean isFull() {
        return this.full;
    }

    public void setFull() {
        this.setFull(true);
    }

    public void setFull(boolean appendCoda) {
        this.bufferLock.lock();
        try {
            this.full = true;
            if (this.fileType == FileSegmentType.COMMIT_LOG && appendCoda) {
                this.appendCoda();
            }
        }
        finally {
            this.bufferLock.unlock();
        }
    }

    public boolean isClosed() {
        return this.closed;
    }

    public void close() {
        this.closed = true;
    }

    public FileSegmentType getFileType() {
        return this.fileType;
    }

    public MessageQueue getMessageQueue() {
        return this.messageQueue;
    }

    public void initPosition(long pos) {
        this.commitPosition = pos;
        this.appendPosition = pos;
    }

    private List<ByteBuffer> rollingUploadBuffer() {
        this.bufferLock.lock();
        try {
            List<ByteBuffer> tmp = this.uploadBufferList;
            this.uploadBufferList = new ArrayList<ByteBuffer>();
            List<ByteBuffer> list = tmp;
            return list;
        }
        finally {
            this.bufferLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void sendBackBuffer(TieredFileSegmentInputStream inputStream) {
        this.bufferLock.lock();
        try {
            List<ByteBuffer> tmpBufferList = inputStream.getUploadBufferList();
            for (ByteBuffer buffer : tmpBufferList) {
                buffer.rewind();
            }
            tmpBufferList.addAll(this.uploadBufferList);
            this.uploadBufferList = tmpBufferList;
            if (inputStream.getCodaBuffer() != null) {
                this.codaBuffer.rewind();
            }
        }
        finally {
            this.bufferLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public AppendResult append(ByteBuffer byteBuf, long timeStamp) {
        if (this.closed) {
            return AppendResult.FILE_CLOSED;
        }
        this.bufferLock.lock();
        try {
            if (this.full || this.codaBuffer != null) {
                AppendResult appendResult = AppendResult.FILE_FULL;
                return appendResult;
            }
            if (this.fileType == FileSegmentType.INDEX) {
                this.beginTimestamp = byteBuf.getLong(4);
                this.endTimestamp = byteBuf.getLong(12);
                this.appendPosition += (long)byteBuf.remaining();
                this.uploadBufferList.add(byteBuf);
                this.setFull();
                AppendResult appendResult = AppendResult.SUCCESS;
                return appendResult;
            }
            if (this.appendPosition + (long)byteBuf.remaining() > this.maxSize) {
                this.setFull();
                AppendResult appendResult = AppendResult.FILE_FULL;
                return appendResult;
            }
            if (this.uploadBufferList.size() > this.storeConfig.getTieredStoreGroupCommitCount() || this.appendPosition - this.commitPosition > (long)this.storeConfig.getTieredStoreGroupCommitSize()) {
                this.commitAsync();
            }
            if (this.uploadBufferList.size() > this.storeConfig.getTieredStoreMaxGroupCommitCount()) {
                logger.debug("TieredFileSegment#append: buffer full: file: {}, upload buffer size: {}", (Object)this.getPath(), (Object)this.uploadBufferList.size());
                AppendResult appendResult = AppendResult.BUFFER_FULL;
                return appendResult;
            }
            if (timeStamp != Long.MAX_VALUE) {
                this.endTimestamp = timeStamp;
                if (this.beginTimestamp == Long.MAX_VALUE) {
                    this.beginTimestamp = timeStamp;
                }
            }
            this.appendPosition += (long)byteBuf.remaining();
            this.uploadBufferList.add(byteBuf);
            AppendResult appendResult = AppendResult.SUCCESS;
            return appendResult;
        }
        finally {
            this.bufferLock.unlock();
        }
    }

    private void appendCoda() {
        if (this.codaBuffer != null) {
            return;
        }
        this.codaBuffer = ByteBuffer.allocate(16);
        this.codaBuffer.putInt(16);
        this.codaBuffer.putInt(-875286124);
        this.codaBuffer.putLong(this.endTimestamp);
        this.codaBuffer.flip();
        this.appendPosition += 16L;
    }

    public ByteBuffer read(long position, int length) {
        return this.readAsync(position, length).join();
    }

    public CompletableFuture<ByteBuffer> readAsync(long position, int length) {
        CompletableFuture<ByteBuffer> future = new CompletableFuture<ByteBuffer>();
        if (position < 0L || length < 0) {
            future.completeExceptionally(new TieredStoreException(TieredStoreErrorCode.ILLEGAL_PARAM, "position or length is negative"));
            return future;
        }
        if (length == 0) {
            future.completeExceptionally(new TieredStoreException(TieredStoreErrorCode.ILLEGAL_PARAM, "length is zero"));
            return future;
        }
        if (position >= this.commitPosition) {
            future.completeExceptionally(new TieredStoreException(TieredStoreErrorCode.ILLEGAL_PARAM, "position is illegal"));
            return future;
        }
        if (position + (long)length > this.commitPosition) {
            logger.warn("TieredFileSegment#readAsync request position + length is greater than commit position, correct length using commit position, file: {}, request position: {}, commit position:{}, change length from {} to {}", new Object[]{this.getPath(), position, this.commitPosition, length, this.commitPosition - position});
            length = (int)(this.commitPosition - position);
            if (length == 0) {
                future.completeExceptionally(new TieredStoreException(TieredStoreErrorCode.NO_NEW_DATA, "request position is equal to commit position"));
                return future;
            }
            if (this.fileType == FileSegmentType.CONSUME_QUEUE && length % 20 != 0) {
                future.completeExceptionally(new TieredStoreException(TieredStoreErrorCode.ILLEGAL_PARAM, "position and length is illegal"));
                return future;
            }
        }
        return this.read0(position, length);
    }

    public boolean needCommit() {
        return this.appendPosition > this.commitPosition;
    }

    public boolean commit() {
        if (this.closed) {
            return false;
        }
        Boolean result = this.commitAsync().join();
        if (!result.booleanValue()) {
            result = this.inflightCommitRequest.join();
        }
        return result;
    }

    public CompletableFuture<Boolean> commitAsync() {
        if (this.closed) {
            return CompletableFuture.completedFuture(false);
        }
        Stopwatch stopwatch = Stopwatch.createStarted();
        if (!this.needCommit()) {
            return CompletableFuture.completedFuture(true);
        }
        try {
            int permits = this.commitLock.drainPermits();
            if (permits <= 0) {
                return CompletableFuture.completedFuture(false);
            }
        }
        catch (Exception e2) {
            return CompletableFuture.completedFuture(false);
        }
        List<ByteBuffer> bufferList = this.rollingUploadBuffer();
        int bufferSize = 0;
        for (ByteBuffer buffer : bufferList) {
            bufferSize += buffer.remaining();
        }
        if (this.codaBuffer != null) {
            bufferSize += this.codaBuffer.remaining();
        }
        if (bufferSize == 0) {
            return CompletableFuture.completedFuture(true);
        }
        TieredFileSegmentInputStream inputStream = new TieredFileSegmentInputStream(this.fileType, this.baseOffset + this.commitPosition, bufferList, this.codaBuffer, bufferSize);
        int finalBufferSize = bufferSize;
        try {
            this.inflightCommitRequest = ((CompletableFuture)((CompletableFuture)this.commit0(inputStream, this.commitPosition, bufferSize, this.fileType != FileSegmentType.INDEX).thenApply(result -> {
                if (result.booleanValue()) {
                    if (this.fileType == FileSegmentType.COMMIT_LOG && bufferList.size() > 0) {
                        this.commitMsgQueueOffset = MessageBufferUtil.getQueueOffset((ByteBuffer)bufferList.get(bufferList.size() - 1));
                    }
                    this.commitPosition += (long)finalBufferSize;
                    return true;
                }
                this.sendBackBuffer(inputStream);
                return false;
            })).exceptionally(e -> this.handleCommitException(inputStream, (Throwable)e))).whenComplete((result, e) -> {
                if (this.commitLock.availablePermits() == 0) {
                    logger.debug("TieredFileSegment#commitAsync: commit cost: {}ms, file: {}, item count: {}, buffer size: {}", new Object[]{stopwatch.elapsed(TimeUnit.MILLISECONDS), this.getPath(), bufferList.size(), finalBufferSize});
                    this.commitLock.release();
                } else {
                    logger.error("[Bug]TieredFileSegment#commitAsync: commit lock is already released: available permits: {}", (Object)this.commitLock.availablePermits());
                }
            });
            return this.inflightCommitRequest;
        }
        catch (Exception e3) {
            this.handleCommitException(inputStream, e3);
            if (this.commitLock.availablePermits() == 0) {
                logger.debug("TieredFileSegment#commitAsync: commit cost: {}ms, file: {}, item count: {}, buffer size: {}", new Object[]{stopwatch.elapsed(TimeUnit.MILLISECONDS), this.getPath(), bufferList.size(), finalBufferSize});
                this.commitLock.release();
            } else {
                logger.error("[Bug]TieredFileSegment#commitAsync: commit lock is already released: available permits: {}", (Object)this.commitLock.availablePermits());
            }
            return CompletableFuture.completedFuture(false);
        }
    }

    private boolean handleCommitException(TieredFileSegmentInputStream inputStream, Throwable e) {
        Throwable cause = e.getCause() != null ? e.getCause() : e;
        this.sendBackBuffer(inputStream);
        long realSize = 0L;
        if (cause instanceof TieredStoreException && ((TieredStoreException)cause).getPosition() > 0) {
            realSize = ((TieredStoreException)cause).getPosition();
        }
        if (realSize <= 0L) {
            realSize = this.getSize();
        }
        if (realSize > 0L && realSize > this.commitPosition) {
            logger.error("TieredFileSegment#handleCommitException: commit failed: file: {}, try to fix position: origin: {}, real: {}", new Object[]{this.getPath(), this.commitPosition, realSize, cause});
            long diff = this.appendPosition - this.commitPosition;
            this.commitPosition = realSize;
            this.appendPosition = realSize + diff;
        } else if (realSize < this.commitPosition) {
            logger.error("[Bug]TieredFileSegment#handleCommitException: commit failed: file: {}, can not fix position: origin: {}, real: {}", new Object[]{this.getPath(), this.commitPosition, realSize, cause});
        }
        return false;
    }

    public static class TieredFileSegmentInputStream
    extends InputStream {
        private final FileSegmentType fileType;
        private final List<ByteBuffer> uploadBufferList;
        private int bufferReadIndex = 0;
        private int readOffset = 0;
        private long commitLogOffset;
        private final ByteBuffer commitLogOffsetBuffer = ByteBuffer.allocate(8);
        private final ByteBuffer codaBuffer;
        private ByteBuffer curBuffer;
        private final int contentLength;
        private int readBytes = 0;

        public TieredFileSegmentInputStream(FileSegmentType fileType, long startOffset, List<ByteBuffer> uploadBufferList, ByteBuffer codaBuffer, int contentLength) {
            this.fileType = fileType;
            this.commitLogOffset = startOffset;
            this.commitLogOffsetBuffer.putLong(0, startOffset);
            this.uploadBufferList = uploadBufferList;
            this.codaBuffer = codaBuffer;
            this.contentLength = contentLength;
            if (uploadBufferList.size() > 0) {
                this.curBuffer = uploadBufferList.get(0);
            }
            if (fileType == FileSegmentType.INDEX && uploadBufferList.size() != 1) {
                logger.error("[Bug]TieredFileSegmentInputStream: index file must have only one buffer");
            }
        }

        public List<ByteBuffer> getUploadBufferList() {
            return this.uploadBufferList;
        }

        public ByteBuffer getCodaBuffer() {
            return this.codaBuffer;
        }

        @Override
        public int available() {
            return this.contentLength - this.readBytes;
        }

        @Override
        public int read() {
            int res;
            if (this.bufferReadIndex >= this.uploadBufferList.size()) {
                return this.readCoda();
            }
            switch (this.fileType) {
                case COMMIT_LOG: {
                    if (this.readOffset >= this.curBuffer.remaining()) {
                        ++this.bufferReadIndex;
                        if (this.bufferReadIndex >= this.uploadBufferList.size()) {
                            return this.readCoda();
                        }
                        this.curBuffer = this.uploadBufferList.get(this.bufferReadIndex);
                        this.commitLogOffset += (long)this.readOffset;
                        this.commitLogOffsetBuffer.putLong(0, this.commitLogOffset);
                        this.readOffset = 0;
                    }
                    if (this.readOffset >= 28 && this.readOffset < 36) {
                        res = this.commitLogOffsetBuffer.get(this.readOffset - 28) & 0xFF;
                        ++this.readOffset;
                        break;
                    }
                    res = this.curBuffer.get(this.readOffset++) & 0xFF;
                    break;
                }
                case CONSUME_QUEUE: {
                    if (!this.curBuffer.hasRemaining()) {
                        ++this.bufferReadIndex;
                        if (this.bufferReadIndex >= this.uploadBufferList.size()) {
                            return -1;
                        }
                        this.curBuffer = this.uploadBufferList.get(this.bufferReadIndex);
                    }
                    res = this.curBuffer.get() & 0xFF;
                    break;
                }
                case INDEX: {
                    if (!this.curBuffer.hasRemaining()) {
                        return -1;
                    }
                    res = this.curBuffer.get() & 0xFF;
                    break;
                }
                default: {
                    throw new IllegalStateException("unknown file type");
                }
            }
            ++this.readBytes;
            return res;
        }

        private int readCoda() {
            if (this.fileType != FileSegmentType.COMMIT_LOG || this.codaBuffer == null) {
                return -1;
            }
            if (!this.codaBuffer.hasRemaining()) {
                return -1;
            }
            ++this.readBytes;
            return this.codaBuffer.get() & 0xFF;
        }
    }

    public static enum FileSegmentType {
        COMMIT_LOG(0),
        CONSUME_QUEUE(1),
        INDEX(2);

        private int type;

        private FileSegmentType(int type) {
            this.type = type;
        }

        public int getType() {
            return this.type;
        }

        public static FileSegmentType valueOf(int type) {
            switch (type) {
                case 0: {
                    return COMMIT_LOG;
                }
                case 1: {
                    return CONSUME_QUEUE;
                }
                case 2: {
                    return INDEX;
                }
            }
            throw new IllegalStateException("Unexpected value: " + type);
        }
    }
}

