import { error, until } from "selenium-webdriver";
import Element from "./Element";
import ErrorProcessor from "../utils/tools/ErrorProcessor";
class LocatorWaitAction {
    constructor(driver, by, timeout = LocatorWaitAction.DEFAULT_TIMEOUT) {
        this.driver = driver;
        this.by = by;
        this.timeout = timeout;
    }
    async absent() {
        await this.driver.wait(async () => (await this.driver.findElements(this.by)).length === 0, this.timeout);
        return Promise.resolve();
    }
    async untilAbsent() {
        return await ErrorProcessor.run(async () => await this.absent(), "Error while waiting until absent " + this.by);
    }
    async isAbsent() {
        try {
            await this.absent();
            return true;
        }
        catch (err) {
            if (err instanceof error.TimeoutError) {
                return false;
            }
            else {
                throw err;
            }
        }
    }
    async present() {
        return new Element(await this.driver.wait(until.elementLocated(this.by), this.timeout));
    }
    async untilPresent() {
        await ErrorProcessor.run(async () => await this.present(), "Error while waiting until present: " + this.by);
        return Promise.resolve();
    }
    async isPresent() {
        try {
            await this.present();
            return true;
        }
        catch (err) {
            if (err instanceof error.TimeoutError) {
                return false;
            }
            else {
                throw err;
            }
        }
    }
    async visible() {
        const webElement = await this.driver.findElement(this.by);
        await this.driver.wait(until.elementIsVisible(webElement), this.timeout);
        return Promise.resolve();
    }
    async untilVisible() {
        return await ErrorProcessor.run(async () => await this.visible(), "Error while waiting until visible: " + this.by);
    }
    async isVisible() {
        try {
            await this.visible();
            return true;
        }
        catch (err) {
            if (err instanceof error.TimeoutError) {
                return false;
            }
            else {
                throw err;
            }
        }
    }
    async value() {
        const webElement = await this.driver.findElement(this.by);
        await this.driver.wait(async () => (await webElement.getAttribute("value")) !== "", this.timeout);
        return await webElement.getAttribute("value");
    }
    async untilHasValue() {
        return await ErrorProcessor.run(async () => await this.value(), "Error while waiting until has value: " + this.by);
    }
    async hasValue() {
        try {
            await this.value();
            return true;
        }
        catch (err) {
            if (err instanceof error.TimeoutError) {
                return false;
            }
            else {
                throw err;
            }
        }
    }
    async enabled() {
        const webElement = await this.driver.findElement(this.by);
        return new Element(await this.driver.wait(until.elementIsEnabled(webElement), this.timeout));
    }
    async untilEnabled() {
        await ErrorProcessor.run(async () => await this.enabled(), "Error while waiting until enabled: " + this.by);
        return Promise.resolve();
    }
}
LocatorWaitAction.DEFAULT_TIMEOUT = 100;
export default LocatorWaitAction;
//# sourceMappingURL=LocatorWaitAction.js.map