import { jsx as _jsx, Fragment as _Fragment } from "react/jsx-runtime";
import { useCallback, useEffect, useMemo, useState } from "react";
import { Button } from "@patternfly/react-core/dist/js/components/Button";
import { Modal, ModalVariant } from "@patternfly/react-core/dist/js/components/Modal";
import { Tooltip } from "@patternfly/react-core/dist/js/components/Tooltip";
import { Wizard } from "@patternfly/react-core/dist/js/components/Wizard";
import { useImportJavaClassesWizardI18n } from "../../i18n";
import { ImportJavaClassesWizardFirstStep } from "./ImportJavaClassesWizardFirstStep";
import { ImportJavaClassesWizardSecondStep } from "./ImportJavaClassesWizardSecondStep";
import { ImportJavaClassesWizardThirdStep } from "./ImportJavaClassesWizardThirdStep";
import { JavaClass } from "./model/JavaClass";
import { DMNSimpleType } from "./model/DMNSimpleType";
import { getJavaClassSimpleName } from "./model/JavaClassUtils";
export const useLanguageServerAvailable = (javaCodeCompletionService) => {
    const [languageServerStatus, setLanguageServerStatus] = useState("loading");
    useEffect(() => {
        try {
            javaCodeCompletionService
                .isLanguageServerAvailable()
                .then((available) => {
                setLanguageServerStatus(available ? "enable" : "disable");
            })
                .catch((reason) => {
                setLanguageServerStatus("error");
                console.error(reason);
            });
        }
        catch (error) {
            setLanguageServerStatus("error");
            console.error(error);
        }
    }, [javaCodeCompletionService]);
    const isLanguageServerDisabled = useMemo(() => {
        return "enable" !== languageServerStatus;
    }, [languageServerStatus]);
    const isLanguageServerLoading = useMemo(() => {
        return "loading" == languageServerStatus;
    }, [languageServerStatus]);
    const isLanguageServerError = useMemo(() => {
        return "error" == languageServerStatus;
    }, [languageServerStatus]);
    return { languageServerStatus, isLanguageServerDisabled, isLanguageServerLoading, isLanguageServerError };
};
export const ImportJavaClassesButton = ({ javaCodeCompletionService, handleButtonClick, }) => {
    const { i18n } = useImportJavaClassesWizardI18n();
    const { isLanguageServerDisabled, isLanguageServerLoading, isLanguageServerError } = useLanguageServerAvailable(javaCodeCompletionService);
    const defineTooltipMessage = useCallback(() => {
        if (isLanguageServerDisabled) {
            return i18n.modalButton.disabledMessage;
        }
        else if (isLanguageServerError) {
            return i18n.modalButton.errorMessage;
        }
        return undefined;
    }, [
        isLanguageServerDisabled,
        isLanguageServerError,
        i18n.modalButton.disabledMessage,
        i18n.modalButton.errorMessage,
    ]);
    return (_jsx(_Fragment, { children: defineTooltipMessage() ? (_jsx(Tooltip, { content: defineTooltipMessage(), children: _jsx(Button, { "data-testid": "modal-wizard-button", isAriaDisabled: isLanguageServerDisabled, isLoading: isLanguageServerLoading, onClick: handleButtonClick, variant: "secondary", children: i18n.modalButton.text }) })) : (_jsx(Button, { "data-testid": "modal-wizard-button", isAriaDisabled: isLanguageServerDisabled, isLoading: isLanguageServerLoading, onClick: handleButtonClick, variant: "secondary", children: i18n.modalButton.text })) }));
};
export const ImportJavaClassesWizard = ({ javaCodeCompletionService, isOpen, onClose, onSave, }) => {
    const { i18n } = useImportJavaClassesWizardI18n();
    const [javaClasses, setJavaClasses] = useState([]);
    const updateJavaFieldsReferences = useCallback((updatedJavaClasses, previousJavaClasses) => {
        const updatedJavaClassesNames = updatedJavaClasses.map((javaClass) => javaClass.name);
        const previousJavaClassesNames = previousJavaClasses.map((javaClass) => javaClass.name);
        const allFields = javaClasses.map((javaClass) => javaClass.fields).flat(1);
        allFields.forEach((field) => {
            if (field.dmnTypeRef === DMNSimpleType.ANY && updatedJavaClassesNames.includes(field.type)) {
                field.dmnTypeRef = getJavaClassSimpleName(field.type);
            }
            else if (previousJavaClassesNames.includes(field.type) && !updatedJavaClassesNames.includes(field.type)) {
                field.dmnTypeRef = DMNSimpleType.ANY;
            }
        });
    }, [javaClasses]);
    const addJavaClass = useCallback((fullClassName) => {
        setJavaClasses((prevState) => {
            if (!prevState.some((javaClass) => javaClass.name === fullClassName)) {
                const updatedSelectedJavaClasses = [...prevState, new JavaClass(fullClassName)];
                updatedSelectedJavaClasses.sort((a, b) => (a.name < b.name ? -1 : 1));
                updateJavaFieldsReferences(updatedSelectedJavaClasses, prevState);
                return updatedSelectedJavaClasses;
            }
            return prevState;
        });
    }, [updateJavaFieldsReferences]);
    const removeJavaClass = useCallback((fullClassName) => {
        setJavaClasses((prevState) => {
            const updatedSelectedJavaClasses = prevState.filter((javaClass) => javaClass.name !== fullClassName);
            updateJavaFieldsReferences(updatedSelectedJavaClasses, prevState);
            return updatedSelectedJavaClasses;
        });
    }, [updateJavaFieldsReferences]);
    const updateSelectedClassesFields = useCallback((fullClassName, fields) => {
        setJavaClasses((prevState) => {
            const updatedJavaClasses = [...prevState];
            const javaClassIndex = updatedJavaClasses.findIndex((javaClass) => javaClass.name === fullClassName);
            if (javaClassIndex > -1) {
                updatedJavaClasses[javaClassIndex].setFields(fields);
            }
            return updatedJavaClasses;
        });
    }, []);
    const isSecondStepActivatable = useCallback(() => {
        return javaClasses.length > 0;
    }, [javaClasses]);
    const isThirdStepActivatable = useCallback(() => {
        return javaClasses.length > 0 && javaClasses.every((javaClass) => javaClass.fieldsLoaded);
    }, [javaClasses]);
    const handleWizardClose = useCallback(() => {
        setJavaClasses([]);
        onClose === null || onClose === void 0 ? void 0 : onClose();
    }, [onClose]);
    const handleWizardSave = useCallback(() => {
        onSave === null || onSave === void 0 ? void 0 : onSave(javaClasses);
        handleWizardClose();
    }, [javaClasses, handleWizardClose, onSave]);
    const steps = [
        {
            canJumpTo: true,
            component: (_jsx(ImportJavaClassesWizardFirstStep, { javaCodeCompletionService: javaCodeCompletionService, onAddJavaClass: addJavaClass, onRemoveJavaClass: removeJavaClass, selectedJavaClasses: javaClasses })),
            enableNext: isSecondStepActivatable(),
            hideBackButton: true,
            name: i18n.modalWizard.firstStep.stepName,
        },
        {
            canJumpTo: isSecondStepActivatable(),
            component: (_jsx(ImportJavaClassesWizardSecondStep, { javaCodeCompletionService: javaCodeCompletionService, onAddJavaClass: addJavaClass, onSelectedJavaClassedFieldsLoaded: updateSelectedClassesFields, selectedJavaClasses: javaClasses })),
            enableNext: isThirdStepActivatable(),
            name: i18n.modalWizard.secondStep.stepName,
        },
        {
            canJumpTo: isThirdStepActivatable(),
            component: _jsx(ImportJavaClassesWizardThirdStep, { selectedJavaClasses: javaClasses }),
            name: i18n.modalWizard.thirdStep.stepName,
            nextButtonText: i18n.modalWizard.thirdStep.nextButtonText,
        },
    ];
    return (_jsx(Modal, { description: i18n.modalWizard.description, isOpen: isOpen, onClose: handleWizardClose, title: i18n.modalWizard.title, variant: ModalVariant.large, children: _jsx(Wizard, { className: "import-java-classes", height: 600, onClose: handleWizardClose, onSave: handleWizardSave, steps: steps }) }));
};
//# sourceMappingURL=ImportJavaClassesWizard.js.map