import { GraphQL } from "../graphql";
import { OperationType, } from "@kie-tools/runtime-tools-shared-gateway-api/dist/types";
import { buildProcessListWhereArgument } from "./QueryUtils";
import axios from "axios";
export const getProcessInstances = async (offset, limit, filters, sortBy, client) => {
    return new Promise((resolve, reject) => {
        client
            .query({
            query: GraphQL.GetProcessInstancesDocument,
            variables: {
                where: buildProcessListWhereArgument(filters),
                offset: offset,
                limit: limit,
                orderBy: sortBy,
            },
            fetchPolicy: "network-only",
            errorPolicy: "all",
        })
            .then((value) => {
            resolve(value.data.ProcessInstances);
        })
            .catch((reason) => {
            reject({ errorMessage: JSON.stringify(reason) });
        });
    });
};
export const getChildProcessInstances = async (rootProcessInstanceId, client) => {
    return new Promise((resolve, reject) => {
        client
            .query({
            query: GraphQL.GetChildInstancesDocument,
            variables: {
                rootProcessInstanceId,
            },
        })
            .then((value) => {
            resolve(value.data.ProcessInstances);
        })
            .catch((reason) => reject(reason));
    });
};
export const performMultipleCancel = async (jobsToBeActioned, client) => {
    const multipleCancel = jobsToBeActioned.map((job) => {
        return new Promise((resolve, reject) => {
            client
                .mutate({
                mutation: GraphQL.JobCancelDocument,
                variables: {
                    jobId: job.id,
                },
                fetchPolicy: "no-cache",
            })
                .then((value) => {
                resolve({ successJob: job });
            })
                .catch((reason) => {
                job.errorMessage = JSON.stringify(reason.message);
                reject({ failedJob: job });
            });
        });
    });
    return Promise.all(multipleCancel.map((mc) => mc.catch((error) => error))).then((result) => {
        return Promise.resolve(result);
    });
};
export const jobCancel = async (job, client) => {
    let modalTitle;
    let modalContent;
    return new Promise((resolve, reject) => {
        client
            .mutate({
            mutation: GraphQL.JobCancelDocument,
            variables: {
                jobId: job.id,
            },
            fetchPolicy: "no-cache",
        })
            .then((value) => {
            modalTitle = "success";
            modalContent = `The job: ${job.id} is canceled successfully`;
            resolve({ modalTitle, modalContent });
        })
            .catch((reason) => {
            modalTitle = "failure";
            modalContent = `The job: ${job.id} failed to cancel. Error message: ${reason.message}`;
            reject({ modalTitle, modalContent });
        });
    });
};
export const handleJobReschedule = async (job, repeatInterval, repeatLimit, scheduleDate, client) => {
    let modalTitle;
    let modalContent;
    let parameter = {};
    if (repeatInterval === null && repeatLimit === null) {
        parameter = {
            expirationTime: new Date(scheduleDate),
        };
    }
    else {
        parameter = {
            expirationTime: new Date(scheduleDate),
            repeatInterval,
            repeatLimit,
        };
    }
    return new Promise((resolve, reject) => {
        client
            .mutate({
            mutation: GraphQL.HandleJobRescheduleDocument,
            variables: {
                jobId: job.id,
                data: JSON.stringify(parameter),
            },
            fetchPolicy: "no-cache",
        })
            .then((value) => {
            modalTitle = "success";
            modalContent = `Reschedule of job: ${job.id} is successful`;
            resolve({ modalTitle, modalContent });
        })
            .catch((reason) => {
            modalTitle = "failure";
            modalContent = `Reschedule of job ${job.id} failed. Message: ${reason.message}`;
            reject({ modalTitle, modalContent });
        });
    });
};
export const handleProcessSkip = async (processInstance, client) => {
    return new Promise((resolve, reject) => {
        client
            .mutate({
            mutation: GraphQL.AbortProcessInstanceDocument,
            variables: {
                processId: processInstance.id,
            },
            fetchPolicy: "no-cache",
        })
            .then((value) => {
            resolve(value.data);
        })
            .catch((reason) => reject(reason));
    });
};
export const handleProcessRetry = async (processInstance, client) => {
    return new Promise((resolve, reject) => {
        client
            .mutate({
            mutation: GraphQL.RetryProcessInstanceDocument,
            variables: {
                processId: processInstance.id,
            },
            fetchPolicy: "no-cache",
        })
            .then((value) => {
            resolve(value.data);
        })
            .catch((reason) => reject(reason));
    });
};
export const handleProcessAbort = async (processInstance, client) => {
    return new Promise((resolve, reject) => {
        client
            .mutate({
            mutation: GraphQL.AbortProcessInstanceDocument,
            variables: {
                processId: processInstance.id,
            },
            fetchPolicy: "no-cache",
        })
            .then((value) => {
            resolve(value.data);
        })
            .catch((reason) => reject(reason));
    });
};
export const handleProcessMultipleAction = async (processInstances, operationType, client) => {
    let operation;
    return new Promise(async (resolve, reject) => {
        const successProcessInstances = [];
        const failedProcessInstances = [];
        switch (operationType) {
            case OperationType.ABORT:
                operation = handleProcessAbort;
                break;
            case OperationType.SKIP:
                operation = handleProcessSkip;
                break;
            case OperationType.RETRY:
                operation = handleProcessRetry;
                break;
        }
        for (const processInstance of processInstances) {
            await operation(processInstance, client)
                .then(() => {
                successProcessInstances.push(processInstance);
            })
                .catch((error) => {
                processInstance.errorMessage = error.message;
                failedProcessInstances.push(processInstance);
            });
        }
        resolve({ successProcessInstances, failedProcessInstances });
    });
};
export const handleNodeTrigger = async (processInstance, node, client) => {
    return new Promise((resolve, reject) => {
        client
            .mutate({
            mutation: GraphQL.HandleNodeTriggerDocument,
            variables: {
                processId: processInstance.id,
                nodeId: node.id,
            },
            fetchPolicy: "no-cache",
        })
            .then((value) => {
            resolve(value.data);
        })
            .catch((reason) => reject(reason));
    });
};
export const handleProcessVariableUpdate = async (processInstance, updatedJson, client) => {
    return new Promise((resolve, reject) => {
        client
            .mutate({
            mutation: GraphQL.HandleProcessVariableUpdateDocument,
            variables: {
                processId: processInstance.id,
                processInstanceVariables: JSON.stringify(updatedJson),
            },
            fetchPolicy: "no-cache",
        })
            .then((value) => {
            resolve(JSON.parse(value.data.ProcessInstanceUpdateVariables));
        })
            .catch((reason) => reject(reason));
    });
};
export const handleNodeInstanceCancel = async (processInstance, node, client) => {
    return new Promise((resolve, reject) => {
        client
            .mutate({
            mutation: GraphQL.HandleNodeInstanceCancelDocument,
            variables: {
                processId: processInstance.id,
                nodeInstanceId: node.id,
            },
            fetchPolicy: "no-cache",
        })
            .then((value) => {
            resolve();
        })
            .catch((reason) => reject(JSON.stringify(reason.message)));
    });
};
export const handleNodeInstanceRetrigger = async (processInstance, node, client) => {
    return new Promise((resolve, reject) => {
        client
            .mutate({
            mutation: GraphQL.HandleNodeInstanceRetriggerDocument,
            variables: {
                processId: processInstance.id,
                nodeInstanceId: node.id,
            },
            fetchPolicy: "no-cache",
        })
            .then((value) => {
            resolve();
        })
            .catch((reason) => reject(JSON.stringify(reason.message)));
    });
};
export const getSVG = async (processInstance, client) => {
    return client
        .query({
        query: GraphQL.GetProcessInstanceSvgDocument,
        variables: {
            processId: processInstance.id,
        },
        fetchPolicy: "network-only",
    })
        .then((value) => {
        return { svg: value.data.ProcessInstances[0].diagram };
    })
        .catch((reason) => {
        return { error: reason.message };
    });
};
export const getProcessDetails = async (id, client) => {
    return new Promise((resolve, reject) => {
        client
            .query({
            query: GraphQL.GetProcessInstanceByIdDocument,
            variables: {
                id,
            },
            fetchPolicy: "network-only",
        })
            .then((value) => {
            resolve(value.data.ProcessInstances[0]);
        })
            .catch((error) => {
            reject(error["graphQLErrors"][0]["message"]);
        });
    });
};
export const getJobs = async (id, client) => {
    return client
        .query({
        query: GraphQL.GetJobsByProcessInstanceIdDocument,
        variables: {
            processInstanceId: id,
        },
        fetchPolicy: "network-only",
    })
        .then((value) => {
        return value.data.Jobs;
    })
        .catch((error) => {
        return error;
    });
};
export const getTriggerableNodes = async (processInstance, client) => {
    return client
        .query({
        query: GraphQL.GetProcessDefinitionNodesDocument,
        variables: {
            processId: processInstance.processId,
        },
        fetchPolicy: "no-cache",
    })
        .then((value) => {
        return value.data.ProcessDefinitions[0].nodes;
    })
        .catch((reason) => {
        return reason;
    });
};
export const getJobsWithFilters = async (offset, limit, filters, orderBy, client) => {
    try {
        const response = await client.query({
            query: GraphQL.GetJobsWithFiltersDocument,
            variables: {
                values: filters,
                offset: offset,
                limit: limit,
                orderBy,
            },
            fetchPolicy: "network-only",
        });
        return Promise.resolve(response.data.Jobs);
    }
    catch (error) {
        return Promise.reject(error);
    }
};
export const getForms = (baseUrl, formFilter) => {
    return new Promise((resolve, reject) => {
        axios
            .get(new URL(`forms/list`, baseUrl).toString(), {
            params: {
                names: formFilter.join(";"),
            },
        })
            .then((result) => {
            resolve(result.data);
        })
            .catch((error) => reject(error));
    });
};
export const getFormContent = (baseUrl, formName) => {
    return new Promise((resolve, reject) => {
        axios
            .get(new URL(`forms/${formName}`, baseUrl).toString())
            .then((result) => {
            resolve(result.data);
        })
            .catch((error) => reject(error));
    });
};
export const saveFormContent = (baseUrl, formName, content) => {
    return new Promise((resolve, reject) => {
        axios
            .post(new URL(`forms/${formName}`, baseUrl).toString(), content)
            .then((result) => {
            resolve();
        })
            .catch((error) => reject(error));
    });
};
export const getProcessDefinitions = (client) => {
    return new Promise((resolve, reject) => {
        client
            .query({
            query: GraphQL.GetProcessDefinitionsDocument,
            fetchPolicy: "network-only",
            errorPolicy: "all",
        })
            .then((value) => {
            var _a;
            resolve(((_a = value.data.ProcessDefinitions) !== null && _a !== void 0 ? _a : []).map((item) => {
                return {
                    processName: item.id,
                    endpoint: item.endpoint,
                };
            }));
        })
            .catch((reason) => {
            reject({ errorMessage: JSON.stringify(reason) });
        });
    });
};
export const getProcessSchema = (processDefinitionData) => {
    return new Promise((resolve, reject) => {
        axios
            .get(`${processDefinitionData.endpoint}/schema`)
            .then((response) => {
            if (response.status === 200) {
                resolve(response.data);
            }
        })
            .catch((error) => {
            reject(error);
        });
    });
};
export const getCustomForm = (processDefinitionData) => {
    return new Promise((resolve, reject) => {
        const lastIndex = processDefinitionData.endpoint.lastIndexOf(`/${processDefinitionData.processName}`);
        const baseEndpoint = processDefinitionData.endpoint.slice(0, lastIndex);
        axios
            .get(`${baseEndpoint}/forms/${processDefinitionData.processName}`)
            .then((response) => {
            if (response.status === 200) {
                resolve(response.data);
            }
        })
            .catch((error) => {
            reject(error);
        });
    });
};
export const startProcessInstance = (formData, businessKey, processDefinitionData) => {
    return new Promise((resolve, reject) => {
        const requestURL = `${processDefinitionData.endpoint}${businessKey.length > 0 ? `?businessKey=${businessKey}` : ""}`;
        axios
            .post(requestURL, formData, {
            headers: {
                "Content-Type": "application/json",
            },
        })
            .then((response) => {
            resolve(response.data.id);
        })
            .catch((error) => reject(error));
    });
};
//# sourceMappingURL=apis.js.map