/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite3.internal.index;

import java.util.ArrayList;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import org.apache.ignite3.internal.catalog.Catalog;
import org.apache.ignite3.internal.catalog.CatalogService;
import org.apache.ignite3.internal.catalog.descriptors.CatalogIndexDescriptor;
import org.apache.ignite3.internal.catalog.descriptors.CatalogIndexStatus;
import org.apache.ignite3.internal.catalog.descriptors.CatalogTableDescriptor;
import org.apache.ignite3.internal.catalog.descriptors.CatalogZoneDescriptor;
import org.apache.ignite3.internal.catalog.events.CatalogEvent;
import org.apache.ignite3.internal.catalog.events.RemoveIndexEventParameters;
import org.apache.ignite3.internal.catalog.events.StartBuildingIndexEventParameters;
import org.apache.ignite3.internal.close.ManuallyCloseable;
import org.apache.ignite3.internal.components.NodeProperties;
import org.apache.ignite3.internal.event.EventListener;
import org.apache.ignite3.internal.failure.FailureContext;
import org.apache.ignite3.internal.failure.FailureProcessor;
import org.apache.ignite3.internal.failure.FailureType;
import org.apache.ignite3.internal.hlc.ClockService;
import org.apache.ignite3.internal.hlc.HybridTimestamp;
import org.apache.ignite3.internal.index.IndexBuilder;
import org.apache.ignite3.internal.index.IndexManagementUtils;
import org.apache.ignite3.internal.index.IndexManager;
import org.apache.ignite3.internal.lang.NodeStoppingException;
import org.apache.ignite3.internal.logger.IgniteLogger;
import org.apache.ignite3.internal.logger.Loggers;
import org.apache.ignite3.internal.network.ClusterService;
import org.apache.ignite3.internal.network.InternalClusterNode;
import org.apache.ignite3.internal.placementdriver.PlacementDriver;
import org.apache.ignite3.internal.placementdriver.PrimaryReplicaAwaitTimeoutException;
import org.apache.ignite3.internal.placementdriver.ReplicaMeta;
import org.apache.ignite3.internal.placementdriver.event.PrimaryReplicaEvent;
import org.apache.ignite3.internal.placementdriver.event.PrimaryReplicaEventParameters;
import org.apache.ignite3.internal.replicator.PartitionGroupId;
import org.apache.ignite3.internal.replicator.ReplicationGroupId;
import org.apache.ignite3.internal.replicator.TablePartitionId;
import org.apache.ignite3.internal.replicator.ZonePartitionId;
import org.apache.ignite3.internal.storage.MvPartitionStorage;
import org.apache.ignite3.internal.storage.StorageClosedException;
import org.apache.ignite3.internal.storage.engine.MvTableStorage;
import org.apache.ignite3.internal.storage.index.IndexStorage;
import org.apache.ignite3.internal.util.CompletableFutures;
import org.apache.ignite3.internal.util.ExceptionUtils;
import org.apache.ignite3.internal.util.IgniteSpinBusyLock;
import org.apache.ignite3.internal.util.IgniteUtils;

class IndexBuildController
implements ManuallyCloseable {
    private static final IgniteLogger LOG = Loggers.forClass(IndexBuildController.class);
    private final IndexBuilder indexBuilder;
    private final IndexManager indexManager;
    private final CatalogService catalogService;
    private final ClusterService clusterService;
    private final PlacementDriver placementDriver;
    private final ClockService clockService;
    private final FailureProcessor failureProcessor;
    private final NodeProperties nodeProperties;
    private final IgniteSpinBusyLock busyLock = new IgniteSpinBusyLock();
    private final AtomicBoolean closeGuard = new AtomicBoolean();
    private final Set<ReplicationGroupId> primaryReplicaIds = ConcurrentHashMap.newKeySet();

    IndexBuildController(IndexBuilder indexBuilder, IndexManager indexManager, CatalogService catalogService, ClusterService clusterService, PlacementDriver placementDriver, ClockService clockService, FailureProcessor failureProcessor, NodeProperties nodeProperties) {
        this.indexBuilder = indexBuilder;
        this.indexManager = indexManager;
        this.catalogService = catalogService;
        this.clusterService = clusterService;
        this.placementDriver = placementDriver;
        this.clockService = clockService;
        this.failureProcessor = failureProcessor;
        this.nodeProperties = nodeProperties;
    }

    public void start() {
        IgniteUtils.inBusyLock(this.busyLock, this::addListeners);
    }

    @Override
    public void close() {
        if (!this.closeGuard.compareAndSet(false, true)) {
            return;
        }
        this.busyLock.block();
        this.indexBuilder.close();
    }

    private void addListeners() {
        this.catalogService.listen(CatalogEvent.INDEX_BUILDING, EventListener.fromConsumer(this::onIndexBuilding));
        this.catalogService.listen(CatalogEvent.INDEX_REMOVED, EventListener.fromConsumer(this::onIndexRemoved));
        this.placementDriver.listen(PrimaryReplicaEvent.PRIMARY_REPLICA_ELECTED, EventListener.fromConsumer(this::onPrimaryReplicaElected));
    }

    private void onIndexBuilding(StartBuildingIndexEventParameters parameters) {
        IgniteUtils.inBusyLockAsync(this.busyLock, () -> {
            Catalog catalog = this.catalogService.catalog(parameters.catalogVersion());
            assert (catalog != null) : "Failed to find a catalog for the specified version [version=" + parameters.catalogVersion() + ", earliestVersion=" + this.catalogService.earliestCatalogVersion() + ", latestVersion=" + this.catalogService.latestCatalogVersion() + "].";
            CatalogIndexDescriptor indexDescriptor = catalog.index(parameters.indexId());
            assert (indexDescriptor != null) : "Failed to find an index descriptor for the specified index [indexId=" + parameters.indexId() + ", catalogVersion=" + parameters.catalogVersion() + "].";
            assert (catalog.table(indexDescriptor.tableId()) != null) : "Failed to find a table descriptor for the specified index [indexId=" + parameters.indexId() + ", tableId=" + indexDescriptor.tableId() + ", catalogVersion=" + parameters.catalogVersion() + "].";
            CatalogZoneDescriptor zoneDescriptor = catalog.zone(catalog.table(indexDescriptor.tableId()).zoneId());
            assert (zoneDescriptor != null) : "Failed to find a zone descriptor for the specified table [indexId=" + parameters.indexId() + ", tableId=" + indexDescriptor.tableId() + ", catalogVersion=" + parameters.catalogVersion() + "].";
            ArrayList<CompletionStage> startBuildIndexFutures = new ArrayList<CompletionStage>();
            for (ReplicationGroupId primaryReplicationGroupId : this.primaryReplicaIds) {
                boolean needToProcessPartition;
                int partitionId;
                int zoneId;
                if (primaryReplicationGroupId instanceof ZonePartitionId) {
                    ZonePartitionId zoneReplicaId = (ZonePartitionId)primaryReplicationGroupId;
                    zoneId = zoneReplicaId.zoneId();
                    partitionId = zoneReplicaId.partitionId();
                    needToProcessPartition = zoneReplicaId.zoneId() == zoneDescriptor.id();
                } else {
                    TablePartitionId partitionReplicaId = (TablePartitionId)primaryReplicationGroupId;
                    needToProcessPartition = partitionReplicaId.tableId() == indexDescriptor.tableId();
                    if (!needToProcessPartition) continue;
                    zoneId = catalog.table(partitionReplicaId.tableId()).zoneId();
                    partitionId = partitionReplicaId.partitionId();
                }
                if (!needToProcessPartition) continue;
                CompletionStage startBuildIndexFuture = this.indexManager.getMvTableStorage(parameters.causalityToken(), indexDescriptor.tableId()).thenCompose(mvTableStorage -> {
                    HybridTimestamp buildAttemptTimestamp = this.clockService.now();
                    return this.awaitPrimaryReplica(primaryReplicationGroupId, buildAttemptTimestamp).thenAccept(replicaMeta -> this.tryScheduleBuildIndex(zoneId, indexDescriptor.tableId(), partitionId, primaryReplicationGroupId, indexDescriptor, (MvTableStorage)mvTableStorage, (ReplicaMeta)replicaMeta, buildAttemptTimestamp));
                });
                startBuildIndexFutures.add(startBuildIndexFuture);
            }
            return CompletableFutures.allOf(startBuildIndexFutures);
        }).whenComplete((res, ex) -> {
            if (ex != null && !ExceptionUtils.hasCause(ex, NodeStoppingException.class, StorageClosedException.class)) {
                this.failureProcessor.process(new FailureContext(FailureType.CRITICAL_ERROR, (Throwable)ex));
            }
        });
    }

    private void onIndexRemoved(RemoveIndexEventParameters parameters) {
        IgniteUtils.inBusyLockAsync(this.busyLock, () -> {
            this.indexBuilder.stopBuildingIndexes(parameters.indexId());
            return CompletableFutures.nullCompletedFuture();
        }).whenComplete((res, ex) -> {
            if (ex != null) {
                this.failureProcessor.process(new FailureContext(FailureType.CRITICAL_ERROR, (Throwable)ex));
            }
        });
    }

    private void onPrimaryReplicaElected(PrimaryReplicaEventParameters parameters) {
        IgniteUtils.inBusyLockAsync(this.busyLock, () -> {
            if (IndexManagementUtils.isLocalNode(this.clusterService, parameters.leaseholderId())) {
                assert (parameters.groupId() instanceof ZonePartitionId || parameters.groupId() instanceof TablePartitionId) : "Primary replica ID must be of type ZonePartitionId or TablePartitionId [groupId=" + parameters.groupId() + ", class=" + parameters.groupId().getClass().getSimpleName() + "].";
                this.primaryReplicaIds.add(parameters.groupId());
                Catalog catalog = this.catalogService.catalog(this.catalogService.latestCatalogVersion());
                if (parameters.groupId() instanceof ZonePartitionId) {
                    assert (this.nodeProperties.colocationEnabled()) : "Primary replica ID must be of type ZonePartitionId";
                    ZonePartitionId primaryReplicaId = (ZonePartitionId)parameters.groupId();
                    CatalogZoneDescriptor zoneDescriptor = catalog.zone(primaryReplicaId.zoneId());
                    if (zoneDescriptor == null) {
                        return CompletableFutures.nullCompletedFuture();
                    }
                    ArrayList<CompletionStage> indexFutures = new ArrayList<CompletionStage>();
                    for (CatalogTableDescriptor tableDescriptor : catalog.tables(zoneDescriptor.id())) {
                        CompletionStage future = this.indexManager.getMvTableStorage(parameters.causalityToken(), tableDescriptor.id()).thenCompose(mvTableStorage -> {
                            HybridTimestamp buildAttemptTimestamp = this.clockService.now();
                            return this.awaitPrimaryReplica(primaryReplicaId, buildAttemptTimestamp).thenAccept(replicaMeta -> this.tryScheduleBuildIndexesForNewPrimaryReplica(catalog, tableDescriptor, primaryReplicaId, (MvTableStorage)mvTableStorage, (ReplicaMeta)replicaMeta, buildAttemptTimestamp));
                        });
                        indexFutures.add(future);
                    }
                    return CompletableFutures.allOf(indexFutures);
                }
                TablePartitionId primaryReplicaId = (TablePartitionId)parameters.groupId();
                CatalogTableDescriptor tableDescriptor = catalog.table(primaryReplicaId.tableId());
                if (tableDescriptor == null) {
                    return CompletableFutures.nullCompletedFuture();
                }
                return this.indexManager.getMvTableStorage(parameters.causalityToken(), primaryReplicaId.tableId()).thenCompose(mvTableStorage -> {
                    HybridTimestamp buildAttemptTimestamp = this.clockService.now();
                    return this.awaitPrimaryReplica(primaryReplicaId, buildAttemptTimestamp).thenAccept(replicaMeta -> this.tryScheduleBuildIndexesForNewPrimaryReplica(catalog, tableDescriptor, primaryReplicaId, (MvTableStorage)mvTableStorage, (ReplicaMeta)replicaMeta, buildAttemptTimestamp));
                });
            }
            this.stopBuildingIndexesIfPrimaryExpired(parameters.groupId());
            return CompletableFutures.nullCompletedFuture();
        }).whenComplete((res, ex) -> {
            if (ex != null && !ExceptionUtils.hasCause(ex, NodeStoppingException.class, StorageClosedException.class)) {
                this.failureProcessor.process(new FailureContext(FailureType.CRITICAL_ERROR, (Throwable)ex));
            }
        });
    }

    private void tryScheduleBuildIndexesForNewPrimaryReplica(Catalog catalog, CatalogTableDescriptor tableDescriptor, ReplicationGroupId primaryReplicaId, MvTableStorage mvTableStorage, ReplicaMeta replicaMeta, HybridTimestamp buildAttemptTimestamp) {
        IgniteUtils.inBusyLock(this.busyLock, () -> {
            if (this.isLeaseExpired(replicaMeta, buildAttemptTimestamp)) {
                LOG.info("Lease has expired (on new primary), stopping build index process [groupId={}, localNode={}, primaryReplica={}.", primaryReplicaId, this.localNode(), replicaMeta);
                this.stopBuildingIndexesIfPrimaryExpired(primaryReplicaId);
                return;
            }
            for (CatalogIndexDescriptor indexDescriptor : catalog.indexes(tableDescriptor.id())) {
                if (indexDescriptor.status() == CatalogIndexStatus.BUILDING) {
                    this.scheduleBuildIndex(tableDescriptor.zoneId(), tableDescriptor.id(), ((PartitionGroupId)primaryReplicaId).partitionId(), indexDescriptor, mvTableStorage, IndexBuildController.enlistmentConsistencyToken(replicaMeta), this.clockService.current());
                    continue;
                }
                if (indexDescriptor.status() != CatalogIndexStatus.AVAILABLE) continue;
                this.scheduleBuildIndexAfterDisasterRecovery(tableDescriptor.zoneId(), tableDescriptor.id(), ((PartitionGroupId)primaryReplicaId).partitionId(), indexDescriptor, mvTableStorage, IndexBuildController.enlistmentConsistencyToken(replicaMeta));
            }
        });
    }

    private void tryScheduleBuildIndex(int zoneId, int tableId, int partitionId, ReplicationGroupId primaryReplicaId, CatalogIndexDescriptor indexDescriptor, MvTableStorage mvTableStorage, ReplicaMeta replicaMeta, HybridTimestamp buildAttemptTimestamp) {
        assert (!(primaryReplicaId instanceof ZonePartitionId) ? ((TablePartitionId)primaryReplicaId).tableId() == tableId && ((TablePartitionId)primaryReplicaId).partitionId() == partitionId : ((ZonePartitionId)primaryReplicaId).zoneId() == zoneId && ((ZonePartitionId)primaryReplicaId).partitionId() == partitionId) : "Primary replica identifier mismatched [zoneId=" + zoneId + ", tableId=" + tableId + ", partitionId=" + partitionId + ", primaryReplicaId=" + primaryReplicaId + ", primaryReplicaCls=" + primaryReplicaId.getClass().getSimpleName() + "].";
        IgniteUtils.inBusyLock(this.busyLock, () -> {
            if (this.isLeaseExpired(replicaMeta, buildAttemptTimestamp)) {
                LOG.info("Lease has expired, stopping build index process [groupId={}, localNode={}, primaryReplica={}.", primaryReplicaId, this.localNode(), replicaMeta);
                this.stopBuildingIndexesIfPrimaryExpired(primaryReplicaId);
                return;
            }
            this.scheduleBuildIndex(zoneId, tableId, partitionId, indexDescriptor, mvTableStorage, IndexBuildController.enlistmentConsistencyToken(replicaMeta), this.clockService.current());
        });
    }

    private void stopBuildingIndexesIfPrimaryExpired(ReplicationGroupId replicaId) {
        if (this.primaryReplicaIds.remove(replicaId)) {
            if (replicaId instanceof ZonePartitionId) {
                ZonePartitionId zoneId = (ZonePartitionId)replicaId;
                this.indexBuilder.stopBuildingZoneIndexes(zoneId.zoneId(), zoneId.partitionId());
            } else {
                TablePartitionId partitionId = (TablePartitionId)replicaId;
                this.indexBuilder.stopBuildingTableIndexes(partitionId.tableId(), partitionId.partitionId());
            }
        }
    }

    private CompletableFuture<ReplicaMeta> awaitPrimaryReplica(ReplicationGroupId replicaId, HybridTimestamp timestamp) {
        return ((CompletableFuture)this.placementDriver.awaitPrimaryReplica(replicaId, timestamp, 10L, TimeUnit.SECONDS).handle((replicaMeta, throwable) -> {
            if (throwable != null) {
                Throwable unwrapThrowable = ExceptionUtils.unwrapCause(throwable);
                if (unwrapThrowable instanceof PrimaryReplicaAwaitTimeoutException) {
                    return this.awaitPrimaryReplica(replicaId, timestamp);
                }
                return CompletableFuture.failedFuture(unwrapThrowable);
            }
            return CompletableFuture.completedFuture(replicaMeta);
        })).thenCompose(Function.identity());
    }

    private void scheduleBuildIndex(int zoneId, int tableId, int partitionId, CatalogIndexDescriptor indexDescriptor, MvTableStorage mvTableStorage, long enlistmentConsistencyToken, HybridTimestamp initialOperationTimestamp) {
        MvPartitionStorage mvPartition = IndexBuildController.mvPartitionStorage(mvTableStorage, zoneId, tableId, partitionId);
        IndexStorage indexStorage = IndexBuildController.indexStorage(mvTableStorage, partitionId, indexDescriptor);
        this.indexBuilder.scheduleBuildIndex(zoneId, tableId, partitionId, indexDescriptor.id(), indexStorage, mvPartition, this.localNode(), enlistmentConsistencyToken, initialOperationTimestamp);
    }

    private void scheduleBuildIndexAfterDisasterRecovery(int zoneId, int tableId, int partitionId, CatalogIndexDescriptor indexDescriptor, MvTableStorage mvTableStorage, long enlistmentConsistencyToken) {
        MvPartitionStorage mvPartition = IndexBuildController.mvPartitionStorage(mvTableStorage, zoneId, tableId, partitionId);
        IndexStorage indexStorage = IndexBuildController.indexStorage(mvTableStorage, partitionId, indexDescriptor);
        this.indexBuilder.scheduleBuildIndexAfterDisasterRecovery(zoneId, tableId, partitionId, indexDescriptor.id(), indexStorage, mvPartition, this.localNode(), enlistmentConsistencyToken, this.clockService.current());
    }

    private InternalClusterNode localNode() {
        return IndexManagementUtils.localNode(this.clusterService);
    }

    private boolean isLeaseExpired(ReplicaMeta replicaMeta, HybridTimestamp buildAttemptTimestamp) {
        return !IndexManagementUtils.isPrimaryReplica(replicaMeta, this.localNode(), buildAttemptTimestamp);
    }

    private static long enlistmentConsistencyToken(ReplicaMeta replicaMeta) {
        return replicaMeta.getStartTime().longValue();
    }

    private static MvPartitionStorage mvPartitionStorage(MvTableStorage mvTableStorage, int zoneId, int tableId, int partitionId) {
        MvPartitionStorage mvPartition = mvTableStorage.getMvPartition(partitionId);
        assert (mvPartition != null) : "Partition storage is missing [zoneId=" + zoneId + ", tableId=" + tableId + ", partitionId=" + partitionId + "].";
        return mvPartition;
    }

    private static IndexStorage indexStorage(MvTableStorage mvTableStorage, int partitionId, CatalogIndexDescriptor indexDescriptor) {
        IndexStorage indexStorage = mvTableStorage.getIndex(partitionId, indexDescriptor.id());
        assert (indexStorage != null) : "Index storage is missing [partitionId=" + partitionId + ", indexId=" + indexDescriptor.id() + "].";
        return indexStorage;
    }
}

