package Unicode::Collate::CJK::Zhuyin;

use 5.006;
use strict;
use warnings;

our $VERSION = '1.14';

my %u2p;
my $wt = 0x8000;

while (<DATA>) {
    last if /^__END__/;
    my @c = split;
    for my $c (@c) {
	next if !$c;
	$u2p{hex($c)} = $wt if $c !~ /-/;
	$wt++;
    }
}

sub weightZhuyin {
    my $u = shift;
    return exists $u2p{$u} ? $u2p{$u} : undef;
}

1;
__DATA__
FDD0-3105
516B 4EC8 6252 6733 7390 593F 5C9C 82AD 5CC7 67ED
75A4 54F5 5DFC 634C 7C91 7F93 8686 91DB 91DF 3B6D
8C5D 9C83 4C3E
53D0 72AE 629C 577A 59AD 62D4 70A6 7679 80C8 8307
83DD 8A59 8DCB 8EF7 98B0 58A2 9B43 9F25
628A 94AF 9200 9776
575D 5F1D 7238 57BB 8019 8DC1 9C85 9C8C 43AC 9B8A
8987 77F2 9738 58E9 705E 6B1B
5DF4 53ED 5427 7B06 7D26 7F62 9B5E 7F77
7676 5E17 62E8 6CE2 7677 73BB 525D 5265 54F1 76CB
7835 94B5 997D 7D34 7F3D 889A 88B0 83E0 7886 9262
50E0 5D93 64A5 64AD 9911 9B81 8E73 9A4B 9C4D
4EE2 4F2F 72BB 8091 9A73 5E1B 72DB 74DD 4FBC 52C3
80C9 82E9 4EB3 632C 6D61 74DF 79E1 90E3 94B9 94C2
3A8D 3DBF 3E40 3F0E 8116 8236 88AF 535A 6E24 88B9
9E41 613D 640F 733C 8467 9238 9251 998E 50F0 3B0D
717F 7254 7B94 824A 4B6F 999B 99C1 8514 8E23 92CD
9548 464F 294D0 999E 99EE 8C70 2836D 5697 61EA 3A67
3B27 7C19 894F 939B 993A 9D53 72A6 7921 9AC6 9AC9
6B02 7934 896E 946E
8DDB 7BA5 7C38
5B79 6A97 7CEA 8B52 8617
2E8A 535C 5575 8421 818A
6300 63B0 64D8
767D
767E 4F70 67CF 6822 636D 74F8 7CA8 7D54 6446 64FA
896C
5E8D 62DD 8D25 62DC 6557 7308 7A17 86FD 7CBA 3525
8D01 97DB
7AE1 85AD
5351 676F 9642 76C3 686E 60B2 63F9 6911 7891 7986
9E4E 9303 85E3 9D6F
5317 3933 9273
8D1D 5B5B 72C8 8C9D 5907 6601 726C 90B6 80CC 82DD
94A1 4FFB 500D 36DD 6096 72FD 90E5 505D 5079 3AF2
6896 73FC 88AB 5099 60EB 7119 7432 8EF0 8F88 9101
50C3 6102 3ED7 789A 3E22 7295 84D3 8A96 9781 8919
8F29 92C7 9AB3 618A 7CD2 97B4 943E
5457 5504 7999
52F9 5305 5B62 67B9 80DE 82DE 7B23 7172 9F85 8912
8554 95C1 8943 9F59
7A87 5AD1 96F9 420F 8584 2688A 3FFA
5B9D 6009 9971 4FDD 9E28 5BB2 73E4 5821 5822 5AAC
5BDA 8446 98FD 99C2 9CF5 7DE5 8913 9D07 8CF2 4B4B
5BF3 5BF6 28C26 974C 4D10 28C3B
52FD 62A5 62B1 8C79 8DB5 94C7 86AB 888C 5831 83E2
924B 9C8D 9764 66B4 9AB2 9AF1 8663 9B91 5124 66D3
7206 5FC1 9464 9E14
4F68 85F5
6273 653D 73ED 822C 9881 6591 642C 6592 9812 7622
9CFB 4232 878C 8929 764D 8FAC
5742 5C85 962A 6604 677F 7248 74EA 94A3 7C84 8228
9211 8742 9B6C 95C6
529E 534A 4F34 5762 59C5 6011 62CC 7ECA 67C8 79DA
6E74 7D46 9261 977D 8FA6 74E3
626E 8781
5954 6CCD 8D32 681F 7287 951B 931B
672C 82EF 5959 755A 7FC9 694D
574B 574C 5034 6379 6873 6E00 7B28 26BC0 9029 64AA
7356 8F3D
90A6 57B9 5E2E 6360 6D5C 6886 90AB 5E47 5E5A 7E0D
5E6B 97A4
7ED1 7D81 699C 7253 8180 9AC8
3B4B 73A4 868C 508D 68D2 68D3 8C24 585D 6412 7A16
84A1 872F 78C5 9551 8255 8B17 938A
4F3B 794A 595F 5D29 7D63 958D 50B0 5D6D 75ED 5623
7DB3
752D
57C4 57F2 7EF7 7423 742B 83F6 7E43 979B
6CF5 8FF8 902C 5874 955A 750F 295F4 4B70 8E66 93F0
882F
63FC
5C44 506A 6BF4 6945 903C 8C4D 8795 9CBE 939E 9D56
9C0F
8378 9F3B
5315 6BD4 3832 5936 673C 4F4A 5421 59A3 6C98 7595
5F7C 67C0 79D5 4FFE 7B14 7C83 822D 555A 3A8F 7B46
7B84 805B 9119 8C8F
5E01 5FC5 6BD5 95ED 4F56 3629 5752 5E87 8BD0 3620
59BC 602D 6036 39D9 6788 7540 90B2 54D4 67F2 6BD6
73CC 75AA 82FE 6BD9 72F4 7B13 7C8A 835C 94CB 965B
5A62 5EB3 655D 6890 7562 8890 9587 9589 581B 5F3B
5F3C 610A 610E 6E62 7695 7B5A 8406 8A56 8CB1 8CC1
8D51 6ED7 714F 75F9 75FA 7764 8177 870C 8DF8 924D
959F 98F6 55F6 5E63 5F0A 5F43 6EED 78A7 7B85 7B86
7DBC 84D6 88E8 999D 6F77 719A 7358 7359 3EEB 84FD
853D 480B 912A 99DC 9AF2 58C1 5B16 5EE6 3D65 7BE6
7F7C 89F1 9B85 6583 6FDE 7BF3 7E2A 81C2 859C 907F
9946 5970 74A7 8E55 9128 9AC0 7E74 895E 93CE 4557
8963 8E83 8E84 97B8 97E0 8D14 9434 9A46 9B53 9DDD
9DE9 9F0A
5302 841E 5E64 5B36 8945
618B 87DE 9CD6 9C49 9F08 864C 9F9E
5225 522B 5487 43DF 8382 86C2 5FB6 8952 8E69
762A 765F 3FDC
5F46 4318
706C 6753 6807 98D1 9A89 9ADF 6DF2 5F6A 730B 813F
98A9 5882 5E56 647D 6EEE 98AE 9AA0 6A19 719B 8198
8508 762D 78E6 9556 98D9 98DA 5126 98B7 700C 8B24
7202 81D5 85E8 8D06 93E2 7A6E 9573 98C6 98C7 98C8
9A43 9463 4BBD 9A6B
8868 5A4A 88F1 8AD8 9336 6AA6 893E
4FF5 9CD4 9C3E
98CA
8FB9 8FBA 782D 7B3E 63D9 7335 7F16 7178 7251 7502
7BAF 7DE8 8759 9089 937D 9CCA 97AD 908A 9BFE 9BFF
7C69
8D2C 6241 7A86 533E 60FC 8CB6 8439 78A5 7A28 7CC4
890A 9D18 85CA
535E 5F01 5325 5FED 6283 6C73 6C74 91C6 53D8 73A3
82C4 4FBF 5909 662A 3B53 898D 5FA7 7F0F 904D 959E
8FA1 7DF6 8251 8FA7 8FA8 8FA9 8FAB 8FAE 8FAF 8B8A
5CC5 709E
6C43 90A0 73A2 780F 5BBE 5F6C 68B9 50A7 658C 6915
6EE8 3EDE 7F24 69DF 7478 8C69 8CD3 8CD4 9554 5110
6FD2 6FF1 8668 8C73 6AB3 74B8 7015 9726 7E7D 944C
986E
6448 6BA1 8191 9AE9 64EF 9B02 6BAF 81CF 9ACC 9B13
9AD5 9B22
6C1E 6FF5
51AB 4ECC 4ED2 6C37 51B0 5175 63A4
4E19 6032 62A6 79C9 90B4 9643 661E 663A 67C4 70B3
82EA 997C 772A 7A89 3A00 86C3 6452 7980 7A1F 9235
927C 9905 9920 979E
5E76 34C8 4E26 4F75 5E77 5EB0 5002 6824 75C5 7ADD
504B 50A1 5BCE 68C5 8A81 9BA9 9750
57AA 9786 92F2
5CEC 5EAF 6661 900B 923D 8AA7
9CEA 8F50 91AD
535F 248F0 8865 54FA 6355 55B8 88DC 9D4F
4E0D 5E03 4F48 5425 6B65 5498 3635 6016 62AA 6B68
6B69 3CCD 67E8 949A 52CF 57D4 57D7 6091 6357 94B8
57E0 8379 90E8 74FF 8E04 8500 90F6 9914 9922 7BF0
7C3F
FDD0-3106
5991 25419 241B5 7685 8DB4 8225 556A 8469
6777 722C 63B1 7436 7B62 6F56
5E0A 5E15 6015 8899
948B 5761 5CA5 6CCA 9887 6E8C 9255 9817 93FA
5A46 5619 8522 9131 76A4 8B08 6AC7
53F5 5C00 94B7 7B38 99CA
5CB6 7087 6540 6622 6D26 73C0 8FEB 70DE 7834 7836
91D9 7C95 84AA 9B44 9197 4A96
6CFC 6872 6F51
62CD
4FF3 5F98 6392 7305 68D1 724C 8F2B 7C30 7C32 72A4
5EF9
54CC 6D3E 3B5B 6E43 848E 9383
5478 36B0 600C 80A7 67F8 80DA 8843 9185
962B 57F9 966A 6BF0 8D54 952B 88F4 88F5 8CE0 99CD
4FD6
4F02 6C9B 4F69 5E14 59F5 65BE 65C6 6D7F 73EE 914D
7B29 4283 8F94 99B7 5D8F 9708 8F61
84DC
629B 62CB 812C
5228 5486 5789 5E96 72CD 70B0 722E 530F 888D 8EF3
9784 9E83 9E85
8DD1
5945 6CE1 70AE 75B1 76B0 7832 9EAD 791F 792E
8422 891C
5256 5A1D
6294 6299 634A 638A 88D2 7B81 9307
5485 54E3 5A44 7283 5ECD
3434 7705 7819 7568 6F58 6500
723F 6D00 76D8 8DD8 5ABB 5E4B 642B 69C3 84B0 76E4
78D0 7E0F 78FB 8E52 23066 700A 87E0 8E63 939C 97B6
51B8 5224 6C9C 62DA 6CEE 708D 53DB 7249 76FC 7554
8041 88A2 8A4A 6EBF 9816 92EC 897B 947B
9D65
55B7 5674 6B55
74EB 76C6 6E53 8450
5460 7FF8
55AF
4E53 6C97 80EE 96F1 6EC2 8196 4A26 9736
5390 5E9E 5396 65C1 9004 823D 5ACE 5FAC 249ED 8783
9CD1 9F8E 9F90
55D9 802A 89AB
7090 80A8 80D6
5309 36C1 6026 62A8 6072 39F8 7830 6888 70F9 7851
8EEF 959B 6F30 562D 6F8E 78DE
8283 670B 7AFC 5017 6337 580B 5F38 8391 5F6D 68DA
6916 787C 7A1D 9E4F 5873 6189 69F0 6A25 71A2 84EC
28083 8F23 7BE3 81A8 930B 97F8 9AFC 87DA 87DB 9B05
4D36 97FC 9D6C 9A2F 7E84 9B14 945D
6367 6DCE 768F 527B
63BD 692A 78B0 8E2B
7BF7
4E15 4F13 4F3E 6279 7EB0 576F 62AB 62B7 708B 72C9
90B3 7812 20731 6082 79DB 79E0 7D15 94CD 65C7 7FCD
801A 8C7E 9208 921A 9239 925F 9294 5288 78C7 99D3
9AEC 567C 930D 9B7E 9B8D 61B5 7914 7915 9739
76AE 34DF 9630 5CAF 6787 6BDE 72D3 80B6 8298 6BD7
6BD8 75B2 868D 5564 57E4 5D25 26A2D 86BD 86BE 8C7C
2821A 90EB 9674 7137 7435 813E 8157 9C8F 7F74 818D
8731 9B6E 58C0 7BFA 87B7 8C94 9D67 7F86 6707 9F19
5339 5E80 758B 4EF3 572E 82C9 8134 75DE 92A2 8AC0
9D04 64D7 567D 7656 4C26 56AD
5C41 6DE0 6E12 63CA 91FD 5AB2 5AD3 7765 8F9F 6F4E
7A2B 50FB 6FBC 568A 7513 4444 7588 8B6C 95E2 9DFF
9E0A
698C
6C15 6487 22CC2 6486 66BC 77A5
4E3F 82E4 9405
5AF3
527D 6153 7F25 98D8 65DA 7FF2 87B5 72A5 98C3 98C4
9B52
5AD6 74E2 7AC2 85B8 95DD
6B8D 5F6F 779F 7BFB 7E39 91A5 76AB 9860
7968 50C4 52E1 560C 5FB1 6F02 3B13
56E8 504F 5AA5 728F 7BC7 7FE9 9342 9DA3
9A88 80FC 8141 6944 6969 8CC6 8DF0 8ADA 8E41 99E2
9ABF 9A08
8991 8C1D 8CB5 8ADE
7247 9A97 9A17 9A19
9B78
59D8 62FC 3866 7917 7A66 99AA 9A5E
73AD 8D2B 5A26 8CA7 7415 5AD4 9891 983B 5B2A 7371
85B2 56AC 77C9 8819 98A6 9870
54C1 6980
725D 6C56 8058
4E52 7539 4FDC 5A09 6D84 782F 8060 8275 7AEE 9829
29A4D
5E73 8BC4 51ED 546F 576A 6CD9 5C4F 5E21 67B0 6D34
73B6 80D3 82F9 90F1 3EB8 3EC2 8353 5C5B 5E32 6DDC
74F6 86B2 5E48 7129 7F3E 840D 86E2 8A55 7501 84F1
8EFF 9C86 51F4 4202 617F 7BB3 8F27 6191 9B83 6A98
7C08 860B
5CBC 5840
4EC6 6534 6251 9660 5657 64B2 6F7D 64C8 9BC6
3EAA 530D 812F 8386 83D0 83E9 8461 50D5 84B1 84B2
917A 58A3 735B 749E 6FEE 77A8 7A59 423B 9564 258C8
7E80 8965 93F7
5724 6734 5703 6D66 70F3 666E 6EA5 8C31 8AE9 6A38
6C06 6A8F 9568 8B5C 8E7C 9420
94FA 8216 8217 92EA 3B25 7011 66DD
5DEC 5DED 99C7 8D0C
FDD0-3107
5463
5988 5B56 5ABD 5B24 5B37
9EBB 75F2 7298 8534 87C7
9A6C 3437 739B 7801 8682 99AC 6EA4 746A 78BC 879E
93B7 9C22 9DCC
72B8 6769 7943 9581 9A82 551B 508C 7341 561C 3A38
69AA 7770 79A1 7F75 372B 99E1 7923 9B15
4E87 5417 55CE 561B 5AF2 9064 87C6
6478
8C1F 998D 5AEB 9EBD 6469 6479 6A21 819C 6A45 78E8
7CE2 56A4 64F5 8B28 56A9 56B0 9943 8611 5298 9ACD
9B54 40FA 995D
62B9 61E1 4A4B
672B 52B0 573D 59BA 5E13 6B7E 6B7F 6B81 6CAB 248FB
5E1E 6629 67BA 8309 964C 551C 768C 771C 773F 781E
79E3 773D 7C96 7D48 8388 83AB 6E50 86E8 8C83 8C8A
55FC 587B 5BDE 6F20 734F 84E6 9286 977A 5AFC 66AF
3C73 9ED9 7790 9546 9B69 58A8 763C 7799 9ED8 700E
87D4 8B29 8C98 7205 85E6 93CC 7933 7E86 9A40 8031
5E85 603D 5C1B 9B79 9EBF
4E48 9EBC 5692 569C 6FF9 7666
57CB 3725 85B6 973E
4E70 836C 8CB7 562A 8552 9DF6
52A2 4F45 58F2 8FC8 9EA6 5356 8109 8108 9EA5 8847
52F1 8CE3 9081 9721 9722 2A133
5445 5746 6C92 6CA1 679A 73AB 6802 7709 82FA 5A12
8104 6885 73FB 8122 8393 5833 5A92 5D4B 6E44 6E48
7338 7742 90FF 6963 6973 7164 7442 847F 587A 69D1
7996 9176 9545 9E5B 92C2 9709 7A48 5FBE 9387 77C0
6517 9DA5 862A 9EF4
6BCE 6BCF 51C2 7F8E 6334 6D7C 5A84 5D44 6E3C 5ABA
815C 9541 5B0D 71D8 9382 9EE3
59B9 62BA 6CAC 65C0 6627 3B51 771B 7959 8882 5A9A
5BD0 75D7 8DCA 715D 9B3D 7778 97CE 7BC3 875E 9B45
8EBE
732B 8C93
6BDB 77DB 6786 7266 8305 8306 65C4 7F5E 515E 8EDE
9155 5825 6E35 28968 951A 9AE6 5AF9 6C02 729B 8765
9AF3 87CA 9328 9D9C
5187 536F 5918 4E6E 623C 36B9 5CC1 6CD6 6634 94C6
7B37 84E9
5183 7683 5190 82BC 5192 67D5 770A 8302 8D38 8004
88A4 8992 5AA2 5E3D 8CBF 6117 6693 6959 6BF7 7441
843A 7780 8C8C 911A 8750 912E 61CB
54DE
725F 4F94 52BA 3B4C 6048 6D20 7738 8C0B 86D1 7F2A
8E0E 927E 8B00 77B4 7E46 936A 9D3E 9EB0
67D0 26CD1
5ADA 989F
59CF 6097 34BC 86EE 50C8 8C29 6172 9992 6A20 7792
779E 9794 25C91 8B3E 9945 9CD7 9862 9B17 9B18 9C3B
883B
5C58 6E80 774C 6EE1 6EFF 87A8 87CE 8954 93CB 77D5
66FC 3B05 5881 5E54 6162 6471 6F2B 734C 7F26 9124
69FE 71B3 8504 8513 3862 6FB7 9558 7E35 93DD 2914D
4555 2414E
8630
95E8 626A 73A7 9494 9580 9585 636B 83DB 748A 9346
4EB9 864B
95F7 7116 60B6 66AA 71DC 61D1 61E3
4EEC 5011 691A
7264
5402 5FD9 6C52 9099 5C28 6757 6767 8292 6C13 76F2
7B00 54E4 5A0F 5EAC 3936 607E 6D5D 72F5 832B 727B
3ECA 91EF 94D3 75DD 786D 86D6 420D 23677 92E9 99F9
29DF6
83BD 7865 833B 83BE 58FE 6F2D 3B12 87D2 880E
753F 867B 51A1 8394 840C 8420 4004 76DF 8499 8771
511A 6A57 750D 77A2 8544 9133 5E6A 61DE 66DA 6FDB
9138 6726 6AAC 6C0B 4443 4445 9BCD 77C7 791E 9E72
8268 8609 77D2 973F 9740 995B 986D 9F06 9E0F
52D0 731B 74FE 4045 9530 824B 8722 9333 61DC 7374
61F5 9BED 8813
5B5F 20D7C 68A6 6E95 5922 5923 44DD 9725 3734
63B9 64DD
54AA 772F 7787
519E 5F25 7F59 7962 8FF7 7315 8C1C 8A78 84BE 5F4C
64DF 7CDC 7E3B 8B0E 919A 9E8A 9E8B 79B0 9761 3737
7030 737C 9E9B 6202 6520 74D5 957E 7222 4577 863C
91BE 91BF 9E0D 91C4
7C73 8288 4F8E 6CB5 7F8B 5F2D 6D23 6549 772B 8112
6E33 845E 92A4 851D 6FD4 5B4A 7056
5196 7CF8 6C68 6C95 5B93 6CCC 89C5 5CDA 5BBB 7955
79D8 5BC6 6DE7 6DFF 8993 8994 5E42 8C27 5853 5E4E
3A20 899B 5627 6993 6EF5 6F1E 7190 871C 6A12 8524
9F0F 51AA 5E66 6FD7 8B10 6AC1 7C1A 85CC 7F83
4E5C 5400 54A9 54F6 5B6D
706D 70D5 8995 6423 6EC5 8511 9D13 7BFE 858E 5E6D
61F1 6AD7 881B 884A 9456 9C74
55B5
82D7 3464 5A8C 63CF 9E4B 7784 7DE2 9D93 9C59
676A 7707 79D2 6DFC 6E3A 7F08 7BCE 7DF2 85D0 9088
5999 5E99 7385 7AD7 5EBF 5EDF
8C2C 8B2C
5B80 8287 7720 5A42 7EF5 5A94 68C9 7DBF 7DDC 81F1
8752 5B35 6AB0 6ACB 77C8 77CA 77CF
4E0F 6C45 514D 6C94 9EFE 52C9 7704 5A29 3743 506D
5195 52D4 6E11 5595 6110 6E4E 7F05 7D7B 817C 8442
9EFD 3EF0 7DEC 9EAB 6FA0 9BB8
9763 9762 7CC6 9EAA 9EBA 9EB5
6C11 59C4 5CB7 5FDE 600B 65FB 65FC 73C9 76FF 7807
82E0 7F60 5D0F 636A 7418 7F17 656F 7449 75FB 7888
9231 7DCD 4AB8 7DE1 9309 9D16 9372
76BF 51BA 5221 95F5 62BF 6CEF 52C4 6543 95FD 60AF
654F 7B22 60FD 6E63 9594 610D 3B06 668B 95A9 50F6
615C 61AB 6F63 7C22 2704B 9CD8 8820 9C35 2A134
578A 7B3D
540D 660E 9E23 6D3A 7700 51A5 8317 6719 7733 94ED
5AC7 6E9F 733D 910D 669D 69A0 84C2 9298 9CF4 7791
879F 89AD 4AE4
4F72 59F3 51D5 614F 9169
547D 6927 8A7A
63B5
6BEA 58B2 6C01
6BCD 4EA9 7261 5776 59C6 5CD4 7273 7546 7552 80DF
755D 755E 782A 756E 27FF9 20E96 9267 8E07
6728 4EEB 6730 76EE 6C90 72C7 7091 7267 82DC 6BE3
869E 94BC 83AF 96EE 52DF 3708 6958 7766 926C 5893
5E55 5E59 6154 6155 66AE 8252 9702 7A46 7E38 97AA
51E9 62C7
FDD0-3108
53D1 6CB7 767A 50A0 767C 5F42 9166 91B1
4E4F 4F10 59C2 57A1 6D4C 75BA 7F5A 9600 6830 781D
8337 7B4F 7782 7F70 95A5 7F78 6A43 85C5
4F71 6CD5 704B
73D0 743A 9AEA 9AEE 855F 27B0B
9345
4ECF 5772
68BB
98DE 5983 975E 98DB 5561 5A53 6E04 7EEF 291EB 6249
7306 83F2 975F 7DCB 871A 88F6 970F 9CB1 9925 99A1
9A11 9A1B 98DD
80A5 6DDD 8153 4208 8730 87E6
670F 532A 8BFD 595C 60B1 6590 68D0 69A7 7FE1 8AB9
7BDA 855C 2913C
5420 5E9F 676E 6CB8 72D2 80BA 82BE 6632 80C7 8D39
4FF7 5255 539E 75BF 5C5D 966B 5EC3 3E43 8409 8CBB
75F1 9544 5EE2 66CA 4935 7648 9F23 6FF7 4C01 6AE0
9BE1 9428 9745
5A54 6683
7D11 88E6
7F36 5426 599A 7F39 7F3B 6B95 96EC 9D00
5E06 8A09 756A 52EB 5643 5B0F 5E61 61A3 65D9 8543
65DB 7E59 7FFB 85E9 8F53 98BF 7C53 98DC 9C55
51E1 51E2 51E3 3836 5FDB 674B 77FE 7C75 9492 67C9
70E6 8227 7B32 68E5 6E22 7169 7DD0 58A6 6A0A 6A4E
71D4 74A0 81B0 7E41 85A0 7FB3 894E 8E6F 702A 703F
792C 9407 9422 8629 881C 9DED
53CD 6255 8FD4 484A 91E9
201A9 6C3E 72AF 597F 6C4E 996D 6CDB 8D29 7548 8303
4000 8ED3 5A4F 68B5 3D00 76D5 7B35 8CA9 8EEC 98EF
98F0 6EFC 5B0E 7BC4
8224
5206 5429 5E09 7EB7 6610 6C1B 82AC 54DB 517A 7D1B
7FC2 886F 515D 8A1C 915A 68FB 9216 96F0 3B1F 6706
71D3 9934 9959
575F 59A2 5C8E 6C7E 670C 678C 7083 80A6 7F92 86A0
86A1 68A4 68FC 711A 999A 84B6 58B3 5E69 6FC6 9B75
6A68 71CC 8561 8C6E 96AB 9F22 7FB5 9F16 8C76 8F52
943C 99A9 9EC2
7C89 9EFA
4EFD 5F05 594B 5FFF 79CE 507E 6124 7CAA 50E8 61A4
596E 81B9 7CDE 9CBC 7035 9C5D
7AD5 8EAE
531A 65B9 6C78 90A1 678B 7265 82B3 94AB 8684 6DD3
9201 9D0B
59A8 9632 623F 80AA 57C5 9C82 9B74 9C1F
4EFF 8BBF 5F77 7EBA 6609 6618 74EC 7706 5023 65CA
7D21 822B 8A2A 9AE3 9DAD
653E 8DBD
574A 580F 933A
4E30 98CE 4EF9 51E8 51EC 59A6 6CA3 6CA8 51EE 67AB
5C01 75AF 76FD 781C 98A8 36D4 5CEF 5CF0 44A0 5051
687B 70FD 5D36 7326 950B 6953 728E 8451 8702 760B
78B8 50FC 7BC8 92D2 9137 6A92 95CF 8C50 93E0 34A5
5BF7 3826 7043 9146 973B 8634 882D 974A 98CC 9EB7
51AF 5906 6340 6D72 9022 5838 99AE 7D98 8242 3980
6453 6F28 3D6F
8BBD 552A 8982 8AF7
51E4 5949 752E 4FF8 6E57 7128 7148 7F1D 8D57 9CEF
9CF3 9D0C 8CF5 7E2B
7412 6E84 93BD 8615
8985
4F15 544B 598B 909E 59C7 739E 80A4 6024 67CE 7806
57BA 5A10 5C03 8342 886D 65C9 3B6A 7D28 8374 8DBA
9EB8 75E1 7A03 8DD7 9207 7B5F 7D92 5B75 8C67 911C
6577 819A 9CFA 9EA9 7CD0 9EAC 9EB1 61EF
4E40 5DFF 5F17 4F0F 51EB 7536 4F5B 51B9 521C 5B5A
6276 5488 5CAA 5F7F 602B 62C2 670D 678E 6CED 7EC2
7ECB 8299 82A3 4FD8 5798 3A84 67EB 6C1F 6D11 70A5
73B8 7549 7550 7F58 82FB 8300 97E8 54F9 683F 6D6E
7829 7953 832F 86A8 90DB 5310 6874 6DAA 70F0 7408
7B26 7B30 7D31 7D3C 7FC7 8274 83A9 8659 5E45 68F4
7D65 7F66 83D4 7CB0 7D8D 8240 844D 8709 8F90 9258
925C 98AB 9CE7 6991 798F 7A2A 7B99 97CD 5E5E 6F93
8760 9AF4 8AE8 8E3E 8F3B 9B84 9D14 7641 29732 9EFB
8946 9D69 9D9D
5452 629A 4E76 5E9C 5F23 62CA 65A7 4FCC 4FDB 80D5
9CEC 4FEF 90D9 91DC 91E1 636C 8F85 7124 76D9 8151
6ECF 8705 42E8 8150 8F14 5638 64A8 64AB 982B 9B34
7C20 9EFC
2ECF 2ED6 961D 7236 8BA3 4ED8 5987 8D1F 577F 3914
7ACE 961C 9644 9A78 590D 5CCA 8A03 8CA0 8D74 3931
7954 86A5 965A 5069 51A8 526F 5A66 86B9 889D 5A8D
5BCC 5FA9 3DC6 79FF 86D7 8A42 8D4B 5711 6931 7F1A
8179 842F 9C8B 8D59 4B7B 3B3C 7DEE 875C 876E 8907
8914 8CE6 99D9 5B14 7E1B 8567 8F39 9B92 8CFB 9351
9362 9CC6 8986 99A5 4600 9C12 204D7
592B 752B 5490 915C 5085 6928 88B1 8984 79A3 9BB2
FDD0-3109
5491 54D2 8037 8345 7B1A 55D2 642D 2143F 6498 5660
265BF 8921 939D
8FBE 547E 59B2 601B 6C93 8FD6 709F 7F8D 7557 7563
7B2A 8359 5273 5312 7B54 8A5A 9039 9054 9618 977C
9791 8598 9389 87FD 8E82 943D 97C3 9F96 9F98
6253
5927 6C4F 7714
57AF 58B6 7629 71F5 7E68
561A
392B 6074 6DC2 60EA 68CF 951D 5FB3 5FB7 9340
5730 7684 5F97 8126
5446 5454 7343 61DB
6B79 50A3 902E
3432 4EE3 8F6A 20C3A 5788 5CB1 5E12 7519 7ED0 9A80
5E26 5F85 6020 67CB 6B86 73B3 8D37 8FE8 5E2F 8ED1
57ED 5E36 7D3F 888B 8EDA 3ED6 8CB8 8EE9 7447 5ED7
4206 53C7 3BC2 7DFF 66C3 9D0F 6234 26102 825C 9EDB
7C24 8E5B 703B 9734 9EF1 8976 9746
9B98
2E88 5200 5202 53E8 5FC9 6737 6C18 8220 91D6 9C7D
9B5B
636F
5BFC 5C9B 5CF6 6363 7977 6417 7982 3800 5D8B 5D8C
3A36 969D 5C0E 58D4 5DB9 64E3 8E48 96AF 79B1
5230 5012 60BC 7118 76D7 76DC 83FF 9053 7A32 7B8C
7FE2 7A3B 885C 5675 885F 6AA4 71FE 4326 7FFF 8EC7
74D9 7E9B
5C76 9666 6921 69DD
543A 5517 515C 90FD 5160 8538 6A77 7BFC
6296 9627 6793 67A1 551E 86AA 9661 9204
6597 8C46 6D62 90D6 997E 9B25 68AA 6BED 8130 8373
9017 9158 75D8 9597 7AA6 4B26 9B26 9916 65A3 2109D
95D8 4546 7AC7 9B2A 9B2C 9B2D
4E67 8254
4E39 5989 5355 62C5 5358 7708 7803 803C 803D 449F
8043 8EAD 90F8 55AE 5A85 6B9A 7605 5330 7BAA 9815
510B 52EF 891D 9132 64D4 6BAB 7649 7514 7C1E 894C
8078
4F14 5210 628C 73AC 74ED 80C6 75B8 7D1E 8874 63B8
8D55 4EB6 64A2 64A3 6FB8 9ED5 81BD 9EEE
65E6 4F46 5E0E 6C8A 72DA 8BDE 67E6 758D 5556 5557
5F39 60EE 6DE1 86CB 557F 5F3E 6C2E 8145 840F 89DB
35D6 7A9E 50E4 8711 99BE 9AE7 563E 5649 5F48 619A
8A95 61BA 66BA 6FB9 84DE 9D20 79AB 99F3 765A 56AA
7E75 8D09 972E 994F 4A35
6CF9
6265 627D
5F53 73F0 7B5C 88C6 7576 5679 6FA2 74AB 7C39 8261
87F7 8960
6321 515A 8C20 64CB 8B61 9EE8 6529 7059 6B13 8B9C
6C39 51FC 5735 5B95 7800 57B1 6863 8361 5A78 6113
83EA 5D63 74FD 903F 96FC 78AD 5105 6F52 778A 8DA4
58CB 8569 6A94 7497 76EA 7911 7C1C 862F 95E3
94DB 943A
706F 767B 8C4B 5654 5B01 71C8 7492 7AF3 7C26 89B4
8E6C
6729 7B49 6225
9093 51F3 58B1 5D9D 9127 96A5 77AA 78F4 956B 6AC8
9419
8260
6C10 4EFE 4F4E 5943 5F7D 7F9D 889B 5824 8D86 9684
6EF4 6A00 955D 78FE 9349 97AE
72C4 5EF8 7C74 82D6 8FEA 5519 654C 6DA4 6891 7B1B
837B 89CC 976E 99B0 9AE2 5600 5AE1 6ECC 7FDF 9814
6575 850B 8510 4A00 5681 7BF4 4A24 8C74 8E62 9B04
85E1 93D1 42AE 7CF4 89BF 9E10
538E 5758 8BCB 5467 5E95 5F24 62B5 62DE 90B8 963A
67E2 7274 830B 7825 57DE 638B 3B7D 83E7 89DD 8A46
8EE7 805C 9AB6
5754 5F1F 65F3 6755 7393 601F 4FE4 5E1D 57CA 5A23
5059 5547 5572 688A 710D 73F6 7731 7B2C 8C1B 9012
9013 91F1 5A82 68E3 6E27 7747 7976 7F14 83C2 50C0
8163 8482 926A 5891 6455 78B2 7998 8743 905E 58AC
6178 7DE0 8515 9070 5DB3 750B 8AE6 8E36 87AE
9BF3
55F2
7239 8DCC 893A
57A4 5CCC 604E 6315 6633 7ED6 80C5 82F5 8FED 74DE
7723 621C 43F2 8C0D 558B 581E 60F5 63F2 7573 7D70
800B 81F7 8A44 8D83 957B 53E0 6B9C 7243 7252 5D7D
789F 8728 8253 8776 890B 22DA3 8ADC 8E40 9CBD 3B2A
66E1 7589 9C08 758A 6C0E
54CB 800A 7730
5E49 7582
5201 53FC 6C48 866D 51CB 595D 5F34 5F6B 86C1 7431
8C82 7889 9CED 6BA6 7797 96D5 9B89 9CB7 9F26 9BDB
9D70
625A 5C4C
5F14 4F04 540A 9493 7A8E 8A0B 8C03 6389 91E3 94DE
94EB 7AE8 92B1 96FF 9B61 84E7 8ABF 7639 7AB5 92FD
85CB 9443
7C13
4E1F 4E22 20C8D 94E5 92A9
7538 6541 20DA7 6382 508E 53A7 5D6E 6EC7 69C7 69D9
7628 98A0 8E4E 5DC5 985A 985B 766B 5DD3 5DD4 6527
7672 9F7B
5178 594C 70B9 5A70 7320 4344 655F 8DD5 7898 84A7
8E2E 8547 9EDE 56B8
7535 4F43 576B 5E97 963D 57AB 6242 73B7 94BF 5A5D
60E6 6DC0 5960 7414 6BBF 8714 96FB 588A 58C2 6A42
6A5D 6FB1 975B 24A71 765C 7C1F 9A54
6923
4E01 4EC3 53EE 5E04 738E 7594 76EF 9489 8035 8670
914A 91D8 976A
5975 9876 9802 3AC0 9F0E 5D7F 9F11 6FCE 85A1 9424
8BA2 5FCA 9964 77F4 5B9A 8A02 98E3 5576 94E4 6917
815A 7887 952D 78A0 874A 92CC 9320 78F8 9841
8423 8062
53BE 21C2A 5262 9607 7763 561F 918F 95CD
6BD2 72EC 6D9C 8BFB 6E0E 691F 3C69 724D 728A 78A1
88FB 8AAD 8773 7368 9316 51DF 5335 5B3B 7006 6ADD
6BB0 7258 72A2 74C4 76BE 9A33 9EE9 8B80 8C44 8D15
97E3 945F 9AD1 97C7 97E5 9EF7 8B9F
7B03 5835 5E3E 8D4C 743D 7779 89A9 8CED 7BE4
5992 675C 809A 828F 59AC 5EA6 79FA 8370 6E21 976F
9540 29D5A 8799 6BAC 934D 7C35 8827 8839
591A 591B 5484 54C6 7553 525F 5D1C 6387 6560 6BF2
88F0 5689
593A 94CE 526B 6553 655A 55A5 60B3 656A 75E5 922C
596A 51D9 8E31 9BB5 9438
6736 54DA 579B 579C 6305 6306 57F5 7F0D 692F 8D93
8EB1 8EB2 7D9E 492A 619C 4EB8 937A 8EC3 56B2 5972
5234 5241 9973 5C2E 67C1 67EE 70A8 964A 964F 6857
5815 8235 60F0 8DE2 8DE5 8DFA 98FF 3EE7 58AE 58AF
5D9E 9D7D
6735 67A4
5796 5806 5860 5D5F 75FD 78D3 9D2D 941C
3A03 9827
5BF9 961F 514A 514C 5151 5BFE 603C 794B 966E 968A
7893 41CF 7D90 5C0D 619E 619D 6FE7 9566 61DF 3B23
85B1 7029 8B48 8968 9413
8011 5073 526C 5A8F 7AEF 890D 9374
22B43 77ED
6BB5 65AD 5845 7F0E 6934 7145 7456 8176 846E 78AB
953B 7DDE 6BC8 7C16 935B 65B7 8E96 7C6A
5428 60C7 6566 8733 58A9 58AA 64B4 7364 5678 6489
6A54 729C 7905 453B 8E72 8E7E 9A50
233B4 76F9 8DB8 8E89
4F05 56E4 5E89 6C8C 7096 76FE 7818 949D 9007 987F
920D 696F 9041 9813 6F61 906F 71C9 8E32
78B7
4E1C 51AC 549A 36B5 5CBD 6771 6638 6C21 82F3 5032
9E2B 57EC 5A3B 5D20 5D2C 6DB7 7B17 5F9A 6C2D 83C4
29093 8740 9D24 9F15 9BDF 9D87
3A02 8463 58A5 7BBD 8ACC 5B1E 23FC5 856B 61C2
52A8 51BB 4F97 578C 59DB 5CD2 606B 630F 680B 6D1E
80E8 51CD 6219 80F4 8FF5 52D5 7850 68DF 6E69 7D67
8156 50CD 99E7 9718
9B97 9DAB
FDD0-310A
4ED6 5B83 5979 7260 7942 94CA 8DBF 584C 6EBB 6999
891F 5683 95E7
8E79
5854 6E9A 5896 35F3 736D 9CCE 737A 9C28
4EA3 62D3 631E 72E7 95FC 20D32 36E5 5D09 6DBE 3B7C
6428 40BF 8DF6 69BB 6BFE 420B 905D 9062 3493 79A2
8ABB 8E0F 64BB 6FBE 9314 6A7D 6FCC 8E4B 979C 9B99
3FF9 95D2 97B3 56BA 8B76 95E5 2A601 8EA2
4FA4 549C
5FD1 5FD2 7279 8CA3 86AE 94FD 615D 92F1 87A3 87D8
56FC 5B61 80CE
51AD 53F0 65F2 576E 62AC 90B0 67B1 70B1 70B2 82D4
83ED 8DC6 3B03 9C90 7B88 81FA 98B1 421A 99D8 5113
9B90 5B2F 64E1 3E00 6AAF 85B9 7C49
592A 5933 5FF2 6C70 6001 80BD 949B 6CF0 8226 48ED
915E 9226 6E99 614B 71E4
7C8F
5932 5F22 6D9B 7EE6 638F 8A5C 5ACD 5E4D 6146 642F
6ED4 7D5B 69C4 746B 97EC 98F8 4231 7E1A 7E27 6FE4
8B1F 8F41 493E 97B1 97DC 9955
530B 54B7 6D2E 8FEF 6843 9003 5555 68BC 6DD8 413B
7EF9 9676 8404 7979 7DAF 872A 88EA 9780 9184 9789
92FE 932D 99E3 6AAE 9940 9A0A 9F17
8BA8 8A0E
5957
5077 5078 5A7E 5AAE 92C0 936E
4EA0 5934 6295 9AB0 7DF0 982D
59B5 94AD 7D0F 3A97 6568 98F3 9EC8 8623
900F 7D89
574D 3631 62A9 8D2A 6039 75D1 8211 8CAA 644A 6EE9
762B 64F9 6524 7058 7671
575B 212A8 6619 5013 8C08 5A52 60D4 90EF 8983 6983
75F0 952C 8C2D 58B0 58B5 619B 6F6D 8AC7 9188 58C7
66C7 71C2 931F 9924 6A80 78F9 9843 7F48 58DC 85EB
8B5A 8C9A 91B0 8B60 7F4E
5FD0 5766 43D9 94BD 8892 6BEF 83FC 47A1 926D 55FF
61B3 61BB 9193 74AE 8962
53F9 70AD 57EE 63A2 509D 6E60 50CB 5606 78B3 8215
6B4E 8CE7
6C64 5763 94F4 6E6F 5621 8025 528F 7FB0 876A 47B6
4800 9557 859A 8E5A 93DC 940B 97BA 9F1E
9967 5510 5802 508F 557A 3B7B 68E0 347D 5858 370D
642A 6E8F 910C 69B6 6F1F 717B 746D 8185 84CE 969A
6A18 2365F 78C4 799F 7CC3 819B 6A56 7BD6 7CD6 8797
8E3C 7CDB 87B3 8D6F 91A3 9933 424E 9395 9939 95DB
9944 28189 9DB6
4F16 5E11 5018 5052 6DCC 50A5 8EBA 954B 93B2 513B
6203 66ED 7223 77D8 9482
70EB 6465 8D9F 71D9
71A5 81AF 9F1F
75BC 75CB 5E50 817E 8A8A 6F1B 6ED5 7E22 9086 99E6
8B04 512F 85E4 9A30 7C50 9C27 7C58 9A63
972F
8645
5254 68AF 3538 9511 8E22 64FF 9DC8 9DC9
5397 82D0 7EE8 8351 504D 557C 5D39 60FF 63D0 7A0A
7F07 7F64 9E48 55C1 7445 7D88 9046 78AE 5FB2 6F3D
7DF9 876D 8906 92BB 9898 855B 8DA7 8E44 918D 8B15
8E4F 9357 9CC0 9D3A 984C 9BB7 9D5C 9A20 9BF7 9D97
9D99 79B5 9DE4
4F53 632E 8EB0 9AB5 9BA7 8EC6 9AD4
623B 239C2 8FCF 5243 6711 6D1F 501C 608C 6D95 6090
60D5 63A6 9016 60D6 63E5 66FF 9037 6974 88FC 6B52
6BA2 8905 9AF0 3B31 568F 8599 9B00 5694 74CB 7C4A
8DAF 4D91
5C49 5C5C 7B39 5D5C
5E16 6017 8D34 8051 841C 8CBC
94C1 86C8 50E3 9295 92E8 9D29 9421 9435 9A56
546B 98FB 4D34 992E
65EB 4F7B 5EA3 604C 6311 7967 3B38 804E
8280 6761 5CA7 5CB9 8FE2 7952 689D 7B24 8414 47ED
8D92 9F86 84DA 8729 6A24 84E8 92DA 9AEB 9CA6 9797
93A5 9BC8 9F60 9C37
5BA8 6640 6713 3E20 8101 7A95 8A82 4837 7AB1 65A2
5B25
773A 7C9C 7D69 899C 8DF3 7CF6
87A9
5929 5172 5A56 6DFB 915F 9754 42EC 3B32 9EC7 975D
7530 5C47 6CBA 606C 754B 7551 76F7 80CB 7560 751B
751C 6E49 83FE 4852 5861 586B 6437 923F 9617 7DC2
78CC 7AB4 74B3 95D0 9DC6 9DCF
5FDD 6B84 500E 35AD 553A 3649 60BF 6DDF 666A 7420
8146 89CD 75F6 7753 8214 9902 89A5 8CDF 932A 9766
9369
3401 63AD 777C 821A
78B5 9D2B
5385 5E81 6C40 827C 542C 753A 8013 539B 70C3 686F
70F4 7D8E 42FC 9793 8074 5EF0 807C 807D 5EF3
9092 5EF7 4EAD 5EAD 505C 839B 5A77 5D49 6E1F 695F
7B73 8476 8713 69B3 8064 874F 95AE 9706 8AEA 9F2E
2123C 5722 753C 4FB9 5A17 633A 6D8F 6883 70F6 73FD
8121 988B 8247 8A94 9832 2A2B6
51F8 5B8A 79BF 79C3 6022 7A81 550B 6D8B 6378 5817
6E65 75DC 8456 5D80 92F5 9D5A 9F35
56F3 56FE 51C3 5CF9 5EA9 5F92 6087 6348 5C60 688C
3ECC 837C 9014 63EC 3B78 7A0C 83DF 5715 5857 5D5E
23225 760F 7B61 816F 922F 5716 5717 5EDC 6F73 84A4
8DFF 9174 48DD 999F 934E 99FC 9D4C 9D9F 9DCB 9DF5
571F 5721 5410 948D 91F7
514E 5154 8FCC 580D 9D75
6C62 6D82 83B5
4E47 4EDB 8BAC 6258 6261 6C51 9966 6754 4F82 5483
62D5 62D6 6CB0 6329 635D 8A17 6DB6 812B 8131 838C
88A5 98E5 9B60 9A5D
9A6E 4F57 5768 5CAE 6CB1 6CB2 72CF 9640 9641 8FF1
7823 7824 9E35 7D3D 8889 5836 8DCE 9161 99B1 69D6
78A2 4B7E 99C4 99DE 6A50 9B80 9D15 9F27 9A28 9F0D
9A52 9F09
5F75 59A5 5EB9 5AA0 692D 6955 5AF7 6A62 9D4E 9B0C
9C16
67DD 6BE4 553E 841A 8DC5 6BFB 7BA8 8600 7C5C
9A7C 99DD
63A8 84F7 85EC
5F1A 4003 9893 5C35 96A4 9839 983A 983D 9B4B 7A68
8E6A 8608 3FD7
4FC0 50D3 817F 8E46 9ABD
4FBB 5A27 9000 717A 86FB 8715 892A 99FE
6E4D 732F 7153 474E 8C92 4D4E
56E2 56E3 629F 5278 5718 6171 6476 6F19 69EB 7BFF
6AB2 93C4 7CF0 9DD2 9DFB
7583
5F56 6E6A 8916
541E 5451 6D92 554D 671C 711E 564B 66BE 3B3F 9ED7
5C6F 5749 5FF3 9968 829A 8C58 8C5A 8ED8 98E9 9C80
9B68 3A54 9715 81C0 81CB
6C3D 757D
65FD
56F2 70B5 75CC 901A 55F5 84EA
4EDD 540C 4F5F 5F64 5CC2 5E9D 54C3 5CDD 72EA 664D
6850 6D75 70D4 783C 833C 8692 772E 79F1 94DC 7AE5
7CA1 26A34 7B69 8A77 8D68 916E 9256 50EE 52ED 9275
9285 9907 9C96 6F7C 735E 66C8 6723 6A66 6C03 71D1
729D 81A7 3F3F 77B3 9BA6 2648D 4D00
7EDF 6345 3A8C 6876 7B52 7D71 7D82 6A0B
6078 75DB 8855 615F 6185
FDD0-310B
55EF
62CF 62FF 6310 55F1 954E 93BF
4E78 54EA 96EB
59A0 7EB3 90A3 80AD 94A0 5A1C 7D0D 8872 637A 7B1D
25E49 88A6 8C7D 8EDC 8C80 9209 9779 84B3 24E3B 9B76
422B
7592 8BB7 6290 7732 8A25
5436 5450 5462
8149 718B 6468 5B7B
4E43 5976 6C16 827F 7593 59B3 5EFC 5037 8FFA 91E2
5B2D
5948 67F0 800F 8010 217AB 6E3F 8418 9F10 879A 8926
933C
5A1E 9981 812E 8147 9912 9BBE 9BD8
5167 5185 343B 6C1D 9317
5B6C
5476 6013 6320 5CF1 7847 94D9 7331 86F2 8A49 7899
6493 5DA9 61B9 87EF 5912 8B4A 9403 5DCE
57B4 607C 60A9 8111 5318 5816 60F1 5AD0 7459 8166
78AF 4700 7376 737F
95F9 5A65 6DD6 9599 9B27 81D1
8133
7FBA
5542
69C8 8028 7373 6ABD 9392 941E 8B73
56E1
7537 678F 67AC 4FBD 5357 67DF 5A1A 7558 96BE 83AE
5583 6694 6960 8AF5 96E3 27574
8D67 63C7 6E73 8169 8433 4212 877B 6201
266DA 5A7B 3B2E
9056
6041 5AE9 5AF0
56D4
4E6A 56A2 8B68 56CA 8830 9B1E 9995 6B1C 9962
64C3 66E9 652E 7062
513E 9F49
80FD
20E4C 28D99
59AE
5C3C 576D 37BE 6029 6CE5 7C7E 502A 5C54 79DC 94CC
57FF 5A57 6DE3 730A 86AD 4626 90F3 68FF 249A4 46CF
8DDC 815D 8063 873A 89EC 8C8E 8F17 9713 9CB5 9BD3
9BE2 9E91 9F6F 81E1
4F31 4F60 62B3 62DF 72D4 67C5 82E8 65CE 6672 5B74
922E 999C 5117 511E 64EC 96AC 6AB7 85BF 807B
5C70 6C3C 4F32 23CB7 6635 80D2 8FE1 7724 9006 533F
5804 60C4 6135 6EBA 7768 817B 5ADF 66B1 8ABD 4065
7E0C 81A9 5B3A 2A65B
88AE
634F 63D1
82F6
5E07 26612 573C 67BF 6D85 75C6 8042 81EC 9667 556E
60D7 55A6 655C 6E7C 83CD 9689 55EB 5D72 8E02 565B
6470 69F7 8E17 954A 954D 5DAD 7BDE 81F2 931C 989E
8E51 5699 8076 93B3 95D1 5B7C 3738 5B7D 6AF1 7C4B
56C1 8616 9F67 23033 7CF1 8825 9448 7CF5 56D3 8B98
8EA1 9477 9873 9480 4BC0
5DD5
9E1F 8311 3B64 8885 9CE5 5ACB 88CA 6A22 8526 5B1D
892D 5B32 3735
5C3F 8132
599E
2EA7 725B 6C7C
5FF8 626D 72C3 7EBD 7084 94AE 7D10 83A5 9215 9775
8842 42F4
725C
62C8 852B
5E74 79CA 79E5 9C87 9B8E 9CB6 9ECF 9BF0
6D8A 637B 6DF0 713E 8DC8 8F87 8F97 649A 64B5 78BE
8F26 7C10 8E4D 6506 8E68 8E8E
5344 5EFF 5FF5 59E9 5538 57DD 824C 9F30
54D6 9D47
56DC 60A8 42FB
62F0
810C
5B22 5B43
917F 91B8 91C0
5A18
5B81 549B 62E7 72DE 67E0 82E7 804D 5BCD 5BD5 752F
5BD7 5BDC 5BE7 511C 51DD 5680 5B23 64F0 7370 6AB8
85B4 8079 27B48 944F 9B21 9E0B
6A63 77C3
4F5E 4FAB 6CDE 6FD8
6F9D
5974 5B65 9A7D 7B2F 99D1
4F2E 52AA 5F29 782E 80EC
6012 5089 6419
90CD 632A 689B 50A9 513A
6A60
8BFA 558F 611E 2272A 63BF 6426 903D 9518 643B 6992
7A2C 7CD1 8AFE 8E43 61E6 61E7 7CE5 7A64 7CEF
597B
6E1C 3B09 6696 7156 7157 992A
9EC1
519C 4FAC 54DD 6D53 8113 79FE 8FB2 5102 8FB3 5665
6FC3 6A82 71F6 81BF 857D 79AF 7A60 895B 91B2 6B01
7E77
5F04 630A 7651 9F48
5973 9495 7C79 91F9
6C91 6067 6712 8844 46BC
759F 8650 7878 7627
FDD0-310C
5783 62C9 67C6 7FCB 83C8 641A 908B
65EF 524C 782C 63E6 78D6
5587 85DE
814A 63E7 694B 760C 8721 874B 8FA2 8FA3 8772 81C8
650B 7209 81D8 9B0E 74CE 9574 9BFB 881F 945E
5566 6E82 97A1 56B9
56D6
808B
4EC2 3539 4E50 53FB 5FC7 6250 6C3B 961E 738F 827B
6CD0 7AFB 7833 697D 97F7 6A02 7C15 39A1 9CD3 9C33
4E86 9979 990E
6765 4F86 4FEB 5008 5D03 5F95 6D9E 5A61 5D0D 5EB2
5FA0 68BE 6DF6 730D 83B1 90F2 68F6 741C 7B59 840A
9028 94FC 7B82 42F1 28090 9338 9A0B 9BE0 9D86 9EB3
553B 8D49 7750 775E 8D56 8CDA 6FD1 8CF4 983C 9842
21096 765E 9D63 7028 702C 7C41 6AF4 3E0A 85FE 7669
7C5F 8970
52D2
96F7 5AD8 7F27 757E 8502 64C2 3D62 6A91 7E32 790C
956D 6AD1 74C3 7FB8 7927 7E8D 7F4D 881D 9433 8632
8F60 513D 58E8 9458 9741 6B19 8646 7E9D 9F3A
53BD 8012 8BD4 5792 7D6B 8142 50AB 8A84 6A0F 78CA
78E5 854C 5121 4242 857E 58D8 3D7D 7657 6AD0 85DF
7928 7045 8B84 863D 9478 9E13
6CEA 6D21 7C7B 6D99 6DDA 7D2F 28024 9179 9287 981B
982A 9311 48AE 6502 98A3 985E 426A 7E87 8631 79B7
5841 561E 9C69
635E 6488
52B3 52B4 7262 7A82 54F0 5520 5D02 6D76 52DE 75E8
94F9 50D7 562E 37A0 5D97 61A5 24A42 7646 78F1 7C29
87E7 91AA 9412 985F 9ADD
8002 8001 4F6C 54BE 59E5 6045 72EB 6833 8356 94D1
41ED 92A0 6F66 6A51 8F51
6D9D 70D9 8022 916A 5AEA 61A6 6F87 8EBC 6A6F 802E
8EC2
73EF 7853 266E8 7CA9 86EF 6725 9BB1
779C
5245 5A04 507B 5A41 6E87 50C2 697C 848C 5ED4 617A
6F0A 6A13 71A1 8027 851E 877C 9071 802C 825B 87BB
8B31 8EC1 9AC5 97BB 9ACF
5D5D 6402 587F 5D81 645F 7BD3 750A 7C0D
3537 964B 5C5A 6F0F 7618 9542 763A 763B 93E4
55BD 560D
5170 5C9A 62E6 680F 5A6A 60CF 5D50 9611 847B 84DD
8C30 53B1 6F9C 5116 6593 7BEE 8934 61E2 71E3 71F7
9567 95CC 74BC 85CD 8955 8B4B 2814F 5E71 6514 703E
7C43 7E7F 8964 6595 6B04 7046 862D 7937 7C63 8974
56D2 7061 8B95 8E9D 6B17 9484 97CA
89C8 6D68 36E6 63FD 7F06 6984 6F24 7F71 9182 58C8
61D2 89A7 3A5C 64E5 5B3E 61F6 5B44 89BD 432B 5B4F
652C 7060 56D5 6B16 9872 7E9C
70C2 6EE5 71D7 5682 6FEB 7201 721B 74D3 7224 946D
7CF7
7226 897D
5577
52C6 90CE 6B34 72FC 90DE 9606 658F 6879 3C62 7405
5ACF 5ECA 7860 7A02 9512 6994 746F 7B64 8246 84C8
870B 8EB4 8782 92C3 93AF 99FA
6717 6716 70FA 365F 5871 6A03 84E2 8A8F 3BBE 6724
57CC 5D00 6D6A 3AF0 83A8 20E98 8497 95AC
5525 90D2
3604
5D1A 5844 68F1 695E 7890 7A1C 8F18 4244 8590
51B7
5030 580E 6123 7756 8E1C
5215 675D 5398 5253 9A8A 60A1 68A8 68A9 68B8 7281
740D 79BB 7C9A 8372 55B1 68C3 3D1D 7282 83DE 9E42
527A 6F13 3E48 775D 7B63 7F21 8243 870A 20ED7 20F1D
5AE0 5B77 76E0 8C8D 3992 6A06 7483 7CCE 84E0 92EB
9CA1 9ECE 7F79 853E 26E99 9305 7BF1 7E2D 4527 87CD
8935 569F 8B27 91A8 91D0 96E2 6584 3C00 74C8 85DC
908C 93EB 9BEC 9D79 9EE7 56C4 8821 9A39 5B4B 5EF2
6521 7055 5299 863A 9457 7A72 7C6C 7E9A 9A6A 9C7A
9E1D
793C 91CC 4FDA 5CDB 5CE2 5A0C 5CF2 6D6C 7406 9026
9502 7CB4 88CF 8C4A 92F0 9CA4 5163 6FA7 79AE 9BC9
87F8 91B4 9CE2 9090 9C67 6B1A
529B 5386 5389 5C74 7ACB 540F 6738 4E3D 5229 52B1
5456 575C 6CA5 4F8B 3578 5CA6 623E 67A5 6CB4 75A0
82C8 96B6 4FD0 4FEA 3926 680E 75AC 7805 82D9 8D72
8F79 3466 550E 60A7 6817 681B 6D96 7301 73D5 783A
783E 79DD 8318 8354 90E6 5533 5A6F 7B20 7C92 7C9D
8137 8385 8389 86B8 86CE 5088 51D3 53A4 68D9 75E2
86E0 8A48 8DDE 96F3 53AF 585B 6144 642E 6EA7 925D
9CE8 3B0F 66A6 6B74 746E 7D9F 849A 849E 8727 53B2
8777 66C6 6B77 7BE5 4509 96B7 29D98 9D17 52F5 3EFA
78FF 96B8 9B01 512E 5DC1 6FFF 7658 9398 56A6 58E2
650A 66DE 6AD4 6ADF 701D 7204 72A1 74C5 8807 9E97
6AEA 720F 74D1 76AA 76ED 77CB 792A 792B 79B2 85F6
34A7 5137 7667 7930 7CF2 8823 882B 9DC5 9E9C 56C7
6526 2789D 89FB 8E92 8F62 9148 6B10 8B88 2829B 8F63
652D 74E5 9742 299C3 9C71 9C73 974B
674E 6803 54E9 5A33 72F8 88E1 6AAA 9BCF
4FE9 5006
5217 52A3 51BD 52BD 39DC 59F4 6312 6D0C 54F7 57D2
57D3 3920 3B5E 6835 6D56 70C8 8322 8FFE 6369 3B69
730E 811F 86DA 88C2 716D 7759 8057 8D94 5DE4 98B2
5120 9BA4 9D37 64F8 7375 72A3 8E90 9B1B 3B2F 9B23
9C72
6BDF 54A7 6318 70EE 731F
64A9 8E7D
8FBD 7597 804A 50DA 5BE5 5D7A 6180 6F3B 818B 5639
5AFD 5BEE 5D9A 5D9B 6579 7360 7F2D 66B8 71CE 7499
81AB 907C 3D73 7642 9E69 5C6A 5EEB 7C1D 7E5A 87DF
8C42 8CFF 8E58 9410 85D4 98C9 9ACE 9DEF
53FE 4126 948C 91D5 911D 61AD 84FC 77AD 66E2 957D
7212
5C25 5C26 7093 6599 5C1E 5ED6 6482 7AB7 9563
720E
6E9C 7198 8E53
5218 6CA0 7544 6D41 6D4F 7409 7559 65C8 7571 786B
88D7 28EE7 5AB9 5D67 65D2 84C5 998F 9A9D 69B4 7460
84A5 905B 98D7 5289 746C 7624 78C2 954F 99E0 9E60
6A4A 74A2 7581 9560 7645 87C9 295E9 99F5 56A0 61F0
700F 938F 93A6 9E8D 85F0 93D0 98C0 9A2E 98C5 9C21
9DB9 9A51
39D5 67F3 6801 73CB 687A 7EFA 950D 925A 98F9 7DB9
71AE 7F76 92F6 6A6E 5B3C 7F80
516D 7542 7FCF 586F 5EC7 6F91 78DF 9E68 9724 993E
96E1 9402 98C2 9B38 9DDA
685E
5941 5E18 601C 8FDE 6D9F 68BF 83B2 467A 9023 8054
4EB7 5EC9 6E93 5332 55F9 5969 6169 6F23 7191 88E2
899D 5286 5333 5652 5AFE 6190 69E4 78CF 806B 84EE
9CA2 6FC2 6FD3 806E 878A 71EB 7E3A 7FF4 806F 81C1
8595 8933 8B30 8E65 938C 9570 6AE3 7C3E 880A 9B11
4965 942E 9C31 7C62 7C68
655B 740F 8138 88E3 6459 7489 8539 5B1A 6582 3BEC
81C9 913B 7FB7 895D 861E
7EC3 70BC 604B 6D70 6B93 581C 3703 5AA1 3C68 6E45
94FE 50C6 695D 7149 7453 8430 6F4B 7DF4 6FB0 932C
6BAE 934A 93C8 3D91 7032 9C0A 861D 6200 373B 7E9E
8068
62CE
53B8 6797 90BB 4E34 51A7 77DD 5549 5D0A 6DCB 667D
7433 7CA6 75F3 7884 7B96 7CBC 5D99 6F7E 735C 9130
96A3 65B4 66BD 71D0 7498 8F9A 9074 9716 77B5 78F7
81E8 7E57 7FF7 9E90 8F54 58E3 7036 93FB 9CDE 9A4E
9C57 9E9F
342D 3A06 83FB 4E83 51DB 51DC 649B 5EE9 5EEA 61CD
61D4 6F9F 6A81 6AA9 765B 765D
541D 6061 608B 8D41 711B 8CC3 50EF 853A 6A49 81A6
95B5 7510 7584 8E78 85FA 8E8F 8E99 8EAA 8F65
826F 4FCD 51C9 6881 6DBC 690B 8F8C 7CAE 7CB1 589A
7DA1 8E09 6A11 8F2C 7CE7
4E21 4E24 34B3 5169 5521 5562 639A 813C 7DC9 873D
88F2 9B49 9B4E
4EAE 54F4 60A2 8C05 8F86 55A8 667E 6E78 91CF 8F0C
8AD2 8F1B 9344
7177 7C17
3007 5222 7075 56F9 577D 590C 59C8 5CBA 5F7E 6CE0
72D1 6624 670E 67C3 73B2 82D3 51CC 74F4 768A 7831
79E2 7ADB 94C3 9E30 5A48 6395 68C2 6DE9 740C 7B2D
7D37 7EEB 4345 7F9A 7FCE 8046 8232 86C9 8851 9675
83F1 8A45 8DC9 8EE8 797E 9234 959D 96F6 9F84 420A
7DBE 88EC 8506 970A 99D6 6FAA 9302 9B7F 9CAE 9D12
9E77 3B21 71EF 4241 4516 8576 971B 971D 9F62 9BEA
5B41 9143 9F61 6AFA 8626 91BD 9748 2331F 6B1E 7227
24AE9 9EA2 4585 9F97
5CAD 963E 888A 9886 9818 5DBA
4EE4 53E6 5464 70A9
4F36 84E4 9717 702E
565C 64B8
5362 5E90 5786 6CF8 7089 82A6 3B54 680C 80EA 8F73
9E2C 7388 823B 9885 9C88 9B72 76E7 56A7 58DA 3820
5EEC 650E 6ADA 7018 7379 74B7 66E5 6AE8 7210 74D0
81DA 8606 77D1 7C5A 7E91 7F4F 826B 8826 8F64 946A
9871 9AD7 9C78 9E15 9EF8
5364 864F 63B3 9E75 7875 9C81 865C 5877 6EF7 6A10
84FE 9B6F 64C4 6A79 78E0 9565 5695 64FC 7002 6AD3
6C0C 8263 93C0 826A 942A 9465
5725 752A 4F93 5774 5F54 5F55 9646 5CCD 52CE 8D42
8F82 5A3D 6DD5 6DE5 6E0C 7849 9678 9E7F 3A90 6902
742D 7984 4402 83C9 902F 50C7 5279 52E0 76DD 7769
788C 797F 7A11 8CC2 8DEF 5876 5ED8 645D 6F09 7B93
7CB6 21014 622E 6A1A 719D 8194 850D 89EE 8DA2 8E1B
8F98 9181 3BDD 6F5E 7A4B 9304 9332 9334 7490 7C0F
8557 87B0 8E57 8F46 9A04 298F1 9E6D 7C2C 7C36 93D5
9BE5 9D66 9D71 9E93 93F4 9732 9A3C 7C59 9DFA 8642
67A6 822E 9229 6F9B 6C07
7F57 5570 9831 56C9
7F56 7321 8136 6924 841D 8999 903B 8161 9523 7BA9
9AA1 9559 87BA 7F85 89B6 93CD 5138 89BC 9A3E 21922
651E 7380 6B0F 268C7 863F 908F 9A58 9E01 7C6E 947C
9960
5246 502E 84CF 88F8 8EB6 7630 3A61 3C01 8803 81DD
66EA 7673
6CFA 5CC8 6D1B 7EDC 9A86 6D1C 73DE 8366 7866 7B3F
7D61 55E0 843D 645E 6F2F 7296 927B 96D2 99F1 9BA5
9D3C 9D45 6FFC 277CC 7E99
5A08 5B6A 5CE6 631B 683E 9E3E 8114 6EE6 92AE 9D49
571D 5971 5B4C 5B7F 5DD2 6523 66EB 6B12 7053 7F89
81E0 571E 7064 864A 947E 7674 7675 9E1E
5375
4E71 91E0 4E82
62A1 6384
4ED1 4F26 56F5 6CA6 7EB6 4F96 8F6E 502B 5707 5A68
5D18 5D19 60C0 6DEA 966F 68C6 3DCD 8140 83D5 4201
7DB8 8726 8E1A 8F2A 9300 9BE9
57E8 7896 7A10 8023
8BBA 6EA3 8AD6
78EE
9F99 5C78 5499 6CF7 663D 680A 73D1 80E7 830F 772C
783B 7ADC 7B3C 804B 6E70 9686 6EDD 20F90 5D90 6F0B
3863 7BED 856F 9F8D 7643 56A8 5DC3 5DC4 7027 7C3C
93E7 66E8 6727 3C0D 6AF3 7216 74CF 8622 9733 77D3
7931 7932 9F92 7C60 807E 882A 882C 8971 8C45 8E98
9468 9747 4BBE 9A61 9E17
5784 5785 62E2 9647 3D33 7BE2 5131 58DF 58E0 650F
96B4 7AC9 9F93
54E2 6335 6887 5FBF 8D1A
69DE 7ABF
9A74 95FE 90D8 6988 99BF 6C00 81A2 95AD 85D8 9DDC
9A62
5415 5442 4FA3 4FB6 6314 36CE 634B 635B 65C5 3B5A
68A0 7963 7A06 94DD 5C61 7F15 7D7D 5C62 8182 5C65
891B 92C1 8190 5122 7A5E 7E37 8938 7A6D
5BFD 578F 5F8B 8651 7387 7EFF 20DC8 5D42 6C2F 6EE4
844E 7DA0 7DD1 616E 7BBB 819F 52F4 7E42 6FFE 6AD6
7208 9462
7112
7567 950A 7A24 3500 5719 3A3C 92DD 92E2 64FD
FDD0-310D
65EE 5477 560E 5620
9486 5C1C 5676 9337
5C15 738D
5C2C 9B40
6208 4EE1 572A 72B5 7EA5 20BEB 6213 8090 726B 7599
54AF 7271 54E5 80F3 9E3D 5272 6401 88BC 20E9D 6ED2
6228 6B4C 29FB7 9D10 9D1A 64F1 8B0C 9D3F 93B6
5444 3837 4F6E 530C 630C 9601 9769 654B 3B58 683C
8316 9B32 6105 81F5 86D2 55DD 5865 6EC6 845B 88D3
89E1 9694 643F 69C5 8188 95A3 95A4 4213 9549 9788
97D0 7366 8AFD 8F35 9ABC 9BAF 97DA 97B7 9A14 4601
8F55
54FF 8238
4E2A 5404 867C 500B 784C 94EC 55F0 7B87
5F41 6ACA
4F85 8BE5 5793 59DF 5CD0 90C2 9654 6650 8344 8D45
7561 7974 7D6F 8A72 8C65 8CC5
5FCB 6539 7D60
4E10 4E62 5303 5304 9623 675A 9499 76D6 6461 6E89
9223 6224 6982 8462 8CCC 9691 6F11 84CB 3562 69E9
69EA 74C2
7ED9 7D66
768B 7F94 7F99 9AD8 7690 2690E 9AD9 81EF 6EDC 69D4
777E 818F 69F9 6A70 7BD9 7CD5 993B 6ADC 9DCE 9F1B
9DF1
5930 6772 83D2 641E 7F1F 66A0 69C0 69C1 3D46 7A3E
7A3F 9550 7E1E 6ABA 85C1 85F3
543F 544A 52C2 53DD 8BF0 90DC 796E 7970 9506 7170
7B76 8AA5 799E 92EF
97DF
52FE 4F5D 6C9F 94A9 88A7 7F11 920E 6E9D 9264 7DF1
7BDD 8920 97B2 97DD
5CA3 72D7 82B6 67B8 73BD 8007 82DF 7B31 8008 8009
86BC 8C7F
5778 6784 8BDF 8D2D 57A2 59E4 5193 8329 591F 5920
8A3D 5ABE 5F40 6406 8A6C 96CA 69CB 7179 89CF 9058
6480 89AF 8CFC
7518 5FD3 653C 6746 3DA5 7395 809D 8289 8FC0 5769
6CD4 77F8 4E79 67D1 7AFF 82F7 75B3 9150 4E7E 7C93
4E81 51F2 5C32 5C34 7B78 6F27 9CF1 5C36 5C37 9B50
4EE0 625E 76AF 79C6 8866 8D76 687F 7B34 6562 7A08
611F 8D95 6F89 64C0 6A44 7C33 9C14 9CE1 9C64
5E72 65F0 6C75 76F0 7EC0 501D 51CE 6DE6 7D3A 8A4C
5E79 9AAD 69A6 6A8A 8D11 8D63 8D1B 7068
6839 8DDF
54CF
826E
4E98 4E99 831B 63EF
5188 7F53 51AE 521A 6760 7EB2 809B 5CA1 3B4E 7268
7598 77FC 7F38 94A2 525B 7F61 5808 6386 91ED 68E1
7285 583D 7DB1 7F41 92FC 93A0
5C97 3F58 5D17 6E2F
7135 7B7B 69D3 6205 6206
522F 5E9A 754A 6D6D 3E74 8015 6404 713F 7D5A 83EE
8D53 9E52 7DEA 7E06 7FAE 8CE1 7FB9 9D8A
54FD 57C2 5CFA 632D 7EE0 803F 90E0 231B3 6897 8384
3F98 7D86 4304 9CA0 9ABE 9BC1
66F4 5829 6685 4C4D 4C6D
63B6 6929
4F30 5471 59D1 5B64 6CBD 6CD2 67E7 82FD 8F71 5502
7F5B 9E2A 7B1F 86C4 83F0 89DA 8EF1 8EF2 8F9C 9164
9232 7B8D 7B9B 5AF4 6A6D 443B 9B95 9D23
9DBB
5903 53E4 6262 6C69 8BC2 8C37 80A1 726F 2EE3 5503
4026 7F5F 7F96 94B4 9AA8 5552 6DC8 8135 86CA 86CC
8A41 9E44 5C33 6132 6BC2 9237 9989 9F13 9F14 560F
6996 69BE 76B7 84C7 7A40 9E58 7CD3 7E0E 6FF2 76BC
81CC 85A3 8F42 7014 76EC 77BD 9936 4D9C 8831
56FA 6545 51C5 987E 580C 5D13 5D2E 688F 727F 68DD
96C7 75FC 797B 7A12 9522 50F1 932E 9CB4 9BDD 9867
5495 5CE0 9027 50A6 83C7 7BD0
74DC 522E 80CD 681D 20D6F 9E39 8052 44B7 6B44 7171
8D8F 5280 7DFA 92BD 98B3 8E3B 9D30 9A27
53E7 518E 5250 526E 5BE1 4211
5366 576C 8BD6 6302 5569 639B 7F63 7D53 7F6B 8A7F
8902
98AA
5459 54BC 57DA 5D1E 90ED 581D 921B 9505 5613 588E
7611 8748 5F49 35FB 6FC4 87C8 934B 5F4D
56EF 56F6 56FB 56FD 5700 570B 5E3C 8158 5E57 6156
6F0D 805D 8195 852E 8662 9998 27C12 4190
679C 60C8 6DC9 7313 9983 6901 83D3 7CBF 7DB6 873E
88F9 69E8 8F20 9301 991C 9439
8FC7 904E
556F
4E56 63B4 6451
62D0 67B4 67FA 27285 7B89
592C 53CF 602A 39D4 6060
5F52 572D 59AB 9F9F 89C4 7688 90BD 95FA 5E30 73EA
80FF 8325 4E80 7845 4145 7A90 898F 5080 5AAF 691D
88BF 5EC6 90CC 5AE2 646B 7470 95A8 9C91 5B00 69FB
69FC 749D 81AD 879D 9F9C 9BAD 5DC2 6B78 9B36 74CC
9A29 9B39 6AF7
5B84 6C3F 6739 8F68 5E8B 4F79 5326 8BE1 20C53 579D
59FD 6051 6531 7678 8ECC 9652 5EAA 9B3C 532D 796A
6677 6E40 436F 86EB 3533 89E4 8A6D 53AC 3A33 4925
87E1 77A1 7C0B
6530 523D 523F 660B 7094 67DC 8D35 3A88 6842 6867
7324 7B40 8CB4 8DEA 5331 84D5 528A 528C 5DA1 648C
69F6 4210 4434 6A9C 77B6 7C02 6AC3 7650 79AC 8958
9CDC 97BC 9C56 9C65
6922
5173 89C2 5B98 51A0 898C 500C 20D28 68FA 7AA4 8484
95A2 761D 764F 89B3 95D7 9CCF 95DC 9C25 9C5E 89C0
839E 9986 742F 75EF 7B66 7BA1 8F28 8218 9327 9928
9CE4
6BCC 4E31 8D2F 6CF4 60BA 60EF 63BC 6DAB 8CAB 60B9
3D26 3BA1 797C 6163 645C 6F45 6A0C 9066 76E5 7F46
465B 96DA 93C6 704C 721F 74D8 9E73 77D4 7936 7F50
9475 9C79 9E1B
4E28 200CC 886E 60C3 7EF2 889E 88AC 8F8A 6EDA 6EFE
7DC4 84D8 78D9 8509 8F25 9CA7 25566 9B8C 9BC0
68CD 7754 3665 7774 74AD 8B34
5149 706E 4F8A 7097 709B 54A3 5799 59EF 6D38 6844
70E1 80F1 832A 8F04 50D9 92A7 9EC6 299C9
5E7F 5E83 72B7 5EE3 81E9 7377 2A1F5
4FC7 73D6 901B 81E6 6497 29448
709A 6B1F
5DE5 5F13 516C 53B7 529F 653B 675B 4F9B 739C 7CFC
80B1 5BAB 5BAE 606D 8EAC 9F9A 5311 5868 5E4A 6129
89E5 7195 8EB3 78BD 9AF8 4CCD 89F5 9F8F 9F94
5EFE 5DE9 6C5E 62F1 39EC 62F2 3B5F 6831 73D9 3F26
8F01 92DB 978F 4C4B
5171 8D21 7FBE 551D 8CA2 83BB 3536
86A3 6150
FDD0-310E
5494 5496 5580 8849 64D6
5361 4F67 80E9 9272
57B0 88C3
533C 3E2F 22208 67EF 7241 73C2 79D1 80E2 82DB 8F72
75B4 7822 8DB7 68F5 8EFB 988F 55D1 6415 7290 7A1E
7AA0 842A 9233 69BC 4426 9897 6A16 778C 78D5 4300
874C 9312 8596 9198 9846 9AC1 791A
58F3 63E2 6BBC 7FD7
53EF 5777 37B9 5CA2 3ABC 70A3 6E07 5D51 6564 6E34
5DB1 790D
514B 523B 524B 52C0 52CA 5BA2 606A 20CED 5A14 5C05
8BFE 20DB2 5801 6C2A 9A92 7F02 6119 6E98 951E 78A6
7DD9 8250 8AB2 790A 9A0D 2A0ED
5D59
5F00 5952 63E9 950E 958B 4924 9426
51EF 5240 57B2 607A 95FF 94E0 51F1 5274 6168 584F
5D66 6137 6977 8488 8F06 5605 669F 9534 4057 9347
93A7 95D3 98BD
5FFE 708C 708F 6B2C 70D7 52D3 6112 613E 938E
5C3B 9ADB
4E02 6537 8003 62F7 6D18 6832 70E4 7A01 9C93 71FA
94D0 7292 92AC 9760 9BB3 9BCC
62A0 26B05 82A4 770D 527E 5F44 6473 7798 4071
53E3 24618 52B6
53E9 6263 6542 51A6 5BBC 5BC7 3C2F 91E6 7A9B 7B58
6EF1 7789 8532 853B 7C06 9DC7
520A 681E 52D8 9F9B 582A 5D41 6221 9F95
519A 574E 4F83 780D 5058 57F3 60C2 83B0 6B3F 586A
6B41 69DB 8F21 6ABB 9851 7AF7 8F57
770B 884E 5D01 5888 78E1 77B0 95DE 77D9
808E 80AF 80BB 57A6 6073 5543 8C64 9F88 58BE 9339
61C7 9F66
63AF 88C9 8903
5FFC 95F6 780A 7C87 5EB7 21413 5ADD 5D7B 6177 6F2E
69FA 7A45 41B2 7CE0 8EBF 93EE 9C47
625B 6443
4849
4EA2 4F09 531F 56E5 6297 72BA 909F 7095 94AA 9227
958C
52A5 542D 5751 5994 962C 6333 7841 727C 785C 94FF
787B 647C 8A99 92B5 40D8 935E 93D7
625D 5233 77FB 67AF 80D0 90C0 54ED 684D 5800 5D2B
35C4 5710 8DCD 7A9F 9AB7 26846 9BAC
72DC 82E6
3838 5E93 4FC8 7ED4 5EAB 79D9 8DB6 7105 55BE 7D5D
88B4 88E4 7614 9177 5EE4 8932 56B3
5938 59F1 8A87
4F89 54B5 57AE 9299
3404 630E 80EF 8DE8 9ABB
823F
6269 62E1 62EC 6304 6870 7B48 86DE 9614 843F 8440
5ED3 9822 9AFA 6FF6 95CA 979F 61D6 64F4 9729 97B9
9B20
97D5
39DF 84AF 64D3
5DDC 51F7 5757 5FEB 4FA9 54D9 72EF 90D0 810D 584A
3C6E 7B77 9C99 5108 58A4 5672 5EE5 736A 9136 81BE
65DD 7CE9 9C60
5726
4E8F 5232 5CBF 609D 76D4 7AA5 8067 7ABA 8667 95DA
985D 5DCB 862C
594E 6646 9804 9997 55B9 63C6 45AF 9035 9108 9A99
6223 668C 694F 6951 8475 9697 777D 9B41 8770 4906
982F 367A 9368 9377 6AC6 85C8 9A24 5914 8637 8641
5DD9 2A1B4 72AA 8EA8
7143 8DEC 980D 8E5E
5C2F 532E 6B33 559F 6126 6E83 8143 9988 5ABF 6127
8489 7786 5633 5B07 6192 6F70 7BD1 8069 4646 6A3B
806D 8562 8B09 7C23 8075 993D 7C44 9400 994B 944E
4C0E
5BBD 5BDB 5BEC 81D7 9ACB 9AD6
6B35 6B3E 6B40 7ABE 4540
7ABD 9467
5764 6606 5803 5A6B 5D10 5D11 665C 7311 711C 7428
83CE 88C8 9AE0 8C87 951F 9AE1 9E4D 872B 88E9 9AE8
747B 890C 918C 9315 9CB2 9A09 9BE4 9D7E 9DA4
6083 6346 9603 58F8 68B1 7871 7975 7A07 58FC 7A1B
7D91 88CD 95AB 95B8 9F6B
56F0 6D83 3AFB 774F
5812 5C21 6F49 71B4
5321 344C 52BB 8BD3 5329 54D0 6047 6D2D 90BC 6846
7844 7B50 44B0 8A86 8EED
5FF9 6282 72C2 8BF3 8ED6 8A91 9D5F
593C 5123 61ED
535D 5739 7EA9 909D 51B5 65F7 5CB2 6CC1 77FF 663F
8D36 7716 7736 7D56 8CBA 8EE6 9271 58D9 8E80 913A
9ECB 61EC 66E0 720C 77CC 7926 7A6C 7E8A 945B
783F 7D4B 7B7A
7A7A 369A 5025 57EA 5D06 60BE 6DB3 787F 7B9C 9313
9D7C
5B54 233FA 6050
63A7 26048 979A
8EBB
FDD0-310F
5677
54C8 94EA
86E4
5964
4E37
8BC3 62B2 6B31 559D 8A36 55EC 881A
79BE 5408 4F55 52BE 5392 548A 548C 59C0 6CB3 5CC6
3A83 66F7 67C7 72E2 76C7 7C7A 7D07 90C3 9602 9978
54EC 3A89 6546 6838 76C9 76CD 555D 3B71 6DB8 76D2
79F4 8377 86B5 9F81 60D2 6E2E 83CF 8402 8A38 988C
6941 6BFC 8A65 8C88 47A6 8F05 924C 9616 9C84 3561
7186 9E56 9EA7 6F95 3FE5 981C 7BD5 7FEE 879B 9B7A
7909 95D4 97A8 9F55 8988 9DA1 76AC 9449 9FA2
4F6B 578E 8D3A 7103 8894 8CC0 55C3 7142 788B 7187
8D6B 3D51 8910 9E64 3B1E 7A52 7FEF 58D1 764B 8B1E
7200 9DAE 9DB4 974E 9E16 974F
7CAD 974D
548D 54B3 55E8
8FD8 5B69 9826 9AB8 9084
6D77 80F2 70F8 917C 91A2
4EA5 598E 39E1 9A87 5BB3 6C26 55D0 9900 99ED 26926
995A
5870 56A1
9ED2 9ED1 563F 6F76 28B46
8320 84BF 5686 8585 85A7
6BDC 869D 6BEB 6903 55E5 7346 8C89 5651 7354 8C6A
295B8 5637 3B14 734B 8AD5 512B 568E 58D5 3819 6FE0
7C47 8814 8B79
597D 90DD
53F7 36AA 3740 660A 6626 79CF 54E0 5CFC 604F 608E
6D69 8017 6667 3B36 6DCF 5090 7693 6EC8 8055 865F
9117 66A4 66AD 6F94 769C 769E 66CD 76A1 76A5 8583
93AC 98A2 704F 9865 9C1D 705D
7AD3
9F41
4FAF 3927 77E6 5589 5E3F 7334 9107 3B0B 8454 760A
777A 7BCC 7CC7 7FED 9ABA 7FF5 936D 9931 9BF8
543C 72BC
540E 539A 5795 5F8C 6D09 90C8 9005 5820 8C5E 9C8E
9C98 9B9C 9C5F
5019
4F44 70B6 9878 3937 86B6 9163 9807 5AE8 8C3D 61A8
99A0 6B5B 9F3E
9097 542B 51FD 5481 80A3 90AF 51FE 8677 5505 5705
5A22 6D5B 5D21 6657 6892 6DB5 7113 7400 5BD2 5D45
3B80 97E9 751D 7B68 4204 872C 6F8F 92E1 9B7D 97D3
4E06 5388 7F55 6D6B 558A 961A 850A 3E01 8C43 9B2B
361A
6C49 5C7D 6C57 95EC 65F1 5CBE 54FB 57BE 608D 634D
6D86 7302 3A8B 6658 6665 710A 839F 91EC 9588 7694
7745 83E1 50BC 86FF 9894 99AF 6F22 872D 8C8B 6496
66B5 71AF 92B2 92CE 61BE 64BC 24CC9 7FF0 8792 9837
9844 99FB 96D7 701A 8B40 862B 9DBE
516F 7233
62EB 75D5 978E
4F77 5F88 72E0 8A6A
6068
592F 3C20
65BB 676D 82C0 8FD2 7ED7 73E9 7B10 822A 86A2 4615
9883 8CA5 7B55 7D4E 980F 28048 9B67
6C86
57B3
4EA8 54FC 6099 5548 811D
3530 59EE 6046 6052 6841 70C6 80FB 9E3B 6A2A 6A6B
8861 9D34 8605 9445
583C
6DA5 9D46
4E6F 5322 864D 547C 5780 5FFD 6612 66F6 6CD8 6057
70C0 82F8 8F77 532B 553F 60DA 39FE 6DF4 8656 8EE4
5611 5BE3 6EF9 44E4 96D0 5E60 622F 6B51 27A53 81B4
8B3C
56EB 6287 5F27 72D0 80E1 58F6 74F3 96BA 20D31 58F7
659B 7100 35C5 5596 58FA 5AA9 6E56 7322 7D57 6430
695C 7173 745A 846B 561D 2430D 9E55 69F2 7BB6 851B
8774 885A 9B71 7E20 879C 9190 9836 89F3 9378 992C
4B4C 9D60 702B 9B0D 9C17 9D98 9DA6
4E55 6C7B 864E 6D52 4FFF 7425 8400 865D 6EF8
4E65 4E92 5F16 6236 6237 6238 3E26 51B1 51B4 5E0D
62A4 6C8D 6CAA 8290 5CB5 6019 623D 6608 6791 6018
795C 7B0F 5A5F 6248 74E0 55C0 695B 7D94 96FD 5AED
5AEE 6462 6EEC 9120 69F4 71A9 8530 9CF8 7C04 9359
569B 9E71 8B77 9CE0 97C4 9800 9C6F 9E0C
4E4E 7C90 552C 7CCA 933F 9BF1
82B1 82B2 54D7 5629 848A 9335
534E 3572 3B49 59E1 9A85 91EA 91EB 94E7 83EF 6433
6ED1 733E 380F 398A 64B6 78C6 92D8 8550 8796 8B41
93F5 9A4A 9DE8
5316 5212 3566 593B 3577 6779 753B 8BDD 6866 5A73
5D0B 756B 7575 89DF 8A71 5283 6466 5AFF 5B05 69EC
6F85 6A3A 8AE3 7E63 8219 9ECA 8B6E
57D6 5A72 691B 7874 7CC0 8AAE 748D
5419 5268 8020 952A 5290 9343 5684 8C41 6509 9A1E
4F78 6D3B 79EE 79F3
706B 4F19 90A9 94AC 9225 5925 6F37
6C8E 6216 8D27 549F 7809 4100 4FF0 6347 7713 9584
20D7E 639D 83B7 8CA8 60D1 6E71 7978 65E4 6947 3A2F
798D 84A6 35F2 596F 970D 6FE9 7372 8B0B 6AB4 956C
56AF 7016 77C6 7A6B 4C5B 66E4 802F 81DB 8267 85FF
8816 56BF 7668 77D0 944A 9743
6000 5F8A 6DEE 69D0 8E1D 61D0 8922 8931 3733 3822
61F7 7024 6AF0 8032 8639
574F 54B6 8AD9 58CA 58DE 863E
7070 39D1 8BD9 54B4 6062 62FB 6325 6D03 867A 6656
70E3 73F2 8886 8C57 5A4E 5645 5A88 63EE 3DC7 7FDA
8F89 6689 694E 7147 8A7C 9693 5E51 7773 7988 649D
5655 7FEC 8918 8F1D 9EBE 5FBD 7008 96B3 8633 9C34
56D8 56DE 56EC 4F6A 5EFB 605B 6D04 5EFD 70E0 8334
8698 8FF4 75D0 9025 86D4 86D5 8716 9BB0
6094 6BC0 6BC1 26799 6BC7 6A93 71EC 8B6D
5349 6C47 4F1A 8BB3 6CCB 54D5 21726 6D4D 7ED8 8294
8BF2 605A 6075 70E9 835F 8D3F 5F57 6666 79FD 5599
60E0 6E4F 7D75 7F0B 7FD9 9613 532F 5F59 5F5A 3963
6703 6ED9 8A6F 8CC4 9892 50E1 5612 3A39 8AA8 571A
5BED 6167 6193 66B3 69E5 6F53 7623 8527 49E5 5666
5B12 5FBB 6A5E 6BA8 6FAE 6FCA 7369 8559 8AF1 982E
71F4 74AF 7BF2 8588 8589 992F 5696 77BA 7A62 7E62
87EA 3B29 6AD8 7E6A 7FFD 85F1 8B53 5136 93F8 95E0
5B48 942C 9767 8B7F 986A
5C77 7073 74A4 61F3
72BF 6B53 9D05 3E55 9D4D 56BE 61FD 737E 9144 8B99
8C9B 9A69
73AF 5CD8 6D39 72DF 90C7 6853 8341 5BCF 7D59 8408
8411 96C8 7D84 7FA6 8C86 926E 953E 571C 5B1B 5BF0
6FB4 7F33 49A1 961B 74B0 8C72 9370 956E 9E6E 7CEB
7E6F 8F58 9436 95E4 9B1F 74DB 4D09
7F13 4220 7DE9 650C
5E7B 5942 8092 5950 5BA6 5524 6362 6D63 6DA3 70C9
60A3 6899 7115 559A 559B 5D48 610C 63DB 6E19 75EA
7746 902D 3B07 3B0A 7165 744D 8C62 6F36 7613 69F5
9CA9 64D0 6FA3 9BC7 85E7 9C00
6B22 77A3 6B61
660F 662C 8364 5A5A 60DB 6DBD 960D 68D4 6B99 7767
8477 776F 95BD
5FF6 6D51 68A1 9984 581A 6E3E 743F 9B42 991B 7E49
8F4B 4B9D 9F32
9BF6
8BE8 4FD2 5031 5702 638D 6DF7 711D 6EB7 6141 89E8
8AE2
5DDF 3835 8093 8841 8352 671A 5843 614C 3B3B
7687 505F 51F0 55A4 582D 5A93 5D32 5FA8 60F6 6E5F
968D 9EC3 9EC4 697B 714C 745D 845F 9051 953D 58B4
6F62 735A 7BC1 7BCA 824E 8757 71BF 749C 8AFB 7640
78FA 417F 7A54 9360 992D 9CC7 7C27 87E5 97F9 8DAA
9A1C 9404 9C09 5164 9C51 9DEC
6033 604D 70BE 5BBA 6644 595B 8C0E 5E4C 8A64 7180
4420 8B0A 6ACE
6130 3A2A 6EC9 69A5 769D 66C2 93A4 76A9
6643 7E28
53FF 543D 544D 7074 8F70 54C4 8A07 70D8 8EE3 63C8
6E39 7122 7861 8C3E 29420 8F37 85A8 9367 569D 8F5F
4EDC 5F18 5985 7EA2 5430 5B8F 6C6F 7392 7EAE 95F3
5B96 6CD3 57AC 5A02 6D2A 7AD1 427A 7D05 82F0 8679
5CF5 6D64 7D18 7FC3 803E 836D 7854 7D2D 8C39 9E3F
6E31 7AE4 7CA0 921C 958E 7D8B 7FDD 8452 8453 8C3C
6F42 28306 9277 9783 9B5F 92D0 5F4B 9710 857B 971F
9D3B 9EC9 2410F 9ECC
664E 3B34 55CA
8BA7 8A0C 95A7 6494 6F8B 6F92 92BE 95C2 9B28
FDD0-3110
4E0C 22A66 8BA5 51FB 5209 53FD 9965 4E69 520F 673A
7391 808C 573E 77F6 9E21 6785 82A8 54AD 59EB 525E
5527 59EC 5C50 79EF 7B04 8FF9 98E2 57FA 7EE9 559E
5D46 5D47 6567 671E 7284 7B53 7F09 8D4D 52E3 55D8
7578 7A18 8DE1 8DFB 9CEE 50DF 20F3B 6BC4 7B95 9288
5630 69E3 757F 7A3D 7DDD 89ED 8CEB 8EB8 9F51 58BC
6A5F 6FC0 74A3 7A4D 9324 64CA 78EF 79A8 7C0A 7E3E
7F81 8940 8CF7 96AE 6AC5 802D 8E5F 96DE 8B4F 913F
97F2 9D8F 8B64 9416 9951 8E8B 97BF 9DC4 9F4E 7F87
9447 8640 8989 9459 9F4F 7F88 9E04 898A
4EBC 53CA 4F0B 5409 7EA7 5373 5C8C 5F76 5FE3 6781
6C72 7680 4E9F 4F76 8BD8 9491 537D 59DE 6025 72E4
90C6 63E4 75BE 768D 7B08 7D1A 810A 4798 506E 5359
5EB4 3B72 710F 89D9 8C3B 35CA 68D8 6E52 96C6 5849
5AC9 6131 6222 696B 6975 6B9B 8D8C 69C9 8024 818C
84BA 92A1 35F1 6483 6F57 7620 799D 7BBF 8E16 9E61
5DAF 6A76 6FC8 8540 878F 64EE 6A9D 857A 8E50 9353
85C9 894B 8265 7C4D 8F5A 93F6 4CED 9735 9DBA 9DD1
96E6 96E7
51E0 5DF1 4E2E 5980 72B1 6CF2 866E 6324 638E 9C7E
5E7E 39B8 621F 9218 5D74 9E82 9B62 64A0 64E0 7A56
87E3 9B55
2E95 5F50 5F51 65E1 8BA1 8BB0 4F0E 7EAA 5756 5993
5FCC 6280 5242 5B63 82B0 9645 54DC 578D 5CDC 65E2
6D0E 6D4E 7D00 830D 8A08 5264 7D12 7EE7 8324 8360
89CA 8A18 5048 5BC2 5BC4 3831 5F9B 394D 60B8 65E3
689E 6E08 796D 5848 60CE 3ED1 4400 81EE 517E 75F5
7D99 846A 8507 88DA 66A8 6F03 6F08 7A29 7A4A 84DF
8900 8A8B 8DFD 969B 9701 9B3E 9C9A 66A9 7A37 8AC5
9CAB 5180 5291 66C1 7A44 9AFB 568C 6A95 6FDF 7F7D
858A 89AC 6AB5 3E04 85BA 9D4B 9F4C 61FB 7660 7A67
7E4B 9AA5 9BDA 7031 7E7C 860E 29949 9C40 862E 973D
9C36 9C3F 863B 9C6D 9A65
4EBD 8F91 6A2D 8F2F 5EED 766A
52A0 4E6B 5939 4F3D 593E 4F73 62B8 62C1 6CC7 3E62
67B7 6BE0 6D43 73C8 8304 8FE6 57C9 5BB6 6D79 75C2
689C 7B33 801E 8888 50A2 7333 8DCF 20EA2 728C 8175
846D 926B 5609 927F 9553 8C6D 8C91 29D9B 93B5 4552
9E9A
573F 5FE6 6274 90CF 550A 605D 835A 90DF 621B 83A2
94D7 621E 86F1 88B7 988A 3B96 86FA 88CC 8DF2 9782
9904 92CF 982C 9830 9D36 9D4A
7532 4EEE 5CAC 53DA 73BE 80DB 659A 8D3E 94BE 5047
5A7D 5FA6 659D 6935 8CC8 9240 698E 69DA 7615 6A9F
4EF7 9A7E 67B6 22C4C 5AC1 5E4F 69A2 50F9 99D5
7A3C 7CD8
7596 9636 7686 63A5 63B2 23D8F 75CE 79F8 5588 5826
5A98 63ED 6904 6E5D 813B 83E8 8857 968E 55DF 5AC5
716F 40C8 7A2D 8754 64D1 7664 8B2F 9D9B
5369 536A 5B51 5C10 8BA6 8282 5226 5227 52AB 5C8A
523C 351A 52BC 6605 6770 758C 3636 36C3 62EE 6D01
7ED3 5022 6840 8871 8A10 8FFC 507C 5A55 5D28 6377
83AD 5091 55BC 3A17 4039 7D50 7D5C 88BA 9889 5D65
3A29 696C 6976 6ED0 776B 7BC0 8710 874D 8A70 9263
9B5D 622A 69A4 78A3 7AED 26D74 9C92 6F54 7FAF 84F5
8AB1 8E15 978A 5E6F 937B 9B9A 5DC0 6AED 881E 8818
883D
6BD1 5A8E 89E3 89E7 98F7 6A9E
4E2F 4ECB 5424 5C95 3839 5E8E 6212 5C46 5C4A 73A0
82A5 754C 754D 75A5 780E 8BEB 501F 6088 86A7 8878
5FA3 583A 6950 743E 86F6 7297 8AA1 9AB1 9B6A 892F
9385 8EA4
59D0 685D
4EA4 827D 8281 59E3 5A07 5CE7 6D47 90CA 9A84 80F6
832D 832E 6912 7126 86DF 8DE4 50EC 5604 213ED 8660
9C9B 5B0C 5D95 5DA3 618D 6F86 81A0 71CB 81B2 8549
7901 7A5A 9BAB 9D41 9E6A 7C25 87ED 8F47 940E 9DCD
9A55 9DE6 9DEE
3B42 81EB 89D2 4F7C 4FA5 6054 6322 72E1 7EDE 997A
6341 6648 70C4 768E 77EB 811A 94F0 6405 6E6B 7D5E
527F 656B 6E6C 714D 8173 8CCB 50E5 6477 669E 8E0B
9278 9903 510C 528B 5FBA 649F 64B9 5FBC 61BF 657D
657F 71DE 7F34 96A6 66D2 74AC 77EF 76A6 87DC 7E73
8B51 5B42 3A70 27088 652A 705A 9C4E
53EB 544C 5CE4 630D 8A06 73D3 7A8C 8F7F 8F83 654E
6559 7A96 3C3E 6ED8 8F03 5602 5626 65A0 6F16 9175
564D 5DA0 6F50 566D 5B13 7365 85E0 8DAD 8F4E 91AE
8B65 3B2D 76AD 91C2
9D64 6AF5 7E90
4E29 52FC 7EA0 673B 725E 7A76 7CFA 9E20 7CFE 8D73
28CCA 9604 557E 63C2 63EA 841B 63EB 9CE9 644E 6A1B
9B0F 9B2E
4E5D 4E45 4E46 4E63 597A 7078 7396 820F 97ED 7D24
9152 9579 97EE
531B 65E7 3EA9 81FC 548E 759A 4192 67E9 67FE 5003
6344 6855 5313 53A9 6551 5AA8 5C31 5EC4 8205 50E6
5ECF 5ED0 6166 6BA7 9E6B 820A 5336 9BE6 9E94 9F68
9DF2
6C63 6766 6B0D
620B 5978 5C16 5E75 575A 6B7C 95F4 51BF 6214 73AA
80A9 8270 59E6 59E7 517C 76D1 5042 5805 60E4 3B74
730F 7B3A 8C5C 6E54 724B 7F04 83C5 83FA 9593 641B
6937 693E 714E 728D 744A 788A 7F23 844C 8C63 76E3
7777 7B8B 84B9 6A2B 719E 7DD8 9CA3 9CFD 9E63 71B8
2585D 7BEF 7E11 8551 8573 8271 9930 99A2 9E89 7010
97AC 97AF 9CD2 791B 4320 89B8 9D73 7038 9427 6AFC
6BB2 9DBC 97C0 9C39 56CF 8643 946F 97C9
56DD 62E3 67A7 4FED 67EC 5039 6338 6361 7B15 8327
51CF 526A 6898 68C0 583F 63C0 63C3 691C 6E1B 6E55
7751 7877 8A43 950F 5F3F 6695 7450 7B67 7B80 7D78
88E5 8C2B 8DBC 6229 622C 3A35 78B1 5109 7FE6 64BF
6AA2 2372D 8B07 8E47 77BC 7906 7C21 85C6 8947 8949
8B2D 7E6D 9B0B 9E78 703D 8812 9417 9C0E 5297 9E7B
7C5B 8B7E 9E7C 897A
89C1 4EF6 898B 996F 5251 5EFA 6D0A 726E 8D31 4FF4
5263 682B 6DA7 73D4 8230 8350 5065 5271 5FA4 6E10
8C0F 91FC 5BCB 6E85 81F6 88B8 8DF5 65D4 6957 6BFD
249E3 8171 8465 8CCE 9274 952E 50ED 3A34 6997 6F38
528D 528E 3D4E 6F97 7BAD 7CCB 852A 8AD3 8CE4 8D9D
8E10 8E3A 5292 5294 8AEB 92FB 991E 77B7 78F5 85A6
87B9 9373 9375 64F6 6FFA 7E5D 3C04 89B5 93E9 7033
8266 8B7C 8F5E 9431 9451 9452 946C 9473
5F45 58B9 6A7A 7900 6BB1
5DFE 4ECA 65A4 9485 5153 91D1 6D25 77DC 89D4 57D0
73D2 7D1F 8355 887F 60CD 583B 7B4B 91FF 5D9C 9E76
9EC5 895F
4EC5 5C3D 4FAD 537A 5DF9 7D27 5807 83EB 50C5 53AA
8C28 9526 5AE4 5ED1 6F0C 76E1 7DCA 9991 69FF 747E
84F3 5118 9326 8B39 9949
4F12 52A4 52B2 5997 6783 8FD1 8FDB 52C1 6D55 6649
664B 6D78 70EC 8369 8D46 552B 740E 7972 9032 5BD6
6422 6E8D 7981 7F19 9773 5890 3A37 3B10 669C 7468
50F8 51DA 6B4F 6BA3 89D0 5664 3B1C 6FC5 74A1 7E09
8CEE 568D 5B27 3BF2 6FDC 71FC 74B6 85CE 89B2 8D10
9F7D
91D2 781B 743B 58D7
6C5F 59DC 5C06 6D46 7555 8333 8C47 5C07 757A 8441
646A 7FDE 50F5 6F3F 8780 58C3 7F30 6A7F 6BAD 8591
87BF 9CC9 7585 7913 7586 7E6E 97C1 9C42
8BB2 5956 6868 508B 5968 848B 596C 69F3 734E 8199
8523 8029 4243 8B1B 985C
531E 5905 5F1C 6D1A 7EDB 964D 5F36 7D73 88B6 52E5
9171 5D79 647E 6EF0 5F4A 236A6 729F 7CE1 7CE8 91A4
8B3D 91AC
5320 6762 6AE4
5755 5759 5DE0 4EAC 6CFE 7ECF 4EB0 79D4 830E 6D87
834A 5A5B 60CA 65CC 65CD 7304 7D4C 8396 6676 7A09
8148 8346 83C1 7CB3 7D93 844F 5162 7CBE 8059 9CB8
9D5B 9BE8 9D81 9D84 9E96 9F31 9A5A 9EA0
4E95 4E3C 522D 362B 5753 5B91 6C6B 9631 6C6C 80BC
5244 7A7D 9888 666F 981A 5106 5E5C 61AC 66BB 71DB
749F 9838 61BC 74A5 87FC 4718 8B66
598C 51C0 5F2A 5F84 4FD3 205CA 6D44 80EB 8FF3 501E
51C8 5F33 5F91 75C9 7ADE 5A59 5A67 6871 68B7 6DE8
811B 9015 291D5 7ADF 75D9 7AE7 9753 50B9 656C 3B0C
7AEB 9756 5883 734D 8AA9 8E01 9759 477C 975A 955C
975C 66D4 6FEA 701E 93E1 7AF6 7AF8
775B 6A78 71DD
51E5 521F 6285 530A 5C45 62D8 6CC3 72D9 9A79 82F4
6336 75BD 75C0 7717 7820 7F5D 5A35 5A6E 5D0C 63AC
68AE 6DBA 9671 6910 741A 8152 83F9 8D84 8DD4 9514
96CE 824D 871B 88FE 8E18 8E19 92E6 99D2 9B88 9D21
97A0 97AB 9D8B
26951 5C40 3632 6CE6 4FB7 72CA 6854 6BE9 5579 5A45
6DD7 7117 6908 6BF1 6E68 83CA 90F9 7291 8F02 50EA
7CB7 8DFC 8ACA 8D9C 8EB9 95B0 6A58 6A8B 99F6 4CD4
9D59 8E6B 9D74 5DC8 9DAA 861C 9F33 9A67
5480 5F06 6CAE 4E3E 6319 8392 6907 6989 6998 7B65
9F83 849F 8065 8209 8E3D 64E7 9F5F 6AF8 6B05
53E5 5DE8 4E6C 5DEA 8BB5 5177 59D6 5CA0 6007 6010
601A 62D2 62E0 6D30 90AD 661B 6B6B 70AC 82E3 949C
4FF1 5028 5036 51A3 5267 79EC 57E7 57FE 60E7 636E
7C94 801F 86B7 8893 728B 8A4E 8DD9 8DDD 98D3 8C66
9245 952F 5BE0 6133 7AAD 805A 8661 5287 52EE 5C66
8E1E 99CF 58C9 61C5 64DA 6FBD 7AB6 422E 92F8 9B94
5C68 907D 98B6 8C97 7C34 8E86 91B5 61FC 943B 3B2C
77E9 7220 8977
5658 6485 64A7 5C69 8E7B
4E85 2010C 5B52 5B53 51B3 5214 6C12 8BC0 6289 6C7A
5F21 3B48 6CEC 73A6 73A8 82B5 6317 73CF 75A6 7804
7EDD 8673 89C9 5014 6354 6B2E 8697 5D1B 6398 658D
6877 6B8C 8990 89D6 8A23 8D7D 8DB9 4848 5095 53A5
3B7E 7133 7D55 7D76 899A 8D89 902B 920C 37F2 5282
3522 52EA 7474 8C32 99C3 5DA5 61B0 71A6 7234 7357
761A 9D02 9D03 5671 61A0 6A5B 6A5C 855D 8568 7235
81C4 9562 87E8 87E9 5C6B 8B4E 8E76 8E77 9D8C 5337
56BC 7211 77CD 89BA 940D 941D 721D 89FC 5F4F 6204
652B 7383 9DE2 6B14 77E1 9FA3 28C1C 8C9C 8EA9 9481
59E2 5A1F 6350 6D93 7106 8127 74F9 9E43 88D0 52EC
48FA 954C 93B8 9D51 942B 8832
5377 545F 5E23 57CD 6372 83E4 9529 3DF7 9308 81C7
52B5 5946 5DFB 5026 52CC 684A 72F7 7EE2 96BD 6DC3
7737 774A 7D6D 7F65 45AD 9104 7760 7D79 96CB 98EC
21840 617B 8528 990B 7367 7E33 7F82
519B 208D5 541B 5747 36AC 6C6E 59F0 8ECD 94A7 8690
8880 687E 76B2 8399 83CC 921E 7885 76B8 76B9 89A0
9281 929E 9CAA 9E87 9355 9BB6 9E8F 9E95
5441 4FCA 3F59 57C8 5CFB 6343 6D5A 90E1 9656 9982
9A8F 35A5 6659 710C 73FA 2797A 347A 68DE 756F 7AE3
3766 41F9 7B98 7B9F 8720 28318 5101 3559 9915 5BEF
61CF 71C7 6FEC 99FF 9D54 9D58 6508 4551 651F
2E86 5182 518B 5770 6243 57DB 7D45 99C9 99EB 860F
8614
518F 56E7 38A0 6CC2 7085 4FB0 70AF 8FE5 6D7B 9008
70F1 489B 7A98 988E 715A 7D97 50D2 715B 71B2 6F83
8927
FDD0-3111
4E03 6C8F 8FC9 59BB 67D2 501B 51C4 6816 6864 5A38
60BD 687C 6DD2 90EA 6532 671F 68F2 6B3A 3C66 840B
86E3 50DB 5601 617D 69BF 6F06 7DC0 617C 69ED 25250
8AC6 8AFF 970B 45E9 8E4A 3B24 9B4C 4953 93DA 9D88
4E93 9F50 573B 5C90 5C93 5FEF 4E9D 5176 5947 6589
6B67 7541 7941 80B5 82AA 4FDF 75A7 7947 7948 7AD2
5258 658A 65C2 8006 8110 8691 8694 869A 9880 57FC
5D0E 5E3A 6391 6DC7 7309 7566 8DC2 8EDD 91EE 9A90
9A91 68CA 68CB 7426 742A 8401 8415 86F4 612D 7881
7895 797A 44C5 951C 980E 65D7 7CB8 7DA5 7DA6 7DA8
44EB 871D 871E 9B3F 9F4A 7482 8E11 79A5 8572 9321
9CAF 61E0 6FDD 2A5C6 6AB1 6AC0 24A8C 81CD 85C4 9A0E
9A0F 9CCD 4C07 9BD5 9D78 9D80 9E92 7E83 8269 8604
8810 9B10 9C2D 7382 9EA1
4E5E 4F01 5C7A 5C82 9094 542F 5447 675E 7398 76C0
8291 5518 8C48 8D77 5553 5554 5A4D 555F 7EEE 6675
68E8 410E 7DAE 7DBA 8AEC 95D9
6C14 8BAB 5FD4 6C17 6C54 5F03 6C7D 77F5 8FC4 546E
6CE3 7081 76F5 829E 54A0 5951 780C 6814 6B2B 6C23
8A16 552D 5921 68C4 6E46 6E47 789B 847A 6456 66A3
78B6 5650 6187 7508 27858 5668 61A9 78DC 78E7 78E9
468D 7F4A 87FF 9F1C
7F3C 621A 6E0F 7DD5 8904 87A7 7C2F 7C31 7C4F
6390 845C
62E4
8DD2 9160
5736 51BE 34E4 5E22 6070 6D3D 6B8E 7848 6118 9AC2
9790
767F 21D90 49A7 807A
4E14
5207 59BE 602F 5327 7A83 90C4 20CB5 608F 6308 6D2F
3932 60EC 6DC1 7B21 611C 86EA 3966 6705 7BA7 7DC1
9532 21424 7BCB 8E25 7A55 9365 85D2 9BDC 9411 7ACA
2383A
82C6 503F 5AAB 7C61
6084 7857 90FB 5D6A 8DF7 5281 6572 6BC3 8E0D 9121
9125 9539 589D 981D 58BD 5E67 22FE3 6A47 71C6 7F32
9AB9 78FD 936B 936C 7E51 8DAC 8E7A 9430
4E54 4FA8 6865 834D 835E 785A 55AC 83EC 50D1 8C2F
563A 5AF6 376F 6194 9792 6A35 6A4B 854E 7644 4009
77A7 7904 4397 8DAB 85EE 9408 97BD 9866 2710D
5DE7 91E5 6100 9ADC
4FCF 8BEE 5CED 5E29 7A8D 9657 6BBB 7FD8 8A9A 9ADA
50FA 64AC 64BD 9798 97D2 7AC5 7FF9 8B59 8E88 3681
69D7 729E
3400 4E18 4E20 20C0B 5775 36B1 3CCB 90B1 6058 79CB
79CC 86AF 5A9D 6978 8429 9E59 7BCD 7DE7 84F2 8775
7A50 8DA5 4A02 9CC5 87D7 97A6 97A7 9C0C 9C0D 9D96
8824 9F9D
53F4 56DA 624F 72B0 738C 6C53 808D 6C42 866C 6CC5
42B5 866F 4FC5 89D3 8A04 8A05 914B 91D3 5512 3797
3939 3B5D 6D57 7D0C 91DA 6882 6B8F 6BEC 7403 838D
8D47 900E 9011 5D37 5DEF 3962 6E1E 6E6D 76B3 76DA
5DF0 716A 7D7F 86F7 88D8 9052 89E9 8CD5 7486 8764
92B6 9194 9B82 9F3D 9BC4 9C3D
641D 7CD7
91FB 8612
5343 4EDF 5731 5732 5977 6266 6C58 9621 4F65 5C8D
6744 6C67 828A 8FC1 6B26 74E9 81E4 833E 948E 62EA
7275 7C81 515B 60AD 8C38 94C5 5A5C 5B6F 727D 91FA
6394 8688 8C26 9206 96C3 50C9 6106 7B7E 925B 9A9E
9E50 6173 6434 6481 7B9E 8AD0 9077 8930 8B19 9845
6AB6 6510 6ACF 7C3D 9D6E 5B45 6513 9A2B 6511 9B1D
9B1C 3A77 7C64 97C6
4EF1 5C92 5FF4 6272 62D1 524D 94A4 6B6C 8654 94B1
94B3 63AE 8EE1 5A8A 63F5 9210 976C 9257 5898 69A9
7B9D 92AD 6F5B 6F5C 7FAC 6A6C 8541 9322 9ED4 9EDA
6FF3 9A1A 9A1D 704A 9C2C
51F5 80B7 6D45 6DFA 8125 55DB 5D70 69CF 8181 8738
9063 8C34 7F31 7E7E 8B74
6B20 520B 3438 82A1 4FD4 5029 6093 831C 5811 5094
5D4C 68C8 6920 614A 7698 5879 371E 37FB 6B49 7DAA
84A8 69E7 7BCF 8533 8F24 5119 7BDF 58CD 7E34 4945
9C1C
7ACF 9386 93F2 7C56 9453
4EB2 4FB5 94A6 887E 9A8E 5A87 5D5A 6B3D 7D85 8A9B
5D94 89AA 9849 99F8 9BBC 5BF4
3626 5E88 3A81 82A9 82B9 57C1 73E1 79E6 8039 8699
6366 7434 7439 83E6 83F3 9219 96C2 52E4 55EA 5AC0
6EB1 79BD 9772 616C 5659 65B3 9CF9 61C4 3A52 64D2
6FBF 763D 8793 61C3 6A8E 9B35 8804 9D6D
5745 6611 7B09 68AB 8D7E 5BD1 9513 5BDD 5BE2 92DF
87BC
5422 5423 3908 628B 6C81 551A 63FF 83E3 29088 6407
64B3 421C 7019 85FD
545B 6215 6217 65A8 67AA 73B1 7F8C 7F97 7310 8DC4
690C 8154 55C6 6EAC 9516 5D88 6227 69CD 7244 7472
7FAB 8723 9535 7BEC 9306 8B12 8E4C 956A 8E61 9397
93D8
4E2C 5F37 5F3A 5899 5AF1 6F12 6A2F 8503 8537 58BB
5B19 5EE7 6AA3 7246 8594 8262 8620
62A2 7F9F 6436 7FA5 588F 7E48 8941 7E66 93F9
709D 5534 7197 7FBB
55F4 7347
72C5 9751 9752 6C22 8F7B 503E 537F 570A 57E5 5BC8
6C2B 6DF8 6E05 90EC 50BE 873B 8F15 4A1D 9CAD 944B
591D 7520 5260 52CD 60C5 6B91 6674 68FE 6C30 6692
845D 6A08 64CF 64CE 3BF3 6AA0 9EE5
9877 82D8 8BF7 5EBC 9803 5ECE 6F00 3DEB 8ACB 6ABE
5E86 51CA 6385 6BB8 7883 7B90 9758 6176 78D8 78EC
7F44 8B26
7858 6AE6
533A 66F2 4F39 4F49 5324 5C96 8BCE 9A71 5765 5C48
5CA8 5CB4 62BE 9639 3B55 6D40 80E0 795B 5340 7D36
86C6 88AA 8EAF 7B41 7CAC 86D0 8A58 8D8B 5D87 6188
420C 99C6 657A 8AB3 957C 99C8 9EB9 9AF7 9B7C 8DA8
9EAF 89B0 8EC0 9EB4 9EE2 89BB 9A45 9C38 9C4B
4F62 52AC 65AA 6710 80CA 9E32 6DED 7D47 7FD1 6E20
83C3 8EE5 844B 24A0E 7496 8556 9D1D 74A9 78F2 87B6
77BF 87DD 9F29 3739 5FC2 7048 8627 6235 6B0B 6C0D
81DE 766F 7C67 8837 8862 8EA3 883C 947A 9E1C
53D6 7AD8 5A36 8A53 7AEC 877A 9F8B 9F72
53BA 53BB 521E 547F 3C26 551F 801D 9612 89D1 8DA3
95B4 9EAE 95C3 9F01 89B7
8FF2 8850
7F3A 9619 849B
7638
5374 537B 57C6 5D05 5BC9 60AB 96C0 7437 785E 786E
9615 5859 6409 76B5 788F 6128 69B7 58A7 6164 78BA
78BB 8D9E 71E9 95CB 7910 95D5 704D 792D
9E4A 9D72
5CD1 5F2E 606E 609B 5708 570F 68EC 99E9 9409
34B0 5168 6743 4F7A 8BE0 59FE 6CC9 6D24 62F3 7277
8343 8F81 5573 57E2 5A58 60D3 75CA 7842 94E8 6E76
7288 7B4C 7D5F 643C 7454 8472 89E0 8A6E 8DE7 8F07
8737 9293 6A29 8E21 7E13 919B 9CC8 9B08 9A21 9C01
5B49 5DCF 9F64 6B0A 98A7 8838 9874
21FE8 72AC 6C71 754E 70C7 7EFB 7DA3 8647
529D 5238 7276 52E7 97CF 52F8
72AD 6926 697E 95CE
590B 56F7 5CEE 9021
5BAD 5E2C 3A8A 7FA3 7FA4 88D9 88E0
828E 5314
536D 5B86 909B 7A77 7A79 8315 684F 4143 7B3B 8D79
60F8 712A 712D 743C 7B47 823C 86E9 86EC 21983 7162
7758 8DEB 928E 778F 7AAE 511D 618C 6A69 749A 74CA
7AC6 85D1 85ED 74D7
718D
FDD0-3112
5915 516E 5FDA 6C50 8980 5438 5E0C 6271 6278 5365
6614 6790 7A78 80B8 80B9 4FD9 5F86 6038 6053 997B
550F 595A 36D3 5C56 6095 6C25 6D60 727A 72F6 90D7
553D 6089 60DC 637F 665E 6878 6B37 6DC5 70EF 7101
7108 740B 7852 8383 8D65 91F8 5092 60C1 6670 6673
3C64 711F 712C 7280 774E 7A00 7C9E 7FD5 823E 83E5
5380 5D60 5FAF 6EAA 7699 910E 9521 50D6 69BD 7155
7184 7188 7199 7DC6 84A0 8725 8C68 990F 563B 564F
5B06 5B09 6F5D 3FB7 761C 78CE 819D 51DE 20615 5DB2
6199 3B1B 6A28 6A40 71B9 71BA 71BB 7AB8 7E18 7FB2
8785 8787 932B 3C46 71E8 406F 77A6 87CB 8C3F 8C40
8C6F 8C95 7CE6 7E65 96DF 9D57 8B46 91AF 93ED 5DC7
66E6 7214 72A7 89F9 96B5 9145 89FD 9F37 8835 9E02
89FF 9474
4E60 90CB 5E2D 7FD2 88AD 89CB 5AB3 693A 3804 5D8D
6F1D 84B5 84C6 89A1 8D98 69E2 3A57 6A84 8582 96B0
8B35 93B4 972B 9CDB 98C1 9A31 9A3D 8972 9C3C 9A68
67B2 6D17 73BA 5F99 94E3 559C 9222 8448 8478 9268
9269 5C63 6F07 6198 84F0 66BF 6B56 8AF0 58D0 79A7
7E30 8B11 87E2 8E5D 74BD 56CD 9C5A 77D6 8EA7
5338 534C 620F 5C43 7CFB 9969 546C 5FE5 602C 77FD
7EC6 4FC2 54A5 6044 76FB 3938 3B61 6B2F 7EE4 90E4
7D30 91F3 960B 55BA 691E 7FD6 8203 8204 8D87 6140
6ECA 7D8C 8D69 588D 7182 7294 798A 7A27 9699 969F
6F5F 89A4 6231 6F99 422A 856E 9ED6 6232 78F6 4B8E
8669 993C 9B29 7E6B 56B1 95DF 3E0D 973C 5C6D 884B
897F 606F 6E13 6A72 72A0 7902 9BD1
8672 75A8 867E 3520 8C3A 5084 9595 7146 7175 98AC
271CD 778E 8766 9C15
5323 4FA0 72CE 4FE0 5CE1 67D9 70A0 72ED 5CFD 70DA
72F9 73E8 965C 7856 796B 7FC8 823A 35C7 7864 967F
656E 6687 7455 7B6A 821D 9050 78AC 8F96 78CD 7E00
7E16 8D6E 9B7B 8578 8F44 935C 971E 938B 9EE0 9A22
9DB7
959C
4E05 4E0B 4E64 5413 759C 590F 3648 7771 5687 61D7
7F45 5913 93BC 93EC
5737 68BA 6E8A
4E9B 3C54 63F3 7332 6954 6B47 874E 880D
52A6 534F 65EA 90AA 5354 80C1 57A5 604A 62F9 631F
594A 5CEB 633E 8105 8107 887A 5055 659C 3B68 8C10
35BF 7FD3 55CB 6136 643A 744E 7D8A 26CC3 3666 7181
818E 52F0 64B7 7DF3 7F2C 8762 978B 9821 64D5 8AE7
71F2 3A66 64F7 97B5 651C 7E88 896D 8B97 9FA4
5199 51A9 5BEB 85DB
4F33 707A 5378 6CC4 6CFB 7EC1 7F37 6D29 70A7 7944
5A0E 5C51 5C53 22B3F 505E 5070 5368 5FA2 68B0 70F2
710E 7D32 4EB5 5A9F 5C5F 6E2B 245C8 79BC 7D4F 7D6C
8C22 50C1 586E 698D 69AD 3D3D 5C67 66AC 7DE4 8909
5667 5DB0 5EE8 61C8 6FA5 736C 7CCF 27735 97F0 71EE
85A2 85A4 893B 8B1D 9082 99F4 7009 97A2 7023 7215
424F 7E72 87F9 880F 9F58 9F5B 9F65 9F42 8E9E
810B 5911
7071 7072 547A 67AD 4FBE 54D3 67B5 9A81 54EE 5BAF
5BB5 5EA8 6D88 7EE1 8653 9E2E 5A4B 689F 7107 7307
900D 75DA 75DF 785D 7863 7A99 8427 9500 63F1 7D83
7FDB 8437 560B 5610 6B4A 6F47 7BAB 8E03 5635 5F47
61A2 3A4B 7362 92B7 9704 81AE 856D 9D1E 7A58 7C18
87C2 87CF 9B48 9D35 56A3 7C2B 85C3 87F0 701F 6AF9
24473 9AC7 56BB 56C2 8828 9A4D 9AD0 6BCA 8648
6D28 3B35 7B05 5D24 6DC6 8A24 90E9 6BBD 7B4A 8AB5
2E8C 2E8D 5C0F 6653 6681 7B71 7B7F 769B 66C9 7BE0
76A2 8B0F
5B5D 8096 52B9 54B2 4FF2 6548 6821 6D8D 7B11 5578
509A 6569 35DB 8A68 5628 8A9F 562F 6B57 71BD 97A9
6585 6586 4567
6077 6EE7
4F11 4FE2 54BB 5EA5 3CDC 4FEE 70CB 70CC 7F9E 8119
8129 9E3A 81F9 8C85 9990 929D 9AE4 6A07 9AF9 9380
9D42 93C5 9948 9C43 98CD
82EC
673D 7D87 6EEB 7CD4
79C0 5CAB 5CC0 73DB 7EE3 7407 8896 9508 55C5 6EB4
7493 890E 890F 92B9 8791 7E4D 7E61 93E5 93FD 9F45
9BB4
353E 4EDA 5C73 5148 597E 7EA4 4F61 5FFA 6C19 6774
79C8 67AE 7946 7C7C 82EE 73D7 6380 83B6 8A2E 94E6
23519 50CA 8DF9 9170 9528 5615 929B 9C9C 97EF 5B10
61B8 66B9 9341 859F 893C 97F1 9BAE 8E6E 99A6 4D4C
5EEF 6515 7E8E 9DB1 8E9A 7E96 8973 9C7B
4F2D 95F2 59B6 5F26 8D24 54B8 6326 80D8 5A34 550C
5563 5A39 5A71 6D8E 7D43 8237 86BF 8854 3B79 75EB
86DD 9591 9592 9E47 5ACC 8858 929C 5AFA 5AFB 61AA
648F 6F96 7509 7A34 8AB8 8CE2 3BD7 3D6A 71C5 8AF4
8F31 918E 7647 764E 77AF 85D6 3C0A 7925 4554 9E79
9E99 8D12 9DF3 9DF4 9DFC
51BC 72DD 39E5 663E 5D04 3B60 6BE8 70CD 7303 86AC
9669 8D7B 967A 7B45 5C1F 5C20 641F 8DE3 3B0E 7992
9291 7BB2 5DAE 736B 96AA 736E 934C 71F9 85D3 9855
5E70 6507 6AF6 8B63 861A 4606 496A 7381 97C5 986F
7066
4F23 53BF 5C98 2070E 73B0 7EBF 81FD 82CB 549E 59ED
5BAA 770C 9650 54EF 57B7 5A0A 5A28 5CF4 3A87 6D80
9665 665B 73FE 784D 83A7 9677 9985 774D 7D64 7F10
7FA1 732E 7CAF 7FA8 817A 8706 49DF 50E9 50F4 21416
7DAB 8AA2 648A 7DDA 92E7 61B2 3B17 6A4C 7E23 930E
9921 58CF 8C4F 493C 9EB2 7017 81D4 737B 7CEE 9F38
4ED9 50F2 7E4A 9466
5FC3 59A1 5FFB 8F9B 90A4 6615 677A 6B23 7098 82AF
76FA 4FFD 3B62 60DE 8A22 920A 950C 65B0 6B46 5EDE
92C5 5B1C 85AA 99A8 946B 99AB
6794 3726 8951 9414
4F08
3430 4F29 56DF 9620 5B5E 3B44 248F2 4FE1 8ED0 812A
8845 8A2B 712E 7161 99B8 820B 9856 91C1
5FC4 567A
4E61 8297 76F8 9999 53A2 554C 90F7 5EC2 6E58 7F03
9109 910A 7A25 8459 9115 7BB1 7DD7 81B7 858C 8944
5FC0 9AA7 9E98 6B00 74D6 9576 9472 9A64
4F6D 74E8 8BE6 5EA0 6819 7965 7D74 7FD4 8A73 8DED
4EAB 4EAF 54CD 9977 6651 98E8 60F3 9284 9909 9C9E
66CF 9B9D 8801 9BD7 97FF 9957 995F 9C76
5411 59E0 5DF7 8683 9879 73E6 5842 7F3F 8856 8C61
9805 842B 50CF 52E8 9297 5D91 6A61 95C0 56AE 87D3
8950 940C 9C4C
697F 9C5C
661F 57B6 9A8D 60FA 7329 714B 7446 8165 86F5 89EA
7BB5 7BC2 9B8F 66D0 89F2 935F 9A02 76A8 9BF9
5211 884C 5F62 90A2 4F80 9649 578B 6D10 90C9 9498
5A19 8365 9658 784E 94CF 9203 6ECE 9276 9292 92DE
3A18 7772 9192 64E4
5174 674F 59D3 5E78 6027 5016 8347 5A5E 60BB 6DAC
8395 7DC8 8208 5B39 81D6
54D8 88C4 8B03
5401 620C 65F4 759E 76F1 6B28 80E5 987B 6647 8A0F
987C 8657 865A 8C1E 5AAD 5E41 63DF 6E51 247E0 865B
88C7 9808 6948 7AA2 980A 5618 9700 5653 589F 5B03
7E03 8751 9B46 6B54 8566 8ADD 8B43 7E7B 9450 9A49
9B1A 9B56
4FC6 5F90 84A3
8BB8 5474 59C1 8BE9 5194 6829 73DD 5066 8A31 668A
8A61 7A30 7CC8 9126 9191 76E8
65ED 4F35 5E8F 6C7F 4F90 5379 6034 6C80 82A7 53D9
6064 662B 6D2B 460F 57BF 6B30 6B88 70C5 73EC 52D6
654D 6558 52D7 70FC 7EEA 7EED 9157 55A3 58FB 5A7F
6702 6E86 7D6E 8A39 6149 7166 8CC9 69D2 6F35 6F4A
76E2 7781 7DD2 805F 84C4 928A 735D 7A38 7DD6 9B63
77B2 85C7 85DA 7E8C 9C6E
8053 7D9A 84FF
524A 75B6 3EE1 8486 9774 35FE 8FA5 8FAA 859B 97BE
7A74 6588 4E74 5B66 5CA4 5CC3 6CF6 8313 9E34 8895
8E05 58C6 5B78 5DA8 6FA9 71E2 3D85 89F7 96E4 9DFD
96EA 9CD5 9C48
8840 3570 5437 5779 72D8 6856 8C11 8D90 8B14 7025
81A4 6A30 825D 8F4C
5405 8F69 660D 5BA3 5F32 8ED2 688B 8C16 55A7 5847
5A97 6103 610B 63CE 6684 714A 7444 8431 8432 777B
84D2 5107 7BAE 7E07 7FE7 8756 92D7 61C1 79A4 8AE0
8AFC 857F 9379 99FD 7FFE 8809 77CE 85FC 8610 8B5E
7384 73B9 3623 75C3 60AC 65CB 7401 8701 5AD9 6F29
66B6 7487 4062 6A88 74BF 61F8
54BA 6645 70DC 9009 3535 9078 9848 7663 766C
6030 6CEB 39E6 6621 70AB 7EDA 7729 94C9 7404 7734
8852 88A8 6E32 7D62 6965 6966 9249 78B9 8519 955F
9799 98B4 7E3C 7E4F 93C7 8B82 8D19
9C1A
5743 52CB 57D9 7104 52DB 5864 718F 7AA8 52F2 8512
52F3 99E8 58CE 736F 85AB 66DB 71FB 81D0 85B0 77C4
7E81 860D 58E6 91BA
5BFB 5EF5 65EC 9A6F 674A 5DE1 7543 8BE2 5CCB 6042
6D35 6D54 7D03 6812 686A 6BE5 73E3 8340 8368 5071
3704 5C0B 5FAA 63D7 295B0 8A62 99B4 69C6 6F43 9C9F
565A 3D4C 6F6F 9129 6533 6A33 71D6 7495 87F3 9C4F
9C58 7065
5342 8BAF 4F28 6C5B 8FC5 4F9A 5DFA 5F87 72E5 6B89
8A0A 8A19 8FFF 900A 595E 5DFD 6BBE 7A04 613B 8CD0
905C 5640 6F60 8548 9D55 720B 9868 9442
8BAD 8A13 5691
51F6 5144 342B 5147 5308 8BBB 5FF7 6C79 54C5 605F
6D36 80F7 80F8 8A29 8A7E 8CEF
96C4 718A
713D
8BC7 7138 8A57 5910 657B 22FCC
FDD0-3113
4E4B 652F 536E 6C41 5431 5DF5 6C65 5767 679D 6CDC
77E5 7EC7 80A2 829D 6800 79D3 79D6 80D1 80DD 5001
75B7 7957 79EA 8102 887C 96BB 6894 796C 6925 81F8
6220 6418 7A19 7D95 69B0 7994 8718 99B6 3BC4 9CF7
9D32 9D44 7E54 9F05 8635
6267 4F84 59B7 76F4 59EA 5024 503C 8040 91DE 57F4
57F7 21E1C 6DD4 804C 8CAD 690D 6B96 7286 7D77 8901
8DD6 55ED 74E1 7983 9244 588C 646D 99BD 5B02 6179
6F10 8E2F 6A34 81B1 5128 7E36 8077 87D9 8E60 8EC4
8E91
5902 6B62 53EA 52A7 65E8 5740 5741 5E0B 627A 6C66
6C9A 7EB8 962F 603E 62A7 23CB5 82B7 54AB 6049 6307
67B3 6D14 780B 7949 8F75 75BB 41DB 7D19 8879 6DFD
8A28 8DBE 8EF9 4920 9EF9 916F 85E2 8967
81F3 9624 5FD7 5FEE 627B 8296 8C78 28468 5236 5394
5781 5E19 5E1C 6CBB 7099 8D28 5CD9 5EA2 5EA4 6303
67E3 6809 6D37 81F4 8FE3 90C5 5A21 5F8F 2267A 631A
664A 684E 72FE 7951 79E9 8D3D 8F7E 965F 4E7F 506B
202B7 5F9D 3A01 63B7 68BD 6956 7318 7564 75D4 79F2
79F7 7A92 7D29 7FD0 889F 88A0 89D7 94DA 9E37 5082
21376 5D3B 5F58 667A 6EDE 75E3 86ED 8EFD 9A98 5BD8
5ECC 6431 6ECD 7A1A 7B6B 7F6E 8DF1 8F0A 9527 96C9
5886 21839 6EEF 6F4C 7590 43BA 88FD 899F 8A8C 928D
5E5F 6184 646F 71AB 7A3A 7DFB 442D 81A3 89EF 8CEA
8E2C 92D5 64F3 65D8 7004 99E4 9D19 5295 61E5 6ADB
7A49 4245 87B2 3609 3731 61EB 64F2 8D04 6ACD 74C6
89F6 9BEF 7929 8C51 9A2D 9A3A 9A47 8E93 9DD9 9455
8C52
51EA 4FE7 5F94 8B22
624E 5412 62AF 5953 6313 67E4 67FB 54F3 5067 55B3
63F8 6E23 6942 5284 6463 76B6 6A1D 89F0 76BB 8B47
9F44 9F47
672D 7534 95F8 86BB 94E1 7160 7250 9598 7B9A 802B
9358 8B57
538F 62C3 82F2 7728 781F 6429 9C8A 9C9D 8E37 9B93
9BBA 4562
4E4D 7079 8BC8 54A4 67DE 6805 70B8 5BB1 75C4 86B1
8A50 643E 6EA0 69A8 27A0A 9705 91A1
8707 55FB 5AEC 906E
5387 6298 6B7D 77FA 7813 7C77 8674 54F2 57D1 3B59
7C8D 5560 608A 6662 6663 3B6F 88A9 8F84 5586 86F0
8A5F 41FD 8C2A 99B2 647A 8F12 78D4 8F19 92B8 8F99
87C4 569E 8B2B 8B3A 9BBF 8F4D 8B81 8B8B
8005 4E7D 556B 9517 7987 8D6D 8936 8975
8FD9 67D8 6D59 6DDB 9019 6A1C 6F6A 9E67 87C5 4826
9DD3
7740 8457 8517
635A 658B 658E 6458 69B8 451D 9F4B 29C73
5B85 6AA1
7A84 9259
503A 7826 50B5 5BE8 7635
5908 7C82
4F4B 948A 59B1 5DF6 62DB 662D 76C4 91D7 5541 924A
99CB 7ABC 9363 76BD
722A 627E 6CBC 7475 4203
53EC 5146 8BCF 679B 5797 70A4 72E3 8D75 7B0A 8081
65D0 68F9 436E 8A54 7167 7F69 4207 8087 8088 8D99
66CC 71F3 9BA1 6AC2 77BE 7F84
2EA5 722B 7F40
5DDE 821F 8BCC 4F9C 5468 6D32 8BEA 70D0 73D8 8F80
5F9F 3A04 63AB 6DCD 77EA 90EE 9E3C 558C 7CA5 8D52
9031 8F08 44DF 9282 8CD9 8F16 970C 76E9 8B05 9D43
9A06 8B78
59AF 8F74 8EF8
8098 759B 666D 83F7 776D 7B92 9BDE
7EA3 4F37 546A 5492 5B99 7EC9 5191 54AE 663C 7D02
80C4 76B1 836E 914E 665D 7C99 3473 8A4B 8464 8A76
7503 50FD 76BA 99CE 5663 243F2 7E10 9AA4 7C40 7C55
7C52 9A5F
5E1A 70BF 99F2
6CBE 6BE1 65C3 6834 7C98 86C5 98E6 60C9 8A40 8D88
8A79 959A 8C35 5661 5DA6 9711 6C08 6C0A 859D 9085
77BB 9E6F 65DC 8B6B 9958 9CE3 9A59 9B59 9C63 9E07
8B9D
65A9 98D0 5C55 76CF 5D2D 65AC 382D 692B 7416 370A
640C 76DE 5D83 5D84 6990 98AD 5AF8 9186 6A4F 406A
8F3E 28AC0 27B06 2217A 9EF5
5360 4F54 6218 6808 685F 7AD9 5061 7EFD 68E7 6E5B
83DA 6226 7DBB 5D98 8F1A 6230 8665 8666 89B1 8F4F
8B67 29947 9A4F 8638
8D1E 3630 9488 4FA6 6D48 73CD 73CE 80D7 8C9E 3598
5E2A 6815 6862 771E 771F 7827 91DD 5075 686D 796F
9159 5BCA 5AC3 21EC8 6438 659F 6968 7349 8474 9049
9241 9755 699B 6B9D 7467 7504 78AA 798E 8496 84C1
6F67 799B 7BB4 6A3C 6FB5 81FB 9331 8F43 937C 85BD
7C48 9C75
3431 8BCA 62AE 6795 5F2B 6623 8F78 5C52 755B 75B9
7715 7D3E 8044 8897 8A3A 8EEB 7D7C 7F1C 88D6 4AB4
7A39 99D7 7E25 9B12 9EF0
5733 7EBC 9635 753D 4FB2 630B 42B6 9E29 632F 6715
681A 7D16 9663 7739 8D48 9156 63D5 5866 7471 8AAB
8CD1 47F4 6576 9547 9707 9D06 93AD 93AE
8419 92F4
5F20 5F35 7AE0 50BD 5887 5ADC 5F70 22554 615E 6F33
7350 7CBB 9123 66B2 6A1F 24364 748B 8501 9067 9926
87D1 9A3F 9C46 9E9E
4EC9 957F 9577 6DA8 638C 6F32 7903
4E08 4ED7 6259 5E10 6756 80C0 8D26 5E33 6DB1 8139
75EE 5D82 5E5B 969C 8CEC 762C 7634 7795
7C80 5E65 93F1 9423
4E89 4F42 59C3 5F81 6014 722D 8BE4 20C9C 5CE5 6323
70A1 72F0 70DD 7710 94B2 57E9 5D1D 5D22 6399 7319
7741 8047 28713 94EE 5A9C 63C1 7B5D 3B39 5FB0 775C
9266 5FB4 84B8 7B8F 5FB5 8E2D 7BDC 931A 9B07 9BD6
7665
6C36 628D 7CFD 62EF 639F 6678 6138 649C 6574
6B63 8BC1 5E27 653F 90D1 75C7 5E40 8A3C 5863 40BB
27D84 4223 8ACD 912D 9D0A 8B49 496D
51E7
6731 52AF 4F8F 8BDB 6D19 90BE 682A 73E0 8331 8BF8
732A 7843 79FC 94E2 7D51 86DB 88BE 8A85 8DE6 69E0
876B 9296 6A65 6F74 8AF8 8C6C 99EF 9BA2 9D38 7026
6AEB 6AE7 9BFA 9F04 8829
7AF9 6CCF 7AFA 70A2 7B01 70DB 7A8B 833F 7B1C 8233
9010 7603 7BC9 71ED 880B 8E85 9C41 5B4E 705F 66EF
6B18 7225 883E
4E36 4E3B 20C0D 5B94 62C4 7F5C 6E1A 967C 716E 8A5D
7151 5631 6FD0 9E88 77A9 529A 56D1 65B8 77DA
4F2B 4F47 4F4F 52A9 7EBB 577E 677C 6CE8 82CE 8D2E
8FEC 9A7B 58F4 67F1 6BB6 70B7 75B0 771D 782B 795D
7ADA 7969 7D35 7D38 7F9C 8387 86C0 5D40 7B51 8A3B
8CAF 8DD3 8EF4 94F8 7B6F 9252 99B5 7BB8 7FE5 6A26
92F3 99D0 7BEB 9714 9E86 9444
58B8
6293 6A9B 81BC 9AFD 7C3B
5353 62D9 70AA 502C 6349 684C 68C1 6DBF 68F3 7A5B
25F1A 4B93 7A71 883F
5734 5F74 6C4B 72B3 707C 53D5 59B0 65AB 6D4A 8301
4E35 6D5E 8BFC 914C 5544 5545 5A3A 39FB 3B6C 68B2
70F5 65B1 666B 6913 7438 4401 251CD 787A 7AA1 7F6C
65B2 69D5 64AF 64C6 799A 4175 8AC1 8AD1 92DC 6FC1
7BE7 289E7 64E2 6580 65B5 6FEF 243F8 6AE1 8B36 956F
9D6B 7042 8817 942F 9432 7C57 9DDF 7C71
5285
7AA7
62FD
8DE9
96B9 8FFD 9A93 9525 9310 9A05 9D7B
6C9D
5760 6858 7B0D 5A37 60F4 7F12 7577 7500 787E 8187
8D58 589C 8AC8 918A 7E0B 9323 991F 7908 8D05 8B75
8F5B 9446
7F00 7DB4
4E13 53C0 5C02 7816 5C08 587C 5AE5 911F 747C 78D7
819E 989B 750E 78DA 8AEF 87E4 9853 9C44
8F6C 5B68 8EE2 7AF1 4871 8F49
7077 556D 581F 7451 815E 8483 50CE 8D5A 64B0 7BC6
9994 7BF9 8CFA 8948 8B54 994C 56C0 7C51
5B92 80AB 8FCD 7A80 8C06 8AC4 8860
51C6 57FB 6E96 7DA7
8A30 7A15
51D6
5986 5E84 599D 5A24 6869 8358 6889 838A 6E77 7CA7
88C5 88DD 6A01 7CDA
58EE 58EF 72B6 72C0 58F5 710B 6F34 649E 6207
5E92
4E2D 4F00 6C77 5223 5990 5F78 5FE0 6CC8 7082 7EC8
26B15 67CA 76C5 949F 41D7 822F 8873 8877 7D42 9221
5E52 953A 8520 92BF 87A4 87BD 937E 9F28 8E71 9418
2915E 7C66
80BF 79CD 51A2 55A0 5C30 585A 585C 6B71 7144 816B
7607 7A2E 8E35 7A5C
4EF2 4F17 5995 72C6 794C 91CD 833D 869B 8876 5045
773E 5839 5A91 7B57 8846 8AE5
8FDA
FDD0-3114
5403 4F99 54E7 5F68 80F5 86A9 9E31 7735 7B1E 55AB
74FB 8A35 55E4 5AB8 75F4 7D7A 645B 5644 8ABA 779D
9D1F 87AD 7661 9F5D 9B51 5F72 9ED0
5F1B 6C60 9A70 577B 5CBB 8FDF 6301 7AFE 830C 6B6D
834E 86B3 8D7F 7B42 8CBE 8D8D 9045 99B3 7B8E 905F
5880 6F26 8E1F 7BEA 9072 8B18 28A32
5C3A 53FA 544E 4F88 5376 9F7F 5791 80E3 6065 3DB4
7C8E 42BC 803B 8687 6B3C 6B6F 88B2 88B3 88ED 9279
9F52 892B
5F73 53F1 65A5 6758 707B 8D64 996C 62B6 52C5 605C
70BD 52D1 7FC4 7FC5 6555 70FE 75D3 557B 3961 6E41
7873 98ED 50BA 75F8 815F 8DEE 9253 96F4 7608 7FE4
9290 6157 618F 761B 7FE8 906B 71BE 61D8 8DA9 4820
994E 9D92 9DD8
599B 9EB6
53C9 6260 6748 2472F 809E 81FF 633F 505B 63D2 63F7
9987 55CF 929F 9538 7580 8256 9364 9937
79C5 579E 67E5 832C 8336 5D56 7339 976B 643D 8A67
5BDF 69CE 78B4 25ED7 6AAB 297D4
8869 8E45 9572 9454
597C 6C4A 5C94 4F98 8BE7 59F9 7D01 5DEE 47D5 8A6B
8F66 4F21 8ECA 4FE5 7817 5513 8397 7868 86FC
626F 5056 47A3 64A6
5C6E 5F7B 577C 8FE0 3934 70E2 3FED 352D 8045 63A3
7869 9819 5FB9 64A4 6F88 3B1A 52F6 77AE 4720 7221
8286 62C6 9497 91F5
4FAA 67F4 8C7A 7961 558D 5115 9F5C
831D
867F 8883 8A0D 7625 8806 56C6
6284 2207E 5F28 600A 6B29 949E 8A2C 712F 24674 8D85
9214 52E6 4AFF
724A 6641 5DE2 5DE3 671D 9F0C 6F05 911B 5632 6A14
6F6E 7AB2 7F7A 8F48 9F02 8B3F
5435 7092 7727 7123 717C 9EA8 5DD0
4EE6 4EEF 8016 89D8
62BD 5A64 640A 7633 7BD8 72A8 72AB
4EC7 601E 4FE6 5E31 6826 60C6 7D2C 7EF8 6906 7574
7D52 83D7 6101 7697 7A20 7B79 9167 7DA2 88EF 8E0C
5114 96D4 568B 5B26 5E6C 61E4 71FD 85B5 96E0 7587
7C4C 8E8A 91BB 8B8E 8B90
4E11 4E12 541C 677B 677D 4FB4 5062 7785 919C 77C1
9B57
81ED 81F0 6BA0 905A
916C
8FBF 89C7 6400 68B4 8998 9246 88E7 92D3 5E68 895C
6519
5A75 8C17 68CE 6E79 998B 7158 7985 7F20 50DD 7351
8749 8A97 5103 5B0B 5EDB 6F79 6F7A 7DFE 6FB6 78DB
92CB 6BDA 79AA 9561 700D 87EC 913D 5133 5296 87FE
56B5 5DC9 703A 9141 6B03 7E8F 7E92 8E94 9575 826C
8B92 9471 995E
4EA7 522C 65F5 4E33 65BA 6D50 5257 8C04 5574 7522
7523 94F2 9610 5277 8487 5D7C 644C 6EFB 563D 5E5D
8AC2 95B3 9AA3 71C0 8546 7C05 5181 7E5F 8B42 8FB4
93DF 95E1 56C5 705B 8B87
5FCF 785F 3B04 6472 61F4 98A4 61FA 7FBC 97C2 986B
58E5
62BB 6375 90F4 741B 55D4 7D9D 778B 8AC3 8CDD 7E1D
8B13
5C18 81E3 5FF1 6C88 6C89 8FB0 9648 8FE7 5BB8 831E
6550 8380 8390 8A26 8C0C 8ED9 9673 6116 63E8 9202
7141 5875 760E 6A04 852F 9703 8AF6 87B4 85BC 9E8E
66DF 9DD0
8DBB 7876 789C 588B 5926 78E3 8E38 9356 8D02 91A6
75A2 886C 9F80 8D81 8D82 6987 9F53 512C 9F54 512D
56AB 8C36 6AEC 896F 8B96
70E5 6668
4F25 660C 5000 5A3C 6DD0 7316 960A 667F 7429 83D6
9520 88EE 9329 95B6 9CB3 4B96 9BE7 2A087 9F1A
4EE7 514F 80A0 82CC 9578 5C1D 507F 5E38 5F9C 74FA
8407 751E 8178 5617 5872 5AE6 747A 8193 92FF 511F
5690 9CBF 93DB 9C68
5382 573A 6636 60DD 5834 50D8 53B0 5EE0 6C05 92F9
6005 739A 7545 5021 9B2F 5531 60B5 713B 7452 66A2
757C 8AAF 97D4
655E 6919 87D0
6CDF 9637 67FD 722F 6D7E 79F0 5041 86CF 36F5 68E6
6E5E 725A 7424 8D6A 50DC 6186 645A 7A31 9757 6490
6491 7DFD 6A55 77A0 8D6C 9833 6A89 7AC0 7A6A 87F6
93F3 93FF 9953
4E1E 6210 673E 5448 627F 67A8 8BDA 4E57 57CE 5A0D
5CF8 6D06 90D5 4E58 57D5 5BAC 6330 665F 73F9 8100
6381 73F5 7A9A 812D 837F 94D6 5818 60E9 68D6 6909
7A0B 7B6C 7D7E 584D 5856 6E97 7880 41F8 88CE 8AA0
757B 9172 92EE 6195 6F82 6F84 6A59 6A99 7013 61F2
2815D 9A2C
4FB1 5F8E 609C 9A8B 5EB1 901E 7748 9A01
79E4
9BCE
51FA 5C80 521D 6474 6A17 8C99 9F63
520D 82BB 9664 53A8 8C60 9504 5AB0 6EC1 8021 870D
8D8E 924F 96CF 3551 7293 84A2 84AD 3861 5EDA 854F
92E4 6A71 7BE8 5E6E 6AC9 96DB 6AE5 85F8 8E70 8E87
9DB5 8E95
51E6 6775 7840 50A8 6918 696E 891A 6FCB 5132 6A9A
790E 9F6D 9E00 9F7C
4E8D 5904 7ACC 6035 62C0 7ECC 8C56 67F7 6B2A 7AD0
4FF6 654A 755C 3647 57F1 73FF 7D40 8655 5097 7421
6410 6EC0 89E6 9110 8E00 95A6 510A 563C 84EB 8AD4
61B7 65B6 6B5C 81C5 9EDC 89F8 77D7
695A 698B 6A7B 74B4 87F5
6B3B 6B58
9034 8E14 6233
2ECC 2ECD 8FB6 8FB5 3C8B 5A15 5A16 60D9 6DB0 7EF0
5A7C 814F 8F8D 44CE 916B 7DBD 8DA0 8F1F 9F8A 64C9
78ED 7E5B 6B60 56BD 9F6A 9461
63E3 640B
8197
555C 562C 81AA 8E39
5439 708A
5782 5015 57C0 6376 68F0 690E 8144 9672 6425 83D9
9524 69CC 7BA0 9318 9840 939A
9FA1
5DDB 5DDD 6C1A 7A7F 5276 732D 744F
4F1D 4F20 8221 8229 8239 570C 50B3 693D 9044 66B7
7BC5 8F32
821B 8348 5598 6B42 50E2 8E33
6C4C 4E32 7394 948F 91E7 8CD7 9DA8
65FE 6776 6625 583E 5A8B 8405 6699 693F 7443 7BBA
877D 6A41 8F34 81A5 6AC4 9C06 9D9E 2A0B9
7EAF 5507 6D71 7D14 9659 6DF3 8123 83BC 6E7B 7289
6EE3 6F18 8493 84F4 9187 9195 931E 9BD9
5046 60F7 8436 7776 8CF0 8822
9E51 9D89
5205 75AE 7A93 7A97 724E 6450 7255 7621 7ABB
5E8A 7240 5647 5E62
95EF 50B8 6464 78E2 95D6
521B 6006 5231 524F 5259 51D4 5275 6134
5145 51B2 5FE1 6C96 833A 6D7A 73EB 7FC0 8202 5603
644F 5FB8 6183 61A7 885D 41B9 7F7F 825F 8E56
866B 5D07 5D08 9680 7DDF 8769 8908 87F2 721E
5BA0 57EB 205A5 5BF5
94F3 63F0 9283
FDD0-3115
5C38 5931 5E08 545E 8671 8BD7 9E24 5C4D 65BD 6D49
72EE 90BF 5E2B 7D41 91F6 6E64 6E7F 921F 6EAE 6EBC
7345 8479 8A69 9247 9248 7461 8492 84CD 9CF2 8768
9CFE 27A79 9CBA 6FD5 26488 8937 9366 2A014 9BF4 9C24
9DB3 8979
5341 9963 77F3 4E6D 8FBB 65F6 20C34 5B9E 5B9F 65F9
59FC 5CD5 70BB 8680 98DF 98E0 57D8 6642 794F 83B3
5BD4 6E5C 5852 6EA1 9048 9250 5BE6 69AF 8494 8755
4215 9CA5 9F2B 9F2D 9C23
53F2 77E2 4E68 8C55 4F7F 59CB 9A76 5158 5BA9 5C4E
7B36 9242 99DB
58EB 6C0F 793B 4E17 4E16 4ED5 5E02 2EAC 793A 534B
5F0F 5FD5 2EAE 4E8A 4F3C 53D3 623A 4E8B 4F8D 52BF
5469 67F9 8BD5 9970 519F 5BA4 6040 6043 62ED 662F
6630 67BE 67FF 7702 89C6 8D33 683B 70D2 770E 7721
8210 8F7C 9002 94C8 8996 8C49 901D 91C8 5A9E 5D3C
5F11 5FA5 63D3 8C25 8CB0 91CA 52E2 55DC 5F12 7757
7B6E 89E2 8A66 8EFE 9230 9243 98FE 8213 8A93 927D
596D 9069 492D 92B4 9919 566C 5B15 6FA8 8ADF 8AE1
991D 87AB 8B1A 907E 7C2D 91CB 896B
4F66 7ACD 8BC6 62FE 5319 5D75 6981 7176 7BD2 9B96
7C42 8B58 9C18
6740 6C99 7EB1 4E77 5239 524E 7802 5526 7300 7C86
7D17 686C 6BBA 6BEE 3E3A 838E 94E9 75E7 7870 715E
88DF 699D 6A27 8531 9B66 9CA8 9BCA 9BCB 93A9
50BB 510D
503D 553C 5551 5565 5E39 53A6 55A2 8410 5EC8 3C3C
6B43 7FDC 4209 7B91 7FE3 95AF 970E
7E4C
5962 731E 8D4A 756C 7572 8F0B 8CD2 8CD6 6AA8
820C 4F58 8675 86C7 86E5 27D73
820D 6368 4B37
538D 8BBE 793E 5399 5C04 6D89 6DBB 6E09 8A2D 8D66
5F3D 6151 6442 6444 6EE0 6174 6475 850E 6B59 97D8
9A07 8802 4713 61FE 651D 7044 9E9D 6B07
820E
3D13 7B5B 917E 7BE9 7C01 7C1B 91C3
7E7A
6652 3B20 95B7 66EC
5F30 634E 70E7 68A2 83A6 713C 7A0D 65D3 7B72 8244
86F8 8F0E 71D2 4230 98B5 9AFE 9BB9
52FA 828D 67D6 73BF 82D5 7AF0 97F6
5C11 28218 4520
52AD 5372 7ECD 90B5 54E8 5A0B 7D39 8891 7744 7DA4
6F72
8571
53CE 6536
624B 5B88 57A8 9996 824F
5BFF 53D7 72E9 517D 552E 6388 6DAD 7EF6 75E9 58FD
7DAC 5900 7626 46F5 7378 93C9
624C 7363
5C71 5F61 9096 5220 522A 6749 59CD 59D7 829F 9490
67F5 72E6 73CA 8222 82EB 886B 57CF 75C1 8ED5 633B
7B18 8120 8DDA 527C 6427 5607 5E53 717D 6F78 6F98
6A86 7E3F 81BB 7FB4 7FB6 9BC5 3C11
27A3E
95EA 9655 9583 965D 3A8E 6671 7154 7752 718C 89A2
8BAA 6C55 38CC 759D 5261 6247 8A15 8D78 639E 91E4
5093 5584 928F 9A9F 50D0 58A0 58A1 6F6C 25531 7F2E
912F 5B17 64C5 6A3F 6B5A 81B3 4C49 78F0 8B06 8D61
4120 7E55 87EE 4947 87FA 8B71 8D0D 9425 994D 9A38
9CDD 7057 9C53 9C54
5738 6763 958A 657E
7533 5C7E 625F 4F38 8EAB 4F81 547B 59BD 7C76 7EC5
8BDC 59FA 67DB 6C20 73C5 7A7C 7C78 5A20 5CF7 7521
7712 7837 6552 6DF1 7D33 8398 515F 68FD 8A37 8460
88D1 8A75 20EDD 7527 84E1 8518 71CA 99EA 9CB9 66D1
8593 9D62 9BF5 9C3A
4EC0 751A 795E 4C20
5F1E 90A5 5BA1 77E4 54C2 77E7 5BB7 8C02 8C09 5A76
6E16 8A20 5BE9 8AD7 9823 9B6B 66CB 9825 77AB 5B38
700B 3C02 89BE 8B85
80BE 4FBA 661A 80C2 6D81 7718 6E17 8124 7973 814E
613C 614E 6939 7606 7F67 8703 8704 6EF2 92E0 762E
5814 698A 9C30
4F24 6B87 5546 89DE 50B7 5892 616F 6EF3 6F21 6BA4
71B5 850F 87AA 89F4 8B2A 9B3A
57A7 6244 664C 8D4F 8CDE 8D18 945C
4E04 4E0A 5C19 5C1A 6066 7EF1 7DD4 979D
4EE9 88F3
5347 751F 544F 58F0 6598 9629 6607 6CE9 72CC 680D
6B85 7272 73C4 82FC 3C61 965E 7B19 9679 6E66 713A
7525 924E 8072 9F2A 9D7F
7EF3 61B4 7E69 8B5D
7701 771A 5057 6E3B
5723 80DC 6660 5270 76DB 5269 52DD 8CB9 5D4A 741E
8056 58AD 69BA 8542 8CF8
7AD4 66FB 6A73
4E66 6BB3 5C17 6292 7EBE 3450 53D4 6778 67A2 59DD
964E 500F 5010 66F8 6B8A 7D13 6393 68B3 6DD1 7102
8ED7 758E 758F 8212 83FD 9103 6445 6BF9 7D80 8F93
8DFE 8E08 6A1E 7479 852C 8F38 6A7E 9B9B 6504 5135
9D68
79EB 5A4C 5B70 8D4E 587E 719F 74B9 8D16
9F21 5C5E 6691 9ECD 668F 7F72 8700 9F20 6F7B 66D9
3EFF 85A5 85AF 7659 85F7 8961 5C6C 8969 9483
672E 672F 620D 675F 6CAD 4FB8 51C1 54B0 6037 6811
7AD6 8FF0 6055 6352 8357 5EB6 5EBB 7D49 8853 5C0C
9683 6570 7AEA 8167 8481 88CB 9265 5885 6F31 6F44
6578 6F8D 8C4E 6A39 6FD6 9330 4240 93E3 9D90 866A
702D 7CEC 8834 9C6A 9C70
5237 5530
800D
8A9C
8BF4 54FE 8AAA 8AAC
5981 70C1 6714 94C4 6B36 7855 77DF 6420 69CA 78A9
84B4 7361 7BBE 26097 9399 720D 9460
8870 6454 2812E
7529
5E05 5E25 2361A 87C0 535B
8C01 813D 8AB0
6C34
5E28 6D97 6D9A 7971 7A05 7A0E 7761 88DE 7793
6C35 6C3A 9596
95E9 62F4 9582 6813
6DAE 8168
542E
987A 821C 9806 6A53 779A 8563 77AC 9B0A
53CC 971C 96D9 5B40 9AA6 5B47 9A3B 6B06 7935 9DDE
9E74 4276 826D 9A66 9E18
723D 587D 6161 6F3A 6A09 7E14 45EE
7040
93EF
FDD0-3116
2E9C 65E5 9A72 56F8 91F0 9224 99B9
60F9
70ED 71B1
5A06 9976 6861 835B 5B08 6A48 8558 8953 9952
6270 96A2 64FE
7ED5 9076 7E5E
53B9 79B8 67D4 5A83 63C9 6E18 7163 7448 4413 8447
7CC5 875A 8E42 8F2E 9352 97A3 74C7 9A25 9C07 9D94
7C88 697A 97D6
2EBC 8089 5B8D 816C
5465 80B0 86A6 887B 8887 86BA 88A1 7136 9AE5 562B
3723 9AEF 71C3 7E4E 4533
5184 5189 59CC 67D3 73C3 82D2 5AA3 6A6A
8485
4EBA 4EBB 4EC1 58EC 5FC8 6732 5FCE 79C2 82A2 9213
9B5C 928B 433E 9D40
5FCD 6820 6823 834F 79F9 8375 68EF 7A14
5203 5204 8BA4 4EDE 4EED 8BB1 4EFB 5C7B 5C83 3834
6268 7EAB 2EB6 2634C 598A 6752 7263 7EB4 8095 8F6B
97E7 996A 59D9 794D 7D09 7D1D 887D 8A12 8ED4 6895
3E3E 8EE0 41EE 7D4D 814D 88B5 976D 9771 97CC 98EA
845A 8A8D 9901
7D9B 8EB5
7A63 5134 52F7 703C 737D 8618 74E4 79B3 7A70 4274
8E9F 9B24
58CC 56B7 58E4 6518 7219 7E95
8BA9 61F9 8B72 8B93
6254
4ECD 20BA8 3B41 8FB8 793D 3EB1 967E
82BF
633C
5982 909A 4F9E 5E24 6847 8339 94F7 6E2A 7B4E 88BD
8498 92A3 8761 5112 8560 9D11 5685 5B2C 5B7A 6FE1
9D3D 66D8 71F8 85B7 8815 8966 98A5 91B9 986C 9C6C
6C5D 8097 4E73 8FB1 910F 64E9
5165 6D33 55D5 5AB7 6EBD 7F1B 84D0 7E1F 8925
6256 6741 8FBC 9CF0 5DBF
637C
53D2 82E5 5F31 504C 6E03 712B 9100 6949 84BB 7BAC
7BDB 7207 9C19 9C2F 9DB8
5D76
5A51 6875 7524 7DCC 8564 4B10
6A64 854A 854B 7E60 8602 8603
6C6D 6798 82AE 4132 868B 9510 745E 777F 8739 92B3
92ED 53E1 289E8 3AAB 58E1
5827 648B 58D6
962E 670A 8F6F 800E 5044 8EDF 5A86 744C 789D 7DDB
44F4 8F2D 74C0 791D
77A4
95F0 6DA6 958F 95A0 6F64 6A4D 81B6
8338
620E 809C 6804 72E8 7ED2 5BB9 3B5C 6BE7 8319 8363
70FF 4846 5D58 41EF 7D68 7FA2 5AB6 5AC6 5D64 6408
6411 6EB6 38D1 6995 69AE 69B5 7194 7462 84C9 7A41
877E 9555 7E19 878D 878E 8923 99E5 9AF6 5B2B 5DB8
7203 9394 701C 5DC6 66E7 8811 2A0FE
5197 5B82 5748 22AE8 5087 8EF5 6C04
9D27
7A43
FDD0-3117
4E72 5B5C 5179 54A8 59D5 59FF 830A 6825 7386 7D0E
8332 8D40 8D44 6DC4 79F6 7F01 8C18 55DE 5B73 5D6B
6914 6E7D 6ECB 7CA2 8F8E 5B76 8458 89DC 8A3E 8CB2
8CC7 8D91 9111 9531 798C 7A35 7DC7 922D 9543 9F87
8F1C 6FAC 8AEE 8DA6 8F3A 9319 9AED 9CBB 9F12 937F
93A1 74BE 983F 983E 9BD4 9D85 9F4D 2A5CB 9C26
84FB
20094 4ED4 5407 674D 3EAD 59C9 59CA 77F7 79C4 5470
7C7D 8014 80CF 8678 79ED 6893 7B2B 91E8 5559 7D2B
6ED3 8A3F 699F
5B57 81EA 8293 39D8 5033 525A 6063 7278 8321 6E0D
7725 7726 80D4 80FE 6F2C 3C74
5B50 5D30 6A74
5E00 531D 6C9E 5482 8FCA 62F6 7D25 7D2E 9254 9B73
81DC 81E2
6742 7838 507A 5592 97F4 96D1 22D8D 5DBB 78FC 894D
96DC 56CB 56D0 96E5
548B
5219 629E 6CA2 62E9 6CCE 6CFD 8D23 5247 8FEE 835D
5536 5567 5E3B 7B2E 8234 8CAC 6EAD 77E0 5616 5AE7
5E58 7BA6 6A0D 8ACE 8D5C 64C7 6FA4 769F 7794 7C00
790B 8B2E 8CFE 880C 8957 9F5A 456A 9F70 9E05
5928 4EC4 5E82 6C44 6603 6617 6351 5D31
4F2C 8536
707D 707E 753E 54C9 683D 70D6 6E3D 83D1 7775 8CF3
5BB0 5D3D 43C1
518D 5728 6257 4FA2 6D05 8F7D 50A4 8F09 9168 510E
7E21
8D3C 621D 8CCA 9C97 9BFD 8808 9C02 9C61
50AE 906D 7CDF 8E67 91A9
51FF 947F
65E9 67A3 86A4 68D7 6FA1 74AA 85BB 7E70 85FB
7076 7681 7682 5515 5523 688D 9020 55BF 8241 6165
566A 71E5 7C09 8B5F 8DAE 8E81 7AC3 7AC8
6806
90B9 9A7A 8BF9 90F0 966C 68F7 68F8 83C6 9112 7B83
7DC5 8ACF 9CB0 9139 9BEB 9EC0 9A36 9F71 9F7A
8D71 8D70
594F 63CD 6971 3D75
9BD0
5142 7CCC 7C2A 7C2E 9415 941F
54B1
661D 6CAF 685A 5BC1 3CEB 63DD 5646 648D 5127 6505
6512 5139 6522 8DB1 7938 8DB2
6682 3506 66AB 8CDB 8D5E 933E 6FFD 8E54 913C 74C9
8D0A 93E8 74D2 373A 7052 8B83 9147 74DA 79B6 8978
8B9A 9961
600E
8C2E 8B56 8B5B
56CE
5328 7242 7F98 8D43 8CCD 81E7 8CD8 8535 8D13 9AD2
8D1C
9A75 99D4
3638 5958 5F09 810F 585F 846C 92BA 81D3 81DF
5897 589E 618E 7F2F 912B 6A67 71B7 7494 77F0 78F3
7F7E 7E52 8B44
9503 92E5 8D60 7511 8D08
9C5B
79DF 8445 84A9
5346 8DB3 5352 54EB 5D12 5D2A 65CF 50B6 7BA4 8E24
8E3F 955E 93C3
8BC5 7EC4 963B 4FCE 723C 73C7 7956 7D44 8A5B 977B
4503
93BA
6628 79E8 637D 838B 690A 7422 7A13 7B70 923C
20087 5DE6 4F50 5528 7E53
4F5C 5750 3634 5C9D 5C9E 600D 963C 4FB3 80D9 5511
5EA7 795A 505A 3930 3B6E 888F 8443 8444 98F5 3600
7CF3
5497 84D9
539C 6718 3B70 55FA 6A36 87D5 7E97
5D8A 5634 5DB5 567F 74BB
682C 7D4A 9154 6700 666C 797D 7A21 7F6A 8FA0 69DC
917B 9189 855E 92F7 930A 6A87 6A8C
67A0 7A5D
94BB 28256 8E9C 947D
7E64 7F35 7E82 7E89 7C6B 7E98
6525 945A
5C0A 58AB 58FF 5D9F 6A3D 9075 7E5C 7F47 940F 9CDF
9C52 9DF7
50D4 5642 6499 453F 8B50
6358 928C
9D8E
5B97 5027 7EFC 9A94 582B 5D4F 5D55 60FE 68D5 7323
8159 6721 6936 847C 5D78 4053 7A2F 7D9C 7DC3 71A7
4226 7DF5 7FEA 876C 8E28 8E2A 78EB 9350 8C75 8E64
9A0C 9B03 9A23 9B09 9B37 9BEE 9BFC 9441
603B 506C 6374 60E3 6121 63D4 6403 50AF 3DD3 6460
7DCF 84D7 7E02 7E3D 93D3
7EB5 662E 75AD 500A 7314 7882 7CBD 7CC9 7632 7E26
931D 7E31 7CED
6F48
FDD0-3118
5472 75B5 8D7C 5068 8D80 8DD0 29A28 9AB4 7E12 8800
9ACA 9F79
8BCD 3639 5790 67CC 73C1 5832 7960 8308 8328 3935
74F7 46D0 8A5E 8F9D 8F9E 6148 7506 78C1 96CC 4A0F
9E5A 7CCD 8FA4 98FA 9908 3602 5B28 6FE8 858B 9D1C
7920 8FAD 9DBF 9DC0
6B64 4F4C 6CDA 73BC 7689 7D2A 9B86
673F 6B21 4F3A 4F7D 523A 523E 5E9B 6828 8326 7D58
83BF 86D3 8D50 8CDC 8786 3A5E
5693 64E6 6503
7924
906A 56C3
518A 518C 4FA7 5395 607B 62FA 6D4B 6547 755F 5074
53A0 7B27 7CA3 5EC1 60FB 6E2C 7B56 8417 7B5E 7B74
8434 5884 3A32 7BA3 84DB 61A1 25C21 7C0E
5072 5A47 731C
624D 72B2 6750 8D22 8CA1 88C1 6EA8 7E94
6BDD 91C7 5038 554B 5BC0 5F69 63A1 776C 8DF4 7DB5
8E29
57F0 68CC 83DC 8521 7E29
64A1 64CD 7CD9
66FA 66F9 5608 5D86 6F15 69FD 84F8 25562 825A 87AC
893F 93EA
8278 8349 613A 61C6 9A32
808F 9135 8959
2EBE 8279
51D1 6E4A 8160 8F8F 8F33
53C2 53C3 53C4 98E1 9A96 53C5 55B0 6E4C 50AA 5B20
9910 9A42
6B8B 8695 60ED 6B98 615A 8745 6159 45DD 5B31 8836
883A
60E8 6701 6158 61AF 7A47 7BF8 9EEA 9EF2
707F 36D1 63BA 5B71 7CB2 647B 6FAF 71E6 74A8 8592
8B32
510F 7218
5D7E
5C91 6D94 7B12 68A3
4ED3 4EFA 4F27 6CA7 82CD 9E27 5009 8231 5096 5D62
6EC4 734A 84BC 8259 87A5 9DAC
85CF 9476
416E 8CF6
6FF8 7F49 6B0C
66FD 564C
5C42 66FE 5C64 5D92 3B1D 7AF2 4255 9A53
8E6D
7C97 89D5 9E81 9E84 9EA4
5F82 6B82
4FC3 731D 8128 9162 7604 8A8E 8D97 5648 61B1 851F
8E27 918B 762F 7C07 7E2C 8E59 4944 9F00 8E74 8E75
9863
6413 7473 64AE 78CB 9073 8E49 919D
3B6B 8658 75E4 7749 77EC 5D6F 5D73 84AB 8516 9E7E
9142 9E7A 8EA6
811E
5249 5252 539D 590E 632B 63AA 839D 83A1 65AE 68E4
902A 9509 9519 84CC 6B75 92BC 932F
5D14 50AC 51D7 7F1E 5894 3720 5D89 615B 6467 69B1
7355 69EF 78EA 7E17 93D9
6F3C 7480 8DA1 76A0
4F1C 5FF0 75A9 5005 7C8B 7D23 7FC6 8103 8106 5550
555B 60B4 6DEC 6BF3 7120 813A 8403 7601 7CB9 7DB7
7FE0 81AC 81B5 6FE2 7AC1 9847 81CE 894A
4E7C
6C46 64BA 92D1 9569 8E7F 651B 8EA5 9479
6AD5 5DD1 6B11 7A73
7A9C 6BA9 71B6 7BE1 7C12 7AC4 7228
6751 90A8 76B4 8E06 6F8A 7AF4
5B58 4F9F 62F5
520C 5FD6
5BF8 540B 7C7F
5306 56EA 56F1 5FE9 679E 82C1 6031 60A4 217DF 68C7
7127 249DA 8471 6F17 8061 9AA2 66B0 6A05 6A2C 719C
747D 7481 7DEB 8066 806A 84EF 8525 779B 71EA 7BF5
8070 87CC 936F 7E71 93E6 9A18 9A44
4ECE 4E1B 5F93 5A43 5B6E 5F96 5F9E 60B0 6DD9 742E
6152 6F0E 6F40 6F68 8AB4 8CE8 8CE9 6A37 53E2 85C2
7047 6B09 721C
6181 8B25
8310
FDD0-3119
53B6 7E9F 4E1D 53F8 3EA8 7CF9 79C1 549D 6CC0 601D
8652 9E36 5AA4 65AF 7D72 7F0C 86F3 6952 3D32 9270
98D4 51D8 53AE 69B9 7997 7F73 8724 9536 5636 565D
21852 5EDD 6495 6F8C 78C3 79A0 25BE8 7DE6 92D6 71CD
856C 8784 452E 87D6 87F4 98B8 9A26 9401 29170 9DE5
9F36 7C6D
6B7B
5DF3 4E96 56DB 5BFA 6C5C 4F40 4FA1 5155 59D2 5B60
676B 6CD7 6CE4 7940 9972 9A77 3B52 67F6 726D 5A30
6D0D 6D98 8082 98E4 7B25 801C 91F2 7AE2 8997 55E3
8086 8C84 9236 923B 98FC 99DF 79A9 5129 857C 7003
4FEC 6056 92AF
4EE8 6331 6332 6492
6D12 8A2F 9778 6F75 7051 8EA0
5345 6CE7 98D2 810E 9212 8428 644B 98AF 99BA 85A9
6AD2 8644
20BBF 96A1
95AA
8272 6D13 681C 6DA9 556C 94EF 6B6E 7417 96ED 55C7
745F 6B70 92AB 6F81 61CE 64CC 6FC7 7637 7A51 6F80
74B1 7012 7A61 7E6C 8F56 93FC 8B45 98CB
6E0B 6FCF 7A6F
6BE2 6122 63CC 585E 6BF8 816E 567B 9CC3 984B 4C04
9C13
4222
55EE 8D5B 50FF 8CFD 7C3A
5625
63BB 9A9A 6145 6414 6E9E 7F2B 7E45 81CA 9CCB 9A12
9A37 9C20 9C62
626B 6383 5AC2 4545
57FD 7619 6C09 77C2 9ADE
87A6
635C 5EC0 998A 55D6 5ECB 641C 6449 6EB2 7340 910B
6457 22CB7 8490 84C3 953C 98D5 422D 8258 878B 9199
93AA 993F 98BC 98BE 9A2A
53DC 53DF 5081 55FE 778D 64DE 85AE 64FB 6AE2 85EA
7C54
8184 7636
55FD
4E09 5F0E 53C1 6BF5 4200 6BFF 7299 9B16
4ED0 4F1E 5098 7CC1 7CC2 9993 7CDD 7CE3 7CE4 7E56
93D2 93FE 9730 994A
4FD5 5E34 60B7 6563 9590
58ED 6BF6 5381 6A75
68EE 692E 69EE 8942
6852 6851
55D3 6421 78C9 892C 98A1 939F 9859
4E27 55AA
69E1
50E7 448F 9B19
82CF 7526 9165 7A23 7AA3 7A4C 6AEF 8607 8613 56CC
4FD7
738A 5919 6CDD 8083 6D2C 6D91 73DF 7D20 5BBF 6880
6B90 7C9B 83A4 901F 9A95 5083 7C9F 8C21 55C9 5850
5851 5ACA 612B 6EAF 6EB8 8085 9E54 50F3 3504 612C
69A1 8186 89EB 8D9A 9061 619F 6A0E 6A15 6F5A 6F65
24882 78BF 850C 906C 92C9 9917 3B18 6A5A 749B 7E24
7C0C 8B16 8E5C 85D7 9A4C 9C50 9DEB
8BC9 8A34 9BC2
5506 36D6 5A11 686B 68AD 838F 509E 7743 55CD 7FA7
644D 7F29 84D1 8D96 7C11 450B 7C14 7E2E 9AFF 9BBB
6240 4E7A 3ABD 5522 7D22 7410 60E2 9501 55E9 22C62
6E91 669B 7463 7485 8928 459B 9388 938D 9396 93BB
93C1
9024 6EB9 8736
7411 55E6
590A 6535 8295 867D 5020 54F8 6D7D 772D 837D 837E
6ED6 7762 7D8F 8470 71A3 6FC9 9796 96D6
7EE5 968B 9040 968F 96A8 74CD
81B8 4B49 7021 9AC4 9AD3
4E97 5C81 7815 795F 8C07 57E3 5D57 21ED5 6B72 6B73
716B 775F 788E 9042 7A42 8AB6 8CE5 5B18 6FBB 96A7
6A85 6A96 71E7 74B2 7A57 7E40 9083 79AD 7A5F 7E50
65DE 7E78 895A 8B62 9406 9429 97E2
72FB 75E0 9178
5334
7958 7B07 7B6D 7B97 849C
5B59 72F2 5B6B 836A 98E7 640E 733B 98F1 69C2 84C0
8575 859E
635F 7B0B 96BC 7B4D 640D 69AB 7BB0 7C28 93A8 9DBD
5FEA 677E 6780 5A00 67D7 502F 51C7 5D27 5EBA 68A5
6DDE 83D8 5D69 7879 8719 61BD 6FCD 290ED 6AA7 9376
9B06
6002 609A 8038 7AE6 50B1 612F 3A26 6964 5D77 616B
8073 99F7
8BBC 5B8B 8BF5 9001 9882 8A1F 980C 8AA6 9938
67A9 93B9
FDD0-311A
5475 963F 9515
55C4
554A
FDD0-311B
5594 5662
54E6
7B7D
FDD0-311C
59B8 59BF 94B6 5A3F 5A40 5C59 75FE
8BB9 542A 56EE 8FD7 4FC4 5A25 5CE8 5CE9 6D90 73F4
83AA 8A1B 7692 774B 4109 920B 9507 9E45 86FE 8A90
78C0 981F 989D 9B64 96B2 984D 9D5D 9D5E 8B4C 21145
9C2A
6799 7808 980B 5641 9A00
5384 5C75 6239 6B7A 5443 5C8B 627C 9628 545D 82CA
8F6D 9638 54A2 54B9 20C65 57A9 59F6 5CC9 7810 530E
6076 7828 4465 8685 997F 5054 537E 580A 60AA 63A0
7565 7846 8C14 8EDB 960F 35C1 582E 5D3F 60E1 6115
6E42 4748 8C5F 8EF6 9102 922A 5EC5 6424 6439 3B99
7427 816D 843C 8A7B 904C 904F 50EB 8741 9537 9E57
981E 989A 9913 9B65 5669 855A 89A8 8AE4 95BC 9929
8C96 9354 9CC4 6B5E 984E 7918 6AEE 9C10 9D9A 8B8D
9F43 9469 9F76 9C77
64DC 9D48
FDD0-311E
54C0 54CE 5509 57C3 5A2D 6328 6B38 55F3 6EBE 92B0
953F 566F 9384
5540 6371 7691 6EB0 560A 6571 6573 769A 764C 9A03
6BD0 6639 5A3E 77EE 853C 8EB7 6FED 972D 85F9 9744
4F0C 827E 3637 7231 7839 784B 35D2 55CC 5867 5AD2
611B 788D 9698 53C6 66A7 7477 95A1 50FE 416C 5B21
61D3 58D2 61DD 66D6 74A6 8586 9932 9D31 76A7 77B9
453D 99A4 7919 8B6A 8B7A 9440 9749
9C6B
FDD0-311F
8BF6 8A92
FDD0-3120
51F9 356D 67EA 688E 8EEA 720A
6556 53AB 55F7 55F8 5D85 5ED2 6EF6 7353 7352 969E
646E 23281 71AC 7488 851C 9068 78DD 7FF1 8071 87AF
7FFA 8B37 8B38 9CCC 93D6 9C32 9DD4 9F07
629D 62D7 82BA 8884 957A 5AAA 5ABC 8956
362D 5C99 6277 5773 5787 5CB0 5961 5965 50B2 5967
9A9C 371C 5AEF 6160 22CC6 58BA 5DB4 61CA 64D9 6FB3
96A9 93CA 9A41
7FF6
FDD0-3121
8BB4 6CA4 6B27 6BB4 74EF 9E25 5878 6F1A 6B50 6BC6
71B0 9D0E 750C 210BF 8B33 6AD9 93C2 9DD7 4972
8192 9F75
20676 5418 5455 5076 8162 5614 3496 8026 8545 85D5
6004 616A
85F2
FDD0-3122
5B89 4F92 5CD6 6849 6C28 5EB5 4002 8C19 5A95 83F4
75F7 8164 843B 844A 9E4C 8A9D 84ED 978C 978D 76E6
8AF3 99A3 76EB 9D6A 97FD 9D95
73B5 557D 96F8 5111
57B5 4FFA 5535 57EF 94F5 63DE 968C 7F6F 92A8
72B4 5CB8 6309 6D1D 6848 80FA 834C 8C7B 5813 5A69
4141 667B 6697 930C 95C7 9B9F 4B97 9EEF
FDD0-3123
5940 6069 217EF 717E 84BD
5CCE
6441 4B53
FDD0-3124
80AE 9AAF
536C 5C87 6602 663B 44A2 3B7F
678A 76CE 91A0
FDD0-3125
97A5
FDD0-3126
513F 800C 5150 4F95 5152 5CCF 6D0F 9651 682D 80F9
834B 5532 9E38 7CAB 804F 88BB 8F00 42E9 9C95 9AF5
96AD 9B9E 9D2F 8F5C
53BC 5C12 5C13 5C14 8033 6D31 8FE9 9975 682E 6BE6
73E5 94D2 723E 990C 99EC 85BE 9087 8DB0
4E8C 5F0D 5F10 4F74 5235 54A1 36C5 8D30 8CAE 8848
8CB3 8A80 927A 6A32
FDD0-3127
4E00 4E4A 5F0C 4F0A 8863 533B 541A 58F1 4F9D 54BF
20C96 36C4 3CD6 6D22 794E 2343F 6098 7317 94F1 58F9
63D6 6B39 86DC 90FC 5ADB 6F2A 7995 7A26 92A5 5B04
566B 5901 747F 9E65 7E44 4AD1 6AB9 6BC9 91AB 9EDF
8B69 9DD6 9EF3
4E41 4EEA 531C 572F 5937 519D 5B90 6C82 8BD2 8FC6
4F87 6021 6CB6 72CB 9974 54A6 59E8 5CD3 5DF8 5F2C
605E 62F8 67C2 73C6 886A 8D3B 8FE4 5BA7 6245 6818
684B 3EBF 74F5 7719 80F0 8A11 8CA4 8FFB 75CD 79FB
801B 8898 51D2 7FA0 8413 86E6 8A51 8A52 8CBD 5A90
6686 6938 8A83 8DE0 9057 9809 9890 98F4 7591 5100
71AA 7BB7 5DAC 5F5B 5F5C 8794 907A 9824 5BF2 5DB7
7C03 984A 294E7 4C4C 5F5D 5F5E 8B3B 93D4 89FA 3C18
8B89 9E03
4E59 5DF2 4EE5 9487 4F41 20BCB 653A 77E3 8094 5EA1
8223 82E1 82E2 8681 91D4 501A 6246 914F 506F 7B16
9018 5D3A 65D1 6905 9CE6 926F 65D6 88FF 8E26 8F22
657C 8798 49E7 6AA5 4B72 7912 8264 87FB 9857 8F59
9F6E
4E42 20086 4E49 4EBF 5F0B 5208 5FC6 808A 827A 8BAE
4EA6 3439 4F07 5C79 5F02 4F3F 4F5A 52AE 5453 5744
5F79 6291 6759 8034 8285 8BD1 9091 4F7E 546D 5479
5CC4 6008 603F 6613 678D 6B25 3CD1 6CC6 7088 79C7
7ECE 82C5 8BE3 9A7F 4FCB 5955 5E1F 5E20 5F08 223D7
67BB 6D02 6D42 73B4 75AB 7FBF 263F8 8F76 3465 5508
57BC 6092 6339 6359 6827 683A 6B2D 6D65 6D73 76CA
8875 8C0A 52DA 57F6 57F8 60A5 639C 6BB9 7570 785B
7F9B 7FCA 7FCC 88A3 8A32 8A33 8C59 8C5B 91F4 966D
96BF 5E46 6561 6679 68ED 6B94 6E59 7132 2497F 433B
86E1 8A4D 8DC7 8EFC 9038 9220 4E84 517F 3534 610F
6EA2 7348 75EC 776A 7AE9 41FC 7F22 7FA9 8084 88D4
88DB 8A63 9AAE 52E9 5AD5 5ED9 698F 7617 8189 8734
977E 99C5 5104 3989 648E 69F8 6BC5 6F69 71A0 71A4
761E 84FA 8ABC 9552 9E5D 9E62 9ED3 5293 3601 571B
58BF 5B11 5B1F 5DA7 61B6 61CC 66C0 6BAA 3D69 6FBA
71BC 71DA 7631 7796 7A53 7E0A 8257 87A0 5BF1 6581
66CE 6A8D 6B5D 71E1 71F1 7FF3 7FFC 81C6 858F 8939
8CF9 9BA8 7654 8D00 93B0 9571 7E76 7E79 85D9 85DD
8C77 972C 9BE3 9D82 9D83 39A4 7037 8B6F 8B70 91B3
91B7 9950 2113B 25725 8619 943F 9DC1 9DCA 56C8 61FF
9A5B 9DE7 9DFE 8649 897C 9F78 8B9B
5307 8864 8FB7 5B9C 7569 692C 841F 9D8D 7C4E
4E2B 5727 538B 5416 5E98 62BC 6792 57AD 9E26 6860
9E2D 57E1 5B72 690F 9D09 930F 9D28 58D3 9D76 941A
7259 4F22 5391 5C88 5393 73A1 82BD 7B0C 869C 5810
5D15 5D16 6DAF 731A 740A 7458 775A 8859 6F04 9F56
2A632
24D13 538A 5E8C 54D1 5516 555E 75D6 96C5 7602 279DD
8565
529C 5720 8F67 4E9A 897E 8BB6 4E9C 72BD 4E9E 8ECB
8FD3 5A05 631C 7811 4FF9 6C29 5A6D 6397 8A1D 94D4
63E0 6C2C 7330 8050 26716 5714 7A0F 7AAB 9F7E
2E84 4E5B 5440
54DF 5537 55B2
503B 6396 668D 6930 564E 6F71 882E
8036 6353 94D8 63F6 91FE 92E3 64E8 9381
4E5F 5414 51B6 57DC 91CE 5622 6F1C 58C4
4E1A 53F6 66F3 9875 66F5 591C 62B4 90BA 4EB1 67BC
9801 6654 67BD 70E8 35A1 5558 6DB2 8C12 5828 6B97
814B 696A 696D 8449 9113 998C 50F7 6B4B 58B7 420E
9765 5DAA 5DAB 64DB 66C4 66C5 6FB2 71C1 2681C 8B01
9134 9923 5688 64EB 66D7 76A3 77B1 9371 64EA 77B8
790F 42A6 9391 9941 9D7A 7217 9437 9768 9A5C 9E08
7237 4EAA 723A
5E7A 592D 5406 5996 6796 6B80 7945 8A1E 5593 6946
8170 847D 4301 4645 9D01 9080
723B 5C27 5C2D 80B4 579A 59DA 5CE3 409A 8F7A 5004
70D1 73E7 7A91 509C 582F 63FA 8C23 8EFA 347E 55C2
5AB1 5FAD 612E 6416 6447 733A 3A31 669A 69A3 7464
7476 9059 9065 929A 98D6 9906 5DA2 5DA4 7AAF 7AB0
4504 991A 7E47 8B20 8B21 26FBE 9390 9CD0 4B19 98BB
8628 908E 9864 9C29
4EF8 5B8E 5C86 62AD 6773 72D5 54AC 67FC 82ED 7711
7A85 7A88 8200 5060 5A79 5D3E 6E94 699A 84D4 9D22
9F3C 95C4 9A15 9F69 9DD5
7A7E 8981 94A5 25052 836F 7A94 888E 7B44 8A4F 846F
718E 899E 977F 735F 9E5E 25AAF 85AC 66DC 71FF 825E
77C5 85E5 8000 7E85 9DC2 8B91 9470
4F18 5FE7 6538 5466 602E 6CD1 5E7D 60A0 900C 9E80
6EFA 6182 512A 5698 7000 913E 6ACC 7E8B 8030
5C22 5C24 7531 6C8B 72B9 3F55 233DE 6CB9 80AC 90AE
6023 65BF 75A3 5CF3 6D5F 79DE 4343 94C0 5064 839C
83B8 86B0 8A27 6E38 7336 9030 90F5 9C7F 6962 7337
904A 923E 9C89 8F0F 99C0 8763 9B77 8555 8F36 9B8B
6AFE
6709 4E23 5363 9149 82C3 3DAD 7F91 5EAE 682F 7F90
6884 8048 811C 83A0 94D5 6E75 870F 7989 92AA 4B00
69F1 7256 9EDD 61EE
2E80 53C8 53F3 5E7C 4F51 4F91 72D6 7CFF 54CA 56FF
59F7 5BA5 5CDF 67DA 7270 8BF1 5500 7950 8FF6 4001
86B4 4EB4 8C81 91C9 916D 8A98 9F2C
53CB 5B67 848F 7257
6079 5266 70DF 73DA 80ED 5063 5571 5D26 393F 6DCA
6DF9 7109 7111 479B 9609 6E6E 7312 814C 83F8 7159
787D 5AE3 6F39 4167 9122 9183 95B9 5B2E 61E8 7BF6
61D5 81D9 9EEB
8BA0 4E25 8A00 5CA9 5EF6 6616 6CBF 708E 82AB 598D
59F8 7814 5A2E 76D0 5A2B 72FF 7402 784F 90D4 9586
960E 5D52 5D53 6E7A 839A 5869 63C5 694C 41FE 8A7D
789E 7B75 7D96 8712 8505 989C 8664 95BB 53B3 6A90
984F 9854 58DB 7C37 56B4 5DCC 6AE9 9EEC 5DD7 58E7
5B4D 5DD6 9E7D 7939 9EA3
5935 6281 6C87 4E75 5156 5944 4FE8 5157 533D 5F07
884D 5043 53A3 63A9 773C 9153 5D43 611D 624A 63DC
3B7A 68EA 6E30 6E37 7430 8412 90FE 693C 7F68 9043
9692 6F14 88FA 5D96 622D 8758 8917 565E 8EBD 9B47
7E2F 6ABF 9A13 9EE1 53B4 2502C 9C0B 9DA0 9EE4 9F5E
9F91 7517 9EED 513C 9869 9F34 5DD8 5DDA 9F39 66EE
9B58 9F74 9EF6
538C 95EB 599F 89C3 726A 54BD 59F2 5F65 5F66 781A
5501 5BB4 664F 8273 898E 9A8C 5050 7114 8C1A 55AD
5830 6565 70FB 7130 7131 786F 9681 96C1 50BF 693B
6E8E 6EDF 8455 9CEB 53AD 5895 66A5 917D 5B0A 8C33
290C0 990D 9D08 3DF3 3DFC 71C4 71D5 8AFA 8D5D 48AD
9B33 66D5 9D33 9A10 56A5 5B3F 8276 8D0B 9140 4A04
4CE1 3B2B 66E3 7213 91B6 9A34 9DC3 7054 8D17 89FE
8B8C 91BC 995C 9A57 9DF0 8277 2A64A 704E 9A60 7067
8B9E 8C53 91C5 8C54 7069
8A01 6A2E 7196 8EC5 6B15
56D9 56E0 9625 9634 4F8C 5794 59FB 39E2 6D07 97F3
9A83 6836 6BB7 6C24 8335 51D0 5A63 4104 79F5 836B
94DF 9670 5591 5819 6114 7B43 7D6A 88C0 967B 9682
6B45 6EB5 6147 647F 7616 798B 92A6 7DF8 852D 9787
8AF2 9712 290EC 99F0 567E 95C9 9720 3D8F 97FE
5198 4E51 3542 541F 72BE 65A6 82C2 57A0 6CFF 3597
5701 5CFE 70CE 72FA 73E2 8A14 8A1A 5A6C 5BC5 5D1F
5D2F 6DEB 8376 8A21 94F6 921D 9F82 6EDB 7892 5924
911E 9280 5656 6BA5 748C 8529 8ABE 569A 6AAD 87EB
4D26 972A 9F57 9DE3
4E5A 200CA 5EF4 5C39 5F15 5432 996E 8693 8D7A 6DFE
920F 9690 98F2 9777 98EE 6704 8F11 96A0 78E4 8D9B
763E 5DBE 6A83 6FE5 6FE6 87BE 96B1 6AFD 861F 766E
8B94
5370 6D15 80E4 57BD 831A 5837 6E5A 730C 5ED5 8491
9173 616D 6196 6197 764A 9BA3 3865 61DA 6ABC 4543
7C8C
592E 5489 59CE 62B0 6CF1 6B83 80E6 770F 79E7 9E2F
9260 96F5 9785 9D26
626C 7F8A 65F8 6768 7080 9626 9633 98CF 4F6F 52B7
6C1C 75A1 9496 579F 5F89 661C 6D0B 7F8F 70CA 73DC
26B75 773B 5D35 5D38 63DA 86D8 967D 656D 6698 694A
716C 760D 7993 3B15 8AF9 8F30 935A 9D39 98BA 940A
9C11 9737 9E09
4EF0 4F52 5771 5C9F 517B 67CD 70B4 6C27 75D2 7D3B
509F 8EEE 6143 6967 6C31 8746 990A 99DA 3526 61E9
6501 4446 7662
600F 6059 6837 7F95 8A47 69D8 6F3E 6A23 7001
594D 7FAA 7922
5E94 5FDC 82F1 685C 5040 5568 5A74 7EEC 83BA 20E04
21398 5A96 6E36 48D0 5AC8 6720 7150 745B 5624 6484
78A4 7F28 7F42 8CCF 9533 6A31 748E 7507 7DD3 8767
7F43 892E 9348 9D2C 9E66 5B30 61C9 81BA 9719 97FA
9E70 7516 9D91 56B6 5B46 5B7E 6516 7F4C 8B4D 9DA7
6AFB 74D4 792F 8621 8B7B 9DAF 944D 7E93 8833 9DEA
9DF9 9E0E 9E1A
76C1 8FCE 76C8 8314 8367 55B6 83B9 86CD 2870F 6E81
6E8B 8424 8426 50CC 584B 6979 6EE2 8425 843E 6F46
7192 84E5 7469 4294 877F 5B34 7E08 87A2 6FD9 6FDA
6FF4 71DF 89AE 8B0D 8D62 7005 85C0 93A3 650D 701B
7020 2294F 702F 7034 8D0F 6AFF 7C5D 7C6F
77E8 6D67 90E2 68AC 988D 9895 9896 646C 4B2C 5F71
6F41 7484 763F 7A4E 9834 5DCA 5EEE 766D
6620 786C 5AB5 668E 81A1 565F 9795 941B 9C66
73F1 6125 8747 7E04 8805 651A 7050 705C 8EC8
FDD0-3128
4E4C 572C 5F19 6C59 6C5A 6C61 545C 5DEB 6747 90AC
5C4B 6D3F 8BEC 94A8 70CF 526D 7A8F 55DA 9114 6B4D
8AA3 7BBC 8790 9D2E 93A2 9C1E
65E0 6BCB 5433 5434 543E 5449 829C 5514 5A2A 6D16
6D6F 90DA 68A7 73F8 8323 8381 7121 7966 94FB 9E40
8708 8A88 7991 7491 856A 87F1 9BC3 9D50 8B55 9F2F
9DE1
4E94 5348 4EF5 59A9 5E91 5FE4 6003 65FF 3B33 6B66
739D 4FAE 4FC9 5035 6342 554E 5A2C 727E 73F7 6440
7894 9E49 7193 7466 821E 5AF5 5EE1 61AE 6F55 511B
6A46 7512 9D61 8E8C
5140 52FF 3433 620A 4F06 5C7C 6264 9622 575E 5C89
674C 5FE2 7269 77F9 82B4 8FD5 537C 6544 8BEF 609E
609F 60AE 7C85 6664 7110 901C 5A7A 5D4D 75E6 9770
9A9B 5862 5966 5D68 6EA9 9696 96FA 96FE 5BE4 7183
8AA4 9E5C 92C8 7AB9 907B 4A01 971A 9F3F 9727 9A16
9F40 8601 9DA9
4E44 52A1 4F0D 52D9 933B
7A75 52B8 6316 6D3C 5A32 7556 7A8A 5AA7 86D9 55D7
6432 6E9B 6F25 7AAA 9F03 6528
5A03
74E6 4F64 5493 39DA 90B7
8049 889C 55E2 817D 8183 896A 97C8 97E4
5C72 54C7 74F2
631D 502D 6DA1 5529 6DB9 83B4 6E26 7327 7A9D 8435
8717 7AA9 8778 8E12 64BE
6211 5A50 6370 4C00
4EF4 6C83 809F 5367 6782 81E5 5053 637E 6DB4 5A89
5E44 63E1 6E25 7125 786A 6943 815B 65A1 7783 20FDF
64ED 6FE3 9F8C 74C1 81D2 96D8 9F77
6B6A 558E 7AF5
5D34
5916 20C3B 591E 9861
5371 5A01 20CFF 70D3 504E 55B4 364E 5A99 6104 63CB
63FB 6E28 840E 9036 9687 9688 5FAE 6933 6972 6EA6
7168 8468 8473 8A74 8732 875B 89A3 71F0 8587 9CC2
9C03 9C04 5DCD
56D7 97E6 5729 56F4 5E0F 6CA9 95F1 8FDD 5CD7 5CDE
6D08 97CB 6845 6DA0 552F 5E37 60DF 7859 7EF4 55A1
570D 5A81 5E43 6E4B 6E88 741F 912C 5D6C 9055 6F4D
7DAD 6F59 6F7F 78D1 84F6 9180 6FF0 934F 95C8 9BA0
7653 4260 89B9 973A 6B08 72A9
5383 4F1F 4F2A 5C3E 7EAC 59D4 3B4F 709C 73AE 829B
82C7 6D27 5A13 5C57 6D58 8BFF 5049 507D 5D23 68B6
75CF 784A 8371 5BEA 5D54 5FAB 6107 7325 9AA9 6690
6932 7152 744B 75FF 41FB 8172 8249 8466 848D 97EA
9AAA 9AAB 50DE 9C94 5130 64B1 78C8 25BE4 7DEF 8AC9
8E13 97D1 9820 3B19 853F 285E8 9361 9BAA 6FFB 85B3
97D9 98B9 58DD 7022 8624 97E1 6596
536B 4E3A 672A 4F4D 5473 70BA 754F 80C3 82FF 53DE
8ECE 5C09 8C13 5582 5AA6 6E2D 3DC9 7232 83CB 715F
78A8 873C 6170 71AD 729A 7DED 851A 885B 61C0 748F
7F7B 885E 8B02 9927 9B87 87B1 893D 9935 8F4A 9B4F
85EF 93CF 9728 9CDA 9956 8636 8B86 8E97 4C81 8B8F
8E9B
6364 7140 732C 589B 7E05 875F 5DB6
5F2F 525C 5A60 5E35 5846 6E7E 873F 6F6B 8C4C 5F4E
58EA 7063
4E38 5213 6C4D 7EA8 5B8C 5C8F 628F 8284 73A9 7D08
6356 987D 70F7 7413 9811 4BC8 7FEB
5B9B 5007 550D 633D 76CC 57E6 5A49 60CB 665A 6669
689A 7EFE 8118 667C 6900 742C 7696 83C0 8416 7579
7755 7897 7DA9 7DB0 8F13 8E20 92C4 92D4 4A4A 3736
4E07 534D 5350 59A7 5FE8 6365 8115 8CA6 8155 842C
8F10 6FAB 933D 858D 87C3 8D03 93AB 4951 8D0E
909C 6764 7B02
6637 6E29 586D 3B08 6985 6EAB 8F92 6B9F 7465 761F
8570 8F3C 8C71 8F40 9CC1 97B0 29948 9C1B 9C2E
5301 6587 5F63 7EB9 7086 739F 82A0 95FB 7D0B 8689
868A 73F3 960C 741D 96EF 7612 805E 99BC 9B70 9CFC
9D0D 87A1 95BA 95BF 87C1 95C5 9F24 95E6
520E 543B 5FDF 6286 5461 80B3 7D0A 687D 8117 7A33
7A4F 7A69
95EE 598F 6C76 554F 6E02 83AC 63FE 6435 9850 74BA
545A 922B 93BE
21BC1 5C23 5C29 5C2A 5C2B 6C6A
4EA1 4EBE 5166 738B 4EFC 5F7A 869F 83A3
34C1 7F52 7F51 5F80 5F83 7F54 5F8D 60D8 6680 68E2
83F5 86E7 8F8B 7DB2 8744 8AB7 8F1E 7007 9B4D
5984 5FD8 65FA 76F3 8FCB 671B 6722 4911
6789 7139
7FC1 55E1 6EC3 9E5F 8789 9393 9DB2
3622 52DC 5963 5855 5D61 66A1 4425 84CA 7788 806C
3732
74EE 8579 7515 7F4B 9F46
FDD0-3129
625C 7EA1 8FC2 8FC3 7A7B 7D06 8676 9653 5539 6DE4
76D3 6BFA 7600 7B8A
4E90 4E8E 4F03 36A5 3B30 9098 4F59 59A4 6275 6745
6B24 7397 7399 65BC 76C2 81FE 9C7C 4E7B 4FDE 516A
79BA 7AFD 4342 8867 5A1B 5A2F 5A31 6859 72F3 8201
8330 8C00 9151 9980 6E14 96E9 9B5A 5823 582C 5D33
5D4E 5D5B 3965 6109 63C4 6970 6E1D 6E61 756D 7862
8174 8438 9685 611A 65D5 3B02 6961 6986 6B48 724F
745C 8245 842E 865E 89CE 903E 9AAC 6F01 776E 7AAC
6B76 7FAD 4433 8753 8915 8ADB 96D3 9918 4C3B 6F9E
8206 854D 89A6 8E30 5B29 87B8 8F3F 935D 28A36 6B5F
74B5 8B23 9BBD 9A1F 9AC3 65DF 7C45 861B 9C05 4CA3
9DE0 9E06
4E0E 4E88 4F1B 5B87 5C7F 7FBD 2ED7 96E8 4FC1 4FE3
79B9 8BED 5704 5CFF 504A 3531 532C 5709 5EBE 6554
7964 3762 659E 9105 50B4 5BD9 6940 7440 7610 842D
38C3 8207 8A9E 7AB3 92D9 9828 9F89 5673 8C90 5DBC
61D9 399B 6594 9E8C 860C 9F6C
8080 7389 9A6D 572B 807F 59AA 5FEC 828B 828C 996B
80B2 3922 6631 72F1 79D7 90C1 4FFC 5CEA 5F67 6D74
7821 831F 94B0 9884 57DF 5809 5FA1 6086 60D0 6B32
6DE2 6DEF 8C15 9608 5585 5590 55A9 55BB 5A80 5BD3
5EBD 68DB 68DC 68EB 7134 7419 77DE 7872 26C40 9033
98EB 99AD 9E46 6108 6BD3 6EEA 715C 3F63 7A22 7F6D
8248 88D5 8A89 9047 923A 9810 5AD7 5D8E 622B 7344
7609 4205 42ED 7DCE 84AE 84E3 871F 872E 463B 8F0D
9289 564A 617E 6F4F 7A36 84F9 92CA 9CFF 224B0 6FA6
71CF 79A6 8AED 8C6B 9079 9325 95BE 9D25 9D2A 5125
2148A 71E0 7BFD 8577 8581 9E6C 7652 7907 7916 7E58
27450 91A7 9B4A 9D52 791C 7A65 9947 6AF2 9731 8B7D
8F5D 942D 6B0E 9A48 9B3B 7C5E 9C4A 9DF8 6B1D 9E12
9FA5 8EC9 9B30 9B31 706A 7C72 7229
6327 8362 6F9A 9BF2
66F0 66F1 7EA6 7D04 7BB9 5F5F 77F1 5F60
2E9D 6708 6209 5216 599C 5C84 6288 216DF 5CB3 73A5
793F 6071 4195 6085 60A6 868E 868F 8ECF 94BA 9605
6373 8DC0 8DC3 4847 7CA4 8D8A 9205 7CB5 925E 4926
95B1 95B2 6A3E 7BD7 5B33 5DBD 294E5 9FA0 3730 3B26
7039 7C46 9EE6 721A 8625 8E8D 79B4 7C65 9E11 7C70
9E19
56E6 9E22 5248 51A4 6081 7722 9E33 5BC3 6DF5 6E01
6E06 6E0A 6E15 60CC 68E9 3F93 847E 870E 9E53 7BA2
84AC 8735 88F7 9CF6 485D 99CC 9D1B 5B3D 4949 9D77
7041 9F18 9F1D
5143 5186 8D20 5458 56ED 6C85 90A7 676C 57A3 7230
8C9F 539F 54E1 5706 7B0E 8696 8881 53A1 570E 63F4
6E72 7328 7F18 8312 9F0B 5712 5713 586C 5AB4 5AC4
6E90 6E92 733F 7342 699E 69AC 849D 8F95 7DE3 7E01
875D 876F 9B6D 4CAE 6A7C 7FB1 8788 8597 8B1C 8F45
9EFF 93B1 6ADE 908D 9A35 9DA2 9DB0 53B5
8FDC 76F6 903A 9060 92FA
5917 8099 59B4 3B47 6028 82D1 57B8 884F 9662 5086
5A9B 63BE 7457 613F 7990 88EB 8911 566E 8924 9858
915B 9228
6655 7F0A 6688 7174 8480 596B 6C32 6C33 8495 7E15
8779 8D5F 9835 8D07 99A7
4E91 52FB 5300 56E9 5998 6C84 7EAD 6600 82B8 7547
7703 79D0 6DA2 7D1C 8018 803A 90E7 96F2 612A 6EB3
7B60 7B7C 9116 69B2 7189 84B7 6F90 92C6 6A52 7BD4
7E1C 8553 9942
5141 593D 628E 72C1 962D 9668 6B92 837A 5597 491E
9217 9695 6B9E 99BB 78D2 891E 8CF1 9723 9F73
5B55 679F 8FD0 607D 90D3 915D 508A 60F2 6120 9106
614D 816A 904B 97EB 97F5 7185 71A8 7DF7 7DFC 4CB0
8574 9196 8580 919E 992B 97D7 85F4 97DE 97FB 860A
62A3 7E67
4F63 62E5 75C8 9095 5EB8 50AD 55C8 96CD 5889 5ADE
6175 6EFD 9118 69E6 5670 58C5 64C1 6FAD 90FA 955B
81C3 7655 96DD 93DE 9CD9 5EF1 7049 9954 9C45 9DDB
7670
5581 63D8 7245 9899 9852 9C2B
6C38 752C 548F 6CF3 4FD1 52C7 52C8 6810 57C7 6080
67E1 6D8C 607F 3D04 509B 60E5 6111 6E67 7867 8A60
584E 5D71 5F6E 6139 86F9 6142 8E0A 799C 9CAC 8E34
9BD2
7528 82DA 919F
603A 783D
__END__

=head1 NAME

Unicode::Collate::CJK::Zhuyin - weighting CJK Unified Ideographs
for Unicode::Collate

=head1 SYNOPSIS

    use Unicode::Collate;
    use Unicode::Collate::CJK::Zhuyin;

    my $collator = Unicode::Collate->new(
        overrideCJK => \&Unicode::Collate::CJK::Zhuyin::weightZhuyin
    );

=head1 DESCRIPTION

C<Unicode::Collate::CJK::Zhuyin> provides C<weightZhuyin()>,
that is adequate for C<overrideCJK> of C<Unicode::Collate>
and makes tailoring of CJK Unified Ideographs in the order
of CLDR's zhuyin (bopomofo) ordering.

=head1 CAVEAT

The zhuyin ordering includes some characters that are not
CJK Unified Ideographs and can't utilize C<weightZhuyin()>
for collation. For them, use C<entry> instead.

=head1 SEE ALSO

=over 4

=item CLDR - Unicode Common Locale Data Repository

L<http://cldr.unicode.org/>

=item Unicode Locale Data Markup Language (LDML) - UTS #35

L<http://www.unicode.org/reports/tr35/>

=item L<Unicode::Collate>

=item L<Unicode::Collate::Locale>

=back

=cut
