#!/usr/bin/perl
#-
#-getweb.pl:  sends formatted web pages via e-mail.
#-
#-Usage:
#-         $0 [-hvD] [-r root]
#-            [(-c | -f spool | -i [-s subject] [-b body])] 
#-            [file ...]
#-
#-Where:
#-          -i : interactive mode (read in command, send result to STDOUT)
#-  -s subject : specify subject on command-line
#-     -b body : specify message body on command-line
#-     -r root : system root, default is /usr/local/getweb
#-
#-    file ... : files containing mail messages or input (default is STDIN)
#-          -c : CGI Web-interface mode
#-    -f spool : read from mail spoolfile 'spool' (example: /var/spool/mail/me)
#-
#-          -h : print this help message and exit
#-          -v : print the version number and exit
#-          -D : print debugging information
#-
#
#   Version:  1.2b1
#   Author: Rolf Nelson
#

=head1 NAME

getweb - command-line for server to provide the GetWeb service

=head1 SYNOPSIS

 Usage:
         $0 [-hvD] [-r root]
            [(-c | -f spool | -i [-s subject] [-b body])] 
            [file ...]

 Where:
          -i : interactive mode (read in command, send result to STDOUT)
  -s subject : specify subject on command-line
     -b body : specify message body on command-line
     -r root : system root, default is /usr/local/getweb

    file ... : files containing mail messages or input (default is STDIN)
          -c : CGI Web-interface mode
    -f spool : read from mail spoolfile 'spool' (example: /var/spool/mail/me)

          -h : print this help message and exit
          -v : print the version number and exit
          -D : print debugging information

by default, getweb.pl processes incoming mail:

  % getweb.pl < myrfc822msg

getweb.pl can also run in a loop from a mail-spool to reduce CPU load:

  % getweb.pl -f /var/spool/mail/getweb

getweb.pl can also be run in manual test-mode with the B<-i> flag:

  % getweb.pl -i
       OR
  % getweb.pl -i -s 'ignore this subject' -b 'GET http://www.healthnet.org/'

getweb.pl can also be run as a CGI script:

  #!/bin/sh
  # put this file in the CGI-bin area, and link HTML docs to this script
  /usr/local/bin/getweb -i -c

The B<-r> option is necessary if your config/ and pub/ directories are
somewhere other than in /usr/local/getweb/.

=head1 DESCRIPTION

GetWeb extends MailBot's message-processing capability to enable email
users to access documents from the World Wide Web.

=head1 SEE ALSO

L<GetWeb::GetWeb>, L<MailBot::MailBot>, L<MailBot::Config>,
L<http://www.healthnet.org/dist/getweb/>

=head1 COPYRIGHT

Copyright (c) 1997 SatelLife.  All rights reserved.

This library is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 AUTHOR

Rolf Nelson <rolf@usa.healthnet.org>

=cut

my $DEFAULT_ROOT = "/usr/local/getweb";

BEGIN {
    push(@INC,".");
}

$ENV{PATH} = "/bin";

use Getopt::Std;
use MailBot::Config;
use GetWeb::GetWeb;
use strict;

# untaint @ARGV to get around libwww-perl bug
grep {/((.|\n)*)/ and $_ = $1} @ARGV;
&procOpts();

my $root = $::opt_r || $DEFAULT_ROOT;
MailBot::Config -> setRoot($root);
my $mailBotConfig = MailBot::Config::current();

if ($::opt_i)
{
    $mailBotConfig -> setInteractive();

    (defined $::opt_s) and
	$mailBotConfig -> setSubject($::opt_s);

    (defined $::opt_b) and
	$mailBotConfig -> setBody($::opt_b);
}

if (defined $::opt_f)
{
    $mailBotConfig -> setMailSpool($::opt_f);
}
elsif ($::opt_c)
{
    $mailBotConfig -> setCGI();
}

my $in = shift @ARGV;
if (defined $in)
{
    open(STDIN,$in) or die "could not open $in for input: $!";
}

my $getweb = new GetWeb::GetWeb;

$getweb -> run;

0;

#----------------------------------------------------------------
# procOpts:  process command-line options
#----------------------------------------------------------------
sub procOpts
{
    ($::opt_v, $::opt_c, $::opt_h, $::opt_D,
     $::opt_i, $::opt_r) = ();  #avoid warning message
    getopts('b:cf:ir:s:hvD') || &showUsage("bad command switches");
    &d();
    $::opt_h && &showUsage();
    $::opt_v && &showVersion();
}   

#----------------------------------------------------------------
# showUsage : display a usage string, then exit.
#----------------------------------------------------------------
sub showUsage
{
    my $errMsg = shift;
    if ($errMsg ne "")
	{
	print STDERR "Usage error: $errMsg\n\n";
	}

    seek(DATA,0,0);
    while (<DATA>)
	{
	if (s/^\#\-//)
	    {
		s/\$0/$0/;
		print STDERR $_ unless /^\-/;
	    }
	}

    exit ($errMsg ne "");
}

#----------------------------------------------------------------
# showVersion : print Version and exit.
#----------------------------------------------------------------
sub showVersion
{
    seek(DATA,0,0);
    while (<DATA>)
	{
	print STDERR $_ if /\s+Version:/;
	}

    exit(0);
}

#----------------------------------------------------------------
# d : print debugging message if -D verbose flag is on.
#----------------------------------------------------------------
sub d
{
    return unless $::opt_D;
    my $msg = shift;
    if ($msg eq "")
	{					       
	print STDERR "found -D flag; running $0 in verbose DEBUG mode.\n";
	}
    else
	{
	print STDERR $msg, "\n";
	}
}

__END__
